/*
 * Socket.c
 *
 * Kevin P. Smith 1/29/89
 *
 * Routines to allow connection to the xtrek server.
 */
#include "copyright2.h"

#include <stdio.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <netinet/in.h>
#include <netdb.h>
#include "Wlib.h"
#include "defs.h"
#include "struct.h"
#include "data.h"
#include "packets.h"

int handleMessage(), handlePhaser(), handlePlyrInfo();
int handlePlayer(), handleSelf(), handleStatus();
int handleWarning(), handleTorp(), handleTorpInfo();
int handlePlanet(), handleQueue(), handlePickok(), handleLogin();
int handlePlasmaInfo(), handlePlasma();
int handleKills(), handleFlags(), handlePStatus();
int handleMotd(), handleMask();
int handleBadVersion(), handlePlanetLoc();
int handleHostile(), handleStats(), handlePlyrLogin();
int handleReserved();

struct packet_handler handlers[] = {
    { 0, NULL },	/* record 0 */
    { sizeof(struct mesg_spacket), handleMessage }, 	    /* SP_MESSAGE */
    { sizeof(struct plyr_info_spacket), handlePlyrInfo },   /* SP_PLAYER_INFO */
    { sizeof(struct kills_spacket), handleKills },	    /* SP_KILLS */
    { sizeof(struct player_spacket), handlePlayer },	    /* SP_PLAYER */
    { sizeof(struct torp_info_spacket), handleTorpInfo },   /* SP_TORP_INFO */
    { sizeof(struct torp_spacket), handleTorp }, 	    /* SP_TORP */
    { sizeof(struct phaser_spacket), handlePhaser },	    /* SP_PHASER */
    { sizeof(struct plasma_info_spacket), handlePlasmaInfo},/* SP_PLASMA_INFO */
    { sizeof(struct plasma_spacket), handlePlasma},	    /* SP_PLASMA */
    { sizeof(struct warning_spacket), handleWarning },	    /* SP_WARNING */
    { sizeof(struct motd_spacket), handleMotd },	    /* SP_MOTD */
    { sizeof(struct you_spacket), handleSelf },		    /* SP_YOU */
    { sizeof(struct queue_spacket), handleQueue },	    /* SP_QUEUE */
    { sizeof(struct status_spacket), handleStatus },	    /* SP_STATUS */
    { sizeof(struct planet_spacket), handlePlanet }, 	    /* SP_PLANET */
    { sizeof(struct pickok_spacket), handlePickok },	    /* SP_PICKOK */
    { sizeof(struct login_spacket), handleLogin }, 	    /* SP_LOGIN */
    { sizeof(struct flags_spacket), handleFlags },	    /* SP_FLAGS */
    { sizeof(struct mask_spacket), handleMask },	    /* SP_MASK */
    { sizeof(struct pstatus_spacket), handlePStatus },	    /* SP_PSTATUS */
    { sizeof(struct badversion_spacket), handleBadVersion },/* SP_BADVERSION */
    { sizeof(struct hostile_spacket), handleHostile },      /* SP_HOSTILE */
    { sizeof(struct stats_spacket), handleStats },	    /* SP_STATS */
    { sizeof(struct plyr_login_spacket), handlePlyrLogin }, /* SP_PL_LOGIN */
    { sizeof(struct reserved_spacket), handleReserved },    /* SP_RESERVED */
    { sizeof(struct planet_loc_spacket), handlePlanetLoc }  /* SP_PLANET_LOC */
};

int sizes[] = {
    0,	/* record 0 */
    sizeof(struct mesg_cpacket), 		/* CP_MESSAGE */
    sizeof(struct speed_cpacket),		/* CP_SPEED */
    sizeof(struct dir_cpacket),			/* CP_DIRECTION */
    sizeof(struct phaser_cpacket),		/* CP_PHASER */
    sizeof(struct plasma_cpacket),		/* CP_PLASMA */
    sizeof(struct torp_cpacket),		/* CP_TORP */
    sizeof(struct quit_cpacket), 		/* CP_QUIT */
    sizeof(struct login_cpacket),		/* CP_LOGIN */
    sizeof(struct outfit_cpacket),		/* CP_OUTFIT */
    sizeof(struct war_cpacket),			/* CP_WAR */
    sizeof(struct practr_cpacket),		/* CP_PRACTR */
    sizeof(struct shield_cpacket),		/* CP_SHIELD */
    sizeof(struct repair_cpacket),		/* CP_REPAIR */
    sizeof(struct orbit_cpacket),		/* CP_ORBIT */
    sizeof(struct planlock_cpacket),		/* CP_PLANLOCK */
    sizeof(struct playlock_cpacket),		/* CP_PLAYLOCK */
    sizeof(struct bomb_cpacket),		/* CP_BOMB */
    sizeof(struct beam_cpacket),		/* CP_BEAM */
    sizeof(struct cloak_cpacket),		/* CP_CLOAK */
    sizeof(struct det_torps_cpacket),		/* CP_DET_TORPS */
    sizeof(struct det_mytorp_cpacket),		/* CP_DET_MYTORP */
    sizeof(struct copilot_cpacket),		/* CP_COPILOT */
    sizeof(struct refit_cpacket),		/* CP_REFIT */
    sizeof(struct tractor_cpacket),		/* CP_TRACTOR */
    sizeof(struct repress_cpacket),		/* CP_REPRESS */
    sizeof(struct coup_cpacket),		/* CP_COUP */
    sizeof(struct socket_cpacket),		/* CP_SOCKET */
    sizeof(struct options_cpacket),		/* CP_OPTIONS */
    sizeof(struct bye_cpacket),			/* CP_BYE */
    sizeof(struct dockperm_cpacket),		/* CP_DOCKPERM */
    sizeof(struct updates_cpacket),		/* CP_UPDATES */
    sizeof(struct resetstats_cpacket),		/* CP_RESETSTATS */
    sizeof(struct reserved_cpacket)		/* CP_RESERVED */
};

#define NUM_PACKETS (sizeof(handlers) / sizeof(handlers[0]) - 1)
#define NUM_SIZES (sizeof(sizes) / sizeof(sizes[0]) - 1)

int serverDead=0;

connectToServer(port)
int port;
{
    int s;
    struct sockaddr_in addr;
    struct sockaddr_in naddr;
    int len;
    int readfds;
    struct timeval timeout;

    serverDead=0;
    if (sock!=-1) {
	shutdown(sock, 2);
	sock= -1;
    }
    sleep(3);		/* I think this is necessary for some unknown reason */

    printf("Waiting for connection. \n");

    if ((s=socket(AF_INET, SOCK_STREAM, 0)) < 0) {
	printf("I can't create a socket\n");
	exit(2);
    }

    addr.sin_family = AF_INET;
    addr.sin_addr.s_addr = INADDR_ANY;
    addr.sin_port = htons(port);

    if (bind(s, &addr, sizeof(addr)) < 0) {
	sleep(10);
	if (bind(s, &addr, sizeof(addr)) < 0) {
	    sleep(10);
	    if (bind(s, &addr, sizeof(addr)) < 0) {
		printf("I can't bind to port!\n");
		exit(3);
	    }
	}
    }

    listen(s,1);

    len=sizeof(naddr);

tryagain:
    timeout.tv_sec=240;	/* four minutes */
    timeout.tv_usec=0;
    readfds=(1<<s);
    if (select(32, &readfds, NULL, NULL, &timeout) == 0) {
	printf("Well, I think the server died!\n");
	exit(0);
    }

    sock=accept(s,&naddr,&len);

    if (sock==-1) {
	extern int errno;
/*	printf("Augh!  Bad accept: %d\n", errno);*/
	goto tryagain;
    }

/*    printf("Got connection.\n");*/

    close(s);
    pickSocket(port);	/* new socket != port */
}

callServer(port, server)
int port;
char *server;
{
    int s;
    struct sockaddr_in addr;
    struct hostent *hp;

    serverDead=0;

/*    printf("Calling %s.\n", server);*/

    if ((s=socket(AF_INET, SOCK_STREAM, 0)) < 0) 
	{
		printf("I can't create a socket\n");
		exit(0);
    }

    addr.sin_family = AF_INET;
    addr.sin_port = htons(port);
    
    if ((addr.sin_addr.s_addr = inet_addr(server)) == -1) 
	{
		if ((hp = gethostbyname(server)) == NULL) 
		{
			printf("Who is %s?\n", server);
			exit(0);
		} else 
		{
			addr.sin_addr.s_addr = *(long *) hp->h_addr;
		}
    }

    if (connect(s, &addr, sizeof(addr)) < 0) 
	{
		printf("Server not listening!\n");
		exit(0);
    }

/*    printf("Got connection.\n");*/

    sock=s;
    pickSocket(port);	/* new socket != port */
}

isServerDead()
{
    return(serverDead);
}

socketPause()
{
    struct timeval timeout;
    int readfds;

    timeout.tv_sec=1;
    timeout.tv_usec=0;
    readfds = 1<<sock;
    select(32,&readfds,0,0,&timeout);
}

readFromServer()
{
    struct timeval timeout;
    int readfds;
    char buf[BUFSIZ*2];
    char *bufptr;
    int size;
    int count;
    int temp;

    if (serverDead) return(0);
    timeout.tv_sec=0;
    timeout.tv_usec=0;
    readfds = 1<<sock;

    if (select(32,&readfds,0,0,&timeout) != 0) 
	{
		/* Read info from the xtrek server */
		count=read(sock,buf,2*BUFSIZ);
		if (count<=0) 
		{
			printf("1) Got read() of %d.  Server is dead\n", count);
			serverDead=1;
			return(0);
		}
		bufptr=buf;
		while (bufptr < buf+count) 
		{
			if (*bufptr < 1 || *bufptr > NUM_PACKETS || handlers[*bufptr].size == 0) 
			{
				printf("Unknown packet type: %d, aborting...\n", *bufptr);
				return(0);
			}
			size=handlers[*bufptr].size;
			while (size>count+(buf-bufptr)) 
			{
				/* We wait for up to ten seconds for rest of packet.
				 * If we don't get it, we assume the server died. 
				 */
				timeout.tv_sec=20;
				timeout.tv_usec=0;
				readfds=1<<sock;
				if (select(32,&readfds,0,0,&timeout) == 0) 
				{
					printf("Packet fragment.  Server must be dead\n");
					serverDead=1;
					return(0);
				}
				temp=read(sock,buf+count,size-(count+(buf-bufptr)));
				count+=temp;
				if (temp<=0) 
				{
					printf("2) Got read() of %d.  Server is dead\n", temp);
					serverDead=1;
					return(0);
				}
			}
			if (handlers[*bufptr].handler != NULL) 
			{
				(*(handlers[*bufptr].handler))(bufptr);
			} else 
			{
				printf("Handler for packet %d not installed...\n", *bufptr);
			}
			bufptr+=size;
			if (bufptr>buf+BUFSIZ) 
			{
				bcopy(buf+BUFSIZ, buf, BUFSIZ);
				if (count==BUFSIZ*2) 
				{
					readfds = 1<<sock;
					if (select(32,&readfds,0,0,&timeout) != 0) 
					{
						temp=read(sock,buf+BUFSIZ,BUFSIZ);
						count=BUFSIZ+temp;
						if (temp<=0) 
						{
							printf("3) Got read() of %d.  Server is dead\n", temp);
							serverDead=1;
							return(0);
						}
					} else 
					{	
						count=BUFSIZ;
					}
				} else 
				{
					count-=BUFSIZ;
				}
				bufptr-=BUFSIZ;
			}
		}
		return(1);
    }
    return(0);
}

handleTorp(packet)
struct torp_spacket *packet;
{
    struct torp *thetorp;

    thetorp= &torps[ntohs(packet->tnum)];
    thetorp->t_x=ntohl(packet->x);
    thetorp->t_y=ntohl(packet->y);
    thetorp->t_dir=packet->dir;
}

handleTorpInfo(packet)
struct torp_info_spacket *packet;
{
    struct torp *thetorp;

    thetorp= &torps[ntohs(packet->tnum)];
    if (thetorp->t_status==TFREE && packet->status) {
	players[thetorp->t_owner].p_ntorp++;
    }
    if (thetorp->t_status && packet->status==TFREE) {
	players[thetorp->t_owner].p_ntorp--;
    }
    thetorp->t_status=packet->status;
    thetorp->t_war=packet->war;
    if (thetorp->t_status == TEXPLODE) {
	thetorp->t_fuse=NUMDETFRAMES;
    }
}

handleStatus(packet)
struct status_spacket *packet;
{
    status->tourn=packet->tourn;
    status->armsbomb=ntohl(packet->armsbomb);
    status->planets=ntohl(packet->planets);
    status->kills=ntohl(packet->kills);
    status->losses=ntohl(packet->losses);
    status->time=ntohl(packet->time);
    status->timeprod=ntohl(packet->timeprod);
}

handleSelf(packet)
struct you_spacket *packet;
{
    me= &players[packet->pnum];
    myship = &(me->p_ship);
    mystats= &(me->p_stats);
    me->p_hostile = packet->hostile;
    me->p_swar = packet->swar;
    me->p_armies = packet->armies;
    me->p_flags = ntohl(packet->flags);
    me->p_damage = ntohl(packet->damage);
    me->p_shield = ntohl(packet->shield);
    me->p_fuel = ntohl(packet->fuel);
    me->p_etemp = ntohs(packet->etemp);
    me->p_wtemp = ntohs(packet->wtemp);
    me->p_whydead = ntohs(packet->whydead);
    me->p_whodead = ntohs(packet->whodead);
}

int handle_player_count = 0;

handlePlayer(packet)
struct player_spacket *packet;
{
    register struct player *pl;

    pl= &players[packet->pnum];
    pl->p_dir=packet->dir;
    pl->p_speed=packet->speed;
    pl->p_x=ntohl(packet->x);
    pl->p_y=ntohl(packet->y);
    redrawPlayer[packet->pnum]=1;

	handle_player_count++;
}

player_messages_count()
{
	return( handle_player_count );
}

handleWarning(packet)
struct warning_spacket *packet;
{
	printf("WARNING:  %s\n", packet->mesg );
}

sendShortPacket(type, state)
char type, state;
{
    struct speed_cpacket speedReq;

    speedReq.type=type;
    speedReq.speed=state;
    sendServerPacket(&speedReq);
}

sendServerPacket(packet)
/* Pick a random type for the packet */
struct player_spacket *packet;
{
    int size;
	
    if (serverDead) return;
    if (packet->type<1 || packet->type>NUM_SIZES || sizes[packet->type]==0) {
	printf("Attempt to send strange packet %d!\n", packet->type);
        return;
    }
    size=sizes[packet->type];
    if (gwrite(sock, packet, size) != size) {
	printf("gwrite failed.  Server must be dead\n");
	serverDead=1;
    }
}

int planet_message_count = 0;

planet_messages_count()
{
	return( planet_message_count );
}
handlePlanet(packet)
struct planet_spacket *packet;
{
    struct planet *plan;

    plan= &planets[packet->pnum];
    plan->pl_owner=packet->owner;
    plan->pl_info=packet->info;
    /* Redraw the planet because it was updated by server */
    plan->pl_flags=(int) ntohs(packet->flags) | PLREDRAW;
    plan->pl_armies=ntohl(packet->armies);
    if (plan->pl_info==0) {
		plan->pl_owner=NOBODY;
    }
	planet_message_count++;
}

handlePhaser(packet)
struct phaser_spacket *packet;
{
    struct phaser *phas;

    phas= &phasers[packet->pnum];
    phas->ph_status=packet->status;
    phas->ph_dir=packet->dir;
    phas->ph_x=ntohl(packet->x);
    phas->ph_y=ntohl(packet->y);
    phas->ph_target=ntohl(packet->target);
}

handleMessage(packet)
struct mesg_spacket *packet;
{
    if (packet->m_from >= MAXPLAYER) 
		packet->m_from=255;
	printf("MESSAGE from %d:  %s.\n", packet->m_from, packet->mesg );
}

handleQueue(packet)
struct queue_spacket *packet;
{
    queuePos = ntohs(packet->pos);

}

sendTeamReq(team,ship)
int team, ship;
{
    struct outfit_cpacket outfitReq;

    outfitReq.type=CP_OUTFIT;
    outfitReq.team=team;
    outfitReq.ship=ship;
    sendServerPacket(&outfitReq);
}

handlePickok(packet)
struct pickok_spacket *packet;
{
    pickOk=packet->state;
}

sendLoginReq(name,pass,login,query)
char *name, *pass;
char *login;
char query;
{
    struct login_cpacket packet;

    strcpy(packet.name, name);
    strcpy(packet.password, pass);
    if (strlen(login)>15) login[15]=0;
    strcpy(packet.login, login);
    packet.type=CP_LOGIN;
    packet.query=query;
    sendServerPacket(&packet);
}

handleLogin(packet)
struct login_spacket *packet;
{
    loginAccept=packet->accept;
    if (packet->accept) {
	bcopy(packet->keymap, mystats->st_keymap, 96);
	mystats->st_flags = ntohl(packet->flags);
	showShields = (me->p_stats.st_flags / ST_SHOWSHIELDS) & 1;
	mapmode = (me->p_stats.st_flags / ST_MAPMODE) & 1;
	namemode = (me->p_stats.st_flags / ST_NAMEMODE) & 1;
	keeppeace = (me->p_stats.st_flags / ST_KEEPPEACE) & 1;
	showlocal = (me->p_stats.st_flags / ST_SHOWLOCAL) & 3;
	showgalactic = (me->p_stats.st_flags / ST_SHOWGLOBAL) & 3;
    }
}

sendTractorReq(state, pnum)
char state;
char pnum;
{
    struct tractor_cpacket tractorReq;

    tractorReq.type=CP_TRACTOR;
    tractorReq.state=state;
    tractorReq.pnum=pnum;
    sendServerPacket(&tractorReq);
}

sendRepressReq(state, pnum)
char state;
char pnum;
{
    struct repress_cpacket repressReq;

    repressReq.type=CP_REPRESS;
    repressReq.state=state;
    repressReq.pnum=pnum;
    sendServerPacket(&repressReq);
}

sendDetMineReq(torp)
short torp;
{
    struct det_mytorp_cpacket detReq;

    detReq.type=CP_DET_MYTORP;
    detReq.tnum=htons(torp);
    sendServerPacket(&detReq);
}

handlePlasmaInfo(packet)
struct plasma_info_spacket *packet;
{
    struct plasmatorp *thetorp;

    thetorp= &plasmatorps[ntohs(packet->pnum)];
    if (!thetorp->pt_status && packet->status) {
	players[thetorp->pt_owner].p_nplasmatorp++;
    }
    if (thetorp->pt_status && !packet->status) {
	players[thetorp->pt_owner].p_nplasmatorp--;
    }
    thetorp->pt_status=packet->status;
    thetorp->pt_war=packet->war;
    if (thetorp->pt_status == PTEXPLODE) {
	thetorp->pt_fuse=NUMDETFRAMES;
    }
}

handlePlasma(packet)
struct plasma_spacket *packet;
{
    struct plasmatorp *thetorp;

    thetorp= &plasmatorps[ntohs(packet->pnum)];
    thetorp->pt_x=ntohl(packet->x);
    thetorp->pt_y=ntohl(packet->y);
}

handleFlags(packet)
struct flags_spacket *packet;
{
    players[packet->pnum].p_flags=ntohl(packet->flags);
    redrawPlayer[packet->pnum]=1;
}

handleKills(packet)
struct kills_spacket *packet;
{
    players[packet->pnum].p_kills=ntohl(packet->kills)/100.0;
    updatePlayer[packet->pnum]=1;
}

handlePStatus(packet)
struct pstatus_spacket *packet;
{
    if (packet->status==PEXPLODE) {
	players[packet->pnum].p_explode=0;
    }
    /* Ignore DEAD status.
     * Instead, we treat it as PEXPLODE.
     * This gives us time to animate all the frames necessary for 
     *  the explosions at our own pace.
     */
    if (packet->status==PDEAD) {
	packet->status=PEXPLODE;
    }
    players[packet->pnum].p_status=packet->status;
    redrawPlayer[packet->pnum]=1;
    updatePlayer[packet->pnum]=1;
}

handleMotd(packet)
struct motd_spacket *packet;
{
/* ignore it! */	
/*	printf("MOTD %s\n", packet->line); */
}

sendMessage(mes, group, indiv)
char *mes;
int group, indiv;
{
    struct mesg_cpacket mesPacket;

    mesPacket.type=CP_MESSAGE;
    mesPacket.group=group;
    mesPacket.indiv=indiv;
    strcpy(mesPacket.mesg, mes);
    sendServerPacket(&mesPacket);
}

handleMask(packet)
struct mask_spacket *packet;
{
    tournMask=packet->mask;
}

sendOptionsPacket()
{
    struct options_cpacket optPacket;

    optPacket.type=CP_OPTIONS;
    optPacket.flags = 
	htonl(ST_MAPMODE * (mapmode!=0) +
	ST_NAMEMODE * namemode +
	ST_SHOWSHIELDS * showShields +
	ST_KEEPPEACE * keeppeace + 
	ST_SHOWLOCAL * showlocal +
	ST_SHOWGLOBAL * showgalactic);
    bcopy(mystats->st_keymap, optPacket.keymap, 96);
    sendServerPacket(&optPacket);
}

pickSocket(old)
int old;
{
    int newsocket;
    struct socket_cpacket sockPack;

	/* Ivan Wed Jan 29 14:53:57 1992 */
    /* Egad! */
    newsocket=(getpid() & 32767);
    while (newsocket < 2048 || newsocket==old) 
	{
		newsocket=(newsocket + 10687) & 32767;
    }
    sockPack.type=CP_SOCKET;
    sockPack.socket=htonl(newsocket);
    sockPack.version=(char) SOCKVERSION;
    sendServerPacket(&sockPack);
    /* Did we get new socket # sent? */
    if (serverDead) return;
    nextSocket=newsocket;
}

handleBadVersion(packet)
struct badversion_spacket *packet;
{
    switch(packet->why) {
    case 0:
	printf("Sorry, this is an invalid client version.\n");
	printf("You need a new version of the client code.\n");
	break;
    default:
	printf("Sorry, but you cannot play xtrek now.\n");
	printf("Try again later.\n");
	break;
    }
    exit(1);
}

gwrite(fd, buf, bytes)
int fd;
char *buf;
register int bytes;
{
    long orig = bytes;
    register long n;
    while (bytes) {
        n = write(fd, buf, bytes);
        if (n < 0)
            return(-1);
        bytes -= n;
        buf += n;
    }
    return(orig);
}

handleHostile(packet)
struct hostile_spacket *packet;
{
    register struct player *pl;

    updatePlayer[packet->pnum]=1;
    pl= &players[packet->pnum];
    pl->p_swar=packet->war;
    pl->p_hostile=packet->hostile;
    redrawPlayer[packet->pnum]=1;
}

handlePlyrLogin(packet)
struct plyr_login_spacket *packet;
{
    register struct player *pl;

    updatePlayer[packet->pnum]=1;
    pl= &players[packet->pnum];

    strcpy(pl->p_name, packet->name);
    strcpy(pl->p_monitor, packet->monitor);
    strcpy(pl->p_login, packet->login);
    pl->p_stats.st_rank=packet->rank;
    if (packet->pnum == me->p_no) {
	/* This is me.  Set some stats */
	if (lastRank== -1) {
	    if (loggedIn) {
		lastRank=packet->rank;
	    }
	} else {
	    if (lastRank != packet->rank) {
		lastRank=packet->rank;
		promoted=1;
	    }
	}
    }
}

handleStats(packet)
struct stats_spacket *packet;
{
    register struct player *pl;

    updatePlayer[packet->pnum]=1;
    pl= &players[packet->pnum];
    pl->p_stats.st_tkills=ntohl(packet->tkills);
    pl->p_stats.st_tlosses=ntohl(packet->tlosses);
    pl->p_stats.st_kills=ntohl(packet->kills);
    pl->p_stats.st_losses=ntohl(packet->losses);
    pl->p_stats.st_tticks=ntohl(packet->tticks);
    pl->p_stats.st_tplanets=ntohl(packet->tplanets);
    pl->p_stats.st_tarmsbomb=ntohl(packet->tarmies);
    pl->p_stats.st_sbkills=ntohl(packet->sbkills);
    pl->p_stats.st_sblosses=ntohl(packet->sblosses);
    pl->p_stats.st_armsbomb=ntohl(packet->armies);
    pl->p_stats.st_planets=ntohl(packet->planets);
    pl->p_stats.st_maxkills=ntohl(packet->maxkills) / 100.0;
    pl->p_stats.st_sbmaxkills=ntohl(packet->sbmaxkills) / 100.0;
}

handlePlyrInfo(packet)
struct plyr_info_spacket *packet;
{
    register struct player *pl;
    static int lastship= -1;

    updatePlayer[packet->pnum]=1;
    pl= &players[packet->pnum];
    pl->p_ship.s_type = packet->shiptype;
    pl->p_team=packet->team;
    pl->p_mapchars[0]=teamlet[pl->p_team];
    pl->p_mapchars[1]=shipnos[pl->p_no];
    if (me==pl && lastship!=me->p_ship.s_type) {
/* 	calibrate_stats(); */
/*	redrawTstats(); */
    }
    redrawPlayer[packet->pnum]=1;
}

sendUpdatePacket(speed)
long speed;
{
    struct updates_cpacket packet;

    packet.type=CP_UPDATES;
    timerDelay=speed;
    packet.usecs=htonl(speed);
    sendServerPacket(&packet);
}

handlePlanetLoc(packet)
struct planet_loc_spacket *packet;
{
    struct planet *pl;

    pl= &planets[packet->pnum];
    pl->pl_x=ntohl(packet->x);
    pl->pl_y=ntohl(packet->y);
    strcpy(pl->pl_name, packet->name);
    pl->pl_namelen=strlen(packet->name);
    pl->pl_flags|=PLREDRAW;
    reinitPlanets=1;
}

handleReserved(packet)
struct reserved_spacket *packet;
{
    struct reserved_cpacket response;

    encryptReservedPacket(packet, &response, serverName, me->p_no);
    sendServerPacket(&response);
}

