/* ----------------------------- -*- Mode: C -*- -----------------------------
 * libpqcmds.c - Tcl extensions for Postgres.
 * 
 * %W%
 * 
 * Written by
 * Robin J. Maxwell
 * Open Systems Solutions, Inc
 * Collaboration Software Group.
 * 
 * Copyright (c) 1992 Open Systems Solutions, Inc
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies.  
 * Open Systems Solutions, Inc. and Robin J. Maxwell make no representations
 * about the suitability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 * 
 * Author          : Robin J. Maxwell
 * Created On      : Wed Jul 29 10:29:03 1992
 * Last Modified By: Robin J. Maxwell
 * Last Modified On: Wed Oct 28 16:07:23 1992
 * Update Count    : 10
 * Status          : Unknown, Use with caution!
 */


#include "libpqcmds.h"

/*----------------------------------------------------------------------------
 * Make the specified database the current database.
 *----------------------------------------------------------------------------
 */
int
Tcl_PQsetdbCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    libpqGlob_pt libpqGlobPtr = (libpqGlob_pt) clientData;

    if (argc != 2) {
	Tcl_AppendResult (interp, "wrong # args: ", argv [0], " dbname", 
			  (char *) NULL);
	return TCL_ERROR;
    }
    if (libpqGlobPtr->dbname)
	ckfree(libpqGlobPtr->dbname);
    libpqGlobPtr->dbname = ckalloc(strlen(argv[1]));
    strcpy(libpqGlobPtr->dbname, argv[1]);			 
    PQsetdb(libpqGlobPtr->dbname);
    return TCL_OK;
}

/*----------------------------------------------------------------------------
 * Return the current database being accessed.
 *----------------------------------------------------------------------------
 */
int
Tcl_PQdbCmd (clientData, interp, argc, argv)

    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    char *dbname;
    
    if ((dbname = PQdb()) != NULL)
	strcpy(interp->result, dbname);
    else
	*interp->result = '\0';
    return TCL_OK;
}

/*----------------------------------------------------------------------------
 * Reset the communication port with the back-end in case of errors.
 *----------------------------------------------------------------------------
 */
int
Tcl_PQresetCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    PQreset();
    return TCL_OK;
}

/*----------------------------------------------------------------------------
 * Close communications ports with the back-end.
 *----------------------------------------------------------------------------
 */
int
Tcl_PQFinishCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    PQfinish();
    return TCL_OK;
}

/*----------------------------------------------------------------------------
 * Submit a query to POSTGRES.
 *----------------------------------------------------------------------------
 */
int
Tcl_PQexecCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    if (argc != 2) {
	Tcl_AppendResult (interp, "wrong # args: ", argv [0], " query", 
			  (char *) NULL);
	return TCL_ERROR;
    }
    Tcl_SetResult(interp, PQexec(argv[1]), TCL_STATIC);
    return TCL_OK;
}

/*----------------------------------------------------------------------------
 * Return the number of open portals.
 *----------------------------------------------------------------------------
 */
int
Tcl_PQnportalsCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    int rule_p;
    
    if (argc != 2) {
	Tcl_AppendResult (interp, "wrong # args: ", argv [0], " rule_p", 
			  (char *) NULL);
	return TCL_ERROR;
    }
    
    if (Tcl_GetInt (interp, argv[1], &rule_p) != TCL_OK)
        return TCL_ERROR;
    
    sprintf (interp->result, "%d", PQnportals(rule_p));
    return TCL_OK;
}

/*----------------------------------------------------------------------------
 * Return all portal names.
 *----------------------------------------------------------------------------
 */
int
Tcl_PQnamesCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    register int i;
    int rule_p;
    char *pnames[MAXPORTALS];
    char arrayref[20];
    
    if (argc != 3) {
	Tcl_AppendResult (interp, "wrong # args: ", argv [0], " pnames rule_p", 
			  (char *) NULL);
	return TCL_ERROR;
    }
    
    if (Tcl_GetInt (interp, argv[1], &rule_p) != TCL_OK)
	return TCL_ERROR;
    
    PQnportals(pnames, rule_p);
    for (i=0;i<MAXPORTALS;i++){
	sprintf(arrayref,"%d",i);
	if (Tcl_SetVar2(interp, argv[1], arrayref, pnames[i], 
			TCL_LEAVE_ERR_MSG) == NULL)
	    return TCL_ERROR;
    }
    return TCL_OK;
}

/*----------------------------------------------------------------------------
 * Return the portal buffer handle given a portal name.
 *----------------------------------------------------------------------------
 */
int
Tcl_PQparrayCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    int rule_p;
    int walkKey;
    libpqGlob_pt libpqGlobPtr = (libpqGlob_pt) clientData;
    PBuffer_pt PBufferPtr;
    PortalBuffer *pbuff;
    
    if (argc != 2) {
	Tcl_AppendResult (interp, "wrong # args: ", argv [0], " pname", 
			  (char *) NULL);
	return TCL_ERROR;
    }
    pbuff = (PortalBuffer *)PQparray(argv[1]);
    walkKey = -1;
    while ((PBufferPtr = Tcl_HandleWalk (libpqGlobPtr->tblHdrPtr, 
					 &walkKey)) != NULL){
	if (*PBufferPtr == pbuff)
	    break;
    }
    if (PBufferPtr)
	Tcl_WalkKeyToHandle(libpqGlobPtr->tblHdrPtr, walkKey, interp->result);
    else{
	PBufferPtr = (PBuffer_pt)Tcl_HandleAlloc(libpqGlobPtr->tblHdrPtr,
						 interp->result);
 	*PBufferPtr = pbuff;
    }

    return TCL_OK;
}

/*----------------------------------------------------------------------------
 * Free storage claimed by named portal.
 *----------------------------------------------------------------------------
 */
int
Tcl_PQclearCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    if (argc != 2) {
	Tcl_AppendResult (interp, "wrong # args: ", argv [0], " pname", 
			  (char *) NULL);
	return TCL_ERROR;
    }
    
    PQclear(argv[1]);
    return TCL_OK;
}

/*----------------------------------------------------------------------------
 * Return 1 if an asynchronous portal.
 *----------------------------------------------------------------------------
 */
int
Tcl_PQrulepCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    libpqGlob_pt libpqGlobPtr = (libpqGlob_pt) clientData;
    PBuffer_pt PBufferPtr;
    
    if (argc != 2) {
	Tcl_AppendResult (interp, "wrong # args: ", argv [0], " hPortal", 
			  (char *) NULL);
	return TCL_ERROR;
    }
    if ((PBufferPtr = Tcl_HandleXlate(interp, libpqGlobPtr->tblHdrPtr, 
				      argv[1]))==NULL){
	Tcl_AppendResult (interp, " bad handle: ", argv [0], " ", argv[1], 
			  (char *)NULL);
	return TCL_ERROR;
    }    
    sprintf (interp->result, "%d", PQrulep(*PBufferPtr));
    return TCL_OK;
}

/*----------------------------------------------------------------------------
 * Return the number of instances in a portal buffer.
 *----------------------------------------------------------------------------
 */
int
Tcl_PQntuplesCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    libpqGlob_pt libpqGlobPtr = (libpqGlob_pt) clientData;
    PBuffer_pt PBufferPtr;
    
    if (argc != 2) {
	Tcl_AppendResult (interp, "wrong # args: ", argv [0], " hPortal", 
			  (char *) NULL);
	return TCL_ERROR;
    }
    if ((PBufferPtr = Tcl_HandleXlate(interp, libpqGlobPtr->tblHdrPtr, 
				      argv[1]))==NULL){
	Tcl_AppendResult (interp, " bad handle: ", argv [0], " ", argv[1], 
			  (char *)NULL);
	return TCL_ERROR;
    }    
    
    sprintf (interp->result, "%d", PQntuples(*PBufferPtr));
    return TCL_OK;
}

/*----------------------------------------------------------------------------
 * Return the number of instance groups in a portal buffer.
 *----------------------------------------------------------------------------
 */
int
Tcl_PQngroupsCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    libpqGlob_pt libpqGlobPtr = (libpqGlob_pt) clientData;
    PBuffer_pt PBufferPtr;
    
    if (argc != 2) {
	Tcl_AppendResult (interp, "wrong # args: ", argv [0], " hPortal", 
			  (char *) NULL);
	return TCL_ERROR;
    }
    if ((PBufferPtr = Tcl_HandleXlate(interp, libpqGlobPtr->tblHdrPtr,
				      argv[1])) ==NULL){
	Tcl_AppendResult (interp, " bad handle: ", argv [0], " ", argv[1], 
			  (char *)NULL);
	return TCL_ERROR;
    }    
    
    sprintf (interp->result, "%d", PQngroups(*PBufferPtr));
    return TCL_OK;
}

/*----------------------------------------------------------------------------
 * Return the number of instances in an instance group.
 *----------------------------------------------------------------------------
 */
int
Tcl_PQntuplesGroupCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    int group_index;
    libpqGlob_pt libpqGlobPtr = (libpqGlob_pt) clientData;
    PBuffer_pt PBufferPtr;
    
    if (argc != 3) {
	Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
			  " hPortal group_index", (char *) NULL);
	return TCL_ERROR;
    }
    if (Tcl_GetInt (interp, argv[2], &group_index) != TCL_OK)
        return TCL_ERROR;
    if ((PBufferPtr = Tcl_HandleXlate(interp, libpqGlobPtr->tblHdrPtr, 
				      argv[1]))==NULL){
	Tcl_AppendResult (interp, " bad handle: ", argv [0], " ", argv[1], 
			  (char *)NULL);
	return TCL_ERROR;
    }    
    
    sprintf (interp->result, "%d", PQntuplesGroup(*PBufferPtr, group_index));
    return TCL_OK;
}

/*----------------------------------------------------------------------------
 * Return the number of fields in an instance group.
 *----------------------------------------------------------------------------
 */
int
Tcl_PQnfieldsGroupCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    int group_index;
    libpqGlob_pt libpqGlobPtr = (libpqGlob_pt) clientData;
    PBuffer_pt PBufferPtr;
    
    if (argc != 3) {
	Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
			  " hPortal group_index", (char *) NULL);
	return TCL_ERROR;
    }
    if (Tcl_GetInt (interp, argv[2], &group_index) != TCL_OK)
        return TCL_ERROR;
    if ((PBufferPtr = Tcl_HandleXlate(interp, libpqGlobPtr->tblHdrPtr,
				      argv[1]))==NULL){
	Tcl_AppendResult (interp, " bad handle: ", argv [0], " ", argv[1], 
			  (char *)NULL);
	return TCL_ERROR;
    }    

    sprintf (interp->result, "%d", PQnfieldsGroup(*PBufferPtr, group_index));
    return TCL_OK;
}

/*----------------------------------------------------------------------------
 * Return the field name given the group and field index.
 *----------------------------------------------------------------------------
 */
int
Tcl_PQfnameGroupCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    int group_index;
    int field_number;
    libpqGlob_pt libpqGlobPtr = (libpqGlob_pt) clientData;
    PBuffer_pt PBufferPtr;
    
    if (argc != 4) {
	Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
			  " hPortal group_index field_number", 
			  (char *) NULL);
	return TCL_ERROR;
    }
    if ((PBufferPtr = Tcl_HandleXlate(interp, libpqGlobPtr->tblHdrPtr,
				      argv[1]))==NULL){
	Tcl_AppendResult (interp, " bad handle: ", argv [0], " ", argv[1], 
			  (char *)NULL);
	return TCL_ERROR;
    }    
    if (Tcl_GetInt (interp, argv[2], &group_index) != TCL_OK)
        return TCL_ERROR;
    if (Tcl_GetInt (interp, argv[3], &field_number) != TCL_OK)
        return TCL_ERROR;
    
    strcpy(interp->result, PQfnameGroup(*PBufferPtr, group_index, 
					field_number));
    return TCL_OK;
}

/*----------------------------------------------------------------------------
 * Return the field number (index) given the group index and field name.
 *----------------------------------------------------------------------------
 */
int
Tcl_PQfnumberGroupCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    int group_index;
    libpqGlob_pt libpqGlobPtr = (libpqGlob_pt) clientData;
    PBuffer_pt PBufferPtr;
    
    if (argc != 4) {
	Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
			  " hPortal group_index field_name", 
			  (char *) NULL);
	return TCL_ERROR;
    }
    if ((PBufferPtr = Tcl_HandleXlate(interp, libpqGlobPtr->tblHdrPtr,
				      argv[1]))==NULL){
	Tcl_AppendResult (interp, " bad handle: ", argv [0], " ",  argv[1], 
			  (char *)NULL);
	return TCL_ERROR;
    }    
    if (Tcl_GetInt (interp, argv[2], &group_index) != TCL_OK)
        return TCL_ERROR;
    
    sprintf (interp->result, "%d", PQfnumberGroup(*PBufferPtr, group_index,
						  argv[3]));
    return TCL_OK;
}

/*----------------------------------------------------------------------------
 * Returns the index of the group that a particular instance is in.
 *----------------------------------------------------------------------------
 */
int
Tcl_PQgetgroupCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    int tuple_index;
    libpqGlob_pt libpqGlobPtr = (libpqGlob_pt) clientData;
    PBuffer_pt PBufferPtr;
    
    if (argc != 3) {
	Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
			  " hPortal tuple_index", (char *) NULL);
	return TCL_ERROR;
    }
    if ((PBufferPtr = Tcl_HandleXlate(interp, libpqGlobPtr->tblHdrPtr,
				      argv[1]))==NULL){
	Tcl_AppendResult (interp, " bad handle: ", argv [0], " ",  argv[1], 
			  (char *)NULL);
	return TCL_ERROR;
    }    
    
    if (Tcl_GetInt (interp, argv[2], &tuple_index) != TCL_OK)
        return TCL_ERROR;
    
    sprintf (interp->result, "%d", PQgetgroup(*PBufferPtr,tuple_index));
    return TCL_OK;
}

/*----------------------------------------------------------------------------
 * Return the number of fields in an instance.
 *----------------------------------------------------------------------------
 */
int
Tcl_PQnfieldsCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    int tuple_index;
    libpqGlob_pt libpqGlobPtr = (libpqGlob_pt) clientData;
    PBuffer_pt PBufferPtr;
    
    if (argc != 3) {
	Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
			  " hPortal tuple_index", (char *) NULL);
	return TCL_ERROR;
    }
    if ((PBufferPtr = Tcl_HandleXlate(interp, libpqGlobPtr->tblHdrPtr,
				      argv[1]))==NULL){
	Tcl_AppendResult (interp, " bad handle: ", argv [0], " ",  argv[1], 
			  (char *)NULL);
	return TCL_ERROR;
    }    
    if (Tcl_GetInt (interp, argv[2], &tuple_index) != TCL_OK)
        return TCL_ERROR;
    
    sprintf (interp->result, "%d", PQnfields(*PBufferPtr,tuple_index));
    return TCL_OK;
}

/*----------------------------------------------------------------------------
 * Return the field index of a given field name within and instance.
 *----------------------------------------------------------------------------
 */
int
Tcl_PQfnumberCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    int tuple_index;
    
    if (argc != 4) {
	Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
			  " PortalBuffer tuple_index field_name", 
			  (char *) NULL);
	return TCL_ERROR;
    }
    if (Tcl_GetInt (interp, argv[2], &tuple_index) != TCL_OK)
        return TCL_ERROR;
    
    sprintf (interp->result, "%d", PQfnumber(argv[1], tuple_index, argv[3]));
    return TCL_OK;
}

/*----------------------------------------------------------------------------
 * Return the name of a field.
 *----------------------------------------------------------------------------
 */
int
Tcl_PQfnameCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    int tuple_index;
    int field_number;
    libpqGlob_pt libpqGlobPtr = (libpqGlob_pt) clientData;
    PBuffer_pt PBufferPtr;
    
    if (argc != 4) {
	Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
			  " hPortal tuple_index field_number", 
			  (char *) NULL);
	return TCL_ERROR;
    }
    if ((PBufferPtr = Tcl_HandleXlate(interp, libpqGlobPtr->tblHdrPtr,
				      argv[1]))==NULL){
	Tcl_AppendResult (interp, " bad handle: ", argv [0], " ",  argv[1], 
			  (char *)NULL);
	return TCL_ERROR;
    }    
    if (Tcl_GetInt (interp, argv[2], &tuple_index) != TCL_OK)
        return TCL_ERROR;
    if (Tcl_GetInt (interp, argv[3], &field_number) != TCL_OK)
        return TCL_ERROR;
    
    strcpy(interp->result, PQfname(*PBufferPtr, tuple_index, field_number));
    return TCL_OK;
}

/*----------------------------------------------------------------------------
 * Return the type of a field.
 *----------------------------------------------------------------------------
 */
int
Tcl_PQftypeCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    int tuple_index;
    int field_number;
    libpqGlob_pt libpqGlobPtr = (libpqGlob_pt) clientData;
    PBuffer_pt PBufferPtr;
    
    if (argc != 4) {
	Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
			  " hPortal tuple_index field_number", 
			  (char *) NULL);
	return TCL_ERROR;
    }
    if ((PBufferPtr = Tcl_HandleXlate(interp, libpqGlobPtr->tblHdrPtr,
				      argv[1]))==NULL){
	Tcl_AppendResult (interp, " bad handle: ", argv [0], " ",  argv[1], 
			  (char *)NULL);
	return TCL_ERROR;
    }    
    if (Tcl_GetInt (interp, argv[2], &tuple_index) != TCL_OK)
        return TCL_ERROR;
    if (Tcl_GetInt (interp, argv[3], &field_number) != TCL_OK)
        return TCL_ERROR;
    
    sprintf (interp->result, "%d", PQftype(*PBufferPtr,tuple_index, 
					   field_number));
    return TCL_OK;
}

/*----------------------------------------------------------------------------
 * Return 1 if two instances have the same attributes.
 *----------------------------------------------------------------------------
 */
int
Tcl_PQsametypeCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    int tuple_index1;
    int tuple_index2;
    libpqGlob_pt libpqGlobPtr = (libpqGlob_pt) clientData;
    PBuffer_pt PBufferPtr;
    
    if (argc != 4) {
	Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
			  " hPortal tuple_index1 tuple_index2", 
			  (char *) NULL);
	return TCL_ERROR;
    }
    if ((PBufferPtr = Tcl_HandleXlate(interp, libpqGlobPtr->tblHdrPtr,
				      argv[1]))==NULL){
	Tcl_AppendResult (interp, " bad handle: ", argv [0], " ",  argv[1], 
			  (char *)NULL);
	return TCL_ERROR;
    }    
    if (Tcl_GetInt (interp, argv[2], &tuple_index1) != TCL_OK)
        return TCL_ERROR;
    if (Tcl_GetInt (interp, argv[3], &tuple_index2) != TCL_OK)
        return TCL_ERROR;
    
    sprintf (interp->result, "%d", PQsametype(*PBufferPtr, tuple_index1, 
					      tuple_index2)); 
    return TCL_OK;
}

/*----------------------------------------------------------------------------
 * Return an attribute (field) value.
 *----------------------------------------------------------------------------
 */
int
Tcl_PQgetvalueCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    int tuple_index;
    int field_number;
    libpqGlob_pt libpqGlobPtr = (libpqGlob_pt) clientData;
    PBuffer_pt PBufferPtr;
    
    if (argc != 4) {
	Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
			  " hPortal tuple_index field_number", 
			  (char *) NULL);
	return TCL_ERROR;
    }
    if ((PBufferPtr = Tcl_HandleXlate(interp, libpqGlobPtr->tblHdrPtr,
				      argv[1]))==NULL){
	Tcl_AppendResult (interp, " bad handle: ", argv [0], " ",  argv[1], 
			  (char *)NULL);
	return TCL_ERROR;
    }    
    if (Tcl_GetInt (interp, argv[2], &tuple_index) != TCL_OK)
        return TCL_ERROR;
    if (Tcl_GetInt (interp, argv[3], &field_number) != TCL_OK)
        return TCL_ERROR;
    
    Tcl_SetResult(interp, PQgetvalue(*PBufferPtr, tuple_index, 
				     field_number), TCL_VOLATILE);
    return TCL_OK;
}

/*----------------------------------------------------------------------------
 * Make the specified database the current database.
 *----------------------------------------------------------------------------
 */
int
Tcl_PQtraceCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    PQtrace();
    return TCL_OK;
}

/*----------------------------------------------------------------------------
 * Disable tracing.
 *----------------------------------------------------------------------------
 */
int
Tcl_PQuntraceCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    PQuntrace();
    return TCL_OK;
}

/*----------------------------------------------------------------------------
 * Cleanup global data.
 *----------------------------------------------------------------------------
 */
static void
libPQCleanUp (clientData)
    ClientData clientData;
{
    libpqGlob_pt    libpqGlobPtr = (libpqGlob_pt) clientData;
    PBuffer_pt PBufferPtr;
    int            walkKey;
    
    libpqGlobPtr->useCount--;
    if (libpqGlobPtr->useCount > 0)
        return;
    
    if (libpqGlobPtr->dbname)
	ckfree(libpqGlobPtr->dbname);
    
#ifdef NEED_LIBPQCLEANUP
    walkKey = -1;
    while ((PBufferPtr = Tcl_HandleWalk (libpqGlobPtr->tblHdrPtr, 
					 &walkKey)) != NULL);
#endif
    
    Tcl_HandleTblRelease (libpqGlobPtr->tblHdrPtr);
    ckfree ((char *) libpqGlobPtr);
}

/*----------------------------------------------------------------------------
 * Initialize environment for libPQ commands.
 *----------------------------------------------------------------------------
 */
void
Tcl_PQlibinit(interp)
Tcl_Interp *interp;
{
    libpqGlob_pt    libpqGlobPtr;
    void_pt        fileCbTblPtr;
    
    libpqGlobPtr = (libpqGlob_pt) ckalloc (sizeof (libpqGlob_t));
    libpqGlobPtr->useCount=0;
    libpqGlobPtr->dbname=NULL;
    libpqGlobPtr->tblHdrPtr = 
        Tcl_HandleTblInit ("Portal", sizeof (PBuffer_pt), MAXPORTALS);
    
    Tcl_CreateCommand (interp, "PQsetdb", Tcl_PQsetdbCmd,
		       (ClientData)libpqGlobPtr, libPQCleanUp);
    libpqGlobPtr->useCount++;
    Tcl_CreateCommand (interp, "PQdb", Tcl_PQdbCmd,
		       (ClientData)libpqGlobPtr, libPQCleanUp);
    libpqGlobPtr->useCount++;
    Tcl_CreateCommand (interp, "PQreset", Tcl_PQresetCmd,
		    (ClientData)libpqGlobPtr, libPQCleanUp);
    libpqGlobPtr->useCount++;
    Tcl_CreateCommand (interp, "PQfinish", Tcl_PQFinishCmd,
		       (ClientData)libpqGlobPtr, libPQCleanUp);
    libpqGlobPtr->useCount++;
    Tcl_CreateCommand (interp, "PQexec", Tcl_PQexecCmd,
		       (ClientData)libpqGlobPtr, libPQCleanUp);
    libpqGlobPtr->useCount++;
    Tcl_CreateCommand (interp, "PQnportals", Tcl_PQnportalsCmd,
		       (ClientData)libpqGlobPtr, libPQCleanUp);
    libpqGlobPtr->useCount++;
    Tcl_CreateCommand (interp, "PQnames", Tcl_PQnamesCmd,
		       (ClientData)libpqGlobPtr, libPQCleanUp);
    libpqGlobPtr->useCount++;
    Tcl_CreateCommand (interp, "PQparray", Tcl_PQparrayCmd,
		       (ClientData)libpqGlobPtr, libPQCleanUp);
    libpqGlobPtr->useCount++;
    Tcl_CreateCommand (interp, "PQclear", Tcl_PQclearCmd,
		       (ClientData)libpqGlobPtr, libPQCleanUp);
    libpqGlobPtr->useCount++;
    Tcl_CreateCommand (interp, "PQrulep", Tcl_PQrulepCmd,
		       (ClientData)libpqGlobPtr, libPQCleanUp);
    libpqGlobPtr->useCount++;
    Tcl_CreateCommand (interp, "PQntuples", Tcl_PQntuplesCmd,
		       (ClientData)libpqGlobPtr, libPQCleanUp);
    libpqGlobPtr->useCount++;
    Tcl_CreateCommand (interp, "PQngroups", Tcl_PQngroupsCmd,
		       (ClientData)libpqGlobPtr, libPQCleanUp);
    libpqGlobPtr->useCount++;
    Tcl_CreateCommand (interp, "PQntuplesGroup", Tcl_PQntuplesGroupCmd,
		       (ClientData)libpqGlobPtr, libPQCleanUp);
    libpqGlobPtr->useCount++;
    Tcl_CreateCommand (interp, "PQnfieldsGroup", Tcl_PQnfieldsGroupCmd,
		       (ClientData)libpqGlobPtr, libPQCleanUp);
    libpqGlobPtr->useCount++;
    Tcl_CreateCommand (interp, "PQfnameGroup", Tcl_PQfnameGroupCmd,
		       (ClientData)libpqGlobPtr, libPQCleanUp);
    libpqGlobPtr->useCount++;
    Tcl_CreateCommand (interp, "PQfnumberGroup", Tcl_PQfnumberGroupCmd,
		       (ClientData)libpqGlobPtr, libPQCleanUp);
    libpqGlobPtr->useCount++;
    Tcl_CreateCommand (interp, "PQgetgroup", Tcl_PQgetgroupCmd,
		       (ClientData)libpqGlobPtr, libPQCleanUp);
    libpqGlobPtr->useCount++;
    Tcl_CreateCommand (interp, "PQnfields", Tcl_PQnfieldsCmd,
		       (ClientData)libpqGlobPtr, libPQCleanUp);
    libpqGlobPtr->useCount++;
    Tcl_CreateCommand (interp, "PQfnumber", Tcl_PQfnumberCmd,
		       (ClientData)libpqGlobPtr, libPQCleanUp);
    libpqGlobPtr->useCount++;
    Tcl_CreateCommand (interp, "PQfname", Tcl_PQfnameCmd,
		       (ClientData)libpqGlobPtr, libPQCleanUp);
    libpqGlobPtr->useCount++;
    Tcl_CreateCommand (interp, "PQftype", Tcl_PQftypeCmd,
		       (ClientData)libpqGlobPtr, libPQCleanUp);
    libpqGlobPtr->useCount++;
    Tcl_CreateCommand (interp, "PQsametype", Tcl_PQsametypeCmd,
		       (ClientData)libpqGlobPtr, libPQCleanUp);
    libpqGlobPtr->useCount++;
    Tcl_CreateCommand (interp, "PQgetvalue", Tcl_PQgetvalueCmd,
		       (ClientData)libpqGlobPtr, libPQCleanUp);
    libpqGlobPtr->useCount++;
    Tcl_CreateCommand (interp, "PQtrace", Tcl_PQtraceCmd,
		       (ClientData)libpqGlobPtr, libPQCleanUp);
    libpqGlobPtr->useCount++;
    Tcl_CreateCommand (interp, "PQuntrace", Tcl_PQuntraceCmd,
		       (ClientData)libpqGlobPtr, libPQCleanUp);
    libpqGlobPtr->useCount++;
}
