#
# Generic Registrar Client
# ------------------------
#
# This is not a runnable program itself, but can be sourced by "specific"
# Registrar Client programs (e.g. openrc.tcl).
#
# A Registrar Client connects to the central Registrar (see reg.tcl), and
# sends requests to modify the lists of conferences and their users that the
# Registrar maintains.  The Registrar Client acts as the "controller" in
# an open protocol, implementing specific policies that determine how users 
# can join and leave conferences.  It is possible to define many such
# Registrar Clients, offering different semantics and interfaces.
#
# This file contains generic routines which interface with the Registrar,
# for sending and receiving messages, etc.  It is up to particular Registrar
# Clients to implement at least the following routines which are called by
# the generic Registrar Client code (see openrc.tcl) for one implementation:
#
#     foundNewConf     - called when a new conference has been created
#     foundNewUser     - called when a new user is found for a conference
#     foundDeletedUser - called when a user is deleted from a conference
#     foundDeletedConf - called when a conference is deleted
#
# This file also contains routines which spawn and then interact with
# particular groupware applications (termed conferences).  This subsumes
# the functionality of the Coordinator object from the InterViews 
# implementation.  These routines forward directives from the registration
# system (such as requests to connect to a new user) to the appropriate
# conference application.
#


#
# temporary hack - force everyone to call us so that we'll be sourced and
#   the stuff below will be run.  eventually move that into initGenericRC
#   properly
#

proc initGenericRC {} {

}

initGkErrorHandling 

catch {source ~/.tclgkrc}

foreach i $argv {
    if {[string range $i 0 1] == "-p"} {
	set regportnum [string range $i 2 [expr [string length $i]-1]]
    }
    if {[string range $i 0 1] == "-h"} {
	set reghostname [string range $i 2 [expr [string length $i]-1]]
    }
}

if {[info exists myusername] == 0} {
    set myusername [exec whoami]
}
if {[info exists internetdomain] == 0} {
    set internetdomain ""
}
if {[info exists regportnum] == 0} {
    set regportnum 9068
}
if {[info exists reghostname] == 0} {
    set reghostname manaslu
}

#
# connect up to a particular central Registrar.  Change this to connect
#    to a different Registrar (e.g. for offsite conferences).  This will
#    eventually be expanded so that a command line flag can override this.
#

set registrar [MakeRPCClient $reghostname $regportnum]


#
# set up a server so that programs can connect to us (typically the groupware
#    conferences that we spawn, but conceivably other Registrar Clients also)
#

set myport [MakeRPCServer 0]
set hostprefix [exec hostname]
set host $hostprefix$internetdomain


#
# conflist and userlist() essentially echo the lists maintained by the
#    central Registrar.  
#

set conflist {}
set userlist(0) {}


#
# This is a generic proc used to update both the conference lists and the
#    user lists.  Assume we want to update the list of conferences.  The 
#    scenario here is as follows:
#      
#       1. The central Registrar calls the Registrar Client's "conflist"
#          proc, passing it the most up to date list of conferences.  
#          (This is invoked from the Registrar's disp_conference proc,
#          typically because one of the Registrar Clients called
#          PollConference to request the conference list).
#
#       2. Our conflist proc calls updatelist, passing parameters as follows:
#
#           list: $confs              - the newly updated list 
#                                       (a list of keyed lists)
#           oldlist: conflist         - the most recent local copy of the list
#           root: conflist            - ditto (used to declare list as global)
#           key: confnum              - the key from each keyed list whose 
#                                       value is a unique id number identifying
#                                       the list element
#           newproc: foundNewConf     - a proc to call if a new item is found
#           delproc: foundDeletedConf - a proc to call if an item was deleted
#
#       3. The proc basically compares the new list with the old list.
#          If items appear in the new list but not the old list, the
#          newproc is called.  If items appear in the old list but not the
#          new list, the delproc is called.  
#
#       4. This routine does NOT change oldlist (e.g. conflist) so that its 
#          contents reflect that of the new list.  Such changes can of course
#          be made by the newproc and delprocs if appropriate for the 
#          particular registration policy.  While this seems weird, its
#          actually an important part of what gives the open protocol its
#          flexibility.
#

proc updatelist {list oldlist root key newproc delproc} {
    global $root
    set newlist {}
    set gonelist {}
    set lst [eval [set cmd "concat \$$oldlist"]]

    set i 0; while {$i < [llength $list]} {
	set x [lindex $list $i]
	set cnfnum [keylget x $key]
	set pattern "*\{$key $cnfnum\}*"
	set posn [lsearch $lst $pattern]
	if {$posn == -1} { lappend newlist $x }
	incr i
    }

    set i 0; while {$i < [llength $lst]} {
	set x [lindex $lst $i]
	set cnfnum [keylget x $key]
	set pattern "*\{$key $cnfnum\}*"
	set posn [lsearch $list $pattern]
	if {$posn == -1} { lappend gonelist $cnfnum }
	incr i
    }

    foreach i $newlist { eval $newproc [list $i] }
    foreach i $gonelist { eval $delproc [list $i] }
}


#
# update the conference list in response to new info from the registrar
#

proc conflist confs {
    updatelist $confs conflist conflist confnum foundNewConf foundDeletedConf
}


#
# update the userlist for a particular conference in response to new info
#    from the registrar.  
#

proc userlist {conf users} { 
    global userlist
    if {[info exists userlist($conf)] == 0} {
	set userlist($conf) ""
    }
    set zzz "foundDeletedUser $conf"
    updatelist $users userlist($conf) userlist usernum foundNewUser $zzz
}


#
# a utility proc that given a unique conference number will locate the
#    keyed list containing the conference's information and return it to
#    the caller

proc findconfinfo confnum {  global conflist
    foreach i $conflist {
	set conf [keylget i confnum]
	if {$conf == $confnum} { return $i }
    }
    return ""
}



#
# These routines are used to send messages to the central Registrar.  They
#    can be called by routines in the specific Registrar Client.  All are
#    just for convenience and reasonably straightforward.  See openrc.tcl
#    for examples of how they can be used
#

proc callJoinConference confnum {	global myport registrar host myusername
	keylset user confnum $confnum userid [exec whoami] host $host port $myport \
	    username $myusername
	RDO $registrar add_user $user
}

proc callJoinConf user { global registrar; RDO $registrar add_user $user }
proc callLeaveConference {conf user} { global registrar; RDO $registrar delete_user $conf $user }
proc callNewConference conf { global registrar; RDO $registrar new_conference $conf }
proc callDeleteConference confnum { global registrar; RDO $registrar delete_conference $confnum }
proc PollConferences {} { global registrar; RDO $registrar disp_conference }
proc PollUsers confnum { global registrar; RDO $registrar disp_users $confnum }


#
# The rest of this file deals with connections to the conferences that we
#    create - the so-called "Coordinator" responsibilities.  
#
# The first thing we keep is a list of all the conferences connected to us.
#    Each conference is represented as a keyed list, with information about
#    the conference.  This information is transmitted by the conference
#    after it has been created and connected back to us. 
#

set conferences ""


#
# Given a unique conference number, return the keyed list containing that
#    conference's information
#

proc findconf confnum {  global conferences
    foreach i $conferences {
	set conf [keylget i confnum]
	if {$conf == $confnum} { return $i }
    }
    return ""
}


#
# This routine creates a groupware application process (conference).  
#    The first parameter is a keyed list containing information about the
#    conference, while the second specifies the user number which will be
#    given the local user of the conference.
#
# This routine assumes the existence of an option database (ie. .Xdefaults)
# containing a block of entries like the following:
#
#    *conferenceTypes:       3
#    *conf1-desc:            Brainstorming Conference
#    *conf1-prog:            dpwish -f brainstorm.tcl
#    *conf2-desc:            GroupSketch
#    *conf2-prog:            dpwish -f sketch.tcl
#    *conf3-desc:            PostIt Notes
#    *conf3-prog:            dpwish -f postit.tcl
#
# The conferenceTypes resource specifies the number of different conferences
# which exist, while the remaining pairs of resources specify the type of the
# conference and the actual command used to start the conference application.
#
# This routine looks at the confType key in the 'conf' parameter, and finds
# the conf?-desc resource which matches it.  The corresponding conf?-prog
# resource gives the command line of the program to execute.  This is
# supplemented with information needed by the conference application, so that
# the resulting information includes (at a minimum):
#
#    confname   - the name assigned to the conference
#    confType   - the type of the conference (used above to run the program)
#    originator - the host and port number (e.g. janu9865) of the Registrar
#                 client which first created the conference
#    confnum    - the unique conference id number assigned by the Registrar 
#    reghost    - the host of the Registrar Client (i.e. us), which the new
#                 conference will connect to as soon as it is created
#    regport    - the port number of the Registrar Client (i.e. us)
#    usernum    - the unique user number assigned by the Registrar
#    userid     - the userid (login) of the user
#
# The complete command line (name of program plus above information) is then
# exec'ed to start up the actual conference application.
#
# NOTE: we need to put a timer on this and check to make sure the conference
#       really did get created and connect back up; if not, delete it via
#       the Registrar
#
# CHANGED to use gk_program variable rather than Xdefaults

proc createConference {conf usernum} { global myport myusername host gk_program

#    set i 1; while {$i <= [option get . conferenceTypes conferenceTypes]} {
#	if {[keylget conf confType] == [option get . conf${i}-desc blah]} {
#	    set cmdline [option get . conf${i}-prog blah]
#	    keylset conf reghost $host regport $myport usernum $usernum \
#		userid [exec whoami] username $myusername
#	    set cmd [concat "exec " $cmdline " " $conf " &"]
#	    eval $cmd
#	}
#	incr i
#    }

     set cmdline $gk_program([keylget conf confType])
     keylset conf reghost $host regport $myport usernum $usernum \
        userid [exec whoami] username $myusername
     set cmd [concat $cmdline " " $conf " &"]
     eval $cmd

}


#
# One of the first things that the new conference will do is connect back
#    up to us and call this method, which essentially tells us which conference
#    is hooked up (so we can associate the file description with the conference
#    number.  This routine also will send any pending messages to the 
#    newly connected conference (see the toConf routine, immediately below).
#

proc remoteInfo whom {  global conferences rpcFile pending
    keylset whom filedesc $rpcFile
    lappend conferences $whom
    if {[info exists pending([keylget whom confnum])]} {
	foreach i $pending([keylget whom confnum]) {
	    eval [concat RDO [keylget whom filedesc] $i]
	}
    } 
}


#
# This routine is used to send messages to a particular conference.  This
#    routine is used rather than sending directly, because it may be desirable
#    to send messages to the conference *before* its actually been created
#    and connected back to us (remember there is a time lag between when we
#    ask it to be created and it really is there).  If the conference is
#    present and can accept messages, the message is send immediately.  If
#    not, the message is queued up in the pending() array, whose contents
#    will be sent to the conference when it does hook up, via the 
#    remoteInfo method above.

proc toConf {confnum msg} { global pending
    set conf [findconf $confnum]
    if {$conf == ""} {
	lappend pending($confnum) $msg
    } else {
	eval [concat RDO [keylget conf filedesc] $msg]
    }
}


#
# This routine is used to tell the local instantiation of a particular
#    conference to connect up to the indicated user (e.g. because that
#    user just joined the conference.
#
# Things here are not quite as straightforward as they seem.  While the
# 'user' parameter does contain a host and a port number, this host and
# port number is *not* the host and port of the conference application
# that our conference should connect to, but in fact the host and port
# number of that user's Registrar Client.  (There's a reason for this,
# but don't worry about it).
#
# Therefore, we have to first locate the actual host and port number of
# the user's conference application.  This is done (surprise) through the
# user's Registrar Client, in particular the address routine (below).  
# After retrieving the correct host and port number, we then instruct
# our conference to connect to the new user's conference application.
#
# We do some retry things in case the remote conference hasn't been created
# yet.  Not well tested yet.
#

proc joinTo {user conf} {
    set remote [MakeRPCClient [keylget user host] [keylget user port]]
    set addr [RPC $remote address [keylget conf confnum]]
    CloseRPC $remote
    if {$addr == ""} {
	after 1000 joinTo [list $user] [list $conf]
    } else {
	toConf [keylget conf confnum] [concat connectTo \{$addr\}]
    }
}


#
# Given the unique conference number, return the host and port number of 
#    the corresponding conference.  Used by the above routine.
#

proc address confnum {
    set theconf [findconf $confnum]
    if {$theconf == ""} {
	return ""
    } else {
	return $theconf
    }
}


#
# This is called (via routines in errors.tcl) if a socket connection closes.
#    Presumably the other end of the socket was one of the groupware 
#    conference applications we spawned.  Call the "userLeft" routine (below)
#    to do some clean up.
#

proc socket_closed filedesc {  global conferences
    foreach i $conferences {
	if {[keylget i filedesc] == $filedesc} {
	    userLeft [keylget i confnum] [keylget i usernum]
	}
    }
}


#
# After a conference is killed (gracefully or otherwise), first tell the
#    central Registrar that the user has left the conference.  If the user
#    was the last user in the conference, tell the Registrar to delete the
#    conference, since no one is left.
#

proc userLeft {confid userid} {  global userlist
    callLeaveConference $confid $userid
    if {[llength $userlist($confid)] < 2} {
	callDeleteConference $confid
    }
    PollUsers $confid
    PollConferences
}


