#---------------------------------------------------------------------------
#
#	Timeline Display Widget
#
#---------------------------------------------------------------------------

defwidget Timeline Window {
	{cw 200}
	{ch 200}
	{chartx 300}
	{fac 1.2}
}

# Default unit system description:
#	o  number of base units per displayed vertical bar;
#	o  number of units per vertical bar;
#	o  name of units;
#	o  short name of units.
#	o  rest: special options passed to the formatter.
#
Timeline slot scale {
		{1 1 second s}
		{1 5 second s}
        	{1 10 second s}
        	{1 30 second s}
        	{60 1 minute min}
        	{60 5 minute min}
        	{60 10 minute min}
        	{60 30 minute min}
        	{3600 1 hour h}
        	{3600 3 hour h}
        	{3600 6 hour h}
        	{3600 12 hour h}
        	{86400 1 day d}
        	{86400 7 day d}
	        {86400 10 day d}
       		{86400 30 day d}
        	{31536000 1 year yr}
		}

defmethod Timeline new {name args} {

  args	{text Timeline} {layout choose} {units 50} {sep 5} {type times} \
	factor scale actions initmethod init embedded buttons scaleinfo \
	{textfont bold} {chartfont bold} {scalefont small} {split false} \
	{showscale true} {showstretch true} format

  set stuff [app(get) $initmethod $init]

  # Add default help action
  if { [assoc Help $actions] == {} } {
    lappend actions [list Help {} Timeline/help]
  }

  # Create toplevel window
  if { $embedded == "true" } {
    Frame new $name -relief flat
  } {
    Toplevel new $name \
	-title $text \
	-resizable true \
	-buttons $buttons \
	-handler [list $name _do] \
	-actions $actions
  }
  defsuper $name Timeline

  if { $scaleinfo == {} } {
    set scaleinfo [$self slot scale]
  }
  $name slot _scaleinfo $scaleinfo
  set split [expr {$split == "true"}]
  $name slot _split $split

  if { $format == {} } {
    set format defaultFormat
  }
  $name slot format $format

  $name slot _list $stuff
  $name slot _fonts [list $textfont $chartfont $scalefont]

  $name slot _listaction [lrange [assoc List $actions] 1 end]
  $name slot _chartaction [lrange [assoc Chart $actions] 1 end]
  $name slot _type $type
  $name slot _units $units
  $name slot _sep $sep

  if { $factor == {} } {
    if { $type == "times" } {
      set factor 1.0
    } {
      set factor 1.3
    }
  }
  if { $factor < 1.0 } { set factor 1.0 }
  $name slot _fac $factor
  $name slot _fac2 1

  if { $scale == {} } {
    set max 0
    foreach item [$name slot _list] {
      foreach spec [lrange $item 2 end] {
	set length [expr [lindex $spec 0]+[lindex $spec 1]]
	if { $length > $max } {
	  set max $length
	}
      }
    }
    set max [expr $max/30]
    $name slot _scale [expr [llength [$name slot _scaleinfo]]-1]
    set pos 0
    foreach i [$name slot _scaleinfo] {
      if { [expr [lindex $i 0]*[lindex $i 1]] > $max } {
	$name slot _scale $pos
	break
      }
      incr pos
    }
  } {
    $name slot _scale 3
    set pos 0
    foreach i [$name slot _scaleinfo] {
      if { [lindex $i 2] == $scale } {
	$name slot _scale $pos
	break
      }
      incr pos
    }
  }
  $name slot _defaultscale [$name slot _scale]

  # Vertical scrollbar
  if { $split } {
    Scrollbar new $name.vscroll -dir vertical \
	-action [list $name _yscroll]
  } {
    Scrollbar new $name.vscroll -dir vertical \
	-action [list $name.left.c yview]
  }

  set bits(-) [the(image) button-less]
  set bits(0) [the(image) button-zero]
  set bits(+) [the(image) button-more]

  # Display for current scaling/stretching factors
  if { $showscale == "true" } {
    Frame new $name.s1 -relief flat
    Pair new $name.s1.scale -layout {left} -label "Current Scale:"
    foreach i {- 0 +} {
      Button new *$name.s1 -bitmap @$bits($i) -layout {left padx 10} \
	-relief raised -action [list $name Scale$i]
    }
    $name.s1 layout {top fillx}
  }
  if { $showstretch == "true" } {
    Frame new $name.s2 -relief flat
    Pair new $name.s2.stretch -layout {left} -label "Current Stretch:"
    foreach i {- 0 +} {
      Button new *$name.s2 -bitmap @$bits($i) -layout {left padx 10} \
	-relief raised -action [list $name Stretch$i]
    }
    $name.s2 layout {top fillx}
  }

  # List of item labels
  if { $split } {
    set xscroll [list $name _xscroll left right]
    set leftlayout {left filly frame nw}
    set width [$self slot cw]
  } {
    set xscroll [list $name.left.c xview]
    set leftlayout {left expand fill frame nw}
    set width [expr [$self slot cw]+[$self slot chartx]]
  }
  Canvas new $name.left -layout $leftlayout \
	-width $width -height [$self slot ch] \
	-xscroll $xscroll \
	-yscroll [list $name.vscroll set] \
	-scroll [list [Timeline slot cw] [Timeline slot ch]]
  $name.left slot _lastx 0

  $name.left.c bind item(list) <Any-Enter> \
	[list $name _event left enter]
  $name.left.c bind item(list) <Any-Leave> \
	[list $name _event left leave]
  $name.left.c bind item(list) <1> \
	[list $name _event left select]

  if { $split } {
    Canvas new $name.right -layout {left expand fill frame nw} \
	-width [Timeline slot chartx] -height [$self slot ch] \
	-xscroll [list $name _xscroll right left] \
	-yscroll [list $name.vscroll set] \
	-scroll [list [Timeline slot cw] [Timeline slot ch]]
    $name.right slot _lastx 0
    set win right

    bind $name.left.c <Button-2> \
	[list + $name.right.c scan mark 0 %y]
    bind $name.left.c <Button2-Motion> \
	[list + $name.right.c scan dragto 0 %y]

    bind $name.right.c <Button-2> \
	[list + $name.left.c scan mark 0 %y]
    bind $name.right.c <Button2-Motion> \
	[list + $name.left.c scan dragto 0 %y]
  } {
    set win left
  }

  $name.$win.c bind item(chart) <Any-Enter> \
	[list $name _event right enter]
  $name.$win.c bind item(chart) <Any-Leave> \
	[list $name _event right leave]
  $name.$win.c bind item(chart) <1> \
	[list $name _event right select]

  # Pack the vertical scroll bar
  $name.vscroll layout {right filly}

  # Draw the timeline
  $name _drawText
  $name _drawChart

  $name layout $layout
}

#---------------------------------------------------------------------------

defmethod Timeline _drawText {} {

  set fnt [lindex [$self slot _fonts] 0]
  $self.left.c delete item(list)
  $self.left.c delete item(shadow)

  set fntX [Font slot $fnt]
  set fntXw [Font slot $fnt,w]
  set fntXh [Font slot $fnt,h]

  # Determine the maximum widths of ids and labels
  set xe $fntXw
  set xte $xe
  foreach item [$self slot _list] {
    set size [textsize $fntX [lindex $item 1]]
    if { $xe < [lindex $size 0] } {
      set xe [lindex $size 0]
    }
    set size [textsize $fntX [lindex $item 0]]
    if { $xte < [lindex $size 0] } {
      set xte [lindex $size 0]
    }
  }
  set pos 0
  set fac [expr [Timeline slot fac]*[$self slot _fac]*[$self slot _fac2]]
  set height [expr $fntXh*$fac+[$self slot _sep]]
  set x $fntXw
  set ye [expr $height+$fntXh]
  set y [expr $ye-$fntXh*[Timeline slot fac]]
  set xe [expr 2*$fntXw+$xe]
  set xt [expr $xe+$fntXw]
  set xte [expr $xt+$xte+2*$fntXw]

  set colfg [Color slot fg]
  set colbgb [Color slot bg,button]

  foreach item [$self slot _list] {

    $self.left.c newRectShadow $xt $y $xte $ye -noitem true \
	-tagsshadow item(shadow)
    $self.left.c newRectShadow $x $y $xe $ye -noitem true \
	-tagsshadow item(shadow)

    # Draw the actual label and its rectangle
    $self.left.c create rectangle $xt $y $xte $ye \
	-width 1 -outline $colfg -fill $colbgb \
	-tags [list item(list) rect [list pos $pos]]
    $self.left.c create text [expr $xt+$fntXw] [expr ($y+$ye)/2] \
	-anchor w -text [lindex $item 0] -font $fntX \
	-tags {item(list) text}

    # Draw the small rectangle with the item id
    $self.left.c create rectangle $x $y $xe $ye \
	-width 1 -outline $colfg -fill $colbgb \
	-tags {item(list) rect2}
    $self.left.c create text [expr ($x+$xe)/2] [expr ($y+$ye)/2] \
	-anchor center -text [lindex $item 1] -font $fntX \
	-tags {item(list) text2}

    set y [expr $y+$height]
    set ye [expr $ye+$height]
    incr pos
  }

  # Adjust the sizes. Return max_y because we need it for $self.left.c.
  # Allow for extra space for the time scale labels.
  #
  set sfont [lindex [$self slot _fonts] 2]
  set max_y [expr $ye-$height+[$self slot _sep]+3*[Font slot $sfont,w]]
  $self slot _max_y $max_y

  if { [$self slot _split] } {
    $self slot _min_x 0
  } {
    $self slot _min_x [expr -$fntXw-$xte]
    $self.left.c move item(list) [$self slot _min_x] 0
    $self.left.c move item(shadow) [$self slot _min_x] 0
  }

  return $self
}

defmethod Timeline _drawChart {} {

  set sfont [lindex [$self slot _fonts] 2]
  set cfont [lindex [$self slot _fonts] 1]
  set lfont [lindex [$self slot _fonts] 0]

  set colfg [Color slot fg]
  set colbg [Color slot bg]
  set colbb [Color slot bg,button]

  set split [$self slot _split]

  # Cleanup the canvas
  if { $split } {
    set canvas $self.right.c
  } {
    set canvas $self.left.c
  }
  $canvas delete item(chart)

  # Display the current scaling
  set scale [lindex [$self slot _scaleinfo] [$self slot _scale]]
  set scale "[lindex $scale 1] [lindex $scale 2]"
  if { [lindex $scale 0] != 1 } {
    append scale "s"
  }
  if { [winfo exists $self.s1] } {
    $self.s1.scale set "$scale per vertical bar"
  }
  if { [winfo exists $self.s2] } {
    $self.s2.stretch set [format "Stretching factor %0.1f" [$self slot _fac2]]
  }

  # Compute the scaling factor
  set sc [lindex [$self slot _scaleinfo] [$self slot _scale]]
  set fact [expr 1.0*[lindex $sc 0]*[lindex $sc 1]/[$self slot _units]]

  # Compute the horizontal size of the timeline display
  set max 0
  foreach item [$self slot _list] {
    foreach spec [lrange $item 2 end] {
      set length [expr [lindex $spec 0]+[lindex $spec 1]]
      if { $max < $length } {
	set max $length
      }
    }
  }
  set max_x [expr $max/$fact+3*[$self slot _units]]
  if { $max_x < 200 } {
    set max_x 200
  }
  set right [expr $max_x+5*[Font slot $lfont,w]]
  $canvas adjust [list [$self slot _min_x] 0 $right [$self slot _max_y]]
  $self.left.c xview 0
  $canvas xview 0

  set offset [$self slot _units]

  # Draw the vertical lines
  set dash [the(image) dash1]
  set sfontX [Font slot $sfont]
  set sfontXh [Font slot $sfont,h]
  set sep [$self slot _sep]
  set tmax_y [expr [$self slot _max_y]+$sep-$sfontXh*5/2]
  set extra 0

  for {set i 0} {$i < $max_x} {set i [expr $i+[$self slot _units]]} {
    set x [expr $i+$offset]
    $canvas create line $x 0 $x [expr $tmax_y+$extra] \
	-tags item(chart) -width 1 -fill $colfg -stipple @$dash

    $canvas create text $x [expr $tmax_y+$extra] \
	-tags item(chart) -font $sfontX -anchor n \
	-text [$self [$self slot format] [expr $i/[$self slot _units]] $sc]
    set extra [expr $sfontXh-$extra]
  }

  set lfontX [Font slot $lfont]
  set lfontXh [Font slot $lfont,h]

  # Draw the horizontal lines and all timeline elements
  set pos 0
  set fac [expr [Timeline slot fac]*[$self slot _fac]*[$self slot _fac2]]
  set height [expr $lfontXh*$fac+[$self slot _sep]]
  set y [expr [$self slot _sep]+$lfontXh]
  set ye [expr $height+$lfontXh]
  set ym [expr ($y+$ye)/2]

  set xb0 [expr [$self slot _units]/2]
  set xb1 [expr [$self slot _units]*3/4]

  set before [the(image) timeline-before]
  set after [the(image) timeline-after]
  set gray [the(image) timeline-gray]

  set type [$self slot _type]

  foreach item [$self slot _list] {

    $canvas create line $offset $ye $max_x $ye \
	-tags item(chart) -width 1 -fill $colfg -stipple @$dash

    $canvas create polygon \
	$xb0 [expr $y+3] $xb1 $ym $xb0 [expr $ye-3] \
	-fill $colbb -tags {item(chart) pointer}
    $canvas create line \
	$xb0 [expr $y+3] $xb1 $ym $xb0 [expr $ye-3] $xb0 [expr $y+3] \
	-fill $colfg -tags {item(chart)}

    set subpos 0
    foreach spec [lrange $item 2 end] {
      set x [expr [lindex $spec 0]/$fact+$offset]
      set xe [expr $x+[lindex $spec 1]/$fact]

      case $type {
      {times} {
	  #-- Display timeline items {START DURATION BEFORE AFTER}
	  #
	  $canvas create rectangle $x $y $xe $ye \
		-fill $colbb -width 1 \
		-tags [list item(chart) rect [list pos $pos $subpos]] \

	  $canvas create text [expr ($x+$xe)/2] $ym \
		-tags {item(chart) text} -font [Font slot $cfont] \
		-anchor center -text [lindex $item 1]

	  if { [lindex $spec 2] > 0 } {
	    $canvas create rectangle \
		[expr $x-[lindex $spec 2]/$fact] $y $x $ye \
		-tags {item(chart) before} -fill $colbg -width 1

	    $canvas create rectangle \
		[expr $x-[lindex $spec 2]/$fact] $y $x $ye \
		-tags {item(chart) before} -fill $colfg -width 1 \
		-stipple @$before
	  }

	  if { [lindex $spec 3] > 0 } {
	    $canvas create rectangle \
		$xe $y [expr $xe+[lindex $spec 3]/$fact] $ye \
		-tags {item(chart) after} -fill $colbg -width 1
	    $canvas create rectangle \
		$xe $y [expr $xe+[lindex $spec 3]/$fact] $ye \
		-tags {item(chart) after} -fill $colfg -width 1 \
		-stipple @$after
	  }
	}

      {usage} {
	  #-- Display resource utilization items {START DURATION HEIGHT}
	  #
	  set ry [expr $ye-($ye-$y)*[lindex $spec 2]/[lindex $item 1]]
	  $canvas create rectangle $x $ry $xe $ye \
		-fill $colbb -outline $colfg \
		-width 1 \
		-tags [list item(chart) rect [list pos $pos $subpos]]
	  $canvas create rectangle $x $ry $xe $ye \
		-fill $colfg -outline $colfg -width 1 \
		-tags {item(chart) rectx} \
		-stipple @$gray
	}

      }
      incr subpos
    }
    set y [expr $y+$height]
    set ye [expr $ye+$height]
    set ym [expr $ym+$height]
    incr pos
  }

  return $self
}

defmethod Timeline defaultFormat {units scaleinfo} {

  # *** This needs some application-dependent reworking!! ***
  return "[expr $units*[lindex $scaleinfo 1]] [lindex $scaleinfo 3]"
}

#---------------------------------------------------------------------------

defmethod Timeline _xscroll {win1 win2 pos} {

  $self.$win1 slot _lastx $pos
  $self.$win1.c xview $pos
  $self.$win2.c xview [$self.$win2 slot _lastx]
}

defmethod Timeline _yscroll {pos} {

  $self.left.c yview $pos
  $self.right.c yview $pos
}

#---------------------------------------------------------------------------

defmethod Timeline Scale- {} {

  set scale [$self slot _scale]
  if { $scale > 0 } {
    incr scale -1
    $self slot _scale $scale
    $self _drawChart
  }
}

defmethod Timeline Scale0 {} {

  $self slot _scale [$self slot _defaultscale]
  $self _drawChart
}

defmethod Timeline Scale+ {} {

  set scale [$self slot _scale]
  if { $scale < [expr [llength [$self slot _scaleinfo]]-1] } {
    incr scale
    $self slot _scale $scale
    $self _drawChart
  }
}

defmethod Timeline Stretch- {} {

  set fac2 [$self slot _fac2]
  if { $fac2 >= 1 } {
    set fac2 [expr $fac2-0.5]
    if { $fac2 < 1 } {
      set fac2 [expr 1.0/[Timeline slot fac]]
    }
    $self slot _fac2 $fac2
    $self _drawText
    $self _drawChart
  }
}

defmethod Timeline Stretch0 {} {

  if { [$self slot _fac2] != 1 } {
    $self slot _fac2 1
    $self _drawText
    $self _drawChart
  }
}

defmethod Timeline Stretch+ {} {

  set fac2 [$self slot _fac2]
  if { $fac2 < 6 } {
    $self slot _fac2 [expr $fac2+0.5]
    $self _drawText
    $self _drawChart
  }
}

#---------------------------------------------------------------------------

defmethod Timeline _do {button action} {

  case $button {
    {Help Dismiss Print} {
	$self $button $action
      }
    default {
	eval [concat $action [list $self]]
      }
  }
}

defmethod Timeline _event {side event} {

  if { [$self slot _split] } {
    set canvas $self.$side.c
  } {
    set canvas $self.left.c
  }

  set id [$canvas find withtag current]
  set tags [$canvas gettags $id]

  foreach tag $tags {
    if { $tag == "text" } {
      return [$self _handle $canvas $side $event [expr $id-1]]
    }
    if { $tag == "rect" } {
      return [$self _handle $canvas $side $event $id]
    }
    if { $tag == "text2" } {
      return [$self _handle $canvas $side $event [expr $id-3]]
    }
    if { $tag == "rect2" } {
      return [$self _handle $canvas $side $event [expr $id-2]]
    }
    if { $tag == "rectx" } {
      return [$self _handle $canvas $side $event [expr $id-1]]
    }
    if { $tag == "pointer" } {
      return [$self _pointer $canvas $event $id]
    }
  }
}

defmethod Timeline _handle {canvas side event id} {

  # `id' is the canvas item id of the first rectangle
  case $event in {
  {enter} {
      $canvas itemconfigure $id -fill [Color slot bg,active]
    }
  {leave} {
      $canvas itemconfigure $id -fill [Color slot bg,button]
    }
  {select} {
      set tags [$canvas gettags $id]
      set pos [assoc pos $tags]
      if { $side == "left" && [$self slot _listaction] != {} } {
	eval [concat [$self slot _listaction] $pos \
		     [list [lindex [$self slot _list] [lindex $pos 0]]]]
      }
      if { $side == "right" && [$self slot _chartaction] != {} } {
	eval [concat [$self slot _chartaction] $pos \
		     [list [lindex [$self slot _list] [lindex $pos 0]]]]
      }
    }
  }

  incr id

  # `id' now is the id of the first text item
  case $event in {
  {enter} {
      $canvas itemconfig $id -fill [Color slot fg,active]
    }
  {leave} {
      $canvas itemconfig $id -fill [Color slot fg]
    }
  }

  if { $side == "right" } {
    return $self
  }

  incr id 

  # `id' now is the id of the second rectangle item
  case $event in {
  {enter} {
      $canvas itemconfigure $id -fill [Color slot bg,active]
    }
  {leave} {
      $canvas itemconfigure $id -fill [Color slot bg,button]
    }
  }

  incr id

  # `id' now is the id of the second text item
  case $event in {
  {enter} {
      $canvas itemconfig $id -fill [Color slot fg,active]
    }
  {leave} {
      $canvas itemconfig $id -fill [Color slot fg]
    }
  }

  return $self
}

defmethod Timeline _pointer {canvas event id} {

  case $event {
  {enter} {
      $canvas itemconfigure $id -fill [Color slot bg,active]
      return
    }

  {leave} {
      $canvas itemconfigure $id -fill [Color slot bg,button]
      return
    }

  {select} {
      incr id 2
      set tags [$canvas gettags $id]

      set pos [assoc pos $tags]
      if { $pos == {} } {
	return
      }

      set scrollincr [lindex [$canvas configure -scrollincrement] 4]
      set x [lindex [$canvas coords $id] 0]
      $canvas xview [format "%0.0f" [expr $x/$scrollincr]]
    }
  }
}

#---------------------------------------------------------------------------

defmethod Timeline Print {action} {

  global system

  if { [$self slot _split] } {
    $self.right.c print -printer $system(printer) -colormode gray
  } {
    $self.left.c print -printer $system(printer) -colormode gray
  }
  return
}

#---------------------------------------------------------------------------

Window addDemo Timeline

defmethod Timeline demo {} {

  set list { \
	{{Integrate Product} 0341 {5 24 5 7}} \
	{{Develop Test Manual} 9432 {20 49 0 10}} \
	{{Integrate Subsystem 1.2} 2393 {100 30 8 0}} \
	{{Mount Antennas} 04.x {50 55 0 0}} \
	{{Periodic Checks} 935 {110 18 4 4} {140 20 0 0} {200 30 10 10}} \
	{{Review} 6 {130 28 * 6}} \
	{{Integration Test} 927 {160 42 6 3 7}} \
	{{Assembly X Integration} 8001 {70 30 0 0}} \
	{{Assembly Y Integration} 8002 {70 25 0 0}} \
	{{Subassembly Z Integration} 8003 {70 39 0 0}} \
	{{Test Subsystem 1.1} 8000 {109 25 0 0}} \
	}

  set scale {
	{1 1 week wk}
        {4 1 month mon}
        {52 1 year yr}
	}

  Timeline new * \
	-layout +100+20 \
	-scaleinfo $scale \
	-text "Timeline for Project" \
	-initmethod list \
	-init $list \
	-actions {{List		{} action List} \
		  {Chart	{} action Chart}}

  set list { \
	{{Adapter Handling Transport} 10 \
		{4 10 10} {14 16 5} {30 10 2}} \
	{{ACDS Integration F} 2 \
		{0 10 2} {10 20 1} {20 100 2}} \
	{{Cleanroom Area} 200 \
		{0 12 20} {12 30 140} {50 100 30}} \
	{{Alignment Engineer} 5 \
		{0 10 2} {10 20 1} {20 100 2}} \
	{{Central Checkout System} 2 \
		} \
	{{Assembly Machine} 3 \
		} \
	{{Robot X.772} 10 \
		} \
	{{Conv. Belt} 1 \
		} \
	{{Electrical Tester} 10 \
		{0 10 2} {10 20 1} {20 100 2}} \
	{{Antenna Checkout System} 3 \
		} \
	}

  Timeline new * \
	-layout -100-20 \
	-text "Resource Utilization" \
	-initmethod list \
	-init $list \
	-type usage \
	-split true \
	-actions {{List		{} action List} \
		  {Chart	{} action Chart}}
}
