 #########################################################################
 #                                                                       #
 # Copyright (C) 1993 by General Electric company.  All rights reserved. #
 #                                                                       #
 # Permission to use, copy, modify, and distribute this                  #
 # software and its documentation for any purpose and without            #
 # fee is hereby granted, provided that the above copyright              #
 # notice appear in all copies and that both that copyright              #
 # notice and this permission notice appear in supporting                #
 # documentation, and that the name of General Electric not be used in   #
 # advertising or publicity pertaining to distribution of the            #
 # software without specific, written prior permission.                  #
 #                                                                       #
 # General Electric makes no representations about the suitability of    #
 # this software for any purpose.  It is provided ``as is''              #
 # without express or implied warranty.                                  #
 #                                                                       #
 # This work was supported in part by the DARPA Initiative in Concurrent #
 # Engineering (DICE) through DARPA Contracts MDA972-88-C-0047 and       #
 # MDA972-92-C-0027.                                                     #
 #                                                                       #
 # This work was supported in part by the Tri-Services Microwave and     #
 # Millimeter-Wave Advanced Computational Environment (MMACE) program    #
 # under Naval Research Laboratory contract N00014-92-C-2044.            #
 #                                                                       #
 #########################################################################


# File:	widget.tcl
#
# Description:
#	Tk library enhancements that are common to multiple sorts of
#	widgets.
#
#	This file includes procedures that allow for multiple bindings
#	on events for widgets, and for bindings on some new events.  The
#	binding scheme implemented here is used for the following
#	general types of events.
#
#c	Destroy
#		Execute an action when a widget is destroyed.  This
#		binding is dispatched through the `bind all <Destroy>'
#		in `init.tcl.'
#c	GainFocus
#		Report that a widget has gained focus from another widget
#		in the same application.  This is a more restrictive event
#		than <FocusIn>; for instance, having the window manager
#		direct focus to the application will not cause <GainFocus>
#		unless the application's focus changes as well.
#c	LoseFocus
#		Report that a widget has lost the focus to another widget in
#		the same application.  Like <GainFocus>, this event is more
#		restrictive than <FocusOut>.
#c	UpdateContent
#		Request that a widget that has a pending change post the
#		change.  This is used for entry widgets that allow the user
#		to edit their contents without posting to the text variable
#		until a change is committed.
#c	Validate
#		Request that a widget check that its content meets constraints,
#		for instance, that an entry's text value is numeric and within
#		its range.
#c	Unmap
#		Notification that a widget is no longer visible.
#
#	The file also implements support for composite widgets that are
#	treated as first-class widgets (by renaming the widget command so that
#	user code may intercept it).
#
# Global variables:
#c	widget:error$w
#		Action that the user has requested for an error message
#		resulting from a failure to validate the content of widget
#		$w.  THIS VARIABLE MAY NOT BE AN ARRAY ELEMENT.
#c	widget_priv(event,$event,$w)
#		Tcl command to execute in response to the event $event
#		occurring on the widget or widget class $w.
#c	widget_type($w)
#		Type of a composite widget $w.
#
#	The following global variables are instantiated once for each widget
#	type.
#
#c	$type_class
#		The widget class for widgets of type $type.
#
#c	$type_commands(subcommand)
#		The name of a Tcl procedure to invoke when handling a
#		subcommand `subcommand' on the widget command corresponding
#		to a widget of type $type.
#
#c	$type_configaction(-flag)
#		The name of a Tcl procedure that will be invoked when the
#		configuration option specified by command-line switch `-flag'
#		changes on a widget of type $type.
#
#c	$type_defaults(-flag)
#		A three-element list that describes command-line switch `-flag'
#		for widgets of type $type.  The elements are as follows.
#			+ The name of the option
#
#			+ The class of the option.
#
#			+ The option's default value.
#
#c	$type_initproc
#		The initialization procedure for widgets of type $type.
#
#	In addition, the following global array will be created once for
#	each instance of a composite widget.
#
#c	$type_config$w(-flag)
#		The current value of the configuration option specified
#		by the command line switch `-flag' on the widget $w, whose
#		type is $type.
#
# Transient procedures:
#
#	The following procedures are created for each type of compound widget.
#
# Procedure: $type
#
# Synopsis:
#	Create a widget of class `$type'
#
# Usage:
#c	$type pathName ?-option value?...
#
# Parameters:
#c	pathName
#		Path name of the widget to create
#
# Options:
#	Options are specified on the `widget_define' command that defined the
#	widget type.
#
# Return value:
#	Path name of the newly created widget.
#
# ------------------------------------------------------------------------
#
# Procedure:	$type:command$command
#
# Synopsis:
#	Process widget command $command on a widget of type $type.
#
# Usage:
#c	$type:command$command pathName ?args?
#
# Parameters:
#c	pathName
#		Path name of the widget
#c	args
#		Other parameters are user-specified by the `widget_subcommand'
#		procedure
#
# Return value:
#	Specified by the user in the `widget_subcommand' procedure
#
# ------------------------------------------------------------------------
#
# Procedure:	$type:config$flag
#
# Synopsis:
#	Process command-line flag $flag in configuring a widget of type $type.
#
# Usage:
#c	$type:config$flag pathName value
#
# Parameters:
#c	pathName
#		Path name of the widget being configured.
#c	value
#		Value of the configuration option being changed.
#
# Return value:
#	Not specified.
#
# ------------------------------------------------------------------------
#
#	In addition, for each composite widget, the Tk widget command is
#	renamed to be
#
#c		$type:$w
#
#	where $type is the type of the widget and $w is the widget path name.
#
#	A new transient procedure $w is created.

 # $Id: widget.tcl,v 1.14 1993/11/01 18:20:46 kennykb Exp $
 # $Source: /homedisk/julius/u0/kennykb/src/tkauxlib_ship/RCS/widget.tcl,v $
 # $Log: widget.tcl,v $
 # Revision 1.14  1993/11/01  18:20:46  kennykb
 # Beta release to be announced on comp.lang.tcl
 #
 # Revision 1.13  1993/10/27  15:52:49  kennykb
 # Package for alpha release to the Net, and for MMACE 931101 release.
 #
 # Revision 1.12  1993/10/26  21:43:14  kennykb
 # Added an Unmap action to the list of actions recognized by widget_bind
 # and widget_event, in order to allow focus management to defocus a
 # window if it s unmapped.
 #
 # Revision 1.11  1993/10/25  16:16:30  kennykb
 # Changed code to make sure that BOTH class bindings AND widget bindings
 # are applied by widget_event.  This is needed, for example, in focus
 # management, where an entry that owns focus is destroyed.  Both the
 # focus Destroy procedure (bound at widget level) and the entry Destroy
 # procedure (bound at Class level) need to get the chance to clean up.
 #
 # Revision 1.10  1993/10/20  19:10:47  kennykb
 # Alpha release #1 was thawed for bug fixes in tk 3.3.  Now frozen again at this
 # point.
 #
 # Revision 1.9  1993/10/20  19:06:24  kennykb
 # Repaired copyright notice so that it doesn't look like structured commentary.
 #
 # Revision 1.8  1993/10/14  18:15:42  kennykb
 # Cleaned up alignment of log messages, to avoid problems extracting
 # structured commentary.
 #
 # Revision 1.7  1993/10/14  18:06:59  kennykb
 # Added GE legal notice to head of file in preparation for release.
 #
 # Revision 1.6  1993/10/14  14:02:02  kennykb
 # Alpha release #1 frozen at this point.
 #
 # Revision 1.5  1993/10/14  13:54:59  kennykb
 # Added "widget_wiatVariable" and the associated procedures.  Changed
 # widget_propagateConfig and widget_propagateAction to be a trifle more
 # efficient.  THey're still FAR too slow.
 #
 # Revision 1.4  1993/07/21  19:44:36  kennykb
 # Finished cleaning up structured commentary.
 #
 # Revision 1.3  1993/07/20  19:17:12  kennykb
 # Improved structured comments.
 # Changed modules through `g' in the alphabet to follow `:' and `_' naming
 # conventions.
 #
 # Revision 1.2  1993/07/16  15:58:00  kennykb
 # Renamed all commands that start with `wiget.' to either `widget_' or
 # `widget:'.  Added the code for creating composite widgets.
 #
 # Revision 1.1  1993/06/03  15:34:10  kennykb
 # Initial revision
 #

# Procedure:	widget:check
#
# Synopsis:
#	Internal procedure to validate the content of a widget.
#
# Usage:
#c	widget:check pathName
#
# Parameters:
#c	pathName
#		Path name of a widget.
#
# Return value:
#	None.
#
# Description:
#	`widget:check' causes the Validate event to occur on a given single
#	widget.  (It uses widget_event (q.v.) to cause the event.)
#
#	If the validation succeeds, widget:check returns.  Otherwise, it
#	uses `widget:error' (q.v.) to display an error message, and resignals
#	the error.

proc widget:check w {
	set status [catch {widget_event $w Validate} message]
	if {$status != 0} {
		global errorInfo
		set info $errorInfo
		widget:error $w "[winlabel $w]: $message"
		error $message $info
	}
}

# Procedure:	widget:checkall
#
# Synopsis:
#	Internal procedure to validate the content of a widget tree.
#
# Usage:
#c	widget:checkall pathName
#
# Parameters:
#c	pathName
#		Path name of a widget.
#
# Return value:
#	None.
#
# Description:
#	`widget:checkall' is used to validate an entire portion of the
#	widget tree at once.  It runs `widget:check' on the specified widget
#	and on all of its subordinates.  It returns normally if all the
#	checks succeed, and otherwise throws an error (and displays a message)
#	detailing the check that failed.

proc widget:checkall w {
	widget:check $w
	foreach c [winfo children $w] {
		if {$c != [winfo toplevel $c]} {
			widget:checkall $c
		}
	}
}

# Procedure:	widget:error
#
# Synopsis:
#	Internal procedure to report a validation error on a widget.
#
# Usage:
#c	widget:error pathName message
#
# Parameters:
#c	pathName
#		Path name of a widget.
#c	message
#		Error message to display
#
# Return value:
#	None.
#
# Description:
#	widget:error makes a modal dialog box describing an error in
#	validating the contents of a widget, and waits for the dialog to be
#	dismissed by the user.

proc widget:error {w message} {
	global widget:error$w
	modalDialog transient choicebox $w.error \
		-text $message \
		-buttons OK \
		-icon rtfm \
		-textvariable widget:error$w
	widget_waitVariable widget:error$w
	unset widget:error$w
	modalDialog.end $w.error
}

# Procedure:	widget_bind
#
# Synopsis:
#	Establish an event handler on a widget.
#
# Usage:	widget_bind pathName event ?action?
#	-or-	widget_bind Class event ?action?
#
# Parameters:
#c	pathName
#		Path name of a widget
#c	Class
#		Widget class
#c	event
#		Event to catch.  May be one of:
#c			Destroy GainFocus
#c			LoseFocus UpdateContent
#c			Validate Unmap
#c	action
#		Tcl command to execute when the specified event occurs
#		on the specified widget.  If the binding is for a class, the
#		widget name will be appended to `action'.
#		If `action' is the null string, any existing binding is
#		removed.  If `action' begins with a plus sign, the specified
#		action is appended to the set of bindings for the widget.
#
# Return value:
#	The Tcl command that will be executed when the specified event occurs.
#
# Description:
#	`widget_bind' establishes a new binding for a given event on a
#	specified widget.  It is used to manage events that do not correspond
#	to X events, or that must be multiply dispatched.
#
#	The events, and their meanings, are as follows.
#
#c	Destroy
#		Widget has been destroyed.
#c	GainFocus
#		Widget has gained the keyboard focus by
#		means of `focus_goTo.'  This is a more
#		restrictive event than <FocusIn>, which
#		seems to happen almost at random.
#	LoseFocus
#		Widget has lost the keyboard focus by
#		means of `focus_goTo.'  This is a more
#		restrictive event than <FocusOut>, which
#		happens for various reasons outside the
#		application's control.
#c	UpdateContent
#		Widget has been requested to update its
#		content because a command is to be executed.
#c	Validate
#		Widget has been requested to check that its
#		content meets constraints.
#c	Unmap
#		Widget has been unmapped from the screen.

proc widget_bind {w event {string ?}} {
	global widget_priv
	case $event in {
		{ Destroy GainFocus 
		  LoseFocus UpdateContent 
		  Validate Unmap } {
		}
		default {
			error "widget_bind: unknown event $event"
		}
	}
	if {$string == "?"} {
		if [info exists widget_priv(event,$event,$w)] {
			set string $widget_priv(event,$event,$w)
		} else {
			set string ""
		}
	} elseif {$string == ""} {
		catch {unset widget_priv(event,$event,$w)}
	} else {
		if {[string index $string 0] == "+"} {
			set string [string range $string 1 end]
			if [info exists widget_priv(event,$event,$w)] {
				append string \n
				append string $widget_priv(event,$event,$w)
			}
		}
		set widget_priv(event,$event,$w) $string
	}
	return $string
}

# Procedure:	widget_addBinding
#
# Synopsis:
#	Add a binding to the list of bindings for a given widget and event.
#
# Usage:
#c	widget_addBinding pathName event action
#
# Parameters:
#c	pathName
#		Path name of a widget
#c	event
#		Event for which to watch (see widget_bind)
#c	action
#		Tcl command to execute when the specified event occurs.
#
# Return value:
#	Complete set of bindings for the specified widget and event.
#
# Description:
#	widget_addBinding adds an action to the list of actions for a
#	specified event on a given widget.  It differs from calling widget_bind
#	with an event beginning with a plus sign in that if the specified
#	action is already on the list of actions to perform, it will not be
#	added a second time.

proc widget_addBinding {w event string} {
	global widget_priv
	if [info exists widget_priv(event,$event,$w)] {
		set curBinding $widget_priv(event,$event,$w)
	} else {
		set curBinding ""
	}
	if {[string first $curBinding $string] >= 0} {
		return $curBinding
	} else {
		return [widget_bind $w $event +$string]
	}
}

# Procedure:	widget_event
#
# Synopsis:
#	Cause an event on a widget.
#
# Usage:
#c	widget_event pathName event
#
# Parameters:
#c	pathName
#		Path name of a widget
#c	event
#		Event that has occurred (see widget_bind)
#
# Return value:
#	None.
#
# Description:
#	widget_event is used when one of the events monitored by widget_bind
#	occurs.  It executes the actions bound to the event for the widget,
#	and for its class.

proc widget_event {w event} {
	global widget_priv
	set action {}
	if [info exists widget_priv(event,$event,$w)] {
		set action $widget_priv(event,$event,$w)\n
	}
	if {[winfo exists $w] 
	    && [info exists widget_priv(event,$event,[winfo class $w])]} {
		append action "$widget_priv(event,$event,[winfo class $w]) $w\n"
	}
	if {$action != ""} {
		return [uplevel #0 $action]
	} else {
		return 0
	}
}

# Procedure:	widget:destroy
#
# Synopsis:
#	Internal procedure executed in response to all <Destroy> events
#	to clean up after widget destruction.
#
# Usage:
#c	widget:destroy pathName
#
# Parameters:
#c	pathName
#		Path name of a widget.
#
# Return value:
#	None.
#
# Description:
#	`widget:destroy' is called when a <Destroy> X event is received for
#	a given widget.  It executes all the `Destroy' bindings that have
#	been established using `widget_bind' for the given widget (and its
#	class).
#
#	If the widget is a composite one (defined using widget_define),
#	the global variables that track its type and configuration are
#	then unset, and the trampoline procedure that processes its
#	widget commands is deleted.
#
#	Finally all `widget_priv' entries belonging to the given widget
#	are deleted.

proc widget:destroy w {
	global widget_priv
	global widget_type
	widget_event $w Destroy
	if [info exists widget_type($w)] {
		set type $widget_type($w)
		unset widget_type($w)
		upvar #0 ${type}_config$w config
		unset config
		if {[info commands ${type}:$w] == "${type}:$w"} {
			rename ${type}:$w {}
		}
		if {[info commands $w] == $w} {
			rename $w {}
		}
	}
	foreach item [array names widget_priv] {
		if [string match *,$w $item] {
			unset widget_priv($item)
		}
	}
}

# Procedure:	widget:unmap
#
# Synopsis:
#	Internal procedure executed in response to all <Unmap> events
#	to clean up after widget disappearance.
#
# Usage:
#c	widget:unmap pathName
#
# Parameters:
#c	pathName
#		Path name of a widget.
#
# Return value:
#	None.
#
# Description:
#	`widget:unmap' is called when an <Unmap> X event is received for
#	a given widget.  It executes all the `Unmap' bindings that have
#	been established using `widget_bind' for the given widget (and its
#	class).

proc widget:unmap w {
	global widget_priv
	widget_event $w Unmap
}

# Procedure:	widget_waitVariable
#
# Synopsis:
#	Wait for a variable
#
# Usage:
#c	widget_waitVariable name
#
# Parameters:
#	name
#		Name of a variable
#
# Return value:
#	None specified.
#
# Description:
#	The `widget_waitVariable' is identical to the Tk command, `tkwait
#	variable', except that it records the name of the variable in an
#	array named `widget_waitVariables'.  The rationale is that a `tkwait
#	variable' hangs the process if the application is destroyed while
#	the `tkwait' is pending.

proc widget_waitVariable {vname} {
	global widget_waitVariables
	global widget_appDestroyed
	if {![info exists widget_waitVariables($vname)]} {
		set widget_waitVariables($vname) 1
	} else {
		incr widget_waitVariables($vname)
	}
	uplevel 1 tkwait variable [list $vname]
	set content [uplevel #0 set $vname]
	if {[incr widget_waitVariables($vname) -1] == 0} {
		unset widget_waitVariables($vname)
	}
	if {[info exists widget_appDestroyed]} {
		error \
"widget_waitVariable $vname terminated prematurely:
   application has been destroyed."
	}
}

# Procedure:	widget:destroyApp
#
# Synopsis:
#	Throw errors at widget_waitVariable when an application is destroyed.
#
# Usage:
#c	widget:destroyApp
#
# Parameters:
#	None
#
# Return value:
#	None specified.
#
# Description:
#	widget:destroyApp is called when the application root window is
#	destroyed.  It sets the values of all variables active in
#	widget_waitVariable, so that all the waits will terminate.

proc widget:destroyApp {} {
	global widget_waitVariables
	global widget_appDestroyed
	set appDestroyed 1
	if [catch {array names widget_waitVariables} names] return
	foreach vname $names {
		widget:destroyApp2 $vname
	}
}
proc widget:destroyApp2 {vname} {
	upvar #0 $vname var
	set var "@APP@DESTROYED@"
}

# Procedure:	widget_checkAndDestroy
#
# Synopsis:
#	Destroy a widget, if the widget exists.
#
# Usage:
#c	widget_checkAndDestroy pathName
#
# Parameters:
#c	pathName
#		Path name of a widget to destroy.
#
# Return value:
#	None.
#
# Description:
#	widget_checkAndDestroy checks if a specified widget exists.  If it
#	exists, it is destroyed, otherwise, nothing happens.
#
# Notes:
#	The commonest use of `widget_checkAndDestroy' is so that destroying a
# 	widget may destroy its parent.  For example, destroying a widget
#	that has been packed into a `transient' frame should destroy the
#	top-level window as well.
#
# Example:
#c	widget_addBinding $w.child Destroy \
#
#c		"widget_checkAndDestroy $w"

proc widget_checkAndDestroy w {
	if [catch {winfo exists $w} exists] return
	if {$exists && ($w == "." || [info commands $w] == $w)} {
		destroy $w
	}
}

# Procedure:	widget_define
#
# Synopsis:
#	Define a new type of composite widget.
#
# Usage:
#c	widget_define type class {config...} initProc
#
# Parameters:
#c	type
#		The type of widget being defined, i.e., the name of the
#		command that will create a widget of that type.
#c	class
#		The resource class corresponding to the type of widget.
#c	config
#		A list of four-element lists giving the widget's configuration
#		parameters.  Each of the lists has, in order:
#
#			+ The command-line flag for the option.
#
#			+ The name (X resource) of the option.
#
#			+ The class (X resource) of the option.
#
#			+ The default value of the option.
#c	initProc
#		The name of a procedure that constructs the widget.  The
#		procedure is passed a single argument -- the name a frame
#		widget in which the composite will be constructed.  It is
#		expected to return the same name.
#
# Return value:
#	None.
#
# Description:
#	widget_define defines a procedure, which in turn is called to 
#	create widgets.  It is the fundamental procedure that is used to
#	make a composite widget.

proc widget_define {type class config initproc} {
	global ${type}_defaults
	global ${type}_class
	global ${type}_initproc
	global ${type}_commands
	foreach item $config {
		set flag [lindex $item 0]
		set ${type}_defaults($flag) [lrange $item 1 3]
	}
	set ${type}_class $class
	set ${type}_initproc $initproc
	set ${type}_commands(configure) widget:configure
	proc $type {w args} \
		"widget:create \$w $type \$args"
}

# Procedure:	widget_subcommand
#
# Synopsis:
#	Define a subcommand for a composite widget's widget command.
#
# Usage:
#c	widget_subcommand type command paramList body
#
# Parameters:
#c	type
#		The type of widget for which the subcommand is being defined.
#c	command
#		The name of the widget command being defined.
#c	paramList
#		The parameters that the widget command expects.  The
#		first parameter is always the widget path name; the remaining
#		parameters are taken from the call to the widget command.
#c	body
#		The body of the procedure that executes the command.
#
# Return value:
#	The name of the procedure that executes the command.  It will have
#	the form
#
#c		type:commandXXX,
#
#	where XXX is the name of the command.
#
# Description:
#	widget_subcommand defines a new widget command for a widget.
#
# Example:
#c	widget_subcommand myWidget show {w string} {
#
#c		$w.label config -text $string
#
#c	}
#
#c	myWidget .foo
#
#c	.foo show "Hi there"

proc widget_subcommand {type command params body} {
	global ${type}_commands
	proc $type:command$command $params $body
	set ${type}_commands($command) $type:command$command
}

# Procedure:	widget_configFlag
#
# Synopsis:
#	Define a configuration option for a composite widget.
#
# Usage:
#c	widget_configFlag type flagName paramList body
#
# Parameters:
#c	type
#		Type of the widget for which the option is being defined.
#c	flagName
#		Name of the option being defined.
#c	paramList
#		Parameters to the function that handles the option.  The
#		function is expected to accept two parameters -- the name
#		of the widget, and the value of the configuration option.
#
# Return value:
#	The name of the function that handles the option.  The function will
#	always have the name
#
#c		type:config-XXX
#
#	where -XXX is the configuration flag being processed.
#
# Description:
#	widget_configFlag defines the function that handles a single
#	configuration option for a widget.  It is responsible for taking
#	any necessary action when the configuration changes.
#
# Example:
#c	widget_configFlag myWidget -background {w color} {
#
#c		widget_propagateConfig $w -background $color
#
#c	}

proc widget_configFlag {type flag params body} {
	global ${type}_configaction
	proc ${type}:config$flag $params $body
	set ${type}_configaction($flag) ${type}:config$flag
}

# Procedure:	widget:create
#
# Synopsis:
#	Internal procedure to create a composite widget defined by
#	widget_define.
#
# Usage:
#c	widget:create pathName type params
#
# Parameters:
#c	pathName
#		Path name of the widget being created.
#c	type
#		Type of the widget being created.
#c	params
#		Configuration options supplied at widget creation time.
#
# Return value:
#	Name of the widget created.
#
# Description:
#	widget:create does the dirty work of a widget command defined by
#	widget_define.  It accepts the wiget, the widget's type, and the
#	configuration parameters.  It creates the frame in which the widget
#	will be constructed, parses configuration parameters (and the
#	corresponding options from the option database), calls the widget
#	initialization procedure, renames the frame command, and puts in its
#	place the composite widget command.  Finally, it calls
#	widget:evalConfig to make the configuration options take effect.

proc widget:create {w type params} {

	upvar #0 ${type}_class class
	global widget_type
	upvar #0 ${type}_initproc initproc

	set widget_type($w) $type

	frame $w -class $class

	widget:getConfig $w $params

	$initproc $w

	rename $w ${type}:$w

	proc $w {command args} "\
		eval \[list widget:command $w \$command\] \$args
	"

	update idletasks

	widget:evalConfig $w

	return $w
}

# Procedure:	widget:command
#
# Synopsis:
#	Internal procedure to evaluate a widget command on a composite widget.
#
# Usage:
#c	widget:command pathName command args
#
# Parameters:
#c	pathName
#		Path name of a composite widget.
#c	command
#		Name of the command to execute.
#c	args
#		Parameters to the command
#
# Return value:
#	Specified by the user.
#
# Description:
#	The widget:command procedure executes a widget command on a
#	composite widget.  It includes unique-prefix matching for
#	subcommand names.

proc widget:command {w command args} {
	global widget_type
	set type $widget_type($w)
	upvar #0 ${type}_commands commands
	set status [catch {widget:matchName $command commands} result]
	if {$status != 0} {
		error "\
$w: invalid command $command" "\
$w: invalid command $command; available commands are:
	[array names commands]"
	}
	eval [list $commands($result) $w] $args

}

# Procedure:	widget:configure
#
# Synopsis:
#	Internal procedure to change or query the configuration of a
#	compound widget.
#
# Usage:
#c	widget:configure pathName params
#
# Parameters:
#c	pathName
#		Path name of a widget.
#c	params
#		If zero-length, query entire configuration.
#
#		If a single flag, query that flag.
#
#		If one or more flag-value pairs, set the corresponding
#		configuration options.
#
# Return value:
#	Result of a query.
#
# Description:
#	widget:configure handles the internals of the `configure' widget
#	command.  It includes unique-prefix matching for option names.

proc widget:configure {w args} {

	global widget_type
	set type $widget_type($w)
	upvar #0 ${type}_defaults defaults
	upvar #0 ${type}_config$w config

	case [llength $args] in {
		0 {
			set r ""
			set id [array startsearch config]
			while {[array anymore config $id]} {
				set name [array nextelement config $id]
				set l $name
				set l [concat $l $defaults($name)]
				lappend l $config($name)
				lappend r $l
			}
			array donesearch config $id
			return $r
		}
		1 {
			set flag [lindex $args 0]
			set status [catch {widget:matchName $flag config} name]
			if {$status == 0} {
				set l $name
				set l [concat $l $defaults($name)]
				lappend l $config($name)
				return $l
			} else {
				error \
"$w: invalid command-line flag $flag" \
"$w: invalid command-line flag $flag;
	available flags are: [lsort [array names config]]"
			}
		}
		default {
			widget:getConfig $w $args
			widget:evalConfig $w
		}
	}
}

# Procedure:	widget:getConfig
#
# Synopsis:
#	Internal procedure to get the configuration of a widget.
#
# Usage:
#c	widget:getConfig pathName args
#
# Parameters:
#c	pathName
#		Path name of the widget being configured.
#c	args
#		Parameters to the `config' widget command, or to the
#		widget creation command.
#
# Return value:
#	Not specified.
#
# Description:
#	widget:getConfig parses the configuration parameters specified
#	on a command line.  If a widget is newly created, it also
#	searches out the options from the option data base.  The
#	global array type_configXXX, where `type' is the widget type and
#	XXX is the widget path name, is filled with the values of configuration
#	flags.

proc widget:getConfig {w params} {

	global widget_type
	set type $widget_type($w)

	upvar #0 ${type}_defaults defaults

	upvar #0 ${type}_config$w config

	if {![info exists config]} {
		set searchid [array startsearch defaults]
		while {[array anymore defaults $searchid]} {
			set flag [array nextelement defaults $searchid]
			set rname [lindex $defaults($flag) 0]
			set rclass [lindex $defaults($flag) 1]
			set value [option get $w $rname $rclass]
			if {$value == ""} {
				set value [lindex $defaults($flag) 2]
			}
			set config($flag) $value
		}
		array donesearch defaults $searchid
	}

	while {$params != ""} {
		set flag [lindex $params 0]
		set value [lindex $params 1]
		set params [lrange $params 2 end]
		set status [catch {widget:matchName $flag config} result]
		if {$status != 0} {
			global errorInfo
			error \
"$type $w: invalid command-line flag $flag" \
"$type $w: invalid command-line flag $flag;
	available flags are: [lsort [array names config]]"
		}
		set config($result) $value
	}
}

# Procedure:	widget:evalConfig
#
# Synopsis:
#	Internal procedure to make configuration changes to a widget
#	take effect.
#
# Usage:
#c	widget:evalConfig pathName
#
# Parameters:
#c	pathName
#		Path name of a widget.
#
# Return value:
#	Not specified.
#
# Description:
#	widget:evalConfig calls all the configuration procedures for a
#	composite widget.  It is invoked whenever the configuration changes,
#	in order to make the changes to the widget's look and feel.

proc widget:evalConfig w {
	global widget_type
	set type $widget_type($w)
	upvar #0 ${type}_config$w config
	upvar #0 ${type}_configaction action
	set id [array startsearch config]
	while {[array anymore config $id]} {
		set item [array nextelement config $id]
		if [info exists action($item)] {
			$action($item) $w $config($item)
		} else {
			widget_propagateConfig $w $item $config($item)
		}
	}
	array donesearch config $id
}

# Procedure:	widget:matchName
#
# Synopsis:
#c	Name matching for widget commands
#
# Usage:
#c	widget:matchName name array
#
# Parameters:
#c	name
#		Name of a subcommand, configuration option, etc.
#c	array
#		Array whose indices are the names of available subcommands,
#		configuration options, etc.
#
# Return value:
#	Name extracted from the available set.
#
# Description:
#	widget:matchName accepts a name, and an array that contains the
#	possible values for the name.  It attempts to match the name to
#	one of the indices in the array.  If the name does not mach exactly,
#	but only one element of the array has the name as a prefix, that
#	array element is returned; this allows for the same unique-prefix
#	rules that Tk uses to match widget commands and command-line flags.

proc widget:matchName {name arrayname} {
	upvar 1 $arrayname array
	if [info exists array($name)] {
		return $name
	}
	set names [array names array]
	set index [lsearch $names $name*]
	if {$index >= 0} {
		set index2 [lsearch [lrange $names [expr $index+1] end] $name*]
		if {$index2 < 0} {
			return [lindex $names $index]
		}
	}
	error "$name not found in $arrayname"
}

# Procedure:	widget_propagateConfig
#
# Synopsis:
#	Propagate a configuration among a widget and all its 
#	children.
#
# Usage:
#c	widget_propagateConfig pathName -flag value
#
# Parameters:
#c	pathName
#		Path name of a widget that may have children.
#c	-flag
#		Configuration option to apply to the widget and all its
#		descendants.
#c	value
#		Value of the option.
#
# Return value:
#	Not specified.
#
# Description:
#	widget_propagateConfig propagates a configuration option to all 
#	the descendants of a widget.  It is used to handle inherited
#	attributes; for instance, the background of a child widget is often
#	inherited from its parent's background.
#
# Example:
#c	widget_propagateConfig .a.b -bg #ffe4c4
#
#		Sets the widget and all of its descendants to have a
#		bisque-colored background.

proc widget_propagateConfig {w flag value} {
	global widget_type
	catch {widget_passCommand $w configure $flag $value}
	foreach c [winfo children $w] {
		if {![info exists widget_type($c)]} {
			widget_propagateConfig $c $flag $value
		}
	}
}

proc widget_propagateAction {w proc args} {
	eval [list $proc $w] $args
	foreach c [winfo children $w] {
		eval [list widget_propagateAction $c $proc] $args
	}
}

# Procedure:	widget_passCommand
#
# Synopsis:
#	Pass a widget command to Tk
#
# Usage:
#c	widget_passCommand pathName command args
#
# Parameters:
#c	pathName
#		Path name of a widget
#c	command
#		Widget command to execute
#c	args
#		Arguments to the command
#
# Return value:
#	Return value of the widget command.
#
# Description:
#	widget_passCommand is used to pass a widget command to Tk, rather than
#	to the interpreter controlling a composite widget.  It accepts the
#	widget command to execute, and returns its result.

proc widget_passCommand {w args} {
	global widget_type
	if [info exists widget_type($w)] {
		return [eval [list $widget_type($w):$w] $args]
	} else {
		return [eval [list $w] $args]
	}
}

	# sample procedures

if [info exists env(WIDGET_TCL_DEMO)] {
 widget_define foo Foo {
	{-background background Background gray95}
	{-borderwidth borderWidth BorderWidth 0}
	{-cursor cursor Cursor {}}
	{-foreground foreground Foreground black}
	{-geometry geometry Geometry {}}
	{-relief relief Relief flat}
 } foo_init

 proc foo_init w {
	pack append $w \
		[frame $w.f1 -geom 1x6 -relief sunken -borderwidth 2] \
			{pady 10 fillx} \
		[label $w.l -text "This is a foo."] \
			{top expand fill} \
		[frame $w.f2 -geom 1x6 -relief sunken -borderwidth 2] \
			{pady 10 fillx}
	return $w
 }

 widget_subcommand foo change {w s} {
	$w.l configure -text "This is a changed foo."
 }

 widget_subcommand foo change_back {w s} {
	$w.l configure -text "This is a foo."
 }

 widget_configFlag foo -background {w bg} {
	widget_propagateConfig $w -background $bg
 }

 widget_configFlag foo -borderwidth {w b} {
	widget_passCommand $w config -borderwidth $b
 }

 widget_configFlag foo cursor {w c} {
	widget_propagateConfig $w -cursor $c
 }

 widget_configFlag foo -foreground {w fg} {
	widget_propagateConfig $w -foreground $fg
 }

 widget_configFlag foo -geometry {w g} {
	widget_passCommand $w config -geometry $g
 }

 widget_configFlag foo -relief {w r} {
	widget_passCommand $w config -relief $r
 }
}
