#ifndef _TkTree_h_
#define _TkTree_h_

/*
 * TkTree.h - class definitions for drawing trees in Tk/Tcl
 * 
 * -----------------------------------------------------------------------------
 * Copyright 1993 Allan Brighton.
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies.  Allan
 * Brighton make no representations about the suitability of this software
 * for any purpose.  It is provided "as is" without express or implied
 * warranty.
 * -----------------------------------------------------------------------------
 */

#include <OS/string.h>

extern  "C" {
#include <tk.h>
}

class TkTreeNode;


/*
 * Class TkTree
 * 
 * This class implements the extended Tk command "tree" for displaying
 * dynamic trees in a canvas.
 */
class TkTree {
private:
    Tk_Window tkwin_;			// widget's Tk window
    Tcl_Interp* interp_;		// current tcl interpreter
    int status_;			// non-zero if OK
    CopyString canvas_;			// parent of tree should be a canvas
    TkTreeNode* root_;			// root of the layout tree
    
    // tree options
    struct TkTreeOptions {
        int parentDistance;		// distance between nodes and parent
        char* layout;			// tree's layout: "horizontal" or "vertical"
        int borderWidth;		// width of border around entire tree
	TkTreeOptions(int p, char* l, int b)
	    :parentDistance(p), layout(l), borderWidth(b) {}
    } options_;
    
    // array of config options for tree widget
    static Tk_ConfigSpec configSpecs_[4];
     
    // tree widget command (generated from tree name)
    static int treeWidgetCmd(ClientData, Tcl_Interp*, int argc, char* argv[]);
    
    // clean up when cmd is destroyed
    static void treeDeleteCmd(ClientData clientData);
    
    // fix the invisible root node of the tree at center left or top
    int fixRootNode();
    
    // find the named node and return a pointer to it or 0 if not found
    TkTreeNode* findNode(const String& tag);
    
    // configure the tree widget with cmd line options
    int configureTree(int argc, char* argv[], int flags = 0);
    
    // used for configure widget command
    int configure(int argc, char* argv[]);
    
    // reconfigure the node (size, options, ...)
    int setupNode(TkTreeNode*, int argc, char* argv[]);
    
    // add a new child node to the parent node in the tree
    int addLink(int argc, char* argv[]);
    
    // move the subtree to a new position in the tree 
    int moveLink(int argc, char* argv[]);
    
    // remove a node and its subtree 
    int rmLink(int argc, char* argv[]);
    
    // recalculate the size and pos and set options
    int nodeConfigure(int argc, char* argv[]);
    
    // command to remove and delete the nodes children
    int prune(int argc, char* argv[]);
    
    // command that returns 1 if the named node is a leaf node (has no children)
    int isLeaf(int argc, char* argv[]);
    
    // command that returns 1 if the named node is a root node (displays no parents)
    int isRoot(int argc, char* argv[]);
    
    // command that sets a new root node
    int root(int argc, char* argv[]);
    
    // command to draw the tree on the canvas
    int draw(int argc, char* argv[]);
    
    // command to return the bounding box of the tree
    int bbox(int argc, char* argv[]);
    
    // command that returns the name of the child node
    int child(int argc, char* argv[]);
    
    // command that returns a list of the child nodes
    int subnodes(int argc, char* argv[]);
    
    // command that returns the name of the sibling node
    int sibling(int argc, char* argv[]);
    
    // command that returns the name of the parent node
    int parent(int argc, char* argv[]);

    // command that returns a list of ancestors of a tree node
    int ancestors(int argc, char* argv[]);
        
    // copy constructor: not defined
    TkTree(const TkTree&);
    
public:
    // initialize the tree with the command line args
    TkTree(Tk_Window, Tcl_Interp*, int argc, char* argv[]);
    
    // destructor - clean up tree nodes when widget cmd is destroyed
    ~TkTree();
    
    // read-only member access
    int status() const { return status_; }
    const String& canvas() const { return canvas_; }
    Tcl_Interp* interp() const { return interp_; }
    int parentDistance() const { return options_.parentDistance; }
    int borderWidth() const { return options_.borderWidth; }
    
    // boolean functions to determine the layout quickly
    int horizontal() const { return (*options_.layout != 'v'); }
    int vertical() const { return (*options_.layout == 'v'); }
};


#endif _TkTree_h_
