static char rcsid[]="$Id: tkCanvRrectangle.c,v 1.1 94/02/09 10:13:57 mangin Exp $";
/* Modified from tkCanvLine.c - Frank Mangin - Fri Jan 21 1994
 *   to implement rotated filled and outlined rectangles
 */

/* 
 * tkCanvRrectangle.c --
 *
 *	This file implements rrectangle items for canvas widgets.
 *
 * Copyright (c) 1991-1993 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE UNIVERSITY OF
 * CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND THE UNIVERSITY OF CALIFORNIA HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

#include <stdio.h>
#include <tkInt.h>
#include "tkCanvas.h"
#include <tkConfig.h>
#include <X11/Xutil.h>
#include <math.h>

#define ZEROF 0.000001
#define FNULL(A) (((A) > -ZEROF) && ((A) < ZEROF))
#define FEQUAL(A,B) FNULL((A)-(B))

/*
 * The structure below defines the record for each rrectangle item.
 */

typedef struct RrectangleItem  {
    Tk_Item header;		/* Generic stuff that's the same for all
				 * types.  MUST BE FIRST IN STRUCTURE. */
    double xc, yc;		/* Coordinates of rectangle center */
    double width, height;	/* dims of rectangle */
    double angle;		/* angle of the width side of the rect (degrees) */
    double sine, cosine;	/* cos and sin of angle */
    int outlineWidth;		/* Width of outline */
    XColor *outlineColor;	/* Color for outline. */
    XColor *fillColor;		/* Color for filling drectangle/doval. */
    Pixmap fillStipple;		/* Stipple bitmap for filling item. */
    GC outlineGC;		/* Graphics context for outline. */
    GC fillGC;			/* Graphics context for filling item. */
    int dashNumber;		/* Number of elements in dashList */
    char *dashList;		/* Specifies dash style. Malloc'd, a value of */
      				/* NULL means no dashes */
} RrectangleItem;

/*
 * Prototypes for procedures defined in this file:
 */

static void		ComputeRrectangleBbox _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    RrectangleItem *rrectanglePtr));
static int		ConfigureRrectangle _ANSI_ARGS_((
			    Tk_Canvas *canvasPtr, Tk_Item *itemPtr, int argc,
			    char **argv, int flags));
static int		CreateRrectangle _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    struct Tk_Item *itemPtr, int argc, char **argv));
static void		DeleteRrectangle _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr));
static void		DisplayRrectangle _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr, Drawable dst));
static int		RrectangleCoords _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr, int argc, char **argv));
static int		RrectangleToArea _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr, double *rectPtr));
static double		RrectangleToPoint _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr, double *coordPtr));
static int		RrectangleToPostscript _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr, Tk_PostscriptInfo *psInfoPtr));
static void		ScaleRrectangle _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr, double originX, double originY,
			    double scaleX, double scaleY));
static void		TranslateRrectangle _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr, double deltaX, double deltaY));
static int              ParseDashes _ANSI_ARGS_((ClientData clientData,
                            Tcl_Interp *interp, Tk_Window tkwin, char *value,
                            char *recordPtr, int offset));
static char *           PrintDashes _ANSI_ARGS_((ClientData clientData,
                            Tk_Window tkwin, char *recordPtr, int offset,
                            Tcl_FreeProc **freeProcPtr));

/*
 * Information used for parsing configuration specs.  If you change any
 * of the default strings, be sure to change the corresponding default
 * values in CreateRrectangle.
 */

static Tk_CustomOption dashesOption = {ParseDashes, PrintDashes, (ClientData) NULL};
     
static Tk_ConfigSpec frectConfigSpecs[] = {
    {TK_CONFIG_DOUBLE, "-width", (char *) NULL, (char *) NULL,
       (char *) NULL, Tk_Offset(RrectangleItem, width), TK_CONFIG_NULL_OK},
    {TK_CONFIG_DOUBLE, "-height", (char *) NULL, (char *) NULL,
       (char *) NULL, Tk_Offset(RrectangleItem, height), TK_CONFIG_NULL_OK},
    {TK_CONFIG_DOUBLE, "-angle", (char *) NULL, (char *) NULL,
       (char *) NULL, Tk_Offset(RrectangleItem, angle), TK_CONFIG_NULL_OK},
    {TK_CONFIG_COLOR, "-fill", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(RrectangleItem, fillColor), TK_CONFIG_NULL_OK},
    {TK_CONFIG_BITMAP, "-stipple", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(RrectangleItem, fillStipple), TK_CONFIG_NULL_OK},
    {TK_CONFIG_CUSTOM, "-tags", (char *) NULL, (char *) NULL,
	(char *) NULL, 0, TK_CONFIG_NULL_OK, &tkCanvasTagsOption},
    {TK_CONFIG_END, (char *) NULL, (char *) NULL, (char *) NULL,
       (char *) NULL, 0, 0}
};

static Tk_ConfigSpec orectConfigSpecs[] = {
    {TK_CONFIG_DOUBLE, "-width", (char *) NULL, (char *) NULL,
       (char *) NULL, Tk_Offset(RrectangleItem, width), TK_CONFIG_NULL_OK},
    {TK_CONFIG_DOUBLE, "-height", (char *) NULL, (char *) NULL,
       (char *) NULL, Tk_Offset(RrectangleItem, height), TK_CONFIG_NULL_OK},
    {TK_CONFIG_DOUBLE, "-angle", (char *) NULL, (char *) NULL,
       (char *) NULL, Tk_Offset(RrectangleItem, angle), TK_CONFIG_NULL_OK},
    {TK_CONFIG_COLOR, "-outline", (char *) NULL, (char *) NULL,
	(char *)NULL, Tk_Offset(RrectangleItem, outlineColor), TK_CONFIG_NULL_OK},
    {TK_CONFIG_CUSTOM, "-tags", (char *) NULL, (char *) NULL,
	(char *) NULL, 0, TK_CONFIG_NULL_OK, &tkCanvasTagsOption},
    {TK_CONFIG_PIXELS, "-outlinewidth", (char *) NULL, (char *) NULL,
	"1", Tk_Offset(RrectangleItem, outlineWidth), TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_CUSTOM, "-dashes", (char *) NULL, (char *) NULL,
       "", Tk_Offset(RrectangleItem, dashList),
       TK_CONFIG_DONT_SET_DEFAULT, &dashesOption},
    {TK_CONFIG_END, (char *) NULL, (char *) NULL, (char *) NULL,
       (char *) NULL, 0, 0}
};

/*
 * The structures below defines the frectangle and orectabgle items type by means
 * of procedures that can be invoked by generic item code.
 */

#define IsFilled(P)   ((P)->header.typePtr == &TkFrectangleType)
#define IsOutlined(P) ((P)->header.typePtr == &TkOrectangleType)

Tk_ItemType TkFrectangleType = {
    "frectangle",				/* name */
    sizeof(RrectangleItem),			/* itemSize */
    CreateRrectangle,				/* createProc */
    frectConfigSpecs,			/* configSpecs */
    ConfigureRrectangle,			/* configureProc */
    RrectangleCoords,				/* coordProc */
    DeleteRrectangle,				/* deleteProc */
    DisplayRrectangle,			/* displayProc */
    0,					/* alwaysRedraw */
    RrectangleToPoint,			/* pointProc */
    RrectangleToArea,				/* areaProc */
    RrectangleToPostscript,			/* postscriptProc */
    ScaleRrectangle,				/* scaleProc */
    TranslateRrectangle,			/* translateProc */
    (Tk_ItemIndexProc *) NULL,		/* indexProc */
    (Tk_ItemCursorProc *) NULL,		/* icursorProc */
    (Tk_ItemSelectionProc *) NULL,	/* selectionProc */
    (Tk_ItemInsertProc *) NULL,		/* insertProc */
    (Tk_ItemDCharsProc *) NULL,		/* dTextProc */
    (Tk_ItemType *) NULL		/* nextPtr */
};

Tk_ItemType TkOrectangleType = {
    "orectangle",				/* name */
    sizeof(RrectangleItem),			/* itemSize */
    CreateRrectangle,				/* createProc */
    orectConfigSpecs,			/* configSpecs */
    ConfigureRrectangle,			/* configureProc */
    RrectangleCoords,				/* coordProc */
    DeleteRrectangle,				/* deleteProc */
    DisplayRrectangle,			/* displayProc */
    0,					/* alwaysRedraw */
    RrectangleToPoint,			/* pointProc */
    RrectangleToArea,				/* areaProc */
    RrectangleToPostscript,			/* postscriptProc */
    ScaleRrectangle,				/* scaleProc */
    TranslateRrectangle,			/* translateProc */
    (Tk_ItemIndexProc *) NULL,		/* indexProc */
    (Tk_ItemCursorProc *) NULL,		/* icursorProc */
    (Tk_ItemSelectionProc *) NULL,	/* selectionProc */
    (Tk_ItemInsertProc *) NULL,		/* insertProc */
    (Tk_ItemDCharsProc *) NULL,		/* dTextProc */
    (Tk_ItemType *) NULL		/* nextPtr */
};


/***************************************************************
 ****          Canvas rrectangle items creation              ****
 ***************************************************************/

void RrectangleItemTypeCreate()
{
  Tk_CreateItemType(&TkFrectangleType);
  Tk_CreateItemType(&TkOrectangleType);
}

/*
 *--------------------------------------------------------------
 *
 * CreateRrectangle --
 *
 *	This procedure is invoked to create a new rrectangle item in
 *	a canvas.
 *
 * Results:
 *	A standard Tcl return value.  If an error occurred in
 *	creating the item, then an error message is left in
 *	canvasPtr->interp->result;  in this case itemPtr is
 *	left uninitialized, so it can be safely freed by the
 *	caller.
 *
 * Side effects:
 *	A new rrectangle item is created.
 *
 *--------------------------------------------------------------
 */

static int
CreateRrectangle(canvasPtr, itemPtr, argc, argv)
    register Tk_Canvas *canvasPtr;	/* Canvas to hold new item. */
    Tk_Item *itemPtr;			/* Record to hold new item;  header
					 * has been initialized by caller. */
    int argc;				/* Number of arguments in argv. */
    char **argv;			/* Arguments describing rrectangle. */
{
    register RrectangleItem *rectPtr = (RrectangleItem *) itemPtr;
    int i;

    if (argc < 4) {
	Tcl_AppendResult(canvasPtr->interp, "wrong # args:  should be \"",
			 Tk_PathName(canvasPtr->tkwin), "\" create ",
			 itemPtr->typePtr->name, " xc yc ?options?",
			 (char *) NULL);
	return TCL_ERROR;
      }

    /*
     * Carry out initialization that is needed to set defaults and to
     * allow proper cleanup after errors during the the remainder of
     * this procedure.
     */

    rectPtr->xc = rectPtr->xc = 0.0;
    rectPtr->width = rectPtr->height = 0.0;
    rectPtr->angle = 0.0;
    rectPtr->sine = 0.0;
    rectPtr->cosine = 1.0;
    rectPtr->outlineWidth = 0;
    rectPtr->outlineColor = NULL;
    rectPtr->fillColor = NULL;
    rectPtr->fillStipple = None;
    rectPtr->outlineGC = None;
    rectPtr->fillGC = None;
    rectPtr->dashNumber = 0;
    rectPtr->dashList = (char *)0;

    /*
     * Count the number of points and then parse them into a point
     * array.  Leading arguments are assumed to be points if they
     * start with a digit or a minus sign followed by a digit.
     */

    if ((TkGetCanvasCoord(canvasPtr, argv[0], &rectPtr->xc) != TCL_OK) ||
	(TkGetCanvasCoord(canvasPtr, argv[1], &rectPtr->yc) != TCL_OK)) {
      return TCL_ERROR;
    }

   if (ConfigureRrectangle(canvasPtr, itemPtr, argc-2, argv+2, 0) != TCL_OK) {
     DeleteRrectangle(canvasPtr, itemPtr);
     return TCL_ERROR;
   }

  return TCL_OK;

  error:
    DeleteRrectangle(canvasPtr, itemPtr);
    return TCL_ERROR;
}

/*
 *--------------------------------------------------------------
 *
 * RrectangleCoords --
 *
 *	This procedure is invoked to process the "coords" widget
 *	command on rrectangles.  It returns or sets the coordinates
 *      of the rectangle center.
 *
 * Results:
 *	Returns TCL_OK or TCL_ERROR, and sets canvasPtr->interp->result.
 *
 * Side effects:
 *	The coordinates for the given item may be changed.
 *
 *--------------------------------------------------------------
 */

static int
RrectangleCoords(canvasPtr, itemPtr, argc, argv)
    register Tk_Canvas *canvasPtr;	/* Canvas containing item. */
    Tk_Item *itemPtr;			/* Item whose coordinates are to be
					 * read or modified. */
    int argc;				/* Number of coordinates supplied in
					 * argv. */
    char **argv;			/* Array of coordinates: x1, y1,
					 * x2, y2, ... */
{
    register RrectangleItem *rectPtr = (RrectangleItem *) itemPtr;
    char xbuf[TCL_DOUBLE_SPACE], ybuf[TCL_DOUBLE_SPACE];
    
    if (argc == 0) {

      Tcl_PrintDouble(canvasPtr->interp,
		      rectPtr->xc, xbuf);
      Tcl_PrintDouble(canvasPtr->interp,
		      rectPtr->yc, ybuf);
      Tcl_AppendResult(canvasPtr->interp, xbuf, " ", ybuf, (char *)0);
      return(TCL_OK);

    } else if (argc == 2) {

      if ((TkGetCanvasCoord(canvasPtr, argv[0], &rectPtr->xc) != TCL_OK) ||
	  (TkGetCanvasCoord(canvasPtr, argv[1], &rectPtr->yc) != TCL_OK)) {
	return(TCL_ERROR);
      }
      ComputeRrectangleBbox(canvasPtr, rectPtr);
      return(TCL_OK);

    } else if ((argc == 1) && (strlen(argv[0]) >= 5) &&
	       !strncmp(argv[0], "-vertices", strlen(argv[0]))) {
      /**  Vertices request  **/
      double x, y, sw2, sh2, cw2, ch2;

      sw2 = 0.5 * rectPtr->sine * rectPtr->width;
      sh2 = 0.5 * rectPtr->sine * rectPtr->height;
      cw2 = 0.5 * rectPtr->cosine * rectPtr->width;
      ch2 = 0.5 * rectPtr->cosine * rectPtr->height;
    
      Tcl_PrintDouble(canvasPtr->interp,
		      rectPtr->xc - cw2 - sh2, xbuf);
      Tcl_PrintDouble(canvasPtr->interp,
		      rectPtr->yc + sw2 - ch2, ybuf);
      Tcl_AppendResult(canvasPtr->interp, xbuf, " ", ybuf, " ", (char *)0);

      Tcl_PrintDouble(canvasPtr->interp,
		      rectPtr->xc + cw2 - sh2, xbuf);
      Tcl_PrintDouble(canvasPtr->interp,
		      rectPtr->yc - sw2 - ch2, ybuf);
      Tcl_AppendResult(canvasPtr->interp, xbuf, " ", ybuf, " ", (char *)0);

      Tcl_PrintDouble(canvasPtr->interp,
		      rectPtr->xc + cw2 + sh2, xbuf);
      Tcl_PrintDouble(canvasPtr->interp,
		      rectPtr->yc - sw2 + ch2, ybuf);
      Tcl_AppendResult(canvasPtr->interp, xbuf, " ", ybuf, " ", (char *)0);

      Tcl_PrintDouble(canvasPtr->interp,
		      rectPtr->xc - cw2 + sh2, xbuf);
      Tcl_PrintDouble(canvasPtr->interp,
		      rectPtr->yc + sw2 + ch2, ybuf);
      Tcl_AppendResult(canvasPtr->interp, xbuf, " ", ybuf, (char *)0);

      return(TCL_OK);
    } else if ((argc == 1) && (strlen(argv[0]) >= 6) &&
	     !strncmp(argv[0], "-config", strlen(argv[0]))) {
    /**  Config request  **/
    Tcl_PrintDouble(canvasPtr->interp,
		    rectPtr->width, xbuf);
    Tcl_PrintDouble(canvasPtr->interp,
		    rectPtr->height, ybuf);
    Tcl_AppendResult(canvasPtr->interp, "-width ", xbuf,
		     " -height ", ybuf, " ", (char *)0);
    
    Tcl_PrintDouble(canvasPtr->interp,
		    rectPtr->angle, xbuf);
    Tcl_AppendResult(canvasPtr->interp, "-angle ", xbuf, (char *)0);

    return(TCL_OK);
  } else if ((argc == 4) && !strcmp("-shape",argv[0])) {
    /* request to change nth coordinate */
      int rank, newrank;
      double newx, newy, newX, newY;
      double deltax, deltay, w2, h2;
      
      if ((Tcl_GetInt(canvasPtr->interp, argv[1], &rank) != TCL_OK) ||
	  (Tcl_GetDouble(canvasPtr->interp, argv[2], &newx) != TCL_OK) ||
	  (Tcl_GetDouble(canvasPtr->interp, argv[3], &newy) != TCL_OK)) {
	Tcl_AppendResult(canvasPtr->interp,
			 "expected <int> <double> <double>, got ",
			 argv[1], argv[2], argv[3], (char *)0);
	return(TCL_ERROR);
      }

      newX = rectPtr->cosine * (newx - rectPtr->xc) -
	rectPtr->sine * (newy - rectPtr->yc);
      newY = rectPtr->sine * (newx - rectPtr->xc) +
	rectPtr->cosine * (newy - rectPtr->yc);

      w2 = rectPtr->width / 2;
      h2 = rectPtr->height / 2;

      newrank = rank;
      if ((rank == 0) || (rank == 1)) {
	deltay = (newY + h2)/2;
	rectPtr->height = fabs(h2 - newY);
	if (newY > h2)
	  newrank = 3 - rank;
      } else {
	deltay = (newY - h2)/2;
	rectPtr->height = fabs(h2 + newY);
	if (newY < -h2)
	  newrank = 3 - rank;
      }
      if ((rank == 1) || (rank == 2)) {
	deltax = (newX - w2)/2;
	rectPtr->width = fabs(w2 + newX);
	if (newX < -w2)
	  newrank = (5-rank)%4;
      } else {
	deltax = (newX + w2)/2;
	rectPtr->width = fabs(w2 - newX);
	if (newX > w2)
	  newrank = (5-rank)%4;
      }

      rectPtr->xc +=
	rectPtr->cosine * deltax + rectPtr->sine * deltay;
      rectPtr->yc +=
	-rectPtr->sine * deltax + rectPtr->cosine * deltay;
      
      ComputeRrectangleBbox(canvasPtr, rectPtr);

      sprintf(canvasPtr->interp->result, "%d", newrank);      
      return(TCL_OK);
      
    } else if ((argc == 4) &&
	       (strlen(argv[0]) >= 4) &&
	       !strncmp(argv[0], "-rotate", strlen(argv[0])) ) {
      /* rotate request */
      double xo, yo, x ,y, angle, s, c;

      if ((Tcl_GetDouble(canvasPtr->interp, argv[1], &xo) != TCL_OK) ||
	  (Tcl_GetDouble(canvasPtr->interp, argv[2], &yo) != TCL_OK) ||
	  (Tcl_GetDouble(canvasPtr->interp, argv[3], &angle) != TCL_OK)) {
	Tcl_AppendResult(canvasPtr->interp,
			 "expected -rotate <double> <double> <double>, got -rotate ",
			 argv[1], argv[2], argv[3], (char *)0);
	return(TCL_ERROR);
      }
      
      rectPtr->angle += angle;
      while (rectPtr->angle <= -90.0) rectPtr->angle += 180.0;
      while (rectPtr->angle > 90.0)   rectPtr->angle -= 180.0;

      rectPtr->cosine = (double)cos(rectPtr->angle * M_PI / 180.0);
      rectPtr->sine   = (double)sin(rectPtr->angle * M_PI / 180.0);
      
      angle *= M_PI / 180.0;
      c = (double)cos(angle);
      s = (double)sin(angle);

      x = rectPtr->xc - xo;
      y = rectPtr->yc - yo;

      rectPtr->xc = xo + c * x + s * y;
      rectPtr->yc = yo - s * x + c * y;

      ComputeRrectangleBbox(canvasPtr, rectPtr);
      return(TCL_OK);
    } else {
      sprintf(canvasPtr->interp->result,
	      "wrong # coordinates:  expected 0 or 4, got %d",
	      argc);
      return TCL_ERROR;
    }
}

/*
 *--------------------------------------------------------------
 *
 * ConfigureRrectangle --
 *
 *	This procedure is invoked to configure various aspects
 *	of a rrectangle item such as its background color.
 *
 * Results:
 *	A standard Tcl result code.  If an error occurs, then
 *	an error message is left in canvasPtr->interp->result.
 *
 * Side effects:
 *	Configuration information, such as colors and stipple
 *	patterns, may be set for itemPtr.
 *
 *--------------------------------------------------------------
 */

static int
ConfigureRrectangle(canvasPtr, itemPtr, argc, argv, flags)
    Tk_Canvas *canvasPtr;	/* Canvas containing itemPtr. */
    Tk_Item *itemPtr;		/* Rrectangle item to reconfigure. */
    int argc;			/* Number of elements in argv.  */
    char **argv;		/* Arguments describing things to configure. */
    int flags;			/* Flags to pass to Tk_ConfigureWidget. */
{
  register RrectangleItem *rectPtr = (RrectangleItem *) itemPtr;
  XGCValues gcValues;
  GC newGC;
  unsigned long mask;
  double lastAngle;
  Tk_ConfigSpec *configSpecs;

  lastAngle = rectPtr->angle;

  if (IsFilled(rectPtr))
    configSpecs = frectConfigSpecs;
  else
    configSpecs = orectConfigSpecs;
  
  if (Tk_ConfigureWidget(canvasPtr->interp, canvasPtr->tkwin,
			 configSpecs, argc, argv, (char *) rectPtr, flags) != TCL_OK) {
    return TCL_ERROR;
  }
  
  /*
   * A few of the options require additional processing, such as
   * graphics contexts.
   */

  if (!IsOutlined(rectPtr) ||
      (rectPtr->outlineColor == NULL)) {
    newGC = None;
  } else {
    gcValues.foreground = rectPtr->outlineColor->pixel;
    gcValues.cap_style = CapProjecting;
    if (rectPtr->outlineWidth < 0) {
      rectPtr->outlineWidth = 1;
    }
    gcValues.line_width = rectPtr->outlineWidth;
    mask = GCForeground|GCCapStyle|GCLineWidth;
    newGC = Tk_GetGC(canvasPtr->tkwin, mask, &gcValues);
  }
  
  if (rectPtr->outlineGC != None) {
    Tk_FreeGC(canvasPtr->display, rectPtr->outlineGC);
  }
  rectPtr->outlineGC = newGC;

  if (!IsFilled(rectPtr) || (rectPtr->fillColor == NULL)) {
    newGC = None;
  } else {
    gcValues.foreground = rectPtr->fillColor->pixel;
    if (rectPtr->fillStipple != None) {
      gcValues.stipple = rectPtr->fillStipple;
      gcValues.fill_style = FillStippled;
      mask = GCForeground|GCStipple|GCFillStyle;
    } else {
      mask = GCForeground;
    }
    newGC = Tk_GetGC(canvasPtr->tkwin, mask, &gcValues);
  }
  if (rectPtr->fillGC != None) {
    Tk_FreeGC(canvasPtr->display, rectPtr->fillGC);
  }
  rectPtr->fillGC = newGC;
  
  /**  Angle stuff  **/
  if (!FEQUAL(rectPtr->angle, lastAngle)) {
    while (rectPtr->angle <= -90.0) rectPtr->angle += 180.0;
    while (rectPtr->angle > 90.0)   rectPtr->angle -= 180.0;
    /*  Recompute cos and sin */
    rectPtr->sine = sin(M_PI*rectPtr->angle/180.0);
    rectPtr->cosine = cos(M_PI*rectPtr->angle/180.0);
  }      
      
  ComputeRrectangleBbox(canvasPtr, rectPtr);

  return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * DeleteRrectangle --
 *
 *	This procedure is called to clean up the data structure
 *	associated with a rrectangle item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Resources associated with itemPtr are released.
 *
 *--------------------------------------------------------------
 */

static void
DeleteRrectangle(canvasPtr, itemPtr)
    Tk_Canvas *canvasPtr;		/* Info about overall canvas widget. */
    Tk_Item *itemPtr;			/* Item that is being deleted. */
{
    register RrectangleItem *rectPtr = (RrectangleItem *) itemPtr;

    if (rectPtr->outlineColor != NULL) {
	Tk_FreeColor(rectPtr->outlineColor);
    }
    if (rectPtr->fillColor != NULL) {
	Tk_FreeColor(rectPtr->fillColor);
    }
    if (rectPtr->fillStipple != None) {
	Tk_FreeBitmap(canvasPtr->display, rectPtr->fillStipple);
    }
    if (rectPtr->outlineGC != None) {
	Tk_FreeGC(canvasPtr->display, rectPtr->outlineGC);
    }
    if (rectPtr->fillGC != None) {
	Tk_FreeGC(canvasPtr->display, rectPtr->fillGC);
    }
}

/*
 *--------------------------------------------------------------
 *
 * ComputeRrectangleBbox --
 *
 *	This procedure is invoked to compute the bounding box of
 *	all the pixels that may be drawn as part of a rrectangle.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The fields x1, y1, x2, and y2 are updated in the header
 *	for itemPtr.
 *
 *--------------------------------------------------------------
 */

#define F2GI(X) ((int)((X) + ((X) > 0 ? 0.5 : -0.5)))

static void
ComputeRrectangleBbox(canvasPtr, rectPtr)
    register Tk_Canvas *canvasPtr;	/* Canvas that contains item. */
    RrectangleItem *rectPtr;		/* Item whose bbox is to be
					 * recomputed. */
{
  int bloat;
  double sw2, sh2, cw2, ch2;
  double X1, Y1, X2, Y2;
  double sb, cb;
  
  sw2 = 0.5 * rectPtr->sine * rectPtr->width;
  sh2 = 0.5 * rectPtr->sine * rectPtr->height;
  cw2 = 0.5 * rectPtr->cosine * rectPtr->width;
  ch2 = 0.5 * rectPtr->cosine * rectPtr->height;
  if (IsOutlined(rectPtr)) {
    cb = 0.5*rectPtr->outlineWidth*rectPtr->cosine;
    sb = 0.5*rectPtr->outlineWidth*rectPtr->sine;
  } else {
    cb = sb = 0.0;
  }
  
  if (rectPtr->angle < 0.0) {
    X1 = rectPtr->xc - cw2 + sh2 + cb - sb;
    X2 = rectPtr->xc + cw2 - sh2 - cb + sb;
    Y1 = rectPtr->yc + sw2 - ch2 - sb + cb;
    Y2 = rectPtr->yc - sw2 + ch2 + sb - cb;
  } else {
    X1 = rectPtr->xc - cw2 - sh2 + cb + sb;
    X2 = rectPtr->xc + cw2 + sh2 - cb - sb;
    Y1 = rectPtr->yc - sw2 - ch2 + sb + cb;
    Y2 = rectPtr->yc + sw2 + ch2 - sb - cb;
  }

  /**  Don't add 0.5 here, or modify the display proc!  **/
  rectPtr->header.x1 = X1 - 2;
  rectPtr->header.x2 = X2 + 2;
  rectPtr->header.y1 = Y1 - 2;
  rectPtr->header.y2 = Y2 + 2;
}
	    

/*
 *--------------------------------------------------------------
 *
 * DisplayRrectangle --
 *
 *	This procedure is invoked to draw a rrectangle item in a given
 *	drawable.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	ItemPtr is drawn in drawable using the transformation
 *	information in canvasPtr.
 *
 *--------------------------------------------------------------
 */

static void
DisplayRrectangle(canvasPtr, itemPtr, drawable)
    register Tk_Canvas *canvasPtr;	/* Canvas that contains item. */
    Tk_Item *itemPtr;			/* Item to be displayed. */
    Drawable drawable;			/* Pixmap or window in which to draw
					 * item. */
{
    register RrectangleItem *rectPtr = (RrectangleItem *) itemPtr;
    Display *display = Tk_Display(canvasPtr->tkwin);
    double x, y, sw, sh, cw, ch;
    unsigned long valuemask;
    XGCValues savedLineStyle;
    XPoint points[5];
    double sb, cb, boffset;

    /*  compute vertices  */
    /*  Take care of X line drawing protocol here  */
    /* we wan't outlined rectangles entirely included in */
    /* [xc-w/2 , xc + w/2]  */
    
    if (IsOutlined(rectPtr)) {
      cb = rectPtr->outlineWidth*rectPtr->cosine;
      sb = rectPtr->outlineWidth*rectPtr->sine;
      boffset = 0.5;
    } else {
      cb = sb = 0.0;
      boffset = 0.0;
    }
    
    sw = rectPtr->sine * rectPtr->width;
    sh = rectPtr->sine * rectPtr->height;
    cw = rectPtr->cosine * rectPtr->width;
    ch = rectPtr->cosine * rectPtr->height;
    
    x = rectPtr->xc + 0.5 * (-cw+cb-sh+sb) - boffset;
    y = rectPtr->yc + 0.5 * (sw-sb-ch+cb) - boffset;
    points[0].x = points[4].x = SCREEN_X(canvasPtr, x);
    points[0].y = points[4].y = SCREEN_Y(canvasPtr, y);
    
    x += cw - cb;
    y += - sw + sb;
    points[1].x = SCREEN_X(canvasPtr, x);
    points[1].y = SCREEN_Y(canvasPtr, y);
    
    x += sh - sb;
    y += ch - cb;
    points[2].x = SCREEN_X(canvasPtr, x);
    points[2].y = SCREEN_Y(canvasPtr, y);
    
    x -= cw - cb;
    y -= - sw + sb;
    points[3].x = SCREEN_X(canvasPtr, x);
    points[3].y = SCREEN_Y(canvasPtr, y);

    if (rectPtr->outlineGC != None) {
      /*  display rectangle as a closed polyline  */

      /* If dash style != solid, then modify the GC dash style.
       * Be sure to reset the dash style when done, since the
       * GC is supposed to be read-only. Note that if an XSetDashes had been
       * performed on the GC before this point, it won't be restored.
       */
      if (rectPtr->dashNumber > 0) {
	XGCValues newLineStyle;

	valuemask = GCLineStyle;
	XGetGCValues(canvasPtr->display, rectPtr->outlineGC,
		     valuemask, &savedLineStyle);
	newLineStyle.line_style = LineOnOffDash;
	XChangeGC(canvasPtr->display, rectPtr->outlineGC,
		  valuemask, &newLineStyle);
	XSetDashes(canvasPtr->display, rectPtr->outlineGC,
		   0, rectPtr->dashList, rectPtr->dashNumber);
      }

      XDrawLines(display, drawable, rectPtr->outlineGC,
		 points, 5, CoordModeOrigin);

      if (rectPtr->dashNumber > 0) {
	XChangeGC(canvasPtr->display, rectPtr->outlineGC,
		  valuemask, &savedLineStyle);
	/* Here's the place for a XSetDashes ( savedDashes ). */
      }
    } else if (rectPtr->fillGC != None) {
      /*  display rectangle as a polygon  */

      if (rectPtr->fillStipple != None) {
	XSetTSOrigin(display, rectPtr->fillGC,
		     -canvasPtr->drawableXOrigin, -canvasPtr->drawableYOrigin);
      }

      XFillPolygon(display, drawable, rectPtr->fillGC,
		   points, 4, Convex, CoordModeOrigin);

      if (rectPtr->fillStipple != None) {
	XSetTSOrigin(display, rectPtr->fillGC, 0, 0);
      }
    }
}

/*
 *--------------------------------------------------------------
 *
 * RrectangleToPoint --
 *
 *	Computes the distance from a given point to a given
 *	rrectangle, in canvas units.
 *
 * Results:
 *	The return value is 0 if the point whose x and y coordinates
 *	are pointPtr[0] and pointPtr[1] is inside the rrectangle.  If the
 *	point isn't inside the rrectangle then the return value is the
 *	distance from the point to the rrectangle.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

	/* ARGSUSED */
static double
RrectangleToPoint(canvasPtr, itemPtr, pointPtr)
    Tk_Canvas *canvasPtr;	/* Canvas containing item. */
    Tk_Item *itemPtr;		/* Item to check against point. */
    double *pointPtr;		/* Pointer to x and y coordinates. */
{
    register RrectangleItem *rectPtr = (RrectangleItem *) itemPtr;
    double xDiff, yDiff, x1, y1, x2, y2, inc, tmp;
    double xp, yp;

    /*  Rotate the point and reuse the code of tkRectOval.c  */
    xp = rectPtr->xc +
      rectPtr->cosine * (pointPtr[0]-rectPtr->xc)
	- rectPtr->sine * (pointPtr[1]-rectPtr->yc);
    yp = rectPtr->yc +
      rectPtr->sine * (pointPtr[0]-rectPtr->xc)
	+ rectPtr->cosine * (pointPtr[1]-rectPtr->yc);

    x1 = rectPtr->xc - 0.5*rectPtr->width;
    y1 = rectPtr->yc - 0.5*rectPtr->height;
    x2 = rectPtr->xc + 0.5*rectPtr->width;
    y2 = rectPtr->yc + 0.5*rectPtr->height;

    if (IsOutlined(rectPtr) && (rectPtr->outlineGC != None)) {
      inc = 0.5*rectPtr->outlineWidth;
      x1 -= inc;
      y1 -= inc;
      x2 += inc;
      y2 += inc;
    }

    /*
     * If the point is inside the rectangle, handle specially:
     * distance is 0 if rectangle is filled, otherwise compute
     * distance to nearest edge of rectangle and subtract width
     * of edge.
     */
    
    if ((xp >= x1) && (xp < x2)
		&& (yp >= y1) && (yp < y2)) {
	if (IsFilled(rectPtr)) {
	  return 0.0;
	}
	xDiff = xp - x1;
	tmp = x2 - xp;
	if (tmp < xDiff) {
	    xDiff = tmp;
	}
	yDiff = yp - y1;
	tmp = y2 - yp;
	if (tmp < yDiff) {
	    yDiff = tmp;
	}
	if (yDiff < xDiff) {
	    xDiff = yDiff;
	}
	xDiff -= rectPtr->outlineWidth;
	if (xDiff < 0.0) {
	    return 0.0;
	}
	return xDiff;
    }

    /*
     * Point is outside rectangle.
     */

    if (xp < x1) {
	xDiff = x1 - xp;
    } else if (xp > x2)  {
	xDiff = xp - x2;
    } else {
	xDiff = 0;
    }

    if (yp < y1) {
	yDiff = y1 - yp;
    } else if (yp > y2)  {
	yDiff = yp - y2;
    } else {
	yDiff = 0;
    }

    return hypot(xDiff, yDiff);
}

/*
 *--------------------------------------------------------------
 *
 * RrectangleToArea --
 *
 *	This procedure is called to determine whether an item
 *	lies entirely inside, entirely outside, or overlapping
 *	a given rectangular area.
 *
 * Results:
 *	-1 is returned if the item is entirely outside the
 *	area, 0 if it overlaps, and 1 if it is entirely
 *	inside the given area.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

	/* ARGSUSED */
static int
RrectangleToArea(canvasPtr, itemPtr, areaPtr)
    Tk_Canvas *canvasPtr;	/* Canvas containing item. */
    Tk_Item *itemPtr;		/* Item to check against rrectangle. */
    double *areaPtr;
{
    register RrectangleItem *rectPtr = (RrectangleItem *) itemPtr;
    double xc, yc;
    double halfWidth;
    double sw2, sh2, cw2, ch2;
    double x1, y1, x2, y2, x3, y3, x4, y4;
    double X1, Y1, X2, Y2;

    if (IsOutlined(rectPtr)) 
      halfWidth = 0.5 * rectPtr->outlineWidth;
    else 
      halfWidth = 0.0;

    xc = rectPtr->xc;
    yc = rectPtr->yc;
    
    X1 = areaPtr[0];
    Y1 = areaPtr[1];
    X2 = areaPtr[2];
    Y2 = areaPtr[3];
    
    /*  Compute vertices coordinates  */
    sw2 = 0.5 * rectPtr->sine * (rectPtr->width + halfWidth);
    sh2 = 0.5 * rectPtr->sine * (rectPtr->height + halfWidth);
    cw2 = 0.5 * rectPtr->cosine * (rectPtr->width + halfWidth);
    ch2 = 0.5 * rectPtr->cosine * (rectPtr->height + halfWidth);
    
    /*  See if rect is entirely in area  */
    if (rectPtr->angle < 0.0) {
      x2 = xc + cw2 - sh2;
      x4 = xc - cw2 + sh2;
      y1 = yc + sw2 - ch2;
      y3 = yc - sw2 + ch2;

      if ((x4 > areaPtr[0]) && (x2 < areaPtr[2]) &&
	  (y1 > areaPtr[1]) && (y3 < areaPtr[3]))
	return 1;

      x1 = xc - cw2 - sh2;
      x3 = xc + cw2 + sh2;
      y2 = yc - sw2 - ch2;
      y4 = yc + sw2 + ch2;
      
    } else {
      x1 = xc - cw2 - sh2;
      x3 = xc + cw2 + sh2;
      y2 = yc - sw2 - ch2;
      y4 = yc + sw2 + ch2;
      
      if ((x1 > areaPtr[0]) && (x3 < areaPtr[2]) &&
	  (y2 > areaPtr[1]) && (y4 < areaPtr[3]))
	return 1;

      x2 = xc + cw2 - sh2;
      x4 = xc - cw2 + sh2;
      y1 = yc + sw2 - ch2;
      y3 = yc - sw2 + ch2;
    }
      
    /* otherwise use XRegion */
    {
      int status;
      Region region;
      XPoint points[4];

      points[0].x = (short)(0.5 + x1);
      points[0].y = (short)(0.5 + y1);
      points[1].x = (short)(0.5 + x2);
      points[1].y = (short)(0.5 + y2);
      points[2].x = (short)(0.5 + x3);
      points[2].y = (short)(0.5 + y3);
      points[3].x = (short)(0.5 + x4);
      points[3].y = (short)(0.5 + y4);

      region = XPolygonRegion(points, 4, EvenOddRule);
      status = XRectInRegion(region,
			     (int)(0.5+X1), (int)(0.5+Y1),
			     (unsigned int)(0.5 + X2 - X1),
			     (unsigned int)(0.5 + Y2 - Y1));
      XDestroyRegion(region);
      
      switch (status) {
      case RectangleIn:    return 0;
      case RectanglePart : return 0;
      case RectangleOut:   return -1;
      }
    }
}


/*
 *--------------------------------------------------------------
 *
 * ScaleRrectangle --
 *
 *	This procedure is invoked to rescale a rrectangle item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The rrectangle referred to by itemPtr is rescaled so that the
 *	following transformation is applied to all point
 *	coordinates:
 *		x' = originX + scaleX*(x-originX)
 *		y' = originY + scaleY*(y-originY)
 *
 *--------------------------------------------------------------
 */

static void
ScaleRrectangle(canvasPtr, itemPtr, originX, originY, scaleX, scaleY)
    Tk_Canvas *canvasPtr;		/* Canvas containing rrectangle. */
    Tk_Item *itemPtr;			/* Rrectangle to be scaled. */
    double originX, originY;		/* Origin about which to scale rect. */
    double scaleX;			/* Amount to scale in X direction. */
    double scaleY;			/* Amount to scale in Y direction. */
{
  RrectangleItem *rectPtr = (RrectangleItem *) itemPtr;

  rectPtr->xc = originX + scaleX*(rectPtr->xc - originX);
  rectPtr->yc = originY + scaleY*(rectPtr->yc - originY);

  if ((rectPtr->angle > -45.0) && (rectPtr->angle < 45.0)) {
    rectPtr->width *= fabs(scaleX);
    rectPtr->height *= fabs(scaleY);
  } else {
    rectPtr->width *= fabs(scaleY);
    rectPtr->height *= fabs(scaleX);
  }
  ComputeRrectangleBbox(canvasPtr, rectPtr);
}

/*
 *--------------------------------------------------------------
 *
 * TranslateRrectangle --
 *
 *	This procedure is called to move a rrectangle by a given amount.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The position of the rrectangle is offset by (xDelta, yDelta), and
 *	the bounding box is updated in the generic part of the item
 *	structure.
 *
 *--------------------------------------------------------------
 */

static void
TranslateRrectangle(canvasPtr, itemPtr, deltaX, deltaY)
    Tk_Canvas *canvasPtr;		/* Canvas containing item. */
    Tk_Item *itemPtr;			/* Item that is being moved. */
    double deltaX, deltaY;		/* Amount by which item is to be
					 * moved. */
{
    RrectangleItem *rectPtr = (RrectangleItem *) itemPtr;

    rectPtr->xc += deltaX;
    rectPtr->yc += deltaY;
    ComputeRrectangleBbox(canvasPtr, rectPtr);
}

/*
 *--------------------------------------------------------------
 *
 * ParseDashes
 *
 *	This procedure is called back during option parsing to
 *	parse dash information
 *
 * Results:
 *	The return value is a standard Tcl result:  TCL_OK means
 *	that the arrow shape information was parsed ok, and
 *	TCL_ERROR means it couldn't be parsed.
 *
 * Side effects:
 *	Dash information in recordPtr is updated.
 *
 *--------------------------------------------------------------
 */

static int
ParseDashes(clientData, interp, tkwin, value, recordPtr, offset)
    ClientData clientData;	/* Not used. */
    Tcl_Interp *interp;		/* Used for error reporting. */
    Tk_Window tkwin;		/* Not used. */
    char *value;		/* Textual specification of arrow shape. */
    char *recordPtr;		/* Pointer to item record in which to
				 * store arrow information. */
    int offset;			/* Offset of shape information in widget
				 * record. */
{
  RrectangleItem *rectPtr = (RrectangleItem *) recordPtr;
  int argc;
  char **argv = NULL;
  
  if (offset != Tk_Offset(RrectangleItem, dashList)) {
    panic("ParseDashes received bogus offset");
  }
  
  if (Tcl_SplitList(interp, value, &argc, &argv) != TCL_OK) {
  syntaxError:
    Tcl_ResetResult(interp);
    Tcl_AppendResult(interp, "bad dash list \"", value,
		     "\": must be even length list of numbers", (char *) NULL);
    if (argv != NULL) {
      ckfree((char *) argv);
    }
    return TCL_ERROR;
  }
  if (argc%2 != 0) {
    goto syntaxError;
  }
  
  if (rectPtr->dashList != (char *)0)
    free(rectPtr->dashList);
  rectPtr->dashList = (char *)0;
  rectPtr->dashNumber = argc;
  
  if (argc > 0) {
    /**  fill in the dashList array  **/
    int ibuf, i;
    rectPtr->dashList = (char *)malloc(argc * sizeof(char));
    for (i = 0; i < argc; i++) {
      if (Tcl_GetInt(interp, argv[i], &ibuf) != TCL_OK) {
	free(rectPtr->dashList);
	rectPtr->dashList = (char *)0;
	goto syntaxError;
      }
      rectPtr->dashList[i] = (char)ibuf;
    }
  }
  
  ckfree((char *) argv);
  
  return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * PrintDashes
 *
 *	This procedure is a callback invoked by the configuration
 *	code to return a printable value describing a dash list.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static char *
PrintDashes(clientData, tkwin, recordPtr, offset, freeProcPtr)
    ClientData clientData;	/* Not used. */
    Tk_Window tkwin;		/* Window associated with rectPtr's widget. */
    char *recordPtr;		/* Pointer to item record containing current
				 * shape information. */
    int offset;			/* Offset of arrow information in record. */
    Tcl_FreeProc **freeProcPtr;	/* Store address of procedure to call to
				 * free string here. */
{
    RrectangleItem *rectPtr = (RrectangleItem *) recordPtr;
    char *buffer;

    if (rectPtr->dashNumber == 0) {
      buffer = ckalloc(1);
      *buffer = '\0';
    } else {
      int i;
      
      buffer = ckalloc((4*rectPtr->dashNumber+1) * sizeof(char));
      sprintf(buffer, "%d", rectPtr->dashList[0]);
      for (i = 1; i < rectPtr->dashNumber; i++)
	sprintf(buffer + strlen(buffer), " %d", rectPtr->dashList[i]);
    }
    *freeProcPtr = (Tcl_FreeProc *) free;
    return buffer;
}

/*
 *--------------------------------------------------------------
 *
 * RrectangleToPostscript --
 *
 *	This procedure is called to generate Postscript for
 *	rrectangle items.
 *
 * Results:
 *	The return value is a standard Tcl result.  If an error
 *	occurs in generating Postscript then an error message is
 *	left in canvasPtr->interp->result, replacing whatever used
 *	to be there.  If no error occurs, then Postscript for the
 *	item is appended to the result.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static int
RrectangleToPostscript(canvasPtr, itemPtr, psInfoPtr)
    Tk_Canvas *canvasPtr;		/* Information about overall canvas. */
    Tk_Item *itemPtr;			/* Item for which Postscript is
					 * wanted. */
    Tk_PostscriptInfo *psInfoPtr;	/* Information about the Postscript;
					 * must be passed back to Postscript
					 * utility procedures. */
{
  register RrectangleItem *rectPtr = (RrectangleItem *) itemPtr;
  char pathCmd[500], string[100];
  double w2, h2;
  double X1, Y1, X2, Y2;
  double sb, cb;
  
  /**  Compute vertices **/
  if (IsOutlined(rectPtr)) {
    cb = 0.5*rectPtr->outlineWidth*rectPtr->cosine;
    sb = 0.5*rectPtr->outlineWidth*rectPtr->sine;
  } else {
    cb = sb = 0.0;
  }
      
  w2 =   0.5*rectPtr->width;
  h2 =   0.5*rectPtr->height;
  
  X1 = rectPtr->xc - rectPtr->cosine * w2 - rectPtr->sine * h2 + cb + sb;
  Y1 = TkCanvPsY(psInfoPtr,
		 rectPtr->yc + rectPtr->sine * w2 - rectPtr->cosine * h2 - sb + cb);
  X2 = rectPtr->xc + rectPtr->cosine * w2 + rectPtr->sine * h2 - cb - sb;
  Y2 = TkCanvPsY(psInfoPtr,
		 rectPtr->yc - rectPtr->sine * w2 + rectPtr->cosine * h2 + sb - cb);

  /*
   * Generate a string that creates a path for the rectangle
   */

  sprintf(pathCmd,
	  "%.15g %.15g moveto %.15g 0 rlineto 0 %.15g rlineto %.15g 0 rlineto closepath\n",
	  X1, Y1, X2-X1, Y2-Y1, X1-X2);

  /*
   * First draw the filled area of the rectangle.
   */

  if (IsFilled(rectPtr)) {
    if (rectPtr->fillColor != NULL) {
      Tcl_AppendResult(canvasPtr->interp, pathCmd, (char *) NULL);
      if (TkCanvPsColor(canvasPtr, psInfoPtr, rectPtr->fillColor)
	  != TCL_OK) {
	return TCL_ERROR;
      }
      if (rectPtr->fillStipple != None) {
	if (TkCanvPsStipple(canvasPtr, psInfoPtr,
			    rectPtr->fillStipple, 1) != TCL_OK) {
	  return TCL_ERROR;
	}
      } else {
	Tcl_AppendResult(canvasPtr->interp, "fill\n", (char *) NULL);
      }
    }
  }
  
  /*
   * Now draw the outline, if there is one.
   */
  
  if (IsOutlined(rectPtr) && (rectPtr->outlineColor != NULL)) {
    Tcl_AppendResult(canvasPtr->interp, pathCmd, (char *) NULL);
    sprintf(string, "%d setlinewidth", rectPtr->outlineWidth);
    Tcl_AppendResult(canvasPtr->interp, string,
		     " 0 setlinejoin 2 setlinecap\n", (char *) NULL);
    if (TkCanvPsColor(canvasPtr, psInfoPtr, rectPtr->outlineColor)
	!= TCL_OK) {
      return TCL_ERROR;
    }
    if (rectPtr->dashNumber > 0) {
      char buffer[256];
      int i;
      sprintf(buffer, "[%d", rectPtr->dashList[0]);
      for (i = 1; i < rectPtr->dashNumber; i++)
	sprintf(buffer + strlen(buffer), " %d", rectPtr->dashList[i]);
      strcat(buffer, "] 0 setdash\n");
      Tcl_AppendResult(canvasPtr->interp, buffer, (char *) NULL);
    }
    Tcl_AppendResult(canvasPtr->interp, "stroke\n", (char *) NULL);
  }
  return TCL_OK;
}
