#! /bin/sh

#
# Copyright (c) 1994 Open Software Foundation, Inc.
# 
# Permission is hereby granted to use, copy, modify and freely distribute
# the software in this file and its documentation for any purpose without
# fee, provided that the above copyright notice appears in all copies, and
# that both the copyright notice and this permission notice appear in
# supporting documentation.  Further, provided that the name of Open
# Software Foundation, Inc. ("OSF") not be used in advertising or
# publicity pertaining to distribution of the software without prior
# written permission from OSF.  OSF makes no representations about the
# suitability of this software for any purpose.  It is provided "AS IS"
# without express or implied warranty.
#

# When ot tools are moved to '/usr/local/bin' get rid of the
# next line.  This is just to work around the fact that the
# /project/tools/bin/... directory name is too long to be given
# with #! notation.
true && eval 'exec otcl /project/tools/bin/i386/otmig ${1+"$@"}'

set argv0 [file tail $argv0]

# otmig
# Migrate CR from one project to another.

# otmig relies on two external interfaces, provided in a
# source/destination project-specific file in /project/ot/tcl.
# This file defines procedures preAdjust and postAdjust.
#
# proc preAdjust generates a string which is sent to the server for
# evaluation before the 'TCL' form of the CR is sent to the destination server.
# The commands generated by preAdjust should include things like the
# following:
# - If any source project fields are missing in the destination project
# 	template, then you should define a null procedure with the name
#	of the abbreviation (keyword) for the field here.  For example,
#	if you are moving a CR from osf1 to osf1_maint, and osf1 has a
#	field with keyword "xnum" and osf1_maint has no such keyword, then
#	return the following in your string
#		proc xnum { args } { }
#	(The 'args' means there will be no argument count checking.)
#
# proc postAdjust generates a string which is sent to the server for
# evaluation after the 'TCL' form of the CR is sent to the destination server.
# Use this for things like the following:
# - You need to set the IDNUM field to 'nnnnnn' (which is required for enter
#   operations).  This is usually 'num' but could be given a project-specific
#   name.
# - If there are any fields in the source destination project metatemplate
#   which are missing in the destination project metatemplate, then you can
#   either 1) rely on default values set in the destination project 
#   metatemplate, or 2) override these defaults in a command you give here.

set otMigUsage [format {usage: %s [option ...] 
  -h 		    print help
  -p <proj1:proj2>  proj1 is source, proj2 is destination
  -n <num>	    enter proj1's CR <num> into proj2
  -x <string>	    update string to apply during enter
} $argv0] 


proc otMigPrintHelp { } {
    global otMigUsage

    puts stdout $otMigUsage nonewline;     return
}

proc otConnect { curPrj } {

    if { $curPrj == "" } {
	error "Requires project specification"
    }
    set msg [project set $curPrj]
    if { [string match {*not found} $msg] } {
	error $msg    
    }
    if { [catch [list remote [list project set $curPrj]] msg] } {
	error $msg
    }

    return
}

proc otMig { objNum srcPrj dstPrj toEval } {

# 'connect' to the project (= connect to the project server).
# That enables the 'remote' command - you can issue commands directly.
# Give the commands
# 	update <objNum>
#	tclString

    if { [catch [list otConnect $srcPrj] msg] } 	   {	error $msg   } 

    set srcObj [project objectName]
    set srcDir [project projectDirectory]

    if { [catch [list source [format "%s/%s/%s_%s.tcl" $srcDir $srcPrj $srcPrj $dstPrj]] msg] } {
	error $msg
    }

# This does a 'query <objNum>' - full, w/ history lines - so we can
# get the history lines of the original CR.

    set queryCmd "control { {fullText 1} {historyLines 1} }; query "
    append queryCmd $objNum
    if { [catch [list remote $queryCmd] msg] }	   	   {	error $msg   }

# Fabricate an OT TCL command ("note append ..") which will add a note
# containing the original CR's history lines.  These must be indented so
# they won't be intepreted as the history lines of the new CR.

    set historyCmd {note append }
    set historyNote [format "\[ot %s public\]\nOriginal history lines: \n" [exec /bin/date +%D]]
    regexp \n..?HISTORY.*$ $msg historyLines
    if { [regsub -all \n $historyLines "\n " newHistoryLines] } {
	set historyLines $newHistoryLines
    }
    append historyNote $historyLines
    append historyCmd [list $historyNote]

# Obtain the CR by checking it out and asking the server for a 'tclString'
# form of the CR.  Then unlock it and disconnect from that server.

    if { [catch [list remote [list update $objNum]] msg] } {  error $msg     }
    if { [catch [list remote [list tclString]] tclData] }  {  error $tclData }

    if { [regsub {^clear header; } $tclData {} newTclData] } {
	set tclData $newTclData
    }

    if { [catch {remote unlock} msg] } 			   {	  error $msg }

# This is a hack until we have time to fix a bug in cmdCommit.
    if { [catch {remote [list update 0]} msg] } {
    }

    catch {remote serverExit} msg

    if { [catch [list otConnect $dstPrj] msg] } 	        { error $msg } 

    if { [catch [list remote enter] msg] } {
	error $msg
    } else {
	if { $msg != {} } {
	    error $msg
	}
    }

    if { [catch preAdjust preAdjustReturn] } {
    } else {
	if { $preAdjustReturn != {} } {
	    if { [catch [list remote $preAdjustReturn] msg] } {
		error $msg
	    }
	}
    }

    if { [catch [list remote $tclData] msg] } {
	error $msg
    }

    if { [catch postAdjust postAdjustReturn] } {
    } else {
	if { $postAdjustReturn != {} } {
	    if { [catch [list remote $postAdjustReturn] msg] } {
		error $msg
	    }
	}
    }

    if { $toEval != {} } {
	if { [catch [list remote $toEval] msg] } {
	    error $msg
	}
    }

    if { [ catch [list remote $historyCmd] histMsg ] } {
	error $histMsg
    } 

    if { [ catch {remote commit} commitMsg ] } {
	error $commitMsg
    }

    if { $commitMsg == {} } {
	if { [catch [list remote {format "%s %s" [project objectName] [num]}] msg] } {
	    return [format "CR moved from %s to %s" $srcPrj $dstPrj]
	} else {
	    return [format "Moved %s %s %s to %s %s" $srcPrj $srcObj $objNum $dstPrj $msg]
	}
    }
    error $commitMsg

}

if { $argc == 0 } {
    otMigPrintHelp
    exit 1
}

set i 0
set curPrj {}
set srcDst {}
set srcPrj {}
set dstPrj {}
set objNum {}
set toEval {}

while { $i < [llength $argv] } {

    switch -glob -- [lindex $argv $i] {
	{-p}	{ set curPrj [lindex $argv [incr i]]			}
	{-p*}	{ set curPrj [string range [lindex $argv $i] 2 end]	}
	{-n}	{ set objNum [lindex $argv [incr i]]			}
	{-x}	{ set toEval [lindex $argv [incr i]]			}
	*	{ otMigPrintHelp ; exit 1				}
    } 
    incr i

}

set srcDst [split $curPrj :]
if { [llength $srcDst] != 2 } {
    puts stderr "need source and destination project (e.g. osf1:osf1_maint)"
    exit 1
}

if { $objNum == {} } {
    puts stderr "need object number"
    exit 1
}

set srcPrj [lindex $srcDst 0]
set dstPrj [lindex $srcDst 1]

set otMigStat [catch [list otMig $objNum $srcPrj $dstPrj $toEval] msg]

if { $otMigStat } {
    puts stderr $msg
} else {
    puts stdout $msg
}
catch {remote serverExit} msg

exit $otMigStat
