#!/usr/local/bin/wish -f
#
# VERSION
# tkInfo version 0.4
#
# Tk script to read GNU "info" files and display them.  tkInfo can be
# used stand alone (via WISH), or embedded within an application to provide
# integrated, on-line help.
#
# The info file format supports a robust hypercard system.
# Info files can be created manually with any text editor, with the
# support of the emacs "info" package, or with the GNU "makeinfo" program.
# The makeinfo program produces a set of info files from TeX source that
# use the GNU texinfo format.  (The one that uses "@" everywhere.)
#
# For more information, please obtain the GNU "texinfo" package.  This
# may be obtained via anonymous ftp from "prep.ai.mit.edu" in
# "pub/gnu/texinfo-2.16.tar.z".  Also consider the "latexinfo" package
# available from "csc-sun.math.utah.edu" in "pub/tex/pub/latexinfo".
#
# In order to use tkinfo, in stand-alone
# mode, you must have a valid info tree on your system.  Most gnu emacs
# installations have an info tree (try looking in /usr/local/lib/emacs).
# For embedded mode, you must create info files; the texinfo distribution
# will explain how to do this.
#
# This release works with tcl6.6/tk3.1.  It should also work with
# tcl6.7/tk3.2.  The key-binding in this release are not as robust
# as I would like, and more menus could be added.
#
# Author: Kennard White <kennard@ohm.berkeley.edu>
#
# begun 13/Jan/1993.
# RCS: $Header: /vol/cooley/cooley1/kennard/src/tkgraph/lib/RCS/tkinfo.tcl,v 1.9 93/05/18 15:39:43 kennard Exp Locker: kennard $
#
# See below for copyright.  Basically you can re-distribute this any way
# you like, just dont sue me and dont pretend you wrote tkInfo.
# Portions of the GNU "makeinfo" is GNU copylefted, but
# only the author of the document needs "makeinfo": the author can freely
# redistribute the info files produced by "makeinfo" (the ones read by
# this script).
#


# Stand-alone usage, via WISH:
# usage: tkinfo [-dir dir1] [-dir dir2] ... [-infofile filePath] [node]
# Note that the "-file" argument is taken by WISH, and thus cant be used
# to specify the info file name.
# Environment variables:
#    INFOPATH	A colon (``:'') seperated list of directories to search
#		for info files.  Equivalent to each -dir option above.
#    INFOSUFFIX	A colon seperated list of file suffixes to try when searching
#		for an info file.  tkinfo will automatically try the
#		suffixes .z and .Z and deal with the compressed files
#		if required.
# The default INFOPATH and INFOSUFFIX is defined in tkiInit() below.

# Embedded usage:
# Set the global variable ``tkiEmbed''.  Source this file.
# If your info files are installed in a non-standard place,
# call proc tkiAddInfoPaths.  It takes one argument, which should
# be a directory to search for info files.
# To popup an info window, call tkiGetAndDisplay,
# passing it a nodeSpec (see below) of what to display.
# The nodeSpec may be an empty string, in which case the highest level
# node in the system is displayed.

# This script requires "topgetopt.tcl", which should be installed in
# the same directory as this script.  In stand-alone mode, this file is
# directly source'd. In embedded mode, tkinfo depends on the auto-load
# feature (so sure to update the tclIndex).

# This file is organized in 3 parts:
#   1)  This comment area, which briefly describes the info file format
#	and this implementation.
#   2)	Code for parsing info files.  This code is independent of tk.
#   3)	Code for displaying info files/nodes.  This calls the parsing
#	functions and then displays the results in various tk widgets.
#
# The program provides on-line help of itself: node (builtin)QuickHelp,
# and the texinfo distribution contains a tutorial: node (info)Top.

# tkInfo requires the following global variables:
#  tki		This is a huge array where essentially all the loaded info-files
#		are stored.  It also contains some configuration state.
#		The contents of this is described below.
#  .tki##	Each toplevel info window has a global variable associated
#		with it.  The name of the variable is the same as the
#		toplevel window name, which is ".tki" followed by some number.
#  tkiEmbed	tkInfo can operate stand-alone (like the "info" program) or
#		embedded (part of your application).  Embedded mode is
#		used iff this variable exists.  When this file is
#		sourced in the stand-alone case, the argv options will be
#		parsed (see tkiBoot() below) and a new toplevel window
#	 	will be opened.

# The core structure of an info file is a {node}.  Each info file consists
# of a set of nodes seperated by a magic character.  Each nodes consists of
# of a header, body, and menu.  There are also special nodes that contain
# control information used to reference "split" files and speed up access.
# The first half of the code below parses this file format.
# A node may be specified in one of several ways (called a {nodeSpec}):
#	(filename)nodename
#	nodename		The given node within the current file.
#	(filename)		The "Top" node of the file.
#	(filename)*		The entire file.


# In the implementation below, the info format consists of {nodes} stored 
# in files.  A given info file has three identifiers associated with it:
#  -	The {filename}, which is the name used either by the user to
#	reference a file, or by one info file to reference another.
#	Such a reference could be complete UNIX path name (either
#	absolute or relative), or may be a partial specification (see below).
#  -	The {filepath}, which is a valid UNIX path name to access the
#	file.  The filepath is derived from the filename.  If the filename
#	is already a valid path, no work needs be done.  Otherwise,
#	the filepath is formed by prepending a path prefix and appending
#	a file suffix.  These are defined by the INFOPATH and INFOSUFFIX
#	variables.
#  -	The {filekey}, which is an internal, auto-generated token associated
#	with each file.


#
# The global array "tki" contains the following elements:
#   file-$fileName	The fileKey for $fileName.
#   fileinfo-$fileKey	The fileinfo struct for $fileKey. Each fileinfo is
#			{ fileKey fileName filePath pntKey wholeB }
#   incore-$fileKey	Boolean 0/1; true if file has been loaded into core.
#   nodesinfo-$fileKey	A list of nodeinfo for every node in $fileKey.
#			Each nodeinfo is a list { idx node file up prev next }.
#   nodesbody-$fileKey	A list of the textual body for every node in $fileKey.
#   indirf-$fileKey	List of indirect-file-info for $fileKey.  Each
#			info is a list { indirFileKey byteOfs }.
#   indirn-$fileKey	List of indirect-node-info for $fileKey.  Each
#			info is a list { nodeName byteOfs fileKey }.
#   xrefinfo-$fileKey-$nodeIdx
#			Contains information on all cross reference
#			pointers within the node's body text.
#   menuinfo-$fileKey-$nodeIdx
#			Contains information on all menu entries
#			within the node's menu text.
#
#
# Notes (some important, some not).
# 1.	Because of the graphical system, there may be several paraell
#	info windows active.  These windows must operate independently.
#	Because of this, there can be no concept of the "current file"
#	or "current node" within the tkinfo core.  Rather, this information
#	must be maintained by the window.
# 2.	Because of #1, we must maintain multiple files in core.  Currently
#	we never flush.
# 3.	The background color used in tkiInit() is BISQUE1, from tk/defaults.h
# 4.	The byte offsets in the indirect tables are not used as such;
#	this is because we parse the file when loaded.  However, they are
#	used to identify which indirect file the node is in.
# 5.	The function tkiLoadFile() attempts to deal with compressed files.
#	Currently it uses "zcat" for .Z files and "gunzip -c" for .z files.
#	If you have better suggestions, please let me know.
#

# Ignore this, it is used by a custom auto-reload script.
proc tkinfo.tcl { } { }

proc tkiInit { } {
    global tki env auto_path tkiEmbed

    set defInfoPath [list . /usr/tools/gnu/info /usr/gnu/info /usr/local/emacs/info /usr/local/lib/gnumacs/info /usr/local/lib/emacs/info]
    set defInfoSuffix [list "" .info -info]

    if { ! [info exist tki] } {
	set tki(self)		[info script]
	set tki(sn)		0
	set tki(compresscat-Z)	"zcat"
	set tki(compresscat-z)	"gunzip -c"
	set tki(background)	"#ffe4c4"
    	set tki(rawHeadersB)	0
    	set tki(nodeSep)	"\037"
    	set tki(nodeByteSep)	"\177"
	set tki(topLevelFile)	"dir"
	set tki(topLevelNode)	"Top"
	set tki(normCursor)	"left_ptr"
	set tki(waitCursor)	"watch"
	set tki(curWindow)	""
	if [info exist env(INFOPATH)] {
	  tkiAddInfoPaths [split $env(INFOPATH) ":"]
	} else {
	  tkiAddInfoPaths $defInfoPath
	}
	if [info exist env(INFOSUFFIX)] {
	    set tki(infoSuffix) [split $env(INFOSUFFIX) ":"]
	} else {
	    set tki(infoSuffix) $defInfoSuffix
	}
	# We need to be able to find topgetopt.tcl.  Assume its in
	# the same directory as us.  Could just do:
	#	lappend auto_path [file dirname $tki(self)]
	# but this requires the tclIndex to be up-to-date, which is to
	# much to ask of some users.  So just source it here.
	if { "[info procs topgetopt.tcl]"=="" } {
	    set selfdir [file dirname $tki(self)]
	    set otherfile $selfdir/topgetopt.tcl
	    if { [file isfile $otherfile] } {
		if { [info exist tkiEmbed] }	{ lappend auto_path $selfdir } \
		else				{ source $otherfile }
	    }
	}
	_tkiBuiltinFile
	rename _tkiBuiltinFile ""
    }
}

proc tkiAddInfoPaths { newPaths } {
    global tki

    if { ! [info exist tki(infoPath) ] } {
	set tki(infoPath) ""
    }
    if { [llength $newPaths] > 0 } {
	set tki(infoPath) [eval linsert {$tki(infoPath)} 0 $newPaths]
    }
    # Kill off null paths
    while 1 {
	set idx [lsearch $tki(infoPath) ""]
	if { $idx < 0 }	break
	set tki(infoPath) [lreplace $tki(infoPath) $idx $idx]
    }
    # Kill off duplicate paths
    for {set idx 0} {$idx < [llength $tki(infoPath)]} {incr idx} {
	set path [lindex $tki(infoPath) $idx]
	while 1 {
	    set dup [lsearch [lrange $tki(infoPath) [expr {$idx+1} ] end] $path]
	    if { $dup < 0 } break
	    set tki(infoPath) [lreplace $tki(infoPath) $dup $dup]
	}
    }
}

proc tkiUninit { } {
    global tki
    catch {unset tki}
}

proc tkiReset { } {
    tkiUninit
    tkiInit
}

proc _tkiNull { args } {
}

proc tkiStatus { msg } {
    global tki
    if { "$tki(curWindow)"=="" } {
        puts stdout "tkInfo: $msg"
    } else {
	$tki(curWindow).s.status conf -text "$msg"
	# idletasks should be sufficient, but the geometry management
	# apparently needs some X-events to make the redisplay occur
	update
    }
}

proc tkiWarning { msg } {
    # Warnings allways go to stderr
    puts stderr "tkInfo Warning: $msg"
}

proc tkiError { msg } {
    global tki
    if { "$tki(curWindow)"=="" } {
        puts stdout "tkInfo Error: $msg"
    } else {
	$tki(curWindow).s.status conf -text "Error: $msg"
    }
}

proc tkiGetSN { } {
    global tki

    incr tki(sn)
    return $tki(sn)
}

#
# This proc is called once during initialization, and then destroyed.
# (It is destroyed to save memory).
# Currently we fake all the appropriate table entires to create a "builtin"
# file.  It might be easier, however, to just pass one large text string
# into the parser and have it be dealt with like any other file.
#
proc _tkiBuiltinFile { } {
    global tki

    set fileKey			builtin
    set tki(file-$fileKey)	$fileKey
    set tki(fileinfo-$fileKey)	[list $fileKey $fileKey $fileKey "" 0]
    set tki(incore-$fileKey)	1
    set tki(nodesinfo-$fileKey) ""
    set tki(nodesbody-$fileKey) ""

    tkiFileParseNode $fileKey "
File: builtin, Node: Top, Up: (dir)Top
This is the builtin info on tkInfo itself.
* Menu:
* Copyright::
* About::
* ToDo::
* QuickHelp::
* Source Code: ($tki(self))*
"

    tkiFileParseNode $fileKey {
File: builtin, Node: Copyright, Up: Top, Next: About
Copyright for tkInfo
--------------------
This copyright applies to the tkInfo system only.  If tkInfo is
embedded within a larger system, that system will most likely have
a different copyright.

Sorry this is so long.  Basically, do whatever you want with this
software, just don't sue me and don't pretend you wrote it -- kennard.

Copyright (c) 1993 The Regents of the University of California.
All rights reserved.

Permission is hereby granted, without written agreement and without
license or royalty fees, to use, copy, modify, and distribute this
software and its documentation for any purpose, provided that the above
copyright notice and the following two paragraphs appear in all copies
of this software.

IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY 
FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES 
ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF 
THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF 
SUCH DAMAGE.

THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY OF
CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
ENHANCEMENTS, OR MODIFICATIONS.
    }
#VERSION
    tkiFileParseNode $fileKey {
File: builtin, Node: About, Up: Top, Prev: Copyright, Next: ToDo
About tkInfo-0.4
----------------
tkInfo version 0.4 (based on tcl-6.6 and tk-3.1) by Kennard White

tkInfo displays "info" files.  "Info" files provide a hyper-text
capability that is ideal for on-line help.  The format is suitable for
both tty-based systems and graphical systems.  In addition, the same
document source can produce both a "nice" hardcopy manual and Info
files (see texinfo below).

Info files can be created manually using any text-editor (vi,emacs), or
from TeX sources using the GNU texinfo style format via the "makeinfo"
program.  There is an emacs package for editing and displaying Info
files.  There is also a standalone C program for displaying info files
called "info".  See FSF/GNU texinfo-2.12 at ftp@prep.ai.mit.edu.

tkInfo is the work of Kennard White.  For information, comments,
code-contributions or bug reports please contact me at:
	e-mail: kennard@ohm.Berkeley.EDU	phone: 510/643-6686
    	s-mail:	Kennard White, Box #115
		207 Cory Hall
		UC Berkeley
		Berkeley, CA 94720, USA

RCS: $Revision: 1.9 $, $Date: 93/05/18 15:39:43 $
    }

    tkiFileParseNode $fileKey {
File: builtin, Node: ToDo, Up: Top, Prev: About, Next: QuickHelp
tkInfo is still incomplete.  The following is my todo list.

Better text widget bindings (page-up,page-down,top-of-node,bottom-of-node).
Better menu bars, and more user-configurability (colors, fonts, etc).
Add option to allow all the "*note:" to not be drawn on the screen,
	or change them to "see also".
Implement all the currently unimplemented bindings.
Implement a tcl-only, tty-based interface?
Implement stat'ing of the source files with auto-reload.
Profile the whole mess: speed up file loading and node formating.
Figure out some huiristic for un-loading files to save memory.
    }

    tkiFileParseNode $fileKey {
File: builtin, Node: QuickHelp, Up: Top, Prev: ToDo
		tkInfo Commands

Use scroll bar on right side to scroll through the node text.  Cross references
and menu entires are shown highlighted (blue foreground).  Press the left mouse
button over highlighted text to see that node.

Accelerator keys:
?	Show this quick help message.
h	Invoke the Info (emacs/tty) tutorial.
n       Move to the "next" node of this node.
p       Move to the "previous" node of this node.
u       Move "up" from this node.
b	Move "back" to last node you were at, toggle based.
l       Move to the "last" node you were at, stack based.
d       Move to the "directory" node.  Equivalent to `gDIR'.
q	Quit info window.

Advanced commands:
g,(	Goto given node (Ctrl-g to abort,Ctrl-u to erase line).
1-5	menu (not implemented).
m/f	menu/footnote (not implemented).
s	Search (not implemented).
p	Print (not implemented).
!	Issue tcl command.

Scrolling commands:
SPACE, Ctrl-V, DELETE, b	(not implemented)

Go "up" from this node to obtain more information on tkInfo.
    }
}



proc _tkiFileFindSuf { fileName } {
    global tki

    foreach suf $tki(infoSuffix) {
	foreach extrasuf {"" .Z .z} {
	    set filePath "$fileName$suf$extrasuf"
	    if { [file isfile $filePath] } {
		return $filePath
	    }
	}
    }
    return ""
}

#
# Given {fileName} (see intro section above), find the cooresponding
# filepath.  The filepath of {pntFileKey}, if specified, is
# used as a starting point for locating {fileName}.
# Returns the file path if found, else empty string.
#
proc tkiFileFind { fileName {pntFileKey ""} } {
    global tki

    case [string index $fileName 0] {
      "/ . ~" {
	# Should be valid UN*X path modulo suffix
	set filePath [_tkiFileFindSuf $fileName]
	if { "$filePath"!="" } { return $filePath }
      }
      default {
	# Try all the infopaths, and all suffixs
	set pp ""
	if { "$pntFileKey"!="" } {
	    set pp [file dirname [lindex $tki(fileinfo-$pntFileKey) 2]]
	}
	foreach prepath "$pp $tki(infoPath)" {
# puts stdout "Searching dir ``$prepath''"
	    if { ! [file isdir $prepath] } continue
	    set filePath [_tkiFileFindSuf $prepath/$fileName]
	    if { "$filePath"!="" } { return $filePath }
	    set filePath [_tkiFileFindSuf $prepath/[string tolower $fileName]]
	    if { "$filePath"!="" } { return $filePath }
	}
      }
    }
    return ""
}

#
# Given {fileName}, find the coorsponding filepath via tkiFindFile().
# Create a {fileKey} for the file, and make the appropriate table entries.
# Note that {fileName} must be just that, and not a filekey.
#
proc tkiFileAdd { fileName {pntFileKey ""} {wholeB 0} } {
    global tki

    if { [info exist tki(file-$fileName)] } {
	return $tki(file-$fileName)
    }
    set filePath [tkiFileFind $fileName $pntFileKey]
    if { "$filePath"=="" } { return "" }
    set fileKey fk[tkiGetSN]
    set tki(file-$fileName) $fileKey
    set tki(fileinfo-$fileKey)  [list $fileKey $fileName $filePath $pntFileKey $wholeB]
    set tki(incore-$fileKey) 0
    return $fileKey
}

proc tkiFileGet { fileSpec {pntFileKey ""} {wholeB 0} } {
    global tki

    if { [info exist tki(fileinfo-$fileSpec)] } {
	set fileKey $fileSpec
    } else {
	if { [info exist tki(file-$fileSpec)] } {
	    set fileKey $tki(file-$fileSpec)
	} else {
	    set fileKey [tkiFileAdd $fileSpec $pntFileKey $wholeB]
	    if { "$fileKey"=="" } {
		tkiError "Can't locate info file ``$fileSpec''."
		return ""
	    }
	}
    }
    set fileinfo $tki(fileinfo-$fileKey)
    if { ! $tki(incore-$fileKey) } {
        tkiFileLoad $fileKey [lindex $fileinfo 1] [lindex $fileinfo 2] [lindex $fileinfo 4]
    }
    return $fileKey
}

proc _tkiFileLoadIndirectTbl { fileKey lines } {
    global tki

    set indirinfos ""
    foreach line $lines {
	if { "$line"!="" } {
	    set pair [split $line ":"]
	    if { [llength $pair] != 2 } {
		tkiWarning "$fileKey has bad file-indirect line ``$line''"
		continue
	    }
	    set indirKey [tkiFileAdd [lindex $pair 0] $fileKey]
	    if { "$indirKey"=="" } {
		tkiError "Can't locate indirect file ``[lindex $pair 0]''."
		continue
	    }
	    set byteOfs [string trim [lindex $pair 1]]
	    lappend indirinfos [list $indirKey $byteOfs]
	}
    }
    set tki(indirf-$fileKey) $indirinfos
#puts stdout "IndirectTbl $fileKey: $indirinfos"
}

proc _tkiFileLookupIndir { indirf byte } {
    set lastKey ""
    foreach fi $indirf {
	if { [lindex $fi 1] > $byte } break
	set lastKey [lindex $fi 0]
    }
    return $lastKey
}

proc _tkiFileLoadTagTbl { fileKey lines } {
    global tki

    set subkey [lindex $lines 0]
    if { "$subkey"!="(Indirect)" } return
    set indirf $tki(indirf-$fileKey)
    set indirinfos ""
    foreach line [lrange $lines 1 end] {
	if { "$line"=="" } continue
	set pair [split $line $tki(nodeByteSep)]
	if { [llength $pair] != 2 } {
	    tkiWarning "$fileKey has bad tag-indirect line ``$line''"
	    continue
	}
	set nodeName [string tolower [string trim [string range [lindex $pair 0] 5 end]]]
	set byteOfs [string trim [lindex $pair 1]]
	set indirFile [_tkiFileLookupIndir $indirf $byteOfs]
	lappend indirinfos [list $nodeName $byteOfs $indirFile]
# puts stdout "$fileKey: tag [list $nodeName $byteOfs $indirFile]"
    }
    set tki(indirn-$fileKey) $indirinfos
}

proc tkiFileParseNode { fileKey node } {
    global tki

    set lines [split $node "\n"]
    set keyline [string trim [lindex $lines 1]]
    case $keyline {
      { {[Ii]ndirect:} } {
	_tkiFileLoadIndirectTbl $fileKey [lrange $lines 2 end]
	return "IndirectTable"
      }
      { {[Tt]ag [Tt]able:} } {
	_tkiFileLoadTagTbl $fileKey [lrange $lines 2 end]
	return "TagTable"
      }
      { {[Ee]nd [Tt]ag [Tt]able} } {
	return "EndTagTable"
      }
    }
    # Some screwed up files omit the ``,'' for the file key.
    regsub "(File:\[^,\]*)Node:" $keyline "\\1,Node:" keyline
    set nodekey ""; set filekey ""
    set nextkey ""; set prevkey ""; set upkey ""
    foreach key [split $keyline ",\t"] {
	set key [string trim $key]
	case $key {
	  "File:*" { set filekey [string trim [string range $key 5 end]] }
	  "Node:*" { set nodekey [string trim [string range $key 5 end]] }
	  "Up:*"   { set upkey   [string trim [string range $key 3 end]] }
	  "Prev:*" { set prevkey [string trim [string range $key 5 end]] }
	  "Next:*" { set nextkey [string trim [string range $key 5 end]] }
	}
    }
    if { "$nodekey" == "" } { return "" }
    lappend tki(nodesinfo-$fileKey) [list [llength $tki(nodesinfo-$fileKey)] $nodekey $filekey $upkey $prevkey $nextkey]
    lappend tki(nodesbody-$fileKey) $node
    return $nodekey
}

proc tkiFileLoad { fileKey fileName filePath wholeB } {
    global tki

    case $filePath in {
      *.Z	{ set fp "|$tki(compresscat-Z) $filePath" }
      *.z	{ set fp "|$tki(compresscat-z) $filePath" }
      default	{ set fp $filePath }
    }
    if [catch {open $fp "r"} fid] {
	tkiError "Can't open ``$fp''."
	return ""
    }
    if { $wholeB } {
	set node_Top [list 0 Top $fileName "" "" ""]
    	set tki(nodesinfo-$fileKey) [list $node_Top]
    	set tki(nodesbody-$fileKey) [list [read $fid]]
        close $fid
        set tki(incore-$fileKey) 1
	return $fileKey
    }
    set nodelist [split [read $fid] $tki(nodeSep)]
    close $fid
    tkiStatus "Loading $fileName"
    set nodecnt 0
    set tki(nodesinfo-$fileKey) ""
    set tki(nodesbody-$fileKey) ""
    foreach node $nodelist {
	incr nodecnt
	if { $nodecnt==1 || [string length $node] < 10 } continue
	set nodeName [tkiFileParseNode $fileKey $node]
	if { "$nodeName" == "" } {
	    puts stdout "Warning: node #$nodecnt of file $filePath is bogus"
	    continue
	}
    }
    set tki(incore-$fileKey) 1
    return $fileKey
}

#
# This is the core search function.  It attempts to locate {nodeSpec}
# where ever it is.  {fileSpec} is a default file name that is used
# only if {nodeSpec} doesn't contain a reference.
# Returns a list {nodeIdx fileKey}, where {nodeIdx} is the index of the
# node within {fileKey}.
#
# As discussed in the intro above, at this level we cannot allow any concept of
# "current file" or "current node": it is up to the caller to maintain
# that information and pass up the appropriate arguments.
#
proc tkiGetNodeRef { nodeSpec {fileSpec ""} {pntFileKey ""} } {
    global tki

    if { "[string index $nodeSpec 0]"=="(" } {
	set ridx [string first ")" $nodeSpec]
	if { $ridx <= 0 } {
	    error "Malformed nodespec ``$nodeSpec''"
	}
	set fileSpec [string range $nodeSpec 1 [expr $ridx-1]]
	set nodeSpec [string trim [string range $nodeSpec [expr $ridx+1] end]]
    }
    if { "$fileSpec"=="" } {
	set fileSpec $tki(topLevelFile)
    }
    if { "$nodeSpec"=="" } {
	set nodeSpec $tki(topLevelNode)
    }
    set wholeB 0
    if { "$nodeSpec"=="*" } {
	set wholeB 1
	set nodeSpec Top
    }
    set fileKey [tkiFileGet $fileSpec $pntFileKey $wholeB]
    if { "$fileKey"=="" } { return "" }
    set fileName [lindex $tki(fileinfo-$fileKey) 1]
    set realPntKey [lindex $tki(fileinfo-$fileKey) 3]
    tkiStatus "Searching for ``$nodeSpec'' in $fileName"
    set nodeSpec [string tolower $nodeSpec]

    # Popup to our indirect-parent, if it has a tag table
    if { "$pntFileKey"=="" && "$realPntKey"!="" 
      && [info exist tki(indirn-$realPntKey)] } {
	return [tkiGetNodeRef $nodeSpec $realPntKey]
    }

    #  Use index on this file, pushdown to our children
    if { [info exist tki(indirn-$fileKey)] } {
	# Use node index (indirect)
	foreach indir $tki(indirn-$fileKey) {
	    if { [string match $nodeSpec [lindex $indir 0]] } {
		set nodeRef [tkiGetNodeRef $nodeSpec [lindex $indir 2] $fileKey]
	        if { "$nodeRef"!="" } { return $nodeRef }
		tkiWarning "$fileKey: Incorrect tag table"; break
	    }
	}
    }

    # Brute force on this file
    if { [info exist tki(nodesinfo-$fileKey)] } {
	foreach nodeinfo $tki(nodesinfo-$fileKey) {
	    if { [string match $nodeSpec [string tolower [lindex $nodeinfo 1]]] } {
		return [list [lindex $nodeinfo 0] $fileKey]
	    }
	}
    }

    # Look for node in all indirect files (brute force)
    if { [info exist tki(indirf-$fileKey)] } {
	foreach indir $tki(indirf-$fileKey) {
# puts stdout "Searching $indir"
	    set nodeRef [tkiGetNodeRef $nodeSpec [lindex $indir 0] $fileKey]
	    if { "$nodeRef"!="" } { return $nodeRef }
	}
    }

    # Look for node in my parent, but only if not called from my pnt
    if { "$pntFileKey"=="" } {
	if { "$realPntKey"!="" } {
	    return [tkiGetNodeRef $nodeSpec $realPntKey]
	}
    }
    return ""
}

#
# Parse a nody-body and identify the cross references.
#
proc tkiNodeParseBody { nodeName fileKey bodytext } {
    set nl "node ``($fileKey)$nodeName''"

    # There are two forms:
    #	*note nodeSpec::terminator			(form 1)
    #   *note label: nodeSpec terminator		(form 2)
    # Terminator is ``.'' or ``,'', forms may wrap accross lines.
    set rp1 "\*(note\[ \t\n\]*)(\[^:\]+)::"
    set sp1 "x\\1\037c\\2\037dxx"
    set rp2 "\*(note\[ \t\n\]*)(\[^:\]+)(:\[ \t\n\]*)(\(\[^ \t\n)\]+\))?(\[^.,\]*)\[.,\]"
    set sp2 "x\\1\037a\\2\037b\\3\037c\\4\\5\037dx"

    regsub -all -nocase $rp1 $bodytext $sp1 bodytext
    regsub -all -nocase $rp2 $bodytext $sp2 bodytext
    set xrefinfo ""
    set curIdx 1
    foreach seg [split $bodytext "\037"] {
	set stIdx $curIdx
	set curIdx [expr { $curIdx + [string length $seg] - 1 }]
	if { "[string index $seg 0]"!="c" } continue
	set toNode [string range $seg 1 end]
#puts stdout "tkiNodeParseBody:1 ``$toNode''"
	regsub -all "\[ \t\n\]+" $toNode " " toNode
#puts stdout "tkiNodeParseBody:2 ``$toNode''"
	lappend xrefinfo [list [llength $xrefinfo] $toNode $stIdx $curIdx]
    }
    return $xrefinfo
}

#
# Parse the menu and extract the keywords
#
proc tkiNodeParseMenu { nodeName fileKey menutext } {
    global tki

    # There are two forms:
    #	* nodeSpec::	comments...			(form 1)
    #   * label: nodeSpec[ \t.,] comments...		(form 2)
    set rp1 "(\*\[ \t\]*)(\[^:\]+)::"
    set sp1 "\\1\037A\\2\037B"
    # rp2 = "* ws label: ws", rp2a="rp2 nodename ws", rp2b="rp2 (file)node ws"
    set rp2 "(\*\[ \t\]*)(\[^:\]+)(:\[ \t\]*)(\(\[^ \t)\]+\))?(\[^\t.,\]*)"
    set sp2 "\\1\037A\\2\037B\\3\037C\\4\\5\037D"

    set menuinfo ""
    set linecnt 0; set menucnt 0
    set nl "node ``($fileKey)$nodeName''"
    foreach line [split $menutext "\n"] {
	incr linecnt
	if { "[string index $line 0]"!="*" 
	  || "[string range $line 0 6]"=="* Menu:" } continue
	incr menucnt
#puts stdout "Try rp $line"
	if { [regsub $rp1 $line $sp1 prsline] } {
	    set nBeg [expr { [string first "\037A" $prsline] + 0 } ]
	    set nEnd [expr { [string first "\037B" $prsline] - 3 } ]
	} else {
	    if { [regsub $rp2 $line $sp2 prsline] } {
	        set nBeg [expr { [string first "\037C" $prsline] - 4 } ]
	        set nEnd [expr { [string first "\037D" $prsline] - 7 } ]
	    } else {
		tkiWarning "$nl has bad menu (line $linecnt)"
		continue
	    }
	}
	set toNode [string range $line $nBeg $nEnd]
	lappend menuinfo [list $linecnt $menucnt $toNode $nBeg $nEnd]
    }
    return $menuinfo
}

proc _tkiDpyWinAction { w action {extra ""} } {
    upvar #0 $w wvars
    global tki

    set toNode ""
    set toFile $wvars(fileKey) 
    case $action {
      quit {
	unset wvars
	destroy $w
	# XXX: !!This is a major hack!!
	global tkiEmbed
	if { ![info exist tkiEmbed] && "[winfo children .]"=="" } {
	    destroy .
	}
	return
      }
      goto {
	  set dd $w.s
	  $dd.status conf -text [expr { "[string index $extra 0]"=="!" ? "Enter cmd: " : "Enter node: " } ]
	  if { "$extra"!="" } {
	      $dd.goto delete 0 end
	      $dd.goto insert end $extra
	  }
	  pack append $dd $dd.goto { left expand fillx }
	  focus $dd.goto
      }
      last {
	  set idx [expr { [llength $wvars(lastNodes)] - 2 } ]
	  if { $idx >= 0 } {
	      set lastinfo [lindex $wvars(lastNodes) $idx]
	      set wvars(lastNodes) [lreplace $wvars(lastNodes) $idx end]
	      set toFile [lindex $lastinfo 0]
	      set toNode [lindex $lastinfo 1]
	  }
      }
      back {
	if { [info exist wvars(back)] } {
	    set toFile [lindex $wvars(back) 0]
	    set toNode [lindex $wvars(back) 1]
	}
      }

      up   { set toNode [lindex $wvars(info) 3] }
      prev { set toNode [lindex $wvars(info) 4] }
      next { set toNode [lindex $wvars(info) 5] }
      menu {
	if { $wvars(menuB) } {
            $w.main.text yview menu.first
	}
      }
    }
    if { "$toNode"!="" } {
        tkiDpyIdle $toNode $toFile $w
    }
}

proc _tkiWinGotoOk { w } {
    upvar #0 $w wvars
    set dd $w.s
    focus $w.main.text
    pack unpack $dd.goto
    set node [string trim [$dd.goto get]]
    case $node {
      "" {
        $dd.status conf -text $wvars(status)
      }
      "!*" {
	if [catch [string range $node 1 end] error] {
	    puts stdout "Error: $error"
	} else {
	    puts stdout [expr { "$error"=="" ? "Ok" : "$error" }]
	}
        $dd.status conf -text $wvars(status)
	set node ""
      }
    }
    tkiDpyIdle $node $wvars(fileKey) $w
}

proc _tkiWinGotoAbort { w } {
    upvar #0 $w wvars
    focus $w.main.text
    set dd $w.s
    pack unpack $dd.goto
    $dd.status conf -text $wvars(status)
}

#
# Note that its not safe to change the command of the buttons while
# the button press is active.  Thus we must use idle-time handler.
#
proc tkiMakeDpyWin { } {
    global tki

    while 1 {
        set w .tki[tkiGetSN]
	if { [catch {winfo parent $w}] } break
    }
    upvar #0 $w wvars
    set wvars(info) ""
    set wvars(fileKey) ""
    set wvars(infonode) "(builtin)Top"

    toplevel $w
    wm minsize $w 20 20
    set dd $w.title; pack append $w [frame $dd] { top fillx }

    set dd $w.main; pack append $w [frame $dd] { top expand fill }
    pack append $dd [text $dd.text -state disabled -width 80 -wrap word] \
      { left expand fill }
    pack append $dd [scrollbar $dd.vsb -orient vert -com "$dd.text yview"] \
      { left fill }
    $dd.text config -yscroll "$dd.vsb set"

    set dd $w.div1; pack append $w [frame $dd -bd 1 -rel sunken \
      -height 3 -width 10] { top fillx }
    set dd $w.s; pack append $w [frame $dd] { top fillx }
    pack append $dd [label $dd.status -anc w] { left fillx }
    entry $dd.goto -rel sunken
    bind $dd.goto <Return> "_tkiWinGotoOk $w"
    bind $dd.goto <Escape> "_tkiWinGotoAbort $w"
    bind $dd.goto <Any-Control-g> "_tkiWinGotoAbort $w"

    set dd $w.buts; pack append $w [frame $dd] { top fillx }
    pack append $dd [button $dd.quit -text "Quit Info" \
      -com "_tkiDpyWinAction $w quit"] { left exp fill }
    pack append $dd [button $dd.menu -text "Show Menu" \
      -com "_tkiDpyWinAction $w menu"] { left exp fill }
    pack append $dd [button $dd.prev -text "Prev Node" \
      -com "_tkiDpyWinAction $w prev"] { left exp fill }
    pack append $dd [button $dd.up   -text "Up Node" \
      -com "_tkiDpyWinAction $w up"] { left exp fill }
    pack append $dd [button $dd.next -text "Next Node" \
      -com "_tkiDpyWinAction $w next"] { left exp fill }
    pack append $dd [button $dd.back -text "Back Node" \
      -com "_tkiDpyWinAction $w back" -state disabled] { left exp fill }

    set tw $w.main.text
    foreach win "$w.main.text $w.s.status" {
	bind $win q "$dd.quit invoke"
	bind $win m "$dd.menu invoke"
	bind $win p "$dd.prev invoke"
	bind $win u "$dd.up   invoke"
	bind $win n "$dd.next invoke"
	bind $win b "$dd.back invoke"
	bind $win l "_tkiDpyWinAction $w last"
	bind $win ? [list tkiDpyIdle "(builtin)QuickHelp" "" $w]
	bind $win h [list tkiDpyIdle "(info)Help" "" $w]
	bind $win d [list tkiDpyIdle "(dir)Top" "" $w]
	bind $win g "_tkiDpyWinAction $w goto"
	bind $win ( "_tkiDpyWinAction $w goto ("
	bind $win ! "_tkiDpyWinAction $w goto !"
	bind $win <Key-Help> "tkiContextHelp %W"
    }

    bind $w <Any-Enter> "focus $w.main.text"
    bind $w <Any-Leave> "focus none"

    set tki(curWindow) $w
    return $w
}

#
# This is more subtle than one might think.  Note that the text index
# "+1line" wont work on the last line of text, because the newline is
# considered part of the previous line.  Thus we use "lineend" instead.
#
proc _tkiTextTrim { w idx } {
    while 1 {
	set nidx [$w index "$idx lineend"]
	if { "[string trim [$w get $idx $nidx]]"!="" || "[$w index end]"=="1.0" } break
	$w delete $idx "$nidx +1char"
    }
}


# Modified version of ouster's version
proc _tkiTextInsertWithTags { w index text args } {
    set start [$w index $index]
    $w insert $start $text
    foreach tag $args {
    	$w tag add $tag $start insert
    }
}

proc _tkiConfChainButton { w which toNode } {
    $w.buts.$which conf -state [expr { "$toNode"=="" ? "disabled" : "normal" } ]
}

proc _tkiDpyNodeBody { w nodeName nodeIdx fileKey bodytext } {
    global tki

    set tw $w.main.text
    $tw insert end $bodytext

    if { [info exist tki(xrefinfo-$fileKey-$nodeIdx)] } {
	set xrefinfo $tki(xrefinfo-$fileKey-$nodeIdx)
    } else {
        set xrefinfo [tkiNodeParseBody $nodeName $fileKey $bodytext]
	set tki(xrefinfo-$fileKey-$nodeIdx) $xrefinfo
    }
    set ms "1.0"
    foreach xi $xrefinfo {
	# xi = { xrefidx toNode startIdx endIdx }
	set xrefidx [lindex $xi 0]
	set toNode [lindex $xi 1]
        $tw tag add xrefkey "$ms +[lindex $xi 2] c" "$ms +[lindex $xi 3] c"
        $tw tag add xref$xrefidx "$ms +[lindex $xi 2] c" "$ms +[lindex $xi 3] c"
	$tw tag bind xref$xrefidx <Button> [list tkiDpyIdle $toNode $fileKey $w]
    }
    $tw tag conf xrefkey -fore blue

    _tkiTextTrim $tw 1.0
    if { ! $tki(rawHeadersB) } {
	$tw delete 1.0 "1.0 +1line"
        _tkiTextTrim $tw 1.0
    }
}

proc _tkiDpyNodeMenu { w nodeName nodeIdx fileKey menutext } {
    global tki
#puts stdout "_tkiDpyNodeMenu: $nodeName"
    if { [info exist tki(menuinfo-$fileKey-$nodeIdx)] } {
	set menuinfo $tki(menuinfo-$fileKey-$nodeIdx)
    } else {
        set menuinfo [tkiNodeParseMenu $nodeName $fileKey $menutext]
	set tki(menuinfo-$fileKey-$nodeIdx) $menuinfo
    }
    set tw $w.main.text
    _tkiTextInsertWithTags $tw end $menutext menu
    foreach mi $menuinfo {
	# mi = { lineidx menuidx toNode nBeg nEnd }
	set lineidx [lindex $mi 0]
	set menuidx [lindex $mi 1]
	set toNode [lindex $mi 2]
	set ms "menu.first +$lineidx lines -1 lines"
	$tw tag add menukey "$ms +[lindex $mi 3] c" "$ms +[lindex $mi 4] c +1 c"
	$tw tag add menu$menuidx "$ms linestart" "$ms lineend"
	$tw tag bind menu$menuidx <Button> [list tkiDpyIdle $toNode $fileKey $w]
#	$tw tag bind menu$menuidx <Key> {puts stdout "key %A press"}
    }
    $tw tag conf menukey -fore blue
}

proc tkiDisplayNode { w fileKey info body } {
    global tki; upvar #0 $w wvars

    if { [info exist wvars(info)] } {
	set wvars(back) [list $wvars(fileKey) [lindex $wvars(info) 1]]
        $w.buts.back conf -state normal
    }
    lappend wvars(lastNodes) [list $fileKey [lindex $info 1]]

    set wvars(info) $info
    set wvars(fileKey) $fileKey
    set nodeIdx [lindex $info 0]
    set nodeSpec "([lindex $info 2])[lindex $info 1]"
    tkiStatus "Formating $nodeSpec"
    set tw $w.main.text
    $tw conf -cursor $tki(waitCursor)
    set menuidx -1
    if { ! [lindex $tki(fileinfo-$fileKey) 4] } {
	set menuidx [string first "\n* Menu:" $body]
	if { $menuidx > 0 } {
	    set menutext [string range $body [expr {$menuidx+1}] end]
	    set body [string range $body 0 $menuidx]
	}
    }

    #
    # Config the up,prev,next headers
    #
    set nodeName [lindex $info 1]
    _tkiConfChainButton $w up   [lindex $info 3]
    _tkiConfChainButton $w prev [lindex $info 4]
    _tkiConfChainButton $w next [lindex $info 5]
    if { $menuidx < 0 } {
	$w.buts.menu conf -state disabled
	set wvars(menuB) 0
    } else {
	$w.buts.menu conf -state normal
	set wvars(menuB) 1
    }

    #
    # Config the main section
    #
    $tw conf -state normal
    $tw delete 1.0 end
    _tkiDpyNodeBody $w $nodeName $nodeIdx $fileKey $body
    if { [info exist menutext] } {
	_tkiDpyNodeMenu $w $nodeName $nodeIdx $fileKey $menutext
    }
    $tw mark set insert 1.0
    $tw mark set anchor insert
    $tw tag remove sel 1.0 end
    $tw conf -state disabled -cursor $tki(normCursor)
    set wvars(status) "$nodeSpec                           Press ? for help."
    tkiStatus $wvars(status)

    # This is really gross
    focus $tw
    after 1 [list $tw tag remove sel 1.0 end]
}

proc tkiGetAndDisplay { nodeSpec {fileSpec ""} {w ""} } {
    global tki
    if { ! [info exist tki] } { tkiInit }
    if { "$w"=="" } {
	set w [tkiMakeDpyWin]
    }
    set tki(curWindow) $w
    set nodeRef [tkiGetNodeRef $nodeSpec $fileSpec]
    if { "$nodeRef"=="" } {
	tkiError "Can't locate info node ``$nodeSpec''"
        set tki(curWindow) ""
	return ""
    }
    set nodeIdx [lindex $nodeRef 0]
    set fileKey [lindex $nodeRef 1]
    tkiDisplayNode $w $fileKey [lindex $tki(nodesinfo-$fileKey) $nodeIdx] \
      [lindex $tki(nodesbody-$fileKey) $nodeIdx]
    set tki(curWindow) ""
    return $nodeRef
}

proc _tkiDpyIdle_Do { nodeSpec fileSpec w } {
    global tki

    update
    set tw $w.main.text
    tkiGetAndDisplay $nodeSpec $fileSpec $w
    $tw conf -cursor $tki(normCursor)
}

proc tkiDpyIdle { nodeSpec {fileSpec ""} {w ""} } {
    global tki
    if { "$w"!="" } {
	set tw $w.main.text
	if { "[lindex [$tw conf -cursor] 4]"=="$tki(waitCursor)" } return
	$tw conf -cursor $tki(waitCursor)
    }
    after 1 [list _tkiDpyIdle_Do $nodeSpec $fileSpec $w]
}

proc tkiInfo { nodeSpec } {
    tkiGetAndDisplay $nodeSpec
    return ""
}

#
# Do stand-alone help window
# The -node option is for compatibility to the info program only.
#
proc tkiInfoWindow { args } {
    global tki

    if { ! [info exist tki] } { tkiInit }

    set w ""
    set nodeSpec ""
    set fileSpec ""
    set fileSpec2 ""
    set dirList ""
    set opt_list { 
      { "window" w } 
      { "dir" dirList append } 
      { "file" fileSpec }
      { "infofile" fileSpec2 }
      { "node" nodeSpec append }
    }

    set args [topgetopt $opt_list $args]

    if { "$dirList"!="" } {
	set infoPaths ""
	foreach dir $dirList {
	    eval lappend infoPaths [split $dir ":"]
	}
	tkiAddInfoPaths $infoPaths
    }
    if { "$fileSpec"=="" } {
	set fileSpec $fileSpec2
    }
    if { "$fileSpec"!="" } {
	tkiAddInfoPaths [file dirname $fileSpec]
    }
    if { "$args"!="" } {
	eval lappend nodeSpec $args
    }
    if { [llength $nodeSpec] > 1 } {
	error "tkiInfoWindow: Only one node may be specified"
    }
    tkiGetAndDisplay [lindex $nodeSpec 0] $fileSpec $w
    return ""
}

#
# Start at window {w}, and traverse up the window tree looking for a variable
# of the form "$w(infonode)".  If found, a window displaying that node
# will be generated.  {fileSpec} may be used to augment the infonode,
# and {infowin} may specific a pre-existing info window returned by
# tkiGetAndDisplay().
# 
proc tkiContextHelp { w {fileSpec ""} {infowin ""} } {
    for {} {"$w"!=""} {set w [winfo parent $w]} {
	# Line below is kludgy, b/c I cant see any other way to do it.
	if [uplevel #0 [list info exist ${w}(infonode)]] {
	    upvar #0 $w wvars 
    	    return [tkiGetAndDisplay $wvars(infonode) $fileSpec $infowin]
	}
    }
    if { "$fileSpec"!="" } {
    	return [tkiGetAndDisplay Top $fileSpec $infowin]
    }
    return [tkiGetAndDisplay (builtin)QuickHelp "" $infowin]
}

#
# We are operating in one of two modes:
#   1)  Stand-alone.  Popup an initial window, filling it according to argv.
#	Kill the stupid "." window.
#   2)	Embedded within a larger application.  Don't do anything automatically;
#	instead, let that application's startup script handle things.
#
# We are operating in embedded mode iff the global tkiEmbed exists.
#
proc tkiBoot { } {
    global argv tkiEmbed

    if { [info exist tkiEmbed] } return
    wm withdraw .
    if { "[lindex $argv 0]"!="" && [file isfile [lindex $argv 0]] } {
	# Some wishs pass the filename as argv[0].  Kill it off.
	set argv [lreplace $argv 0 0]
    }
    eval tkiInfoWindow $argv
}

tkiReset
tkiBoot
