/*
 * tmMakeEvent.c --
 *	This module contains routines to create an X event
 *	used by the Tm toolkit.
 *
 * Copyright 1993 Jan Newmarch, University of Canberra.
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that the above copyright
 * notice appear in all copies.  The author
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 */

#include <sys/time.h>
#include <ctype.h>
#include <X11/keysym.h>
#include "tm.h"
#include "tmFuncs.h"

/*
 *--------------------------------------------------------------
 *
 * Tm_ExtractArg --
 *	find a name and its value in a list
 *
 * Results:
 *	returns a pointer to the value
 *
 * Side effects:
 *	none
 *--------------------------------------------------------------
 */

static char *
Tm_ExtractArg(name, argc, argv)
    char *name;
    int argc;
    char *argv[];
{
   int n;

    for (n = 0; n < argc; n++) {
	if (strcmp(name, argv[n]) == 0) {
	    if (++n < argc) {
		return argv[n];
	    } else {
		return NULL;
	    }
	}
    }
    return NULL;
}

/*
 *--------------------------------------------------------------
 *
 * Tm_MakeXEvent --
 *
 *	Create an X event that can be used to send to another window
 *
 * Results:
 *	a Tcl result value
 *
 * Side effects:
 *	returns a filled in event in xev
 *
 * Restrictions:
 *	only handles a small number of X types
 *--------------------------------------------------------------
 */
int
Tm_MakeXEvent(w, interp, xev, argc, argv)
    Widget w;
    Tcl_Interp *interp;
    XEvent *xev;
    int argc;
    char **argv;
{
    char *type;
    char *value;
    struct timeval  tp;
    Display *display;
    Position root_x, root_y;
    XRectangle rect;


    /* let them know we made this thing up */
    xev->xany.send_event = True;

    /* fill in the rest of the xany values.
     */
    xev->xany.display = display = XtDisplay(w);
    xev->xany.window = XtWindow(w);

    if ((type = Tm_ExtractArg("-type", argc, argv)) == NULL) {
        type = "ClientMessage";
    }

    if (type[0] == 'B') {
        if (strcmp(type, "ButtonRelease") == 0) {
	    xev->type = ButtonRelease;
	} else
	if (strcmp(type, "ButtonPress") == 0) {
	    xev->type = ButtonPress;
	} else {
	    sprintf(interp->result, "illegal type %s\n", type);
	    return TCL_ERROR;
	}

	xev->xbutton.time = XtLastTimestampProcessed(display);

	if ((value = Tm_ExtractArg("-x", argc, argv)) != NULL) {
	    xev->xbutton.x = atoi(value);
	} else {
	    xev->xbutton.x = 0;
	}

	if ((value = Tm_ExtractArg("-y", argc, argv)) != NULL) {
	    xev->xbutton.y = atoi(value);
	} else {
	    xev->xbutton.y = 0;
	}

 	XtTranslateCoords(XtParent(w), xev->xbutton.x,  xev->xbutton.y,
			&root_x, &root_y);
	xev->xbutton.x_root = root_x;
	xev->xbutton.y_root = root_y;

	return TCL_OK;
    }

    if (type[0] == 'K') {
	unsigned int mask;
	KeySym keysym;
	KeyCode keycode;

        if (strcmp(type, "KeyRelease") == 0) {
	    xev->type = KeyRelease;
	} else
	if (strcmp(type, "KeyPress") == 0) {
	    xev->type = KeyPress;
	} else {
	    sprintf(interp->result, "illegal type %s\n", type);
	    return TCL_ERROR;;
	}

	xev->xkey.time = XtLastTimestampProcessed(display);

	if ((value = Tm_ExtractArg("-x", argc, argv)) != NULL) {
	    xev->xkey.x = atoi(value);
	} else {
	    xev->xkey.x = 0;
	}

	if ((value = Tm_ExtractArg("-y", argc, argv)) != NULL) {
	    xev->xkey.y = atoi(value);
	} else {
	    xev->xkey.y = 0;
	}

	if ((value = Tm_ExtractArg("-keysym", argc, argv)) != NULL) {
	    if ((keysym = XStringToKeysym(value)) == NoSymbol) {
		sprintf(interp->result, "illegal keysym\"%s\"", value);
		return TCL_ERROR;
	    }
	    if ((keycode = XKeysymToKeycode(display, keysym)) == 0) {
		sprintf(interp->result, "no keycode for keysym \"%s\"", value);
		return TCL_ERROR;
	    }

	    mask = 0;
	    /* is this what we should do? */
	    if (keysym >= XK_A && keysym <= XK_Z)
		mask |= ShiftMask;
	    
	    xev->xkey.keycode = keycode;
	    xev->xkey.state = mask;
	} else {
	    sprintf(interp->result, "Key event must have a keysym");
	    return TCL_ERROR;
	}

	return TCL_OK;
    }

    if (strcmp(type, "ClientMessage") == 0) {
	return TCL_OK;
    }

    sprintf(interp->result, "illegal type %s\n", type);
    return TCL_ERROR;;
}

