/*
 * Copyright (c) 1990, 1991 Stanford University
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the name
 * Stanford may not be used in any advertising or publicity relating to
 * the software without the specific, prior written permission of
 * Stanford.
 *
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * IN NO EVENT SHALL STANFORD BE LIABLE FOR ANY SPECIAL, INCIDENTAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT
 * ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF LIABILITY,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */
/*
 * Copyright 1992 by the National Optical Astronomy Observatories(*)
 *
 * Permission to use, copy, and distribute
 * is hereby granted without fee, providing that the above copyright
 * notice appear in all copies and that both the copyright notice and this
 * permission notice appear in supporting documentation.
 *
 * This software is provided "as is" without any express or implied warranty.
 *
 * (*) Operated by the Association of Universities for Research in
 *     Astronomy, Inc. (AURA) under cooperative agreement with the
 *     National Science Foundation.
 */
/* Program: dispatch.c
 *      This file contains rpc send and receive commands and the
 *      dispatch table that lets users override the system functions.
 *      Some of this is based on George Drapeau's MAEstro Receiver.c.
 *
 * Created: K. Gillies 26 June 1992
 * SCCS INFO
 *      @(#)dispatch.c	1.1 9/11/92
 *
 */
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/socket.h>
#include "tcl.h"
#include "tclIPC.h"            /* RPC stuff */
#include "tclipcP.h"           /* Private structures */
#include "tclipc.h"            /* Public interface */

/* This is required by the new RPCgen.  It may not be needed in the future */
int _rpcsvcdirty; 	       /* Still serving ? */

/* This is the size of descriptor table for later use by select().
 * It's been removed since it's an expensive system call.
 */
extern int ipcDescriptorTableSize;

/* Used to tell select() how long to wait until it returns */
/* This can be set by the user */
struct timeval waitTime, *waitTimePtr = &waitTime;

/* In file prototypes */
static evalcmd_res *tclCmdReceive _ANSI_ARGS_((Tcl_Interp*, cmdText));
static void ReceiverListenForMessages _ANSI_ARGS_((void));
static void checkIncomingConnections _ANSI_ARGS_((int newfd));
static void destroyAConnection _ANSI_ARGS_((int dyingfd));

/* The default dispatch table */
static DispatchTable ReceiverDispatchTable = 
{
  tclCmdReceive,
  ReceiverListenForMessages,
  checkIncomingConnections,
  destroyAConnection,
};

/*
 *--------------------------------------------------------------
 *
 * BuildDispatchTable
 *
 *      Called to set the routines that the library uses to receive
 *      messages and to check for incoming messages.
 *
 * Results:
 *      void
 *--------------------------------------------------------------
 */
void BuildDispatchTable(table)
DispatchTable* table;
{
  if (table == NULL)
    return;
  /* Set the receive function */
  if (table->cmdRecvPtr)
    ReceiverDispatchTable.cmdRecvPtr = table->cmdRecvPtr;
  /* Set the listen function */
  if (table->listenFunctionPtr)
    ReceiverDispatchTable.listenFunctionPtr = table->listenFunctionPtr;
  /* Set the connect function */
  if (table->connectFunctionPtr)
    ReceiverDispatchTable.connectFunctionPtr = table->connectFunctionPtr;
  /* Set the die function */
  if (table->destroyConnectFunctionPtr)
    ReceiverDispatchTable.destroyConnectFunctionPtr = 
      table->destroyConnectFunctionPtr;
  return;
}

/*
 *--------------------------------------------------------------
 *
 * tclCmdReceive
 *
 *      This is the "STOCK" message receiver.  The state of the
 *      target interpreter is saved (as in Tk send).  The correct
 *      interpreter is selected at a higher level.  The user
 *      can replace this routine.
 *
 * Results:
 *      the error and the result string
 *--------------------------------------------------------------
 */
static evalcmd_res* tclCmdReceive(interp, cmd)
Tcl_Interp *interp;
cmdText cmd;
{
  static evalcmd_res result = { 0, NULL };
  char *oldResultString;
  Tcl_FreeProc *oldFreeProc;

  /* Since I'm using the interp->result, I don't free it */
#ifdef DEBUG
  fprintf(stderr, "STOCK receiver\n");
#endif
  /* Save and restore the current interpreter state.. like tkSend */
  oldResultString = interp->result;
  oldFreeProc = interp->freeProc;
  interp->freeProc = 0;
  result.errno = Tcl_GlobalEval(interp, cmd);
  result.text = interp->result;
  interp->result = oldResultString;
  interp->freeProc = oldFreeProc;

  return (&result);
}

/*
 *--------------------------------------------------------------
 *
 * ipcCheckForIncomingConnections
 *
 *      If not interactive, call the wait function directly.
 *
 * Results:
 *
 *--------------------------------------------------------------
 */
void ipcCheckIncomingConnections(newfd)
int newfd;
{
  ReceiverDispatchTable.connectFunctionPtr(newfd);
}

/*
 *--------------------------------------------------------------
 *
 * checkForIncomingConnections
 *
 *      The default monitor connections function.
 *
 * Results:
 *
 *--------------------------------------------------------------
 */
static void checkIncomingConnections(newfd)
int newfd;
{
#ifdef DEBUG
  fprintf(stderr, "New connection: %d\n", newfd);
#endif
}

/*
 *--------------------------------------------------------------
 *
 * ipcDestroyConnections
 *
 *      Give the User a chance to remove connections.
 *
 * Results:
 *
 *--------------------------------------------------------------
 */
void ipcDestroyConnections(dyingfd)
int dyingfd;
{
  ReceiverDispatchTable.destroyConnectFunctionPtr(dyingfd);
}

/*
 *--------------------------------------------------------------
 *
 * destroyAConnection
 *
 *      The default destroy connections function.
 *
 * Results:
 *
 *--------------------------------------------------------------
 */
static void destroyAConnection(dyingfd)
int dyingfd;
{
  /* Default does nothing */
#ifdef DEBUG
  fprintf(stderr, "destroy connection: %d\n", dyingfd);
#endif
}

/*
 *--------------------------------------------------------------
 *
 * ipcWaitForMessages
 *
 *      If not interactive, call the wait function directly.
 *
 * Results:
 *
 *--------------------------------------------------------------
 */
void ipcWaitForMessages()
{
  ReceiverDispatchTable.listenFunctionPtr();
}

/*
 *--------------------------------------------------------------
 *
 * setReceiveTimeout
 *
 *      If not interactive, call the wait function directly.
 *
 * Results:
 *
 *--------------------------------------------------------------
 */
void setReceiveTimeout(secs, usecs)
long secs;
long usecs;
{
  if (secs == WAITFOREVER || usecs == WAITFOREVER) {
    waitTimePtr = (struct timeval *)NULL;
  } else {
    /* Set up a timeval to wait for 1.0 seconds */
    waitTimePtr->tv_sec = secs;
    waitTimePtr->tv_usec = usecs;
  }
}

/*
 *--------------------------------------------------------------
 *
 * ipcListenOnceOnSocket
 *
 * Use this routine to check for incoming messages and service
 * them for one file descriptor.

 * Results:
 *      void
 *--------------------------------------------------------------
 */
void ipcListenOnceOnSocket(fd)
int fd;
{
  fd_set rset;

  FD_ZERO(&rset);
  FD_SET(fd, &rset);
  /* Now look at just the one connection */
  svc_getreqset(&rset);	    /* Handle the incoming RPC requests */
}

/*
 *--------------------------------------------------------------
 *
 * ReceiverListenForMessages
 *
 *      This is the default routine that is used to check for incoming
 *      RPC messages.  It is meant to be used by application
 *	programmers who do not have a toolkit that does automatic
 *	RPC dispatching (Sun's XView toolkit does RPC dispatching
 *	automatically).  If the programmer's toolkit does not support
 *	RPC dispatching, the programmer should tell the toolkit to call
 *	this function every so often.
 *
 * Results:
 *      Returns -1 if no app exists or 0 if ok.
 *--------------------------------------------------------------
 */
static void ReceiverListenForMessages()
{
  /* Variable holding list of RPC sockets to listen... */
  /* ...on.  Used as an argument to select() to determine... */
  /* ...whether there are requests on any RPC socket ... */
  /* ...waiting to be processed. */
  fd_set rpcSockets;

  /* Return value from select, tells how many files had... */
  /* ...input ready to be processed. */
  int numSockets = 0;
  
  /* Get the list of RPC sockets accepting requests */
  rpcSockets = svc_fdset;

  /* Check for input on RPC sockets */
 
  numSockets = select(ipcDescriptorTableSize,
		      &rpcSockets, (int*)0,(int*)0,
		      waitTimePtr);
  /* Based on how many sockets had input, determine what to do */
  switch (numSockets) {
  case -1:			    /* An error occurred */
    break;			    /* For now, don't do anything */
    
  case 0:			    /* There was no input to process */
    break;
    
  default:			    /* There was input to process */
    svc_getreqset(&rpcSockets);	    /* Handle the incoming RPC requests */
    break;
  }
  return;
}

/*
 *--------------------------------------------------------------
 *
 * receiverAlarmTimeout
 *
 *      The non-window programs receive by polling using the
 *      alarm signal.  This is the signal handler.  Called from
 *      tclipc.c.
 *
 * Results:
 *      
 *--------------------------------------------------------------
 */
void receiverAlarmTimeout(sig)
int sig;
{
  int oldmask;
  
  /* Make sure no more alarms happend during an alarm response */
  oldmask = sigblock(sigmask(SIGALRM));

  ReceiverDispatchTable.listenFunctionPtr();

  sigsetmask(oldmask);
}

/*
 *--------------------------------------------------------------
 *
 * cmdsend_4_svc
 *
 *      This is the first handler of incoming messages.  It is called
 *      from the RPC server code.  This code finds the correct
 *      target interpreter and calls the user routine.
 *      If the cached target port is not correct, the handle table
 *      is searched for a match.
 *
 * Results:
 *      Result of user function call.
 *--------------------------------------------------------------
 */
evalcmd_res *cmdsend_4_svc(argp, rqstp)
cmdType *argp;
struct svc_req *rqstp;
{
  /* These are the last port and interp cached values */
  static int lastPort = -1;     
  static Tcl_Interp *lastInterp = NULL;

  if (argp->destinationPort != lastPort) {   
    /* Search handle table if cached value is not correct */
    ipcGetInterp(argp->destinationPort, &lastInterp);
    lastPort = argp->destinationPort;
  }
#ifdef DEBUG
  printf("Send to port: %d\n", argp->destinationPort);
#endif
  if (lastInterp == NULL) {
    assert(lastInterp != NULL);
    fprintf(stderr, "!!!!-Could not find an interpreter-!!!!\n");
    /* Don't exit here, maybe return this message? */
    return NULL;
  }
  return ReceiverDispatchTable.cmdRecvPtr(lastInterp, argp->cmd);
}
