/*
 * Copyright 1992 by the National Optical Astronomy Observatories(*)
 *
 * Permission to use, copy, and distribute
 * is hereby granted without fee, providing that the above copyright
 * notice appear in all copies and that both the copyright notice and this
 * permission notice appear in supporting documentation.
 *
 * This software is provided "as is" without any express or implied warranty.
 *
 * (*) Operated by the Association of Universities for Research in
 *     Astronomy, Inc. (AURA) under cooperative agreement with the
 *     National Science Foundation.
 */
/* Program: plist.c
 *      This file is contains wrappers for the collection file
 *      that have been specialized for the PortManager needs.
 *
 * Created: K. Gillies 26 June 1992
 * SCCS INFO
 *      @(#)plist.c	1.1 9/11/92
 */
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <sys/types.h>
#include <sys/time.h>
#include "tclExtend.h"
#include "tclipc.h" 
#include "plist.h"

/* Private function declarations */
static Port *pListNewPortEntry _ANSI_ARGS_((Port *oldPort));

/*
 *--------------------------------------------------------------
 *
 * pListInit
 *
 *      Allocate a new list and return a pointer to it.
 *
 * Results:
 *      Argument newlist is set to the newly allocated list.
 */
void pListInit(newlist)
PortManList *newlist;
{
  Collection temp;

  *newlist = (CollectionCreate(&temp) != NOERROR) ? NULL : temp;
}

/*
 *--------------------------------------------------------------
 *
 * pListPrint
 *
 *      A diagnostic routine to dump the contents of the
 *      PortManager receivers list.
 *
 * Results:
 *      void
 */
void pListPrint(ports_inuse)
PortManList ports_inuse;
{
  generic_ptr member;
  Port *port;

  if (CollectionEmpty(ports_inuse)) {
    printf("No applications are currently registered.\n");
    return;
  }

  printf("List of currenly open applications:\n");
  CollectionForEach(ports_inuse, member) {
    port = (Port *)member;
    printf("App Name: %s, Host Name: %s, Port Number: %d.\n",
	   port->appName,
	   port->hostName,
	   port->portNumber);
  } CollectionNext();
}

/*
 *--------------------------------------------------------------
 *
 * pListPortInList
 *
 *      Check the PortManList for an entry that matches the
 *      name, host, and port that are passed to the function.
 *      This function is not finished and is currently not used.
 *
 * Results:
 *      A pointer to a matching Port. 
 */
Port *pListPortInList(ports_inuse, name, host, port)
PortManList ports_inuse;
char *name;
char *host;
int port;
{  
  generic_ptr member;
  Port *info;

  /* Check to see if server by name is in list already */
  CollectionForEach(ports_inuse, member) {
    info = (Port *)member;
    if (!strcmp(name, info->appName))
      return info;
  } CollectionNext();
  return NULL;
}

/*
 *--------------------------------------------------------------
 *
 * pListNewPortEntry
 *
 *      Allocate a new Port structure for the PortManager list and
 *      copy the contents of the Port that is passed as argument oldPort.
 *
 * Results:
 *      A pointer to the new Port. 
 */
static Port *pListNewPortEntry(oldPort)
Port *oldPort;
{
  Port *newPort;

  newPort = (Port *)ckalloc(sizeof(Port));
  assert(newPort != NULL);

  /* Setup the arguments */
  newPort->appName = strdup(oldPort->appName);
  assert(newPort->appName != NULL);

  newPort->hostName = strdup(oldPort->hostName);
  assert(newPort->hostName != NULL);

  newPort->portNumber = oldPort->portNumber;

  return newPort;
}    

/*
 *--------------------------------------------------------------
 *
 * pListAddPortToList
 *
 *      Add a new entry to the PortManager list and make its
 *      contents a copy of the port argument.
 *
 * Results:
 *      void
 */
void pListAddPortToList(ports_inuse, port)
PortManList ports_inuse;
Port *port;
{
  Port *newport;

  newport = pListNewPortEntry(port);
  /* Place it at the end of the open list */
  (void)(CollectionAppend(ports_inuse, (generic_ptr)newport));
}

/*
 *--------------------------------------------------------------
 *
 * pListRemovePortFromList
 *
 *      Find a Port in the PortManager list that matches the
 *      contents of portToRemove and remove it from the list.
 *
 * Results:
 *      void
 */
void pListRemovePortFromList(ports_inuse, portToRemove)
PortManList ports_inuse;
Port *portToRemove;
{
  Collection member;
  Port *port;

  /* Go through the list and free portToRemove if present */
  CollectionForEachNode(ports_inuse, member) {
    port = (Port *)member->info;
    if ((port->portNumber == portToRemove->portNumber) &&
	(strcmp(port->appName, portToRemove->appName) == 0) &&
	(strcmp(port->hostName, portToRemove->hostName) == 0)) {
#ifdef DEBUG 
      printf("Deleting app info: %s %s %d\n", 
	      port->appName,
	      port->hostName,
	      port->portNumber);
#endif 
      assert (CollectionDelete(ports_inuse, member) != ERROR);

      /* Free the info */
      free(port->appName);
      free(port->hostName);
      free((void *)port); 
      /* Found one so leave */
      break;
    }
  } CollectionNext();
}

/*
 *--------------------------------------------------------------
 *
 * pListScanForDeadApps
 *
 *      This function scans through a list of Ports, looking for
 *      applications that are no longer listening on their
 *      Receiver's for whatever reason (probably because the app
 *      is no longer running).  The function goes through the
 *      list passed in as argument, and tries to check each item on
 *      the list.  If each item on the list is still
 *      listening for messages, then this function will scan the whole
 *      list.  But, if one app on the list doesn't respond,
 *      this function will remove its entry from the port list, then
 *      this function will return.  So this function removes
 *      at most one application from the portList, assuming that this
 *      function will be called repeatedly.
 *      In addition, not too much time should be spent in this function,
 *      since it takes away from time this app should
 *      be listening for messages.  So, when appropriate, this function
 *      calculates the time it has been executing; when
 *      it has been executing longer than a specified time ("maxTimePassed"),
 *      the function returns.
 *
 * Results:
 *      void
 */
void pListScanForDeadApps(ports_inuse)
PortManList ports_inuse;
{
  int result;
  /* Max number of seconds to be spent in this func. */
  int	maxTimePassed = 1;
  /* Used to calculate time has spent in this func. */
  struct timeval startTime, now;
  generic_ptr member;
  Port *thisPort;

  /* If the port list is empty, return */
  if (CollectionEmpty(ports_inuse))
    return;

  result = gettimeofday(&now,(struct timezone*)NULL);
  if (result != 0) {	
    fprintf(stderr, "Could not get the time of day.  Exiting.\n");
    exit(result);
  }
  /* Initialize time this function started */
  startTime.tv_sec = now.tv_sec;
  startTime.tv_usec = now.tv_usec;

  /* Go through the list and free portToRemove if present */
  CollectionForEach(ports_inuse, member) {
    thisPort = (Port *)member;

    /* Create a new Sender to communicate with this application */
    if (thisPort->portNumber != PortMgrPortNumber) {
      result = SenderAppCheck(thisPort);
      /* If the creation failed, assume the application is... gone.*/
      if (result != TCL_OK) {
	/* Remove this port from the list of active ports */
	pListRemovePortFromList(ports_inuse, thisPort);
	return;
      }
#ifdef DEBUG
      fprintf(stderr, "Checked: %d\n", thisPort->portNumber);
#endif
      /* Get current time for later comparison with startTime */
      result = gettimeofday(&now, (struct timezone*)NULL);
      /* Was there a problem getting the time of day? */
      /* Yes, print an error and exit the program */
      if (result != 0) {
	fprintf(stderr, "Could not get the time of day.  Exiting.\n");
	exit(result);
      }
      /* Has the maximum allowed time elapsed in this function?
       * If yes, leave. */
      if ((now.tv_sec - startTime.tv_sec) >= maxTimePassed)
	return;
    }
  } CollectionNext();
}
