/*-
 * Copyright (c) 1993, 1994 Michael B. Durian.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by Michael B. Durian.
 * 4. The name of the the Author may be used to endorse or promote 
 *    products derived from this software without specific prior written 
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
#include <assert.h>
#include <stdlib.h>
#include "EventTree.h"
#include "Note.h"
#include "MetaEOT.h"

EventTree::EventTree() : curr_node(0), curr_event(0)
{

	head = make_rb();
	assert(head != 0);
}

EventTree::EventTree(const EventTree &t)
{

	CopyTree(t);
}

EventTree::~EventTree()
{

	DeleteTree();
}

/*
 * returns node for time, or node with smallest time greter than time
 * returns 0 if there are no nodes with time >= specified time
 */
Event *
EventTree::GetEvents(unsigned long time)
{
	EventTreeNode *n;

	n = rb_find_ukey(head, time);
	if (n == head)
		return (0);

	curr_node = n;
	curr_event = 0;
	return ((Event *)n->v.val);
}

Event *
EventTree::NextEvent(void)
{

	if (curr_node == 0) {
		// at very beginning
		if ((curr_node = head->c.list.flink) == 0)
			return (0);
		curr_event = (Event *)curr_node->v.val;
	} else if (curr_event != 0 && curr_event->GetNextEvent() != 0) {
		curr_event = curr_event->GetNextEvent();
	} else {
		// move to next node
		curr_node = rb_next(curr_node);
		if (curr_node != head->c.list.flink && curr_node != head)
			curr_event = (Event *)curr_node->v.val;
		else {
			// reached end
			curr_node = 0;
			curr_event = 0;
		}
	}
	return (curr_event);
}

Event *
EventTree::NextEvent(const Event *event)
{
	Event *e, *next;
	EventTreeNode *place;
	unsigned long time;

	// no-brainer if it isn't the last event in a node
	if ((e = event->GetNextEvent()) != 0)
		return (e);

	// otherwise we need to find it's node
	time = event->GetTime();
	/* find where it is or where it should go */
	place = rb_find_ukey(head, time);
	assert(place != 0);
	e = (Event *)place->v.val;
	if (e == 0 || e->GetTime() != time)
		return (0);

	// place is the correct node - find the event
	for (; e != 0 && !(*e == *event); e = e->GetNextEvent());
	if (e == 0) {
		// didn't find event
		return (0);
	}

	if ((next = e->GetNextEvent()) == 0) {
		// next event is in next node
		place = rb_next(place);
		if (place == head->c.list.flink && place != head)
			return (0);
		next = (Event *)place->v.val;
	}
	return (next);
}

Event *
EventTree::NextEvents(void)
{

	if (curr_node == 0) {
		// at very beginning
		if ((curr_node = head->c.list.flink) == 0)
			return (0);
		// so we know to start at node beginning
		curr_event = 0;
		return ((Event *)curr_node->v.val);
	} else {
		// move to next node
		curr_node = rb_next(curr_node);
		if (curr_node != head->c.list.flink && curr_node != head) {
			curr_event = 0;
			return ((Event *)curr_node->v.val);
		} else {
			// reached end
			curr_node = 0;
			curr_event = 0;
			return (0);
		}
	}
}

Event *
EventTree::NextEvents(const Event *event)
{
	Event *e;
	EventTreeNode *place, *next_node;
	unsigned long time;

	if (event == 0) {
		if (head->c.list.flink == head)
			return (0);
		else
			return ((Event *)head->c.list.flink->v.val);
	}

	time = event->GetTime();
	/* find where it is or where it should go */
	place = rb_find_ukey(head, time);
	assert(place != 0);
	e = (Event *)place->v.val;
	if (e == 0 || e->GetTime() != time)
		return (0);

	// place is the correct node - move to next
	next_node = rb_next(place);
	if (next_node == head->c.list.flink || next_node == head)
		return (0);

	return ((Event *)next_node->v.val);
}

Event *
EventTree::PrevEvent(void)
{

	if (curr_node == 0) {
		curr_node = head->c.list.blink;
		for (curr_event = (Event *)curr_node->v.val;
		    curr_event->GetNextEvent() != 0;
		    curr_event = curr_event->GetNextEvent());
	} else if (curr_event != 0 && curr_event->GetPrevEvent() != 0) {
		curr_event = curr_event->GetPrevEvent();
	} else {
		curr_node = rb_prev(curr_node);
		if (curr_node == head || curr_node == head->c.list.blink) {
			// we wrapped
			curr_node = 0;
			curr_event = 0;
		} else {
			// find last event in new node
			for (curr_event = (Event *)curr_node->v.val;
			    curr_event->GetNextEvent() != 0;
			    curr_event = curr_event->GetNextEvent());
		}
	}
	return (curr_event);
}

Event *
EventTree::PrevEvent(const Event *event)
{
	Event *e, *prev;
	EventTreeNode *place;
	unsigned long time;

	// a no-brainer if it's not the first event
	if ((e = event->GetPrevEvent()) != 0)
		return (e);

	// we're going to need to find the node to back up
	time = event->GetTime();
	// find where it is
	place = rb_find_ukey(head, time);
	assert(place != 0);
	e = (Event *)place->v.val;
	if (e == 0 || e->GetTime() != time)
		return (0);

	// we know that place is at the correct time
	// now find the event
	for (e = (Event *)place->v.val; e != 0 && !(*e == *event);
	    e = e->GetNextEvent());
	if (e == 0) {
		// didn't find the event
		return (0);
	}

	// we now have the correct event
	if ((prev = e->GetPrevEvent()) == 0) {
		// back up a node
		place = rb_prev(place);
		if (place == head || place == head->c.list.blink) {
			// we wrapped
			return (0);
		}
		// find last event in new node
		for (prev = (Event *)place->v.val; prev->GetNextEvent() != 0;
		    prev = prev->GetNextEvent());
	}
	return (prev);
}

Event *
EventTree::PrevEvents(void)
{

	if (curr_node == 0) {
		curr_node = head->c.list.blink;
		curr_event = 0;
		return ((Event *)curr_node->v.val);
	} else {
		curr_node = rb_prev(curr_node);
		if (curr_node == head || curr_node == head->c.list.blink) {
			// we wrapped
			curr_node = 0;
			curr_event = 0;
			return (0);
		} else {
			curr_node = 0;
			return ((Event *)curr_node->v.val);
		}
	}
}

Event *
EventTree::PrevEvents(const Event *event)
{
	Event *e;
	EventTreeNode *place, *prev_node;
	unsigned long time;

	if (event == 0) {
		if (head->c.list.blink == head)
			return (0);
		else
			return ((Event *)head->c.list.blink->v.val);
	}

	time = event->GetTime();
	/* find where it is or where it should go */
	place = rb_find_ukey(head, time);
	assert(place != 0);
	e = (Event *)place->v.val;
	if (e == 0 || e->GetTime() != time)
		return (0);

	// place is the correct node - move to prev
	prev_node = rb_prev(place);
	if (prev_node == head->c.list.blink || prev_node == head)
		return (0);

	return ((Event *)prev_node->v.val);
}

Event *
EventTree::GetFirstEvent(void)
{

	if ((curr_node = head->c.list.flink) == 0 || curr_node == head) {
		curr_node = 0;
		curr_event = 0;
	} else
		curr_event = (Event *)curr_node->v.val;
	return (curr_event);
}

Event *
EventTree::GetFirstEvents(void)
{

	curr_event = 0;
	if ((curr_node = head->c.list.flink) == 0 || curr_node == head) {
		curr_node = 0;
		return (0);
	} else {
		return ((Event *)curr_node->v.val);
	}
}

Event *
EventTree::GetLastEvent(void)
{

	if ((curr_node = head->c.list.blink) == 0 || curr_node == head) {
		curr_node = 0;
		curr_event = 0;
	} else
		for (curr_event = (Event *)curr_node->v.val;
		    curr_event->GetNextEvent() != 0;
		    curr_event = curr_event->GetNextEvent());
	return (curr_event);
}

Event *
EventTree::GetLastEvents(void)
{

	curr_event = 0;
	if ((curr_node = head->c.list.blink) == 0 || curr_node == head) {
		curr_node = 0;
		return (0);
	} else {
		return ((Event *)curr_node->v.val);
	}
}

EventTreeNode *
EventTree::GetFirstNode(void)
{

	if (head->c.list.flink == 0 || head->c.list.flink == head)
		return (0);
	else
		return (head->c.list.flink);
}

EventTreeNode *
EventTree::GetLastNode(void)
{

	if (head->c.list.blink == 0 || head->c.list.blink == head)
		return (0);
	else
		return (head->c.list.blink);
}

EventTreeNode *
EventTree::NextNode(const EventTreeNode *n)
{

	if (n == 0 || n->c.list.flink == 0 || n->c.list.flink == head)
		return (0);
	else
		return (n->c.list.flink);
}

EventTreeNode *
EventTree::PrevNode(const EventTreeNode *n)
{

	if (n == 0 || n->c.list.blink == 0 || n->c.list.blink == head)
		return (0);
	else
		return (n->c.list.blink);
}

unsigned long
EventTree::GetStartTime(void)
{

	if (head->c.list.flink == 0 || head->c.list.flink == head)
		return (0);
	return (head->c.list.flink->k.ukey);
}

unsigned long
EventTree::GetEndTime(void)
{

	if (head->c.list.blink == 0 || head->c.list.blink == head)
		return (0);
	return (head->c.list.blink->k.ukey);
}

int
EventTree::Add(const EventTree &event_tree, unsigned long start, double scalar)
{
	EventType etype;
	EventTree *et = (EventTree *)&event_tree;
	Event *e, *new_e, *mod_e, *off, *new_off;

	for (e = et->GetFirstEvent(); e != 0; e = et->NextEvent(e)) {
		etype = e->GetType();
		// skip not off side of note pairs since they've
		// already been inserted
		if ((etype == NOTEOFF || (etype == NOTEON &&
		    ((NoteEvent *)e)->GetVelocity() == 0)) &&
		    ((NoteEvent *)e)->GetNotePair() != 0)
			continue;
		if ((mod_e = e->Dup()) == 0)
			return (0);
		mod_e->SetTime((long)(e->GetTime() * scalar) + start);
		new_e = PutEvent(*mod_e);
		delete mod_e;
		if (new_e == 0)
			return (0);
		if (etype == NOTEON && (off = ((NoteEvent *)e)->GetNotePair())
		    != 0) {
			if ((mod_e = off->Dup()) == 0)
				return (0);
			mod_e->SetTime((long)(off->GetTime() * scalar) + start);
			new_off = PutEvent(*mod_e);
			delete mod_e;
			if (new_off == 0)
				return (0);
			((NoteEvent *)new_e)->SetNotePair((NoteEvent *)new_off);
			((NoteEvent *)new_off)->SetNotePair((NoteEvent *)new_e);
		}
	}
	return (1);
}

EventTree *
EventTree::GetRange(unsigned long start, unsigned long end) const
{
	EventType etype;
	EventTree *nt, *this_ptr;
	Event *e;
	NoteEvent *np, *new_e1, *new_e2;

	if (start >= end)
		return (0);
	this_ptr = (EventTree *)this;
	nt = new EventTree;
	if (nt == 0)
		return (0);
	e = this_ptr->GetEvents(start);
	for (e = this_ptr->GetEvents(start); e != 0 && e->GetTime() < end;
	    e = this_ptr->NextEvent(e)) {
		etype = e->GetType();
		// skip note off halfs
		if ((etype == NOTEOFF || (etype == NOTEON &&
		    ((NoteEvent *)e)->GetVelocity() == 0)) &&
		    (np = ((NoteEvent *)e)->GetNotePair()) != 0) {
			continue;
		} else if (etype == NOTEON && ((NoteEvent *)e)->GetVelocity()
		    != 0 && (np = ((NoteEvent *)e)->GetNotePair()) != 0) {
			// copy in note off half and link
			if ((new_e1 = (NoteEvent *)nt->PutEvent(*e)) == 0)
				return (0);
			if ((new_e2 = (NoteEvent *)nt->PutEvent (*np)) == 0)
				return (0);
			new_e1->SetNotePair(new_e2);
			new_e2->SetNotePair(new_e1);
		} else {
			if (nt->PutEvent(*e) == 0)
				return (0);
		}

	}
	return (nt);
}

int
EventTree::DeleteRange(unsigned long start, unsigned long end)
{
	EventType etype;
	Event *e, *next, *np;

	if (start >= end)
		return (0);

	e = GetEvents(start);
	while (e != 0 && e->GetTime() < end) {
		etype = e->GetType();
		if ((etype == NOTEOFF || (etype == NOTEON &&
		    ((NoteEvent *)e)->GetVelocity() == 0)) &&
		    (np = ((NoteEvent *)e)->GetNotePair()) != 0 &&
		    np->GetTime() < start) {
			// don't remove noteoff halfs when note
			// starts before range
			e = NextEvent(e);
			continue;
		} else if (etype == NOTEON && ((NoteEvent *)e)->GetVelocity()
		    != 0 && (np = ((NoteEvent *)e)->GetNotePair()) != 0
		    && np->GetTime() >= end) {
			// remove note off even though it falls outside
			// of range
			if (!DeleteEvent(*np))
				return (0);
		}
		next = NextEvent(e);
		if (!DeleteEvent(*e))
			return (0);
		e = next;
	}
	return (1);
}

Event *
EventTree::PutEvent(const Event &event)
{
	unsigned long time;
	EventTreeNode *place, *new_node;
	Event *e, *new_event, *first, *next, *prev, *last;

	time = event.GetTime();
	/* find where it is or where it should go */
	place = rb_find_ukey(head, time);
	assert(place != 0);
	e = (Event *)place->v.val;
	if (e != 0 && e->GetTime() == time) {
		/* find end of event list or duplicate */
		first = e;
		for (; e->GetNextEvent() != 0 && !(*e == event);
		    e = e->GetNextEvent());
		/* don't insert duplicate events */
		if (*e == event)
			return (0);
		last = e;
		new_event = event.Dup();
		assert(new_event != 0);
		switch (new_event->GetType()) {
		case NOTEOFF:
			/* at beginning, but after other existing NoteOffs */
			for (e = first; e != 0 && e->GetType() == NOTEOFF;
			    e = e->GetNextEvent());
			if (e == 0) {
				new_event->SetNextEvent(0);
				new_event->SetPrevEvent(last);
				last->SetNextEvent(new_event);
			} else {
				prev = e->GetPrevEvent();
				new_event->SetNextEvent(e);
				new_event->SetPrevEvent(prev);
				e->SetPrevEvent(new_event);
				if (prev == 0)
					place->v.val = (char *)new_event;
				else
					prev->SetNextEvent(new_event);
			}
			break;
		case NOTEON:
			/* at end, but before EOT if there is one */
			if (last->GetType() == METAENDOFTRACK) {
				prev = last->GetPrevEvent();
				new_event->SetNextEvent(last);
				new_event->SetPrevEvent(prev);
				last->SetPrevEvent(new_event);
				if (prev == 0)
					place->v.val = (char *)new_event;
				else
					prev->SetNextEvent(new_event);
			} else {
				new_event->SetNextEvent(0);
				new_event->SetPrevEvent(last);
				last->SetNextEvent(new_event);
				break;
			}
			break;
		case METAENDOFTRACK:
			/* always at very end */
			new_event->SetNextEvent(0);
			new_event->SetPrevEvent(last);
			last->SetNextEvent(new_event);
			break;
		default:
			/* all others go in middle */
			/* search from rear to preserve order */
			for (e = last; e != 0 &&
			    (e->GetType() == METAENDOFTRACK ||
			    e->GetType() == NOTEON); e = e->GetPrevEvent());
			if (e == 0) {
				new_event->SetPrevEvent(0);
				new_event->SetNextEvent(first);
				place->v.val = (char *)new_event;
				first->SetPrevEvent(new_event);
			} else {
				next = e->GetNextEvent();
				new_event->SetPrevEvent(e);
				new_event->SetNextEvent(next);
				e->SetNextEvent(new_event);
				if (next != 0)
					next->SetPrevEvent(new_event);
			}
			break;
		}
		curr_node = place;
	} else {
		new_event = event.Dup();
		assert(new_event != 0);
		new_node = rb_insert_b(place, (char *)time, (char *)new_event);
		curr_node = new_node;
	}
	curr_event = new_event;
	return (new_event);
}

void
EventTree::RewindEvents(void)
{

	curr_node = 0;
	curr_event = 0;
}

int
EventTree::DeleteEvent(const Event &event)
{
	Event *e, *events, *prev;
	EventTreeNode *n;

	n = rb_find_ukey(head, event.GetTime());
	if (n == 0)
		return (0);
	events = (Event *)n->v.val;
	if (events == 0 || events->GetTime() != event.GetTime())
		return (0);
	/* find matching event */
	for (e = events, prev = 0; e != 0 && !(*e == event);
	    prev = e, e = e->GetNextEvent());
	if (e == 0)
		return (0);
	if (prev != 0 || e->GetNextEvent() != 0) {
		if (e->GetNextEvent() != 0)
			e->GetNextEvent()->SetPrevEvent(prev);

		if (prev != 0)
			prev->SetNextEvent(e->GetNextEvent());
		else
			n->v.val = (char *)e->GetNextEvent();
		// check to see if curr_event needs to move
		if (curr_event == e) {
			if ((curr_event = e->GetNextEvent()) == 0) {
				curr_node = rb_next(n);
				if (curr_node != head->c.list.flink &&
				    curr_node != head)
					curr_event = (Event *)curr_node->v.val;
				else {
					curr_node = 0;
					curr_event = 0;
				}
			}
			
		}
	} else {
		if (curr_node == n) {
			if ((curr_node = rb_next(n)) != head->c.list.flink
			    && curr_node != head)
				curr_event = (Event *)curr_node->v.val;
			else {
				curr_node = 0;
				curr_event = 0;
			}
		}
		rb_delete_node(n);
	}
	delete e;
	return (1);
}

EventTree &
EventTree::operator=(const EventTree &t)
{

	CopyTree(t);
	return (*this);
}

/* private functions */
void
EventTree::DeleteTree(void)
{
	Event *e;

	RewindEvents();
	/* get each event in order and delete it */
	while ((e = NextEvent()) != 0)
		DeleteEvent(*e);
		
	/* then free head */
	free(head);
	head = 0;
	curr_node = 0;
	curr_event = 0;
}

void
EventTree::CopyTree(const EventTree &t)
{
	Event *e, *events, *new_e;
	EventTreeNode *next;

	/* we must clear out the current tree */
	DeleteTree();

	head = make_rb();
	if (t.head->c.list.flink == 0)
		return;
	next = t.head->c.list.flink;
	// default curr_event and curr_node to 0
	curr_event = 0;
	curr_node = 0;
	// get each event in order and insert them into new tree
	for(;;) {
		events = (Event *)next->v.val;
		for (e = events; e != 0; e = e->GetNextEvent()) {
			new_e = PutEvent(*e);
			assert(new_e != 0);
			if (t.curr_event == e)
				curr_event = new_e;
			if (t.curr_node == next) {
				/*
				 * since we go in order, the current
				 * node is the last in the linked list
				 */
				curr_node = head->c.list.blink;
			}
		}
		if (next == t.head->c.list.blink)
			break;
		next = rb_next(next);
	}
	// catch cases where curr_node and/or curr_event are 0
}

ostream &
operator<<(ostream &os, const EventTree &t)
{
	EventTreeNode *n;
	Event *e, *events;

	n = t.head->c.list.flink;
	for (;;) {
		events = (Event *)n->v.val;
		for (e = events; e != 0; e = e->GetNextEvent())
			os << *e << endl;
		if (n == t.head->c.list.blink)
			break;
		n = rb_next(n);
	}
	return (os);
}
