/****************************************************************************
 *
 *	C Implementation of theObjects
 *	Juergen Wagner, J_Wagner@iao.fhg.de
 *
 * 93-07-26 J_Wagner@iao.fhg.de
 *	First version.
 *
 * 93-08-07 J_Wagner@iao.fhg.de
 *	Fixed a typo.
 *
 * 93-08-09 J_Wagner@iao.fhg.de
 *	Added The_getpidCmd.
 *
 ****************************************************************************/

# include <stdio.h>
# include <string.h>
# include <stdlib.h>
# include <tcl.h>
# include <tk.h>
# include <tkInt.h>
# include <regexp.h>
# include <tclInt.h>

/*# define DEBUG_TRAVERSAL /**/

# define MAXKEY	100

# define SLOT_VAR(var, name)	sprintf(var, "_o:%s", name)

static count = 0;
static char tmp[1024];

/*
 *----------------------------------------------------------------------
 *
 * anon ?root?
 *
 *	Generate a new number.
 *
 * Results:
 *      The procedure returns standard Tcl results.
 *
 * Performance gain (on a Sparc 10/30):
 *
 *	anon
 *
 *	Tcl version	 230 usec/call
 *	  C version	  30 usec/call
 *
 *----------------------------------------------------------------------
 */

int
The_anonCmd(clientData, interp, argc, argv)
  ClientData clientData;                /* Main window of interp. */
  Tcl_Interp *interp;                   /* Current interpreter. */
  int argc;                             /* Number of arguments. */
  char **argv;                          /* Argument strings. */
{
  if (argc > 2) {
    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
                     " ?prefix?\"", (char *) NULL);
    return(TCL_ERROR);
  }

  sprintf(tmp, "%s._%d", (argc == 2) ? argv[1] : "", ++count);
  Tcl_SetResult(interp, tmp, TCL_VOLATILE);
  return(TCL_OK);
}

/*
 * The_getpid ?-parent?
 *
 *	Determine the process id of this process or of its parent.
 *
 * Results:
 *	A string containing the requested process id.
 *
 */

int
The_getpidCmd(clientData, interp, argc, argv)
  ClientData clientData;                /* Main window of interp. */
  Tcl_Interp *interp;                   /* Current interpreter. */
  int argc;                             /* Number of arguments. */
  char **argv;                          /* Argument strings. */
{
  if (argc == 1) {
    sprintf(tmp, "%d", getpid());
  } else if (argc > 2) {
    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
                     " ?-parent?\"", (char *) NULL);
    return(TCL_ERROR);
  } else if (strcmp(argv[1], "-parent")) {
    Tcl_AppendResult(interp, "syntax: ", argv[0], " ?-parent?", NULL);
    return(TCL_ERROR);
  } else {
    sprintf(tmp, "%d", getppid());
  }
    
  Tcl_SetResult(interp, tmp, TCL_VOLATILE);
  return(TCL_OK);
}

/*
 * The_argsCmd args...
 *
 *	Argument parsing routine. This parses the arguments found in "args"
 *	in the next higher level into variables also set in the next higher
 *	level. The procedure takes a list of option names and default values.
 *
 * Performance gain (on a Sparc 10/30):
 *
 *	proc foo {args} {
 *	  args foo {} {bar 9} {baz 3 4}
 *	  list $foo $bar $baz
 *	}
 *	foo -foo 9
 *
 *	Tcl version	7600 usec/call
 *	C version	 400 usec/call
 */

int
The_argsCmd(clientData, interp, argc, argv)
  ClientData clientData;                /* Main window of interp. */
  Tcl_Interp *interp;                   /* Current interpreter. */
  int argc;                             /* Number of arguments. */
  char **argv;                          /* Argument strings. */
{
  char *value, *arguments;
  char *key[MAXKEY], *def[MAXKEY];
  char *option;
  int listc[MAXKEY], argsc, i, j, found, opt;
  char **listv[MAXKEY], **argsv;

  /* Adjust the window name if it is a '*' */
  value = Tcl_GetVar(interp, "name", 0);
  if (value && value[0] == '*') {
    if ( !value[1] || (value[1] == '.' && ! value[2]) ) {
      value = "*";
    }
    sprintf(tmp, "%s._%d", &value[1], ++count);
    Tcl_SetVar(interp, "name", tmp, 0);
    Tcl_ResetResult(interp);
  }

  /* Initialize defaults for all variables */
  j = 0;
  for (i = 1; i < argc; i++) {
    if (Tcl_SplitList(interp, argv[i], &listc[i], &listv[i]) != TCL_OK) {
      return(TCL_ERROR);
    }
    if (listc[i] == 0) {
      continue;
    }
    if (listc[i] > 1) {
      key[j] = listv[i][0];
      def[j] = listv[i][1];
    } else {
      key[j] = listv[i][0];
      def[j] = "";
    }
    Tcl_SetVar(interp, key[j], def[j], 0);
    j++;
  }

  /* Parse the argument list */
  arguments = Tcl_GetVar(interp, "args", 0);
  if ( arguments ) {
    if (Tcl_SplitList(interp, arguments, &argsc, &argsv) != TCL_OK) {
      return(TCL_ERROR);
    }
    if (argsc & 1) {
      Tcl_AppendResult(interp,
		       "Argument list has an odd number of elements: \n  ",
		       arguments, NULL);
      return(TCL_ERROR);
    }
    for (i = 0; i < argsc; i++) {
      found = 0;
      option = argsv[i++];

      if ( *option != '-' ) {
	Tcl_AppendResult(interp, "Options must start with '-': ", option,
			 NULL);
	return(TCL_ERROR);
      }

      for (opt = 0; opt < j; opt++) {
	if ( !strcmp(&option[1], key[opt]) ) {
	  found = 1;
	  Tcl_SetVar(interp, key[opt], argsv[i], 0);
	  break;
	}
      }

      if ( found == 0 ) {
	Tcl_AppendResult(interp, "Unknown option: ", option,
			 " (", argsv[i], ")", NULL);
	return(TCL_ERROR);
      }
    }
  }

  /* Return empty list */
  for (i = 1; i < argc; i++) {
    if (listc[i])
      free((char *) listv[i]);
  }
  Tcl_ResetResult(interp);
  return(TCL_OK);
}

/*
 * The_defobjectCmd name ?super? ?slots?
 *
 *	Define a new object.
 *
 * Performance gain (on a Sparc 10/30):
 *
 *	defobject foo {} {{a 9} {b esjfsd} c}
 *	(without the call to defmethod)
 *
 *	Tcl version	3100 usec/call
 *	C version	 280 usec/call
 */

static char *vanilla = "vanilla-object";

int
The_defobjectCmd(clientData, interp, argc, argv)
  ClientData clientData;                /* Main window of interp. */
  Tcl_Interp *interp;                   /* Current interpreter. */
  int argc;                             /* Number of arguments. */
  char **argv;                          /* Argument strings. */
{
  char var[512], *value;
  char *name, *super = "", *slots = "";
  int superc, slotsc, slotc, i, j, nsupers, found;
  char **superv, **slotsv, **slotv;
  char *supers[MAXKEY];

  if ( argc < 2 || argc > 4 ) {
    Tcl_AppendResult(interp, "wrong # of args: ", argv[0],
		     "name ?super? ?slots?", NULL);
    return(TCL_ERROR);
  }

  name = argv[1];
  if ( argc > 2 ) {
    super = argv[2];
    if ( argc == 4 ) {
      slots = argv[3];
    }
  }

  if (Tcl_SplitList(interp, super, &superc, &superv) != TCL_OK) {
    return(TCL_ERROR);
  }
  if (Tcl_SplitList(interp, slots, &slotsc, &slotsv) != TCL_OK) {
    return(TCL_ERROR);
  }

  if ( strcmp(name, vanilla) && superc == 0 ) {
    if (superc)
      free((char *) superv);
    super = vanilla;
    if (Tcl_SplitList(interp, super, &superc, &superv) != TCL_OK) {
      return(TCL_ERROR);
    }
  }

  SLOT_VAR(var, name);

  /* Optionally notify the user */
  {
    char *verbose;

    verbose = Tcl_GetVar2(interp, "system", "verbose", TCL_GLOBAL_ONLY);
    if ( verbose && !strcmp(verbose, "1") ) {
      printf("**-- Creating object %s {%s}\n", name, super);
    }
  }

  /* Initialize the slots */
  for (i = 0; i < slotsc; i++) {
    if (Tcl_SplitList(interp, slotsv[i], &slotc, &slotv) != TCL_OK ||
	slotc == 0) {
      return(TCL_ERROR);
    }
    value = (slotc == 1) ? "" : slotv[1];
    Tcl_SetVar2(interp, var, slotv[0], value, TCL_GLOBAL_ONLY);
    if (slotc)
      free((char *) slotv);
  }
  if (slotsc)
    free((char *) slotsv);

  /* Initialize the super-object list */
  value = Tcl_GetVar2(interp, var, "", TCL_GLOBAL_ONLY);
  Tcl_ResetResult(interp);
  if ( value ) {
    if (Tcl_SplitList(interp, value, &slotsc, &slotsv) != TCL_OK) {
      return(TCL_ERROR);
    }
    for (nsupers = 0; nsupers < slotsc; nsupers++) {
      supers[nsupers] = slotsv[nsupers];
    }
    if (slotsc)
      free((char *) slotsv);
  } {
    nsupers = 0;
  }

  /* Augment the list of super-objects */
  for (i = 0; i < superc; i++) {
    found = 0;
    for (j = 0; j < nsupers; j++) {
      if ( !strcmp(superv[i], supers[j]) ) {
	found = 1;
	break;
      }
    }
    if ( !found ) {
      supers[nsupers++] = superv[i];
    }
  }
  if (superc)
    free((char *) superv);
  value = Tcl_Merge(nsupers, supers);
  Tcl_SetVar2(interp, var, "", value, TCL_GLOBAL_ONLY);
  free(value);

  argc = 2;
  argv[0] = "defmethod";
  argv[1] = name;
  if (The_defmethodCmd(clientData, interp, argc, argv) != TCL_OK) {
    return(TCL_ERROR);
  }

  Tcl_SetResult(interp, name, TCL_VOLATILE);
  return(TCL_OK);
}

/*
 * The_defmethodCmd name ?method? ?args? ?body?
 *
 *	Define a new message handler.
 *
 */

static int
The_checkproc(interp, command)
Tcl_Interp *interp;
char *command;
{
  Interp *ip = (Interp *) interp;
  Tcl_HashEntry *ent = Tcl_FindHashEntry(&ip->commandTable, command);

  return(ent != (Tcl_HashEntry *) 0);
}

static char *
The_methodFinder(interp, object, method)
Tcl_Interp *interp;
char *object, *method;
{
  char **listv;
  int listc;
  char *super;
  char proc[256];

# ifdef DEBUG_TRAVERSAL
  printf("find %s %s\n", object, method);
# endif
  sprintf(proc, "_method(%s,%s)", object, method);
  if (The_checkproc(interp, proc)) {
    char *res = (char *) malloc(strlen(proc)+1);

# ifdef DEBUG_TRAVERSAL
    printf("** found: %s\n", proc);
# endif
    strcpy(res, proc);
    return(res);
  }

  SLOT_VAR(proc, object);
  super = Tcl_GetVar2(interp, proc, "", TCL_GLOBAL_ONLY);
  if (super) {
    if (Tcl_SplitList(interp, super, &listc, &listv) != TCL_OK) {
      return(NULL);
    }
    for (; listc; listc--, listv++) {
      super = The_methodFinder(interp, *listv, method);
      if (super) {
	return(super);
      }
    }
    if (listc)
      free((char *) listv);
  }

  return(NULL);
}

/*
 * The_methodInterpreter object message args...
 *
 *	Invokes the right handler procedure for the given message.
 *
 * Performance gain (on a Sparc 10/30):
 *
 *	defmethod foo list {x} {list $x + 3}
 *	defmethod {bar foo} lost {x} {list $x - 3}
 *	defmethod {baz bar} last {x} {list $x = $x}
 *	baz list 9
 *
 *	(without the call to defmethod)
 *
 *	Tcl version	4500 usec/call
 *	C version	 250 usec/call
 */
static int
The_methodInterpreter(clientData, interp, argc, argv)
  ClientData clientData;                /* Main window of interp. */
  Tcl_Interp *interp;                   /* Current interpreter. */
  int argc;                             /* Number of arguments. */
  char **argv;                          /* Argument strings. */
{
  char *handler;
  char *form;
  int result;

  Tcl_ResetResult(interp);

  if ( argc < 2 ) {
    Tcl_AppendResult(interp, "wrong # of arguments: ", argv[0],
		     " method args...", NULL);
    return(TCL_ERROR);
  }

  if (!strcmp(argv[1], "slot")) {
    static char var[256];

    SLOT_VAR(var, argv[0]);
    switch (argc) {
    case 3: {
      form = Tcl_GetVar2(interp, var, argv[2], TCL_GLOBAL_ONLY);
      if (!form)
	form = "";
      Tcl_SetResult(interp, form, TCL_VOLATILE);
      return(TCL_OK);
    }
    case 4: {
      form = Tcl_SetVar2(interp, var, argv[2], argv[3], TCL_GLOBAL_ONLY);
      Tcl_SetResult(interp, argv[3], TCL_VOLATILE);
      return(TCL_OK);
    }
    default:
      Tcl_AppendResult(interp, "wrong # of arguments: ", argv[0],
		      " slot slotname ?slotvalue", NULL);
      return(TCL_ERROR);
    }
  }

  handler = The_methodFinder(interp, argv[0], argv[1]);
  if (handler) {
    /* Built "concat [list $handler $name] $args" */
# ifdef DEBUG_TRAVERSAL
    printf("%s.%s: %s\n", argv[0], argv[1], handler);
# endif
    result = Tcl_VarEval(interp, handler, " ", argv[0], " ",
			 form = Tcl_Merge(argc-2, argv+2), NULL);
    free(handler);
    free(form);
    return(result);
  }
  if ( strcmp(argv[1], "DEFAULT") ) {
    handler = The_methodFinder(interp, argv[0], "DEFAULT");
    if (handler) {
      /* Built "concat [list $handler $name $method] $args" */
# ifdef DEBUG_TRAVERSAL
      printf("*%s.%s: %s\n", argv[0], argv[1], handler);
# endif
      result = Tcl_VarEval(interp, handler, " ", argv[0], " ", argv[1], " ",
			   form = Tcl_Merge(argc-2, argv+2), NULL);
      free(handler);
      free(form);
      return(result);
    }
  }
  Tcl_ResetResult(interp);
  Tcl_AppendResult(interp, "Object '", argv[0], "' doesn't handle message {",
		   Tcl_Merge(argc-1, argv+1), "}.", NULL);
  return(TCL_ERROR);
}

int
The_defmethodCmd(clientData, interp, argc, argv)
  ClientData clientData;                /* Main window of interp. */
  Tcl_Interp *interp;                   /* Current interpreter. */
  int argc;                             /* Number of arguments. */
  char **argv;                          /* Argument strings. */
{
  int listc;
  char **listv, *super;

  Tcl_ResetResult(interp);

  if ( argc != 2 && argc != 5 ) { 
    Tcl_AppendResult(interp, "wrong # of arguments: ", argv[0],
		     " name ?method args body?", NULL);
    return(TCL_ERROR);
  }

  if (Tcl_SplitList(interp, argv[1], &listc, &listv) != TCL_OK) {
    return(TCL_ERROR);
  }
  if (listc == 1) {
    super = "";
  } {
    super = Tcl_Merge(listc-1, listv+1);
  }

  if (argc > 2) {
    if (!strcmp(argv[2], "slot")) {
      /* We have hard-coded the "slot" method, so prohibit redefinition! */
      char *verbose;

      verbose = Tcl_GetVar2(interp, "system", "verbose", TCL_GLOBAL_ONLY);
      if ( verbose && !strcmp(verbose, "1") ) {
	fprintf(stderr,
		"WARNING: Attempted to redefine 'slot' method of %s.\n",
		argv[1]);
	fprintf(stderr, "\tThis definition will be ignored!\n");
	free(super);
	if (listc)
	  free((char *) listv);
	return(TCL_OK);
      }
    }
    if (Tcl_VarEval(interp, "proc _method(", listv[0], ",", argv[2],
		    ") {self ", argv[3], "} {", argv[4], "}", NULL)
	!= TCL_OK) {
      return(TCL_ERROR);
    }
  }

  if (!The_checkproc(interp, listv[0])) {
    char var[512], *argv2[3];

    Tcl_CreateCommand(interp, listv[0], The_methodInterpreter,
		      clientData, (Tcl_CmdDeleteProc *) 0);

    SLOT_VAR(var, listv[0]);
    if (!Tcl_GetVar2(interp, var, "", TCL_GLOBAL_ONLY)) {
      argv2[0] = "defobject";
      argv2[1] = listv[0];
      argv2[2] = super;
      The_defobjectCmd(clientData, interp, 3, argv2);
    }
  }

  free(super);
  if (listc)
    free((char *) listv);
  return(TCL_OK);
}
