/*
 *=============================================================================
 *                                  tSippPoly.c
 *-----------------------------------------------------------------------------
 * Tcl commands to manage SIPP polygons and surfaces.
 *-----------------------------------------------------------------------------
 * Copyright 1992-1993 Mark Diekhans
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies.  Mark Diekhans makes
 * no representations about the suitability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 *-----------------------------------------------------------------------------
 * $Id: tSippPoly.c,v 5.0 1994/09/05 01:23:23 markd Rel $
 *=============================================================================
 */

#include "tSippInt.h"

/*
 * Flags from parsed options to SippPolygonPush
 */
#define POLYGON_NORM     0x01   /* Normal supplied               */
#define POLYGON_TEX      0x02   /* Textures supplied             */
#define POLYGON_CLOCK    0x04   /* Clockwise push of coordinates */

/*
 * Internal prototypes.
 */
static void
ClearPolygonStack ();

static void
BindSurfaceToHandle _ANSI_ARGS_((tSippGlob_pt    tSippGlobPtr,
                                 Surface         *surfacePtr));

static void
SurfaceHandleCleanup _ANSI_ARGS_((tSippGlob_pt   tSippGlobPtr));

static bool
ConvertVertexData _ANSI_ARGS_((tSippGlob_pt  tSippGlobPtr,
                               char         *listStr,
                               int           numTriples,
                               Vector        vertexData []));

static bool
PushVertexList _ANSI_ARGS_((tSippGlob_pt    tSippGlobPtr,
                            unsigned        options,
                            char           *vertexList));

/*=============================================================================
 * ClearPolygonStack --
 *    Clear the polygon stack if it contains any polygons.
 *-----------------------------------------------------------------------------
 */
static void
ClearPolygonStack ()
{
    Surface       *surfacePtr;
    Surf_desc_hdr  surfDescHdr;

    surfDescHdr.ref_count = 1;
    surfDescHdr.free_func = NULL;

    /*
     * Pop any pending polygons into a surface, then delete it.
     */

    surfacePtr = surface_create (&surfDescHdr, NULL);
    if (surfacePtr != NULL)
        surface_unref (surfacePtr);
}

/*=============================================================================
 * BindSurfaceToHandle --
 *   Assigns a handle to the specified surface.
 * Parameters:
 *   o tSippGlobPtr (I) - Pointer to the Tcl SIPP globals. The handle is
 *     returned in interp->result.
 *   o surfacePtr (I) - A pointer to the surface.
 *-----------------------------------------------------------------------------
 */
static void
BindSurfaceToHandle (tSippGlobPtr, surfacePtr)
    tSippGlob_pt    tSippGlobPtr;
    Surface         *surfacePtr;
{
    Surface  **surfaceEntryPtr;

    surfaceEntryPtr = (Surface **)
        Tcl_HandleAlloc (tSippGlobPtr->surfaceTblPtr, 
                         tSippGlobPtr->interp->result);
    *surfaceEntryPtr = surfacePtr;

}

/*=============================================================================
 * SurfaceHandleCleanup --
 *    Delete all surface handles that are defined.  Note that if the surface
 * is not associated with an object, memory will be lost.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static void
SurfaceHandleCleanup (tSippGlobPtr)
    tSippGlob_pt   tSippGlobPtr;
{
    int       walkKey = -1;
    Surface **surfaceEntryPtr;

    while (TRUE) {
        surfaceEntryPtr = Tcl_HandleWalk (tSippGlobPtr->surfaceTblPtr,
                                          &walkKey);
        if (surfaceEntryPtr == NULL)
            break;

        surface_unref (*surfaceEntryPtr);
        Tcl_HandleFree (tSippGlobPtr->surfaceTblPtr, surfaceEntryPtr);
    }

}

/*=============================================================================
 * TSippSurfaceHandleToPtr --
 *   Utility procedure to convert a surface handle to a surface pointer.
 *   For use of by functions outside of this module.
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o handle (I) - A surface handle.
 * Returns:
 *   A pointer to the surface, or NULL if an error occured.
 *-----------------------------------------------------------------------------
 */
Surface *
TSippSurfaceHandleToPtr (tSippGlobPtr, handle)
    tSippGlob_pt    tSippGlobPtr;
    char           *handle;
{
    Surface **surfaceEntryPtr;

    surfaceEntryPtr = (Surface **)
        Tcl_HandleXlate (tSippGlobPtr->interp, 
                         tSippGlobPtr->surfaceTblPtr, handle);
    if (surfaceEntryPtr == NULL)
        return NULL;
    return *surfaceEntryPtr;

}

/*=============================================================================
 * ConvertVertexData --
 *   Convert a list contain either vertex data.  This consists of the specified
 * number of triples of numbers.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o listStr (I) - A Tcl list containing the vertex values to convert.
 *   o numTriples (I) - Number of triples to expect.
 *   o vertex (O) - Array that the data is returned in.
 * Returns:
 *   TRUE if the list and numbers are valid, FALSE if there is an error.
 *-----------------------------------------------------------------------------
 */
static bool
ConvertVertexData (tSippGlobPtr, listStr, numTriples, vertexData)
    tSippGlob_pt  tSippGlobPtr;
    char         *listStr;
    int           numTriples;
    Vector        vertexData [];
{
    int      dataArgc,  coordArgc, idx;
    char   **dataArgv, **coordArgv;

    if (Tcl_SplitList (tSippGlobPtr->interp, listStr,
                       &dataArgc, &dataArgv) != TCL_OK)
        return FALSE;

    if (dataArgc != numTriples) {
        char numStr [32];
        sprintf (numStr, "%d", numTriples);
        Tcl_AppendResult (tSippGlobPtr->interp, "data for each vertex ",
                          "should consist of ", numStr, " sets of ",
                          "coordinates or vectors, got \"", listStr, "\"",
                          (char *) NULL);
        goto errorCleanup;
    }

    for (idx = 0; idx < numTriples; idx++) {
        if (Tcl_SplitList (tSippGlobPtr->interp, dataArgv [idx], &coordArgc,
                           &coordArgv) != TCL_OK)
            goto errorCleanup;

        if (coordArgc != 3) {
            Tcl_AppendResult (tSippGlobPtr->interp, "coordinate or vector ",
                              " must be a list of three numbers, got \"",
                              dataArgv [idx], "\"", (char *) NULL);
            goto errorCleanup2;
        }

        if (Tcl_GetDouble (tSippGlobPtr->interp, coordArgv [0],
                           &(vertexData [idx].x)) != TCL_OK)
           goto errorCleanup2;
        if (Tcl_GetDouble (tSippGlobPtr->interp, coordArgv [1],
                           &(vertexData [idx].y)) != TCL_OK)
           goto errorCleanup2;
        if (Tcl_GetDouble (tSippGlobPtr->interp, coordArgv [2],
                           &(vertexData [idx].z)) != TCL_OK)
           goto errorCleanup2;

        sfree (coordArgv);
    }
    sfree (dataArgv);
    return TRUE;

errorCleanup2:
    sfree (coordArgv);
errorCleanup:
    sfree (dataArgv);
    return FALSE;

}

/*=============================================================================
 * PushVertexList --
 *   Parse, convert a list of vertex coordinates, and optional texture
 * coordinates on the vertex stack.
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o options (I) - Options describing the vertexes supplied.  
 *   o vertexList (I) - A list of lists of vertex coordinates or a list of
 *     vertex, normals and texture coordinates (in some combination).
 * Returns:
 *   TRUE if the list and numbers are valid, FALSE if there is an error.
 *-----------------------------------------------------------------------------
 */
static bool
PushVertexList (tSippGlobPtr, options, vertexList)
    tSippGlob_pt    tSippGlobPtr;
    unsigned        options;
    char           *vertexList;
{
    int      vertexArgc, idx, idxLimit, idxIncr;
    char   **vertexArgv;
    Vector   vertex [3];

    if (Tcl_SplitList (tSippGlobPtr->interp,
                       vertexList, &vertexArgc,
                       &vertexArgv) != TCL_OK)
        return FALSE;

    if (options & POLYGON_CLOCK) {
        idx      = vertexArgc - 1;
        idxLimit = -1;
        idxIncr  = -1;
    } else {
        idx      = 0;
        idxLimit = vertexArgc;
        idxIncr  = 1;
    }

    switch (options & ~POLYGON_CLOCK) {
      case POLYGON_NORM | POLYGON_TEX:
        for (; idx != idxLimit; idx += idxIncr) {
            if (!ConvertVertexData (tSippGlobPtr,
                                    vertexArgv [idx],
                                    3,
                                    vertex))
                goto errorExit;
            vertex_tx_n_push (vertex [0].x, vertex [0].y, vertex [0].z,
                              vertex [1].x, vertex [1].y, vertex [1].z,
                              vertex [2].x, vertex [1].y, vertex [2].z);
        }
        break;

      case POLYGON_NORM:
        for (; idx != idxLimit; idx += idxIncr) {
            if (!ConvertVertexData (tSippGlobPtr,
                                    vertexArgv [idx],
                                    2,
                                    vertex))
                goto errorExit;
            vertex_n_push (vertex [0].x, vertex [0].y, vertex [0].z, 
                           vertex [1].x, vertex [1].y, vertex [1].z);
        }
        break;

      case POLYGON_TEX:
        for (; idx != idxLimit; idx += idxIncr) {
            if (!ConvertVertexData (tSippGlobPtr,
                                    vertexArgv [idx],
                                    2,
                                    vertex))
                goto errorExit;
            vertex_tx_push (vertex [0].x, vertex [0].y, vertex [0].z, 
                            vertex [1].x, vertex [1].y, vertex [1].z);
        }
        break;

      case 0:
        for (; idx != idxLimit; idx += idxIncr) {
            if (!TSippConvertVertex (tSippGlobPtr,
                                     vertexArgv [idx],
                                     &(vertex [0])))
                goto errorExit;
            vertex_push (vertex [0].x, vertex [0].y, vertex [0].z);
        }
        break;
    }

    sfree (vertexArgv);
    return TRUE;

    /*
     * Cleanup adter an error.  We pop any pending vertices into a polygon,
     * then clear the polygon stack.  This is a no-op if none have been pushed.
     */
errorExit:
    polygon_push ();
    ClearPolygonStack ();

    sfree (vertexArgv);
    return FALSE;

}

/*=============================================================================
 * SippPolygonPush --
 *   Implements the command:
 *       SippPolygonPush [-flags] vertexList
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippPolygonPush (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt  tSippGlobPtr = (tSippGlob_pt) clientData;
    unsigned      options = 0, optionsSeen = 0;
    char         *flag;
    int           argIdx = 1;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    /*
     * Parse the flags and validate argument.
     */
    if (argc < 2)
        goto wrongArgs;
    
    while ((argIdx < argc) && (argv [argIdx][0] == '-')) {
        flag = argv [argIdx];
        argIdx++;
        if (STREQU (flag, "-norm")) {
            if (optionsSeen & POLYGON_NORM)
                goto dupNorm;
            options |= POLYGON_NORM;
            optionsSeen |= POLYGON_NORM;
            continue;
        }
        if (STREQU (flag, "-nonorm")) {
            if (optionsSeen & POLYGON_NORM)
                goto dupTex;
            optionsSeen |= POLYGON_NORM;
            continue;
        }
        if (STREQU (flag, "-tex")) {
            if (optionsSeen & POLYGON_TEX)
                goto dupTex;
            options |= POLYGON_TEX;
            optionsSeen |= POLYGON_TEX;
            continue;
        }
        if (STREQU (flag, "-notex")) {
            if (optionsSeen & POLYGON_TEX)
                goto dupTex;
            optionsSeen |= POLYGON_TEX;
            continue;
        }
        if (STREQU (flag, "-clock")) {
            if (options & POLYGON_CLOCK)
                goto dupClock;
            options |= POLYGON_CLOCK;
            optionsSeen |= POLYGON_CLOCK;
            continue;
        }
        if (STREQU (flag, "-counter")) {
            if (options & POLYGON_CLOCK)
                goto dupClock;
            optionsSeen |= POLYGON_CLOCK;
            continue;
        }
        Tcl_AppendResult (interp, "invalid flag \"", flag,
                          "\" expected one of: ",
                          "\"-norm\", \"-nonorm\", ",
                          "\"-tex\", \"-notex\", ",
                          "\"-clock\", or \"-counter\"",
                          (char *) NULL);
        return TCL_ERROR;
    }

    if (argIdx != argc - 1)
        goto wrongArgs;

    if (!PushVertexList (tSippGlobPtr, options, argv [argIdx]))
        return TCL_ERROR;

    polygon_push ();
    return TCL_OK;

  wrongArgs:
    Tcl_AppendResult (interp, "wrong # args: ", argv [0],
                      " [-flags] vertexList", (char *) NULL);
    return TCL_ERROR;

  dupNorm:
    interp->result = "can only specify one of \"-norm\" or \"-nonorm\"";
    return TCL_ERROR;

  dupTex:
    interp->result = "can only specify one of \"-tex\" or \"-notex\"";
    return TCL_ERROR;

  dupClock:
    interp->result = "can only specify one of \"-clock\" or \"-counter\"";
    return TCL_ERROR;
}

/*=============================================================================
 * SippPolygonClear --
 *   Implements the command:
 *       SippPolygonClear
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippPolygonClear (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt  tSippGlobPtr = (tSippGlob_pt) clientData;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if (argc != 1) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], (char *) NULL);
        return TCL_ERROR;
    }

    ClearPolygonStack ();

    return TCL_OK;

}

/*=============================================================================
 * SippSurfaceCreate --
 *   Implements the command:
 *     SippSurfaceCreate shaderhandle
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippSurfaceCreate (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt    tSippGlobPtr = (tSippGlob_pt) clientData;
    Surface        *surfacePtr;
    Shader         *shaderPtr;
    void           *surfDescPtr;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if (argc != 2) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], " shaderhandle",
                          (char *) NULL);
        return TCL_ERROR;
    }                     

    shaderPtr = TSippShaderHandleToPtr (tSippGlobPtr, argv [1],
                                        &surfDescPtr);
    if (shaderPtr == NULL)
        return TCL_ERROR;

    surfacePtr = surface_create (surfDescPtr, shaderPtr);
    if (surfacePtr == NULL) {
        Tcl_AppendResult (interp, "the polygon stack is empty",
                         (char *) NULL);
        return TCL_ERROR;
    }

    BindSurfaceToHandle (tSippGlobPtr, surfacePtr);
    return TCL_OK;

}

/*=============================================================================
 * SippSurfaceUnref --
 *   Implements the command:
 *     SippSurfaceUnref surfacelist|ALL
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippSurfaceUnref (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt    tSippGlobPtr = (tSippGlob_pt) clientData;
    int             idx;
    handleList_t    surfaceList;
    handleList_t    surfaceEntryList;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if (argc != 2) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0],
                          " surfacelist|ALL", (char *) NULL);
        return TCL_ERROR;
    }                     

    if (STREQU (argv [1], "ALL")) {
        SurfaceHandleCleanup (tSippGlobPtr);
        return TCL_OK;
    }

    if (!TSippHandleListConvert (tSippGlobPtr, tSippGlobPtr->surfaceTblPtr,
                                 argv [1], &surfaceList, &surfaceEntryList,
                                 NULL))
        return TCL_ERROR;

    for (idx = 0; idx < surfaceList.len; idx++) {
        surface_unref (surfaceList.ptr [idx]);
        Tcl_HandleFree (tSippGlobPtr->surfaceTblPtr, 
                        surfaceEntryList.ptr [idx]);
    }

    TSippHandleListFree (&surfaceList);
    TSippHandleListFree (&surfaceEntryList);
    return TCL_OK;

}

/*=============================================================================
 * SippSurfaceSetShader --
 *   Implements the command:
 *     SippSurfaceSetShader surfacelist shaderhandle
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippSurfaceSetShader (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt    tSippGlobPtr = (tSippGlob_pt) clientData;
    int             idx;
    handleList_t    surfaceList;
    Shader         *shaderPtr;
    void           *surfDescPtr;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if (argc != 3) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0],
                          "  surfacelist shaderhandle", (char *) NULL);
        return TCL_ERROR;
    }                     

    if (!TSippHandleListConvert (tSippGlobPtr, tSippGlobPtr->surfaceTblPtr,
                                 argv [1], &surfaceList, NULL, NULL))
        return TCL_ERROR;

    shaderPtr = TSippShaderHandleToPtr (tSippGlobPtr, argv [2], &surfDescPtr);
    if (shaderPtr == NULL)
        goto errorExit;

    for (idx = 0; idx < surfaceList.len; idx++)
        surface_set_shader ((Surface *) surfaceList.ptr [idx], surfDescPtr,
                            shaderPtr);

    TSippHandleListFree (&surfaceList);
    return TCL_OK;
errorExit:
    TSippHandleListFree (&surfaceList);
    return TCL_ERROR;

}

/*=============================================================================
 * TSippPolyInit --
 *   Initialized the polygon and surface commands, including creating the 
 *   polygon table.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
void
TSippPolyInit (tSippGlobPtr)
    tSippGlob_pt  tSippGlobPtr;
{
    static tSippTclCmdTbl_t cmdTable [] = {
        {"SippPolygonPush",      (Tcl_CmdProc *) SippPolygonPush},
        {"SippPolygonClear",     (Tcl_CmdProc *) SippPolygonClear},
        {"SippSurfaceCreate",    (Tcl_CmdProc *) SippSurfaceCreate},
        {"SippSurfaceUnref",     (Tcl_CmdProc *) SippSurfaceUnref},
        {"SippSurfaceSetShader", (Tcl_CmdProc *) SippSurfaceSetShader},
        {NULL,                   NULL}
    };

    tSippGlobPtr->surfaceTblPtr = 
        Tcl_HandleTblInit ("surface", sizeof (Surface *), 24);

    TSippInitCmds (tSippGlobPtr, cmdTable);

}

/*=============================================================================
 * TSippPolyCleanUp --
 *   Cleanup the surface table and release all associated resources.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
void
TSippPolyCleanUp (tSippGlobPtr)
    tSippGlob_pt  tSippGlobPtr;
{
    ClearPolygonStack ();

    SurfaceHandleCleanup (tSippGlobPtr);

    Tcl_HandleTblRelease (tSippGlobPtr->surfaceTblPtr);
    tSippGlobPtr->surfaceTblPtr = NULL;

}

