#
# artthread.tcl
#
# Copyright 1995 - Koert Zeilstra
#
# No warranty - use at your own risk
#

#
# TreadedArticleListDialog
#
# Public attributes:
#   -closecommand	command when quit is selected
#   -title		title
#
# Methods:
#
itcl_class ThreadedArticleListDialog \
{
  constructor {config} \
  {
    #
    #  Create a window with the same name as this object
    #
    set class [$this info class]
    ::rename $this $this-tmp-
    ::toplevel $this -class $class
    ::rename $this $this-win-
    ::rename $this-tmp- $this
    wm title $this $title

    frame $this.view
	
    set canvas [canvas $this.view.canvas -width 20c -height 10c \
	-scrollregion {0 0 600 800} \
	-relief sunken -borderwidth 2 \
	-xscrollcommand "$this.view.hscroll set" \
	-yscrollcommand "$this.view.vscroll set"]
    scrollbar $this.view.vscroll -command "$this.view.canvas yview"
    scrollbar $this.view.hscroll -command "$this.view.canvas xview" \
		-orient horizontal
    pack $this.view.hscroll -side bottom -fill x
    pack $this.view.vscroll -side right -fill y
    pack $this.view.canvas -expand yes -fill both
    pack $this.view -expand yes -fill both

    $canvas bind all <Any-ButtonPress-1> "$this highlightItem"
    $canvas bind all <Any-Double-ButtonPress-1> "$this read"

    entry $this.info -width 40
    pack $this.info -side top -fill x

    frame $this.buttons1 -relief sunken -bd 1
    button $this.buttons1.read -text "Read.." -width 9 \
		-command "$this read"
    button $this.buttons1.uudecode -text "Uudecode" -width 9 \
		-command "$this uudecode"
    button $this.buttons1.post -text "Post.." -width 9 \
		-command "$this post"
    pack $this.buttons1.read $this.buttons1.uudecode \
		-side left -padx 3 -pady 2
    pack $this.buttons1.post \
	-side right -padx 3 -pady 2
    pack $this.buttons1 -side top -fill x -pady 1

    frame $this.buttons2 -relief sunken -bd 1
    button $this.buttons2.markall -text "Mark All Read" -width 10 \
		-command "$this markall"
    button $this.buttons2.close -text "Close" -width 10 \
		-command "$this close"
    pack $this.buttons2.close $this.buttons2.markall \
	-side right -padx 3 -pady 2
    pack $this.buttons2 -side top -fill x -pady 1

    set article_status [ArticleList $this.article_status]

    wm protocol . WM_DELETE_WINDOW "$this close"
  }

  destructor \
  { ::rename $this-win- {} }

  method config {config} {}

  method read {} \
  { if {$current_article != ""} \
    { if {$article_dialog == ""} \
      { set article_dialog [ArticleDialog $this.article_dialog \
		-title "Article" -main $main -article_list $this \
		-closecommand "wm withdraw $this.article_dialog"]
      } \
      else \
      { wm deiconify $article_dialog }
      showInfo "loading article $current_article..."
      $article_dialog showArticle $current_group $current_article
      showInfo ""
      $article_status setRead $current_article
    }
  }

  method uudecode {} \
  {
    set nntp_socket [$main getNntpSocket]
    if {$nntp_socket != ""} \
    { showInfo "saving article..."
      dp_send $nntp_socket "group $current_group"
      gets $nntp_socket line
      set tmp_file "/tmp/flash[pid]"
      set fp [::open $tmp_file w]
      dp_send $nntp_socket "article $current_article"
      gets $nntp_socket line
      if {[lindex $line 0] != "220"} \
      { show_error $this [string range 4 end [string trimright $line]]
      } \
      else \
      { set done 0
        while {$done == 0} \
        { gets $nntp_socket line
          set line [string trimright $line]
          if {$line == "."} \
          { set done 1 } \
          else \
          { puts $fp $line }
        }
        ::close $fp
        showInfo "uudecoding..."
        if [catch {exec uudecode $tmp_file} msg] \
        { show_error $this "Uudecode Error" $msg }
        exec rm $tmp_file
        showInfo ""
      }
      $article_status setRead $current_article
    }
  }

  method followup {subject references} \
  { if {$post_dialog == ""} \
    { set post_dialog [PostDialog $this.post_dialog \
		-main $main -title "Post Article" \
		-closecommand "wm withdraw $this.post_dialog"]
    } \
    else \
    { wm deiconify $post_dialog }
    $post_dialog setGroupAndClear $current_group
    $post_dialog setFollowup $subject $references
  }

  method post {} \
  { if {$post_dialog == ""} \
    { set post_dialog [PostDialog $this.post_dialog \
		-main $main -title "Post Article" \
		-closecommand "wm withdraw $this.post_dialog"]
    } \
    else \
    { wm deiconify $post_dialog }
    $post_dialog setGroupAndClear $current_group
  }

  method markall {} \
  { $article_status setReadAll $last_number
    $canvas itemconfigure all -fill $read_color
  }

  method close {} \
  { saveArticleStatus
    if {$article_dialog != ""} \
    { $article_dialog close }
    eval $closecommand
  }

  method saveArticleStatus {} \
  { $group_stat updateGroupReadArticles $current_group \
		[$article_status getArticleList]
  }

  method showInfo {message} \
  { $this.info delete 0 end
    $this.info insert 0 $message
    update idletasks
  }

  method clickArticle {row} \
  { set current_article $article_numbers($row)
    set current_row $row
  }

  method doubleClickArticle {row} \
  { set current_article $article_numbers($row)
    set current_row $row
    read
  }

  method showGroupArticles {group_name} \
  { wm title $this "$title - $group_name"
    $canvas delete all
    $article_status setArticleList [$group_stat getGroupReadArticles \
		$group_name]
    set current_group $group_name
    set current_item ""
    set current_article ""
    catch {unset art_subject}
    catch {unset art_from}
    catch {unset art_idnumber}
    catch {unset art_numberid}
    catch {unset art_children}
    catch {unset art_has_parent}
    set nntp_socket [$main getNntpSocket]
    if {$nntp_socket != ""} \
    { dp_send $nntp_socket "group $current_group"
      gets $nntp_socket line
      set line [string trim $line]
      set first_number [lindex $line 2]
      set last_number [lindex $line 3]
      set lowest [$article_status getLowestNumber]
      if {$lowest < $first_number} \
      { $article_status setLowestNumber $first_number
        set lowest $first_number
      }
      showInfo "Getting subjects"
      getXhdr $nntp_socket $lowest $last_number subject setSubject
      showInfo "Getting authors"
      getXhdr $nntp_socket $lowest $last_number from setFrom
      showInfo "Getting message ids"
      getXhdr $nntp_socket $lowest $last_number Message-Id setMessageId
      showInfo "Getting references"
      getXhdr $nntp_socket $lowest $last_number References setReference
      set x $x_margin
      set y $line_height
      
      set no_of_articles [array size art_subject]
      showInfo "Threading $no_of_articles articles"
      set canvas_height [expr ($no_of_articles+1)*(2*$line_height+$y_margin)]
      $canvas config -scrollregion [list 0 0 600 $canvas_height]

      set ids [array names art_children]
      foreach id $ids \
      { if {$id == "(none)"} \
        { foreach number $art_children($id) \
          { set id $art_numberid($number)
            set y [displayArticle $id $x $y]
          }
        } \
        else \
        { if {[info exists art_idnumber($id)] \
		&& [info exists art_has_parent($art_idnumber($id))]} \
          { set is_child 1 } \
          else \
          { set y [displayArticle $id $x $y] }
        }
      }
    }
    showInfo ""
  }

  method getXhdr {nntp_socket start end field callback} \
  { dp_send $nntp_socket "xhdr $field $start-$end"
    gets $nntp_socket line
    set status [lindex $line 0]
    if {$status < 300} \
    { set done ""
      while {$done == ""} \
      { gets $nntp_socket line
        set line [string trimright $line]
        if {$line == "."} \
        { set done 1} \
        else \
        { set number [lindex $line 0]
          set value [string range $line [string length $number] end]
          if {[$article_status isRead $number] == 0} \
          { $callback $number $value }
        }
      }
    }
  }

  method setSubject {number value} \
  { set art_subject($number) $value }

  method setFrom {number value} \
  { set art_from($number) $value }

  method setMessageId {number value} \
  { set id [string trim $value]
    set art_idnumber($id) $number
    set art_numberid($number) $id
  }

  method setReference {number value} \
  { set reference [lindex [split $value] end]
    if {$reference == ""} \
    { set reference "(none)" }
    if {[info exists art_children($reference)] == 0} \
    { set art_children($reference) {} }
    lappend art_children($reference) $number
    set art_has_parent($number) 1
  }

  method setReferenceOld {number value} \
  { set reference [lindex [split $value] end]
    if {$reference != "" && [info exists art_messageid($reference)]} \
    { set parent $art_messageid($reference)
      if {[info exists art_children($parent)] == 0} \
      { set art_children($parent) {} }
      lappend art_children($parent) $number
      set art_has_parent($number) 1
    }
  }

  method displayArticle {id x y} \
  { if [info exists art_idnumber($id)] \
    { set number $art_idnumber($id)
      set subject $art_subject($number)
      set author $art_from($number)
      $canvas create text $x $y -text $art_subject($number) \
		-anchor nw -tags "art-$number" -font $font 
      set y [expr $y+$line_height]
      $canvas create text [expr $x+$x_margin] $y \
		-text $art_from($number) -anchor nw -font $font \
		-tags "art-$number" -fill $normal_color
      set y [expr $y+$line_height+$y_margin]
      set py [expr $y - $line_height - $y_margin]
    } \
    else \
    { set py $y
    }

    if [info exists art_children($id)] \
    { # set py [expr $y - $line_height - $y_margin]
      foreach number $art_children($id) \
      { set px [expr $x - $x_indent/2]
        $canvas create line $px $py $px [expr $y+$line_height] \
		[expr $x+$x_indent] [expr $y+$line_height] \
		-tags "art-$number" -fill $normal_color
        set id $art_numberid($number)
        set y [displayArticle $id [expr $x+$x_indent] $y]
      }
    }

    return $y
  }

  method highlightItem {} \
  { unhighlightItem
    set current_item [lindex [$canvas gettags current] 0]
    $canvas itemconfigure $current_item -fill $highlight_color
    set current_article [lindex [split $current_item "-"] 1]
  }

  method unhighlightItem {} \
  { if {$current_item != ""} \
    { if [$article_status isRead $current_article] \
      { $canvas itemconfigure $current_item -fill $read_color } \
      else \
      { $canvas itemconfigure $current_item -fill $normal_color }
    }
    set current_item ""
  }
  method takeFocus {} \
  {
  }

  public closecommand ""
  public main ""
  public title ""
  public group_stat ""
  public line_height 12
  public x_indent 15
  public x_margin 15
  public y_margin 5
  public highlight_color "blue"
  public normal_color "black"
  public read_color "yellow"
  public font "-*-helvetica-medium-r-normal-*-14-*-*-*-*-*-*-*"

  protected canvas ""
  protected current_group ""
  protected current_article ""
  protected last_number
  protected current_item ""
  protected no_of_rows 0
  protected article_dialog ""
  protected post_dialog ""
  protected article_numbers ""
  protected art_subject ""
  protected art_from ""
  protected art_idnumber ""
  protected art_numberid ""
  protected art_children ""
  protected art_has_parent ""
  protected article_status ""
}


