# internal.tcl,v 1.1 1995/11/17 00:42:07 steve Exp
#
#	PASTIME Project
#	Cooperative Research Centre for Advanced Computational Systems
#	COPYRIGHT NOTICE AND DISCLAIMER.
#
#	Copyright (c) 1995 ANU and CSIRO
#	on behalf of the participants in
#	the CRC for Advanced Computational Systems (ACSys)
#
# This software and all associated data and documentation ("Software")
# was developed for research purposes and ACSys does not warrant that 
# it is error free or fit for any purpose.  ACSys disclaims all liability
# for all claims, expenses, losses, damages and costs any user may incur 
# as a result of using, copying or modifying the Software.
#
# You may make copies of the Software but you must include all of this
# notice on any copy.
###
# Internal procedure declarations for surfit.  It is safe to re-source this
# file while SurfIt! is running.

# Documented procedures are prefixed with "SurfIt"
# Internal procedures and variables in this module are prefixed with "surfit"

###
### Register the HTML parser as a content-type handler for text/html
###

PRregister_type text/html .html HMparse_html {
    {PRdata $data}
    {PRfile {[$read_handler $read_handler_state]}}
    {PRfd {[$read_handler $read_handler_state]}}
    {PRimage [error "incompatible data"]}
} HMrender {} {hmstart hmcont}

proc HMlink_setup {win href} {
    surfit_setup_anchor $win $href

    regsub -all {%} $href {%%} href2	;# make bind happy
    set top [SurfIt_hyperpage $win]
    $win tag bind L:$href <Enter> "+set surfit_highlightURL($top) \{$href2\}"
    $win tag bind L:$href <Leave> "+set surfit_highlightURL($top) {}"
}

proc HMlink_window_setup {but link} {
    set top [SurfIt_hyperpage $but]
    regsub -all {%} $link {%%} link2	;# make bind happy
    bind $but <Enter> "+set surfit_highlightURL($top) \{$link2\}"
    bind $but <Leave> "+set surfit_highlightURL($top) {}"
}

# Override the library link-callback routine for the sample app.

proc HMlink_callback {win href} {
    global surfit_url_history_idx surfit_currPage
    upvar #0 HM$win var

    # Check if only a destination anchor was specified
    if {[regexp -nocase {^#(.+)} $href all name]} {
	HMgoto $win $name
	return
    }

    set hyperpage [SurfIt_hyperpage $win]
    incr surfit_url_history_idx($hyperpage)
    # Set insert position for applets
    upvar #0 PS$hyperpage newState
    set newState(index) [$win index insert]	;# Assumes Button-1 invokes link
    set newState(referer) $surfit_currPage($hyperpage)
    SurfIt_loadURL [URL_canonicalise $var(base) $href] $hyperpage
}

# Supply an image callback function
# NB. This function bypasses HMgot_image

proc HMset_image {win handle src} {
    global surfit_currPage
    set hyperpage [SurfIt_hyperpage $handle]
    upvar #0 HM$hyperpage var
    upvar #0 PS$hyperpage altvar

    set altvar(referer) $surfit_currPage($hyperpage)
#   loadDocument [URL_canonicalise $var(base) $src] $handle {} fileevent surfit_image_start
    if {[catch {PRloadDocument [URL_canonicalise $var(base) $src] $handle {} fileevent surfit_image_start} result]} {
	puts "unable to load inline due to \"$result\""
	# Do something about not being able to load the image
    }
}

# Callback for loading applets.

proc HMload_applet {win script level {formID {}}} {
    upvar #0 HM$win var

    upvar #0 PS$var(tags)$win appState
    set appState(win) $win
    set appState(index) [$win index $var(S_insert)]
    if {$level == "form"} {
	upvar #0 $formID form
	set appState(form) $formID
	set form(items) {}
    }
    PRloadDocument [URL_canonicalise $var(base) $script] $var(tags)$win {} fileevent surfit_image_start
}

proc surfit_image_start {url win type args} {
    global surfit_feedback surfit_pageStatus
    upvar #0 PR$win var
    upvar #0 PS$win intState

    if {[info exists intState(win)]} {set win $intState(win)}
    set hyperpage [SurfIt_hyperpage $win]
    incr surfit_pageStatus($hyperpage,numdocs)
    # Sometimes the content-length is not set even at this stage.
    # If content-length is subsequently set the feedback will be wrong
    catch {incr surfit_pageStatus($hyperpage,total) $var(HDRcontent-length)}
    SurfIt_setFeedback $hyperpage
}

# Override the default HMsubmit_form proc to handle additional
# features.  If there is an applet attached to the form then
# call into it and override query if necessary

proc HMsubmit_form {win form_id param query} {
    upvar #0 HM[SurfIt_hyperpage $win] var
    upvar #0 AppFormsMap$win formmap

    HMextract_param $param method
    set method [string tolower $method]

    # Call into applet if attached
    if {![catch {set formmap($form_id)} slave]} {
	if {[$slave eval info commands HMsubmit_form] != {}} {
	    if {[catch {$slave eval HMsubmit_form $method \{$query\}} query]} {
		# Cancel the form submission
		puts "HMsubmit_form: submission cancelled due to \"$query\""
		return
	    }
	}
    }

    switch $method {
	get {
	    set result ""
	    set sep ""
	    foreach i $query {
		append result $sep [HMmap_reply $i]
		if {$sep != "="} {set sep =} {set sep &}
	    }
	    if {[HMextract_param $param action]} {
		SurfIt_loadURL [URL_canonicalise $var(base) $action]?$result] [SurfIt_hyperpage $win]
	    }
	}
	put {
	    puts "PUT method not yet supported"
	}
	default {error "unknown method \"$method\""}
    }
}

# Let the attached applet know that the form has been reset
# This gives it a chance to reset its internal state

proc HMreset_form {win form_id} {
    upvar #0 HM[SurfIt_hyperpage $win]
    upvar #0 AppFormsMap$win formmap

    # Call into applet if attached
    if {![catch {set formmap($form_id)} slave]} {
	catch {$slave eval HMreset_form}
    }
}

###
### Load a URL into a hyperpage.
###
### url is the URL of the document to load
### win is the toplevel of the hyperpage to load the document into
###
### NB. That the given URL may not cause anything to be loaded
### (or at least cause a page reset), so don't jump the gun and
### clear the page at this point.
###

# If there is current processing/loading activity then it must
# be terminated before loading the new page.  However, in this case
# we will have been called from an event handler due to the use of
# 'update' and there are still processing commands outstanding.
# We must cause these commands to evaluate before starting the new load.

proc SurfIt_loadURL {url win {type {}}} {
    global surfit_pageStatus

    if {$surfit_pageStatus($win,busy)} {
	set surfit_pageStatus($win,pending) [list $url $type]
	surfit_terminate_loading $win
    } else {
	surfit_doLoadURL $url $win $type
    }
}

# This is the real load procedure, but should not be called directly

proc surfit_doLoadURL {url win {type {}}} {
    upvar #0 HM$win parser
    upvar #0 PR$win prot
    global surfit_feedback surfit_pageStatus

    # Tell the applets about it
    Applet_anchor_activation $win $url

    # Check if a local destination anchor was specified
    if {[regexp -nocase {([^#]*)#(.+)} $url all base name]} {
	if {$base == {} || $base == $parser(base)} {
	    HMgoto $win $name
	    return
	}
    }

    # Invoke pre-load scripts...

    set surfit_feedback($win) "Loading $url"
    array set surfit_pageStatus \
	"$win,busy 1 $win,numdocs 0 $win,total 0 $win,loaded 0"

    PRloadDocument $url $win HM$win fileevent surfit_new_page $type
}

# Clear the hyperpage in readiness for the new document

proc surfit_new_page {url win type args} {
    global surfit_pageStatus surfit_feedback Helpers HTtypes
    upvar #0 PR$win var

    # Handle automatic redirection.
    # It is wrong to do this at the application level - 
    # protocols.tcl should be doing it.

    if {[info exists var(HTTPstatus)]} {
	if {$var(HTTPstatus) == 301 || $var(HTTPstatus) == 302} {
	    if {[info exists var(HDRlocation)]} {
		SurfIt_loadURL $var(HDRlocation) $win
	    } else {
		# Strictly an error, but doc body should contain
		# hyperlink to real doc so let it through
	    }
	}
    }

    # Exceptions are Tcl applets, which must request for 
    # the page to be cleared explicitly, helper applications
    # and hypertools.  We need a more general mechanism!
    if {[info exists Helpers($type)] ||
	[info exists HTtypes($type)]} {
	return
    }
    if {$type != "application/x-tcl"} {
	# Initialise the window
	Applet_destroy_page $win
	# This routine takes care of the rest
	Applet_new_page $url $win
    }

    # Initialise hyperpage load stats
    incr surfit_pageStatus($win,numdocs)
    if {[info exists var(HDRcontent-length)] && \
	$var(HDRcontent-length) != {} && $var(HDRcontent-length) != 0} {
	incr surfit_pageStatus($win,total) $var(HDRcontent-length)
	set surfit_feedback($win) "Displaying $url, $var(HDRcontent-length) bytes"
    }
}

# Special version of surfit_new_page for Applets

proc Applet_new_page {url win} {
    global surfit_currPage surfit_feedback
    global surfit_url_history surfit_url_history_idx
    global surfit_highlightURL

    HMreset_win $win
    upvar #0 HM$win var
    upvar #0 PR$win prot

    set var(base) [URL_makeURI $prot(protocol) $prot(host) $prot(port) $prot(path)]
    surfit_setOpenPage $win $url
    set surfit_highlightURL($win) {}
    update idletasks

    if {$prot(name) != {}} {
	HMgoto $win $prot(name)
    }

    set t [winfo toplevel $win]
    set p $t.top-ctl-panel
    set s $t.second-ctl-panel
    set hyperpage [SurfIt_hyperpage $win]

    # Adjust navigation history
    if {$url == [lindex $surfit_url_history($hyperpage) $surfit_url_history_idx($hyperpage)]} {
	# Do nothing - visiting previous page
    } elseif {$surfit_url_history_idx($hyperpage) == [llength $surfit_url_history($hyperpage)]} {
	lappend surfit_url_history($hyperpage) $url
    } else {
	set surfit_url_history($hyperpage) \
		[lrange $surfit_url_history($hyperpage) 0 [expr $surfit_url_history_idx($hyperpage) - 1]]
	lappend surfit_url_history($hyperpage) $url
    }

    # Adjust button states
    if {$surfit_url_history_idx($hyperpage) == 0} {
	$s.back configure -state disabled
    } elseif {[llength $surfit_url_history($hyperpage)] > 1} {
	$s.back configure -state normal
    }
    if {$surfit_url_history_idx($hyperpage) < [expr [llength $surfit_url_history($hyperpage)] - 1]} {
	$s.forward configure -state normal
    } else {
	$s.forward configure -state disabled
    }
    # These really only needs to happen once...
    if {$surfit_url_history_idx($hyperpage) >= 0} {
	$s.start configure -state normal
    }
    $p.reload configure -state normal
    $p.source configure -state normal
}

# Protocol handler callbacks.  These let us know when loads complete
# or fail

proc PRhandlerFinished {win size} {
    global surfit_pageStatus
    upvar #0 PR$win var
    upvar #0 PS$win intState

    if {[info exists intState(win)]} {
	set win $intState(win)
    }
    set hyperpage [SurfIt_hyperpage $win]
    if {$size != {}} {incr surfit_pageStatus($hyperpage,loaded) $size}
    if {$var(eof)} {
	incr surfit_pageStatus($hyperpage,numdocs) -1
	if {$win == $hyperpage} {
	    set surfit_pageStatus($win,busy) 0
	    # Has a document load been scheduled?
	    if {[info exists surfit_pageStatus($win,pending)]} {
		set url [lindex $surfit_pageStatus($win,pending) 0]
		set type [lindex $surfit_pageStatus($win,pending) 1]
		unset surfit_pageStatus($win,pending)
		surfit_doLoadURL $url $win $type
		return
	    }
	}
    }
    SurfIt_setFeedback $hyperpage
}

proc PRhandlerError {win result} {
    global surfit_pageStatus errorInfo
    upvar #0 PR$win var
    upvar #0 PS$win intState

    set cause $errorInfo	;# Save the text
    if {[info exists intState(win)]} {
	set win $intState(win)
    }
    set hyperpage [SurfIt_hyperpage $win]
    incr surfit_pageStatus($hyperpage,numdocs) -1
    if {$var(HDRcontent-length) != {}} {incr surfit_pageStatus($hyperpage,total) -$var(HDRcontent-length)}
    # Should also subtract the # bytes loaded so far for this document
    SurfIt_setFeedback $hyperpage
    update

    if {$win == $hyperpage} {
	set surfit_pageStatus($win,busy) 0
	if {[tk_dialog [winfo toplevel $win]_error "Load Error" "Error loading document:\n$result" {} 0 "OK" "Stack Trace"] == 1} {
	    # Display the stack trace
	    set top [toplevel [winfo toplevel $win]_error$var(id)]
	    wm title $top "Stack Trace"
	    text $top.text -height 20 -width 80 -yscrollcommand "$top.scroll set"
	    $top.text insert end $cause
	    scrollbar $top.scroll -command "$top.text yview" -orient vertical
	    button $top.dismiss -text "OK" -command "destroy $top"
	    pack $top.dismiss -side bottom
	    pack $top.text -side left -fill both -expand yes
	    pack $top.scroll -side right -fill y
	}

	# Has a document load been scheduled?
	if {[info exists surfit_pageStatus($win,pending)]} {
	    set url [lindex $surfit_pageStatus($win,pending) 0]
	    set type [lindex $surfit_pageStatus($win,pending) 1]
	    unset surfit_pageStatus($win,pending)
	    surfit_doLoadURL $url $win $type
	    return
	}
    }
}

proc SurfIt_setFeedback {win} {
    global surfit_pageStatus surfit_feedback surfit_currPage

    set hyperpage [SurfIt_hyperpage $win]
    set FBwin [winfo toplevel $win].bottom-panel.feedback
    $FBwin configure -fg black
    if {$surfit_pageStatus($hyperpage,numdocs) == 0} {
	set surfit_feedback($hyperpage) "Document done"
	# Let the applets know
	Applet_pageloaded $hyperpage $surfit_currPage($hyperpage)
    } elseif {$surfit_pageStatus($hyperpage,total) > 0} {
	set surfit_feedback($hyperpage) "Loading $surfit_pageStatus($hyperpage,numdocs) documents, \
[expr $surfit_pageStatus($hyperpage,loaded) * 100 / $surfit_pageStatus($hyperpage,total)]% completed"
    } else {
	set surfit_feedback($hyperpage) "Loading $surfit_pageStatus($hyperpage,numdocs) documents"
    }
}

# From html_library/sample.tcl:
# downloading fonts can take a long time.  We'll override the default
# font-setting routine to permit better user feedback on fonts.  We'll
# keep our own list of installed fonts on the side, to guess when delays
# are likely

proc HMset_font {win tag font} {
    global surfit_feedback surfit_Fonts
    set hyperpage [SurfIt_hyperpage $win]
    set oldmessage $surfit_feedback($hyperpage)
    set FBwin [winfo toplevel $win].bottom-panel.feedback
    if {![info exists surfit_Fonts($font)]} {
	set surfit_Fonts($font) 1
	$FBwin configure -fg blue
	set surfit_feedback($hyperpage) "downloading font $font"
	update
    }
    if {[catch {$win tag configure $tag -font $font} fonterr]} {
	$FBwin configure -fg red
	set surfit_feedback($hyperpage) $fonterr
    } else {
	$win tag lower $tag mark
	$FBwin configure -fg black
	set surfit_feedback($hyperpage) $oldmessage
    }
}

proc surfit_terminate_loading {win} {
    set win [SurfIt_hyperpage $win]
    foreach g [info globals PR$win*] {
	upvar #0 $g var
	if {[info exists var(fd)]} {
	    if {[info exists var(url)]} {puts "stopping $g (\"$var(url)\")"} else {puts "stopping processing"}
	    if {[catch {fileevent $var(fd) readable {}} msg]} {
		puts "unable to delete fileevent: \"$msg\""
	    } else {
		puts "fileevent removed"
	    }
	    catch {close $var(fd)} ;# what about cache?
	    unset var(fd)
	}
    }
    HMset_state $win -stop 1
}

# Applet Management: We need to keep track of which applets are attached
# at what level and what they are attached to.

# Applet_created callback procedure to detect when applets are created
# NB. This gets called before the applet's code is evaluated, hence the
# user may at least have some control over the applet!
# Applets are created at the hyperpage level by default.
# NB. #2: Numerals denote menu indexes, so had to add "Applet" to menu item labels

proc Applet_created {win slave} {
    global surfit_Applets
    upvar #0 PR[set win [SurfIt_hyperpage $win]] var

    set surfit_Applets($slave) $win	;# What the slave is attached to
    set surfit_Applets($slave,url) $var(url)

    set menu [winfo toplevel $win].top-ctl-panel.applets.m

    # Hyperpage-level applets are added above the hyperwindow separator
    $menu insert "Window" cascade -label "Applet $slave" -menu [surfit_create_applet_menu $menu $slave $win $var(url)]
}

proc surfit_create_applet_menu {menu slave win url} {
    menu $menu.$slave -tearoff no
    $menu.$slave add command -label "Reload" -state disabled -command "
	interp_delete $slave
	Cache_expire_document $url
	SurfIt_loadURL $url $win
    "
    $menu.$slave add command -label "Suspend" -state disabled
    $menu.$slave add command -label "Delete" -command "interp_delete $slave"

    return $menu.$slave
}

# Define the Applet level change callback procedure

proc Applet_levelChange {slave old new} {
    global surfit_Applets

    set object $surfit_Applets($slave)
    set menu [winfo toplevel $object].top-ctl-panel.applets.m
    set menuitem [$menu entrycget "Applet $slave" -menu]

    if {$old == "hyperpage" && $new == "hyperwindow"} {
	# Simply move the menu
	$menu delete "Applet $slave"
	$menu add cascade -label "Applet $slave" -menu $menuitem
    } else {
	# Either hyperpage->browser or hyperwindow->browser

	# Remove the applet from the old object
	$menu delete "Applet $slave"
	destroy $menuitem

	# Add the applet to the browser object
	set menu .ctl-panel.applets.m
	$menu add cascade -label "Applet $slave" -menu [surfit_create_applet_menu $menu $slave browser $surfit_Applets($slave,url)]
	set surfit_Applets($slave) browser
    }
}

# Override the interp_delete_slave procedure to detect when applets are destroyed

proc interp_delete_slave {slave} {
    global surfit_Applets

    surfit_interp_delete_slave $slave

    if {[set object $surfit_Applets($slave)] == "browser"} {
	set menu .ctl-panel.applets.m
    } else {
	set menu [winfo toplevel $object].top-ctl-panel.applets.m
    }
    $menu delete "Applet $slave"
}

###
### Window management routines
###

proc surfit_create_window {{open_url {}}} {
    global surfit_win_num surfit_url_history surfit_url_history_idx surfit_open_vars
    global surfit_currPage surfit_openPage surfit_feedback surfit_pageStatus

    # Allocate unique name for this hyperdoc window
    set t .hyperdoc[incr surfit_win_num]

    toplevel $t
    wm title $t "Hyperdocument $surfit_win_num"
    wm iconname $t "Hyperdoc $surfit_win_num"
    set hyperpage [SurfIt_hyperpage $t]
    set surfit_url_history($hyperpage) {}
    set surfit_url_history_idx($hyperpage) -1
    set surfit_open_vars($hyperpage) {}
    set surfit_currPage($hyperpage) {}
    set surfit_openPage($hyperpage) {}
    set surfit_pageStatus($hyperpage,busy) 0

    set p [frame $t.top-ctl-panel]

    button $p.dismiss -text "Dismiss" -command "surfit_destroy_window $t"
    pack $p.dismiss -side left
    button $p.reload -text "Reload" -command "
	Cache_expire_document \$surfit_currPage($hyperpage)
	SurfIt_loadURL \$surfit_currPage($hyperpage) $hyperpage
    " -state disabled
    pack $p.reload -side left
    button $p.source -text "Source" -command "surfit_load_source $t" \
	-state disabled
    pack $p.source -side left
    button $p.stop -text "Stop" -command "
	surfit_terminate_loading $hyperpage
	set surfit_pageStatus($hyperpage,busy) 0
	catch {unset surfit_pageStatus($hyperpage,pending)}
	set surfit_feedback($hyperpage) {}
    "
    pack $p.stop -side left
    menubutton $p.applets -text "Applets" -relief raised -menu $p.applets.m
    menu $p.applets.m
    $p.applets.m add command -label "Forms" -state disabled
    $p.applets.m add command -label "Page" -state disabled
    $p.applets.m add command -label "Window" -state disabled
    pack $p.applets -side left -anchor w

    pack $p -side top -anchor nw -fill x

    set p [frame $t.second-ctl-panel]

    button $p.start -text "Start" -command "
	set surfit_url_history_idx($hyperpage) 0
	SurfIt_loadURL \[lindex \$surfit_url_history($hyperpage) 0\] $hyperpage
    " -state disabled
    pack $p.start -side left
    button $p.back -text "Back" -command "
	incr surfit_url_history_idx($hyperpage) -1
	SurfIt_loadURL \[lindex \$surfit_url_history($hyperpage) \$surfit_url_history_idx($hyperpage)\] $hyperpage
    " -state disabled
    pack $p.back -side left
    button $p.forward -text "Forward" -command "
	incr surfit_url_history_idx($hyperpage)
	SurfIt_loadURL \[lindex \$surfit_url_history($hyperpage) \$surfit_url_history_idx($hyperpage)\] $hyperpage
    " -state disabled
    pack $p.forward -side left

    pack $p -side top -anchor nw -fill x

    # Create the information panel
 
    set p [frame $t.info-panel]

    set q [frame $p.currPage]
    frame $q.label -borderwidth 0
    pack $q.label -side left -anchor e
    pack propagate $q.label 0
    button $q.label.l -text "Current Page:" \
	-state disabled -relief flat -borderwidth 0 \
	-disabledforeground black
    pack $q.label.l -fill both -expand yes
    $q.label configure -width [set currPageWidth [winfo reqwidth $q.label.l]] \
	-height [winfo reqheight $q.label.l]
    entry $q.url -textvariable surfit_openPage($hyperpage) -relief raised -border 2
    bind $q.url <Key> "surfit_enable_openlab $q"
    bind $q.url <Control-Key-u> "set surfit_openPage($hyperpage) {}"
    pack $q.url -side left -fill both -expand yes -anchor nw
    pack $q -side top -fill x

    set q [frame $p.highlightURL]
    frame $q.label -width $currPageWidth -borderwidth 0
    pack $q.label -side left -anchor e
    pack propagate $q.label 0
    label $q.label.l -text "URL: "
    pack $q.label.l -fill both -expand yes
    $q.label configure -height [winfo reqheight $q.label.l]
    label $q.url -textvariable surfit_highlightURL($hyperpage) -relief raised -border 2
    pack $q.url -side left -fill x -expand yes -anchor nw
    pack $q -side top -fill x

    pack $p -side top -anchor nw -fill x

    # Create the banner panel

    text $t.banner

    # Create the bottom feedback panel

    set p [frame $t.bottom-panel]

    label $p.feedback -textvariable surfit_feedback($hyperpage) \
	-relief raised -border 2 -anchor nw
    pack $p.feedback -side left -anchor nw -fill both -expand yes

    pack $p -side bottom -anchor sw -fill x

    # Create the main text widget for the hyperdocument
    # If the widget path changes, be sure to update the SurfIt_hyperpage procedure
 
    set p [frame $t.main]
    if {$hyperpage != [text $p.text \
	-xscrollcommand "$p.xscroll set" \
	-yscrollcommand "$p.f.yscroll set"]} {
	# Sanity check
	error "internal error: hyperpage pathname incorrect"
    }

    # Initialise the text widgets for HTML parsing & rendering
    HMinit_win $hyperpage
    HMset_state $hyperpage -symbols \154\155\147\151\152\153\121\122\123\124\125\126\127\130\131\132\133\134\135\136

    # Initialise transparent GIF handling
    global TRANSPARENT_GIF_COLOR; set TRANSPARENT_GIF_COLOR [$hyperpage cget -background]

    scrollbar $p.xscroll -command "$p.text xview" -orient horizontal
    frame $p.f
    frame $p.f.filler -height [expr [lindex [$p.xscroll configure -width] 4] + 10]
    scrollbar $p.f.yscroll -command "$p.text yview"
    $p.f.filler configure -width [lindex [$p.f.yscroll configure -width] 4]
    pack $p.f.filler -side bottom
    pack $p.f.yscroll -side top -fill y -expand yes
    pack $p.f -side right -fill y
    pack $p.xscroll -side bottom -fill x
    pack $p.text -side left -fill both -expand yes
 
    pack $p -side top -fill both -expand yes

    # Explicitly setting the size locks in the size of the info-panel widgets
    bind lockGeom <Visibility> \
	"wm geometry $t \[wm geometry $t\]; bindtags $t \[lrange \[bindtags $t\] 1 end\]"
    bindtags $t "lockGeom [bindtags $t]"

    # If an initial URL was specified, load it now
    if {$open_url != {}} {
	incr surfit_url_history_idx($hyperpage)
	SurfIt_loadURL $open_url $hyperpage
    }

    return $t
}

# Given a window, return the pathname for the text widget in the hyperwindow.
# This pathname is used in many places, so it is critical to get it right!

proc SurfIt_hyperpage {t} {
    return [winfo toplevel $t].main.text
}

# Destroy given window and other storage associated with it
proc surfit_destroy_window {t} {
    global surfit_url_history surfit_url_history_idx surfit_open_vars surfit_currPage

    set hyperpage [SurfIt_hyperpage $t]
    unset surfit_url_history($hyperpage)
    unset surfit_url_history_idx($hyperpage)
    unset surfit_open_vars($hyperpage)
    unset surfit_currPage($hyperpage)
    Applet_destroy_window $hyperpage
    # HMreset_state ...
    # unset state...
    destroy $t
}

proc surfit_load_source {win} {
    global surfit_url_history surfit_url_history_idx surfit_open_vars
    global surfit_currPage

    set hyperpage [SurfIt_hyperpage $win]
    SurfIt_loadURL $surfit_currPage($hyperpage) $hyperpage text/plain
    $win.top-ctl-panel.source configure -text "Render" -command "
	SurfIt_loadURL \$surfit_currPage($hyperpage) $hyperpage
	$win.top-ctl-panel.source configure -text {Source} \
	    -command \"surfit_load_source $win\"
    "
}

proc surfit_enable_openlab {q} {
    set hyperpage [SurfIt_hyperpage $q]

    bind $q.url <Key> {}
    bind $q.url <Key-Return> "
	incr surfit_url_history_idx($hyperpage)
	SurfIt_loadURL \$surfit_openPage($hyperpage) $hyperpage
    "

    $q.label.l configure -state normal -relief raised -borderwidth 3 \
	-text "Open:" -command "
	    incr surfit_url_history_idx($hyperpage)
	    SurfIt_loadURL \$surfit_openPage($hyperpage) $hyperpage
	"
}

proc surfit_setOpenPage {win url} {
    global surfit_currPage surfit_openPage

    set hyperpage [SurfIt_hyperpage $win]
    set q [winfo toplevel $win].info-panel.currPage
    set surfit_currPage($hyperpage) [set surfit_openPage($hyperpage) $url]

    $q.label.l configure -text "Current Page:" -state disabled \
	-relief flat -borderwidth 0
    bind $q.url <Key-Return> {}
    bind $q.url <Key> "surfit_enable_openlab $q"
}
