
/*
 * bltPictureImage.c --
 *
 * This module implements the Tk image interface for picture images.
 *
 *	Copyright 2003-2004 George A Howlett.
 *
 *	Permission is hereby granted, free of charge, to any person
 *	obtaining a copy of this software and associated documentation
 *	files (the "Software"), to deal in the Software without
 *	restriction, including without limitation the rights to use,
 *	copy, modify, merge, publish, distribute, sublicense, and/or
 *	sell copies of the Software, and to permit persons to whom the
 *	Software is furnished to do so, subject to the following
 *	conditions:
 *
 *	The above copyright notice and this permission notice shall be
 *	included in all copies or substantial portions of the
 *	Software.
 *
 *	THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY
 *	KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 *	WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 *	PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS
 *	OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
 *	OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 *	OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 *	SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include "bltInt.h"
#include <bltHash.h>
#include <bltSink.h>
#include "bltSwitch.h"
#include "bltPicture.h"
#include "bltPictureFormats.h"
#include "bltPainter.h"

#include "bltPs.h"
#include <X11/Xutil.h>

#include <sys/types.h>
#include <sys/stat.h>

typedef struct Blt_DataSinkStruct DataSink;

extern Tcl_ObjCmdProc Blt_PictureDrawOp;

/*
 * Default configuration options for picture images. 
 */
#define DEF_PICTURE_ANGLE	"0.0"
#define DEF_PICTURE_ASPECT	"yes"
#define DEF_PICTURE_CACHE	"no"
#define DEF_PICTURE_DITHER	"no"
#define DEF_PICTURE_DATA	(char *)NULL
#define DEF_PICTURE_FILE	(char *)NULL
#define DEF_PICTURE_FILTER	(char *)NULL
#define DEF_PICTURE_GAMMA	"1.0"
#define DEF_PICTURE_HEIGHT	"0"
#define DEF_PICTURE_WIDTH	"0"
#define DEF_PICTURE_WINDOW	(char *)NULL
#define DEF_PICTURE_IMAGE	(char *)NULL
#define DEF_PICTURE_SHARPEN	"no"

#define DEF_PICTURE_OPAQUE	"no"
#define DEF_PICTURE_OPAQUE_BACKGROUND	"white"

#define IMPORTED_NONE		0
#define IMPORTED_FILE		(1<<0)
#define IMPORTED_IMAGE		(1<<1)
#define IMPORTED_WINDOW		(1<<2)
#define IMPORTED_DATA		(1<<3)
#define IMPORTED_MASK	\
	(IMPORTED_FILE|IMPORTED_IMAGE|IMPORTED_WINDOW|IMPORTED_DATA)
/*
 * Various external file/string formats handled by the picture image.
 */
enum PictureExternalFormats {
    PICTURE_XFMT_JPG, /* Read/Write */
    PICTURE_XFMT_PNG, /* Read/Write */
    PICTURE_XFMT_TIF, /* Read/Write */
    PICTURE_XFMT_XPM, /* Read/Write */
    PICTURE_XFMT_XBM, /* Read/Write */
    PICTURE_XFMT_GIF, /* Read */
    PICTURE_XFMT_EPS, /* Write */
    PICTURE_XFMT_PDF, /* Write */
    PICTURE_XFMT_PHOTO, /* Read/Write */
    PICTURE_XFMT_BMP, /* Read/Write (Windows only) */
    PICTURE_XFMT_EMF, /* Read/Write (Windows only) */
    PICTURE_XFMT_WMF, /* Read/Write (Windows only) */
    PICTURE_XFMT_NUMBER_FORMATS
};

/*
 * A PictureImage implements a Tk_ImageMaster for the "picture" image
 * type.  It represents a set of bits (i.e. the picture), some
 * options, and operations (sub-commands) to manipulate the picture.
 *
 * The PictureImage manages the Tcl interface to a picture (using
 * Tk's "image" command).  Pictures and the mechanics of drawing the
 * picture (painters) are orthogonal.  The PictureImage knows nothing
 * about the display type (the display is stored only to free options
 * when it's destroyed).  Information specific to the visual context
 * (combination of display, visual, depth, colormap, and gamma) is
 * store in each cache entry.  The picture image manages the various
 * picture transformations: reading, writing, scaling, rotation, etc.
 */
typedef struct Blt_PictureImageStruct {
    Tk_ImageMaster imgToken;	/* Tk's token for image master.  If
				 * NULL, the image has currently been
				 * deleted. */

    Tcl_Interp *interp;		/* Interpreter associated with the
				 * application using this image. */

    Display *display;		/* Display associated with this
				 * picture image.  This is used
				 * to free configuration options */
    Colormap colormap;

    Tcl_Command cmdToken;	/* Token for image command (used to
				 * delete it when the image goes
				 * away).  NULL means the image
				 * command has already been
				 * deleted. */

    int	flags;			/* Various bit-field flags defined
				   below. */

    Blt_Picture picture;	/* Picture to be displayed. */
    
    /* User-options. */
    double angle;		/* Angle in degrees to rotate the image. */

    int reqWidth, reqHeight;	/* User-requested size of picture. The
				 * picture is scaled accordingly.
				 * These dimensions may or may not be
				 * used, depending upon the -aspect
				 * option. */

    int aspect;			/* If non-zero, indicates to maintain
				 * the minimum aspect ratio while
				 * scaling. The larger dimension is
				 * discarded. */

    int dither;			/* If non-zero, dither the picture 
				 * before drawing. */
    int sharpen;		/* If non-zero, indicates to sharpen
				 * the image. */

    Blt_PictureFilter filter;	/* Filter to use when the picture is
				 * resampled (resized). The same
				 * filter is applied both horizontally
				 * and vertically. */

    double gamma;		/* Gamma correction value of
				 * monitor. In theory, the same
				 * picture image may be displayed on
				 * two monitors simultaneously (using
				 * xinerama).  Here we assume both
				 * monitors will have the same gamma
				 * value. */

    char *name;			/* Name of the image, file, or window
				 * read into the picture image. */

    int readFormat;		/* External format of last image read
				 * into the picture image. */

    int doCache;		/* If non-zero, indicates to generate
				 * a pixmap of the picture and to use
				 * that for redrawing. */

    Blt_HashTable cacheTable;	/* Table of cache entries specific to
				 * each visual context where this
				 * picture is displayed. */

} PictureImage;


/*
 * A PictureCacheKey represents the visual context of a cache
 * entry. type.  It contains information specific to the visual
 * context (combination of display, visual, depth, colormap, and
 * gamma).  It is used as a hash table key for cache entries of
 * picture images.  The same picture may be displayed in more than
 * one visual context.
 */
typedef struct {
    Display *display;		/* Display where the picture will be
				 * drawn. Used to free colors
				 * allocated by the painter. */

    Visual *visualPtr;		/* Visual information for window displaying
				 * the image. */

    Colormap colormap;		/* Colormap used.  This may be the default
				 * colormap, or an allocated private map. */

    int depth;			/* Depth of the display. */

    double gamma;		/* Gamma correction value */
} PictureCacheKey;


/*
 * PictureCaches (image instances in the Tk parlance) represent a
 * picture image in some specific combination of visual, display,
 * colormap, depth, and output gamma.  Cache entries are stored by
 * each picture image.
 *
 * The purpose is to 1) allocated and hold the "painter" specific the
 * visual and 2) provide caching of XImage's (drawn pictures) into
 * pixmaps.  This feature is enabled only for 100% opaque pictures.
 * If the picture must be blended with the current background, there
 * can be no guarantee between redraws that the background will not
 * have changed.  This feature is widget specific. There's no simple
 * way to detect when the pixmap must be redrawn.  In general, we
 * should rely on the widget itself to perform its own caching of
 * complex scenes.
 */
typedef struct {
    Blt_PictureImage image;	/* The picture image represented by
				 * this cache entry. */

    Blt_Painter painter;	/* The painter allocated for this
				 * particular combination of visual,
				 * display, colormap, depth, and
				 * gamma. */

    Display *display;		/* Used to free the pixmap below when
				 * the entry is destroyed. */

    Blt_HashEntry *hashPtr;	/* These two fields allow the cache */
    Blt_HashTable *tablePtr;	/* entry to be deleted from the picture
				 * image's table of entries. */

    Pixmap pixmap;		/* If non-NULL, is a cached pixmap of
				 * the picture. It's recreated each
				 * time the picture changes. */

    int refCount;		/* This entry may be shared by all
				 * clients displaying this picture
				 * image with the same painter. */
} PictureCache;


static Blt_OptionParseProc ObjToFile;
static Blt_OptionPrintProc FileToObj;
static Blt_CustomOption fileOption =
{
    ObjToFile, FileToObj, NULL, (ClientData)0
};

static Blt_OptionParseProc ObjToData;
static Blt_OptionPrintProc DataToObj;
static Blt_CustomOption dataOption =
{
    ObjToData, DataToObj, NULL, (ClientData)0
};

static Blt_OptionParseProc ObjToFilter;
static Blt_OptionPrintProc FilterToObj;
static Blt_CustomOption filterOption =
{
    ObjToFilter, FilterToObj, NULL, (ClientData)0
};

static Blt_OptionParseProc ObjToImage;
static Blt_OptionPrintProc ImageToObj;
static Blt_CustomOption imageOption =
{
    ObjToImage, ImageToObj, NULL, (ClientData)0
};

static Blt_OptionParseProc ObjToWindow;
static Blt_OptionPrintProc WindowToObj;
static Blt_CustomOption windowOption =
{
    ObjToWindow, WindowToObj, NULL, (ClientData)0
};

static Blt_ConfigSpec configSpecs[] =
{
    {BLT_CONFIG_BOOLEAN, "-aspect", (char *)NULL, (char *)NULL, 
	DEF_PICTURE_ASPECT, Blt_Offset(PictureImage, aspect), 
        BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_CUSTOM, "-data", (char *)NULL, (char *)NULL, DEF_PICTURE_DATA, 
	Blt_Offset(PictureImage, picture), 0, &dataOption},
    {BLT_CONFIG_BOOLEAN, "-dither", (char *)NULL, (char *)NULL, 
	DEF_PICTURE_DITHER, Blt_Offset(PictureImage, dither), 
        BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_CUSTOM, "-file", (char *)NULL, (char *)NULL, DEF_PICTURE_DATA, 
	Blt_Offset(PictureImage, picture), 0, &fileOption},
    {BLT_CONFIG_CUSTOM, "-filter", (char *)NULL, (char *)NULL, 
	DEF_PICTURE_FILTER, Blt_Offset(PictureImage, filter), 0, 
        &filterOption},
    {BLT_CONFIG_DOUBLE, "-gamma", (char *)NULL, (char *)NULL, DEF_PICTURE_GAMMA,
	Blt_Offset(PictureImage, gamma), BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_INT_NONNEGATIVE, "-height", (char *)NULL, (char *)NULL,
	DEF_PICTURE_HEIGHT, Blt_Offset(PictureImage, reqHeight), 0},
    {BLT_CONFIG_CUSTOM, "-image", (char *)NULL, (char *)NULL, DEF_PICTURE_IMAGE,
	Blt_Offset(PictureImage, picture), 0, &imageOption},
    {BLT_CONFIG_DOUBLE, "-rotate", (char *)NULL, (char *)NULL, 
	DEF_PICTURE_ANGLE, Blt_Offset(PictureImage, angle), 
	BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_BOOLEAN, "-sharpen", (char *)NULL, (char *)NULL, 
	DEF_PICTURE_SHARPEN, Blt_Offset(PictureImage, sharpen), 
        BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_INT_NONNEGATIVE, "-width", (char *)NULL, (char *)NULL,
	DEF_PICTURE_WIDTH, Blt_Offset(PictureImage, reqWidth), 0},
    {BLT_CONFIG_CUSTOM, "-window", (char *)NULL, (char *)NULL, 
	DEF_PICTURE_WINDOW, Blt_Offset(PictureImage, picture), 0, 
	&windowOption},
    {BLT_CONFIG_END, NULL, NULL, NULL, NULL, 0, 0}
};

extern Blt_SwitchCustom bltColorSwitch;

static Blt_SwitchParseProc BBoxSwitch;
static Blt_SwitchCustom bboxSwitch = {
    BBoxSwitch, NULL, (ClientData)0,
};

static Blt_SwitchParseProc FilterSwitch;
static Blt_SwitchCustom filterSwitch = {
    FilterSwitch, NULL, (ClientData)0,
};

static Blt_SwitchParseProc CompressSwitch;
static Blt_SwitchCustom compressSwitch = {
    CompressSwitch, NULL, (ClientData)0,
};

static Blt_SwitchParseProc BlendingModeSwitch;
static Blt_SwitchCustom blendModeSwitch = {
    BlendingModeSwitch, NULL, (ClientData)0,
};

static Blt_SwitchParseProc ShapeSwitch;
static Blt_SwitchCustom shapeSwitch = {
    ShapeSwitch, NULL, (ClientData)0,
};

static Blt_SwitchParseProc PathSwitch;
static Blt_SwitchCustom pathSwitch = {
    PathSwitch, NULL, (ClientData)0,
};

static Blt_SwitchSpec jpgExportSwitches[] = 
{
    {BLT_SWITCH_CUSTOM, "-bg", Blt_Offset(Blt_PictureExportSwitches, bg), 0, 
	&bltColorSwitch},
    {BLT_SWITCH_OBJ, "-data", 
	Blt_Offset(Blt_PictureExportSwitches, dataObjPtr), 0},
    {BLT_SWITCH_OBJ, "-file", 
	Blt_Offset(Blt_PictureExportSwitches, fileObjPtr), 0},
    {BLT_SWITCH_INT_NONNEGATIVE, "-quality", 
	Blt_Offset(Blt_PictureExportSwitches, quality), 0},
    {BLT_SWITCH_INT_NONNEGATIVE, "-smooth", 
	Blt_Offset(Blt_PictureExportSwitches, smoothing), 0},
    {BLT_SWITCH_FLAG, "-progressive", 
	Blt_Offset(Blt_PictureExportSwitches, flags), 0,
	(ClientData)PICTURE_PROGRESSIVE},
    {BLT_SWITCH_END, NULL, 0, 0}
};

static Blt_SwitchSpec tifExportSwitches[] = 
{
    {BLT_SWITCH_CUSTOM, "-compress", 
	Blt_Offset(Blt_PictureExportSwitches, compress), 0, &compressSwitch},
    {BLT_SWITCH_OBJ, "-data", 
	Blt_Offset(Blt_PictureExportSwitches, dataObjPtr), 0},
    {BLT_SWITCH_OBJ, "-file", 
	Blt_Offset(Blt_PictureExportSwitches, fileObjPtr), 0},
    {BLT_SWITCH_END, NULL, 0, 0}
};

static Blt_SwitchSpec photoExportSwitches[] = 
{
    {BLT_SWITCH_OBJ, "-image", 
	Blt_Offset(Blt_PictureExportSwitches, imageObjPtr), 0},
    {BLT_SWITCH_END, NULL, 0, 0}
};

static Blt_SwitchSpec pngExportSwitches[] = 
{
    {BLT_SWITCH_OBJ, "-data", 
	Blt_Offset(Blt_PictureExportSwitches, dataObjPtr), 0},
    {BLT_SWITCH_OBJ, "-file", 
	Blt_Offset(Blt_PictureExportSwitches, fileObjPtr), 0},
    {BLT_SWITCH_END, NULL, 0, 0}
};

static Blt_SwitchSpec xpmExportSwitches[] = 
{
    {BLT_SWITCH_CUSTOM, "-bg", Blt_Offset(Blt_PictureExportSwitches, bg), 0, 
	&bltColorSwitch},
    {BLT_SWITCH_OBJ, "-data", 
	Blt_Offset(Blt_PictureExportSwitches, dataObjPtr), 0},
    {BLT_SWITCH_OBJ, "-file", 
	Blt_Offset(Blt_PictureExportSwitches, fileObjPtr), 0},
    {BLT_SWITCH_FLAG, "-noquantize", 
	Blt_Offset(Blt_PictureExportSwitches, flags), 0, 
	(ClientData)PICTURE_NO_QUANTIZE},
    {BLT_SWITCH_END, NULL, 0, 0}
};

static Blt_SwitchSpec stdExportSwitches[] = 
{
    {BLT_SWITCH_OBJ, "-data", 
	Blt_Offset(Blt_PictureExportSwitches, dataObjPtr), 0},
    {BLT_SWITCH_OBJ, "-file", 
	Blt_Offset(Blt_PictureExportSwitches, fileObjPtr), 0},
    {BLT_SWITCH_END, NULL, 0, 0}
};

static Blt_SwitchSpec xbmExportSwitches[] = 
{
    {BLT_SWITCH_CUSTOM, "-bg", Blt_Offset(Blt_PictureExportSwitches, bg), 0, 
	&bltColorSwitch},
    {BLT_SWITCH_OBJ, "-data", 
	Blt_Offset(Blt_PictureExportSwitches, dataObjPtr), 0},
    {BLT_SWITCH_OBJ, "-file", 
	Blt_Offset(Blt_PictureExportSwitches, fileObjPtr), 0},
    {BLT_SWITCH_END, NULL, 0, 0}
};


static Blt_SwitchSpec *exportSwitches[] = {
    jpgExportSwitches,		/* JPEG */
    pngExportSwitches,		/* PNG */
    tifExportSwitches,		/* TIF */
    xpmExportSwitches,		/* XPM */
    xbmExportSwitches,		/* XBM */
    NULL,			/* GIF */
    stdExportSwitches,		/* EPS */
    stdExportSwitches,		/* PDF */
    photoExportSwitches,	/* PHOTO */
#ifdef WIN32
    stdExportSwitches,		/* BMP */
    stdExportSwitches,		/* EMF */
    stdExportSwitches,		/* WMF */
#endif
};

typedef struct {
    Pix32 bg, fg;	/* Fg and bg colors. */
    Blt_Gradient gradient;
} GradientSwitches;

static Blt_SwitchSpec gradientSwitches[] = 
{
    {BLT_SWITCH_CUSTOM, "-direction", 
	Blt_Offset(GradientSwitches, gradient.path), 0, &pathSwitch},
    {BLT_SWITCH_CUSTOM, "-shape", Blt_Offset(GradientSwitches, gradient.shape),
	0, &shapeSwitch},
    {BLT_SWITCH_CUSTOM, "-high", Blt_Offset(GradientSwitches, fg), 0, 
	&bltColorSwitch},
    {BLT_SWITCH_CUSTOM, "-low", Blt_Offset(GradientSwitches, bg), 0, 
	&bltColorSwitch},
    {BLT_SWITCH_BOOLEAN, "-logscale", 
	Blt_Offset(GradientSwitches, gradient.logScale),0},
    {BLT_SWITCH_BOOLEAN, "-jitter", 
	Blt_Offset(GradientSwitches, gradient.jitter),0},
    {BLT_SWITCH_END, NULL, 0, 0}
};

static Blt_SwitchSpec jpgImportSwitches[] = 
{
    {BLT_SWITCH_OBJ, "-data", 
	Blt_Offset(Blt_PictureImportSwitches, dataObjPtr), 0},
    {BLT_SWITCH_INT, "-fast", 
	Blt_Offset(Blt_PictureImportSwitches, fast), 0},
    {BLT_SWITCH_OBJ, "-file", 
	Blt_Offset(Blt_PictureImportSwitches, fileObjPtr), 0},
    {BLT_SWITCH_END, NULL, 0, 0}
};

static Blt_SwitchSpec gifImportSwitches[] = 
{
    {BLT_SWITCH_OBJ, "-data", 
	Blt_Offset(Blt_PictureImportSwitches, dataObjPtr), 0},
    {BLT_SWITCH_OBJ, "-file", 
	Blt_Offset(Blt_PictureImportSwitches, fileObjPtr), 0},
    {BLT_SWITCH_INT, "-index", 
	Blt_Offset(Blt_PictureImportSwitches, imageIndex), 0},
    {BLT_SWITCH_END, NULL, 0, 0}
};

static Blt_SwitchSpec photoImportSwitches[] = 
{
    {BLT_SWITCH_OBJ, "-image", 
	Blt_Offset(Blt_PictureImportSwitches, imageObjPtr), 0},
    {BLT_SWITCH_END, NULL, 0, 0}
};

static Blt_SwitchSpec xbmImportSwitches[] = 
{
    {BLT_SWITCH_CUSTOM, "-bg", Blt_Offset(Blt_PictureImportSwitches, bg), 0, 
	&bltColorSwitch},
    {BLT_SWITCH_OBJ, "-data", 
	Blt_Offset(Blt_PictureImportSwitches, dataObjPtr), 0},
    {BLT_SWITCH_CUSTOM, "-fg", Blt_Offset(Blt_PictureImportSwitches, fg), 0, 
	&bltColorSwitch},
    {BLT_SWITCH_OBJ, "-file", 
	Blt_Offset(Blt_PictureImportSwitches, fileObjPtr), 0},
    {BLT_SWITCH_END, NULL, 0, 0}
};

static Blt_SwitchSpec stdImportSwitches[] = 
{
    {BLT_SWITCH_OBJ, "-data", 
	Blt_Offset(Blt_PictureImportSwitches, dataObjPtr), 0},
    {BLT_SWITCH_OBJ, "-file", 
	Blt_Offset(Blt_PictureImportSwitches, fileObjPtr), 0},
    {BLT_SWITCH_END, NULL, 0, 0}
};

static Blt_SwitchSpec *importSwitches[] = {
    jpgImportSwitches,		/* JPEG */
    stdImportSwitches,		/* PNG */
    stdImportSwitches,		/* TIF */
    stdImportSwitches,		/* XPM */
    xbmImportSwitches,		/* XBM */
    gifImportSwitches,		/* GIF */
    NULL,			/* EPS */
    NULL,			/* PDF */
    photoImportSwitches,	/* PHOTO */
#ifdef WIN32
    NULL,			/* BMP */
    NULL,			/* EMF */
    NULL,			/* WMF */
#endif
};

typedef struct {
    BBox bbox;			/* Area to tile. */
    int xOrigin;
    int yOrigin;
} TileSwitches;

static Blt_SwitchSpec tileSwitches[] = 
{
    {BLT_SWITCH_CUSTOM, "-region",  Blt_Offset(TileSwitches, bbox), 0, 
	&bboxSwitch},
    {BLT_SWITCH_INT,    "-xorigin", Blt_Offset(TileSwitches, xOrigin),  0},
    {BLT_SWITCH_INT,    "-yorigin", Blt_Offset(TileSwitches, yOrigin),  0},
    {BLT_SWITCH_END, NULL, 0, 0}
};

typedef struct {
    int invert;			/* Flag. */
    Blt_Picture matte;
} ArithSwitches;

static Blt_SwitchSpec arithSwitches[] = 
{
    {BLT_SWITCH_OBJ, "-matte", Blt_Offset(ArithSwitches, matte), 0},
    {BLT_SWITCH_OBJ, "-invertmatte", Blt_Offset(ArithSwitches, invert), 0},
    {BLT_SWITCH_END, NULL, 0, 0}
};

typedef struct {
    BBox bbox;			/* Area to crop. */
    int nocopy;			/* If non-zero, don't copy the source image. */
} DupSwitches;

#define DUP_NOCOPY	1

static Blt_SwitchSpec dupSwitches[] = 
{
    {BLT_SWITCH_FLAG, "-nocopy", Blt_Offset(DupSwitches, nocopy), 0, NULL, 
	DUP_NOCOPY},
    {BLT_SWITCH_CUSTOM, "-region", Blt_Offset(DupSwitches, bbox), 0, 
	&bboxSwitch},
    {BLT_SWITCH_END, NULL, 0, 0}
};

typedef struct {
    Blt_BlendingMode mode;		/* Blending mode. */
} BlendSwitches;

static Blt_SwitchSpec blendSwitches[] = 
{
    {BLT_SWITCH_CUSTOM, "-mode", Blt_Offset(BlendSwitches, mode), 0, 
	&blendModeSwitch},
    {BLT_SWITCH_END, NULL, 0, 0}
};

typedef struct {
    Blt_PictureFilter vFilter;		/* Color of rectangle. */
    Blt_PictureFilter hFilter;		/* Width of outline. */
    Blt_PictureFilter filter;
} ConvolveSwitches;

static Blt_SwitchSpec convolveSwitches[] = 
{
    {BLT_SWITCH_CUSTOM, "-filter", Blt_Offset(ConvolveSwitches, filter), 0, 
	&filterSwitch},
    {BLT_SWITCH_CUSTOM, "-hfilter", Blt_Offset(ConvolveSwitches, hFilter), 0, 
	&filterSwitch},
    {BLT_SWITCH_CUSTOM, "-vfilter", Blt_Offset(ConvolveSwitches, vFilter), 0, 
	&filterSwitch},
    {BLT_SWITCH_END, NULL, 0, 0}
};

typedef struct {
    BBox from;
    BBox to;
} CopySwitches;

static Blt_SwitchSpec copySwitches[] = 
{
    {BLT_SWITCH_CUSTOM, "-from", Blt_Offset(CopySwitches,from), 0, &bboxSwitch},
    {BLT_SWITCH_CUSTOM, "-to", Blt_Offset(CopySwitches, to), 0, &bboxSwitch},
    {BLT_SWITCH_END, NULL, 0, 0}
};

typedef struct {
    Blt_PictureFilter filter;
    BBox bbox;
} ResampleSwitches;

static Blt_SwitchSpec resampleSwitches[] = 
{
    {BLT_SWITCH_CUSTOM, "-filter", Blt_Offset(ResampleSwitches, filter), 0, 
	&filterSwitch},
    {BLT_SWITCH_CUSTOM, "-from",  Blt_Offset(ResampleSwitches, bbox), 0, 
	&bboxSwitch},
    {BLT_SWITCH_END, NULL, 0, 0}
};

/* 
 * Forward references for Tcl command callbacks used below. 
 */
static Tcl_ObjCmdProc PictureInstCmdProc;
static Tcl_CmdDeleteProc PictureInstCmdDeletedProc;

Blt_Picture
Blt_PictureFromImage(PictureImage *imgPtr)
{
    return imgPtr->picture;
}

void
Blt_NotifyImageChanged(PictureImage *imgPtr)
{
    int width, height;

    width = Blt_PictureWidth(imgPtr->picture);
    height = Blt_PictureHeight(imgPtr->picture);
    Tk_ImageChanged(imgPtr->imgToken, 0, 0, width, height, width, height);
}

Pix32 
Blt_XColorToPix32(XColor *colorPtr)
{
    Pix32 new;

    new.Red = colorPtr->red / 257;
    new.Green = colorPtr->green / 257;
    new.Blue = colorPtr->blue / 257;
    new.Alpha = ALPHA_OPAQUE;
    return new;
}

int
Blt_GetPix32(
    Tcl_Interp *interp, 
    char *string, 
    Pix32 *pixelPtr)
{
    int length;

    length = strlen(string);
    if ((string[0] == '0') && (string[1] == 'x')) {
	unsigned int value;
	char *term;

	value = strtoul(string + 2, &term, 16); 
	if ((term == (string + 1)) || (*term != '\0')) {
	    Tcl_AppendResult(interp, "expected color value but got \"", string,
		"\"", (char *) NULL);
	    return TCL_ERROR;
	}
	pixelPtr->color = value;
    } else {
    	XColor color;
	Tk_Window tkwin;

	tkwin = Tk_MainWindow(interp);
	if (!XParseColor(Tk_Display(tkwin), Tk_Colormap(tkwin), string, 
		&color)) {
	    Tcl_AppendResult(interp, "unknown color name \"", string, "\"",
			     (char *) NULL);
	    return TCL_ERROR;
	}
	*pixelPtr = Blt_XColorToPix32(&color);
    }  
    return TCL_OK;
}

int
Blt_GetPix32FromObj(
    Tcl_Interp *interp, 
    Tcl_Obj *objPtr, 
    Pix32 *pixelPtr)
{
    char *string;
    
    string = Tcl_GetString(objPtr);
    return Blt_GetPix32(interp, string, pixelPtr);
}

int
Blt_GetBBoxFromObjv(
    Tcl_Interp *interp, 
    int objc,
    Tcl_Obj *CONST *objv, 
    BBox *bbPtr)
{
    int left, top, right, bottom;

    if ((objc != 2) && (objc != 4)) {
	Tcl_AppendResult(interp, "wrong # elements in bounding box ", 
		(char *)NULL);
	return TCL_ERROR;
    }
    bbPtr->x = bbPtr->y = bbPtr->width = bbPtr->height = 0;
    if ((Tcl_GetIntFromObj(interp, objv[0], &left) != TCL_OK) ||
	(Tcl_GetIntFromObj(interp, objv[1], &top) != TCL_OK)) {
	return TCL_ERROR;
    }
    bbPtr->x = left, bbPtr->y = top;
    if (objc == 2) {
	return TCL_OK;
    }
    if ((Tcl_GetIntFromObj(interp, objv[2], &right) != TCL_OK) ||
	(Tcl_GetIntFromObj(interp, objv[3], &bottom) != TCL_OK)) {
	return TCL_ERROR;
    }

    /* Flip the coordinates of the bounding box if necessary so that
     * its the upper-left and lower-right corners */
    if (left > right) {
	int tmp;

	tmp = left, left = right, right = tmp;
    }
    if (top > bottom) {
	int tmp;

	tmp = top, top = bottom, bottom = tmp;
    }
    bbPtr->width = right - left + 1;
    bbPtr->height = bottom - top + 1;
    return TCL_OK;
}

static int
GetBBoxFromObj(
    Tcl_Interp *interp, 
    Tcl_Obj *objPtr, 
    BBox *bbPtr)
{
    int objc;
    Tcl_Obj **objv;

    if (Tcl_ListObjGetElements(interp, objPtr, &objc, &objv) != TCL_OK) {
	return TCL_ERROR;
    }
    return Blt_GetBBoxFromObjv(interp, objc, objv, bbPtr);
}

int
Blt_AdjustBBoxToPicture(Blt_Picture picture, BBox *bbPtr)
{
    int pictWidth, pictHeight;

    pictWidth = Blt_PictureWidth(picture);
    pictHeight = Blt_PictureHeight(picture);

    if ((bbPtr->width == 0) || (bbPtr->width > pictWidth)) {
	bbPtr->width = pictWidth;
    }
    if ((bbPtr->height == 0) || (bbPtr->height > pictHeight)) {
	bbPtr->height = pictHeight;
    }

    /* Verify that some part of the bounding box is actually inside
     * the picture. */
    if ((bbPtr->x >= pictWidth)  || ((bbPtr->x + bbPtr->width) <= 0) ||
	(bbPtr->y >= pictHeight) || ((bbPtr->y + bbPtr->height) <= 0)) {
	return FALSE;
    }
    /* If needed, adjust the bounding box so that it resides totally
     * inside the picture */
    if (bbPtr->x < 0) {
	bbPtr->width += bbPtr->x;
	bbPtr->x = 0;
    } 
    if (bbPtr->y < 0) {
	bbPtr->height += bbPtr->y;
	bbPtr->y = 0;
    }
    if ((bbPtr->x + bbPtr->width) > pictWidth) {
	bbPtr->width = pictWidth - bbPtr->x;
    }
    if ((bbPtr->y + bbPtr->height) > pictHeight) {
	bbPtr->height = pictHeight - bbPtr->y;
    }
    return TRUE;
}

int
Blt_ResetPicture(
    Tcl_Interp *interp,
    char *imageName,
    Blt_Picture picture)
{
    Tcl_CmdInfo cmdInfo;

    if (Tcl_GetCommandInfo(interp, imageName, &cmdInfo)) {
	if (cmdInfo.objProc == PictureInstCmdProc) {
	    PictureImage *imgPtr;

	    imgPtr = cmdInfo.objClientData;
	    if (imgPtr->picture != picture) {
		if (imgPtr->picture != NULL) {
		    Blt_FreePicture(imgPtr->picture);
		}
		imgPtr->picture = picture;
	    }
	    Blt_NotifyImageChanged(imgPtr);
	    return TCL_OK;
	}
    }
    Tcl_AppendResult(interp, "can't find picture \"", imageName, "\"", 
	(char *)NULL);
    return TCL_ERROR;
}
    
static int
Blt_ResetPictureFromObj(
    Tcl_Interp *interp,
    Tcl_Obj *objPtr,
    Blt_Picture picture)
{
    char *imageName;

    imageName = Tcl_GetString(objPtr);
    return Blt_ResetPicture(interp, imageName, picture);
}

int
Blt_GetPicture(
    Tcl_Interp *interp,
    char *string,
    Blt_Picture *picturePtr)
{
    Tcl_CmdInfo cmdInfo;

    if (Tcl_GetCommandInfo(interp, string, &cmdInfo)) {
	if (cmdInfo.objProc == PictureInstCmdProc) {
	    PictureImage *imgPtr;

	    imgPtr = cmdInfo.objClientData;
	    *picturePtr = imgPtr->picture;
	    return TCL_OK;
	}
    }
    Tcl_AppendResult(interp, "can't find picture \"", string, "\"",
		     (char *)NULL);
    return TCL_ERROR;
}

int
Blt_GetPictureFromObj(
    Tcl_Interp *interp,
    Tcl_Obj *objPtr,
    Blt_Picture *picturePtr)
{
    char *string;

    string = Tcl_GetString(objPtr);
    return Blt_GetPicture(interp, string, picturePtr);
}

static int 
StringToFormat(Tcl_Interp *interp, char *string, int *fmtPtr)
{
    char c;

    c = tolower(UCHAR(string[0]));
    if ((c == 'j') && ((strcasecmp(string, "jpg") == 0) ||
		       (strcasecmp(string, "jpeg") == 0))) {
	*fmtPtr = PICTURE_XFMT_JPG;
    } else if ((c == 'g') && (strcasecmp(string, "gif") == 0)) {
	*fmtPtr = PICTURE_XFMT_GIF;
    } else if ((c == 'e') && (strcasecmp(string, "eps") == 0)) {
	*fmtPtr = PICTURE_XFMT_EPS;
    } else if ((c == 'p') && (strcasecmp(string, "pdf") == 0)) {
	*fmtPtr = PICTURE_XFMT_PDF;
    } else if ((c == 'p') && (strcasecmp(string, "png") == 0)) {
	*fmtPtr = PICTURE_XFMT_PNG;
    } else if ((c == 't') && (strcasecmp(string, "tif") == 0)) {
	*fmtPtr = PICTURE_XFMT_TIF;
#ifdef WIN32
    } else if ((c == 'b') && (strcasecmp(string, "bmp") == 0)) {
	*fmtPtr = PICTURE_XFMT_BMP;
    } else if ((c == 'w') && (strcasecmp(string, "wmf") == 0)) {
	*fmtPtr = PICTURE_XFMT_WMF;
    } else if ((c == 'e') && (strcasecmp(string, "emf") == 0)) {
	*fmtPtr = PICTURE_XFMT_EMF;
#endif
    } else if ((c == 'x') && (strcasecmp(string, "xpm") == 0)) {
	*fmtPtr = PICTURE_XFMT_XPM;
    } else if ((c == 'x') && (strcasecmp(string, "xbm") == 0)) {
	*fmtPtr = PICTURE_XFMT_XBM;
    } else if ((c == 'p') && (strcasecmp(string, "photo") == 0)) {
	*fmtPtr = PICTURE_XFMT_PHOTO;
    } else {
	Tcl_AppendResult(interp, "Unknown file format \"", string, 
		"\": should be jpg, png, gif, pdf, eps, wmf, emf, xpm, or raw",
		(char *)NULL);
	return TCL_ERROR;
    }
    return TCL_OK;
}

static char *
FormatToString(int fmt)
{
    switch (fmt) {
    case PICTURE_XFMT_JPG:
	return "jpg";
    case PICTURE_XFMT_GIF:
	return "gif";
    case PICTURE_XFMT_TIF:
	return "tif";
    case PICTURE_XFMT_EPS:
	return "eps";
    case PICTURE_XFMT_PDF:
	return "pdf";
    case PICTURE_XFMT_PNG:
	return "png";
    case PICTURE_XFMT_PHOTO:
	return "photo";

#ifdef WIN32
    case PICTURE_XFMT_BMP:
	return "bmp";
    case PICTURE_XFMT_EMF:
	return "emf";
    case PICTURE_XFMT_WMF:
	return "wmf";
#endif
    case PICTURE_XFMT_XPM:
	return "xpm";

    case PICTURE_XFMT_XBM:
	return "xbm";

    default:
	return "unknown picture format";
    }
}

static int
ReadImageFromFile(
    Tcl_Interp *interp, 
    char *fileName, 
    Blt_DataSink destData)
{
    FILE *f;
    size_t nBytes, nRead;
    struct stat sb;

#ifdef WIN32
#define READ_MODE "rb"
#else 
#define READ_MODE "r"
#endif
    f = Blt_OpenFile(interp, fileName, READ_MODE);
    if (f == NULL) {
	return TCL_ERROR;
    }
    if (fstat(fileno(f), &sb)) {
	Tcl_AppendResult(interp, "can't stat \"", fileName, "\": ",
		Tcl_PosixError(interp), (char *)NULL);
	return TCL_ERROR;
    }	
    Blt_SinkInit(destData);
    nBytes = sb.st_size;	/* Size of buffer */
    if (!Blt_SinkSetLength(destData, nBytes)) {
	fclose(f);
	return TCL_ERROR;
    }	
    nRead = fread(Blt_SinkBuffer(destData), sizeof(unsigned char), nBytes, f);
    Blt_SinkSetMark(destData, nRead);
    fclose(f);
    if (nRead != nBytes) {
	Tcl_AppendResult(interp, "short file \"", fileName, "\" : read ", 
		Blt_Itoa(nBytes), " bytes.", (char *)NULL); 
	Blt_SinkFree(destData);
	return TCL_ERROR;
    }	
    return TCL_OK;
}

static int
ReadImageFromOpenChannel(
    Tcl_Interp *interp, 
    char *string,
    Blt_DataSink destData)
{
    size_t nBytes, nRead;
    Tcl_Channel channel;
    int mode;

    channel = Tcl_GetChannel(interp, string, &mode);
    if (channel == NULL) {
	return TCL_ERROR;
    }
    if ((mode & TCL_READABLE) == 0) {
	Tcl_AppendResult(interp, "can't read from \"", string, "\"",
			 (char *)NULL);
	return TCL_ERROR;
    }
    Blt_SinkInit(destData);
    nBytes = 0;
    for (;;) {
#define BUFFER_SIZE (1<<16)
	Blt_SinkSetLength(destData, nBytes + BUFFER_SIZE);
	nRead = Tcl_ReadRaw(channel, Blt_SinkBuffer(destData) + nBytes, 
		BUFFER_SIZE);
	if (nRead == -1) {
	    Tcl_AppendResult(interp, "error reading channel: ",
			Tcl_PosixError(interp), (char *)NULL);
	    Blt_SinkFree(destData);
	    return TCL_ERROR;
	}
	nBytes += nRead;
	if (nRead != BUFFER_SIZE) {
	    break;
	}
    }
    Blt_SinkSetLength(destData, nBytes);
    Tcl_Close(interp, channel);
    return TCL_OK;
}

static int
WriteImageToFile(
    Tcl_Interp *interp, 
    Blt_DataSink srcData,
    char *fileName) 
{
    FILE *f;
    size_t nWritten, nBytes;
    unsigned char *bp;
#ifdef WIN32
#define WRITE_MODE "wb"
#else 
#define WRITE_MODE "w"
#endif
    f = Blt_OpenFile(interp, fileName, WRITE_MODE);
    if (f == NULL) {
	return TCL_ERROR;
    }
    nBytes = Blt_SinkMark(srcData);
    bp = Blt_SinkBuffer(srcData);
    nWritten = fwrite(bp, sizeof(unsigned char), nBytes, f);
    fclose(f);
    if (nWritten != nBytes) {
	Tcl_AppendResult(interp, "short file \"", fileName, (char *)NULL);
	Tcl_AppendResult(interp, "\" : wrote ", Blt_Itoa(nWritten), " of ", 
			 (char *)NULL);
	Tcl_AppendResult(interp, Blt_Itoa(nBytes), " bytes.", (char *)NULL); 
	return TCL_ERROR;
    }	
    return TCL_OK;
}

static int
QueryExternalFormat(Blt_DataSink srcData)
{
#ifdef HAVE_LIBJPEG
    if (Blt_IsJpg(srcData)) {
	return PICTURE_XFMT_JPG;
    } 
#endif /* HAVE_LIBJPEG */

#ifdef HAVE_LIBPNG
    if (Blt_IsPng(srcData)) {
	return PICTURE_XFMT_PNG;
    } 
#endif /* HAVE_LIBPNG */

#ifdef HAVE_LIBTIFF
    if (Blt_IsTif(srcData)) {
	return PICTURE_XFMT_TIF;
    } 
#endif /* HAVE_LIBTIFF */

    if (Blt_IsGif(srcData)) {
	return PICTURE_XFMT_GIF;
    }

    /* Put the ASCII formats last, because their checks are more
     * expensive.  When they fail they read the entire buffer. */
#ifdef HAVE_LIBXPM
    if (Blt_IsXpm(srcData)) {
	return PICTURE_XFMT_XPM;
    }
#endif /* HAVE_LIBXPM */

    if (Blt_IsXbm(srcData)) {
	return PICTURE_XFMT_XBM;
    }
    return -1;
}

static int
ImageToPicture(Tcl_Interp *interp, PictureImage *imgPtr, char *imageName)
{
    Blt_Picture picture;
    Tk_PhotoHandle photo;

    photo = Tk_FindPhoto(interp, imageName);
    if (photo != NULL) {
	picture = Blt_PhotoToPicture(photo);
    } else if (Blt_GetPicture(interp, imageName, &picture) == TCL_OK) {
	picture = Blt_ClonePicture(picture);
    } else {
	return TCL_ERROR;
    }
    if (imgPtr->picture != NULL) {
	Blt_FreePicture(imgPtr->picture);
    }
    imgPtr->picture = picture;
    if (imgPtr->name != NULL) {
	Blt_Free(imgPtr->name);
    }
    imgPtr->name = Blt_Strdup(imageName);
    imgPtr->flags &= ~IMPORTED_MASK;
    imgPtr->flags |= IMPORTED_IMAGE;
    return TCL_OK;
}

static int
WindowToPicture(Tcl_Interp *interp, PictureImage *imgPtr, Tcl_Obj *objPtr)
{
    Blt_Picture picture;
    Window window;
    int width, height;

    if (Blt_GetWindowFromObj(interp, objPtr, &window) != TCL_OK) {
	return TCL_ERROR;
    }
    if (Blt_GetWindowRegion(imgPtr->display, window, (int *)NULL, (int *)NULL,
		&width, &height) != TCL_OK) {
	Tcl_AppendResult(interp, "can't get dimensions of window \"", 
		Tcl_GetString(objPtr), "\"", (char *)NULL);
	return TCL_ERROR;
    }
    /* Depth, visual, colormap */
    picture = Blt_WindowToPicture(imgPtr->display, window, 0, 0, width, 
	height, imgPtr->gamma);
    if (picture == NULL) {
	Tcl_AppendResult(interp, "can't obtain snapshot of window \"", 
		Tcl_GetString(objPtr), "\"", (char *)NULL);
	return TCL_ERROR;
    }
    if (imgPtr->picture != NULL) {
	Blt_FreePicture(imgPtr->picture);
    }
    imgPtr->picture = picture;
    if (imgPtr->name != NULL) {
	Blt_Free(imgPtr->name);
    }
    imgPtr->name = Blt_Strdup(Tcl_GetString(objPtr));
    imgPtr->flags &= ~IMPORTED_MASK;
    imgPtr->flags |= IMPORTED_WINDOW;
    return TCL_OK;
}

static int
PictureToPhoto(Tcl_Interp *interp, Blt_Picture picture, CONST char *photoName)
{
    Tk_PhotoHandle photo;

    photo = Tk_FindPhoto(interp, photoName);
    if (photo == NULL) {
	return TCL_ERROR;
    }
    Blt_PictureToPhoto(picture, photo);
    return TCL_OK;
}

static Blt_Picture
DataToPicture(
    Tcl_Interp *interp, 
    char *fileName,
    int format,
    Blt_DataSink srcData)
{
    Pix32 white, black;
    Blt_PictureImportSwitches switches;

    white.color = 0xFFFFFFFF;
    black.color = 0x00000000;
    switches.fast = 0;
    switch (format) {
#ifdef HAVE_LIBJPEG
    case PICTURE_XFMT_JPG:
	return Blt_JpgToPicture(interp, fileName, srcData, &switches);
#endif /* HAVE_LIBJPEG */

#ifdef HAVE_LIBTIFF
    case PICTURE_XFMT_TIF:
	return Blt_TifToPicture(interp, fileName, srcData);
#endif /* HAVE_LIBTIFF */

#ifdef HAVE_LIBPNG
    case PICTURE_XFMT_PNG:
	return Blt_PngToPicture(interp, fileName, srcData);
#endif /* HAVE_LIBPNG */

#ifdef HAVE_LIBXPM
    case PICTURE_XFMT_XPM:
	return Blt_XpmToPicture(interp, fileName, srcData);
#endif /* HAVE_LIBXPM */

#ifdef notdef
#ifdef WIN32
    case PICTURE_XFMT_EMF:
	return Blt_EmfToPicture(interp, fileName, srcData);

    case PICTURE_XFMT_WMF:
	return Blt_WmfToPicture(interp, fileName, srcData);
#endif
#endif
    case PICTURE_XFMT_XBM:
	return  Blt_XbmToPicture(interp, fileName, srcData, 
		&white, 	/* Foreground color */
		&black);	/* Background color */

    case PICTURE_XFMT_GIF:
	/* By default, use the first image in the file. */
	return Blt_GifToPicture(interp, fileName, srcData, 
		1);		/*Image index */

    default:
	Tcl_AppendResult(interp, "no reader for format \"", 
		FormatToString(format), "\".", (char *)NULL);
    }
    return NULL;
}

static int
PictureToData(
    Tcl_Interp *interp, 
    Blt_Picture picture, 
    int format,
    Blt_DataSink destData)
{
    Blt_Picture tmp;
    int result;
    Pix32 white;
    Blt_PictureExportSwitches switches;

    white.color = 0xFFFFFFFF;
    tmp = NULL;
    if (Blt_PictureFlags(picture) & BLT_PICTURE_PREMULTIPLIED_ALPHAS) {
	/* 
	 * The picture has an alpha burned into the components.
	 * Create a temporary copy removing pre-multiplied alphas.
	 */ 
	tmp = Blt_ClonePicture(picture);
	Blt_UnmultiplyAlphas(tmp);
	picture = tmp;
    }

    /* Default export switch settings. */
    switches.quality = 75;
    switches.smoothing = 0;
    switches.flags = 0;		/* No progressive or compression. */
    switches.bg.color = 0xFFFFFFFF; /* white */

    /* 
     * Based upon the format, convert the picture into a binary buffer.
     */
    switch (format) {
#ifdef HAVE_LIBJPEG
    case PICTURE_XFMT_JPG:
	result = Blt_PictureToJpg(interp, picture, destData, &switches);
	break;
#endif /* HAVE_LIBJPEG */

#ifdef HAVE_LIBTIFF
    case PICTURE_XFMT_TIF:
	result = Blt_PictureToTif(interp, picture, destData,
		0);		/* compression */
	break;
#endif /* HAVE_LIBTIFF */

#ifdef HAVE_LIBPNG
    case PICTURE_XFMT_PNG:
	result = Blt_PictureToPng(interp, picture, destData);
	break;
#endif /* HAVE_LIBPNG */

#ifdef HAVE_LIBXPM
    case PICTURE_XFMT_XPM:
	result = Blt_PictureToXpm(interp, picture, destData,
		TRUE,		/* Allow color quantization. */
		&white);	/* Default background color is white. */
	break;
#endif /* HAVE_LIBXPM */

    case PICTURE_XFMT_XBM:
	result = Blt_PictureToXbm(interp, picture, destData,
		&white);	/* Default background color is white. */
	break;

#ifdef notdef
#ifdef WIN32
    case PICTURE_XFMT_EMF:
	result = Blt_PictureToEmf(interp, picture, destData);
	break;

    case PICTURE_XFMT_WMF:
	result = Blt_PictureToWmf(interp, picture, destData);
	break;
#endif
#endif
    default:
	Tcl_AppendResult(interp, "no writer for format \"", 
		FormatToString(format), "\".", (char *)NULL);
	result = TCL_ERROR;
    }
    if (tmp != NULL) {
	Blt_FreePicture(tmp);
    }
    return result;
}

/*
 *--------------------------------------------------------------------------
 *
 * ObjToFile --
 *
 *	Given a file name, determine the image type and convert into a
 *	picture.
 *
 * Results:
 *	The return value is a standard Tcl result.  
 *
 *--------------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
ObjToFile(
    ClientData clientData,	/* Not used. */
    Tcl_Interp *interp,		/* Interpreter to send results back to */
    Tk_Window tkwin,		/* Not used. */
    Tcl_Obj *objPtr,		/* String representation of value. */
    char *widgRec,		/* Widget record. */
    int offset,			/* Offset to field in structure */
    int flags)	
{
    DataSink srcSink;
    Blt_Picture *picturePtr = (Blt_Picture *)(widgRec + offset);
    Blt_Picture picture;
    PictureImage *imgPtr = (PictureImage *)widgRec;
    char *fileName;
    int format;

    Blt_SinkInit(&srcSink);
    fileName = Tcl_GetString(objPtr);
    if (fileName[0] == '@') {
	if (ReadImageFromOpenChannel(interp, fileName+1, &srcSink) != TCL_OK) {
	    return TCL_ERROR;
	}
    } else {
	if (ReadImageFromFile(interp, fileName, &srcSink) != TCL_OK) {
	    return TCL_ERROR;
	}
    }
    format = QueryExternalFormat(&srcSink);
    if (format == -1) {
	Tcl_AppendResult(interp, "unknown image file format in \"", fileName, 
		"\"", (char *)NULL);
	Blt_SinkFree(&srcSink);
	return TCL_ERROR;
    }
    picture = DataToPicture(interp, fileName, format, &srcSink);
    Blt_SinkFree(&srcSink);
    if (picture == NULL) {
	return TCL_ERROR;
    }
    if (*picturePtr != NULL) {
	Blt_FreePicture(*picturePtr);
    }
    if (imgPtr->name != NULL) {
	Blt_Free(imgPtr->name);
    }
    imgPtr->readFormat = format;
    imgPtr->name = Blt_Strdup(fileName);
    imgPtr->flags &= ~IMPORTED_MASK;
    imgPtr->flags |= IMPORTED_FILE;
    *picturePtr = picture;
    return TCL_OK;
}

/*
 *--------------------------------------------------------------------------
 *
 * FileToObj --
 *
 *	Convert the picture into a Tcl list of pixels.
 *
 * Results:
 *	The string representation of the picture is returned.
 *
 *--------------------------------------------------------------------------
 */
/*ARGSUSED*/
static Tcl_Obj *
FileToObj(
    ClientData clientData,	/* Not used. */
    Tcl_Interp *interp,
    Tk_Window tkwin,		/* Not used. */
    char *widgRec,		/* Widget record */
    int offset,			/* Offset to field in structure */
    int flags)	
{
    PictureImage *imgPtr = (PictureImage *)widgRec;

    if (((imgPtr->flags & IMPORTED_FILE) == 0) || (imgPtr->name == NULL)) {
	return Tcl_NewStringObj("", -1);
    }
    return Tcl_NewStringObj(imgPtr->name, -1);
}

/*
 *--------------------------------------------------------------------------
 *
 * ObjToFilter --
 *
 *	Given a string name, get the resample filter associated with it.
 *
 * Results:
 *	The return value is a standard Tcl result.  
 *
 *--------------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
ObjToFilter(
    ClientData clientData,	/* Not used. */
    Tcl_Interp *interp,		/* Interpreter to send results back to */
    Tk_Window tkwin,		/* Not used. */
    Tcl_Obj *objPtr,		/* String representation of value. */
    char *widgRec,		/* Widget record. */
    int offset,			/* Offset to field in structure */
    int flags)	
{
    Blt_PictureFilter *filterPtr = (Blt_PictureFilter *)(widgRec + offset);
    char *string;

    string = Tcl_GetString(objPtr);
    if (string[0] == '\0') {
	*filterPtr = NULL;
	return TCL_OK;
    }
    return Blt_GetPictureFilterFromObj(interp, objPtr, filterPtr);
}

/*
 *--------------------------------------------------------------------------
 *
 * FilterToObj --
 *
 *	Convert the picture filter into a string Tcl_Obj.
 *
 * Results:
 *	The string representation of the filter is returned.
 *
 *--------------------------------------------------------------------------
 */
/*ARGSUSED*/
static Tcl_Obj *
FilterToObj(
    ClientData clientData,	/* Not used. */
    Tcl_Interp *interp,
    Tk_Window tkwin,		/* Not used. */
    char *widgRec,		/* Widget record */
    int offset,			/* Offset to field in structure */
    int flags)	
{
    Blt_PictureFilter filter = *(Blt_PictureFilter *)(widgRec + offset);

    if (filter == NULL) {
	return Tcl_NewStringObj("", -1);
    }
    return Tcl_NewStringObj(Blt_NameOfPictureFilter(filter), -1);
}

/*
 *--------------------------------------------------------------------------
 *
 * ObjToData --
 *
 *	Given a string of data or binary Tcl_Obj, determine the image
 *	type and convert into a picture.
 *
 * Results:
 *	The return value is a standard Tcl result.  
 *
 *--------------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
ObjToData(
    ClientData clientData,	/* Not used. */
    Tcl_Interp *interp,		/* Interpreter to send results back to */
    Tk_Window tkwin,		/* Not used. */
    Tcl_Obj *objPtr,		/* String representation of value. */
    char *widgRec,		/* Widget record. */
    int offset,			/* Offset to field in structure */
    int flags)	
{
    DataSink data;
    Blt_Picture *picturePtr = (Blt_Picture *)(widgRec + offset);
    Blt_Picture picture;
    PictureImage *imgPtr = (PictureImage *)widgRec;
    int format;

    Blt_SinkInit(&data);
    if ((objPtr->typePtr != NULL) && 
	(strcmp(objPtr->typePtr->name, "bytearray") == 0)) {
	int length;

	data.bytes = Tcl_GetByteArrayFromObj(objPtr, &length);
	Blt_SinkSetMark(&data, length);
	data.nBytes = length;
    } else if (Blt_DecodeBase64(interp, Tcl_GetString(objPtr), &data) 
	       != TCL_OK) {
	return TCL_ERROR;
    }
#ifdef notdef
    {
	FILE *f;

	f = fopen("junk.unk", "w");
	fwrite(Blt_SinkBuffer(&data), 1, Blt_SinkMark(&data), f);
	fclose(f);
    }
#endif
    format = QueryExternalFormat(&data);
    if (format == -1) {
	Tcl_AppendResult(interp, "unknown image file format in \"",
			 Tcl_GetString(objPtr), "\"", (char *)NULL);
	Blt_SinkFree(&data);
	return TCL_ERROR;
    }
    picture = DataToPicture(interp, "data string", format, &data);
    Blt_SinkFree(&data);
    if (picture == NULL) {
	return TCL_ERROR;
    }
    if (*picturePtr != NULL) {
	Blt_FreePicture(*picturePtr);
    }
    imgPtr->flags &= ~IMPORTED_MASK;
    imgPtr->flags |= IMPORTED_DATA;
    *picturePtr = picture;
    return TCL_OK;
}

/*
 *--------------------------------------------------------------------------
 *
 * DataToObj --
 *
 *	Convert the picture into a Tcl list of pixels.
 *
 * Results:
 *	The string representation of the picture is returned.
 *
 *--------------------------------------------------------------------------
 */
/*ARGSUSED*/
static Tcl_Obj *
DataToObj(
    ClientData clientData,	/* Not used. */
    Tcl_Interp *interp,
    Tk_Window tkwin,		/* Not used. */
    char *widgRec,		/* Widget record */
    int offset,			/* Offset to field in structure */
    int flags)	
{
    DataSink data;
    PictureImage *imgPtr = (PictureImage *)(widgRec);
    Tcl_Obj *objPtr;
    
    Blt_SinkInit(&data);
    if (PictureToData(interp, imgPtr->picture, imgPtr->readFormat, 
	&data) == TCL_OK) {
	if ((imgPtr->readFormat == PICTURE_XFMT_XBM) ||
	    (imgPtr->readFormat == PICTURE_XFMT_XPM)) {
	    objPtr = Tcl_NewStringObj(Blt_SinkBuffer(&data), 
		Blt_SinkMark(&data));
	    Blt_SinkFree(&data);
	    return objPtr;
	} else {
	    char *string;
	    
	    string = Blt_EncodeBase64(interp, &data);
	    Blt_SinkFree(&data);
	    if (string != NULL) {
		objPtr = Tcl_NewStringObj(string, -1);
		Blt_Free(string);
		return objPtr;
	    }
	}
    }
    return Tcl_NewStringObj("", -1);
}

/*
 *--------------------------------------------------------------------------
 *
 * ObjToImage --
 *
 *	Convert a named image into a picture.
 *
 * Results:
 *	The return value is a standard Tcl result.  
 *
 *--------------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
ObjToImage(
    ClientData clientData,	/* Not used. */
    Tcl_Interp *interp,		/* Interpreter to send results back to */
    Tk_Window tkwin,		/* Used to search for named window. */
    Tcl_Obj *objPtr,		/* String representation of value. */
    char *widgRec,		/* Widget record. */
    int offset,			/* Offset to field in structure */
    int flags)	
{
    PictureImage *imgPtr = (PictureImage *)(widgRec);

    return ImageToPicture(interp, imgPtr, Tcl_GetString(objPtr));
}

/*
 *--------------------------------------------------------------------------
 *
 * ImageToObj --
 *
 *	Convert the named image into a picture.
 *
 * Results:
 *	The string representation of the picture is returned.
 *
 *--------------------------------------------------------------------------
 */
/*ARGSUSED*/
static Tcl_Obj *
ImageToObj(
    ClientData clientData,	/* Not used. */
    Tcl_Interp *interp,
    Tk_Window tkwin,		/* Not used. */
    char *widgRec,		/* Widget record */
    int offset,			/* Offset to field in structure */
    int flags)	
{
    PictureImage *imgPtr = (PictureImage *)widgRec;

    if (((imgPtr->flags & IMPORTED_IMAGE) == 0) || (imgPtr->name == NULL)) {
	return Tcl_NewStringObj("", -1);
    }
    return Tcl_NewStringObj(imgPtr->name, -1);
}

/*
 *--------------------------------------------------------------------------
 *
 * ObjToWindow --
 *
 *	Given a file name, determine the image type and convert 
 *	into a picture.
 *
 * Results:
 *	The return value is a standard Tcl result.  
 *
 *--------------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
ObjToWindow(
    ClientData clientData,	/* Not used. */
    Tcl_Interp *interp,		/* Interpreter to send results back to */
    Tk_Window tkwin,		/* Used to search for named window. */
    Tcl_Obj *objPtr,		/* String representation of value. */
    char *widgRec,		/* Widget record. */
    int offset,			/* Offset to field in structure */
    int flags)	
{
    PictureImage *imgPtr = (PictureImage *)(widgRec);

    return WindowToPicture(interp, imgPtr, objPtr);
}

/*
 *--------------------------------------------------------------------------
 *
 * WindowToObj --
 *
 *	Convert the picture into a Tcl list of pixels.
 *
 * Results:
 *	The string representation of the picture is returned.
 *
 *--------------------------------------------------------------------------
 */
/*ARGSUSED*/
static Tcl_Obj *
WindowToObj(
    ClientData clientData,	/* Not used. */
    Tcl_Interp *interp,
    Tk_Window tkwin,		/* Not used. */
    char *widgRec,		/* Widget record */
    int offset,			/* Offset to field in structure */
    int flags)	
{
    PictureImage *imgPtr = (PictureImage *)widgRec;

    if (((imgPtr->flags & IMPORTED_WINDOW) == 0) || (imgPtr->name == NULL)) {
	return Tcl_NewStringObj("", -1);
    }
    return Tcl_NewStringObj(imgPtr->name, -1);
}

/*
 *--------------------------------------------------------------------------
 *
 * CacheKey --
 *
 *	Returns a key representing a specific visual context for a
 *	PictureImage.  Keys are used to create/find cache entries
 *	stored in the hash table of each PictureImage.
 *
 * Results:
 *	A pointer to a static cache key.  
 *
 * Side effects:
 *	The key is overwritten on each call.  Care must be taken by
 *	the caller to save the key before making additional calls.
 *
 *--------------------------------------------------------------------------
 */
static PictureCacheKey *
CacheKey(Tk_Window tkwin)
{
    static PictureCacheKey key;

    key.display = Tk_Display(tkwin);
    key.visualPtr = Tk_Visual(tkwin);
    key.colormap = Tk_Colormap(tkwin);
    key.depth = Tk_Depth(tkwin);
    return &key;
}

/*
 *--------------------------------------------------------------------------
 *
 * DestroyCache --
 *
 *	This procedure is a callback for Tcl_EventuallyFree to release
 *	the resources and memory used by a PictureCache entry. The
 *	entry is destroyed only if noone else is currently using the 
 *	entry (using reference counting).
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The cache entry is possibly deallocated.
 *
 *--------------------------------------------------------------------------
 */
static void
DestroyCache(DestroyData data)
{
    PictureCache *cachePtr = (PictureCache *)data;
    
    if (cachePtr->refCount <= 0) {
	if (cachePtr->pixmap != None) {
	    Tk_FreePixmap(cachePtr->display, cachePtr->pixmap);
	}
	if (cachePtr->painter != NULL) {
	    Blt_FreePainter(cachePtr->painter);
	}
	if (cachePtr->hashPtr != NULL) {
	    Blt_DeleteHashEntry(cachePtr->tablePtr, cachePtr->hashPtr);
	}
	Blt_Free(cachePtr);
    }
}

/*
 *--------------------------------------------------------------------------
 *
 * GetPictureCache --
 *
 *	This procedure is called for each use of a picture image in a
 *	widget. 
 *
 * Results:
 *	The return value is a cache entry for the visual contex, which
 *	will be passed back to us in calls to DisplayPictureImage and 
 *	FreeCache.
 *
 * Side effects:
 *	A new cache entry is possibly allocated (or shared if one already
 *	exists).
 *
 *--------------------------------------------------------------------------
 */
static ClientData
GetPictureCache(
    Tk_Window tkwin,		/* Window in which the picture will be
				 * displayed. */
    ClientData clientData)	/* Pointer to picture image for the
				 * image. */
{
    PictureImage *imgPtr = clientData;
    PictureCache *cachePtr;
    Blt_HashEntry *hPtr;
    int isNew;

    hPtr = Blt_CreateHashEntry(&imgPtr->cacheTable, (char *)CacheKey(tkwin), 
	&isNew);
    if (isNew) {
	cachePtr = Blt_Malloc(sizeof(PictureCache));	
	if (cachePtr == NULL) {
	    return NULL;	/* Can't allocate memory. */
	}
	cachePtr->painter = Blt_GetPainter(tkwin, imgPtr->gamma);
	cachePtr->image = imgPtr;
	cachePtr->display = Tk_Display(tkwin);
	cachePtr->pixmap = None;
	cachePtr->hashPtr = hPtr;
	cachePtr->tablePtr = &imgPtr->cacheTable;
	cachePtr->refCount = 0;
	Blt_SetHashValue(hPtr, cachePtr);

	if (imgPtr->picture != NULL) {
	    Blt_NotifyImageChanged(imgPtr);
	}
    } 
    cachePtr = Blt_GetHashValue(hPtr);
    cachePtr->refCount++;
    return cachePtr;
}

/*
 *--------------------------------------------------------------------------
 *
 * FreePictureCache --
 *
 *	This procedure is called when a widget ceases to use a
 *	particular instance of a picture image.  We don't actually
 *	get rid of the cache entry until later because we may be about
 *	to get this instance again.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Internal data structures get freed.
 *
 *--------------------------------------------------------------------------
 */
/*ARGSUSED*/
static void
FreePictureCache(
    ClientData clientData,	/* Pointer to cache structure */
    Display *display)		/* Not used. */
{
    PictureCache *cachePtr = clientData;

    cachePtr->refCount--;
    if (cachePtr->refCount <= 0) {
	/* 
	 * Right now no one is using the entry. But delay the removal
	 * of the cache entry in case it's reused shortly afterwards.
	 */
 	Tcl_EventuallyFree(cachePtr, DestroyCache);
    }    
}

/*
 *--------------------------------------------------------------------------
 *
 * DeletePictureImage --
 *
 *	This procedure is called by the Tk image code to delete the
 *	master structure (PictureImage) for a picture image.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Resources associated with the picture image are freed.
 *
 *--------------------------------------------------------------------------
 */
static void
DeletePictureImage(
    ClientData clientData)	/* Pointer to PhotoMaster structure
				 * for image.  Must not have any more
				 * instances. */
{
    Blt_HashEntry *hPtr;
    Blt_HashSearch cursor;
    PictureImage *imgPtr = clientData;
 
    for (hPtr = Blt_FirstHashEntry(&imgPtr->cacheTable, &cursor); 
	hPtr != NULL; hPtr = Blt_NextHashEntry(&cursor)) {
	PictureCache *cachePtr;

	cachePtr = Blt_GetHashValue(hPtr);
	cachePtr->hashPtr = NULL; /* Flag for FreeCache. */
	DestroyCache((DestroyData)cachePtr);
    }
    imgPtr->imgToken = NULL;
    if (imgPtr->cmdToken != NULL) {
	Tcl_DeleteCommandFromToken(imgPtr->interp, imgPtr->cmdToken);
    }
    Blt_DeleteHashTable(&imgPtr->cacheTable);
    Blt_FreeOptions(configSpecs, (char *)imgPtr, imgPtr->display, 0);
    Blt_Free(imgPtr);
}

static int 
ConfigurePictureImage(
    Tcl_Interp *interp, 
    PictureImage *imgPtr, 
    int objc, 
    Tcl_Obj *CONST *objv, 
    int flags) 
{
    int width, height;

    if (Blt_ConfigureWidgetFromObj(interp, Tk_MainWindow(interp), configSpecs, 
	objc, objv, (char *)imgPtr, flags) != TCL_OK) {
	return TCL_ERROR;
    }

    if (imgPtr->picture == NULL) {
	width = (imgPtr->reqWidth == 0) ? 16 : imgPtr->reqWidth;
	height = (imgPtr->reqHeight == 0) ? 16 : imgPtr->reqHeight;
	imgPtr->picture = Blt_CreatePicture(width, height);
    }

    if (imgPtr->angle != 0.0) {
	Blt_Picture rotate;

	/* Rotate the picture */
	rotate = Blt_RotatePicture(imgPtr->picture, imgPtr->angle);
	Blt_FreePicture(imgPtr->picture);
	imgPtr->picture = rotate;
    }

    width = (imgPtr->reqWidth == 0) ? 
	Blt_PictureWidth(imgPtr->picture) : imgPtr->reqWidth;
    height = (imgPtr->reqHeight == 0) ? 
	Blt_PictureHeight(imgPtr->picture) : imgPtr->reqHeight;

    if (imgPtr->aspect) {
	double sx, sy, scale;

	sx = (double)width / (double)Blt_PictureWidth(imgPtr->picture);
	sy = (double)height / (double)Blt_PictureHeight(imgPtr->picture);
	scale = MIN(sx, sy);
	width = (int)(Blt_PictureWidth(imgPtr->picture) * scale + 0.5);
	height = (int)(Blt_PictureHeight(imgPtr->picture) * scale + 0.5);
    }	    
    if ((Blt_PictureWidth(imgPtr->picture) != width) || 
	(Blt_PictureHeight(imgPtr->picture) != height)) {
	Blt_Picture resize;

	/* Scale the picture */
	if (imgPtr->filter == NULL) {
	    resize = Blt_ScalePicture(imgPtr->picture, 0, 0,
		Blt_PictureWidth(imgPtr->picture), 
		Blt_PictureHeight(imgPtr->picture), width, height);
	} else {
	    resize = Blt_CreatePicture(width, height);
	    Blt_ResamplePicture(resize, imgPtr->picture, imgPtr->filter, 
		imgPtr->filter);
	}	
	Blt_FreePicture(imgPtr->picture);
	imgPtr->picture = resize;
    }
    if (imgPtr->sharpen > 0) {
	Blt_Picture sharpen;

	sharpen = Blt_ClonePicture(imgPtr->picture);
	Blt_SharpenPicture2(imgPtr->picture, sharpen);
	Blt_FreePicture(imgPtr->picture);
	imgPtr->picture = sharpen;
    }
    Blt_NotifyImageChanged(imgPtr);
    return TCL_OK;
}    

/*
 *--------------------------------------------------------------------------
 *
 * CreatePictureImage --
 *
 *	This procedure is called by the Tk image code to create
 *	a new photo image.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	The data structure for a new photo image is allocated and
 *	initialized.
 *
 *--------------------------------------------------------------------------
 */
static int
CreatePictureImage(
    Tcl_Interp *interp,		/* Interpreter to report errors back to. */
    char *name,			/* Name to use for image command. */
    int objc,			/* # of option arguments. */
    Tcl_Obj *CONST *objv,	/* Option arguments (doesn't include 
				 * image name or type). */
    Tk_ImageType *typePtr,	/* Not used. */
    Tk_ImageMaster imgToken,	/* Token for image, to be used by us
				 * in later callbacks. */
    ClientData *clientDataPtr)	/* Store manager's token for image
				 * here; it will be returned in later
				 * callbacks. */
{
    PictureImage *imgPtr;
    Tk_Window tkwin;

    /*
     * Allocate and initialize the photo image master record.
     */

    imgPtr = Blt_Calloc(1, sizeof(PictureImage));
    imgPtr->imgToken = imgToken;
    imgPtr->interp = interp;
    imgPtr->gamma = 1.0;
    imgPtr->cmdToken = Tcl_CreateObjCommand(interp, name, PictureInstCmdProc,
	    imgPtr, PictureInstCmdDeletedProc);
    imgPtr->aspect = TRUE;
    imgPtr->dither = 2;
    tkwin = Tk_MainWindow(interp);
    imgPtr->display = Tk_Display(tkwin);
    imgPtr->colormap = Tk_Colormap(tkwin);
#ifdef notdef
    imgPtr->typePtr = typePtr;
#endif
    Blt_InitHashTable(&imgPtr->cacheTable, sizeof(PictureCacheKey));

    /*
     * Process configuration options given in the image create command.
     */
    if (ConfigurePictureImage(interp, imgPtr, objc, objv, 0) != TCL_OK) {
	DeletePictureImage(imgPtr);
	return TCL_ERROR;
    }
    *clientDataPtr = imgPtr;
    Tcl_SetStringObj(Tcl_GetObjResult(interp), name, -1);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------------------
 *
 * DisplayPictureImage --
 *
 *	This procedure is invoked to draw a photo image.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	A portion of the image gets rendered in a pixmap or window.
 *
 *--------------------------------------------------------------------------
 */
static void
DisplayPictureImage(
    ClientData clientData,	/* Pointer to token structure for the
				 * picture to be displayed. */
    Display *display,		/* Display on which to draw picture. */
    Drawable drawable,		/* Pixmap or window in which to draw
				 * image. */
    int x, int y,		/* Upper-left corner of region within
				 * image to draw. */
    int width, int height,	/* Dimensions of region within image
				 * to draw. */
    int drawX, int drawY)	/* Coordinates within drawable that
				 * correspond to imageX and imageY. */
{
    PictureCache *cachePtr = clientData;
    PictureImage *imgPtr;

    imgPtr = cachePtr->image;
    if (imgPtr->picture == NULL) {
	return;
    }
    if ((cachePtr->pixmap != None) && (Blt_PictureIsDirty(imgPtr->picture))) {
	Tk_FreePixmap(display, cachePtr->pixmap);
	cachePtr->pixmap = None;
    }
    if ((imgPtr->doCache) && (Blt_PictureIsOpaque(imgPtr->picture))) {
	if (cachePtr->pixmap == None) {
	    Pixmap pixmap;

	    /* Save the entire picture in the pixmap. */
	    pixmap = Tk_GetPixmap(display, drawable, 
		Blt_PictureWidth(imgPtr->picture), 
		Blt_PictureHeight(imgPtr->picture),
		Blt_PainterDepth(cachePtr->painter));
	    Blt_PaintPicture(cachePtr->painter, drawable, imgPtr->picture,
		0, 0, Blt_PictureWidth(imgPtr->picture), 
		Blt_PictureHeight(imgPtr->picture), 0, 0, imgPtr->dither);
	    cachePtr->pixmap = pixmap;
	}
	/* Draw only the area that need to be repaired. */
	XCopyArea(display, cachePtr->pixmap, drawable, 
		Blt_PainterGC(cachePtr->painter), x, y, (unsigned int)width, 
			(unsigned int)height, drawX, drawY);
    } else {
	unsigned int flags;

	if (cachePtr->pixmap != None) {
	    /* Kill the cached pixmap. */
	    Tk_FreePixmap(display, cachePtr->pixmap);
	    cachePtr->pixmap = None;
	}
	flags = 0;
	if ((imgPtr->dither == 1) || ((imgPtr->dither == 2) && 
		(Blt_PainterDepth(cachePtr->painter) < 15))) {
	    flags |= BLT_PAINTER_DITHER;
	}
	Blt_PaintPicture(cachePtr->painter, drawable, imgPtr->picture, x, y, 
		width, height, drawX, drawY, flags);
    }
}

/*
 *--------------------------------------------------------------------------
 *
 * PictureImageToPostScript --
 *
 *	This procedure is called to output the contents of a photo
 *	image in PostScript by calling the Tk_PostscriptPhoto
 *	function.
 *
 * Results:
 *	Returns a standard Tcl return value.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------------------
 */
static int
PictureImageToPostScript(
    ClientData clientData,	/* Master picture image. */
    Tcl_Interp *interp,		/* Interpreter to return generated
				   PostScript. */
    Tk_Window tkwin,
    Tk_PostscriptInfo psInfo,	/* Not used.  Only useful for Tk internal
				 * Photo and Bitmap image types.  */
    int x, int y,		/* First pixel to output */
    int width, int height,	/* Width and height of area */
    int prepass)
{
    PictureImage *imgPtr = clientData;

    if (prepass) {
	return TCL_OK;
    }
    if (Blt_PictureIsOpaque(imgPtr->picture)) {
	Blt_PostScript ps;

	ps = Blt_GetPostScript(interp, tkwin);
	Blt_PictureToPostScript(ps, imgPtr->picture, (double)x, (double)y);
	Blt_PostScriptToInterp(interp, ps);
	Blt_FreePostScript(ps);
    } else {
	Blt_HashEntry *hPtr;
	Blt_Picture bg;
	Blt_PostScript ps;
	PictureCache *cachePtr;

	cachePtr = NULL;
	hPtr = Blt_FindHashEntry(&imgPtr->cacheTable,(char *)CacheKey(tkwin));
	if (hPtr != NULL) {
	    cachePtr = Blt_GetHashValue(hPtr);
	}
	if ((cachePtr != NULL) && (cachePtr->pixmap != None)) {
	    bg = Blt_DrawableToPicture(tkwin, cachePtr->pixmap, x, y, width, 
		height, imgPtr->gamma); 
	} else {
	    bg = Blt_DrawableToPicture(tkwin, Tk_WindowId(tkwin), x, y, width, 
		height, imgPtr->gamma); 
	}
	if (bg == NULL) {
	    return TCL_ERROR;
	}
	Blt_BlendPictureArea(bg, imgPtr->picture, 0, 0, width, height, 0, 0);
	ps = Blt_GetPostScript(interp, tkwin);
	Blt_PictureToPostScript(ps, bg, (double)x, (double)y);
	Blt_PostScriptToInterp(interp, ps);
	Blt_FreePostScript(ps);
	Blt_FreePicture(bg);
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * BBoxSwitch --
 *
 *	Convert a Tcl_Obj list of 2 or 4 numbers into representing 
 *	a bounding box structure.
 *
 * Results:
 *	The return value is a standard Tcl result.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
BBoxSwitch(
    ClientData clientData,	/* Not used. */
    Tcl_Interp *interp,		/* Interpreter to send results back to */
    char *switchName,		/* Not used. */
    Tcl_Obj *objPtr,		/* String representation */
    char *record,		/* Structure record */
    int offset,			/* Offset to field in structure */
    int flags)	
{
    BBox *bbPtr = (BBox *)(record + offset);

    if (GetBBoxFromObj(interp, objPtr, bbPtr) != TCL_OK) {
	return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * FilterSwitch --
 *
 *	Convert a Tcl_Obj representing a 1D image filter.
 *
 * Results:
 *	The return value is a standard Tcl result.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
FilterSwitch(
    ClientData clientData,	/* Not used. */
    Tcl_Interp *interp,		/* Interpreter to send results back to */
    char *switchName,		/* Not used. */
    Tcl_Obj *objPtr,		/* String representation */
    char *record,		/* Structure record */
    int offset,			/* Offset to field in structure */
    int flags)	
{
    Blt_PictureFilter *filterPtr = (Blt_PictureFilter *)(record + offset);

    return Blt_GetPictureFilterFromObj(interp, objPtr, filterPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * CompressSwitch --
 *
 *	Convert a Tcl_Obj representing a TIFF compression name.
 *
 * Results:
 *	The return value is a standard Tcl result.
 *
 *----------------------------------------------------------------------
 */
static char *compressNames[] =  {
    "lzw",			/* Lempel-Ziv & Welch */
    "ojpeg",			/* !6.0 JPEG */
    "jpeg",			/* %JPEG DCT compression */
    "next",			/* NeXT 2-bit RLE */
    "packbits",			/* Macintosh RLE */
    "thunderscan",		/* ThunderScan RLE */
    "pixarfilm",		/* Pixar companded 10bit LZW */
    "pixarlog",			/* Pixar companded 11bit ZIP */
    "deflate",			/* Deflate compression */
    "adobe_deflate",		/* Adobe's deflate */
    "dcs",			/* Kodak DCS encoding */
    "sgilog",			/* SGI Log Luminance RLE */
    "sgilog24",			/* SGI Log 24-bit packed */
};

static int nCompressNames = sizeof(compressNames) / sizeof(char *);

/*ARGSUSED*/
static int
CompressSwitch(
    ClientData clientData,	/* Not used. */
    Tcl_Interp *interp,		/* Interpreter to send results back to */
    char *switchName,		/* Not used. */
    Tcl_Obj *objPtr,		/* String representation */
    char *record,		/* Structure record */
    int offset,			/* Offset to field in structure */
    int flags)	
{
    int *compressPtr = (int *)(record + offset);
    int i;
    char *string;
    char c;

    string = Tcl_GetString(objPtr);
    c = string[0];
    if (c == '\0') {
	*compressPtr = 0;
	return TCL_OK;
    } 
    for (i = 0; i < nCompressNames; i++) {
	if ((c == compressNames[i][0]) && 
	    (strcasecmp(string, compressNames[i]) == 0)) {
	    *compressPtr = i + 1;
	    return TCL_OK;
	}
    }
    Tcl_AppendResult(interp, "unknown TIFF compression mode \"", string, "\"",
		     (char *) NULL);
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * BlendingModeSwitch --
 *
 *	Convert a Tcl_Obj representing a blending mode.
 *
 * Results:
 *	The return value is a standard Tcl result.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
BlendingModeSwitch(
    ClientData clientData,	/* Not used. */
    Tcl_Interp *interp,		/* Interpreter to send results back to */
    char *switchName,		/* Not used. */
    Tcl_Obj *objPtr,		/* String representation */
    char *record,		/* Structure record */
    int offset,			/* Offset to field in structure */
    int flags)	
{
    Blt_BlendingMode *modePtr = (Blt_BlendingMode *)(record + offset);
    char *string;
    int length;
    char c;

    string = Tcl_GetString(objPtr);
    c = string[0];
    length = strlen(string);
    if ((c == 'n') && (strncmp(string, "normal", length) == 0)) {
	*modePtr = BLT_BLEND_NORMAL;
    } else if ((c == 'm') && (strncmp(string, "multiply", length) == 0)) {
	*modePtr = BLT_BLEND_MULTIPLY;
    } else if ((c == 's') && (strncmp(string, "screen", length) == 0)) {
	*modePtr = BLT_BLEND_SCREEN;
    } else if ((c == 'd') && (length > 1) && 
	       (strncmp(string, "darken", length) == 0)) {
	*modePtr = BLT_BLEND_DARKEN;
    } else if ((c == 'l') && (strncmp(string, "lighten", length) == 0)) {
	*modePtr = BLT_BLEND_LIGHTEN;
    } else if ((c == 'd') && (length > 1) && 
	       (strncmp(string, "difference", length) == 0)) {
	*modePtr = BLT_BLEND_DIFFERENCE;
    } else if ((c == 'h') && (strncmp(string, "hardlight", length) == 0)) {
	*modePtr = BLT_BLEND_HARDLIGHT;
    } else if ((c == 's') && (strncmp(string, "softlight", length) == 0)) {
	*modePtr = BLT_BLEND_SOFTLIGHT;
    } else if ((c == 'c') && (length > 5) && 
	       (strncmp(string, "colordodge", length) == 0)) {
	*modePtr = BLT_BLEND_COLORDODGE;
    } else if ((c == 'c') && (length > 5) && 
	       (strncmp(string, "colorburn", length) == 0)) {
	*modePtr = BLT_BLEND_COLORBURN;
    } else if ((c == 'o') && (strncmp(string, "overlay", length) == 0)) {
	*modePtr = BLT_BLEND_OVERLAY;
    } else {
	Tcl_AppendResult(interp, "unknown blending mode \"", string, "\": ",
		"should be normal, mulitply, screen, darken, lighten, ",
		"or difference", (char *) NULL);
	return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * ShapeSwitch --
 *
 *	Convert a Tcl_Obj representing a gradient shape.
 *
 * Results:
 *	The return value is a standard Tcl result.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
ShapeSwitch(
    ClientData clientData,	/* Not used. */
    Tcl_Interp *interp,		/* Interpreter to send results back to */
    char *switchName,		/* Not used. */
    Tcl_Obj *objPtr,		/* String representation */
    char *record,		/* Structure record */
    int offset,			/* Offset to field in structure */
    int flags)	
{
    Blt_GradientShape *shapePtr = (Blt_GradientShape *)(record + offset);
    char *string, **p;
    int shape;
    static char *gradientShapes[] = {
	"linear",
	"bilinear",
	"radial",
	"rectangular",
	(char *)NULL
    };
    string = Tcl_GetString(objPtr);
    for (shape = 0, p = gradientShapes; *p != NULL; p++, shape++) {
	if (strcmp(string, *p) == 0) {
	    *shapePtr = (Blt_GradientShape)shape;
	    return TCL_OK;
	}
    }
    Tcl_AppendResult(interp, "unknown gradient type \"", string, "\"",
		     (char *) NULL);
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * PathSwitch --
 *
 *	Convert a Tcl_Obj representing a gradient path.
 *
 * Results:
 *	The return value is a standard Tcl result.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
PathSwitch(
    ClientData clientData,	/* Not used. */
    Tcl_Interp *interp,		/* Interpreter to send results back to */
    char *switchName,		/* Not used. */
    Tcl_Obj *objPtr,		/* String representation */
    char *record,		/* Structure record */
    int offset,			/* Offset to field in structure */
    int flags)	
{
    Blt_GradientPath *pathPtr = (Blt_GradientPath *)(record + offset);
    char *string, **p;
    int path;
    static char *gradientPaths[] = {
	"x", "y", "xy", "yx", (char *)NULL
    };

    string = Tcl_GetString(objPtr);
    for (path = 0, p = gradientPaths; *p != NULL; p++, path++) {
	if (strcmp(string, *p) == 0) {
	    *pathPtr = (Blt_GradientPath)path;
	    return TCL_OK;
	}
    }
    Tcl_AppendResult(interp, "unknown gradient type \"", string, "\"",
		     (char *) NULL);
    return TCL_ERROR;
}

/*
 *--------------------------------------------------------------------------
 *
 * ArithOp --
 *
 *	$image arith op $src -from { 0 0 100 100 } -to { 0 0 }
 *
 *--------------------------------------------------------------------------
 */
/*ARGSUSED*/
static int 
ArithOp(
    ClientData clientData,	/* Information about picture cmd. */
    Tcl_Interp *interp,		/* Interpreter to report errors back to. */
    int objc,			/* Not used. */
    Tcl_Obj *CONST *objv)
{
    Blt_Picture src;
    Blt_PictureArithOps op;
    PictureImage *imgPtr = clientData;
    Pix32 scalar;
    char *string;
    char c;
    int len;
    ArithSwitches switches;

    src = NULL;
    string = Tcl_GetString(objv[2]);
    if ((string[0] == '0') && (string[1] == 'x')) {
	if (Blt_GetPix32(interp, string, &scalar) != TCL_OK) {
	    return TCL_ERROR;
	}
    } else if (Blt_GetPicture(interp, string, &src) != TCL_OK) {
	return TCL_ERROR;
    }
    string = Tcl_GetString(objv[1]);
    op = 0;
    c = string[0];
    len = strlen(string);
    if ((c == 'a') && (len > 1) && (strncmp(string, "add", len) == 0)) {
	op = PICTURE_ARITH_ADD;
    } else if ((c == 's') && (strncmp(string, "subtract", len) == 0)) {
	op = PICTURE_ARITH_SUB;
    } else if ((c == 'a') && (len > 1) && (strncmp(string, "and", len) == 0)) {
	op = PICTURE_ARITH_AND;
    } else if ((c == 'o') && (strncmp(string, "or", len) == 0)) {
	op = PICTURE_ARITH_OR;
    } else if ((c == 'n') && (len > 1) && (strncmp(string, "nand", len) == 0)) {
	op = PICTURE_ARITH_NAND;
    } else if ((c == 'n') && (len > 1) && (strncmp(string, "nor", len) == 0)) {
	op = PICTURE_ARITH_NOR;
    } else if ((c == 'x') && (strncmp(string, "xor", len) == 0)) {
	op = PICTURE_ARITH_XOR;
    } else if ((c == 'm') && (len > 1) && (strncmp(string, "max", len) == 0)) {
	op = PICTURE_ARITH_MAX;
    } else if ((c == 'm') && (len > 1) && (strncmp(string, "min", len) == 0)) {
	op = PICTURE_ARITH_MIN;
    }	

    memset(&switches, 0, sizeof(switches));
    if (Blt_ParseSwitches(interp, arithSwitches, objc - 3, objv + 3, &switches, 
	BLT_SWITCH_DEFAULTS) < 0) {
	return TCL_ERROR;
    }
    if (switches.matte == NULL) {
	if (src == NULL) {
	    Blt_ApplyScalarToPicture(imgPtr->picture, &scalar, op);
	} else {
	    Blt_ApplyPictureToPicture(imgPtr->picture, src, op);
	}	
    } else {
	if (src == NULL) {
	    Blt_ApplyScalarToPictureWithMatte(imgPtr->picture, &scalar, 
		switches.matte, switches.invert, op);
	} else {
	    Blt_ApplyPictureToPictureWithMatte(imgPtr->picture, src, 
		switches.matte, switches.invert, op);
	}	
    }
    Blt_NotifyImageChanged(imgPtr);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------------------
 *
 * BlankOp --
 *	
 *	Resets the picture at its current size to blank (by default 
 *	white, fully opaque) pixels.  
 *
 *		$image blank -color #000000 -region { 0 0 20 20 }
 * Results:
 *	Returns a standard Tcl return value. If an error occured parsing
 *	the pixel.
 *
 *
 * Side effects:
 *	A Tk_ImageChanged notification is triggered.
 *
 *--------------------------------------------------------------------------
 */
static int
BlankOp(
    ClientData clientData,	/* Information about picture cmd. */
    Tcl_Interp *interp,		/* Current interpreter. */
    int objc,			/* Number of arguments. */
    Tcl_Obj *CONST *objv)	/* Argument objects. */
{
    PictureImage *imgPtr = clientData;
    Pix32 bg;

    bg.color = 0x00000000;
    if ((objc == 3) && (Blt_GetPix32FromObj(interp, objv[2], &bg)!= TCL_OK)) {
	return TCL_ERROR;
    }
    Blt_BlankPicture(imgPtr->picture, &bg);
    Blt_NotifyImageChanged(imgPtr);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------------------
 *
 * BlendOp --
 *
 *--------------------------------------------------------------------------
 */
/*ARGSUSED*/
static int 
BlendOp(
    ClientData clientData,	/* Information about picture cmd. */
    Tcl_Interp *interp,		/* Interpreter to report errors back to. */
    int objc,			/* Not used. */
    Tcl_Obj *CONST *objv)
{
    BlendSwitches switches;
    Blt_Picture fg, bg;
    PictureImage *imgPtr = clientData;

    if ((Blt_GetPictureFromObj(interp, objv[2], &bg) != TCL_OK) ||
	(Blt_GetPictureFromObj(interp, objv[3], &fg) != TCL_OK)) {
	return TCL_ERROR;
    }
    switches.mode = BLT_BLEND_NORMAL;
    if (Blt_ParseSwitches(interp, blendSwitches, objc - 4, objv + 4, 
	&switches, BLT_SWITCH_DEFAULTS) < 0) {
	return TCL_ERROR;
    }
    Blt_CopyPicture(imgPtr->picture, bg);
    Blt_BlendPictures2(imgPtr->picture, fg, switches.mode);
    Blt_NotifyImageChanged(imgPtr);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------------------
 *
 * Blend2Op --
 *
 *--------------------------------------------------------------------------
 */
/*ARGSUSED*/
static int 
Blend2Op(
    ClientData clientData,	/* Information about picture cmd. */
    Tcl_Interp *interp,		/* Interpreter to report errors back to. */
    int objc,			/* Not used. */
    Tcl_Obj *CONST *objv)
{
    Blt_Picture fg, bg;
    BlendSwitches switches;
    PictureImage *imgPtr = clientData;

    if ((Blt_GetPictureFromObj(interp, objv[2], &bg) != TCL_OK) ||
	(Blt_GetPictureFromObj(interp, objv[3], &fg) != TCL_OK)) {
	return TCL_ERROR;
    }
    switches.mode = BLT_BLEND_NORMAL;
    if (Blt_ParseSwitches(interp, blendSwitches, objc - 4, objv + 4, 
	&switches, BLT_SWITCH_DEFAULTS) < 0) {
	return TCL_ERROR;
    }
    Blt_CopyPicture(imgPtr->picture, bg);
    Blt_BlendPictures(imgPtr->picture, fg);
    Blt_NotifyImageChanged(imgPtr);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------------------
 *
 * BlurOp --
 *
 *--------------------------------------------------------------------------
 */
/*ARGSUSED*/
static int 
BlurOp(
    ClientData clientData,	/* Information about picture cmd. */
    Tcl_Interp *interp,		/* Interpreter to report errors back to. */
    int objc,			/* Not used. */
    Tcl_Obj *CONST *objv)
{
    PictureImage *imgPtr = clientData;

    Blt_BlurPicture(imgPtr->picture);
    Blt_NotifyImageChanged(imgPtr);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------------------
 *
 * CgetOp --
 *
 *	Returns the value of the configuration option specified.
 *
 * Results:
 *	Returns a standard Tcl return value.  If TCL_OK, the value of
 *	the picture configuration option specified is returned in the
 *	interpreter result.  Otherwise an error message is returned.
 * 
 *--------------------------------------------------------------------------
 */
static int
CgetOp(
    ClientData clientData,	/* Information about picture cmd. */
    Tcl_Interp *interp,		/* Current interpreter. */
    int objc,			/* Number of arguments. */
    Tcl_Obj *CONST *objv)	/* Argument objects. */
{
    PictureImage *imgPtr = clientData;
    
    return Blt_ConfigureValueFromObj(interp, Tk_MainWindow(interp), configSpecs,
	(char *)imgPtr, objv[2], 0);
}


/*
 *--------------------------------------------------------------------------
 *
 * ConfigureOp --
 *
 * Results:
 *	Returns a standard Tcl return value.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------------------
 */
static int
ConfigureOp(
    ClientData clientData,	/* Information about picture cmd. */
    Tcl_Interp *interp,		/* Current interpreter. */
    int objc,			/* Number of arguments. */
    Tcl_Obj *CONST *objv)	/* Argument objects. */
{
    PictureImage *imgPtr = clientData;
    Tk_Window tkwin;
    unsigned int flags;
    
    flags = BLT_CONFIG_OBJV_ONLY;
    tkwin = Tk_MainWindow(interp);
    if (objc == 2) {
	return Blt_ConfigureInfoFromObj(interp, tkwin, configSpecs,
	    (char *)imgPtr, (Tcl_Obj *)NULL, flags);
    } else if (objc == 3) {
	return Blt_ConfigureInfoFromObj(interp, tkwin, configSpecs,
	    (char *)imgPtr, objv[2], flags);
    } else {
	if (ConfigurePictureImage(interp, imgPtr, objc - 2, objv + 2, flags) 
	    != TCL_OK) {
	    return TCL_ERROR;
	}
    }
    return TCL_OK;
}


/*
 *--------------------------------------------------------------------------
 *
 * ConvolveOp --
 *
 *--------------------------------------------------------------------------
 */
/*ARGSUSED*/
static int 
ConvolveOp(
    ClientData clientData,	/* Information about picture cmd. */
    Tcl_Interp *interp,		/* Interpreter to report errors back to. */
    int objc,			/* Not used. */
    Tcl_Obj *CONST *objv)
{
    Blt_Picture src;
    ConvolveSwitches switches;
    PictureImage *imgPtr = clientData;

    if (Blt_GetPictureFromObj(interp, objv[2], &src) != TCL_OK) {
	return TCL_ERROR;
    }
    switches.vFilter = bltBoxFilter;
    switches.hFilter = bltBoxFilter;
    switches.filter = NULL;
    if (Blt_ParseSwitches(interp, convolveSwitches, objc - 3, objv + 3, 
	&switches, BLT_SWITCH_DEFAULTS) < 0) {
	return TCL_ERROR;
    }
    if (switches.filter != NULL) {
	switches.hFilter = switches.vFilter = switches.filter;
    }
    Blt_ConvolvePicture(imgPtr->picture, src, switches.vFilter, 
	switches.hFilter);
    Blt_NotifyImageChanged(imgPtr);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------------------
 *
 * CopyOp --
 *
 * Results:
 *	Returns a standard Tcl return value.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------------------
 */
static int
CopyOp(
    ClientData clientData,	/* Information about picture cmd. */
    Tcl_Interp *interp,		/* Current interpreter. */
    int objc,			/* Number of arguments. */
    Tcl_Obj *CONST *objv)	/* Argument objects. */
{
    Blt_Picture src;
    CopySwitches switches;
    PictureImage *imgPtr = clientData;

    if (Blt_GetPictureFromObj(interp, objv[2], &src) != TCL_OK) {
	return TCL_ERROR;
    }
    switches.from.x = switches.from.y = 0;
    switches.from.width = Blt_PictureWidth(src);
    switches.from.height = Blt_PictureHeight(src);
    switches.to.x = switches.to.y = 0;
    switches.to.width = Blt_PictureWidth(src);
    switches.to.height = Blt_PictureHeight(src);

    if (Blt_ParseSwitches(interp, copySwitches, objc - 3, objv + 3, &switches, 
	BLT_SWITCH_DEFAULTS) < 0) {
	return TCL_ERROR;
    }
    if (!Blt_AdjustBBoxToPicture(src, &switches.from)) {
	Tcl_AppendResult(interp, "impossible coordinates for region", 
			 (char *)NULL);
	return TCL_ERROR;
    }
    if (!Blt_AdjustBBoxToPicture(imgPtr->picture, &switches.to)) {
	Tcl_AppendResult(interp, "impossible coordinates for region", 
			 (char *)NULL);
	return TCL_ERROR;
    }
    Blt_CopyPictureArea(imgPtr->picture, src, switches.from.x, 
	switches.from.y, switches.from.width, switches.from.height,
	switches.to.x, switches.to.y);
    Blt_NotifyImageChanged(imgPtr);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------------------
 *
 * CropOp --
 *
 * Results:
 *	Returns a standard Tcl return value.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------------------
 */
static int
CropOp(
    ClientData clientData,	/* Information about picture cmd. */
    Tcl_Interp *interp,		/* Current interpreter. */
    int objc,			/* Number of arguments. */
    Tcl_Obj *CONST *objv)	/* Argument objects. */
{
    BBox bbox;
    Blt_Picture picture;
    PictureImage *imgPtr = clientData;

    if (Blt_GetBBoxFromObjv(interp, objc - 2, objv + 2, &bbox) != TCL_OK) {
	return TCL_ERROR;
    }
    if (!Blt_AdjustBBoxToPicture(imgPtr->picture, &bbox)) {
	Tcl_AppendResult(interp, "impossible coordinates for region", 
			 (char *)NULL);
	return TCL_ERROR;
    }
    picture = Blt_CreatePicture(bbox.width, bbox.height);
    Blt_CopyPictureArea(picture, imgPtr->picture, bbox.x, bbox.y, bbox.width,
	bbox.height, 0, 0);
    Blt_FreePicture(imgPtr->picture);
    imgPtr->picture = picture;
    Blt_NotifyImageChanged(imgPtr);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------------------
 *
 * DupOp --
 *
 *	Creates a duplicate of the current picture in a new picture
 *	image.  The name of the new image is returned.
 *
 * Results:
 *	Returns a standard Tcl return value.  If TCL_OK, The name of
 *	the new image command is returned via the interpreter result.
 *	Otherwise an error message is returned.
 *
 * Side effects:
 *	A new image command is created.
 *
 *--------------------------------------------------------------------------
 */
static int
DupOp(
    ClientData clientData,	/* Information about picture cmd. */
    Tcl_Interp *interp,		/* Current interpreter. */
    int objc,			/* Number of arguments. */
    Tcl_Obj *CONST *objv)	/* Argument objects. */
{
    Blt_Picture picture;
    DupSwitches switches;
    PictureImage *imgPtr = clientData;
    char *imageName;

    memset(&switches, 0, sizeof(switches));
    if (Blt_ParseSwitches(interp, dupSwitches, objc - 2, objv + 2, &switches, 
	BLT_SWITCH_DEFAULTS) < 0) {
	return TCL_ERROR;
    }
    if (!Blt_AdjustBBoxToPicture(imgPtr->picture, &switches.bbox)) {
	Tcl_AppendResult(interp, "impossible coordinates for region", 
			 (char *)NULL);
	return TCL_ERROR;
    }
    /* 
     * Create a new picture image. Let Tk automatically generate the
     * command name. 
     */
    if (Tcl_Eval(interp, "image create picture") != TCL_OK) {
	return TCL_ERROR;
    }
    /* The interpreter result now contains the name of the image. */
    imageName = Tcl_GetString(Tcl_GetObjResult(interp));

    /* 
     * Save the image name and reset the result just in case
     * Blt_ResetPicture returns an error. 
     */
    imageName = Blt_Strdup(imageName);
    Tcl_ResetResult(interp);	
    
    picture = Blt_CreatePicture(switches.bbox.width, switches.bbox.height);
    if (switches.nocopy) {	/* Set the picture to a blank image. */
	Blt_BlankPicture(picture, 0x00000000);
    } else {			/* Copy region into new picture. */
	Blt_CopyPictureArea(picture, imgPtr->picture, switches.bbox.x,
	    switches.bbox.y, switches.bbox.width, switches.bbox.height, 0, 0);
    }

    if (Blt_ResetPicture(interp, imageName, picture) != TCL_OK) {
	Blt_Free(imageName);
	Blt_FreePicture(picture);
	return TCL_ERROR;
    }
    Tcl_SetObjResult(interp, Tcl_NewStringObj(imageName, -1));
    Blt_Free(imageName);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------------------
 *
 * ExportOp --
 *
 * Results:
 *	Returns a standard Tcl return value.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------------------
 */
static int
ExportOp(
    ClientData clientData,	/* Information about picture cmd. */
    Tcl_Interp *interp,		/* Current interpreter. */
    int objc,			/* Number of arguments. */
    Tcl_Obj *CONST *objv)	/* Argument objects. */
{
    Blt_Picture picture, tmp;
    DataSink data;
    Blt_PictureExportSwitches switches;
    PictureImage *imgPtr = clientData;
    int format, result;

    if (objc == 2) {
	for (format = 0; format < PICTURE_XFMT_NUMBER_FORMATS; format++) {
	    if (exportSwitches[format] != NULL) {
		Tcl_AppendElement(interp, FormatToString(format));
	    }
	}
	return TCL_OK;
    }
    if (StringToFormat(interp, Tcl_GetString(objv[2]), &format) != TCL_OK) {
	return TCL_ERROR;
    }
    if (exportSwitches[format] == NULL) {
	Tcl_AppendResult(interp, "no export registered for \"", 
			 FormatToString(format), "\"", (char *)NULL);
	return TCL_ERROR;
    }
    memset(&switches, 0, sizeof(switches));
    switches.bg.color = 0xFFFFFFFF; /* Default bgcolor is white. */
    bltColorSwitch.clientData = imgPtr;
    if (Blt_ParseSwitches(interp, exportSwitches[format], objc - 3, objv + 3, 
	&switches, BLT_SWITCH_DEFAULTS) < 0) {
	return TCL_ERROR;
    }
    if ((switches.dataObjPtr != NULL) && (switches.fileObjPtr != NULL)) {
	Tcl_AppendResult(interp, "more than one import destination: ",
		"use only one -file or -data flag.", (char *)NULL);
	return TCL_ERROR;
    }
    Blt_SinkInit(&data);
    picture = imgPtr->picture;
    tmp = NULL;
    if (Blt_PictureFlags(picture) & BLT_PICTURE_PREMULTIPLIED_ALPHAS) {
	/* 
	 * The picture has an alpha burned into the components.
	 * Create a temporary copy removing pre-multiplied alphas.
	 */ 
	tmp = Blt_ClonePicture(picture);
	Blt_UnmultiplyAlphas(tmp);
	picture = tmp;
    }

    /* 
     * Based upon the format, convert the picture into a binary buffer
     * and then convert to a base64.
     */
    switch (format) {
#ifdef HAVE_LIBJPEG
    case PICTURE_XFMT_JPG:
	result = Blt_PictureToJpg(interp, picture, &data, &switches);
	break;
#endif /* HAVE_LIBJPEG */

#ifdef HAVE_LIBTIFF
    case PICTURE_XFMT_TIF:
	result = Blt_PictureToTif(interp, picture, &data, switches.compress);
	break;
#endif /* HAVE_LIBTIFF */

#ifdef HAVE_LIBPNG
    case PICTURE_XFMT_PNG:
	result = Blt_PictureToPng(interp, picture, &data);
	break;
#endif /* HAVE_LIBPNG */

#ifdef HAVE_LIBXPM
    case PICTURE_XFMT_XPM:
	result = Blt_PictureToXpm(interp, picture, &data, 
		((switches.flags & PICTURE_NO_QUANTIZE) == 0), 
		&switches.bg);
	break;
#endif /* HAVE_LIBXPM */

    case PICTURE_XFMT_XBM:
	result = Blt_PictureToXbm(interp, picture, &data, &switches.bg);
	break;

#ifdef notdef
#ifdef WIN32
    case PICTURE_XFMT_EMF:
	result = Blt_PictureToEmf(interp, picture, &data);
	break;

    case PICTURE_XFMT_WMF:
	result = Blt_PictureToWmf(interp, picture, &data);
	break;
#endif
#endif

    case PICTURE_XFMT_PHOTO:
	result = TCL_ERROR;
	if (switches.imageObjPtr == NULL) {
	    Tcl_AppendResult(interp, "no photo name specified with \"-image\"",
			     (char *)NULL);
	}  else {
	    Tk_PhotoHandle photo;

	    photo = Tk_FindPhoto(interp, Tcl_GetString(switches.imageObjPtr));
	    if (photo == NULL) {
		Tcl_AppendResult(interp, "can't find photo \"",
				 Tcl_GetString(switches.imageObjPtr), "\"", 
				 (char *)NULL);
	    } else {
		Blt_PictureToPhoto(picture, photo);
		result = TCL_OK;
	    }
	}
	break;

    default:
	Tcl_AppendResult(interp, "unknown image format \"", 
		Tcl_GetString(objv[2]), "\"", (char *)NULL);
	goto error;
    }

    if (tmp != NULL) {
	Blt_FreePicture(tmp);
	tmp = NULL;
    }
    if (result != TCL_OK) {
	Tcl_AppendResult(interp, "can't convert \"", 
		Tcl_GetString(objv[2]), "\"", (char *)NULL);
	goto error;
    }
    if (switches.fileObjPtr != NULL) {
	result = WriteImageToFile(interp, &data, 
		Tcl_GetString(switches.fileObjPtr));
    } else {
	if ((format == PICTURE_XFMT_XBM) || (format == PICTURE_XFMT_XPM)) {
	    Tcl_SetObjResult(interp, 
		Tcl_NewStringObj(Blt_SinkBuffer(&data),
		Blt_SinkMark(&data)));
	} else {
	    char *string;

	    string = Blt_EncodeBase64(interp, &data);
	    if (string == NULL) {
		goto error;
	    }
	    Tcl_SetObjResult(interp, Tcl_NewStringObj(string, -1));
	    Blt_Free(string);
	}
    } 
    Blt_SinkFree(&data);
    return TCL_OK;
 error:
    if (tmp != NULL) {
	Blt_FreePicture(tmp);
    }
    Blt_SinkFree(&data);
    return TCL_ERROR;
}

/*
 *--------------------------------------------------------------------------
 *
 * FadeOp --
 *
 *--------------------------------------------------------------------------
 */
/*ARGSUSED*/
static int 
FadeOp(
    ClientData clientData,	/* Information about picture cmd. */
    Tcl_Interp *interp,		/* Interpreter to report errors back to. */
    int objc,			/* Not used. */
    Tcl_Obj *CONST *objv)
{
    Blt_Picture src;
    PictureImage *imgPtr = clientData;
    double fade;

    if (Blt_GetPictureFromObj(interp, objv[2], &src) != TCL_OK) {
	return TCL_ERROR;
    }
    if (Tcl_GetDoubleFromObj(interp, objv[3], &fade) != TCL_OK) {
	return TCL_ERROR;
    }
    Blt_FadePicture(imgPtr->picture, src, fade);
    Blt_NotifyImageChanged(imgPtr);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------------------
 *
 * FlipOp --
 *
 *	Flips the picture either horizontally or vertically.
 *
 * Results:
 *	Returns a standard Tcl return value.  If TCL_OK, the
 *	components of the pixel are returned as a list in the
 *	interpreter result.  Otherwise an error message is returned.
 *
 *---------------------------------------------------------------------------
 */
static int
FlipOp(
    ClientData clientData,	/* Information about picture cmd. */
    Tcl_Interp *interp,		/* Current interpreter. */
    int objc,			/* Number of arguments. */
    Tcl_Obj *CONST *objv)	/* Argument objects. */
{
    PictureImage *imgPtr = clientData;
    char *string;
    int isVertical;

    string = Tcl_GetString(objv[2]);
    if ((string[0] ==  'x') && (string[1] == '\0')) {
	isVertical = FALSE;
    } else if ((string[0] ==  'y') && (string[1] == '\0')) {
	isVertical = TRUE;
    } else {
	Tcl_AppendResult(interp, "bad flip argument \"", string, 
		"\": should be x or y", (char *)NULL);
	return TCL_ERROR;
    }
    Blt_FlipPicture(imgPtr->picture, isVertical);
    Blt_NotifyImageChanged(imgPtr);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------------------
 *
 * GetOp --
 *
 *	Returns the RGBA components of the pixel at the specified
 *	coordinate.
 *
 * Results:
 *	Returns a standard Tcl return value.  If TCL_OK, the
 *	components of the pixel are returned as a list in the
 *	interpreter result.  Otherwise an error message is returned.
 *
 *---------------------------------------------------------------------------
 */
static int
GetOp(
    ClientData clientData,	/* Information about picture cmd. */
    Tcl_Interp *interp,		/* Current interpreter. */
    int objc,			/* Number of arguments. */
    Tcl_Obj *CONST *objv)	/* Argument objects. */
{
    PictureImage *imgPtr = clientData;
    Pix32 *pixelPtr;
    char string[20];
    int x, y;

    if ((Tcl_GetIntFromObj(interp, objv[2], &x) != TCL_OK) ||
	(Tcl_GetIntFromObj(interp, objv[3], &y) != TCL_OK)) {
	return TCL_ERROR;
    }
    if ((x < 0) || (x >= Blt_PictureWidth(imgPtr->picture))) {
	Tcl_AppendResult(interp, "x-coordinate \"", Tcl_GetString(objv[2]),
		"\" is out of range", (char *)NULL);
	return TCL_ERROR;
    }
    if ((y < 0) || (y >= Blt_PictureHeight(imgPtr->picture))) {
	Tcl_AppendResult(interp, "y-coordinate \"", Tcl_GetString(objv[3]),
		"\" is out of range", (char *)NULL);
	return TCL_ERROR;
    }
    pixelPtr = Blt_PicturePixel(imgPtr->picture, x, y);
    sprintf(string, "#%02x%02x%02x%02x", 
	    pixelPtr->Alpha, pixelPtr->Red, pixelPtr->Green, pixelPtr->Blue);
    Tcl_SetObjResult(interp, Tcl_NewStringObj(string, -1));
    return TCL_OK;
}

/*
 *--------------------------------------------------------------------------
 *
 * GradientOp --
 *
 *	Flips the picture either horizontally or vertically.
 *
 * Results:
 *	Returns a standard Tcl return value.  If TCL_OK, the
 *	components of the pixel are returned as a list in the
 *	interpreter result.  Otherwise an error message is returned.
 *
 *---------------------------------------------------------------------------
 */
static int
GradientOp(
    ClientData clientData,	/* Information about picture cmd. */
    Tcl_Interp *interp,		/* Current interpreter. */
    int objc,			/* Number of arguments. */
    Tcl_Obj *CONST *objv)	/* Argument objects. */
{
    PictureImage *imgPtr = clientData;
    GradientSwitches switches;

    memset(&switches, 0, sizeof(switches));
    bltColorSwitch.clientData = imgPtr;
    switches.fg.color = 0xFFFFFFFF;
    switches.bg.color = 0xFF000000;
    switches.gradient.shape = BLT_GRADIENT_SHAPE_LINEAR;
    switches.gradient.path = BLT_GRADIENT_PATH_X; 
    switches.gradient.logScale = TRUE;
    switches.gradient.jitter = FALSE;
   if (Blt_ParseSwitches(interp, gradientSwitches, objc - 2, objv + 2, 
	&switches, BLT_SWITCH_DEFAULTS) < 0) {
	return TCL_ERROR;
    }
    Blt_GradientPicture(imgPtr->picture, &switches.fg, &switches.bg, 
	&switches.gradient);
    if ((switches.bg.Alpha != 0xFF) || (switches.fg.Alpha != 0xFF)) {
	imgPtr->picture->flags |= BLT_PICTURE_BLEND;
    }
    Blt_PremultiplyAlphas(imgPtr->picture);
    Blt_NotifyImageChanged(imgPtr);
    return TCL_OK;
}


/*
 *--------------------------------------------------------------------------
 *
 * GreyscaleOp --
 *
 *--------------------------------------------------------------------------
 */
/*ARGSUSED*/
static int 
GreyscaleOp(
    ClientData clientData,	/* Information about picture cmd. */
    Tcl_Interp *interp,		/* Interpreter to report errors back to. */
    int objc,			/* Not used. */
    Tcl_Obj *CONST *objv)
{
    Blt_Picture src, dest;
    PictureImage *imgPtr = clientData;

    if (Blt_GetPictureFromObj(interp, objv[2], &src) != TCL_OK) {
	return TCL_ERROR;
    }
    dest = Blt_GreyscalePicture(src);
    if (imgPtr->picture != NULL) {
	Blt_FreePicture(imgPtr->picture);
    } 
    imgPtr->picture = dest;
    Blt_NotifyImageChanged(imgPtr);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------------------
 *
 * HeightOp --
 *	Returns the current height of the picture.
 *
 *--------------------------------------------------------------------------
 */
static int
HeightOp(
    ClientData clientData,	/* Information about picture cmd. */
    Tcl_Interp *interp,		/* Current interpreter. */
    int objc,			/* # of arguments. */
    Tcl_Obj *CONST *objv)	/* Argument vector. */
{
    PictureImage *imgPtr = clientData;
    int newHeight;

    newHeight = 0;
    if (objc == 3) {
	int width;

	if (Tcl_GetIntFromObj(interp, objv[2], &newHeight) != TCL_OK) {
	    return TCL_ERROR;
	}
	width = Blt_PictureWidth(imgPtr->picture);

	if (!Blt_ReallocatePicture(imgPtr->picture, width, newHeight, TRUE)) {
	    Tcl_AppendResult(interp, "can't allocate memory for new width",
			     (char *)NULL);
	    return TCL_ERROR;
	}
	Blt_NotifyImageChanged(imgPtr);
    } 
    if (imgPtr->picture != NULL) {
	newHeight = Blt_PictureHeight(imgPtr->picture);
    }
    Tcl_SetIntObj(Tcl_GetObjResult(interp), newHeight);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------------------
 *
 * ImportOp --
 *
 *	Imports an image source into a picture.  The image source can
 *	be a file, base64 string, or binary Tcl_Obj.  This performs 
 *	basically the same function as "configure".  The only extra 
 *	utility this routine has is the ability to pass extra flags to 
 *	the various converters, something that can't be done really be 
 *	done with 
 *		$image configure -file file.jpg
 *
 * Results:
 *	Returns a standard Tcl return value.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------------------
 */
static int
ImportOp(
    ClientData clientData,	/* Information about picture cmd. */
    Tcl_Interp *interp,		/* Current interpreter. */
    int objc,			/* Number of arguments. */
    Tcl_Obj *CONST *objv)	/* Argument objects. */
{
    DataSink srcSink;
    Blt_Picture picture;
    Blt_PictureImportSwitches switches;
    PictureImage *imgPtr = clientData;
    char *string;
    int format;

    if (objc == 2) {
	for (format = 0; format < PICTURE_XFMT_NUMBER_FORMATS; format++) {
	    if (importSwitches[format] != NULL) {
		Tcl_AppendElement(interp, FormatToString(format));
	    }
	}
	return TCL_OK;
    }
    if (StringToFormat(interp, Tcl_GetString(objv[2]), &format) != TCL_OK) {
	return TCL_ERROR;
    }
    if (importSwitches[format] == NULL) {
	Tcl_AppendResult(interp, "no import registered for \"", 
			 FormatToString(format), "\"", (char *)NULL);
	return TCL_ERROR;
    }
    memset(&switches, 0, sizeof(switches));
    if (Blt_ParseSwitches(interp, importSwitches[format], objc - 3, objv + 3, 
	&switches, BLT_SWITCH_DEFAULTS) < 0) {
	return TCL_ERROR;
    }
    if ((switches.dataObjPtr != NULL) && (switches.fileObjPtr != NULL)) {
	Tcl_AppendResult(interp, "more than one import source: ",
		"use only one -file or -data flag.", (char *)NULL);
	return TCL_ERROR;
    }
    string = NULL;
    Blt_SinkInit(&srcSink);
    if (switches.dataObjPtr != NULL) {
	if ((switches.dataObjPtr->typePtr != NULL) && 
	    (strcmp(switches.dataObjPtr->typePtr->name, "bytearray") == 0)) {
	    int length;

	    srcSink.bytes = Tcl_GetByteArrayFromObj(switches.dataObjPtr, 
		&length);
	    Blt_SinkSetMark(&srcSink, length);
	} else if (Blt_DecodeBase64(interp, Tcl_GetString(switches.dataObjPtr), 
		&srcSink) != TCL_OK) {
	    return TCL_ERROR;
	}
	string = "data buffer";
    } else if (switches.fileObjPtr != NULL) {
	string = Tcl_GetString(switches.fileObjPtr);
	if (ReadImageFromFile(interp, string, &srcSink) != TCL_OK) {
	    return TCL_ERROR;
	}
    }
    switch (format) {

#ifdef HAVE_LIBJPEG
    case PICTURE_XFMT_JPG:
	picture = Blt_JpgToPicture(interp, string,  &srcSink, &switches);
	break;
#endif /* HAVE_LIBJPEG */

#ifdef HAVE_LIBTIFF
    case PICTURE_XFMT_TIF:
	picture = Blt_TifToPicture(interp, string, &srcSink);
	break;
#endif /* HAVE_LIBTIFF */

#ifdef HAVE_LIBPNG
    case PICTURE_XFMT_PNG:
	picture = Blt_PngToPicture(interp, string, &srcSink);
	break;
#endif /* HAVE_LIBPNG */

#ifdef HAVE_LIBXPM
    case PICTURE_XFMT_XPM:
	picture = Blt_XpmToPicture(interp, string, &srcSink);
	break;
#endif /* HAVE_LIBXPM */

    case PICTURE_XFMT_XBM:
	picture = Blt_XbmToPicture(interp, string, &srcSink, &switches.fg, 
		&switches.bg);
	break;

#ifdef notdef

#ifdef WIN32
    case PICTURE_XFMT_EMF:
	picture = Blt_EmfToPicture(interp, string, &srcSink);
	break;

    case PICTURE_XFMT_WMF:
	picture = Blt_WmfToPicture(interp, string, &srcSink);
	break;

#endif
#endif

    case PICTURE_XFMT_GIF:
	picture = Blt_GifToPicture(interp, string, &srcSink, 
		switches.imageIndex);
	break;

    case PICTURE_XFMT_PHOTO:
	picture = NULL;
	if (switches.imageObjPtr == NULL) {
	    Tcl_AppendResult(interp, "no photo name specified with \"-image\"",
			     (char *)NULL);
	}  else {
	    Tk_PhotoHandle photo;

	    photo = Tk_FindPhoto(interp, Tcl_GetString(switches.imageObjPtr));
	    if (photo == NULL) {
		Tcl_AppendResult(interp, "can't find photo \"",
				 Tcl_GetString(switches.imageObjPtr), "\"", 
				 (char *)NULL);
	    } else {
		picture = Blt_PhotoToPicture(photo);
	    }
	}
	break;

    default:
	picture = NULL;
    }
    Blt_SinkFree(&srcSink);
    if (picture == NULL) {
	return TCL_ERROR;
    }
    if (imgPtr->picture != NULL) {
	Blt_FreePicture(imgPtr->picture);
    }
    imgPtr->picture = picture;
    imgPtr->readFormat = format;
    imgPtr->flags &= ~IMPORTED_MASK;
    if (imgPtr->name != NULL) {
	Blt_Free(imgPtr->name);
    }
    if (switches.fileObjPtr == NULL) {
	imgPtr->name = NULL;
	imgPtr->flags |= IMPORTED_DATA;
    } else {
	imgPtr->name = Blt_Strdup(Tcl_GetString(switches.fileObjPtr));
	imgPtr->flags |= IMPORTED_FILE;
    }
    Blt_NotifyImageChanged(imgPtr);
    return TCL_OK;
}


/*
 *--------------------------------------------------------------------------
 *
 * InfoOp --
 *
 *	Reports the basic information about a picture.  
 *
 * Results:
 *	Returns a standard Tcl return value.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------------------
 */
static int
InfoOp(
    ClientData clientData,	/* Information about picture cmd. */
    Tcl_Interp *interp,		/* Current interpreter. */
    int objc,			/* Number of arguments. */
    Tcl_Obj *CONST *objv)	/* Argument objects. */
{
    PictureImage *imgPtr = clientData;
    int nColors;
    char *string;
    Tcl_Obj *listObjPtr, *objPtr;

    nColors = Blt_QueryColors(imgPtr->picture, (Blt_HashTable *)NULL);
    listObjPtr = Tcl_NewListObj(0, (Tcl_Obj **) NULL);

    objPtr = Tcl_NewStringObj("colors", -1);
    Tcl_ListObjAppendElement(interp, listObjPtr, objPtr);
    objPtr = Tcl_NewIntObj(nColors);
    Tcl_ListObjAppendElement(interp, listObjPtr, objPtr);

    objPtr = Tcl_NewStringObj("type", -1);
    Tcl_ListObjAppendElement(interp, listObjPtr, objPtr);
    string = Blt_PictureIsColor(imgPtr->picture) ? "color" : "greyscale";
    objPtr = Tcl_NewStringObj(string, -1);
    Tcl_ListObjAppendElement(interp, listObjPtr, objPtr);

    objPtr = Tcl_NewStringObj("opacity", -1);
    Tcl_ListObjAppendElement(interp, listObjPtr, objPtr);
    if (Blt_PictureIsBlended(imgPtr->picture)) {
	string = "blended";
    } else if (Blt_PictureIsMasked(imgPtr->picture)) {
	string = "masked";
    } else {
	string = "full";
    }
    objPtr = Tcl_NewStringObj(string, -1);
    Tcl_ListObjAppendElement(interp, listObjPtr, objPtr);

    objPtr = Tcl_NewStringObj("width", -1);
    Tcl_ListObjAppendElement(interp, listObjPtr, objPtr);
    objPtr = Tcl_NewIntObj(Blt_PictureWidth(imgPtr->picture));
    Tcl_ListObjAppendElement(interp, listObjPtr, objPtr);
    
    objPtr = Tcl_NewStringObj("height", -1);
    Tcl_ListObjAppendElement(interp, listObjPtr, objPtr);
    objPtr = Tcl_NewIntObj(Blt_PictureHeight(imgPtr->picture));
    Tcl_ListObjAppendElement(interp, listObjPtr, objPtr);

    Tcl_SetObjResult(interp, listObjPtr);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------------------
 *
 * MultiplyOp --
 *
 *	$image multiply scalar
 *
 *--------------------------------------------------------------------------
 */
/*ARGSUSED*/
static int 
MultiplyOp(
    ClientData clientData,	/* Information about picture cmd. */
    Tcl_Interp *interp,		/* Interpreter to report errors back to. */
    int objc,			/* Not used. */
    Tcl_Obj *CONST *objv)
{
    PictureImage *imgPtr = clientData;
    double scalar;

    if (Tcl_GetDoubleFromObj(interp, objv[2], &scalar) != TCL_OK) {
	return TCL_ERROR;
    }
    Blt_MultiplyPixels(imgPtr->picture, (float)scalar);
    Blt_NotifyImageChanged(imgPtr);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------------------
 *
 * PutOp --
 *
 * Results:
 *	Returns a standard Tcl return value.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------------------
 */
static int
PutOp(
    ClientData clientData,	/* Information about picture cmd. */
    Tcl_Interp *interp,		/* Current interpreter. */
    int objc,			/* Number of arguments. */
    Tcl_Obj *CONST *objv)	/* Argument objects. */
{
    return TCL_OK;
}

/*
 *--------------------------------------------------------------------------
 *
 * QuantizeOp --
 *
 *	$dest quantize $src 256
 *
 *--------------------------------------------------------------------------
 */
/*ARGSUSED*/
static int 
QuantizeOp(
    ClientData clientData,	/* Information about picture cmd. */
    Tcl_Interp *interp,		/* Interpreter to report errors back to. */
    int objc,			/* Not used. */
    Tcl_Obj *CONST *objv)
{
    Blt_Picture src, dest;
    PictureImage *imgPtr = clientData;
    int nColors;

    if (Blt_GetPictureFromObj(interp, objv[2], &src) != TCL_OK) {
	return TCL_ERROR;
    }
    if (Tcl_GetIntFromObj(interp, objv[3], &nColors) != TCL_OK) {
	return TCL_ERROR;
    }
    dest = Blt_QuantizePicture(src, nColors);
    if (dest == NULL) {
	return TCL_ERROR;
    }
    if (imgPtr->picture != NULL) {
	Blt_FreePicture(imgPtr->picture);
    } 
    imgPtr->picture = dest;
    Blt_NotifyImageChanged(imgPtr);
    return TCL_OK;
}


/*
 *-------------------------------------------------------------------------- 
 *
 * ResampleOp --
 *
 *--------------------------------------------------------------------------
 */
/*ARGSUSED*/
static int 
ResampleOp(
    ClientData clientData,	/* Information about picture cmd. */
    Tcl_Interp *interp,		/* Interpreter to report errors back to. */
    int objc,			/* Not used. */
    Tcl_Obj *CONST *objv)
{
    Blt_Picture src, tmp;
    Blt_PictureFilter hFilter, vFilter;
    PictureImage *imgPtr = clientData;
    ResampleSwitches switches;
    int destWidth, destHeight;

    if (Blt_GetPictureFromObj(interp, objv[2], &src) != TCL_OK) {
	return TCL_ERROR;
    }
    switches.bbox.x = switches.bbox.y = 0;
    switches.bbox.width = Blt_PictureWidth(src);
    switches.bbox.height = Blt_PictureHeight(src);

    switches.filter = NULL;
    if (Blt_ParseSwitches(interp, resampleSwitches, objc - 3, objv + 3, 
	&switches, BLT_SWITCH_DEFAULTS) < 0) {
	return TCL_ERROR;
    }
    if (!Blt_AdjustBBoxToPicture(src, &switches.bbox)) {
	Tcl_AppendResult(interp, "impossible coordinates for region", 
			 (char *)NULL);
	return TCL_ERROR;
    }
    destWidth = Blt_PictureWidth(imgPtr->picture);
    destHeight = Blt_PictureHeight(imgPtr->picture);
    if (switches.filter == NULL) {
	if (switches.bbox.width < destWidth) {
	    hFilter = bltMitchellFilter;
	} else {
	    hFilter = bltBoxFilter;
	}
	if (switches.bbox.height < destHeight) {
	    vFilter = bltMitchellFilter;
	} else {
	    vFilter = bltBoxFilter;
	}
    } else {
	hFilter = vFilter = switches.filter;
    }
    tmp = Blt_CreatePicture(switches.bbox.width, switches.bbox.height);
    Blt_CopyPictureArea(tmp, src, switches.bbox.x, switches.bbox.y, 
	switches.bbox.width, switches.bbox.height, 0, 0);
    Blt_ResamplePicture(imgPtr->picture, tmp, vFilter, hFilter);
    Blt_FreePicture(tmp);
    Blt_NotifyImageChanged(imgPtr);
    return TCL_OK;
}


/*
 *--------------------------------------------------------------------------
 *
 * RotateOp --
 *
 *--------------------------------------------------------------------------
 */
/*ARGSUSED*/
static int 
RotateOp(
    ClientData clientData,	/* Information about picture cmd. */
    Tcl_Interp *interp,		/* Interpreter to report errors back to. */
    int objc,			/* Not used. */
    Tcl_Obj *CONST *objv)
{
    Blt_Picture src, dest;
    PictureImage *imgPtr = clientData;
    double angle;

    if (Blt_GetPictureFromObj(interp, objv[2], &src) != TCL_OK) {
	return TCL_ERROR;
    }
    if (Tcl_GetDoubleFromObj(interp, objv[3], &angle) != TCL_OK) {
	char *string;

	string = Tcl_GetString(objv[3]);
	if (Tcl_ExprDouble(interp, string, &angle) != TCL_OK) {
	    return TCL_ERROR;
	}
    }
    dest = Blt_RotatePicture(src, angle);
    if (imgPtr->picture != NULL) {
	Blt_FreePicture(imgPtr->picture);
    } 
    imgPtr->picture = dest;
    Blt_NotifyImageChanged(imgPtr);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------------------
 *
 * SelectOp --
 *
 * Results:
 *	Returns a standard Tcl return value.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------------------
 */
static int
SelectOp(
    ClientData clientData,	/* Information about picture cmd. */
    Tcl_Interp *interp,		/* Current interpreter. */
    int objc,			/* Number of arguments. */
    Tcl_Obj *CONST *objv)	/* Argument objects. */
{
    Blt_Picture src;
    PictureImage *imgPtr = clientData;
    Pix32 lower, upper;
    unsigned char tmp;

    if (Blt_GetPictureFromObj(interp, objv[2], &src) != TCL_OK) {
	return TCL_ERROR;
    }
    if (Blt_GetPix32FromObj(interp, objv[3], &lower) != TCL_OK) {
	return TCL_ERROR;
    }
    if (objc == 5) {
	if (Blt_GetPix32FromObj(interp, objv[4], &upper) != TCL_OK) {
	    return TCL_ERROR;
	}
    } else {
	upper.color = lower.color;
    }
    if (lower.Red > upper.Red) {
	tmp = lower.Red, lower.Red = upper.Red, upper.Red = tmp;
    }
    if (lower.Green > upper.Green) {
	tmp = lower.Green, lower.Green = upper.Green, upper.Green = tmp;
    }
    if (lower.Blue > upper.Blue) {
	tmp = lower.Blue, lower.Blue = upper.Blue, upper.Blue = tmp;
    }
    if (lower.Alpha > upper.Alpha) {
	tmp = lower.Alpha, lower.Alpha = upper.Alpha, upper.Alpha = tmp;
    }
    Blt_SelectPixels(imgPtr->picture, src, &lower, &upper);
    Blt_NotifyImageChanged(imgPtr);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------------------
 *
 * SnapOp --
 *
 * Results:
 *	Returns a standard Tcl return value.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------------------
 */
static int
SnapOp(
    ClientData clientData,	/* Information about picture cmd. */
    Tcl_Interp *interp,		/* Current interpreter. */
    int objc,			/* Number of arguments. */
    Tcl_Obj *CONST *objv)	/* Argument objects. */
{
    return TCL_OK;
}


/*
 *--------------------------------------------------------------------------
 *
 * TileOp --
 *
 * Results:
 *	Returns a standard Tcl return value.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------------------
 */
static int
TileOp(
    ClientData clientData,	/* Information about picture cmd. */
    Tcl_Interp *interp,		/* Current interpreter. */
    int objc,			/* Number of arguments. */
    Tcl_Obj *CONST *objv)	/* Argument objects. */
{
    PictureImage *imgPtr = clientData;
    Blt_Picture src;
    TileSwitches switches;

    if (Blt_GetPictureFromObj(interp, objv[2], &src) != TCL_OK) {
	return TCL_ERROR;
    }
    switches.xOrigin = switches.yOrigin = 0;
    switches.bbox.x = switches.bbox.y = 0;
    switches.bbox.width = Blt_PictureWidth(imgPtr->picture);
    switches.bbox.height = Blt_PictureHeight(imgPtr->picture);

    if (Blt_ParseSwitches(interp, tileSwitches, objc - 3, objv + 3, &switches, 
	BLT_SWITCH_DEFAULTS) < 0) {
	return TCL_ERROR;
    }
    if (!Blt_AdjustBBoxToPicture(imgPtr->picture, &switches.bbox)) {
	Tcl_AppendResult(interp, "impossible coordinates for region", 
			 (char *)NULL);
	return TCL_ERROR;
    }
    Blt_TilePicture(imgPtr->picture, src, switches.xOrigin, switches.yOrigin, 
            switches.bbox.x, switches.bbox.y, switches.bbox.width,
	    switches.bbox.height);
    Blt_NotifyImageChanged(imgPtr);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------------------
 *
 * WidthOp --
 *	Returns the current width of the picture.
 *
 *--------------------------------------------------------------------------
 */
static int
WidthOp(
    ClientData clientData,	/* Information about picture cmd. */
    Tcl_Interp *interp,		/* Current interpreter. */
    int objc,			/* # of arguments. */
    Tcl_Obj *CONST *objv)	/* Argument vector. */
{
    PictureImage *imgPtr = clientData;
    int width;

    if (objc == 3) {
	int height;

	if (Tcl_GetIntFromObj(interp, objv[2], &width) != TCL_OK) {
	    return TCL_ERROR;
	}
	if (width < 0) {
	    Tcl_AppendResult(interp, "bad width \"", Tcl_GetString(objv[2]), 
			     "\"", (char *)NULL);
	    return TCL_ERROR;
	}
	height = Blt_PictureHeight(imgPtr->picture);
	if (!Blt_ReallocatePicture(imgPtr->picture, width, height, TRUE)) {
	    Tcl_AppendResult(interp, "can't allocate memory for new width",
			     (char *)NULL);
	}
	Blt_NotifyImageChanged(imgPtr);
    } 
    width = Blt_PictureWidth(imgPtr->picture);
    Tcl_SetIntObj(Tcl_GetObjResult(interp), width);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------------------
 *
 * Picture instance sub-command specification:
 *
 *	- Name of the sub-command.
 *	- Minimum number of characters needed to unambiguously
 *        recognize the sub-command.
 *	- Pointer to the function to be called for the sub-command.
 *	- Minimum number of arguments accepted.
 *	- Maximum number of arguments accepted.
 *	- String to be displayed for usage (arguments only).
 *
 *--------------------------------------------------------------------------
 */
static Blt_OpSpec pictureInstOps[] =
{
    {"add",       2, (Blt_Op)ArithOp,     3, 3, "image|color",},
    {"and",       2, (Blt_Op)ArithOp,     3, 3, "image|color",},
    {"blank",     3, (Blt_Op)BlankOp,     2, 3, "?color?",},
    {"blend",     3, (Blt_Op)BlendOp,     4, 0, "bg fg ?switches?",},
    {"blur",      3, (Blt_Op)BlurOp,      2, 2, "",},
    {"bold",      2, (Blt_Op)Blend2Op,    4, 0, "bg fg ?switches?",},
    {"cget",      2, (Blt_Op)CgetOp,      3, 3, "option",},
    {"configure", 4, (Blt_Op)ConfigureOp, 2, 0, "?option value?...",},
    {"convolve",  4, (Blt_Op)ConvolveOp,  3, 0, "src ?switches?",},
    {"copy",      3, (Blt_Op)CopyOp,      3, 0, "srcPict ?switches?",},
    {"crop",      3, (Blt_Op)CropOp,      2, 0, "?switches?",},
    {"draw",      2, (Blt_Op)Blt_PictureDrawOp, 2, 0, "?args?",},
    {"dup",       2, (Blt_Op)DupOp,       2, 0, "?switches?",},
    {"export",    1, (Blt_Op)ExportOp,    2, 0, "format ?switches?...",},
    {"fade",      2, (Blt_Op)FadeOp,      4, 4, "src factor",},
    {"flip",      2, (Blt_Op)FlipOp,      3, 0, "x|y",},
    {"get",       2, (Blt_Op)GetOp,       4, 4, "x y",},
    {"gradient",  3, (Blt_Op)GradientOp,  2, 0, "?switches?",},
    {"greyscale", 3, (Blt_Op)GreyscaleOp, 3, 3, "src",},
    {"height",    1, (Blt_Op)HeightOp,    2, 3, "?newHeight?",},
    {"import",    2, (Blt_Op)ImportOp,    2, 0, "format ?switches?...",},
    {"info",      2, (Blt_Op)InfoOp,      2, 2, "info",},
    {"max",	  2, (Blt_Op)ArithOp,     3, 3, "image|color",},
    {"min",	  2, (Blt_Op)ArithOp,     3, 3, "image|color",},
    {"multiply",  2, (Blt_Op)MultiplyOp,  3, 3, "float",},
    {"nand",      2, (Blt_Op)ArithOp,     3, 3, "image|color",},
    {"nor",       2, (Blt_Op)ArithOp,     3, 3, "image|color",},
    {"or",        1, (Blt_Op)ArithOp,     3, 3, "image|color",},
    {"put",       1, (Blt_Op)PutOp,       2, 0, "color ?window?...",},
    {"quantize",  1, (Blt_Op)QuantizeOp,  4, 4, "src numColors",},
    {"resize",    2, (Blt_Op)ResampleOp,  4, 0, "src ?switches?",},
    {"rotate",    2, (Blt_Op)RotateOp,    4, 4, "src angle",},
    {"select",    2, (Blt_Op)SelectOp,    4, 5, "src color ?color?",},
    {"subtract",  2, (Blt_Op)ArithOp,     3, 3, "image|color",},
    {"tile",      2, (Blt_Op)TileOp,      3, 0, "image ?switches?",},
    {"width",     1, (Blt_Op)WidthOp,     2, 3, "?newWidth?",},
    {"xor",       1, (Blt_Op)ArithOp,     3, 3, "image|color ?switches?",},
};

static int nPictureInstOps = sizeof(pictureInstOps) / sizeof(Blt_OpSpec);

static int
PictureInstCmdProc(
    ClientData clientData,	/* Information about picture cmd. */
    Tcl_Interp *interp,		/* Current interpreter. */
    int objc,			/* Number of arguments. */
    Tcl_Obj *CONST *objv)	/* Argument objects. */
{
    Blt_Op proc;

    proc = Blt_GetOpFromObj(interp, nPictureInstOps, pictureInstOps, 
	BLT_OP_ARG1, objc, objv, 0);
    if (proc == NULL) {
	return TCL_ERROR;
    }
    return (*proc) (clientData, interp, objc, objv);
}

/*
 *--------------------------------------------------------------------------
 *
 * PictureInstCmdDeleteProc --
 *
 *	This procedure is invoked when a picture command is deleted.
 *
 * Results:
 *	None.
 *
 *--------------------------------------------------------------------------
 */
static void
PictureInstCmdDeletedProc(ClientData clientData) /* Pointer to record. */
{
    PictureImage *imgPtr = clientData;

    imgPtr->cmdToken = NULL;
    if (imgPtr->imgToken != NULL) {
	Tk_DeleteImage(imgPtr->interp, Tk_NameOfImage(imgPtr->imgToken));
    }
}

/*
 *--------------------------------------------------------------------------
 *
 * SharpenOp --
 *
 *--------------------------------------------------------------------------
 */
/*ARGSUSED*/
static int 
SharpenOp(
    Tcl_Interp *interp,		/* Interpreter to report errors back to. */
    int objc,			/* Not used. */
    Tcl_Obj *CONST *objv)
{
    return TCL_OK;
}


/*
 *--------------------------------------------------------------
 *
 * Picture instance sub-command specification:
 *
 *	- Name of the sub-command.
 *	- Minimum number of characters needed to unambiguously
 *        recognize the sub-command.
 *	- Pointer to the function to be called for the sub-command.
 *	- Minimum number of arguments accepted.
 *	- Maximum number of arguments accepted.
 *	- String to be displayed for usage (arguments only).
 *
 *--------------------------------------------------------------
 */
static Blt_OpSpec pictureOps[] =
{
    {"sharpen",   1, (Blt_Op)SharpenOp,   4, 0, "src dest ?switches?",},
#ifdef notdef
    {"blur",      1, (Blt_Op)BlurOp,      4, 0, "src dest ?switches?",},
    {"brighten"   1, (Blt_Op)BrightenOp,  4, 0, "src dest ?switches?",},
    {"darken"     1, (Blt_Op)BrightenOp,  4, 0, "src dest ?switches?",},
    {"medianf"    1, (Blt_Op)MedianOp,    4, 0, "src dest ?switches?",},
    {"translate", 1, (Blt_Op)TranslateOp, 4, 0, "src dest ?switches?",},
#endif
};
static int nPictureOps = sizeof(pictureOps) / sizeof(Blt_OpSpec);

/*
 *--------------------------------------------------------------------------
 *
 * PictureImageCmdProc --
 *
 *--------------------------------------------------------------------------
 */
/*ARGSUSED*/
static int 
PictureImageCmdProc(
    ClientData clientData,	/* Not used. */
    Tcl_Interp *interp,
    int objc,			/* Not used. */
    Tcl_Obj *CONST *objv)
{
    Blt_Op proc;

    proc = Blt_GetOpFromObj(interp, nPictureOps, pictureOps, BLT_OP_ARG1, 
	objc, objv, 0);
    if (proc == NULL) {
	return TCL_ERROR;
    }
    return (*proc)(interp, objc, objv);
}

/*
 *--------------------------------------------------------------------------
 *
 * Blt_PictureInit --
 *
 *	This procedure is invoked to initialize the "tree" command.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Creates the new command and adds a new entry into a global Tcl
 *	associative array.
 *
 *--------------------------------------------------------------------------
 */
int
Blt_PictureInit(Tcl_Interp *interp)
{
    static Blt_InitCmdSpec cmdSpec = { 
	"picture", PictureImageCmdProc, 
    };
    /* cmdSpec.clientData = GetTreeCmdInterpData(interp); */
    return Blt_InitCmd(interp, "blt", &cmdSpec);
}

/*
 *--------------------------------------------------------------------------
 *
 * Blt_RegisterPictureImageType --
 *
 *	Registers the "picture" image type with Tk.
 *
 *--------------------------------------------------------------------------
 */
void
Blt_RegisterPictureImageType() 
{
    static Tk_ImageType imageType = {
	"picture",		
	CreatePictureImage,
	GetPictureCache,		
	DisplayPictureImage,	
	FreePictureCache,	
	DeletePictureImage,	
	PictureImageToPostScript,	
	(Tk_ImageType *)NULL	/* nextPtr */
    };
    Tk_CreateImageType(&imageType);
}
