namespace eval ::kt2c {
    variable Platform ""
    variable Compiler ""
    variable Linker   ""

    proc FileHeader {} {
	global tcl_platform
	
	set retval    "\#include <assert.h>\n"
	
	if {[string equal $tcl_platform(os) Linux]} {
	    append retval "\#include <values.h>\n"
	}

	append retval "\#include \"kproc.h\"\n"
	append retval "\#include \"tcl.h\"\n"
	append retval "\#include \"tclCompile.h\"\n"
	
	return $retval
    }

    proc LibraryInitialiser {libname code} {
	global tcl_platform

	if {[string equal $tcl_platform(platform) windows]} {
	    set retval "__declspec(dllexport) "
	}

	append retval "int ${libname}_Init(Tcl_Interp *interp)\n"
	append retval "{\n"
	append retval ${code}
	append retval "    return 0;\n"
	append retval "}\n"

	return $retval
    }

    proc Compile {filename} {
	variable Platform

	if {[string length $Platform] == 0} {
	    GuessPlatform
	}

	if {[string length $Platform] == 0} {
	    error "Unknown platform."
	}

	puts "Compiling for $Platform"

	return [Compile_${Platform} $filename]
    }

    proc Compile_SunOS_SunPro {filename} {
	variable Compiler
	variable Linker
	variable Prefix

	set base [file rootname $filename]

	# Do not increase the optimisation level beyond -xO2 -- various
	# SunPro compilers have very ugly optimisation bugs. :-(
	exec $Compiler -xO2 -I${Prefix}/include -I${Prefix}/tcl8.4/generic -c \
	    ${filename}
	exec $Linker -G -o lib${base}.so ${base}.o -L${Prefix}/lib -ltcl8.4 \
	    -R${Prefix}/lib
	return [file join [pwd] lib${base}.so]
    }

    proc Compile_Posix_GCC {filename} {
	variable Compiler
	variable Linker
	variable Prefix

	set base [file rootname $filename]

	exec $Compiler -O2 -I${Prefix}/include -I${Prefix}/tcl8.4/generic -c \
	    ${filename}

	exec $Linker -G -o lib${base}.so ${base}.o -L${Prefix}/lib -ltcl8.4 \
	    -R${Prefix}/lib
	return [file join [pwd] lib${base}.so]
    }

    proc Compile_Win32_MSVC {filename} {
	variable Compiler
	variable Linker
	variable Prefix

	set base [file rootname $filename]

	exec $Compiler -nologo -O2 -I${Prefix}\\include \
	    -I${Prefix}\\tcl8.4\\generic -c ${filename}
	exec $Linker -nologo -dll -out:${base}.dll -libpath:${Prefix}\\lib \
	    ${base}.obj tcl8.4.lib
	return [file join [pwd] ${base}.dll]
    }

    proc GuessPlatform {} {
	variable ::tcl_platform
	variable Platform
	variable Compiler
	variable Linker

	set Platform ""
	set Compiler ""
	set Linker   ""
	
	if {[string equal $tcl_platform(os) SunOS]} {
	    return [GuessPlatform_SunOS]	    
	}
	if {[string equal $tcl_platform(os) Linux]} {
	    return [GuessPlatform_Posix_GCC]
	}
	if {[string equal $tcl_platform(platform) windows]} {
	    return [GuessPlatform_Win32]
	}
    }

    proc GuessPlatform_SunOS {} {
	variable ::env
	variable Platform
	variable Compiler
	variable Linker

	# Look for the SunPro compiler
	foreach path [split $::env(PATH) :] {
	    # Reject anything in /usr/ucb -- the compiler that lives in
	    # there is intentionally crippled
	    if {[string equal -length 8 $path "/usr/ucb"]} { continue }
		
	    if {[file exists [file join $path cc]]} {
		set Compiler [file join $path cc]
	    }

	    if {[file exists [file join $path ld]]} {
		set Linker [file join $path ld]
	    }

	    # If we found both the compiler and linker, we're done.
	    if {[string length $Compiler] != 0 &&
		[string length $Linker] != 0} { 
		set Platform SunOS_SunPro
		return $Platform
	    }
	}

	# Try looking for a GNU compiler instead
	return [GuessPlatform_Posix_GCC]
    }

    proc GuessPlatform_Posix_GCC {} {
	variable ::env
	variable Platform
	variable Compiler
	variable Linker

	# Look for the gcc compiler
	foreach path [split $::env(PATH) :] {
	    if {[file exists [file join $path gcc]]} {
		set Compiler [file join $path gcc]
	    }

	    if {[file exists [file join $path ld]]} {
		set Linker [file join $path ld]
	    }

	    # If we found both the compiler and linker, we're done.
	    if {[string length $Compiler] != 0 &&
		[string length $Linker] != 0} { 
		set Platform Posix_GCC
		return $Platform
	    }
	}

	error "Cannot identify platform."
    }

    proc GuessPlatform_Win32 {} {
	variable ::env
	variable Platform
	variable Compiler
	variable Linker

	# Look for the Microsoft Visual C++ compiler
	foreach path [split $::env(PATH) ";"] {
	    if {[file exists [file join $path cl.exe]]} {
		set Compiler [file join $path cl]
	    }

	    if {[file exists [file join $path link.exe]]} {
		set Linker [file join $path link]
	    }

	    # If we found both the compiler and linker, we're done.
	    if {[string length $Compiler] != 0 &&
		[string length $Linker] != 0} { 
		set Platform Win32_MSVC
		return $Platform
	    }
	}

	# Look for the Borland C++ compiler
	foreach path [split $::env(PATH) ";"] {
	    if {[file exists [file join $path bcc32.exe]]} {
		set Compiler [file join $path bcc32]
	    }

	    if {[file exists [file join $path ilink32.exe]]} {
		set Linker [file join $path ilink32]
	    }

	    # If we found both the compiler and linker, we're done.
	    if {[string length $Compiler] != 0 &&
		[string length $Linker] != 0} { 
		set Platform Win32_Borland
		return $Platform
	    }
	}

	# Look for the Cygwin compiler
	foreach path [split $::env(PATH) ";"] {
	    if {[file exists [file join $path gcc.exe]]} {
		set Compiler [file join $path gcc]
	    }

	    if {[file exists [file join $path ld.exe]]} {
		set Linker [file join $path ld]
	    }

	    # If we found both the compiler and linker, we're done.
	    if {[string length $Compiler] != 0 &&
		[string length $Linker] != 0} { 
		set Platform Win32_GCC
		return $Platform
	    }
	}

	error "Cannot identify platform."
    }
}
