
/*
 *
 * bltDataTable.c --
 *
 *	Copyright 1998-2005 George A Howlett.
 *
 *	Permission is hereby granted, free of charge, to any person
 *	obtaining a copy of this software and associated documentation
 *	files (the "Software"), to deal in the Software without
 *	restriction, including without limitation the rights to use,
 *	copy, modify, merge, publish, distribute, sublicense, and/or
 *	sell copies of the Software, and to permit persons to whom the
 *	Software is furnished to do so, subject to the following
 *	conditions:
 *
 *	The above copyright notice and this permission notice shall be
 *	included in all copies or substantial portions of the
 *	Software.
 *
 *	THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY
 *	KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 *	WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 *	PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS
 *	OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
 *	OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 *	OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 *	SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include <bltInt.h>
#include <bltDataTable.h>
#include <bltNsUtil.h>
/*
 *  array or row pointers
 *   _
 *	index map  [ ][ ][ ][ ][ ][ ][ ][ ][ ][ ]
 *                        |
 *	                  v
 *      header  [   ]<->[   ] label, offset, index, flags
 *      freelist 
 *	tuples 
 *	   
 *  |_---> [row index
 *  |_     [#columns             array of Tcl_Objs
 *  |_     [tuple pointer ---> [ . . . . . ]
 *  |_
 *  |_
 *  |_
 *  |_
 *  |_
 *  |_
 *  |_
 *  |_
 *  |_
 *  |_
 *  |_
 *  |_
 *  |_
 *  |_
 *  |_
 *  |_
 *  |_
 *  |_
 *
 */
#define RowData(c)		(&((c)->tableObjPtr->row))
#define ColumnData(c)		(&((c)->tableObjPtr->column))
#define ClientName(c)		((c)->name)
#define NumRows(c)		((c)->tableObjPtr->row.nUsed)
#define NumColumns(c)		((c)->tableObjPtr->column.nUsed)
#define RowOffset(c,i)		((c)->tableObjPtr->row.map[(i) - 1])
#define ColumnOffset(c,i)	((c)->tableObjPtr->column.map[(i) - 1])
#define RowIndex(c,o)		((c)->tableObjPtr->row.headers[(o)].index)
#define ColumnIndex(c,o)	((c)->tableObjPtr->column.headers[(o)].index)
#define NumColumnsAllocated(c)	((c)->tableObjPtr->column.nAllocated)
#define NumRowsAllocated(c)	((c)->tableObjPtr->row.nAllocated)
#define EmptyValue(c)		((c)->emptyValue)
#define RowMap(c)		((c)->tableObjPtr->row.map)
#define ColumnMap(c)		((c)->tableObjPtr->column.map)
#define RowTags(c)		((c)->rowTags)
#define ColumnTags(c)		((c)->columnTags)
#define TableObj(c)		((c)->tableObjPtr)

#define DATATABLE_THREAD_KEY		"BLT DataTable Data"
#define DATATABLE_MAGIC			((unsigned int) 0xfaceface)
#define DATATABLE_DESTROYED		(1<<0)

#define TABLE_ALLOC_MAX_DOUBLE_SIZE	(1<<16)
#define TABLE_ALLOC_MAX_CHUNK		(1<<16)
#define TABLE_NOTIFY_ANY		(-1)

typedef struct {
    unsigned int flags;
    long nRows, nCols;
    long mtime, ctime;
    char *fileName;
    long nLines;
    int argc;
    char **argv;
    Blt_HashTable rowIndices, colIndices;
} RestoreData;

typedef struct Blt_DataTableStruct DataTable;
typedef struct Blt_DataTableTagsStruct Tags;
typedef struct Blt_DataTableTraceStruct Trace;
typedef struct Blt_DataTableNotifierStruct Notifier;

/*
 * Blt_DataTableTagsStruct --
 *
 *	Structure representing tags used by a client of the table.
 *
 *	Table rows and columns may be tagged with strings.  A row may
 *	have many tags.  The same tag may be used for many rows.  Tags
 *	are used and stored by clients of a table.  Tags can also be
 *	shared between clients of the same table.
 *	
 *	Both rowTable and columnTable are hash tables keyed by the
 *	physical row or column location in the table respectively.
 *	This is not the same as the client's view (the order of rows
 *	or columns as seen by the client).  This is so that clients
 *	(which may have different views) can share tags without
 *	sharing the same view.
 */
struct Blt_DataTableTagsStruct {
    Blt_HashTable rowTable;	/* Table of row indices.  Each entry
				 * is a hash table itself of tag
				 * names. */

    Blt_HashTable columnTable;	/* Table of column indices.  Each
				 * entry is a hash table itself of tag
				 * names. */

    int refCount;		/* Tracks the number of clients
				 * currently using these tags. If
				 * refCount goes to zero, this means
				 * the table can safely be freed. */
};

typedef struct {
    Blt_HashTable clientTable;	/* Tracks all table clients. */
    unsigned int nextId;
    Tcl_Interp *interp;
} DataTableInterpData;

typedef struct {
    unsigned long row, column;
} RowColumnKey;

typedef struct TableHeader {
    char *label;		/* Label of row column header */
    unsigned int flags;		/* Flags indicating type of column. */
    long index;			/* Reverse lookup offset-to-index. */
} TableHeader;

/*
 * RowColumn --
 *
 *	Structure representing a row or column in the table. 
 */
typedef struct {
    CONST char *name;
    long nAllocated;		/* Length of allocated array
				 * below. May exceed the number of
				 * row or column headers used. */
    long nUsed;

    TableHeader *headers;	/* Array of row or column headers. */
    long *map;			/* Current view the above array. */
    
    Blt_Chain *freeList;	/* Tracks free row or column
				 * headers.  */

    Blt_HashTable labels;	/* Hash table of labels. Maps labels
				 * to table offsets. */

    unsigned long nextId;	/* Used to generate default labels. */
} RowColumn;

/*
 * TableObject --
 *
 *	Structure representing a table object. 
 *
 *	The table object is an array of tuples. A tuple itself is an
 *	array of Tcl_Objs representing data for each column (or key).
 *	Value are identified by their column name.  A tuple does not
 *	need to contain values for all columns.  Undefined values
 *	(Tcl_Objs) are NULL.
 *
 *	A table object can be shared by several clients.  When a
 *	client wants to use a table object, it is given a token that
 *	represents the table.  The object tracks its clients by its
 *	token. When all clients have released their tokens, the tuple
 *	object is automatically destroyed.
 */
typedef struct {
    Blt_Chain *clients;		/* List of clients using this table */

    unsigned long mtime, ctime;

    RowColumn row, column;

    Tcl_Obj ***tuples;		/* Array of tuple pointers */

    unsigned int notifyFlags;	/* Notification flags. See definitions
				 * below. */
    int notifyHold;

    
    unsigned int flags;		/* Internal flags. See definitions
				 * below. */

} TableObject;

/*
 * Blt_DataTableStruct --
 *
 *	A client is uniquely identified by a combination of its name
 *	and the originating namespace.  Two table objects in the same
 *	interpreter can have similar names but must reside in
 *	different namespaces.
 *
 *	Two or more clients can share the same table object.  Each
 *	client structure which acts as a ticket for the underlying
 *	table object.  Clients can designate notifier routines that
 *	are automatically invoked by the table object whenever the
 *	table is changed is specific ways by other clients.
 */

struct Blt_DataTableStruct {
    unsigned int magic;		/* Magic value indicating whether a
				 * generic pointer is really a
				 * datatable token or not. */
    Tcl_Interp *interp;
    char *name;			/* Fully namespace-qualified name of
				 * the client. */
    Blt_HashTable *tablePtr;
    Blt_HashEntry *hPtr;

    Blt_ChainLink *linkPtr;	/* Pointer into the server's chain of
				 * clients. */

    TableObject *tableObjPtr;	/* Pointer to the structure containing
				 * the master information about the
				 * table used by the client.  If NULL,
				 * this indicates that the table has
				 * been destroyed (but as of yet, this
				 * client hasn't recognized it). */
    char *emptyValue;

    Blt_HashTable *rowTags;
    Blt_HashTable *columnTags;

    Blt_Chain *traces;		/* Chain of traces. */
    Blt_Chain *notifiers;	/* Chain of event handlers. */
    Tags *tagsPtr;
};

static Tcl_InterpDeleteProc DataTableInterpDeleteProc;
static void DestroyDataTable(DataTable *dtPtr);

static int
ResizeMap(RowColumn *rcPtr, long oldLen, long newLen)
{
    long *map;
    long *ip, *iend;

    map = Blt_Realloc(rcPtr->map, sizeof(long) * newLen);
    if (map == NULL) {
	return FALSE;
    }
    for (ip = map + oldLen, iend = map + newLen; ip < iend; ip++) {
	*ip = -1;		/* Initialize new slots in the map.  */
    }
    rcPtr->map = map;
    return TRUE;
}

static void
UnsetLabel(RowColumn *rcPtr, long offset)
{
    TableHeader *headerPtr;
    Blt_HashEntry *hPtr;

    headerPtr = rcPtr->headers + offset;
    if (headerPtr->label == NULL) {
	return;
    }
    hPtr = Blt_FindHashEntry(&rcPtr->labels, headerPtr->label);
    if (hPtr != NULL) {
	Blt_Chain *chainPtr;
	Blt_ChainLink *linkPtr;
	
	chainPtr = Blt_GetHashValue(hPtr);
	for (linkPtr = Blt_ChainFirstLink(chainPtr); linkPtr != NULL;
	     linkPtr = Blt_ChainNextLink(linkPtr)) {
	    long off;
	    
	    off = (long)Blt_ChainGetValue(linkPtr);
	    if (off == offset) {
		Blt_ChainDeleteLink(chainPtr, linkPtr);
		break;
	    }
	}
	if (Blt_ChainGetLength(chainPtr) == 0) {
	    Blt_ChainDestroy(chainPtr);
	    Blt_DeleteHashEntry(&rcPtr->labels, hPtr);
	}
    }	
    headerPtr->label = NULL;
}

static void
SetHeaderLabel(RowColumn *rcPtr, long offset, CONST char *newLabel)
{
    Blt_Chain *chainPtr;
    Blt_HashEntry *hPtr;
    TableHeader *headerPtr;
    int isNew;
    
    headerPtr = rcPtr->headers + offset;
    if (headerPtr->label != NULL) {
	UnsetLabel(rcPtr, offset);
    }
    if (newLabel == NULL) {
	return;
    }
    hPtr = Blt_CreateHashEntry(&rcPtr->labels, newLabel, &isNew);
    chainPtr = Blt_GetHashValue(hPtr);
    if (chainPtr == NULL) {
	chainPtr = Blt_ChainCreate();
	Blt_SetHashValue(hPtr, chainPtr);
    }
    headerPtr->label = Blt_GetHashKey(&rcPtr->labels, hPtr);
    if (!isNew) {
	Blt_ChainLink *linkPtr;
	
	for (linkPtr = Blt_ChainFirstLink(chainPtr); linkPtr != NULL;
	     linkPtr = Blt_ChainNextLink(linkPtr)) {
	    long off;
	    
	    off = (long)Blt_ChainGetValue(linkPtr);
	    if (off == offset) {
		return;		/* It's already there. */
	    }
	}
    }
    Blt_ChainAppend(chainPtr, (ClientData)offset);
}

#ifdef notdef
static CONST char *
GetLabel(RowColumn *rcPtr, long offset)
{
    return rcPtr->headers[offset].label;
}
#endif

static int
SetLabel(
    Tcl_Interp *interp, 
    RowColumn *rcPtr, 
    long offset, 
    CONST char *label)
{
    if ((offset < 0) || (offset >= rcPtr->nAllocated)) {
	if (interp != NULL) {
	    Tcl_AppendResult(interp, "bad ", rcPtr->name, " \"", 
		Blt_Ltoa(offset), "\"", (char *)NULL);
	}
	return TCL_ERROR;
    }
    if (label[0] == '\0') {
	if (interp != NULL) {
	    Tcl_AppendResult(interp, "bad ", rcPtr->name, " label \"", 
		label, "\": can't be empty.", (char *)NULL);
	}
	return TCL_ERROR;
    }
    if (label[0] == '-') {
	if (interp != NULL) {
	    Tcl_AppendResult(interp, "bad label \"", label, 
			     "\": can't start with a '-'.", (char *)NULL);
	}
	return TCL_ERROR;
    }
    if (isdigit(label[0])) {
	if (interp != NULL) {
	    Tcl_AppendResult(interp, "bad label \"", label, 
			     "\": can't start with a digit.", (char *)NULL);
	}
	return TCL_ERROR;
    }
    if (strchr(label, ':') != NULL) {
	if (interp != NULL) {
	    Tcl_AppendResult(interp, "bad ", rcPtr->name, " label \"", 
		label, "\": can't contain range specifier ':'.", (char *)NULL);
	}
	return TCL_ERROR;
    }
    SetHeaderLabel(rcPtr, offset, label);
    return TCL_OK;
}

static void
GetNextLabel(RowColumn *rcPtr, long offset)
{
    char label[200];

    for(;;) {
	int isNew;
	Blt_HashEntry *hPtr;

	sprintf(label, "%c%ld", rcPtr->name[0], rcPtr->nextId++);
	hPtr = Blt_CreateHashEntry(&rcPtr->labels, label, &isNew);
	if (isNew) {
	    SetHeaderLabel(rcPtr, offset, label);
	    return;
	}
    }
}

static long
GrowHeaders(RowColumn *rcPtr, long n)
{
    TableHeader *headers;
    size_t i;
    size_t reqLen, oldLen, newLen;

    assert(n >= 0);
    if (n == 0) {
	return rcPtr->nAllocated;
    }
    oldLen = rcPtr->nAllocated;
    reqLen = n + rcPtr->nAllocated;

    newLen = rcPtr->nAllocated;
    if (newLen == 0) {
	newLen = 1;
    }
    if (reqLen < TABLE_ALLOC_MAX_DOUBLE_SIZE) {
	while (newLen < reqLen) {
	    newLen += newLen;
	}
    } else {
	while (newLen < reqLen) {
	    newLen += TABLE_ALLOC_MAX_CHUNK;
	}
    }
    if (rcPtr->headers == NULL) {
	headers = Blt_Malloc(newLen * sizeof(TableHeader));
    } else {
	headers = Blt_Realloc(rcPtr->headers, newLen * sizeof(TableHeader));
    }
    if (headers == NULL) {
	return -1;
    }
    rcPtr->headers = headers;
    rcPtr->nAllocated = newLen;
    for (i = oldLen; i < newLen; i++) {
	/* Add the new row to the free list.  */
	Blt_ChainAppend(rcPtr->freeList, (ClientData)i);
	
	/* Initialize the new slots. */
	rcPtr->headers[i].label = NULL;
	rcPtr->headers[i].flags = 0;
	rcPtr->headers[i].index = -1;
    }
    ResizeMap(rcPtr, oldLen, newLen);
    return newLen;
}

static int
GrowRows(TableObject *tableObjPtr, long n)
{
    long oldLen, newLen;
    Tcl_Obj ***tuples, ***tp, ***tend;

    oldLen = tableObjPtr->row.nAllocated;
    if (GrowHeaders(&tableObjPtr->row, n) < 0) {
	return FALSE;
    }
    newLen = tableObjPtr->row.nAllocated;
    tuples = tableObjPtr->tuples;
    if (tuples == NULL) {
	tuples = Blt_Malloc(newLen * sizeof(Tcl_Obj **));
    } else {
	tuples = Blt_Realloc(tuples, newLen * sizeof(Tcl_Obj **));
    }
    if (tuples == NULL) {
	return FALSE;
    }
    tableObjPtr->tuples = tuples;
    for (tp = tuples + oldLen, tend = tuples + newLen; tp < tend; tp++) {
	*tp = NULL;
    }
    return TRUE;
}

static int
GrowColumns(TableObject *tableObjPtr, long n)
{
    long oldLen, newLen;
    Tcl_Obj ***tp, ***tend;

    oldLen = tableObjPtr->column.nAllocated;
    newLen = GrowHeaders(&tableObjPtr->column, n);
    if (newLen < 0) {
	return FALSE;
    }
    /* Resize all the tuples. */
    for (tp = tableObjPtr->tuples, tend = tp + tableObjPtr->row.nAllocated; 
	 tp < tend; tp++) {
	if (*tp != NULL) {
	    Tcl_Obj **tuple, **vp, **vend;
	    
	    tuple = Blt_Realloc(*tp, newLen * sizeof(Tcl_Obj *));
	    for (vp = tuple + oldLen, vend = tuple + newLen; vp < vend; vp++) {
		*vp = NULL;
	    }
	    *tp = tuple;
	}
    }
    return TRUE;
}

static void
ExtendHeaders(RowColumn *rcPtr, size_t n, long *indices)
{
    Blt_ChainLink *linkPtr;
    long nextIndex;
    size_t i;

    /* 
     * At this point, we're guarenteed to have as many free
     * rows/columns in the table as requested.
     */
    linkPtr = Blt_ChainFirstLink(rcPtr->freeList);
    nextIndex = rcPtr->nUsed; 
    for (i = 0; i < n; i++) {
	Blt_ChainLink *nextPtr;
	long offset;

	offset = (long)Blt_ChainGetValue(linkPtr);

	/* Initialize new row or column. */
	rcPtr->headers[offset].label = NULL;
	rcPtr->headers[offset].flags = 0;
	/* GetNextLabel(rcPtr, offset); */
	if (indices != NULL) {
	    indices[i] = offset;
	}
	rcPtr->map[nextIndex] = offset;
	nextIndex++;
	rcPtr->headers[offset].index = nextIndex;

	/* Remove the entry from the free list. */
	nextPtr = Blt_ChainNextLink(linkPtr);
	Blt_ChainDeleteLink(rcPtr->freeList, linkPtr);
	linkPtr = nextPtr;
    }
    rcPtr->nUsed += n;
}

static int
ExtendRows(Blt_DataTable table, size_t n, long *indices)
{
    RowColumn *rcPtr;
    size_t nFree;

    rcPtr = RowData(table);
    nFree = Blt_ChainGetLength(rcPtr->freeList);
    if (n > nFree) {
	if (!GrowRows(TableObj(table), n - nFree)) {
	    return FALSE;
	}
    }
    ExtendHeaders(rcPtr, n, indices);
    return TRUE;
}

static int
ExtendColumns(Blt_DataTable table, size_t n, long *indices)
{
    RowColumn *rcPtr;
    size_t nFree;

    rcPtr = ColumnData(table);
    nFree = Blt_ChainGetLength(rcPtr->freeList);
    if (n > nFree) {
	if (!GrowColumns(TableObj(table), n - nFree)) {
	    return FALSE;
	}
    }
    ExtendHeaders(rcPtr, n, indices);
    return TRUE;
}

/*
 * --------------------------------------------------------------
 *
 * NewTableObject --
 *
 *	Creates and initializes a new table object. 
 *
 * Results:
 *	Returns a pointer to the new object is successful, NULL
 *	otherwise.  If a table object can't be generated,
 *	interp->result will contain an error message.
 *
 * -------------------------------------------------------------- 
 */
static TableObject *
NewTableObject(Tcl_Interp *interp)
{
    TableObject *tableObjPtr;

    tableObjPtr = Blt_Calloc(1, sizeof(TableObject));
    if (tableObjPtr == NULL) {
	return NULL;
    }
    tableObjPtr->clients = Blt_ChainCreate();

    tableObjPtr->row.name = "row";
    tableObjPtr->row.nextId = 1;
    tableObjPtr->row.freeList = Blt_ChainCreate();
    Blt_InitHashTable(&tableObjPtr->row.labels, BLT_STRING_KEYS);

    tableObjPtr->column.name = "column";
    tableObjPtr->column.nextId = 1;
    tableObjPtr->column.freeList = Blt_ChainCreate();
    Blt_InitHashTable(&tableObjPtr->column.labels, BLT_STRING_KEYS);
    return tableObjPtr;
}

/*
 *----------------------------------------------------------------------
 *
 * FreeTrace --
 *
 *	Deletes a trace.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	Memory is deallocated for the trace.
 *
 *----------------------------------------------------------------------
 */
static void
FreeTrace(Trace *tp)
{
    if (tp->deleteProc != NULL) {
	(*tp->deleteProc)(tp->clientData);
    }
    if (tp->rowTag != NULL) {
	Blt_Free(tp->rowTag);
    }
    if (tp->colTag != NULL) {
	Blt_Free(tp->colTag);
    }
    Blt_DeleteHashTable(&tp->activeTable);
    if (tp->linkPtr != NULL) {
	Blt_ChainDeleteLink(tp->chainPtr, tp->linkPtr);
    }
    Blt_Free(tp);
}

static void
FreeTraces(Blt_Chain *chainPtr)
{
    Blt_ChainLink *linkPtr;

    for (linkPtr = Blt_ChainFirstLink(chainPtr); linkPtr != NULL; 
	 linkPtr = Blt_ChainNextLink(linkPtr)) {
	Trace *tp;
	
	tp = Blt_ChainGetValue(linkPtr);
	tp->linkPtr = NULL;
	FreeTrace(tp);
    }
    Blt_ChainDestroy(chainPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * NotifyIdleProc --
 *
 *	Used to invoke event handler routines at some idle point.
 *	This routine is called from the Tcl event loop.  Errors
 *	generated by the event handler routines are backgrounded.
 *	
 * Results:
 *	None.
 *
 *---------------------------------------------------------------------- 
 */
static void
NotifyIdleProc(ClientData clientData)
{
    Notifier *np = clientData;
    int result;

    np->flags &= ~TABLE_NOTIFY_PENDING;

    np->flags |= TABLE_NOTIFY_ACTIVE;
    result = (*np->proc)(np->clientData, &np->event);
    np->flags &= ~TABLE_NOTIFY_ACTIVE;

    if (result == TCL_ERROR) {
	Tcl_BackgroundError(np->interp);
    }
}

static void
FreeNotifier(Notifier *np) 
{
    np->flags |= TABLE_NOTIFY_DESTROYED;
    if (np->deleteProc != NULL) {
	(*np->deleteProc)(np->clientData);
    }
    if (np->flags & TABLE_NOTIFY_PENDING) {
	Tcl_CancelIdleCall(NotifyIdleProc, np);
    }
    if (np->tag != NULL) {
	Blt_Free(np->tag);
    }
    if (np->linkPtr != NULL){
	Blt_ChainDeleteLink(np->chainPtr, np->linkPtr);
    }
    Blt_Free(np);
}

static void
FreeNotifiers(Blt_Chain *chainPtr)
{
    Blt_ChainLink *linkPtr;

    for (linkPtr = Blt_ChainFirstLink(chainPtr); linkPtr != NULL; 
	 linkPtr = Blt_ChainNextLink(linkPtr)) {
	Notifier *np;

	np = Blt_ChainGetValue(linkPtr);
	np->linkPtr = NULL;
	FreeNotifier(np);
    }
    Blt_ChainDestroy(chainPtr);
}


/*
 *----------------------------------------------------------------------
 *
 * DumpTags --
 *
 *	Retrieves all tags for a given row or column into a tcl list.  
 *
 * Results:
 *	Returns the number of tags in the list.
 *
 *---------------------------------------------------------------------- 
 */
static void
DumpTags(Blt_HashTable *tagTablePtr, long offset, Blt_Chain *chainPtr)
{
    Blt_HashEntry *hPtr;
    Blt_HashSearch cursor;

    for (hPtr = Blt_FirstHashEntry(tagTablePtr, &cursor); hPtr != NULL; 
	 hPtr = Blt_NextHashEntry(&cursor)) {
	Blt_HashEntry *h2Ptr;
	Blt_HashTable *tablePtr;

	tablePtr = Blt_GetHashValue(hPtr);
	h2Ptr = Blt_FindHashEntry(tablePtr, (char *)offset);
	if (h2Ptr != NULL) {
	    Blt_ChainAppend(chainPtr, Blt_GetHashKey(tagTablePtr, hPtr));
	}
    }
}

/*
 *----------------------------------------------------------------------
 *
 * ClearRowTags --
 *
 *	Removes all tags for a given row.  
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *      All tags associcated with the row are freed.
 *
 *---------------------------------------------------------------------- 
 */
static void
ClearRowTags(Blt_DataTable table, long row)
{
    Blt_HashEntry *hPtr;
    Blt_HashSearch cursor;

    for (hPtr = Blt_FirstHashEntry(RowTags(table), &cursor); hPtr != NULL; 
	 hPtr = Blt_NextHashEntry(&cursor)) {
	Blt_HashEntry *h2Ptr;
	Blt_HashTable *tagTablePtr;

	tagTablePtr = Blt_GetHashValue(hPtr);
	h2Ptr = Blt_FindHashEntry(tagTablePtr, (char *)row);
	if (h2Ptr != NULL) {
	    Blt_DeleteHashEntry(tagTablePtr, h2Ptr);
	}
    }
}

/*
 *----------------------------------------------------------------------
 *
 * ClearColumnTags --
 *
 *	Removes all tags for a given column.  
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *      All tags associcated with the column are freed.
 *
 *---------------------------------------------------------------------- 
 */
static void
ClearColumnTags(Blt_DataTable table, long column)
{
    Blt_HashEntry *hPtr;
    Blt_HashSearch cursor;

    for (hPtr = Blt_FirstHashEntry(ColumnTags(table), &cursor); hPtr != NULL; 
	hPtr = Blt_NextHashEntry(&cursor)) {
	Blt_HashEntry *h2Ptr;
	Blt_HashTable *tagTablePtr;

	tagTablePtr = Blt_GetHashValue(hPtr);
	h2Ptr = Blt_FindHashEntry(tagTablePtr, (char *)column);
	if (h2Ptr != NULL) {
	    Blt_DeleteHashEntry(tagTablePtr, h2Ptr);
	}
    }
}

static void
FreeTuple(Tcl_Obj **tuple, long length)
{
    if (tuple != NULL) {
	long i;

	for (i = 0; i < length; i++) {
	    if (tuple[i] != NULL) {
		Tcl_DecrRefCount(tuple[i]);
	    }
	}
	Blt_Free(tuple);
    }
}


/*
 * ----------------------------------------------------------------------
 *
 * DestroyTableObject --
 *
 *	Destroys the table object.  This is the final clean up of the
 *	object.  The object's entry is removed from the hash table of
 *	tables.
 *
 * Results: 
 *	None.
 *
 * ---------------------------------------------------------------------- 
 */
static void
DestroyTableObject(TableObject *tableObjPtr)
{
    Blt_ChainLink *linkPtr;

    tableObjPtr->flags |= DATATABLE_DESTROYED;

    /* 
     * Remove any clients that remain. Make sure we dereference the
     * client's tableObjPtr so we don't call this routine recursively.
     */
    for (linkPtr = Blt_ChainFirstLink(tableObjPtr->clients); linkPtr != NULL;
	 linkPtr = Blt_ChainNextLink(linkPtr)) {
	DataTable *dtPtr;

	dtPtr = Blt_ChainGetValue(linkPtr);
	dtPtr->tableObjPtr = NULL;
	dtPtr->linkPtr = NULL;
	DestroyDataTable(dtPtr);
    }
    Blt_ChainDestroy(tableObjPtr->clients);

    /* Free the headerss containing row and column info. */
    /* Free the data in each row. */
    if (tableObjPtr->tuples != NULL) {
	Tcl_Obj ***tp, ***tend;

	for (tp = tableObjPtr->tuples, tend = tp + tableObjPtr->row.nAllocated;
	     tp < tend; tp++) {
	    if (*tp != NULL) {
		FreeTuple(*tp, tableObjPtr->column.nAllocated);
	    }
	}
	Blt_Free(tableObjPtr->tuples);
    }
    if (tableObjPtr->row.headers != NULL) {
	Blt_Free(tableObjPtr->row.headers);
    }
    Blt_DeleteHashTable(&tableObjPtr->row.labels);
    Blt_ChainDestroy(tableObjPtr->row.freeList);
    Blt_Free(tableObjPtr->row.map);

    if (tableObjPtr->column.headers != NULL) {
	Blt_Free(tableObjPtr->column.headers);
    }
    Blt_DeleteHashTable(&tableObjPtr->column.labels);
    Blt_ChainDestroy(tableObjPtr->column.freeList);
    Blt_Free(tableObjPtr->column.map);

    Blt_Free(tableObjPtr);
}

/*
 * -----------------------------------------------------------------------
 *
 * DataTableInterpDeleteProc --
 *
 *	This is called when the interpreter hosting the table object
 *	is deleted from the interpreter.  
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Destroys all remaining tables and removes the hash table
 *	used to register table names.
 *
 * ------------------------------------------------------------------------
 */
/* ARGSUSED */
static void
DataTableInterpDeleteProc(ClientData clientData, Tcl_Interp *interp)
{
    DataTableInterpData *dataPtr;
    Blt_HashEntry *hPtr;
    Blt_HashSearch cursor;
    
    dataPtr = clientData;
    for (hPtr = Blt_FirstHashEntry(&dataPtr->clientTable, &cursor);
	 hPtr != NULL; hPtr = Blt_NextHashEntry(&cursor)) {
	DataTable *dtPtr;

	dtPtr = Blt_GetHashValue(hPtr);
	dtPtr->hPtr = NULL;
	dtPtr->tableObjPtr = NULL;
	DestroyDataTable(dtPtr);
    }
    Blt_DeleteHashTable(&dataPtr->clientTable);
    Tcl_DeleteAssocData(interp, DATATABLE_THREAD_KEY);
    Blt_Free(dataPtr);
}

/*
 * --------------------------------------------------------------
 *
 * GetDataTableInterpData --
 *
 *	Creates or retrieves data associated with tuple data objects
 *	for a particular thread.  We're using Tcl_GetAssocData rather
 *	than the Tcl thread routines so BLT can work with pre-8.0 
 *	Tcl versions that don't have thread support.
 *
 * Results:
 *	Returns a pointer to the tuple interpreter data.
 *
 * -------------------------------------------------------------- 
 */
static DataTableInterpData *
GetDataTableInterpData(Tcl_Interp *interp)
{
    Tcl_InterpDeleteProc *proc;
    DataTableInterpData *dataPtr;

    dataPtr = (DataTableInterpData *)
	Tcl_GetAssocData(interp, DATATABLE_THREAD_KEY, &proc);
    if (dataPtr == NULL) {
	dataPtr = Blt_Malloc(sizeof(DataTableInterpData));
	assert(dataPtr);
	dataPtr->interp = interp;
	Tcl_SetAssocData(interp, DATATABLE_THREAD_KEY, 
		DataTableInterpDeleteProc, dataPtr);
	Blt_InitHashTable(&dataPtr->clientTable, BLT_STRING_KEYS);
    }
    return dataPtr;
}


CONST char *
Blt_DataTableNameOfColumnType(int type)
{
    switch (type & TABLE_COLUMN_TYPE_MASK) {
    case TABLE_COLUMN_TYPE_STRING:
	return "string";
    case TABLE_COLUMN_TYPE_INTEGER:
	return "integer";
    case TABLE_COLUMN_TYPE_IMAGE:
	return "image";
    case TABLE_COLUMN_TYPE_DOUBLE:
	return "double";
    default:
	return "???";
    }
}

int
Blt_DataTableParseColumnType(CONST char *typeName)
{
    if (strcmp(typeName, "string") == 0) {
	return TABLE_COLUMN_TYPE_STRING;
    } else if (strcmp(typeName, "integer") == 0) {
	return TABLE_COLUMN_TYPE_INTEGER;
    } else if (strcmp(typeName, "image") == 0) {
	return TABLE_COLUMN_TYPE_IMAGE;
    } else if (strcmp(typeName, "double") == 0) {
	return TABLE_COLUMN_TYPE_DOUBLE;
    } else {
	return TABLE_COLUMN_TYPE_UNKNOWN;
    }
}


/*
 *----------------------------------------------------------------------
 *
 * NewTags --
 *
 *---------------------------------------------------------------------- 
 */
static Tags *
NewTags(void)
{
    Tags *tagsPtr;

    tagsPtr = Blt_Malloc(sizeof(Tags));
    if (tagsPtr != NULL) {
	Blt_InitHashTable(&tagsPtr->rowTable, BLT_STRING_KEYS);
	Blt_InitHashTable(&tagsPtr->columnTable, BLT_STRING_KEYS);
	tagsPtr->refCount = 1;
    }
    return tagsPtr;
}

/*
 * ----------------------------------------------------------------------
 *
 * FindClientInNamespace --
 *
 *	Searches for a table client in a given namespace.
 *
 * Results:
 *	Returns a pointer to the table client if found, otherwise NULL.
 *
 * ----------------------------------------------------------------------
 */
static DataTable *
FindClientInNamespace(DataTableInterpData *dataPtr, Blt_ObjectName *objNamePtr)
{
    Tcl_DString ds;
    char *qualName;
    Blt_HashEntry *hPtr;

    qualName = Blt_MakeQualifiedName(objNamePtr, &ds);
    hPtr = Blt_FindHashEntry(&dataPtr->clientTable, qualName);
    Tcl_DStringFree(&ds);
    if (hPtr == NULL) {
	return NULL;
    }
    return Blt_GetHashValue(hPtr);
}

/*
 * ----------------------------------------------------------------------
 *
 * GetDataTable --
 *
 *	Searches for the table client associated by the name given.
 *
 * Results:
 *	Returns a pointer to the table client if found, otherwise NULL.
 *
 * ----------------------------------------------------------------------
 */
static Blt_DataTable
GetDataTable(DataTableInterpData *dataPtr, CONST char *name, unsigned int flags)
{
    Blt_ObjectName objName;
    Blt_DataTable table;
    Tcl_Interp *interp;

    table = NULL;
    interp = dataPtr->interp;
    if (!Blt_ParseObjectName(interp, name, &objName, BLT_NO_DEFAULT_NS)) {
	return NULL;
    }
    if (objName.nsPtr != NULL) { 
	table = FindClientInNamespace(dataPtr, &objName);
    } else { 
	if (flags & NS_SEARCH_CURRENT) {
	    /* Look first in the current namespace. */
	    objName.nsPtr = Tcl_GetCurrentNamespace(interp);
	    table = FindClientInNamespace(dataPtr, &objName);
	}
	if ((table == NULL) && (flags & NS_SEARCH_GLOBAL)) {
	    objName.nsPtr = Tcl_GetGlobalNamespace(interp);
	    table = FindClientInNamespace(dataPtr, &objName);
	}
    }
    return table;
}

static void
DestroyDataTable(DataTable *dtPtr)
{
    if (dtPtr->magic != DATATABLE_MAGIC) {
	fprintf(stderr, "invalid table object token 0x%lx\n", 
		(unsigned long)dtPtr);
	return;
    }
    /* Remove any traces that were set by this client. */
    FreeTraces(dtPtr->traces);
    /* Also remove all event handlers created by this client. */
    FreeNotifiers(dtPtr->notifiers);

    if (dtPtr->tagsPtr != NULL) {
	Blt_DataTableReleaseTags(dtPtr);
    }
    if (dtPtr->hPtr != NULL) {
	Blt_DeleteHashEntry(dtPtr->tablePtr, dtPtr->hPtr);
    }
    if ((dtPtr->tableObjPtr != NULL) && (dtPtr->linkPtr != NULL)) {
	TableObject *tableObjPtr;

	tableObjPtr = dtPtr->tableObjPtr;
	/* Remove the client from the server's list */
	Blt_ChainDeleteLink(tableObjPtr->clients, dtPtr->linkPtr);
	if (Blt_ChainGetLength(tableObjPtr->clients) == 0) {
	    DestroyTableObject(tableObjPtr);
	}
    }
    dtPtr->magic = 0;
    Blt_Free(dtPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * NewDataTable --
 *
 *	Creates a new table client.  Clients shared a tuple data
 *	object.  They individually manage traces and events on tuple
 *	objects.  Returns a pointer to the malloc'ed structure.  This
 *	is passed to the client as a tuple token.
 *	
 * Results:
 *	A pointer to a DataTable is returned.  If one can't
 *	be allocated, NULL is returned.  
 *
 *---------------------------------------------------------------------- 
 */
static DataTable *
NewDataTable(
    DataTableInterpData *dataPtr, 
    TableObject *tableObjPtr,	/* Table object serving this client. */
    CONST char *qualName)	/* Full namespace qualified name of
				 * table. */
{
    DataTable *dtPtr;
    int isNew;

    dtPtr = Blt_Calloc(1, sizeof(DataTable));
    if (dtPtr == NULL) {
	return NULL;
    }
    dtPtr->magic = DATATABLE_MAGIC;
    dtPtr->interp = dataPtr->interp;
    /* Add client to table object's list of clients. */
    dtPtr->linkPtr = Blt_ChainAppend(tableObjPtr->clients, dtPtr);

    /* By default, use own sets of tags. */
    dtPtr->tagsPtr = NewTags();
    dtPtr->rowTags = &dtPtr->tagsPtr->rowTable;
    dtPtr->columnTags = &dtPtr->tagsPtr->columnTable;

    dtPtr->tablePtr = &dataPtr->clientTable;

    /* Add new entry to global client hash table.  */
    dtPtr->hPtr = Blt_CreateHashEntry(&dataPtr->clientTable, qualName, &isNew);
    dtPtr->name = Blt_GetHashKey(&dataPtr->clientTable, dtPtr->hPtr);
    Blt_SetHashValue(dtPtr->hPtr, dtPtr);

    dtPtr->notifiers = Blt_ChainCreate();
    dtPtr->traces = Blt_ChainCreate();

    /* Create client row and column maps corresponding to the current
     * table object. */
    dtPtr->tableObjPtr = tableObjPtr;
    return dtPtr;
}

static long
FindLabel(RowColumn *rcPtr, CONST char *label)
{
    Blt_HashEntry *hPtr;
    Blt_Chain *chainPtr;
    Blt_ChainLink *linkPtr;

    hPtr = Blt_FindHashEntry(&rcPtr->labels, label);
    if (hPtr == NULL) {
	return -1;
    }
    chainPtr = Blt_GetHashValue(hPtr);
    assert(chainPtr);
    linkPtr = Blt_ChainFirstLink(chainPtr);
    return (long)Blt_ChainGetValue(linkPtr);
}

static void
SetType(RowColumn *rcPtr, long column, int type)
{
    rcPtr->headers[column].flags &= ~TABLE_COLUMN_TYPE_MASK;
    rcPtr->headers[column].flags |= (type & TABLE_COLUMN_TYPE_MASK);
}


static void
ResetInverseMap(RowColumn *rcPtr, long *map)
{
    long i, j;

    /* Reset the inverse offset-to-index map. */
    for (i = 0; i < rcPtr->nAllocated; i++) {
	rcPtr->headers[i].index = -1;
    }
    for (i = 0, j = 1; i < rcPtr->nUsed; i++, j++) {
	rcPtr->headers[map[i]].index = j;
    }
}

static Tcl_Obj **
AllocateRow(TableObject *tableObjPtr, long row)
{
    if (tableObjPtr->tuples[row] == NULL) {
	Tcl_Obj **tuple;

	tuple = Blt_Calloc(tableObjPtr->column.nAllocated, sizeof(Tcl_Obj *));
	tableObjPtr->tuples[row] = tuple;
    }
    return tableObjPtr->tuples[row];
}

static void
DeleteHeader(RowColumn *rcPtr, long offset)
{
    long *p, *q, *qend;

    /* If there is a label is associated with the column, free it. */
    if (rcPtr->headers[offset].label != NULL) {
	UnsetLabel(rcPtr, offset);
    }
    /* Compress the index-to-offset map. */
    for (q = rcPtr->map + rcPtr->headers[offset].index, p = q - 1,
	     qend = rcPtr->map + rcPtr->nUsed; q < qend; p++, q++) {
	*p = *q;
    }
    *q = -1;
    rcPtr->nUsed--;
    ResetInverseMap(rcPtr, rcPtr->map);
}


/*
 *----------------------------------------------------------------------
 *
 * ClearRowNotifiers --
 *
 *	Removes all event handlers set for this row.  Note that this
 *	doesn't remove handlers triggered by row tags.  Row traces are
 *	stored in a chain.
 *
 *---------------------------------------------------------------------- 
 */
static void
ClearRowNotifiers(DataTable *dtPtr, long row)
{
    Blt_ChainLink *linkPtr, *nextPtr;

    for (linkPtr = Blt_ChainFirstLink(dtPtr->notifiers); linkPtr != NULL;
	 linkPtr = nextPtr) {
	Notifier *np;

	nextPtr = Blt_ChainNextLink(linkPtr);
	np = Blt_ChainGetValue(linkPtr);
	if ((np->flags & TABLE_NOTIFY_ROW) && (np->offset == row)) {
	    FreeNotifier(np);
	}
    }
}

/*
 *----------------------------------------------------------------------
 *
 * ClearColumnNotifiers --
 *
 *	Removes all event handlers set for this column.  Note that
 *	this doesn't remove handlers triggered by column tags.  Column
 *	traces are stored in a chain.
 *
 *---------------------------------------------------------------------- 
 */
static void
ClearColumnNotifiers(DataTable *dtPtr, long column)
{
    Blt_ChainLink *linkPtr, *nextPtr;

    for (linkPtr = Blt_ChainFirstLink(dtPtr->notifiers); linkPtr != NULL;
	 linkPtr = nextPtr) {
	Notifier *np;

	nextPtr = Blt_ChainNextLink(linkPtr);
	np = Blt_ChainGetValue(linkPtr);
	if (((np->flags & TABLE_NOTIFY_ROW) == 0) && (np->offset == column)) {
	    FreeNotifier(np);
	}
    }
}

/*
 *----------------------------------------------------------------------
 *
 * DoNotify --
 *
 *	Traverses the list of event callbacks for a client and checks
 *	if one matches the given event.  A client may trigger an
 *	action that causes the itself to be notified again.  This can
 *	be prevented by setting the TABLE_NOTIFY_FOREIGN_ONLY bit in
 *	the event handler.
 *
 *	If a matching handler is found, a callback may be called
 *	either immediately or at the next idle time depending upon the
 *	TABLE_NOTIFY_WHENIDLE bit.
 *
 *	Since a handler routine may trigger yet another call to
 *	itself, callbacks are ignored while the event handler is
 *	executing.
 *	
 *---------------------------------------------------------------------- 
 */
static void
DoNotify(DataTable *dtPtr, Blt_DataTableNotifierEvent *eventPtr)
{
    Blt_ChainLink *linkPtr;
    unsigned int eventMask;

    /* Check the client table for matching notifiers.  Issue callbacks
     * indicating that the structure of the table has changed.  */
    eventMask = eventPtr->type & TABLE_NOTIFY_MASK;
    for (linkPtr = Blt_ChainFirstLink(dtPtr->notifiers); linkPtr != NULL; 
	 linkPtr = Blt_ChainNextLink(linkPtr)) {
	Notifier *np;
	int match;

	np = Blt_ChainGetValue(linkPtr);
	if ((np->flags & eventMask) == 0) {
	    continue;		/* Event type doesn't match */
	}
	if ((eventPtr->self) && (np->flags & TABLE_NOTIFY_FOREIGN_ONLY)) {
	    continue;		/* Don't notify yourself. */
	}
	if (np->flags & TABLE_NOTIFY_ACTIVE) {
	    continue;		/* Ignore callbacks that are generated
				 * inside of a notify handler
				 * routine. */
	}
	match = FALSE;
	if (np->tag != NULL) {
	    if (np->flags & TABLE_NOTIFY_ROW) {
		if (eventPtr->offset < 0) {
		    Blt_HashTable *tablePtr;
		    
		    tablePtr = Blt_DataTableRowTagTable(dtPtr, np->tag);
		    if ((tablePtr != NULL) && (tablePtr->numEntries > 0)) {
			match++;
		    }
		} else if (Blt_DataTableHasRowTag(dtPtr, eventPtr->offset, 
						  np->tag)) {
		    match++;
		}
	    } else {
		if (eventPtr->offset < 0) {
		    Blt_HashTable *tablePtr;

		    tablePtr = Blt_DataTableColumnTagTable(dtPtr, np->tag);
		    if ((tablePtr != NULL) && (tablePtr->numEntries > 0)) {
			match++;
		    }
		} else if (Blt_DataTableHasColumnTag(dtPtr, eventPtr->offset, 
			np->tag)) {
		    match++;
		}
	    }
	} else if (np->offset == eventPtr->offset) {
	    match++;		/* Offsets match. */
	} else if (eventPtr->offset < 0) {
	    match++;		/* Event matches any notifier offset. */
	} else if (np->offset < 0) {
	    match++;		/* Notifier matches any event offset. */
	}
	if (!match) {
	    continue;		/* Row or column doesn't match. */
	}
	if (np->flags & TABLE_NOTIFY_WHENIDLE) {
	    if ((np->flags & TABLE_NOTIFY_PENDING) == 0) {
		np->flags |= TABLE_NOTIFY_PENDING;
		np->event = *eventPtr;
		Tcl_DoWhenIdle(NotifyIdleProc, np);
	    }
	} else {
	    int result;
	    
	    np->flags |= TABLE_NOTIFY_ACTIVE;
	    result = (*np->proc)(np->clientData, eventPtr);
	    np->flags &= ~TABLE_NOTIFY_ACTIVE;
	    if (result != TCL_OK) {
		Tcl_BackgroundError(np->interp);
	    }
	}
    }
}
	     
/*
 *----------------------------------------------------------------------
 *
 * NotifyClients --
 *
 *	Traverses the list of event callbacks and checks if one
 *	matches the given event.  A client may trigger an action that
 *	causes the table object to notify it.  This can be prevented
 *	by setting the TABLE_NOTIFY_FOREIGN_ONLY bit in the event
 *	handler.
 *
 *	If a matching handler is found, a callback may be called either
 *	immediately or at the next idle time depending upon the
 *	TABLE_NOTIFY_WHENIDLE bit.  
 *
 *	Since a handler routine may trigger yet another call to
 *	itself, callbacks are ignored while the event handler is
 *	executing.
 *	
 *---------------------------------------------------------------------- 
 */
static void
NotifyClients(DataTable *dtPtr, long offset, unsigned int flags)
{
    Blt_ChainLink *linkPtr, *nextPtr;
    
    for (linkPtr = Blt_ChainFirstLink(dtPtr->tableObjPtr->clients); 
	 linkPtr != NULL; linkPtr = nextPtr) {
	Blt_DataTable table;
	Blt_DataTableNotifierEvent event;
	
	nextPtr = Blt_ChainNextLink(linkPtr);
	table = Blt_ChainGetValue(linkPtr);
	event.type = flags;
	event.table = dtPtr;
	event.offset = offset;
	event.self = (table == dtPtr);
	DoNotify(table, &event);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * ClearRowTraces --
 *
 *	Removes all traces set for this row.  Note that this doesn't
 *	remove traces set for specific cells (row,column).  Row traces
 *	are stored in a chain, which in turn is held in a hash table,
 *	keyed by the row.
 *
 *---------------------------------------------------------------------- 
 */
static void
ClearRowTraces(DataTable *dtPtr, long row)
{
    Blt_ChainLink *linkPtr, *nextPtr;

    for (linkPtr = Blt_ChainFirstLink(dtPtr->traces); linkPtr != NULL;
	 linkPtr = nextPtr) {
	Trace *tp;

	nextPtr = Blt_ChainNextLink(linkPtr);
	tp = Blt_ChainGetValue(linkPtr);
	if (tp->row == row) {
	    FreeTrace(tp);
	}
    }
}

/*
 *----------------------------------------------------------------------
 *
 * ClearColumnTraces --
 *
 *	Removes all traces set for this column.  Note that this doesn't
 *	remove traces set for specific cells (row,column).  Column traces
 *	are stored in a chain, which in turn is held in a hash table,
 *	keyed by the column.
 *
 *---------------------------------------------------------------------- 
 */
static void
ClearColumnTraces(DataTable *dtPtr, long column)
{
    Blt_ChainLink *linkPtr, *nextPtr;

    for (linkPtr = Blt_ChainFirstLink(dtPtr->traces); linkPtr != NULL;
	 linkPtr = nextPtr) {
	Trace *tp;

	nextPtr = Blt_ChainNextLink(linkPtr);
	tp = Blt_ChainGetValue(linkPtr);
	if (tp->column == column) {
	    FreeTrace(tp);
	}
    }
}

/*
 *----------------------------------------------------------------------
 *
 * DoTrace --
 *
 *	Fires a trace set by a client of the table object.  Trace
 *	procedures should return a standard Tcl result.
 *
 *	   TCL_OK	procedure executed sucessfully.
 *	   TCL_ERROR	procedure failed.
 *	   TCL_BREAK	don't execute any further trace procedures.
 *	   TCL_CONTINUE	treat like TCL_OK.
 *
 *	A trace procedure can in turn trigger more traces.  Traces are
 *	prohibited from recusively reentering their own trace
 *	procedures.  A hash table in the trace structure tracks the
 *	cells currently actively traced.  If a cell is already being
 *	traced, the trace procedure is not called and TCL_OK is
 *	blindly returned.
 *
 * Results:
 *	Returns the result of trace procedure.  If the trace is
 *	already active, then TCL_OK is returned.
 *
 * Side Effects:
 *	Traces on the table location may be fired.
 *
 *----------------------------------------------------------------------
 */
static int
DoTrace(
    DataTable *dtPtr,
    Trace *tp,
    long row, long column,	/* Client row, column location. */
    unsigned int flags)
{
    RowColumnKey rcKey;
    Blt_HashEntry *hPtr;
    int result;
    int isNew;

    /* 
     * Check for individual traces on a cell.  Each trace has a hash
     * table that tracks what cells are actively being traced. This is
     * to prevent traces from triggering recursive callbacks.
     */
    rcKey.row = row;
    rcKey.column = column;
    hPtr = Blt_CreateHashEntry(&tp->activeTable, &rcKey, &isNew);
    if (!isNew) {
	return TCL_OK;		/* Trace is already active in cell. */
    }
    result = (*tp->proc)(tp->clientData, dtPtr->interp, 
	RowIndex(dtPtr, row), ColumnIndex(dtPtr, column), flags);
    if (result == TCL_ERROR) {
	Tcl_BackgroundError(dtPtr->interp);
    }
    Blt_DeleteHashEntry(&tp->activeTable, hPtr);
    return result;
}

/*
 *----------------------------------------------------------------------
 *
 * CallTraces --
 *
 *	Examines the traces set for each client of the table object 
 *	and fires any matching traces.  
 *
 *	Traces match on row and column tag and indices and flags.
 *	Traces can match on
 *	     flag		type of trace (read, write, unset, create)
 *	     row index
 *	     column index
 *	     row tag
 *	     column tag
 *
 *	If the TABLE_TRACE_FOREIGN_ONLY is set in the handler, it
 *	means to ignore actions that are initiated by that client of
 *	the object.  Only actions by other clients are handled.
 *
 * Results:
 *	Always returns TCL_OK.
 *
 * Side Effects:
 *	Traces on the tuple table location may be fired.
 *
 *----------------------------------------------------------------------
 */
static void
CallTraces(
    DataTable *dtPtr,
    long row, long column, 
    unsigned int flags)
{
    Blt_ChainLink *linkPtr, *nextPtr;

    for (linkPtr = Blt_ChainFirstLink(dtPtr->traces); linkPtr != NULL;
	 linkPtr = nextPtr) {
	Trace *tp;
	int rowMatch, colMatch;

	nextPtr = Blt_ChainNextLink(linkPtr);
	tp = Blt_ChainGetValue(linkPtr);
	if ((tp->flags & flags) == 0) {
	    continue;		/* Doesn't match trace flags. */
	}
	rowMatch = colMatch = FALSE;
	if (tp->colTag != NULL) {
	    if (Blt_DataTableHasColumnTag(dtPtr, column, tp->colTag)) {
		colMatch++;
	    }
	} else if ((tp->column == column) || (tp->column < 0)) {
	    colMatch++;
	}
	if (tp->rowTag != NULL) {
	    if (Blt_DataTableHasRowTag(dtPtr, row, tp->rowTag)) {
		rowMatch++;
	    }
	} else if ((tp->row == row) || (tp->row < 0)) {
	    rowMatch++;
	}
	if (!rowMatch || !colMatch) {
	    continue;
	}
	if (DoTrace(dtPtr, tp, row, column, flags) == TCL_BREAK) {
	    return;		/* Don't complete traces on break. */
	}
    }
}

/*
 *----------------------------------------------------------------------
 *
 * CallClientTraces --
 *
 *	Examines the traces set for each client of the table object 
 *	and fires any matching traces.  
 *
 *	Traces match on row and column indices and flags.
 *	The order is 
 *	  1. column traces.
 *	  2. row traces.
 *	  3. cell (row,column) traces.
 *
 *	If no matching criteria is specified (no tag, key, or tuple
 *	address) then only the bit flag has to match.
 *
 *	If the TABLE_TRACE_FOREIGN_ONLY is set in the handler, it
 *	means to ignore actions that are initiated by that client of
 *	the object.  Only actions by other clients are handled.
 *
 * Results:
 *	Always returns TCL_OK.
 *
 * Side Effects:
 *	Traces on the tuple table location may be fired.
 *
 *----------------------------------------------------------------------
 */
static void
CallClientTraces(
    TableObject *tableObjPtr,
    long row, long column, 
    unsigned int flags)
{
    Blt_ChainLink *linkPtr, *nextPtr;

    for (linkPtr = Blt_ChainFirstLink(tableObjPtr->clients); linkPtr != NULL;
	 linkPtr = nextPtr) {
	DataTable *dtPtr;

	nextPtr = Blt_ChainNextLink(linkPtr);
	dtPtr = Blt_ChainGetValue(linkPtr);
	CallTraces(dtPtr, row, column, flags);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * GetValue --
 *
 *	Retrieves a value from the selected row, column location in
 *	the table.  The row, column location must be within the actual
 *	table limits, but there does not have to be a value there.
 *
 * Results:
 *	Returns the objPtr representing the value.  If no value
 *	is present, the NULL is returned.
 *
 *----------------------------------------------------------------------
 */
static Tcl_Obj *
GetValue(TableObject *tableObjPtr, long row, long column)
{
    Tcl_Obj *objPtr, **tuple;

    tuple = tableObjPtr->tuples[row];
    if (tuple == NULL) {
	objPtr = NULL;
    } else {
	objPtr = tuple[column];
    }
    return objPtr;
}

/*
 *----------------------------------------------------------------------
 *
 * SetValue --
 *
 *	Sets the value of the selected row, column location in
 *	the table.  The row, column location must be within the actual
 *	table limits.  
 *
 * Results:
 *	Returns the objPtr representing the old value.  If no previous
 *	value was present, the NULL is returned.
 *
 * Side Effects:
 *	New tuples may be allocated created.
 *
 *----------------------------------------------------------------------
 */
static Tcl_Obj *
SetValue(TableObject *tableObjPtr, long row, long column, Tcl_Obj *objPtr)
{
    Tcl_Obj *oldObjPtr, **tuple;

    assert(row < tableObjPtr->row.nAllocated);
    tuple = tableObjPtr->tuples[row];
    if (tuple == NULL) {
	tuple = AllocateRow(tableObjPtr, row);
    }
    assert(column < tableObjPtr->column.nAllocated);
    oldObjPtr = tuple[column];
    if (objPtr != NULL) {
	Tcl_IncrRefCount(objPtr);
    } 
    if (oldObjPtr != NULL) {
	Tcl_DecrRefCount(oldObjPtr);
    }
    tuple[column] = objPtr;
    return oldObjPtr;
}

/*
 *----------------------------------------------------------------------
 *
 * UnsetValue --
 *
 *	Removes the value from the selected row, column location in
 *	the table.  The row, column location must be within the actual
 *	table limits, but it's okay if there isn't a value there to
 *	remove.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	The objPtr representing the value is released.
 *
 *----------------------------------------------------------------------
 */
static void
UnsetValue(TableObject *tableObjPtr, long row, long column)
{
    Tcl_Obj *objPtr, **tuple;

    tuple = tableObjPtr->tuples[row];
    if (tuple == NULL) {
	return;
    }
    objPtr = tuple[column];
    Tcl_DecrRefCount(objPtr);
    tuple[column] = NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * GetArray --
 *
 *	Retrieves the array object from the selected row, column
 *	location in the table.  The row, column location must be
 *	within the actual table limits, but there does not have to be
 *	a value there.  This routine is similar to GetValue but
 *	performs a check if the array object is shared and duplicates
 *	it if necessary.
 *
 * Results:
 *	Returns the objPtr representing the array object.  If no value
 *	is present, the NULL is returned.
 *
 *----------------------------------------------------------------------
 */
static Tcl_Obj *
GetArray(TableObject *tableObjPtr, long row, long column)
{
    Tcl_Obj *arrayObjPtr;

    arrayObjPtr = GetValue(tableObjPtr, row, column);
    if (arrayObjPtr == NULL) {
	return NULL;
    }
    if (Tcl_IsShared(arrayObjPtr)) {
	Tcl_DecrRefCount(arrayObjPtr);
	arrayObjPtr = Tcl_DuplicateObj(arrayObjPtr);
	Tcl_IncrRefCount(arrayObjPtr);
	SetValue(tableObjPtr, row, column, arrayObjPtr);
    }
    return arrayObjPtr;
}

/*
 *----------------------------------------------------------------------
 *
 * SetArray --
 *
 *	Retrieves the array object of the selected row, column
 *	location in the table.  The row, column location must be
 *	within the actual table limits.  If no array object exists at
 *	that location, one is created.
 *
 * Results:
 *	Returns the objPtr representing the array object.  
 *
 * Side Effects:
 *	New tuples may be allocated created.
 *
 *----------------------------------------------------------------------
 */
static Tcl_Obj *
SetArray(TableObject *tableObjPtr, long row, long column)
{
    Tcl_Obj *arrayObjPtr, **tuple;

    tuple = tableObjPtr->tuples[row];
    if (tuple == NULL) {
	tuple = AllocateRow(tableObjPtr, row);
    }
    arrayObjPtr = tuple[column];
    if (arrayObjPtr == NULL) {
	arrayObjPtr = Blt_NewArrayObj(0, (Tcl_Obj **)NULL);
	Tcl_IncrRefCount(arrayObjPtr);
    } else if (Tcl_IsShared(arrayObjPtr)) {
	Tcl_DecrRefCount(arrayObjPtr);
	arrayObjPtr = Tcl_DuplicateObj(arrayObjPtr);
	Tcl_IncrRefCount(arrayObjPtr);
    }
    tuple[column] = arrayObjPtr;
    return arrayObjPtr;
}

static void
UnsetRowValues(Blt_DataTable table, long row)
{
    TableObject *tableObjPtr;
    Tcl_Obj **tuple;
    long i;

    tableObjPtr = TableObj(table);
    for (i = 1; i <= NumColumns(table); i++) {
	long column;

	column = ColumnOffset(table, i);
	if (GetValue(tableObjPtr, row, column) != NULL) {
	    CallClientTraces(tableObjPtr, row, column, TABLE_TRACE_UNSETS);
	    UnsetValue(tableObjPtr, row, column);
	}
    }
    tuple = tableObjPtr->tuples[row];
    if (tuple != NULL) {
	Blt_Free(tuple);
	tableObjPtr->tuples[row] = NULL;
    }
}

static void
UnsetColumnValues(Blt_DataTable table, long column)
{
    TableObject *tableObjPtr;
    long i;

    tableObjPtr = TableObj(table);
    for (i = 1; i <= NumRows(table); i++) {
	long row;

	row = RowOffset(table, i);
	if (GetValue(tableObjPtr, row, column) != NULL) {
	    CallClientTraces(tableObjPtr, row, column, TABLE_TRACE_UNSETS);
	    UnsetValue(tableObjPtr, row, column);
	}
    }
}

static int
CompareDictionaryStrings(
    ClientData clientData, 
    Tcl_Obj *valueObjPtr1, 
    Tcl_Obj *valueObjPtr2)
{
    Blt_DataTable table = clientData;
    char *s1, *s2;

    s1 = s2 = EmptyValue(table);
    if (valueObjPtr1 != NULL) {
	s1 = Tcl_GetString(valueObjPtr1);
    }
    if (valueObjPtr2 != NULL) {
	s2 = Tcl_GetString(valueObjPtr2);
    }
    return Blt_DictionaryCompare(s1, s2);
}

static int
CompareAsciiStrings(
    ClientData clientData, 
    Tcl_Obj *valueObjPtr1, 
    Tcl_Obj *valueObjPtr2)
{
    Blt_DataTable table = clientData;
    char *s1, *s2;

    s1 = s2 = EmptyValue(table);
    if (valueObjPtr1 != NULL) {
	s1 = Tcl_GetString(valueObjPtr1);
    }
    if (valueObjPtr2 != NULL) {
	s2 = Tcl_GetString(valueObjPtr2);
    }
    return strcmp(s1, s2);
}

static int
CompareIntegers(
    ClientData clientData, 
    Tcl_Obj *valueObjPtr1, 
    Tcl_Obj *valueObjPtr2)
{
    long i1, i2;

    if ((valueObjPtr1 != NULL) && (valueObjPtr2 != NULL)) {
	if ((Tcl_GetLongFromObj(NULL, valueObjPtr1, &i1) == TCL_OK) &&
	    (Tcl_GetLongFromObj(NULL, valueObjPtr2, &i2) == TCL_OK)) {
	    return i1 - i2;
	} 
    }
    return CompareDictionaryStrings(clientData, valueObjPtr1, valueObjPtr2);
}

static int
CompareDoubles(
    ClientData clientData, 
    Tcl_Obj *valueObjPtr1, 
    Tcl_Obj *valueObjPtr2)
{
    double d1, d2;

    if ((valueObjPtr1 != NULL) && (valueObjPtr2 != NULL)) {
	if ((Tcl_GetDoubleFromObj(NULL, valueObjPtr1, &d1) == TCL_OK) &&
	    (Tcl_GetDoubleFromObj(NULL, valueObjPtr2, &d2) == TCL_OK)) {
	    return (d1 < d2) ? -1 : (d1 > d2) ? 1 : 0;
	} 
    }
    return CompareDictionaryStrings(clientData, valueObjPtr1, valueObjPtr2);
}

typedef struct {
    Blt_DataTable table;
    Blt_DataTableSortOrder *order;
    long nCompares;
    unsigned int flags;
} DataTableSortData;

static DataTableSortData sortData;

static int
CompareRows(void *a, void *b)
{
    TableObject *tableObjPtr;
    DataTable *dtPtr;
    Blt_DataTableSortOrder *sp, *send;
    long r1, r2;
    long result;

    dtPtr = sortData.table;
    tableObjPtr = dtPtr->tableObjPtr;
 
    r1 = *(long *)a;
    r2 = *(long *)b;
    for (sp = sortData.order, send = sp + sortData.nCompares; sp < send; sp++) {
	TableHeader *columnPtr;
	Tcl_Obj *valueObjPtr1, *valueObjPtr2, **tuple;
	long column;

	column = sp->offset;
	columnPtr = tableObjPtr->column.headers + sp->offset;
	valueObjPtr1 = valueObjPtr2 = NULL;
	tuple = tableObjPtr->tuples[r1];
	if (tuple != NULL) {
	    valueObjPtr1 = tuple[sp->offset];
	}
	tuple = tableObjPtr->tuples[r2];
	if (tuple != NULL) {
	    valueObjPtr2 = tuple[sp->offset];
	}
	result = (*sp->proc)(sp->clientData, valueObjPtr1, valueObjPtr2);
	if (result != 0) {
	    return (sortData.flags & TABLE_SORT_DECREASING) ? -result : result;
	}
    }
    result = RowIndex(dtPtr, r1) - RowIndex(dtPtr, r2);
    return (sortData.flags & TABLE_SORT_DECREASING) ? -result : result;
}

static void
InitSortProcs(DataTable *dtPtr, Blt_DataTableSortOrder *order, long n)
{
    Blt_DataTableSortOrder *sp, *send;
    TableObject *tableObjPtr;

    tableObjPtr = dtPtr->tableObjPtr;
    for (sp = order, send = sp + n; sp < send; sp++) {
	TableHeader *columnPtr;

	if ((sp->proc != NULL) || (sp->type == TABLE_SORT_CUSTOM)) {
	    continue;
	}
	sp->clientData = dtPtr;
	switch (sp->type) {
	case TABLE_SORT_INTEGER:
	    sp->proc = CompareIntegers;
	    break;
	case TABLE_SORT_DOUBLE:
	    sp->proc = CompareDoubles;
	    break;
	case TABLE_SORT_ASCII:
	    sp->proc = CompareAsciiStrings;
	    break;
	case TABLE_SORT_DICTIONARY:
	    sp->proc = CompareDictionaryStrings;
	    break;
	case TABLE_SORT_NONE: 
	    columnPtr = tableObjPtr->column.headers + sp->offset;
	    switch (columnPtr->flags & (TABLE_COLUMN_TYPE_MASK)) {
	    case TABLE_COLUMN_TYPE_INTEGER:
		sp->proc = CompareIntegers;
		break;
	    case TABLE_COLUMN_TYPE_DOUBLE:
		sp->proc = CompareDoubles;
		break;
	    case TABLE_COLUMN_TYPE_STRING:
	    case TABLE_COLUMN_TYPE_UNKNOWN:
	    default:
		sp->proc = CompareDictionaryStrings;
		break;
	    }
	case TABLE_SORT_CUSTOM:
	    break;
	}
    }
}

static long *
SortHeaders(RowColumn *rcPtr, QSortCompareProc *proc)
{
    long i;
    long *map;

    /* Make a copy of the current row map. */
    map = Blt_Malloc(sizeof(long) * rcPtr->nAllocated);
    if (map == NULL) {
	return NULL;
    }
    for (i = 0; i < rcPtr->nAllocated; i++) {
	map[i] = rcPtr->map[i];
    }
    /* Sort the map and return it. */
    qsort((char *)map, rcPtr->nUsed, sizeof(long), proc);
    return map;
}


static void
ReplaceMap(RowColumn *rcPtr, long *map)
{
    ResetInverseMap(rcPtr, map);
    Blt_Free(rcPtr->map);
    rcPtr->map = map;
}

static int
MoveIndices(
    RowColumn *rcPtr,
    long src,			/* Starting source index.  */
    long dest,			/* Starting destination index. */
    long count)			/* # of rows or columns to move. */
{
    long i, j;
    long *newMap;		/* Resulting reordered map. */

#ifdef notdef
    fprintf(stderr, "src=%ld, dest=%ld, count=%ld\n", src, dest, count);
    fprintf(stderr, "%s nUsed=%ld, nAllocated=%ld\n", rcPtr->name,
	    rcPtr->nUsed, rcPtr->nAllocated);
#endif
    if (src == dest) {
	return TRUE;
    }
    src--; dest--;
    newMap = Blt_Malloc(sizeof(long) * rcPtr->nAllocated);
    if (newMap == NULL) {
	return FALSE;
    }
    if (dest < src) {
	/*
	 *     dest   src
	 *      v     v
	 * | | | | | |x|x|x|x| |
	 *  A A B B B C C C C D
	 * | | |x|x|x|x| | | | |
	 *
	 * Section C is the selected region to move.
	 */
	/* Section A: copy everything from 0 to "dest" */
	for (i = 0; i < dest; i++) {
	    newMap[i] = rcPtr->map[i];
	}
	/* Section C: append the selected region. */
	for (i = src, j = dest; i < (src + count); i++, j++) {
	    newMap[j] = rcPtr->map[i];
	}
	/* Section B: shift the preceding indices from "dest" to "src".  */
	for (i = dest; i < src; i++, j++) {
	    newMap[j] = rcPtr->map[i];
	}
	/* Section D: append trailing indices until the end. */
	for (i = src + count; i < rcPtr->nUsed; i++, j++) {
	    newMap[j] = rcPtr->map[i];
	}
    } else if (src < dest) {
	/*
	 *     src        dest
	 *      v           v
	 * | | |x|x|x|x| | | | |
	 *  A A C C C C B B B D
	 * | | | | | |x|x|x|x| |
	 *
	 * Section C is the selected region to move.
	 */
	/* Section A: copy everything from 0 to "src" */
	for (j = 0; j < src; j++) {
	    newMap[j] = rcPtr->map[j];
	}
	/* Section B: shift the trailing indices from "src" to "dest".  */
	for (i = (src + count); j < dest; i++, j++) {
	    newMap[j] = rcPtr->map[i];
	}
	/* Section C: append the selected region. */
	for (i = src; i < (src + count); i++, j++) {
	    newMap[j] = rcPtr->map[i];
	}
	/* Section D: append trailing indices until the end. */
	for (i = dest + count; i < rcPtr->nUsed; i++, j++) {
	    newMap[j] = rcPtr->map[i];
	}
    }
    /* Reset the inverse offset-to-index map. */
    ReplaceMap(rcPtr, newMap);
    return TRUE;
}


/*
 *----------------------------------------------------------------------
 *
 * ParseDumpRecord --
 *
 *	Gets the next full record in the dump string, returning the
 *	record as a list. Blank lines and comments are ignored.
 *
 * Results: 
 *	TCL_RETURN	The end of the string is reached.
 *	TCL_ERROR	An error occurred and an error message 
 *			is left in the interpreter result.  
 *	TCL_OK		The next record has been successfully parsed.
 *
 *----------------------------------------------------------------------
 */
static int
ParseDumpRecord(
    Tcl_Interp *interp,
    char **stringPtr,		/* (in/out) points to current location
				 * in in dump string. Updated after
				 * parsing record. */
    RestoreData *rp)
{
    char *entry, *eol;
    char saved;
    int result;

    entry = *stringPtr;
    /* Get first line, ignoring blank lines and comments. */
    for (;;) {
	char *first;

	first = NULL;
	rp->nLines++;
	/* Find the end of the first line. */
	for (eol = entry; (*eol != '\n') && (*eol != '\0'); eol++) {
	    if ((first == NULL) && (!isspace(UCHAR(*eol)))) {
		first = eol;	/* Track first non-whitespace
				 * character. */
	    }
	}
	if (first == NULL) {
	    if (*eol == '\0') {
		return TCL_RETURN;
	    }
	} else if (*first != '#') {
	    break;		/* Not a comment or blank line. */
	}
	entry = eol + 1;
    }
    saved = *eol;
    *eol = '\0';
    while (!Tcl_CommandComplete(entry)) {
	*eol = saved;
	if (*eol == '\0') {
	    Tcl_AppendResult(interp, "incomplete dump record: \"", entry, 
		"\"", (char *)NULL);
	    return TCL_ERROR;		/* Found EOF (incomplete
					 * entry) or error. */
	}
	/* Get the next line. */
	for (eol = eol + 1; (*eol != '\n') && (*eol != '\0'); eol++) {
	    /*empty*/
	}
	rp->nLines++;
	saved = *eol;
	*eol = '\0';
    }
    if (entry == eol) {
	return TCL_RETURN;
    }
    result = Tcl_SplitList(interp, entry, &rp->argc, &rp->argv);
    *eol = saved;
    *stringPtr = eol + 1;
    return result;
}

/*
 *----------------------------------------------------------------------
 *
 * ReadDumpRecord --
 *
 *	Reads the next full record from the given channel, returning the
 *	record as a list. Blank lines and comments are ignored.
 *
 * Results: 
 *	TCL_RETURN	The end of the file has been reached.
 *	TCL_ERROR	A read error has occurred and an error message 
 *			is left in the interpreter result.  
 *	TCL_OK		The next record has been successfully parsed.
 *
 *----------------------------------------------------------------------
 */
static int
ReadDumpRecord(Tcl_Interp *interp, Tcl_Channel channel, RestoreData *rp)
{
    int result;
    Tcl_DString ds;

    Tcl_DStringInit(&ds);
    /* Get first line, ignoring blank lines and comments. */
    for (;;) {
	char *cp;
	int nBytes;

	Tcl_DStringSetLength(&ds, 0);
	nBytes = Tcl_Gets(channel, &ds);
	if (nBytes < 0) {
	    if (Tcl_Eof(channel)) {
		return TCL_RETURN;
	    }
	    return TCL_ERROR;
	}
	rp->nLines++;
	for (cp = Tcl_DStringValue(&ds); *cp != '\0'; cp++) {
	    if (!isspace(UCHAR(*cp))) {
		break;
	    }
	}
	if ((*cp != '\0') && (*cp != '#')) {
	    break;		/* Not a comment or blank line. */
	}
    }

    Tcl_DStringAppend(&ds, "\n", 1);
    while (!Tcl_CommandComplete(Tcl_DStringValue(&ds))) {
	int nBytes;

	/* Process additional lines if needed */
	nBytes = Tcl_Gets(channel, &ds);
	if (nBytes < 0) {
	    Tcl_AppendResult(interp, "error reading file: ", 
			     Tcl_PosixError(interp), (char *)NULL);
	    Tcl_DStringFree(&ds);
	    return TCL_ERROR;		/* Found EOF (incomplete
					 * entry) or error. */
	}
	rp->nLines++;
	Tcl_DStringAppend(&ds, "\n", 1);
    }
    result = Tcl_SplitList(interp, Tcl_DStringValue(&ds), &rp->argc, &rp->argv);
    Tcl_DStringFree(&ds);
    return result;
}

static void
RestoreError(Tcl_Interp *interp, RestoreData *rp)
{
    Tcl_DString ds;

    Tcl_DStringInit(&ds);
    Tcl_DStringGetResult(interp, &ds);
    Tcl_AppendResult(interp, rp->fileName, ":", 
	Blt_Ltoa(rp->nLines), ": error: ", Tcl_DStringValue(&ds), 
	(char *)NULL);
    Tcl_DStringFree(&ds);
}

static int
RestoreHeader(Tcl_Interp *interp, Blt_DataTable table, RestoreData *rp)
{
    long nCols, nRows;
    unsigned long mtime, ctime;

    /* i rows columns ctime mtime */
    if (rp->argc != 5) {
	RestoreError(interp, rp);
	Tcl_AppendResult(interp, "wrong # elements in restore header.", 
		(char *)NULL);
	return TCL_ERROR;
    }	
    if (Tcl_GetLong(interp, rp->argv[1], &nRows) != TCL_OK) {
	RestoreError(interp, rp);
	return TCL_ERROR;
    }
    if (nRows < 1) {
	RestoreError(interp, rp);
	Tcl_AppendResult(interp, "bad # rows \"", rp->argv[1], "\"", 
			 (char *)NULL);
	return TCL_ERROR;
    }
    if (Tcl_GetLong(interp, rp->argv[2], &nCols) != TCL_OK) {
	RestoreError(interp, rp);
	return TCL_ERROR;
    }
    if (nCols < 1) {
	RestoreError(interp, rp);
	Tcl_AppendResult(interp, "bad # columns \"", rp->argv[2], "\"", 
			 (char *)NULL);
	return TCL_ERROR;
    }
    if ((rp->flags & TABLE_RESTORE_OVERWRITE) == 0) {
	nRows += rp->nRows;
	nCols += rp->nCols;
    }
    if (nCols > NumColumns(table)) {
	long n;

	n = nCols - NumColumns(table);
	if (!GrowColumns(TableObj(table), n)) {
	    RestoreError(interp, rp);
	    Tcl_AppendResult(interp, "can't allocate \"", Blt_Ltoa(n),
			"\"", " extra columns.", (char *)NULL);
	    return TCL_ERROR;
	}
    }
    if (nRows > NumRows(table)) {
	long n;

	n = nRows - NumRows(table);
	if (!GrowRows(TableObj(table), n)) {
	    RestoreError(interp, rp);
	    Tcl_AppendResult(interp, "can't allocate \"", Blt_Ltoa(n),
			     "\"", " extra rows.", (char *)NULL);
	    return TCL_ERROR;
	}
    }
    if (Tcl_GetLong(interp, rp->argv[3], &ctime) != TCL_OK) {
	RestoreError(interp, rp);
	return TCL_ERROR;
    }
    rp->ctime = ctime;
    if (Tcl_GetLong(interp, rp->argv[4], &mtime) != TCL_OK) {
	RestoreError(interp, rp);
	return TCL_ERROR;
    }
    rp->mtime = mtime;
    return TCL_OK;
}

static int
RestoreColumn(Tcl_Interp *interp, Blt_DataTable table, RestoreData *rp)
{
    RowColumn *rcPtr;
    long n, column;
    int type;
    char *label;
    int isNew;
    Blt_HashEntry *hPtr;

    /* c index label type ?tagList? */
    if ((rp->argc < 4) || (rp->argc > 5)) {
	RestoreError(interp, rp);
	Tcl_AppendResult(interp, "wrong # elements in restore column entry", 
		(char *)NULL);
	return TCL_ERROR;
    }	
    if (Tcl_GetLong(interp, rp->argv[1], &n) != TCL_OK) {
	RestoreError(interp, rp);
	return TCL_ERROR;
    }
    if (n < 1) {
	RestoreError(interp, rp);
	Tcl_AppendResult(interp, "bad column index \"", rp->argv[1], 
		"\"", (char *)NULL);
	return TCL_ERROR;
    }
    rcPtr = ColumnData(table);
    label = rp->argv[2];
    column = FindLabel(rcPtr, label);
    if ((column < 0) || ((rp->flags & TABLE_RESTORE_OVERWRITE) == 0)) {
	long newCol;

	if (!ExtendColumns(table, 1, &newCol)) {
	    RestoreError(interp, rp);
	    Tcl_AppendResult(interp, "can't append column", (char *)NULL);
	    return TCL_ERROR;
	}
	if (column < 0) {
	    if (SetLabel(interp, rcPtr, newCol, label) != TCL_OK) {
		RestoreError(interp, rp);
		return TCL_ERROR;
	    }
	}
	column = newCol;
    }
    hPtr = Blt_CreateHashEntry(&rp->colIndices, (char *)n, &isNew);
    Blt_SetHashValue(hPtr, column);

    type = Blt_DataTableParseColumnType(rp->argv[3]);
    if (type == TABLE_COLUMN_TYPE_UNKNOWN) {
	RestoreError(interp, rp);
	Tcl_AppendResult(interp, "bad column type \"", rp->argv[3], 
			 "\"", (char *)NULL);
	return TCL_ERROR;
    }
    SetType(rcPtr, column, type);
    if ((rp->argc == 5) && ((rp->flags & TABLE_RESTORE_NO_TAGS) == 0)) {
	int elc;
	char **elv;
	long i;

	if (Tcl_SplitList(interp, rp->argv[4], &elc, &elv) != TCL_OK) {
	    RestoreError(interp, rp);
	    return TCL_ERROR;
	}
	
	for (i = 0; i < elc; i++) {
	    if (Blt_DataTableSetColumnTag(interp, table, column, elv[i])
		!= TCL_OK) {
		Blt_Free(elv);
		return TCL_ERROR;
	    }
	}
	Blt_Free(elv);
    }
    return TCL_OK;
}

static int
RestoreRow(Tcl_Interp *interp, Blt_DataTable table, RestoreData *rp)
{
    int elc;
    char **elv;
    RowColumn *rcPtr;
    long n, row;
    char *label;
    int isNew;
    Blt_HashEntry *hPtr;

    /* r index label ?tagList? */
    if ((rp->argc < 3) || (rp->argc > 4)) {
	RestoreError(interp, rp);
	Tcl_AppendResult(interp, "wrong # elements in restore row entry", 
			 (char *)NULL);
	return TCL_ERROR;
    }	
    if (Tcl_GetLong(interp, rp->argv[1], &n) != TCL_OK) {
	RestoreError(interp, rp);
	return TCL_ERROR;
    }
    if (n < 1) {
	RestoreError(interp, rp);
	Tcl_AppendResult(interp, "bad row index \"", rp->argv[1], "\"",
		(char *)NULL);
	return TCL_ERROR;
    }
    rcPtr = RowData(table);
    label = rp->argv[2];
    row = FindLabel(rcPtr, label);
    if ((row < 0) || ((rp->flags & TABLE_RESTORE_OVERWRITE) == 0)) {
	long newRow;

	if (!ExtendRows(table, 1, &newRow)) {
	    RestoreError(interp, rp);
	    Tcl_AppendResult(interp, "can't append row", (char *)NULL);
	    return TCL_ERROR;
	}
	if (row < 0) {
	    if (SetLabel(interp, rcPtr, newRow, label) != TCL_OK) {
		RestoreError(interp, rp);
		return TCL_ERROR;
	    }
	}
	row = newRow;
    }
    hPtr = Blt_CreateHashEntry(&rp->rowIndices, (char *)n, &isNew);
    Blt_SetHashValue(hPtr, row);
    if ((rp->argc == 5) && ((rp->flags & TABLE_RESTORE_NO_TAGS) == 0)) {
	long i;

	if (Tcl_SplitList(interp, rp->argv[3], &elc, &elv) != TCL_OK) {
	    RestoreError(interp, rp);
	    return TCL_ERROR;
	}
	for (i = 0; i < elc; i++) {
	    if (Blt_DataTableSetRowTag(interp, table, row, elv[i]) != TCL_OK) {
		Blt_Free(elv);
		return TCL_ERROR;
	    }
	}
	Blt_Free(elv);
    }
    return TCL_OK;
}

static int
RestoreValue(Tcl_Interp *interp, Blt_DataTable table, RestoreData *rp)
{
    Blt_HashEntry *hPtr;
    Tcl_Obj *valueObjPtr;
    int result;
    long row, column;
    long iRow, iColumn;

    /* d row column value */
    if (rp->argc != 4) {
	RestoreError(interp, rp);
	Tcl_AppendResult(interp, "wrong # elements in restore data entry", 
		(char *)NULL);
	return TCL_ERROR;
    }	
    if (Tcl_GetLong(interp, rp->argv[1], &iRow) != TCL_OK) {
	RestoreError(interp, rp);
	return TCL_ERROR;
    }
    hPtr = Blt_FindHashEntry(&rp->rowIndices, (char *)iRow);
    if (hPtr == NULL) {
	RestoreError(interp, rp);
	Tcl_AppendResult(interp, "bad row index \"", rp->argv[1], "\"",
			 (char *)NULL);
	return TCL_ERROR;
    }
    row = (long)Blt_GetHashValue(hPtr);
    if (Tcl_GetLong(interp, rp->argv[2], &iColumn) != TCL_OK) {
	RestoreError(interp, rp);
	return TCL_ERROR;
    }
    hPtr = Blt_FindHashEntry(&rp->colIndices, (char *)iColumn);
    if (hPtr == NULL) {
	RestoreError(interp, rp);
	Tcl_AppendResult(interp, "bad column index \"", rp->argv[2], 
		"\"", (char *)NULL);
	return TCL_ERROR;
    }
    column = (long)Blt_GetHashValue(hPtr);
    valueObjPtr = Tcl_NewStringObj(rp->argv[3], -1);
    Tcl_IncrRefCount(valueObjPtr);
    result = Blt_DataTableSetValue(interp, table, row, column, valueObjPtr);
    Tcl_DecrRefCount(valueObjPtr);
    if (result != TCL_OK) {
	RestoreError(interp, rp);
    }
    return result;
}

/* Public Routines */
CONST char *
Blt_DataTableName(Blt_DataTable table)
{
    return ClientName(table);
}

CONST char *
Blt_DataTableEmptyValue(Blt_DataTable table)
{
    return EmptyValue(table);
}

long 
Blt_DataTableNumRows(Blt_DataTable table)  
{
    return NumRows(table);
}

long 
Blt_DataTableNumColumns(Blt_DataTable table)  
{
    return NumColumns(table);
}

long 
Blt_DataTableRowOffset(Blt_DataTable table, long index)  
{
    if ((index < 1) || (index > NumRows(table))) {
	return -1;
    }
    return RowOffset(table, index);
}

long 
Blt_DataTableColumnOffset(Blt_DataTable table, long index)  
{
    if ((index < 1) || (index > NumColumns(table))) {
	return -1;
    }
    return ColumnOffset(table, index);
}

long 
Blt_DataTableRowIndex(Blt_DataTable table, long offset)  
{
    if ((offset < 0) || (offset >= NumRowsAllocated(table))) {
	return -1;
    }
    return RowIndex(table, offset);
}

long 
Blt_DataTableColumnIndex(Blt_DataTable table, long offset)  
{
    if ((offset < 0) || (offset >= NumColumnsAllocated(table))) {
	return -1;
    }
    return ColumnIndex(table, offset);
}

long *
Blt_DataTableRowMap(Blt_DataTable table)  
{
    return RowMap(table);
}

long *
Blt_DataTableColumnMap(Blt_DataTable table)  
{
    return ColumnMap(table);
}

Blt_HashEntry *
Blt_DataTableFirstRowTag(Blt_DataTable table, Blt_HashSearch *cursorPtr)  
{
    return Blt_FirstHashEntry(RowTags(table), cursorPtr);
}

Blt_HashEntry *
Blt_DataTableFirstColumnTag(Blt_DataTable table, Blt_HashSearch *cursorPtr)  
{
    return Blt_FirstHashEntry(ColumnTags(table), cursorPtr);
}

int 
Blt_DataTableSameTableObject(Blt_DataTable table1, Blt_DataTable table2)  
{
    return TableObj(table1) == TableObj(table2);
}

Blt_Chain *
Blt_DataTableRowTags(Blt_DataTable table, long row)  
{
    Blt_Chain *chainPtr;

    chainPtr = Blt_ChainCreate();
    DumpTags(RowTags(table), row, chainPtr);
    return chainPtr;
}

Blt_Chain *
Blt_DataTableColumnTags(Blt_DataTable table, long column)  
{
    Blt_Chain *chainPtr;

    chainPtr = Blt_ChainCreate();
    DumpTags(ColumnTags(table), column, chainPtr);
    return chainPtr;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_DataTableGetRows --
 *
 *	Returns the id of the first row derived from the given tag,
 *	label or index represented in objPtr.  
 *
 * Results:
 *	Returns the row location of the first item.  If no row 
 *	can be found, then -1 is returned and an error message is
 *	left in the interpreter.
 *
 *---------------------------------------------------------------------- 
 */
int
Blt_DataTableGetRows(
    Tcl_Interp *interp,
    Blt_DataTable table,
    Tcl_Obj *objPtr,
    Blt_DataTableIterator *ip)
{
    long n;
    CONST char *p;
    CONST char *colon, *pend;
    int nBytes;

    ip->table = table;
    ip->type = TABLE_ITER_INDEX;
    ip->tagName = Tcl_GetStringFromObj(objPtr, &nBytes);
    ip->iNext = -1;

    colon = NULL;
    for (p = ip->tagName, pend = p + nBytes; p < pend; p++) {
	if (*p != ':') {
	    continue;
	}
	if (colon != NULL) {
	    if (interp != NULL) {
		Tcl_AppendResult(interp, "bad range \"", Tcl_GetString(objPtr),
			"\": too many range specifiers.", (char *)NULL);
	    }
	    return TCL_ERROR;
	}
	colon = p;
    }
    if (colon != NULL) {
	Tcl_Obj *objPtr1, *objPtr2;
	int result;
	long from, to;

	if ((colon == ip->tagName) || (colon == (pend - 1))) {
	    if (interp != NULL) {
		Tcl_AppendResult(interp, "bad range \"", 
			Tcl_GetString(objPtr), "\": must specify \"from-to\"", 
				 (char *)NULL);
	    }
	    return TCL_ERROR;
	}	    
	objPtr1 = Tcl_NewStringObj(ip->tagName, colon - ip->tagName);
	colon++;
	objPtr2 = Tcl_NewStringObj(colon, pend - colon);
	result = Blt_DataTableGetRow(interp, table, objPtr1, &from);
	if (result == TCL_OK) {
	    result = Blt_DataTableGetRow(interp, table, objPtr2, &to);
	}
	Tcl_DecrRefCount(objPtr1);
	Tcl_DecrRefCount(objPtr2);
	if (result != TCL_OK) {
	    return TCL_ERROR;
	}
	ip->iStart = RowIndex(table, from);
	ip->iEnd = RowIndex(table, to);
	ip->type = TABLE_ITER_RANGE;
	return TCL_OK;
    }
    if (Tcl_GetLongFromObj((Tcl_Interp *)NULL, objPtr, &n) == TCL_OK) {
	if ((n < 1) || (n > NumRows(table))) {
	    if (interp != NULL) {
		Tcl_AppendResult(interp, "bad row index \"", 
			Tcl_GetString(objPtr), "\"", (char *)NULL);
	    }
	    return TCL_ERROR;
	}		
	ip->iStart = ip->iEnd = n;
	return TCL_OK;
    } else if (strcmp(ip->tagName, "all") == 0) {
	ip->type = TABLE_ITER_ALL;
	ip->iStart = 1;
	ip->iEnd = NumRows(table);
	return TCL_OK;
    } else if (strcmp(ip->tagName, "end") == 0) {
	ip->iStart = ip->iEnd = NumRows(table);
	return TCL_OK;
    } else {
	long row;

	row = Blt_DataTableFindRow(table, ip->tagName);
	if (row >= 0) {
	    ip->iStart = ip->iEnd = RowIndex(table, row);
	    return TCL_OK;
	}
	ip->tablePtr = Blt_DataTableRowTagTable(ip->table, ip->tagName);
	if (ip->tablePtr != NULL) {
	    ip->type = TABLE_ITER_TAG;
	    return TCL_OK;
	}
    }
    if (interp != NULL) {
	Tcl_AppendResult(interp, "can't find row tag \"", ip->tagName, 
			 "\" in ", ClientName(table), (char *)NULL);
    }
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_DataTableFirstRow --
 *
 *	Returns the id of the next row derived from the given tag.
 *
 * Results:
 *	Returns the row location of the first item.  If no more rows
 *	can be found, then -1 is returned.
 *
 *----------------------------------------------------------------------
 */
long
Blt_DataTableFirstRow(Blt_DataTableIterator *ip)
{
    if (ip->type == TABLE_ITER_TAG) {
	Blt_HashEntry *hPtr;

	hPtr = Blt_FirstHashEntry(ip->tablePtr, &ip->cursor);
	if (hPtr == NULL) {
	    return -1;
	}
	return (long)Blt_GetHashValue(hPtr);
    } else {
	long offset;

	if (ip->iStart <= ip->iEnd) {
	    offset = RowOffset(ip->table, ip->iStart);
	    ip->iNext = ip->iStart + 1;
	    return offset;
	}
    } 
    return -1;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_DataTableNextRow --
 *
 *	Returns the id of the next row derived from the given tag.
 *
 * Results:
 *	Returns the row location of the first item.  If no more rows
 *	can be found, then -1 is returned.
 *
 *----------------------------------------------------------------------
 */
long
Blt_DataTableNextRow(Blt_DataTableIterator *ip)
{
    if (ip->type == TABLE_ITER_TAG) {
	Blt_HashEntry *hPtr;

	hPtr = Blt_NextHashEntry(&ip->cursor); 
	if (hPtr != NULL) {
	    return (long)Blt_GetHashValue(hPtr);
	}
    } else {
	if (ip->iNext <= ip->iEnd) {
	    long offset;

	    offset = RowOffset(ip->table, ip->iNext);
	    ip->iNext++;
	    return offset;
	}
    }	
    return -1;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_DataTableGetRow --
 *
 *	Gets the row offset associated the given row index, tag, or
 *	label.  This routine is used when you want only one row index.
 *	It's an error if more than one row is specified (e.g. "all"
 *	tag or range "1:4").  It's also an error if the row tag is
 *	empty (no rows are currently tagged).
 *
 *---------------------------------------------------------------------- 
 */
int 
Blt_DataTableGetRow(
    Tcl_Interp *interp, 
    Blt_DataTable table,
    Tcl_Obj *objPtr,
    long *rowPtr)
{
    Blt_DataTableIterator iRow;
    long firstRow, nextRow;

    if (Blt_DataTableGetRows(interp, table, objPtr, &iRow) != TCL_OK) {
	return TCL_ERROR;
    }
    firstRow = Blt_DataTableFirstRow(&iRow);
    if (firstRow < 0) {
	if (interp != NULL) {
	    Tcl_AppendResult(interp, "no rows specified by \"", 
			     Tcl_GetString(objPtr), "\"", (char *)NULL);
	}
	return TCL_ERROR;
    }
    nextRow = Blt_DataTableNextRow(&iRow);
    if (nextRow >= 0) {
	if (interp != NULL) {
	    Tcl_AppendResult(interp, "multiple rows specified by \"", 
			     Tcl_GetString(objPtr), "\"", (char *)NULL);
	}
	return TCL_ERROR;
    }
    *rowPtr = firstRow;
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_DataTableGetColumns --
 *
 *	Initials the table iterator to walk through the columns tagged
 *	by the given tag, label, or index, as represented in objPtr.
 *
 *	Notes: 
 *
 *	1) A tag doesn't need to point to any columns. It can be empty.
 *	This routine does not check if a tag represents any columns,
 *	only that the tag itself exists.
 *
 *	2) If a column label and tag are the same string, the label
 *	always wins.
 *
 *	3) A range of columns can be represented by x:y x-y {x y}
 *
 * Results:
 *	A standard Tcl result.  If there is an error parsing the index
 *	or tag, then TCL_ERROR is returned and an error message is
 *	left in the interpreter.
 *
 *---------------------------------------------------------------------- 
 */
int
Blt_DataTableGetColumns(
    Tcl_Interp *interp,
    Blt_DataTable table,
    Tcl_Obj *objPtr,
    Blt_DataTableIterator *ip)
{
    long n;
    CONST char *p, *colon, *pend;
    int nBytes;

    ip->table = table;
    ip->type = TABLE_ITER_INDEX;
    ip->tagName = Tcl_GetStringFromObj(objPtr, &nBytes);
    ip->iNext = -1;

    colon = NULL;
    for (p = ip->tagName, pend = p + nBytes; p < pend; p++) {
	if (*p != ':') {
	    continue;
	}
	if (colon != NULL) {
	    if (interp != NULL) {
		Tcl_AppendResult(interp, "bad range \"", Tcl_GetString(objPtr),
			"\": too many range specifiers.", (char *)NULL);
	    }
	    return TCL_ERROR;
	}
	colon = p;
    }
    if (colon != NULL) {
	Tcl_Obj *objPtr1, *objPtr2;
	int result;
	long from, to;

	if ((colon == ip->tagName) || (colon == (pend - 1))) {
	    if (interp != NULL) {
		Tcl_AppendResult(interp, "bad range \"", 
			Tcl_GetString(objPtr), "\": must specify \"from-to\"", 
				 (char *)NULL);
	    }
	    return TCL_ERROR;
	}	    
	objPtr1 = Tcl_NewStringObj(ip->tagName, colon - ip->tagName);
	colon++;
	objPtr2 = Tcl_NewStringObj(colon, pend - colon);
	result = Blt_DataTableGetColumn(interp, table, objPtr1, &from);
	if (result == TCL_OK) {
	    result = Blt_DataTableGetColumn(interp, table, objPtr2, &to);
	}
	Tcl_DecrRefCount(objPtr1);
	Tcl_DecrRefCount(objPtr2);
	if (result != TCL_OK) {
	    return TCL_ERROR;
	}
	ip->iStart = ColumnIndex(table, from);
	ip->iEnd = ColumnIndex(table, to);
	ip->type = TABLE_ITER_RANGE;
	return TCL_OK;
    }
    if (Tcl_GetLongFromObj((Tcl_Interp *)NULL, objPtr, &n) == TCL_OK) {
	if ((n < 1) || (n > NumColumns(table))) {
	    if (interp != NULL) {
		Tcl_AppendResult(interp, "bad column index \"", 
			Tcl_GetString(objPtr), "\"", (char *)NULL);
	    }
	    return TCL_ERROR;
	}		
	ip->iStart = ip->iEnd = n;
	return TCL_OK;
    } else if (strcmp(ip->tagName, "all") == 0) {
	ip->type = TABLE_ITER_ALL;
	ip->iStart = 1;
	ip->iEnd = NumColumns(table);
	return TCL_OK;
    } else if (strcmp(ip->tagName, "end") == 0) {
	ip->iStart = ip->iEnd = NumColumns(table);
	return TCL_OK;
    } else {
	long column;

	column = Blt_DataTableFindColumn(table, ip->tagName);
	if (column >= 0) {
	    ip->iStart = ip->iEnd = ColumnIndex(table, column);
	    return TCL_OK;
	}
	ip->tablePtr = Blt_DataTableColumnTagTable(ip->table, ip->tagName);
	if (ip->tablePtr != NULL) {
	    ip->type = TABLE_ITER_TAG;
	    return TCL_OK;
	}
    }
    if (interp != NULL) {
	Tcl_AppendResult(interp, "can't find column tag \"", ip->tagName, 
		"\" in ", ClientName(table), (char *)NULL);
    }
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_DataTableFirstColumn --
 *
 *	Returns the offset of the first column based upon given
 *	iterator.
 *
 * Results:
 *	Returns the column location of the first item.  If no more columns
 *	can be found, then -1 is returned.
 *
 *----------------------------------------------------------------------
 */
long
Blt_DataTableFirstColumn(Blt_DataTableIterator *ip)
{
    if (ip->type == TABLE_ITER_TAG) {
	Blt_HashEntry *hPtr;

	hPtr = Blt_FirstHashEntry(ip->tablePtr, &ip->cursor);
	if (hPtr == NULL) {
	    return -1;
	}
	return (long)Blt_GetHashValue(hPtr);
    } else {
	long offset;

	if (ip->iStart <= ip->iEnd) {
	    offset = ColumnOffset(ip->table, ip->iStart);
	    ip->iNext = ip->iStart + 1;
	    return offset;
	}
    } 
    return -1;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_DataTableNextColumn --
 *
 *	Returns the column location of the next column using the given
 *	iterator.
 *
 * Results:
 *	Returns the column location of the next item.  If no more
 *	columns can be found, then -1 is returned.
 *
 *----------------------------------------------------------------------
 */
long
Blt_DataTableNextColumn(Blt_DataTableIterator *ip)
{
    if (ip->type == TABLE_ITER_TAG) {
	Blt_HashEntry *hPtr;

	hPtr = Blt_NextHashEntry(&ip->cursor); 
	if (hPtr != NULL) {
	    return (long)Blt_GetHashValue(hPtr);
	}
    } else {
	if (ip->iNext <= ip->iEnd) {
	    long offset;

	    offset = ColumnOffset(ip->table, ip->iNext);
	    ip->iNext++;
	    return offset;
	}
    }	
    return -1;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_DataTableGetColumn --
 *
 *---------------------------------------------------------------------- 
 */
int 
Blt_DataTableGetColumn(
    Tcl_Interp *interp, 
    Blt_DataTable table, 
    Tcl_Obj *objPtr,
    long *columnPtr)
{
    Blt_DataTableIterator iColumn;
    long firstColumn, nextColumn;

    if (Blt_DataTableGetColumns(interp, table, objPtr, &iColumn) != TCL_OK) {
	return TCL_ERROR;
    }
    firstColumn = Blt_DataTableFirstColumn(&iColumn);
    if (firstColumn < 0) {
	if (interp != NULL) {
	    Tcl_AppendResult(interp, "no columns specified by \"", 
			     Tcl_GetString(objPtr), "\"", (char *)NULL);
	}
	return TCL_ERROR;
    }
    nextColumn = Blt_DataTableNextColumn(&iColumn);
    if (nextColumn >= 0) {
	if (interp != NULL) {
	    Tcl_AppendResult(interp, "multiple columns specified by \"", 
		Tcl_GetString(objPtr), "\"", (char *)NULL);
	}
	return TCL_ERROR;
    }
    *columnPtr = firstColumn;
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_DataTableDeleteTrace --
 *
 *	Deletes a trace.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	Memory is deallocated for the trace.
 *
 *----------------------------------------------------------------------
 */
void
Blt_DataTableDeleteTrace(Trace *tp)
{
    FreeTrace(tp);
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_DataTableTraces --
 *	
 *	Returns the chain of traces for a particular client.
 *
 * Results:
 *	Returns a pointer to the chain containing the traces for the
 *	given row.  If the row has no traces, then NULL is returned.
 *
 *---------------------------------------------------------------------- 
 */
Blt_Chain *
Blt_DataTableTraces(DataTable *dtPtr)
{
    return dtPtr->traces;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_DataTableClearRowTraces --
 *
 *	Removes all traces set for this row.  Note that this doesn't
 *	remove traces set for specific cells (row,column).  Row traces
 *	are stored in a chain, which in turn is held in a hash table,
 *	keyed by the row.
 *
 *---------------------------------------------------------------------- 
 */
void
Blt_DataTableClearRowTraces(Blt_DataTable table, long row)
{
    if ((row < 0) || (row >= NumRowsAllocated(table))) {
	return;
    }
    ClearRowTraces(table, row);
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_DataTableClearColumnTraces --
 *
 *	Removes all traces set for this column.  Note that this doesn't
 *	remove traces set for specific cells (row,column).  Column traces
 *	are stored in a chain, which in turn is held in a hash table,
 *	keyed by the column.
 *
 *---------------------------------------------------------------------- 
 */
void
Blt_DataTableClearColumnTraces(Blt_DataTable table, long column)
{
    if ((column < 0) || (column >= NumColumnsAllocated(table))) {
	return;
    }
    ClearColumnTraces(table, column);
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_DataTableCreateTrace --
 *
 *	Creates a trace for one or more tuples with one or more column
 *	keys.  Whenever a matching action occurs in the table object,
 *	the specified procedure is executed.
 *
 * Results:
 *	Returns a token for the trace.
 *
 * Side Effects:
 *	Memory is allocated for the trace.
 *
 *---------------------------------------------------------------------- 
 */
Blt_DataTableTrace
Blt_DataTableCreateTrace(
    DataTable *dtPtr,		/* Table to be traced. */
    long row, long column,	/* Offset of cell in table. */
    char *rowTag, char *colTag,
    unsigned int flags,		/* Bit mask indicating what actions to
				 * trace. */
    Blt_DataTableTraceProc *proc, /* Callback procedure for the trace. */
    Blt_DataTableTraceDeleteProc *deleteProc, 
    ClientData clientData)	/* One-word of data passed along when
				 * the callback is executed. */
{
    Trace *tp;

    tp = Blt_Calloc(1, sizeof (Trace));
    if (tp == NULL) {
	return NULL;
    }
    Blt_InitHashTable(&tp->activeTable, sizeof(RowColumnKey) / sizeof(int));
    tp->row = row;
    tp->column = column;
    if (rowTag != NULL) {
	tp->rowTag = Blt_Strdup(rowTag);
    }
    if (colTag != NULL) {
	tp->colTag = Blt_Strdup(colTag);
    }
    tp->flags = flags;
    tp->proc = proc;
    tp->deleteProc = deleteProc;
    tp->clientData = clientData;
    tp->chainPtr = dtPtr->traces;
    tp->linkPtr = Blt_ChainAppend(dtPtr->traces, tp);
    return tp;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_DataTableReleaseTags --
 *
 *	Releases the tag table used by this client.  
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	If no client is using the table, then it is freed.
 *
 *---------------------------------------------------------------------- 
 */
void
Blt_DataTableReleaseTags(DataTable *dtPtr)
{
    Tags *tagsPtr;

    tagsPtr = dtPtr->tagsPtr;
    tagsPtr->refCount--;
    if (tagsPtr->refCount <= 0) {
	Blt_HashEntry *hPtr;
	Blt_HashSearch cursor;

	for (hPtr = Blt_FirstHashEntry(&tagsPtr->rowTable, &cursor); 
	     hPtr != NULL; hPtr = Blt_NextHashEntry(&cursor)) {
	    Blt_HashTable *tablePtr;

	    tablePtr = Blt_GetHashValue(hPtr); 
	    Blt_DeleteHashTable(tablePtr);
	    Blt_Free(tablePtr);
	}
	Blt_DeleteHashTable(&tagsPtr->rowTable);
	dtPtr->rowTags = NULL;
	for (hPtr = Blt_FirstHashEntry(&tagsPtr->columnTable, &cursor); 
	     hPtr != NULL; hPtr = Blt_NextHashEntry(&cursor)) {
	    Blt_HashTable *tablePtr;

	    tablePtr = Blt_GetHashValue(hPtr); 
	    Blt_DeleteHashTable(tablePtr);
	    Blt_Free(tablePtr);
	}
	Blt_DeleteHashTable(&tagsPtr->columnTable);
	Blt_Free(tagsPtr);
	dtPtr->columnTags = NULL;
    }
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_DataTableTagsAreShared --
 *
 *	Returns whether the tag table is shared with another client.
 *
 * Results:
 *	Returns TRUE if the current tag table is shared with another
 *	client, FALSE otherwise.
 *
 *---------------------------------------------------------------------- 
 */
int
Blt_DataTableTagsAreShared(DataTable *dtPtr)
{
    return (dtPtr->tagsPtr->refCount > 1);
}   

/*
 *----------------------------------------------------------------------
 *
 * Blt_DataTableRowTagTable --
 *
 *	Returns the hash table containing row indices for a tag.
 *
 * Results:
 *	Returns a pointer to the hash table containing indices for the
 *	given tag.  If the row has no tags, then NULL is returned.
 *
 *---------------------------------------------------------------------- 
 */
Blt_HashTable *
Blt_DataTableRowTagTable(Blt_DataTable table, CONST char *tagName)		
{
    Blt_HashEntry *hPtr;

    hPtr = Blt_FindHashEntry(RowTags(table), tagName);
    if (hPtr == NULL) {
	return NULL;		/* Row isn't tagged. */
    }
    return Blt_GetHashValue(hPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_DataTableColumnTagTable --
 *
 *	Returns the hash table containing column indices for a tag.
 *
 * Results:
 *	Returns a pointer to the hash table containing indices for the
 *	given tag.  If the tag has no indices, then NULL is
 *	returned.
 *
 *---------------------------------------------------------------------- 
 */
Blt_HashTable *
Blt_DataTableColumnTagTable(Blt_DataTable table, CONST char *tagName)
{
    Blt_HashEntry *hPtr;

    hPtr = Blt_FindHashEntry(ColumnTags(table), tagName);
    if (hPtr == NULL) {
	return NULL;		
    }
    return Blt_GetHashValue(hPtr);
}


/*
 *----------------------------------------------------------------------
 *
 * Blt_DataTableForgetRowTag --
 *
 *	Removes a tag from the row tag table.  Row tags are contained
 *	in hash tables keyed by the tag name.  Each table is in turn
 *	hashed by the row index in the row tag table.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	Entries for the given tag in the corresponding row in hash
 *	tables may be removed. 
 *	
 *---------------------------------------------------------------------- 
 */
int
Blt_DataTableForgetRowTag(
    Tcl_Interp *interp, 
    Blt_DataTable table, 
    CONST char *tagName)
{
    Blt_HashEntry *hPtr;
    Blt_HashTable *tagTablePtr;

    if ((strcmp(tagName, "all") == 0) || (strcmp(tagName, "end") == 0)) {
	return TCL_OK;		/* Can't forget reserved tags. */
    }
    hPtr = Blt_FindHashEntry(RowTags(table), tagName);
    if (hPtr == NULL) {
	if (interp != NULL) {
	    Tcl_AppendResult(interp, "unknown row tag \"", tagName, "\"", 
			     (char *)NULL);
	}
	return TCL_ERROR;	/* No such row tag. */
    }
    tagTablePtr = Blt_GetHashValue(hPtr);
    Blt_DeleteHashTable(tagTablePtr);
    Blt_Free(tagTablePtr);
    Blt_DeleteHashEntry(RowTags(table), hPtr);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_DataTableForgetColumnTag --
 *
 *	Removes a tag from the column tag table.  Column tags are
 *	contained in hash tables keyed by the tag name.  Each table is
 *	in turn hashed by the column offset in the column tag table.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	Entries for the given tag in the corresponding column in hash
 *	tables may be removed.
 *	
 *---------------------------------------------------------------------- 
 */
int
Blt_DataTableForgetColumnTag(
    Tcl_Interp *interp, 
    Blt_DataTable table, 
    CONST char *tagName)
{
    Blt_HashEntry *hPtr;
    Blt_HashTable *tagTablePtr;

    if ((strcmp(tagName, "all") == 0) || (strcmp(tagName, "end") == 0)) {
	return TCL_OK;		/* Can't forget reserved tags. */
    }
    hPtr = Blt_FindHashEntry(ColumnTags(table), tagName);
    if (hPtr == NULL) {
	if (interp != NULL) {
	    Tcl_AppendResult(interp, "unknown column tag \"", tagName, "\"", 
			     (char *)NULL);
	}
	return TCL_ERROR;	/* No such column tag. */
    }
    tagTablePtr = Blt_GetHashValue(hPtr);
    Blt_DeleteHashTable(tagTablePtr);
    Blt_Free(tagTablePtr);
    Blt_DeleteHashEntry(ColumnTags(table), hPtr);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_DataTableSetRowTag --
 *
 *	Associates a tag with a given row.  Individual row tags are
 *	stored in hash tables keyed by the tag name.  Each table is in
 *	turn stored in a hash table keyed by the row location.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	A tag is stored for a particular row.
 *
 *---------------------------------------------------------------------- 
 */
int
Blt_DataTableSetRowTag(
    Tcl_Interp *interp, 
    Blt_DataTable table, 
    long row, 
    CONST char *tagName)
{
    Blt_HashEntry *hPtr;
    Blt_HashTable *tagTablePtr;
    int isNew;

    if ((row < 0) || (row >= NumRowsAllocated(table))) {
	if (interp != NULL) {
	    Tcl_AppendResult(interp, "bad row \"", Blt_Ltoa(row), 
		"\"", (char *)NULL);
	}
	return TCL_ERROR;
    }
    if ((strcmp(tagName, "all") == 0) || (strcmp(tagName, "end") == 0)) {
	return TCL_OK;		/* Don't need to create reserved
				 * tags. */
    }
    if (tagName[0] == '\0') {
	if (interp != NULL) {
	    Tcl_AppendResult(interp, "bad tag \"", tagName, 
		"\": can't be empty.", (char *)NULL);
	}
	return TCL_ERROR;
    }
    if (tagName[0] == '-') {
	if (interp != NULL) {
	    Tcl_AppendResult(interp, "bad tag \"", tagName, 
		"\": can't start with a '-'.", (char *)NULL);
	}
	return TCL_ERROR;
    }
    if (isdigit(tagName[0])) {
	if (interp != NULL) {
	    Tcl_AppendResult(interp, "bad tag \"", tagName, 
		"\": can't start with a digit.", (char *)NULL);
	}
	return TCL_ERROR;
    }
    if (strchr(tagName, ':') != NULL) {
	if (interp != NULL) {
	    Tcl_AppendResult(interp, "bad tag \"", tagName, 
		"\": can't contain range specifier ':'.", (char *)NULL);
	}
	return TCL_ERROR;
    }
    hPtr = Blt_CreateHashEntry(RowTags(table), tagName, &isNew);
    if (hPtr == NULL) {
	if (interp != NULL) {
	    Tcl_AppendResult(interp, "can't add tag \"", tagName, 
			 "\": out of memory", (char *)NULL);
	}
	return TCL_ERROR;
    }
    assert(hPtr);
    if (isNew) {
	tagTablePtr = Blt_Malloc(sizeof(Blt_HashTable));
	Blt_InitHashTable(tagTablePtr, BLT_ONE_WORD_KEYS);
	Blt_SetHashValue(hPtr, tagTablePtr);
    } else {
	tagTablePtr = Blt_GetHashValue(hPtr);
    }
    hPtr = Blt_CreateHashEntry(tagTablePtr, (char *)row, &isNew);
    assert(hPtr);
    if (isNew) {
	Blt_SetHashValue(hPtr, row);
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_DataTableSetColumnTag --
 *
 *	Associates a tag with a given column.  Individual column tags
 *	are stored in hash tables keyed by the tag name.  Each table
 *	is in turn stored in a hash table keyed by the column
 *	location.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	A tag is stored for a particular column.
 *
 *---------------------------------------------------------------------- 
 */
int
Blt_DataTableSetColumnTag(
    Tcl_Interp *interp, 
    Blt_DataTable table, 
    long column, 
    CONST char *tagName)
{
    Blt_HashEntry *hPtr;
    Blt_HashTable *tagTablePtr;
    int isNew;
    
    if ((column < 0) || (column >= NumColumnsAllocated(table))) {
	if (interp != NULL) {
	    Tcl_AppendResult(interp, "bad column \"", Blt_Ltoa(column), 
		"\"", (char *)NULL);
	}
	return TCL_ERROR;
    }
    if ((strcmp(tagName, "all") == 0) || (strcmp(tagName, "end") == 0)) {
	return TCL_OK;			/* Don't create reserved tags. */
    }
    if (tagName[0] == '\0') {
	if (interp != NULL) {
	    Tcl_AppendResult(interp, "bad tag \"", tagName, 
		"\": can't be empty.", (char *)NULL);
	}
	return TCL_ERROR;
    }
    if (tagName[0] == '-') {
	if (interp != NULL) {
	    Tcl_AppendResult(interp, "bad tag \"", tagName, 
		"\": can't start with a '-'.", (char *)NULL);
	}
	return TCL_ERROR;
    }
    if (isdigit(tagName[0])) {
	if (interp != NULL) {
	    Tcl_AppendResult(interp, "bad tag \"", tagName, 
		"\": can't start with a digit.", (char *)NULL);
	}
	return TCL_ERROR;
    }
    if (strchr(tagName, ':') != NULL) {
	if (interp != NULL) {
	    Tcl_AppendResult(interp, "bad tag \"", tagName, 
		"\": can't contain range specifier ':'.", (char *)NULL);
	}
	return TCL_ERROR;
    }
    hPtr = Blt_CreateHashEntry(ColumnTags(table), tagName, &isNew);
    if (hPtr == NULL) {
	if (interp != NULL) {
	    Tcl_AppendResult(interp, "can't add tag \"", tagName, 
			 "\": out of memory", (char *)NULL);
	}
	return TCL_ERROR;
    }
    if (isNew) {
	tagTablePtr = Blt_Malloc(sizeof(Blt_HashTable));
	Blt_InitHashTable(tagTablePtr, BLT_ONE_WORD_KEYS);
	Blt_SetHashValue(hPtr, tagTablePtr);
    } else {
	tagTablePtr = Blt_GetHashValue(hPtr);
    }
    hPtr = Blt_CreateHashEntry(tagTablePtr, (char *)column, &isNew);
    assert(hPtr);
    if (isNew) {
	Blt_SetHashValue(hPtr, column);
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_DataTableHasRowTag --
 *
 *	Checks if a tag is associated with the given row.  
 *
 * Results:
 *	Returns TRUE if the tag is found, FALSE otherwise.
 *
 *---------------------------------------------------------------------- 
 */
int
Blt_DataTableHasRowTag(Blt_DataTable table, long row, CONST char *tagName)
{
    Blt_HashTable *tagTablePtr;
    Blt_HashEntry *hPtr;

    if (strcmp(tagName, "all") == 0) {
	return TRUE;		/* "all" tags matches every row. */
    }
    if (strcmp(tagName, "end") == 0) {
	return (RowIndex(table, row) == NumRows(table));
    }
    tagTablePtr = Blt_DataTableRowTagTable(table, tagName);
    if (tagTablePtr == NULL) {
	return FALSE;
    }
    hPtr = Blt_FindHashEntry(tagTablePtr, (char *)row);
    if (hPtr != NULL) {
	return TRUE;		/* Found tag in row tag table. */
    }
    return FALSE;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_DataTableHasColumnTag --
 *
 *	Checks if a tag is associated with the given column.  
 *
 * Results:
 *	Returns TRUE if the tag is found, FALSE otherwise.
 *
 *---------------------------------------------------------------------- 
 */
int
Blt_DataTableHasColumnTag(Blt_DataTable table, long column, CONST char *tagName)
{
    Blt_HashTable *tagTablePtr;
    Blt_HashEntry *hPtr;

    if (strcmp(tagName, "all") == 0) {
	return TRUE;		/* "all" tags matches every column. */
    }
    if (strcmp(tagName, "end") == 0) {
	return (ColumnIndex(table, column) == NumColumns(table));
    }
    tagTablePtr = Blt_DataTableColumnTagTable(table, tagName);
    if (tagTablePtr == NULL) {
	return FALSE;
    }
    hPtr = Blt_FindHashEntry(tagTablePtr, (char *)column);
    if (hPtr != NULL) {
	return TRUE;		/* Found tag in column tag table. */
    }
    return FALSE;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_DataTableUnsetRowTag --
 *
 *	Removes a tag from a given row.  
 *
 * Results:
 *	A standard Tcl result.  If an error occurred, TCL_ERROR
 *	is returned and the interpreter result contains the error
 *	message.
 *
 * Side Effects:
 *      The tag associcated with the row is freed.
 *
 *---------------------------------------------------------------------- 
 */
int
Blt_DataTableUnsetRowTag(
    Tcl_Interp *interp,
    Blt_DataTable table, 
    long row, 
    CONST char *tagName)
{
    Blt_HashEntry *hPtr;
    Blt_HashTable *tagTablePtr;

    if ((row < 0) || (row >= NumRowsAllocated(table))) {
	if (interp != NULL) {
	    Tcl_AppendResult(interp, "bad row \"", Blt_Ltoa(row), "\"", 
			 (char *)NULL);
	}
	return TCL_ERROR;
    }
    if ((strcmp(tagName, "all") == 0) || (strcmp(tagName, "end") == 0)) {
	return TCL_OK;		/* Can't remove reserved tags. */
    } 
    tagTablePtr = Blt_DataTableRowTagTable(table, tagName);
    if (tagTablePtr == NULL) {
	if (interp != NULL) {
	    Tcl_AppendResult(interp, "unknown row tag \"", tagName, "\"", 
			     (char *)NULL);
	}
	return TCL_ERROR;
    }
    hPtr = Blt_FindHashEntry(tagTablePtr, (char *)row);
    if (hPtr != NULL) {
	Blt_DeleteHashEntry(tagTablePtr, hPtr);
    }
    return TCL_OK;
}    

/*
 *----------------------------------------------------------------------
 *
 * Blt_DataTableUnsetColumnTag --
 *
 *	Removes a tag from a given column.  
 *
 * Results:
 *	A standard Tcl result.  If an error occurred, TCL_ERROR
 *	is returned and the interpreter result contains the error
 *	message.
 *
 * Side Effects:
 *      The tag associcated with the column is freed.
 *
 *---------------------------------------------------------------------- 
 */
int
Blt_DataTableUnsetColumnTag(
    Tcl_Interp *interp,
    Blt_DataTable table, 
    long column, 
    CONST char *tagName)
{
    Blt_HashEntry *hPtr;
    Blt_HashTable *tagTablePtr;

    if ((column < 0) || (column >= NumColumnsAllocated(table))) {
	if (interp != NULL) {
	    Tcl_AppendResult(interp, "bad column \"", Blt_Ltoa(column), "\"", 
			 (char *)NULL);
	}
	return TCL_ERROR;
    }
    if ((strcmp(tagName, "all") == 0) || (strcmp(tagName, "end") == 0)) {
	return TCL_OK;		/* Can't remove reserved tags. */
    } 
    tagTablePtr = Blt_DataTableColumnTagTable(table, tagName);
    if (tagTablePtr == NULL) {
	if (interp != NULL) {
	    Tcl_AppendResult(interp, "unknown column tag \"", tagName, "\"", 
			     (char *)NULL);
	}
	return TCL_ERROR;
    }
    hPtr = Blt_FindHashEntry(tagTablePtr, (char *)column);
    if (hPtr != NULL) {
	Blt_DeleteHashEntry(tagTablePtr, hPtr);
    }
    return TCL_OK;
}    

/*
 *----------------------------------------------------------------------
 *
 * Blt_DataTableClearRowTags --
 *
 *	Removes all tags for a given row.  
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *      All tags associcated with the row are freed.
 *
 *---------------------------------------------------------------------- 
 */
void
Blt_DataTableClearRowTags(Blt_DataTable table, long row)
{
    if ((row < 0) || (row >= NumRowsAllocated(table))) {
	return;
    }
    ClearRowTags(table, row);
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_DataTableClearColumnTags --
 *
 *	Removes all tags for a given column.  
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *      All tags associcated with the column are freed.
 *
 *---------------------------------------------------------------------- 
 */
void
Blt_DataTableClearColumnTags(Blt_DataTable table, long column)
{
    if ((column < 0) || (column >= NumColumnsAllocated(table))) {
	return;
    }
    ClearColumnTags(table, column);
}

/*
 * --------------------------------------------------------------
 *
 * Blt_DataTableGetValue --
 *
 *	Gets a scalar Tcl_Obj value from the table at the designated
 *	row, column location.  "Read" traces may be fired *before* the
 *	value is retrieved.  If no value exists at that location,
 *	*objPtrPtr is set to NULL.
 *
 * Results:
 *	A standard Tcl result.  Returns TCL_OK if successful accessing
 *	the table location.  If an error occurs, TCL_ERROR is returned
 *	and an error message is left in the interpreter.
 *
 * -------------------------------------------------------------- 
 */
int
Blt_DataTableGetValue(
    Tcl_Interp *interp,		/* Interpreter to reports result to. */
    DataTable *dtPtr,	/* Table client. */
    long row, long column,	/* Cell location.  */
    Tcl_Obj **objPtrPtr)	/* (out) Value from table.  */
{
    TableObject *tableObjPtr;	/* Table. */

    tableObjPtr = dtPtr->tableObjPtr;
    if ((row < 0) || (row >= NumRowsAllocated(dtPtr))) {
	Tcl_AppendResult(interp, "bad row \"", Blt_Ltoa(row), "\"", 
			 (char *)NULL);
	return TCL_ERROR;
    }
    if ((column < 0) || (column >= NumColumnsAllocated(dtPtr))) {
	Tcl_AppendResult(interp, "bad column \"", Blt_Ltoa(column), "\"", 
			 (char *)NULL);
	return TCL_ERROR;
    }
    CallClientTraces(tableObjPtr, row, column, TABLE_TRACE_READS);
    *objPtrPtr = GetValue(tableObjPtr, row, column);
    return TCL_OK;
}

/*
 * --------------------------------------------------------------
 *
 * Blt_DataTableSetValue --
 *
 *	Sets a scalar Tcl_Obj value in the table at the designated row
 *	and column.  "Write" and possibly "create" or "unset" traces
 *	may be fired *after* the value is set.  If valueObjPtr is
 *	NULL, this indicates to unset the old value.
 *
 * Results:
 *	A standard Tcl result.  Returns TCL_OK if successful setting
 *	the value at the table location.  If an error occurs,
 *	TCL_ERROR is returned and an error message is left in the
 *	interpreter.
 *
 * -------------------------------------------------------------- 
 */
int
Blt_DataTableSetValue(
    Tcl_Interp *interp,
    DataTable *dtPtr,	/* Table client. */
    long row, long column,	/* Cell location.  */
    Tcl_Obj *objPtr)		/* New value to be set.  If NULL,
				 * indicates to unset the old
				 * value. */
{
    TableObject *tableObjPtr;	/* Table. */
    Tcl_Obj *oldObjPtr;
    unsigned int flags;

    tableObjPtr = dtPtr->tableObjPtr;
    if ((row < 0) || (row >= NumRowsAllocated(dtPtr))) {
	Tcl_AppendResult(interp, "bad row \"", Blt_Ltoa(row), "\"", 
			 (char *)NULL);
	return TCL_ERROR;
    }
    if ((column < 0) || (column >= NumColumnsAllocated(dtPtr))) {
	Tcl_AppendResult(interp, "bad column \"", Blt_Ltoa(column), "\"", 
			 (char *)NULL);
	return TCL_ERROR;
    }
    oldObjPtr = SetValue(tableObjPtr, row, column, objPtr);
    flags = TABLE_TRACE_WRITES;
    if (objPtr == NULL) {
	flags = TABLE_TRACE_UNSETS;
    } else if (oldObjPtr == NULL) {
	flags |= TABLE_TRACE_CREATES;
    } 
    CallClientTraces(tableObjPtr, row, column, flags);
    return TCL_OK;
}

/*
 * --------------------------------------------------------------
 *
 * Blt_DataTableUnsetValue --
 *
 *	Unsets a scalar Tcl_Obj value in the table at the designated
 *	row, column location.  It's okay is there is presently no
 *	value at the location. Unset traces may be fired *before* the
 *	value is unset.
 *
 * Results:
 *	A standard Tcl result.  Returns TCL_OK if successful unsetting
 *	the value at the table location.  If an error occurs,
 *	TCL_ERROR is returned and an error message is left in the
 *	interpreter.
 *
 * -------------------------------------------------------------- 
 */
int
Blt_DataTableUnsetValue(
    Tcl_Interp *interp,
    DataTable *dtPtr,	/* Table client. */
    long row, long column)	/* Cell location.  */
{
    TableObject *tableObjPtr;	/* Table. */

    tableObjPtr = dtPtr->tableObjPtr;
    if ((row < 0) || (row >= NumRowsAllocated(dtPtr))) {
	Tcl_AppendResult(interp, "bad row \"", Blt_Ltoa(row), "\"", 
			 (char *)NULL);
	return TCL_ERROR;
    }
    if ((column < 0) || (column >= NumColumnsAllocated(dtPtr))) {
	Tcl_AppendResult(interp, "bad column \"", Blt_Ltoa(column), "\"", 
			 (char *)NULL);
	return TCL_ERROR;
    }
    if (GetValue(tableObjPtr, row, column) != NULL) {
	CallClientTraces(tableObjPtr, row, column, TABLE_TRACE_UNSETS);
	UnsetValue(tableObjPtr, row, column);
    }
    return TCL_OK;
}

/*
 * --------------------------------------------------------------
 *
 * Blt_DataTableGetArrayValue --
 *
 *	Gets a Tcl_Obj value from array object in the table at the
 *	designated row and column.  Read traces may be fired *before*
 *	the value is accessed.
 *
 * Results:
 *	A standard Tcl result.  If an error occurs, TCL_ERROR is
 *	returned and an error message is left in the interpreter.
 *
 * -------------------------------------------------------------- 
 */
Tcl_Obj *
Blt_DataTableArrayNames(
    Tcl_Interp *interp,
    DataTable *dtPtr,	/* Table client. */
    long row, long column)	/* Cell location.  */
{
    TableObject *tableObjPtr;	/* Table. */
    Tcl_Obj *listObjPtr, *arrayObjPtr;

    tableObjPtr = dtPtr->tableObjPtr;
    if ((row < 0) || (row >= NumRowsAllocated(dtPtr))) {
	Tcl_AppendResult(interp, "bad row \"", Blt_Ltoa(row), "\"", 
			 (char *)NULL);
	return NULL;
    }
    if ((column < 0) || (column >= NumColumnsAllocated(dtPtr))) {
	Tcl_AppendResult(interp, "bad column \"", Blt_Ltoa(column), "\"", 
			 (char *)NULL);
	return NULL;
    }
    listObjPtr = Tcl_NewListObj(0, (Tcl_Obj **) NULL);

    /* Access the data value after traces have been called. */
    arrayObjPtr = GetArray(tableObjPtr, row, column);
    if (arrayObjPtr != NULL) {
	Blt_HashTable *tablePtr;

	if (Blt_GetArrayFromObj(interp, arrayObjPtr, &tablePtr) != TCL_OK) {
	    Tcl_DecrRefCount(listObjPtr);
	    return NULL;
	}
	if (tablePtr != NULL) {
	    Blt_HashEntry *hPtr;
	    Blt_HashSearch cursor;

	    for (hPtr = Blt_FirstHashEntry(tablePtr, &cursor); hPtr != NULL;
		 hPtr = Blt_NextHashEntry(&cursor)) {
		CONST char *key;
		Tcl_Obj *objPtr;

		key = Blt_GetHashKey(tablePtr, hPtr);
		objPtr = Tcl_NewStringObj(key, -1);
		Tcl_ListObjAppendElement(interp, listObjPtr, objPtr);
	    }
	} 
    }
    return listObjPtr;
}

/*
 * --------------------------------------------------------------
 *
 * Blt_DataTableGetArrayValue --
 *
 *	Gets a Tcl_Obj value from array object in the table at the
 *	designated row and column.  Read traces may be fired *before*
 *	the value is accessed.
 *
 * Results:
 *	A standard Tcl result.  If an error occurs, TCL_ERROR is
 *	returned and an error message is left in the interpreter.
 *
 * -------------------------------------------------------------- 
 */
int
Blt_DataTableGetArrayValue(
    Tcl_Interp *interp,
    DataTable *dtPtr,	/* Table client. */
    long row, long column,	/* Cell location.  */
    CONST char *key,
    Tcl_Obj **objPtrPtr)
{
    TableObject *tableObjPtr;	/* Table. */
    Tcl_Obj *arrayObjPtr, *objPtr;

    tableObjPtr = dtPtr->tableObjPtr;
    if ((row < 0) || (row >= NumRowsAllocated(dtPtr))) {
	Tcl_AppendResult(interp, "bad row \"", Blt_Ltoa(row), "\"", 
			 (char *)NULL);
	return TCL_ERROR;
    }
    if ((column < 0) || (column >= NumColumnsAllocated(dtPtr))) {
	Tcl_AppendResult(interp, "bad column \"", Blt_Ltoa(column), "\"", 
			 (char *)NULL);
	return TCL_ERROR;
    }
    CallClientTraces(tableObjPtr, row, column, TABLE_TRACE_READS);
    /* Access the data value after traces have been called. */
    arrayObjPtr = GetArray(tableObjPtr, row, column);
    if (arrayObjPtr == NULL) {
	objPtr = NULL;
    } else {
	Blt_HashTable *tablePtr;

	if (Blt_GetArrayFromObj(interp, arrayObjPtr, &tablePtr) != TCL_OK) {
	    return TCL_ERROR;
	}
	if (tablePtr == NULL) {
	    objPtr = NULL;
	} else {
	    Blt_HashEntry *hPtr;

	    hPtr = Blt_FindHashEntry(tablePtr, key);
	    if (hPtr == NULL) {
		if (interp != NULL) {
		    Tcl_AppendResult(interp, "can't find array element \"", 
				     key, "\"", (char *)NULL);
		}
		return TCL_ERROR;
	    }
	    objPtr = Blt_GetHashValue(hPtr);
	} 
    }
    *objPtrPtr = objPtr;
    return TCL_OK;
}

/*
 * --------------------------------------------------------------
 *
 * Blt_DataTableSetArrayValue --
 *
 *	Sets a Tcl_Obj value in the array object for the table at the
 *	designated row and column.  "Write" and possibly "create"
 *	traces may be fired *after* the value is set in the array.
 *
 * Results:
 *	A standard Tcl result.  Returns TCL_OK if successful setting
 *	the value at the table location.  If an error occurs,
 *	TCL_ERROR is returned and an error message is left in the
 *	interpreter.
 *
 * -------------------------------------------------------------- 
 */
int
Blt_DataTableSetArrayValue(
    Tcl_Interp *interp,
    DataTable *dtPtr,	/* Table client. */
    long row, long column,	/* Cell location.  */
    CONST char *key,		/* Name of array element in cell. */
    Tcl_Obj *valueObjPtr)	/* Value to be stored in the array. */
{
    Blt_HashEntry *hPtr;
    Blt_HashTable *tablePtr;
    TableObject *tableObjPtr;
    Tcl_Obj *arrayObjPtr;
    int isNew;
    unsigned int flags;

    tableObjPtr = dtPtr->tableObjPtr;
    if ((row < 0) || (row >= NumRowsAllocated(dtPtr))) {
	Tcl_AppendResult(interp, "bad row \"", Blt_Ltoa(row), "\"", 
			 (char *)NULL);
	return TCL_ERROR;
    }
    if ((column < 0) || (column >= NumColumnsAllocated(dtPtr))) {
	Tcl_AppendResult(interp, "bad column \"", Blt_Ltoa(column), "\"", 
			 (char *)NULL);
	return TCL_ERROR;
    }
    flags = TABLE_TRACE_WRITES;
    arrayObjPtr = SetArray(tableObjPtr, row, column);
    if (Blt_GetArrayFromObj(interp, arrayObjPtr, &tablePtr) != TCL_OK) {
	return TCL_ERROR;
    }
    Tcl_InvalidateStringRep(arrayObjPtr);
    hPtr = Blt_CreateHashEntry(tablePtr, key, &isNew);
    assert(hPtr);
    Tcl_IncrRefCount(valueObjPtr);
    if (!isNew) {
	Tcl_Obj *oldValueObjPtr;

	/* An element by the same name already exists. Decrement the
	 * reference count of the old value. */

	oldValueObjPtr = Blt_GetHashValue(hPtr);
	if (oldValueObjPtr != NULL) {
	    Tcl_DecrRefCount(oldValueObjPtr);
	} else {
	    flags |= TABLE_TRACE_CREATES;
	}
    } 
    Blt_SetHashValue(hPtr, valueObjPtr);
    CallClientTraces(tableObjPtr, row, column, flags);
    return TCL_OK;
}

/*
 * --------------------------------------------------------------
 *
 * Blt_DataTableUnsetValue --
 *
 *	Unsets a Tcl_Obj value in the array object in the table at the
 *	designated row, column location.  It's okay is there is
 *	presently no value at the location. Unset traces may be fired
 *	*before* the value in the array is unset.
 *
 * Results:
 *	A standard Tcl result.  Returns TCL_OK if successful unsetting
 *	the value at the table location.  If an error occurs,
 *	TCL_ERROR is returned and an error message is left in the
 *	interpreter.
 *
 * -------------------------------------------------------------- 
 */
int
Blt_DataTableUnsetArrayValue( 
    Tcl_Interp *interp,
    DataTable *dtPtr,	/* Table client. */
    long row, long column,	/* Cell location.  */
    CONST char *key)
{
    Blt_HashTable *tablePtr;
    TableObject *tableObjPtr;
    Tcl_Obj *arrayObjPtr;

    tableObjPtr = dtPtr->tableObjPtr;
    if ((row < 0) || (row >= NumRowsAllocated(dtPtr))) {
	Tcl_AppendResult(interp, "bad row \"", Blt_Ltoa(row), "\"", 
			 (char *)NULL);
	return TCL_ERROR;
    }
    if ((column < 0) || (column >= NumColumnsAllocated(dtPtr))) {
	Tcl_AppendResult(interp, "bad column \"", Blt_Ltoa(column), "\"", 
			 (char *)NULL);
	return TCL_ERROR;
    }
    arrayObjPtr = GetArray(tableObjPtr, row, column);
    if (arrayObjPtr == NULL) {
	return TCL_OK;
    }
    if (Blt_GetArrayFromObj(interp, arrayObjPtr, &tablePtr) != TCL_OK) {
	return TCL_ERROR;
    }
    {
	Blt_HashEntry *hPtr;
	Tcl_Obj *valueObjPtr;

	hPtr = Blt_FindHashEntry(tablePtr, key);
	if (hPtr == NULL) {
	    return TCL_OK;		/* Element doesn't exist, Ok. */
	}
	CallClientTraces(tableObjPtr, row, column, TABLE_TRACE_UNSETS);
	valueObjPtr = Blt_GetHashValue(hPtr);
	Tcl_DecrRefCount(valueObjPtr);
	Blt_DeleteHashEntry(tablePtr, hPtr);
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_DataTableCreateObject --
 *
 *	Creates a table object by the designated name.  It's an error
 *	if a table object already exists by that name.  After
 *	successfully creating the object, the caller must then call
 *	Blt_DataTableGetToken to allocate a token to share and manipulate
 *	the object.
 *
 * Results:
 *	A standard Tcl result.  If successful, a new table object is
 *	created and TCL_OK is returned.  If an object already exists
 *	or the table object can't be allocated, then TCL_ERROR is
 *	returned and an error message is left in the interpreter.
 *
 * Side Effects:
 *	A new table object is created.
 *
 *---------------------------------------------------------------------- 
 */
int
Blt_DataTableCreate(
    Tcl_Interp *interp,		/* Interpreter to report errors back to. */
    CONST char *name,		/* Name of tuple in namespace.  Object
				 * must not already exist. */
    DataTable **dtPtrPtr)	/* (out) Client token of newly created
				 * table object.  Releasing the token
				 * will free the tuple.  If NULL, no
				 * token is generated. */
{
    Tcl_DString ds;
    DataTableInterpData *dataPtr;
    TableObject *tableObjPtr;
    DataTable *newClientPtr;
    Blt_ObjectName objName;
    char string[200];
    char *qualName;

    dataPtr = GetDataTableInterpData(interp);
    if (name != NULL) {
	/* Check if a client by this name already exist in the current
	 * namespace. */
	if (GetDataTable(dataPtr, name, NS_SEARCH_CURRENT) != NULL) {
	    Tcl_AppendResult(interp, "a table object \"", name,
		"\" already exists", (char *)NULL);
	    return TCL_ERROR;
	}
    } else {
	/* Generate a unique name in the current namespace. */
	do  {
	    sprintf(string, "datatable%d", dataPtr->nextId++);
	} while (GetDataTable(dataPtr, name, NS_SEARCH_CURRENT) != NULL);
	name = string;
    } 
    /* 
     * Tear apart and put back together the namespace-qualified name 
     * of the object.  This is to ensure that naming is consistent.
     */ 
    if (!Blt_ParseObjectName(interp, name, &objName, 0)) {
	return TCL_ERROR;
    }
    tableObjPtr = NewTableObject(interp);
    if (tableObjPtr == NULL) {
	Tcl_AppendResult(interp, "can't allocate table object.", (char *)NULL);
	Tcl_DStringFree(&ds);
	return TCL_ERROR;
    }
    qualName = Blt_MakeQualifiedName(&objName, &ds);
    newClientPtr = NewDataTable(dataPtr, tableObjPtr, qualName);
    Tcl_DStringFree(&ds);
    if (newClientPtr == NULL) {
	Tcl_AppendResult(interp, "can't allocate table token", (char *)NULL);
	return TCL_ERROR;
    }
    if (dtPtrPtr != NULL) {
	*dtPtrPtr = newClientPtr;
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_DataTableGetToken --
 *
 *	Allocates a token for the table object designated by name.
 *	It's an error if no table object exists by that name.  The
 *	token returned is passed to various routines to manipulate the
 *	object.  Traces and event notifications are also made through
 *	the token.
 *
 * Results:
 *	A new token is returned representing the table object.  
 *
 * Side Effects:
 *	If this is the remaining client, then the table object itself
 *	is freed.
 *
 *---------------------------------------------------------------------- 
 */
int
Blt_DataTableGetToken(
    Tcl_Interp *interp,		/* Interpreter to report errors back to. */
    CONST char *name,		/* Name of table object in namespace. */
    DataTable **dtPtrPtr)
{
    DataTable *dtPtr, *newClientPtr;
    DataTableInterpData *dataPtr;

    dataPtr = GetDataTableInterpData(interp);
    dtPtr = GetDataTable(dataPtr, name, NS_SEARCH_BOTH);
    if ((dtPtr == NULL) || (dtPtr->tableObjPtr == NULL)) {
	Tcl_AppendResult(interp, "can't find a table object \"", name, "\"", 
		(char *)NULL);
	return TCL_ERROR;
    }
    newClientPtr = NewDataTable(dataPtr, dtPtr->tableObjPtr, name);
    if (newClientPtr == NULL) {
	Tcl_AppendResult(interp, "can't allocate token for table \"", name, 
		"\"", (char *)NULL);
	return TCL_ERROR;
    }
    *dtPtrPtr = newClientPtr;
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_DataTableReleaseToken --
 *
 *	Releases the tuple token, indicating this the client is no
 *	longer using the object. The client is removed from the tuple
 *	object's client list.  If this is the last client, then the
 *	object itself is destroyed and memory is freed.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	If this is the remaining client, then the table object itself
 *	is freed.
 *
 *---------------------------------------------------------------------- 
 */
void
Blt_DataTableReleaseToken(DataTable *dtPtr)
{
    if (dtPtr->magic != DATATABLE_MAGIC) {
	fprintf(stderr, "invalid table object token 0x%lx\n", 
		(unsigned long)dtPtr);
	return;
    }
    DestroyDataTable(dtPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_DataTableExists --
 *
 *	Indicates if a table object by the given name exists in either
 *	the current or global namespace.
 *
 * Results:
 *	Returns 1 if a table object exists and 0 otherwise.
 *
 *---------------------------------------------------------------------- 
 */
int
Blt_DataTableExists(
    Tcl_Interp *interp,		/* Interpreter to report errors back to. */
    CONST char *name)		/* Name of table object in the
				 * designated namespace. */
{
    DataTableInterpData *dataPtr;

    dataPtr = GetDataTableInterpData(interp);
    return (GetDataTable(dataPtr, name, NS_SEARCH_BOTH) != NULL);
}

static Notifier *
CreateNotifier(
    Tcl_Interp *interp,
    DataTable *dtPtr,
    unsigned int mask,
    long offset,
    char *tag,
    Blt_DataTableNotifierEventProc *proc,
    Blt_DataTableNotifierDeleteProc *deleteProc,
    ClientData clientData)
{
    Notifier *np;

    np = Blt_Malloc(sizeof (Notifier));
    assert(np);
    np->proc = proc;
    np->deleteProc = deleteProc;
    np->chainPtr = dtPtr->notifiers;
    np->clientData = clientData;
    np->offset = offset;
    np->tag = tag;
    np->flags = mask;
    np->interp = interp;
    np->linkPtr = Blt_ChainAppend(dtPtr->notifiers, np);
    return np;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_DataTableCreateNotifier --
 *
 *	Creates an event handler using the following three pieces of
 *	information: 
 *		1. C function pointer, 
 *		2. one-word of data passed on each call, and 
 *		3. event mask indicating which events are of interest.  
 *	If an event already exists matching all of the above criteria,
 *	it is repositioned on the end of the event handler list.  This
 *	means that it will be the last to fire.
 *
 * Results:
 *      Returns a pointer to the event handler.
 *
 * Side Effects:
 *	Memory for the event handler is possibly allocated.
 *
 *---------------------------------------------------------------------- 
 */
Blt_DataTableNotifier
Blt_DataTableColumnNotifier(
    Tcl_Interp *interp,
    Blt_DataTable table,
    unsigned int mask,
    long column, char *tag,
    Blt_DataTableNotifierEventProc *proc,
    Blt_DataTableNotifierDeleteProc *deletedProc,
    ClientData clientData)
{
    return CreateNotifier(interp, table, mask | TABLE_NOTIFY_COLUMN, column, 
		tag, proc, deletedProc, clientData);
}

Blt_DataTableNotifier
Blt_DataTableRowNotifier(
    Tcl_Interp *interp,
    Blt_DataTable table,
    unsigned int mask,
    long row, char *tag,
    Blt_DataTableNotifierEventProc *proc,
    Blt_DataTableNotifierDeleteProc *deletedProc,
    ClientData clientData)
{
    return CreateNotifier(interp, table, mask | TABLE_NOTIFY_ROW, row, 
		tag, proc, deletedProc, clientData);
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_DataTableDeleteNotifier --
 *
 *	Removes the event handler designated by following three pieces
 *	of information: 
 *	   1. C function pointer, 
 *	   2. one-word of data passed on each call, and 
 *	   3. event mask indicating which events are of interest.
 *
 * Results:
 *      Nothing.
 *
 * Side Effects:
 *	Memory for the event handler is freed.
 *
 *----------------------------------------------------------------------
 */
void
Blt_DataTableDeleteNotifier(Notifier *np)
{
    /* Check if notifier is already being deleted. */
    if ((np->flags & TABLE_NOTIFY_DESTROYED) == 0) {
	FreeNotifier(np);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_DataTableRowLabel --
 *
 *	Returns the label of the row.  If the row offset is invalid or
 *	the row has no label, then NULL is returned.
 *
 * Results:
 *	Returns the label of the row.
 *
 *----------------------------------------------------------------------
 */
CONST char *
Blt_DataTableRowLabel(Blt_DataTable table, long row)
{
    RowColumn *rcPtr;

    rcPtr = RowData(table);
    if ((row < 0) || (row >= rcPtr->nAllocated)) {
	return NULL;
    }
    return rcPtr->headers[row].label;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_DataTableColumnLabel --
 *
 *	Returns the label of the column.  If the column offset is invalid or
 *	the column has no label, then NULL is returned.
 *
 * Results:
 *	Returns the label of the column.
 *
 *----------------------------------------------------------------------
 */
CONST char *
Blt_DataTableColumnLabel(Blt_DataTable table, long column)
{
    RowColumn *rcPtr;

    rcPtr = ColumnData(table);
    if ((column < 0) || (column >= rcPtr->nAllocated)) {
	return NULL;
    }
    return rcPtr->headers[column].label;
}


/*
 *----------------------------------------------------------------------
 *
 * Blt_DataTableFindRow --
 *
 *	Returns the offset of the row given its label.  If the row
 *	label is invalid, then -1 is returned.
 *
 * Results:
 *	Returns the offset of the row or -1 if not found.
 *
 *----------------------------------------------------------------------
 */
long
Blt_DataTableFindRow(Blt_DataTable table, CONST char *label)
{
    return FindLabel(RowData(table), label);
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_DataTableFindColumn --
 *
 *	Returns the offset of the column given its label.  If the
 *	column label is invalid, then -1 is returned.
 *
 * Results:
 *	Returns the offset of the column or -1 if not found.
 *
 *----------------------------------------------------------------------
 */
long
Blt_DataTableFindColumn(Blt_DataTable table, CONST char *label)
{
    return FindLabel(ColumnData(table), label);
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_DataTableColumnType --
 *
 *	Returns the type of the column.  If the column offset is
 *	invalid or the column has no type, then
 *	TABLE_COLUMN_TYPE_UNKNOWN is returned.
 *
 * Results:
 *	Returns the type of the column.
 *
 *----------------------------------------------------------------------
 */
int
Blt_DataTableColumnType(Blt_DataTable table, long column)
{
    RowColumn *rcPtr;

    rcPtr = ColumnData(table);
    if ((column < 0) || (column >= rcPtr->nAllocated)) {
	return TABLE_COLUMN_TYPE_UNKNOWN;
    }
    return (rcPtr->headers[column].flags & TABLE_COLUMN_TYPE_MASK);
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_DataTableSetRowLabel --
 *
 *	Returns the label of the row.  If the row offset is invalid or
 *	the row has no label, then NULL is returned.
 *
 * Results:
 *	Returns the label of the row.
 *
 *----------------------------------------------------------------------
 */
int
Blt_DataTableSetRowLabel(
    Tcl_Interp *interp, 
    Blt_DataTable table, 
    long row, 
    CONST char *label)
{
    if (SetLabel(interp, RowData(table), row, label) != TCL_OK) {
	return TCL_ERROR;
    }
    NotifyClients(table, row, TABLE_NOTIFY_RELABEL | TABLE_NOTIFY_ROW);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_DataTableSetColumnLabel --
 *
 *	Sets the label of the column.  If the column offset is invalid,
 *	then no label is set.
 *
 * Results:
 *	None.
 *
 *----------------------------------------------------------------------
 */
int
Blt_DataTableSetColumnLabel(
    Tcl_Interp *interp, 
    Blt_DataTable table,
    long column, 
    CONST char *label)
{
    if (SetLabel(interp, ColumnData(table), column, label) != TCL_OK) {
	return TCL_ERROR;
    }
    NotifyClients(table, column, TABLE_NOTIFY_RELABEL | TABLE_NOTIFY_COLUMN);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_DataTableSetColumnType --
 *
 *	Sets the type of the column.  If the column offset is invalid,
 *	then no type is set.
 *
 * Results:
 *	None.
 *
 *----------------------------------------------------------------------
 */
int
Blt_DataTableSetColumnType(
    Tcl_Interp *interp, 
    Blt_DataTable table,
    long column, 
    int type)
{
    RowColumn *rcPtr;

    rcPtr = ColumnData(table);
    if ((column < 0) || (column >= rcPtr->nAllocated)) {
	if (interp != NULL) {
	    Tcl_AppendResult(interp, "bad column \"", Blt_Ltoa(column), "\"", 
			     (char *)NULL);
	}
	return TCL_ERROR;
    }
    if ((type & TABLE_COLUMN_TYPE_MASK) == 0) {
	if (interp != NULL) {
	    Tcl_AppendResult(interp, "bad type value \"", Blt_Ltoa(type), "\"", 
			     (char *)NULL);
	}
	return TCL_ERROR;
    }	
    SetType(rcPtr, column, type);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_DataTableValueExists --
 *
 *	Indicates if a value exists for a given row,column offset in
 *	the tuple.  Note that this routine does not fire read traces.
 *
 * Results:
 *	Returns 1 is a value exists, 0 otherwise.
 *
 *----------------------------------------------------------------------
 */
int
Blt_DataTableValueExists(
    DataTable *dtPtr, 
    long row, long column)
{
    TableObject *tableObjPtr;

    tableObjPtr = dtPtr->tableObjPtr;
    if ((row < 0) || (row >= NumRowsAllocated(dtPtr))) {
	return FALSE;		/* Row doesn't exist. */
    }
    if ((column < 0) || (column >= NumColumnsAllocated(dtPtr))) {
	return FALSE;		/* Column doesn't exist. */
    }
    return (GetValue(tableObjPtr, row, column) != NULL);
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_DataTableArrayValueExists --
 *
 *	Indicates if a value exists for a given row,column offset in
 *	the tuple.  Note that this routine does not fire read traces.
 *
 * Results:
 *	Returns 1 is a value exists, 0 otherwise.
 *
 *----------------------------------------------------------------------
 */
int
Blt_DataTableArrayValueExists(
    Tcl_Interp *interp, 
    DataTable *dtPtr, 
    long row, long column, 
    CONST char *key,
    int *boolPtr)
{
    TableObject *tableObjPtr;
    Tcl_Obj *arrayObjPtr;

    tableObjPtr = dtPtr->tableObjPtr;
    if ((row < 0) || (row >= NumRowsAllocated(dtPtr))) {
	return FALSE;		/* Row doesn't exist. */
    }
    if ((column < 0) || (column >= NumColumnsAllocated(dtPtr))) {
	return FALSE;		/* Column doesn't exist. */
    }
    arrayObjPtr = GetValue(tableObjPtr, row, column);
    if (arrayObjPtr == NULL) {
	return FALSE;		/* No array object allocated yet. */
    }
    {
	Blt_HashEntry *hPtr;
	Blt_HashTable *tablePtr;

	if (Blt_GetArrayFromObj(interp, arrayObjPtr, &tablePtr) != TCL_OK) {
	    return FALSE;	/* Not an array element. */
	}
	hPtr = Blt_FindHashEntry(tablePtr, key);
	if (hPtr == NULL) {
	    return FALSE;	/* Element doesn't exist. */
	}
    }
    return TRUE;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_DataTableExtendRows --
 *
 *	Adds new rows to the table.  Rows are slots in an array
 *	of Rows.  The array grows by doubling its size, so there
 *	may be more slots than needed (# rows).  
 *
 * Results:
 *	Returns TCL_OK is the tuple is resized and TCL_ERROR if an
 *	not enough memory was available.
 *
 * Side Effects:
 *	If more rows are needed, the array which holds the tuples is
 *	reallocated by doubling its size.  
 *
 *----------------------------------------------------------------------
 */
int
Blt_DataTableExtendRows(
    Tcl_Interp *interp, 
    Blt_DataTable table, 
    size_t n, 
    long *rows)
{
    size_t i;

    if (!ExtendRows(table, n, rows)) {
	if (interp != NULL) {
	    Tcl_AppendResult(interp, "can't extend table by ", 
		Blt_Ltoa(n), " rows: out of memory.", (char *)NULL);
	}
	return TCL_ERROR;
    }
    for (i = 0; i < n; i++) {
	NotifyClients(table, rows[i], TABLE_NOTIFY_CREATE | TABLE_NOTIFY_ROW);
    }
    return TCL_OK;
}

int
Blt_DataTableDeleteRow(Tcl_Interp *interp, Blt_DataTable table, long row)
{
    if ((row < 0) || (row >= NumRowsAllocated(table))) {
	Tcl_AppendResult(interp, "bad row \"", Blt_Ltoa(row), "\"", 
			 (char *)NULL);
	return TCL_ERROR;
    }
    if (RowIndex(table, row) > 0) {
	DeleteHeader(RowData(table), row);
	UnsetRowValues(table, row);
	NotifyClients(table, row, TABLE_NOTIFY_DELETE | TABLE_NOTIFY_ROW);
	ClearRowTags(table, row);
	ClearRowTraces(table, row);
	ClearRowNotifiers(table, row);
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_DataTableMoveRows --
 *
 *	Move one of more rows to a new location in the tuple.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
int
Blt_DataTableMoveRows(
    Tcl_Interp *interp,
    Blt_DataTable table,
    long iSrc, long iDest, long count)
{
    if (iSrc == iDest) {
	return TCL_OK;		/* Move to the same location. */
    }
    if (!MoveIndices(RowData(table), iSrc, iDest, count)) {
	Tcl_AppendResult(interp, "can't allocate new map for \"", 
		ClientName(table), "\"", (char *)NULL);
	return TCL_ERROR;
    }
    NotifyClients(table, TABLE_NOTIFY_ANY, TABLE_NOTIFY_MOVE|TABLE_NOTIFY_ROW);
    return TCL_OK;
}

void
Blt_DataTableSetRowMap(Blt_DataTable table, long *map)
{
    ReplaceMap(RowData(table), map);
}

long *
Blt_DataTableSortRows(
    Blt_DataTable table, 
    Blt_DataTableSortOrder *order, 
    long nCompares,
    unsigned int flags)
{
    sortData.table = table;
    sortData.order = order;
    sortData.nCompares = nCompares;
    sortData.flags = flags;
    InitSortProcs(table, order, nCompares);
    return SortHeaders(RowData(table), (QSortCompareProc *)CompareRows);
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_DataTableDeleteColumn --
 *
 *	Remove the designated column from the table.  The actual space
 *	contained by the column isn't freed.  The map is compressed.
 *	Tcl_Objs stored as column values are released.  Traces and
 *	tags associated with the column are removed.
 *
 * Side Effects:
 *	Traces may fire when column values are unset.  Also notifier
 *	events may be triggered, indicating the column has been
 *	deleted.
 *
 *----------------------------------------------------------------------
 */
int
Blt_DataTableDeleteColumn(Tcl_Interp *interp, Blt_DataTable table, long column)
{
    if ((column < 0) || (column >= NumColumnsAllocated(table))) {
	Tcl_AppendResult(interp, "bad column \"", Blt_Ltoa(column), "\"", 
		(char *)NULL);
	return TCL_ERROR;
    }
    /* If the column offset has no index (i.e. -1), the column has
     * already been deleted. */
    if (ColumnIndex(table, column) > 0) {
	DeleteHeader(ColumnData(table), column);
	UnsetColumnValues(table, column);
	
	NotifyClients(table, column, TABLE_NOTIFY_DELETE | TABLE_NOTIFY_COLUMN);
	ClearColumnTraces(table, column);
	ClearColumnTags(table, column);
	ClearColumnNotifiers(table, column);
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_DataTableExtendColumns --
 *
 *	Adds new columns to the table.  Columns are slots in an array
 *	of Columns.  The array columns by doubling its size, so there
 *	may be more slots than needed (# columns).  
 *
 * Results:
 *	Returns TCL_OK is the tuple is resized and TCL_ERROR if an
 *	not enough memory was available.
 *
 * Side Effects:
 *	If more columns are needed, the array which holds the tuples is
 *	reallocated by doubling its size.  
 *
 *----------------------------------------------------------------------
 */
int
Blt_DataTableExtendColumns(
    Tcl_Interp *interp,
    Blt_DataTable table, 
    size_t n, 
    long *cols)
{
    size_t i;

    if (!ExtendColumns(table, n, cols)) {
	if (interp != NULL) {
	    Tcl_AppendResult(interp, "can't extend table by ", 
		Blt_Ltoa(n), " columns: out of memory.", (char *)NULL);
	}
	return TCL_ERROR;
    }
    for (i = 0; i < n; i++) {
	SetType(ColumnData(table), cols[i], TABLE_COLUMN_TYPE_STRING);
	NotifyClients(table, cols[i], TABLE_NOTIFY_CREATE|TABLE_NOTIFY_COLUMN);
    }
    return TCL_OK;
}

void
Blt_DataTableSetColumnMap(Blt_DataTable table, long *map)
{
    ReplaceMap(ColumnData(table), map);
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_DataTableMoveColumns --
 *
 *	Move one of more rows to a new location in the tuple.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
int
Blt_DataTableMoveColumns(
    Tcl_Interp *interp,
    Blt_DataTable table,
    long iSrc, long iDest, long count)
{
    if (iSrc == iDest) {
	return TCL_OK;		/* Move to the same location. */
    }
    if (!MoveIndices(ColumnData(table), iSrc, iDest, count)) {
	Tcl_AppendResult(interp, "can't allocate new map for \"", 
		ClientName(table), "\"", (char *)NULL);
	return TCL_ERROR;
    }
    NotifyClients(table, TABLE_NOTIFY_ANY, 
		  TABLE_NOTIFY_MOVE|TABLE_NOTIFY_COLUMN);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_DataTableRestore --
 *
 *	Restores data to the given table based upon the dump string.
 *	The dump string should have been generated by Blt_DataTableDump.
 *	Two bit flags may be set.
 *	
 *	TABLE_RESTORE_NO_TAGS	Don't restore tag information.
 *	TABLE_RESTORE_OVERWRITE	Look for row and columns with the 
 *				same label. Overwrite if necessary.
 *
 * Results:
 *	A standard Tcl result.  If the restore was successful, TCL_OK
 *	is returned.  Otherwise, TCL_ERROR is returned and an error
 *	message is left in the interpreter result.
 *
 * Side Effects:
 *	New row and columns are created in the table and may possibly
 *	generate event notifier or trace callbacks.
 *
 *---------------------------------------------------------------------- 
 */
int
Blt_DataTableRestore(
    Tcl_Interp *interp,
    Blt_DataTable table,
    char *string,
    unsigned int flags)
{
    RestoreData restore;
    int result;

    restore.argc = 0;
    restore.mtime = restore.ctime = 0L;
    restore.argv = NULL;
    restore.fileName = "data string";
    restore.nLines = 0;
    restore.flags = flags;
    restore.nCols = NumColumns(table);
    restore.nRows = NumRows(table);
    Blt_InitHashTableWithPool(&restore.rowIndices, BLT_ONE_WORD_KEYS);
    Blt_InitHashTableWithPool(&restore.colIndices, BLT_ONE_WORD_KEYS);
    result = TCL_ERROR;		
    /* Read dump information */
    for (;;) {
	char c1, c2;

	result = ParseDumpRecord(interp, &string, &restore);
	if (result != TCL_OK) {
	    break;
	}
	if (restore.argc == 0) {
	    continue;
	}
	c1 = restore.argv[0][0], c2 = restore.argv[0][1];
	if ((c1 == 'i') && (c2 == '\0')) {
	    result = RestoreHeader(interp, table, &restore);
	} else if ((c1 == 'r') && (c2 == '\0')) {
	    result = RestoreRow(interp, table, &restore);
	} else if ((c1 == 'c') && (c2 == '\0')) {
	    result = RestoreColumn(interp, table, &restore);
	} else if ((c1 == 'd') && (c2 == '\0')) {
	    result = RestoreValue(interp, table, &restore);
	} else {
	    Tcl_AppendResult(interp, restore.fileName, ":", 
		Blt_Ltoa(restore.nLines), ": error: unknown entry \"", 
		restore.argv[0], "\"", (char *)NULL);
	    result = TCL_ERROR;
	}
	Blt_Free(restore.argv);
	if (result != TCL_OK) {
	    break;
	}
    }
    Blt_DeleteHashTable(&restore.rowIndices);
    Blt_DeleteHashTable(&restore.colIndices);
    if (result == TCL_ERROR) {
	return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_DataTableFileRestore --
 *
 *	Restores data to the given table based upon the dump file
 *	provided. The dump file should have been generated by
 *	Blt_DataTableDump or Blt_DataTableFileDump.  
 *
 *	If the filename starts with an '@', then it is the name of an
 *	already opened channel to be used. Two bit flags may be set.
 *	
 *	TABLE_RESTORE_NO_TAGS	Don't restore tag information.
 *	TABLE_RESTORE_OVERWRITE	Look for row and columns with 
 *				the same label. Overwrite if necessary.
 *
 * Results:
 *	A standard Tcl result.  If the restore was successful, TCL_OK
 *	is returned.  Otherwise, TCL_ERROR is returned and an error
 *	message is left in the interpreter result.
 *
 * Side Effects:
 *	Row and columns are created in the table and may possibly
 *	generate trace or notifier event callbacks.
 *
 *---------------------------------------------------------------------- 
 */
int
Blt_DataTableFileRestore(
    Tcl_Interp *interp,
    Blt_DataTable table,
    char *fileName,
    unsigned int flags)
{
    Tcl_Channel channel;
    RestoreData restore;
    int closeChannel;
    int result;

    closeChannel = TRUE;
    if ((fileName[0] == '@') && (fileName[1] != '\0')) {
	int mode;
	
	channel = Tcl_GetChannel(interp, fileName+1, &mode);
	if (channel == NULL) {
	    return TCL_ERROR;
	}
	if ((mode & TCL_READABLE) == 0) {
	    Tcl_AppendResult(interp, "channel \"", fileName, 
		"\" not opened for reading", (char *)NULL);
	    return TCL_ERROR;
	}
	closeChannel = FALSE;
    } else {
	channel = Tcl_OpenFileChannel(interp, fileName, "r", 0);
	if (channel == NULL) {
	    return TCL_ERROR;	/* Can't open dump file. */
	}
    }
    restore.argc = 0;
    restore.mtime = restore.ctime = 0L;
    restore.argv = NULL;
    restore.fileName = fileName;
    restore.nLines = 0;
    restore.flags = flags;
    restore.nCols = NumColumns(table);
    restore.nRows = NumRows(table);
    Blt_InitHashTableWithPool(&restore.rowIndices, BLT_ONE_WORD_KEYS);
    Blt_InitHashTableWithPool(&restore.colIndices, BLT_ONE_WORD_KEYS);

    /* Process dump information record by record. */
    result = TCL_ERROR;		
    for (;;) {
	char c1, c2;

	result = ReadDumpRecord(interp, channel, &restore);
	if (result != TCL_OK) {
	    break;
	}
	if (restore.argc == 0) {
	    continue;
	}
	c1 = restore.argv[0][0], c2 = restore.argv[0][1];
	if ((c1 == 'i') && (c2 == '\0')) {
	    result = RestoreHeader(interp, table, &restore);
	} else if ((c1 == 'r') && (c2 == '\0')) {
	    result = RestoreRow(interp, table, &restore);
	} else if ((c1 == 'c') && (c2 == '\0')) {
	    result = RestoreColumn(interp, table, &restore);
	} else if ((c1 == 'd') && (c2 == '\0')) {
	    result = RestoreValue(interp, table, &restore);
	} else {
	    Tcl_AppendResult(interp, fileName, ":", Blt_Ltoa(restore.nLines), 
		": error: unknown entry \"", restore.argv[0], "\"", 
		(char *)NULL);
	    result = TCL_ERROR;
	}
	Blt_Free(restore.argv);
	if (result != TCL_OK) {
	    break;
	}
    }
    Blt_DeleteHashTable(&restore.rowIndices);
    Blt_DeleteHashTable(&restore.colIndices);
    if (result == TCL_ERROR) {
	return TCL_ERROR;
    }
    return TCL_OK;
}


#ifdef notdef

typedef struct {
    Tcl_Interp *interp;
    Blt_DataTable table;
    long curRow;		/* Current row. */
    
} FindInfo;

/*
 *----------------------------------------------------------------------
 *
 * NewVar --
 *
 *	Create a new heap-allocated variable that will eventually be
 *	entered into a hashtable.
 *
 * Results:
 *	The return value is a pointer to the new variable structure. It is
 *	marked as a scalar variable (and not a link or array variable). Its
 *	value initially is NULL. The variable is not part of any hash table
 *	yet. Since it will be in a hashtable and not in a call frame, its
 *	name field is set NULL. It is initially marked as undefined.
 *
 * Side effects:
 *	Storage gets allocated.
 *
 *----------------------------------------------------------------------
 */

static Var *
NewVar(CONST char *label, Tcl_Obj *objPtr)
{
    Var *varPtr;

    varPtr = Blt_Malloc(sizeof(Var));
    varPtr->value.objPtr = objPtr;
    if (objPtr != NULL) {
	Tcl_IncrRefCount(objPtr);
    }
    varPtr->name = label;
    varPtr->refCount = 1;  /* protect from being deleted */
    /*
     *  NOTE:  Tcl reports a "dangling upvar" error for variables
     *         with a null "hPtr" field.  Put something non-zero
     *         in here to keep Tcl_SetVar2() happy.  The only time
     *         this field is really used is it remove a variable
     *         from the hash table that contains it in CleanupVar,
     *         but since these variables are protected by their
     *         higher refCount, they will not be deleted by CleanupVar
     *         anyway.  These variables are unset and removed in
     *         ItclFreeObject().
     */
    varPtr->hPtr = 0x01;
    varPtr->nsPtr = NULL;
    varPtr->tracePtr = NULL;
    varPtr->searchPtr = NULL;
    varPtr->flags = (VAR_SCALAR | VAR_IN_HASHTABLE);
    return varPtr;
}

static Var *
GetVar(FindInfo *findPtr, CONST char *label, Tcl_Obj *objPtr)
{
    Blt_HashEntry *hPtr;
    int isNew;
    Var *varPtr;

    hPtr = Blt_CreateHashEntry(&findPtr->varTable, label, &isNew);
    if (isNew) {
	varPtr = NewVar(label, objPtr);
	Blt_SetHashValue(varPtr);
    } else {
	varPtr = Blt_GetHashValue(varPtr);
	varPtr->value.objPtr = objPtr;
    }
    return varPtr;
}

static void
FreeInfo(FindInfo *findPtr)
{
    Blt_HashEntry *hPtr;
    Blt_HashSearch cursor;

    for (hPtr = Blt_FirstHashEntry(&findPtr->varTable, &cursor); hPtr != NULL; 
	 hPtr = Blt_FindNextHashEntry(&cursor)) {
	varPtr = Blt_GetHashValue(hPtr);
	varPtr->refCount--;
	if (varPtr->refcount > 1) {
	    Tcl_DecrRefCount(varPtr->objPtr);
	    Blt_Free(varPtr);
	}
    }
    Blt_DeleteHashTable(&findPtr->varTable);
}

static int
ColumnFindVarResolver(
    Tcl_Interp *interp,		/* current interpreter */
    CONST char *name,		/* Variable name being resolved */
    Tcl_Namespace *nsPtr,	/* Current namespace context */
    int flags,			/* TCL_LEAVE_ERR_MSG => leave error message */
    Tcl_Var *rPtr)		/* (out) Resolved variable. */ 
{
    FindInfo *findPtr;
    Tcl_HashEntry *hPtr;
    Tcl_CallFrame *framePtr;
    Blt_DataTable table;

    framePtr = Tcl_GetCurrentCallVarFrame(interp);
    hPtr = Blt_FindHashEntry(&findInfoTable, framePtr);
    if (hPtr == NULL) {
	return TCL_CONTINUE;
    }
    findPtr = Blt_GetHashValue(hPtr);
    table = findPtr->table;
    if (TclGetLong((Tcl_Interp *)NULL, name, &n) == TCL_OK) {
	if ((n < 1) || (n > NumColumns(table))) {
	    if (flags & TCL_LEAVE_ERR_MSG) {
		Tcl_AppendResult(interp, "bad column index \"", name,
			"\"", (char *)NULL);
	    }
	    return TCL_ERROR;
	}		
	col = ColumnOffset(table, n);
    } else if ((col = Blt_DataTableFindColumn(table, name)) >= 0) {
	n = ColumnIndex(table, col);
    } else {
	return TCL_CONTINUE;
    }
 resolve:
    {
	Tcl_Obj *valueObjPtr;
	int result;
	Tcl_Interp *errInterp;

	if (flags & TCL_LEAVE_ERR_MSG) {
	    errInterp = interp;
	} else {
	    errInterp = NULL;
	}
	result = Blt_DataTableGetValue(errInterp, table, findPtr->curRow, col, 
				       &valueObjPtr);
	if (result != TCL_OK) {
	    return TCL_ERROR;	/* No such table location. */
	}
	if (valueObjPtr == NULL) {
	    return TCL_ERROR;	/* Empty value. */
	}
	*rPtr = GetVar(findPtr, name, valueObjPtr);
    }
    return TCL_OK;
}

static void
DoColumnExpr(Blt_DataTable table, char *expr)
{
    Tcl_CallFrame frame;
    FindInfo find;

    /*
     *  Install a variable resolution procedure to handle row, column
     *  values elsewhere within the interpreter.
     */
    Tcl_AddInterpResolvers(interp, ClientName(table), 
	(Tcl_ResolveCmdProc*)NULL, ColumnFindVarResolver, 
	(Tcl_ResolveCompiledVarProc*)NULL);

    find.curRow = 0;
    Blt_HashTableInit(&find.varTable, BLT_ONE_WORD_KEYS);
    
    hPtr = Blt_CreateHashEntry(&findTable, &frame, &isNew);
    assert(!isNew);
    Blt_SetHashValue(hPtr, &find);
    Tcl_PushCallFrame(interp, &frame, Tcl_GetCurrentNamespace(interp),
        /* isProcCallFrame */ FALSE);
    /* compile expression into tokens */
    for (iRow = 1; iRow <= NumRows(table); iRow++) {
	find.curRow = RowIndex(table, iRow);
	/* Execute tokens. */
    }
    Tcl_PopCallFrame(interp);
    FreeFindInfo(&find);
    Tcl_RemoveInterpResolvers(interp, ClientName(table));
}

#endif
