
/*
 * bltDataTable.h --
 *
 *	Copyright 1998-2004 George A Howlett.
 *
 *	Permission is hereby granted, free of charge, to any person
 *	obtaining a copy of this software and associated documentation
 *	files (the "Software"), to deal in the Software without
 *	restriction, including without limitation the rights to use,
 *	copy, modify, merge, publish, distribute, sublicense, and/or
 *	sell copies of the Software, and to permit persons to whom the
 *	Software is furnished to do so, subject to the following
 *	conditions:
 *
 *	The above copyright notice and this permission notice shall be
 *	included in all copies or substantial portions of the
 *	Software.
 *
 *	THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY
 *	KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 *	WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 *	PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS
 *	OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
 *	OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 *	OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 *	SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#ifndef _BLT_DATATABLE_H
#define _BLT_DATATABLE_H

#include <bltChain.h>
#include <bltHash.h>

typedef struct Blt_DataTableStruct *Blt_DataTable;
typedef struct Blt_DataTableIteratorStruct Blt_DataTableIterator;
typedef struct Blt_DataTableNotifierStruct *Blt_DataTableNotifier;
typedef struct Blt_DataTableTagDataTableStruct Blt_DataTableTagTable;
typedef struct Blt_DataTableTraceStruct *Blt_DataTableTrace;

EXTERN void Blt_DataTableReleaseTags(Blt_DataTable table);

EXTERN int Blt_DataTableExists(Tcl_Interp *interp, CONST char *name);
EXTERN int Blt_DataTableCreate(Tcl_Interp *interp, CONST char *name,
	Blt_DataTable *tablePtr);
EXTERN int Blt_DataTableGetToken(Tcl_Interp *interp, CONST char *name, 
	Blt_DataTable *tablePtr);
EXTERN void Blt_DataTableReleaseToken(Blt_DataTable table);

EXTERN int Blt_DataTableSameTableObject(Blt_DataTable table1, 
	Blt_DataTable table2);

EXTERN CONST char *Blt_DataTableName(Blt_DataTable table);
EXTERN long Blt_DataTableNumRows(Blt_DataTable table);
EXTERN long Blt_DataTableNumColumns(Blt_DataTable table);
EXTERN long Blt_DataTableRowOffset(Blt_DataTable table, long index);
EXTERN long Blt_DataTableColumnOffset(Blt_DataTable table, long index);
EXTERN long Blt_DataTableRowIndex(Blt_DataTable table, long offset);
EXTERN long Blt_DataTableColumnIndex(Blt_DataTable table, long offset);

EXTERN long Blt_DataTableFindRow(Blt_DataTable table, CONST char *label);
EXTERN long Blt_DataTableFindColumn(Blt_DataTable table, CONST char *label);
EXTERN CONST char *Blt_DataTableRowLabel(Blt_DataTable table, long row);
EXTERN CONST char *Blt_DataTableColumnLabel(Blt_DataTable table, long column);
EXTERN int Blt_DataTableSetRowLabel(Tcl_Interp *interp, Blt_DataTable table, 
	long row, CONST char *label);
EXTERN int Blt_DataTableSetColumnLabel(Tcl_Interp *interp, Blt_DataTable table, 
	long column, CONST char *label);

EXTERN int Blt_DataTableParseColumnType(CONST char *typeName);

EXTERN CONST char *Blt_DataTableNameOfColumnType(int type);

EXTERN int Blt_DataTableColumnType(Blt_DataTable table, long column);
EXTERN int Blt_DataTableSetColumnType(Tcl_Interp *interp, Blt_DataTable table, 
	long column, int type);

EXTERN int Blt_DataTableSetColumnTag(Tcl_Interp *interp, Blt_DataTable table, 
	long column, CONST char *tagName);
EXTERN int Blt_DataTableSetRowTag(Tcl_Interp *interp, Blt_DataTable table, 
	long row, CONST char *tagName);

EXTERN int Blt_DataTableExtendRows(Tcl_Interp *interp, Blt_DataTable table, 
	size_t n, long *rows);
EXTERN int Blt_DataTableExtendColumns(Tcl_Interp *interp, Blt_DataTable table, 
	size_t n, long *columms);
EXTERN int Blt_DataTableDeleteRow(Tcl_Interp *interp, Blt_DataTable table, 
	long row);
EXTERN int Blt_DataTableDeleteColumn(Tcl_Interp *interp, Blt_DataTable table, 
	long column);
EXTERN int Blt_DataTableMoveRows(Tcl_Interp *interp, Blt_DataTable table, 
	long from, long to, long n);
EXTERN int Blt_DataTableMoveColumns(Tcl_Interp *interp, Blt_DataTable table, 
	long from, long to, long n);


EXTERN int Blt_DataTableGetValue(Tcl_Interp *interp, Blt_DataTable table, 
	long row,  long column, Tcl_Obj **objPtrPtr);
EXTERN int Blt_DataTableSetValue(Tcl_Interp *interp, Blt_DataTable table, 
	long row,  long column, Tcl_Obj *objPtr);
EXTERN int Blt_DataTableUnsetValue(Tcl_Interp *interp, Blt_DataTable table, 
	long row,  long column);
EXTERN int Blt_DataTableValueExists(Blt_DataTable table, long row, long column);
EXTERN int Blt_DataTableGetArrayValue(Tcl_Interp *interp, Blt_DataTable table, 
	long row,  long column, CONST char *key, Tcl_Obj **objPtrPtr);
EXTERN int Blt_DataTableSetArrayValue(Tcl_Interp *interp, Blt_DataTable table, 
	long row,  long column, CONST char *key, Tcl_Obj *objPtr);
EXTERN int Blt_DataTableUnsetArrayValue(Tcl_Interp *interp, 
	Blt_DataTable table, long row,  long column, CONST char *key);
EXTERN int Blt_DataTableArrayValueExists(Tcl_Interp *interp, 
	Blt_DataTable table, long row,  long column, CONST char *key, 
	int *booleanPtr);

EXTERN Blt_HashTable *Blt_DataTableRowTagTable(Blt_DataTable table, 
	CONST char *tagName);
EXTERN Blt_HashTable *Blt_DataTableColumnTagTable(Blt_DataTable table, 
	CONST char *tagName);
EXTERN Blt_Chain *Blt_DataTableRowTags(Blt_DataTable table, long row);
EXTERN Blt_Chain *Blt_DataTableColumnTags(Blt_DataTable table, long column);

EXTERN int Blt_DataTableHasRowTag(Blt_DataTable table, long column, 
	CONST char *tagName);
EXTERN int Blt_DataTableHasColumnTag(Blt_DataTable table, long column, 
	CONST char *tagName);
EXTERN void Blt_DataTableAddColumnTag(Blt_DataTable table, long column, 
	CONST char *tagName);
EXTERN void Blt_DataTableAddRowTag(Blt_DataTable table, long row, 
	CONST char *tagName);
EXTERN int Blt_DataTableForgetRowTag(Tcl_Interp *interp, Blt_DataTable table, 
	CONST char *tagName);
EXTERN int Blt_DataTableForgetColumnTag(Tcl_Interp *interp, Blt_DataTable table,
	CONST char *tagName);
EXTERN int Blt_DataTableUnsetRowTag(Tcl_Interp *interp, Blt_DataTable table, 
	long row, CONST char *tagName);
EXTERN int Blt_DataTableUnsetColumnTag(Tcl_Interp *interp, Blt_DataTable table,
	long column, CONST char *tagName);
EXTERN Blt_HashEntry *Blt_DataTableFirstRowTag(Blt_DataTable table, 
	Blt_HashSearch *cursorPtr);
EXTERN Blt_HashEntry *Blt_DataTableFirstColumnTag(Blt_DataTable table, 
	Blt_HashSearch *cursorPtr);

EXTERN Blt_HashTable *Blt_DataTableTagHashTable(Blt_DataTable table, 
	CONST char *tagName);
EXTERN int Blt_DataTableTagTableIsShared(Blt_DataTable table);

#define TABLE_COLUMN_TYPE_UNKNOWN	(0) 
#define TABLE_COLUMN_TYPE_STRING	(1<<0)
#define TABLE_COLUMN_TYPE_INTEGER	(1<<1)
#define TABLE_COLUMN_TYPE_DOUBLE	(1<<2)
#define TABLE_COLUMN_TYPE_IMAGE	(1<<3)
#define TABLE_COLUMN_TYPE_MASK	\
	(TABLE_COLUMN_TYPE_INTEGER | TABLE_COLUMN_TYPE_IMAGE | \
	 TABLE_COLUMN_TYPE_DOUBLE | TABLE_COLUMN_TYPE_STRING)

/*
 * Blt_DataTableIterator --
 *
 *	Structure representing a trace used by a client of the table.
 *
 *	Table rows and columns may be tagged with strings.  A row may
 *	have many tags.  The same tag may be used for many rows.  Tags
 *	are used and stored by clients of a table.  Tags can also be
 *	shared between clients of the same table.
 *	
 *	Both rowTable and columnTable are hash tables keyed by the
 *	physical row or column location in the table respectively.
 *	This is not the same as the client's view (the order of rows
 *	or columns as seen by the client).  This is so that clients
 *	(which may have different views) can share tags without
 *	sharing the same view.
 */

typedef enum { 
    TABLE_ITER_INDEX, TABLE_ITER_RANGE, TABLE_ITER_ALL, TABLE_ITER_TAG, 
} Blt_DataTableTagIterType;

struct Blt_DataTableIteratorStruct {
    Blt_DataTable table;	/* Table that we're iterating over. */

    Blt_DataTableTagIterType type; /* Type of iteration:
				 * TABLE_ITER_TAG	by row or column tag.
				 * TABLE_ITER_ALL	by every row or column.
				 * TABLE_ITER_INDEX	single item: either 
				 *			label or index.
				 * TABLE_ITER_RANGE	over a consecutive 
				 *			range of indices.
				 */

    long iStart;		/* Starting index.  Starting point of
				 * search, saved if iterator is
				 * reused.  Used for TABLE_ITER_ALL and
				 * TABLE_ITER_INDEX searches. */
    long iEnd;			/* Ending index (inclusive). */

    long iNext;			/* Next index. */

    /* For tag-based searches. */
    CONST char *tagName;	/* If non-NULL, is the tag that we are
				 * currently iterating over. */

    Blt_HashTable *tablePtr;	/* Pointer to tag hash table. */
    Blt_HashSearch cursor;	/* Search iterator for tag hash table. */

};

EXTERN int Blt_DataTableGetRows(Tcl_Interp *interp, Blt_DataTable table, 
	Tcl_Obj *objPtr, Blt_DataTableIterator *iter);

EXTERN int Blt_DataTableGetColumns(Tcl_Interp *interp, Blt_DataTable table,
	Tcl_Obj *objPtr, Blt_DataTableIterator *iter);

EXTERN long Blt_DataTableFirstRow(Blt_DataTableIterator *iter);

EXTERN long Blt_DataTableFirstColumn(Blt_DataTableIterator *iter);

EXTERN long Blt_DataTableNextRow(Blt_DataTableIterator *iter);

EXTERN long Blt_DataTableNextColumn(Blt_DataTableIterator *iter);

EXTERN int Blt_DataTableGetRow(Tcl_Interp *interp, Blt_DataTable table, 
	Tcl_Obj *objPtr, long *rowPtr);

EXTERN int Blt_DataTableGetColumn(Tcl_Interp *interp, Blt_DataTable table, 
	Tcl_Obj *objPtr, long *columnPtr);

/*
 * Blt_DataTableTrace --
 *
 *	Structure representing a trace used by a client of the table.
 *
 *	Table rows and columns may be tagged with strings.  A row may
 *	have many tags.  The same tag may be used for many rows.  Tags
 *	are used and stored by clients of a table.  Tags can also be
 *	shared between clients of the same table.
 *	
 *	Both rowTable and columnTable are hash tables keyed by the
 *	physical row or column location in the table respectively.
 *	This is not the same as the client's view (the order of rows
 *	or columns as seen by the client).  This is so that clients
 *	(which may have different views) can share tags without
 *	sharing the same view.
 */

typedef int (Blt_DataTableTraceProc)(ClientData clientData, Tcl_Interp *interp, 
	long row, long column, unsigned int flags);

typedef void (Blt_DataTableTraceDeleteProc)(ClientData clientData);

struct Blt_DataTableTraceStruct {
    unsigned int flags;
    char *rowTag, *colTag;
    long row, column;
    Blt_DataTableTraceProc *proc;
    Blt_DataTableTraceDeleteProc *deleteProc;
    ClientData clientData;
    Blt_HashTable activeTable;	/* Table of cells currently active in
				 * this trace. */
    Blt_Chain *chainPtr;
    Blt_ChainLink *linkPtr;
};

#define TABLE_TRACE_UNSETS	(1<<3)
#define TABLE_TRACE_WRITES	(1<<4)
#define TABLE_TRACE_READS	(1<<5)
#define TABLE_TRACE_CREATES	(1<<6)
#define TABLE_TRACE_ALL		(TABLE_TRACE_UNSETS | TABLE_TRACE_WRITES | \
				 TABLE_TRACE_READS  | TABLE_TRACE_CREATES)
#define TABLE_TRACE_MASK	(TRACE_ALL)

#define TABLE_TRACE_FOREIGN_ONLY (1<<8)
#define TABLE_TRACE_ACTIVE	(1<<9)

EXTERN void Blt_DataTableClearRowTags(Blt_DataTable table, long row);

EXTERN void Blt_DataTableClearColumnTags(Blt_DataTable table, long column);

EXTERN void Blt_DataTableClearRowTraces(Blt_DataTable table, long row);

EXTERN void Blt_DataTableClearColumnTraces(Blt_DataTable table, long column);

EXTERN Blt_DataTableTrace Blt_DataTableCreateTrace(Blt_DataTable table, 
	long row, long column, char *rowTag, char *columnTag, 
	unsigned int mask, Blt_DataTableTraceProc *proc, 
	Blt_DataTableTraceDeleteProc *deleteProc, ClientData clientData);

EXTERN void Blt_DataTableDeleteTrace(Blt_DataTableTrace trace);

/*
 * Blt_DataTableNotifierEvent --
 *
 *	Structure representing a trace used by a client of the table.
 *
 *	Table rows and columns may be tagged with strings.  A row may
 *	have many tags.  The same tag may be used for many rows.  Tags
 *	are used and stored by clients of a table.  Tags can also be
 *	shared between clients of the same table.
 *	
 *	Both rowTable and columnTable are hash tables keyed by the
 *	physical row or column location in the table respectively.
 *	This is not the same as the client's view (the order of rows
 *	or columns as seen by the client).  This is so that clients
 *	(which may have different views) can share tags without
 *	sharing the same view.
 */
typedef struct {
    Tcl_Interp *interp;		/* Interpreter to report to */
    Blt_DataTable table;	/* Table object client that received
				 * the event. */
    long offset;		/* Row or column offset. */
    int self;			/* Indicates if this table client
				 * generated the event. */
    int type;			/* Indicates type of event received. */
} Blt_DataTableNotifierEvent;

typedef int (Blt_DataTableNotifierEventProc)(ClientData clientData, 
	Blt_DataTableNotifierEvent *eventPtr);
typedef void (Blt_DataTableNotifierDeleteProc)(ClientData clientData);

struct Blt_DataTableNotifierStruct {
    Blt_DataTable table;
    Blt_ChainLink *linkPtr;
    Blt_Chain *chainPtr;
    Blt_DataTableNotifierEvent event;
    Blt_DataTableNotifierEventProc *proc;
    Blt_DataTableNotifierDeleteProc *deleteProc;
    ClientData clientData;
    Tcl_Interp *interp;
    long offset;
    char *tag;
    unsigned int flags;
};

#define TABLE_NOTIFY_CREATE	(1<<0)
#define TABLE_NOTIFY_DELETE	(1<<1)
#define TABLE_NOTIFY_RELABEL	(1<<2)
#define TABLE_NOTIFY_MOVE	(1<<3)
#define TABLE_NOTIFY_SORT	(1<<4)
#define TABLE_NOTIFY_PACK	(1<<5)
#define TABLE_NOTIFY_ALL	\
	 (TABLE_NOTIFY_CREATE | TABLE_NOTIFY_DELETE | \
	  TABLE_NOTIFY_RELABEL | TABLE_NOTIFY_MOVE | TABLE_NOTIFY_SORT)

#define TABLE_NOTIFY_ROW	(1<<6)
#define TABLE_NOTIFY_COLUMN	(0)
#define TABLE_NOTIFY_TYPE_BIT	(1<<6)

#define TABLE_NOTIFY_EVENT_MASK TABLE_NOTIFY_ALL
#define TABLE_NOTIFY_MASK	(TABLE_NOTIFY_ALL | TABLE_NOTIFY_TYPE_BIT)

#define TABLE_NOTIFY_WHENIDLE		(1<<10)
#define TABLE_NOTIFY_FOREIGN_ONLY	(1<<11)
#define TABLE_NOTIFY_PENDING		(1<<12)
#define TABLE_NOTIFY_ACTIVE		(1<<13)
#define TABLE_NOTIFY_DESTROYED		(1<<14)

EXTERN Blt_DataTableNotifier Blt_DataTableRowNotifier(Tcl_Interp *interp, 
	Blt_DataTable table, unsigned int mask, long row, char *tag, 
	Blt_DataTableNotifierEventProc *proc, 
	Blt_DataTableNotifierDeleteProc *deleteProc, ClientData clientData);

EXTERN Blt_DataTableNotifier Blt_DataTableColumnNotifier(Tcl_Interp *interp, 
	Blt_DataTable table, unsigned int mask, long column, char *tag, 
	Blt_DataTableNotifierEventProc *proc, 
	Blt_DataTableNotifierDeleteProc *deleteProc, ClientData clientData);

EXTERN void Blt_DataTableDeleteNotifier(Blt_DataTableNotifier notifier);

/*
 * Blt_DataTableSortOrder --
 *
 *	Structure representing a trace used by a client of the table.
 *
 *	Table rows and columns may be tagged with strings.  A row may
 *	have many tags.  The same tag may be used for many rows.  Tags
 *	are used and stored by clients of a table.  Tags can also be
 *	shared between clients of the same table.
 *	
 *	Both rowTable and columnTable are hash tables keyed by the
 *	physical row or column location in the table respectively.
 *	This is not the same as the client's view (the order of rows
 *	or columns as seen by the client).  This is so that clients
 *	(which may have different views) can share tags without
 *	sharing the same view.
 */

typedef int (Blt_DataTableSortProc) _ANSI_ARGS_((ClientData clientData, 
	Tcl_Obj *valueObjPtr1, Tcl_Obj *valueObjPtr2));

typedef struct {
    int type;
    Blt_DataTableSortProc *proc; /* Procedures to be called to compare two
				 * entries in the same row or column. */
    ClientData clientData;	/* One word of data passed to the sort
				 * comparison procedure above. */
    long offset;		/* Row or column to be compared. */
} Blt_DataTableSortOrder;

#define TABLE_SORT_NONE		0
#define TABLE_SORT_INTEGER	(1<<0)
#define TABLE_SORT_DOUBLE	(1<<1)
#define TABLE_SORT_DICTIONARY   (1<<2)
#define TABLE_SORT_ASCII	(1<<3)
#define TABLE_SORT_CUSTOM	(1<<4)
#define TABLE_SORT_DECREASING	(1<<6)

EXTERN long *Blt_DataTableSortRows(Blt_DataTable table, 
	Blt_DataTableSortOrder *order, long nCompares, unsigned int flags);

EXTERN long *Blt_DataTableRowMap(Blt_DataTable table);
EXTERN long *Blt_DataTableColumnMap(Blt_DataTable table);

EXTERN void Blt_DataTableSetRowMap(Blt_DataTable table, long *map);
EXTERN void Blt_DataTableSetColumnMap(Blt_DataTable table, long *map);

#define TABLE_RESTORE_NO_TAGS	    (1<<0)
#define TABLE_RESTORE_OVERWRITE	    (1<<1)

EXTERN int Blt_DataTableRestore(Tcl_Interp *interp, Blt_DataTable table, 
	char *string, unsigned int flags);
EXTERN int Blt_DataTableFileRestore(Tcl_Interp *interp, Blt_DataTable table, 
	char *fileName, unsigned int flags);
EXTERN int Blt_DataTableDump(Tcl_Interp *interp, Blt_DataTable table, 
	long *rowMap, long *colMap, Tcl_DString *dsPtr);
EXTERN int Blt_DataTableFileDump(Tcl_Interp *interp, Blt_DataTable table, 
	long *rowMap, long *colMap, CONST char *fileName);

#endif /* BLT_TABLE_H */
