
/*
 * bltGrMarker.c --
 *
 * This module implements markers for the BLT graph widget.
 *
 *	Copyright 1993-2004 George A Howlett.
 *
 *	Permission is hereby granted, free of charge, to any person
 *	obtaining a copy of this software and associated documentation
 *	files (the "Software"), to deal in the Software without
 *	restriction, including without limitation the rights to use,
 *	copy, modify, merge, publish, distribute, sublicense, and/or
 *	sell copies of the Software, and to permit persons to whom the
 *	Software is furnished to do so, subject to the following
 *	conditions:
 *
 *	The above copyright notice and this permission notice shall be
 *	included in all copies or substantial portions of the
 *	Software.
 *
 *	THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY
 *	KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 *	WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 *	PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS
 *	OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
 *	OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 *	OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 *	SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include "bltGraph.h"
#include "bltImage.h"
#include "bltPicture.h"
#include "bltPainter.h"
#include "bltChain.h"
#include "bltGrElem.h"
#include "bltBitmap.h"

#define GETBITMAP(b) \
	(((b)->destBitmap == None) ? (b)->srcBitmap : (b)->destBitmap)

#define MAX_OUTLINE_POINTS	12

/* Map graph coordinates to normalized coordinates [0..1] */
#define NORMALIZE(A,x) 	(((x) - (A)->axisRange.min) / (A)->axisRange.range)

#define DEF_MARKER_ANCHOR	"center"
#define DEF_MARKER_BACKGROUND	RGB_WHITE
#define DEF_MARKER_BITMAP	(char *)NULL
#define DEF_MARKER_CAP_STYLE	"butt"
#define DEF_MARKER_COORDS	(char *)NULL
#define DEF_MARKER_DASHES	(char *)NULL
#define DEF_MARKER_DASH_OFFSET	"0"
#define DEF_MARKER_ELEMENT	(char *)NULL
#define DEF_MARKER_FOREGROUND	RGB_BLACK
#define DEF_MARKER_FILL_COLOR	RGB_RED
#define DEF_MARKER_FONT		STD_FONT
#define DEF_MARKER_GAP_COLOR	RGB_PINK
#define DEF_MARKER_HEIGHT	"0"
#define DEF_MARKER_HIDE		"no"
#define DEF_MARKER_JOIN_STYLE	"miter"
#define DEF_MARKER_JUSTIFY	"left"
#define DEF_MARKER_LINE_WIDTH	"1"
#define DEF_MARKER_MAP_X	"x"
#define DEF_MARKER_MAP_Y	"y"
#define DEF_MARKER_NAME		(char *)NULL
#define DEF_MARKER_OUTLINE_COLOR RGB_BLACK
#define DEF_MARKER_PAD		"4"
#define DEF_MARKER_ANGLE	"0.0"
#define DEF_MARKER_SCALE	"1.0"
#define DEF_MARKER_STATE	"normal"
#define DEF_MARKER_STIPPLE	(char *)NULL
#define DEF_MARKER_TEXT		(char *)NULL
#define DEF_MARKER_UNDER	"no"
#define DEF_MARKER_WIDTH	"0"
#define DEF_MARKER_WINDOW	(char *)NULL
#define DEF_MARKER_XOR		"no"
#define DEF_MARKER_X_OFFSET	"0"
#define DEF_MARKER_Y_OFFSET	"0"

#define DEF_MARKER_TEXT_TAGS	"Text all"
#define DEF_MARKER_IMAGE_TAGS	"Image all"
#define DEF_MARKER_BITMAP_TAGS	"Bitmap all"
#define DEF_MARKER_WINDOW_TAGS	"Window all"
#define DEF_MARKER_POLYGON_TAGS	"Polygon all"
#define DEF_MARKER_LINE_TAGS	"Line all"

static Blt_OptionParseProc ObjToCoordinates;
static Blt_OptionPrintProc CoordinatesToObj;
static Blt_OptionFreeProc FreeCoordinates;
static Blt_CustomOption coordsOption =
{
    ObjToCoordinates, CoordinatesToObj, FreeCoordinates, (ClientData)0
};
static Blt_OptionFreeProc FreeColorPair;
static Blt_OptionParseProc ObjToColorPair;
static Blt_OptionPrintProc ColorPairToObj;
static Blt_CustomOption colorPairOption =
{
    ObjToColorPair, ColorPairToObj, FreeColorPair, (ClientData)0
};

extern Blt_CustomOption bltXAxisOption;
extern Blt_CustomOption bltYAxisOption;

typedef Marker *(MarkerCreateProc) _ANSI_ARGS_((void));
typedef void (MarkerDrawProc) _ANSI_ARGS_((Marker *markerPtr, 
	Drawable drawable));
typedef void (MarkerFreeProc) _ANSI_ARGS_((Graph *graphPtr, Marker *markerPtr));
typedef int (MarkerConfigProc) _ANSI_ARGS_((Marker *markerPtr));
typedef void (MarkerMapProc) _ANSI_ARGS_((Marker *markerPtr));
typedef void (MarkerPostScriptProc) _ANSI_ARGS_((Marker *markerPtr, 
	Blt_PostScript ps));
typedef int (MarkerPointProc) _ANSI_ARGS_((Marker *markerPtr, 
	Point2D *samplePtr));
typedef int (MarkerRegionProc) _ANSI_ARGS_((Marker *markerPtr, 
	Region2D *extsPtr, int enclosed));

typedef struct {
    Blt_ConfigSpec *configSpecs; /* Marker configuration specifications */

    MarkerConfigProc *configProc;
    MarkerDrawProc *drawProc;
    MarkerFreeProc *freeProc;
    MarkerMapProc *mapProc;
    MarkerPointProc *pointProc;
    MarkerRegionProc *regionProc;
    MarkerPostScriptProc *postscriptProc;

}  MarkerClass;



/*
 * -------------------------------------------------------------------
 *
 * Marker --
 *
 *	Structure defining the generic marker.  In C++ parlance this
 *	would be the base class from which all markers are derived.
 *
 *	This structure corresponds with the specific types of markers.
 *	Don't change this structure without changing the individual
 *	marker structures of each type below.
 *
 * ------------------------------------------------------------------- 
 */
struct MarkerStruct {
    GraphObject object;		/* Must be first field in marker. */

    MarkerClass *classPtr;

    Blt_HashEntry *hashPtr;

    Blt_ChainLink *linkPtr;

    char *elemName;		/* Element associated with marker */

    Axis2D axes;

    Point2D *worldPts;		/* Coordinate array to position marker */

    int nWorldPts;		/* Number of points in above array */

    int drawUnder;		/* If non-zero, draw the marker
				 * underneath any elements. This can
				 * be a performance penalty because
				 * the graph must be redraw entirely
				 * each time the marker is redrawn. */

    int clipped;		/* Indicates if the marker is totally
				 * clipped by the plotting area. */

    unsigned int flags;		


    int xOffset, yOffset;	/* Pixel offset from graph position */

    int state;
};

/*
 * -------------------------------------------------------------------
 *
 * TextMarker --
 *
 * -------------------------------------------------------------------
 */
typedef struct {
    Marker base;

    /* Fields specific to text markers. */
#ifdef notdef
    char *textVarName;		/* Name of variable (malloc'ed) or
				 * NULL. If non-NULL, graph displays
				 * the contents of this variable. */
#endif
    char *string;		/* Text string to be display.  The string
				 * make contain newlines. */

    Tk_Anchor anchor;		/* Indicates how to translate the given
				 * marker position. */

    Point2D anchorPt;		/* Translated anchor point. */

    int width, height;		/* Dimension of bounding box.  */

    TextStyle style;		/* Text attributes (font, fg, anchor, etc) */

    TextLayout *textPtr;	/* Contains information about the layout
				 * of the text. */
    Point2D outline[5];
    XColor *fillColor;
    GC fillGC;
} TextMarker;


static Blt_ConfigSpec textConfigSpecs[] =
{
    {BLT_CONFIG_ANCHOR, "-anchor", "anchor", "Anchor", DEF_MARKER_ANCHOR, 
	Blt_Offset(TextMarker, anchor), 0},
    {BLT_CONFIG_COLOR, "-background", "background", "MarkerBackground",
	(char *)NULL, Blt_Offset(TextMarker, fillColor), BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_SYNONYM, "-bg", "background", "Background", (char *)NULL, 0, 0},
    {BLT_CONFIG_LIST, "-bindtags", "bindTags", "BindTags", DEF_MARKER_TEXT_TAGS,
	Blt_Offset(TextMarker, base.object.tags), BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_CUSTOM, "-coords", "coords", "Coords", DEF_MARKER_COORDS, 
	Blt_Offset(TextMarker, base.worldPts), BLT_CONFIG_NULL_OK, 
        &coordsOption},
    {BLT_CONFIG_STRING, "-element", "element", "Element",
	DEF_MARKER_ELEMENT, Blt_Offset(TextMarker, base.elemName), 
        BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_SYNONYM, "-fg", "foreground", "Foreground", (char *)NULL, 0, 0},
    {BLT_CONFIG_SYNONYM, "-fill", "background", (char *)NULL, (char *)NULL, 
	0, 0},
    {BLT_CONFIG_FONT, "-font", "font", "Font", 	DEF_MARKER_FONT, 
	Blt_Offset(TextMarker, style.font), 0},
    {BLT_CONFIG_COLOR, "-foreground", "foreground", "Foreground",
	DEF_MARKER_FOREGROUND, Blt_Offset(TextMarker, style.color), 0},
    {BLT_CONFIG_JUSTIFY, "-justify", "justify", "Justify",
	DEF_MARKER_JUSTIFY, Blt_Offset(TextMarker, style.justify),
	BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_BOOLEAN, "-hide", "hide", "Hide", DEF_MARKER_HIDE, 
	Blt_Offset(TextMarker, base.object.hidden), 
        BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_CUSTOM, "-mapx", "mapX", "MapX", DEF_MARKER_MAP_X, 
	Blt_Offset(TextMarker, base.axes.x), 0, &bltXAxisOption},
    {BLT_CONFIG_CUSTOM, "-mapy", "mapY", "MapY", DEF_MARKER_MAP_Y, 
	Blt_Offset(TextMarker, base.axes.y), 0, &bltYAxisOption},
    {BLT_CONFIG_STRING, "-name", (char *)NULL, (char *)NULL, DEF_MARKER_NAME, 
	Blt_Offset(TextMarker, base.object.name), BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_SYNONYM, "-outline", "foreground", (char *)NULL, (char *)NULL, 
	0, 0},
    {BLT_CONFIG_PAD, "-padx", "padX", "PadX", DEF_MARKER_PAD, 
	Blt_Offset(TextMarker, style.padX), BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_PAD, "-pady", "padY", "PadY", DEF_MARKER_PAD, 
	Blt_Offset(TextMarker, style.padY), BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_DOUBLE, "-rotate", "rotate", "Rotate", DEF_MARKER_ANGLE, 
	Blt_Offset(TextMarker, style.angle), BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_STATE, "-state", "state", "State", DEF_MARKER_STATE, 
	Blt_Offset(TextMarker, base.state), BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_STRING, "-text", "text", "Text", DEF_MARKER_TEXT, 
	Blt_Offset(TextMarker, string), BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_BOOLEAN, "-under", "under", "Under", DEF_MARKER_UNDER, 
	Blt_Offset(TextMarker, base.drawUnder), BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_PIXELS_ANY, "-xoffset", "xOffset", "XOffset", 
	DEF_MARKER_X_OFFSET, Blt_Offset(TextMarker, base.xOffset),
	BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_PIXELS_ANY, "-yoffset", "yOffset", "YOffset", 
	DEF_MARKER_Y_OFFSET, Blt_Offset(TextMarker, base.yOffset),
	BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_END, NULL, NULL, NULL, NULL, 0, 0}
};

/*
 * -------------------------------------------------------------------
 *
 * WindowMarker --
 *
 * -------------------------------------------------------------------
 */
typedef struct {
    Marker base;

    /* Fields specific to window markers. */

    char *pathName;		/* Name of child widget to be displayed. */
    Tk_Window tkwin;		/* Window to display. */
    int reqWidth, reqHeight;	/* If non-zero, this overrides the size 
				 * requested by the child widget. */

    Tk_Anchor anchor;		/* Indicates how to translate the given
				 * marker position. */

    Point2D anchorPt;		/* Translated anchor point. */
    int width, height;		/* Current size of the child window. */

} WindowMarker;

static Blt_ConfigSpec windowConfigSpecs[] =
{
    {BLT_CONFIG_ANCHOR, "-anchor", "anchor", "Anchor",
	DEF_MARKER_ANCHOR, Blt_Offset(WindowMarker, anchor), 0},
    {BLT_CONFIG_LIST, "-bindtags", "bindTags", "BindTags",
	DEF_MARKER_WINDOW_TAGS, Blt_Offset(WindowMarker, base.object.tags),
	BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_CUSTOM, "-coords", "coords", "Coords", DEF_MARKER_COORDS, 
	Blt_Offset(WindowMarker, base.worldPts), BLT_CONFIG_NULL_OK, 
	&coordsOption},
    {BLT_CONFIG_STRING, "-element", "element", "Element", DEF_MARKER_ELEMENT, 
	Blt_Offset(WindowMarker, base.elemName), BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_PIXELS_POSITIVE, "-height", "height", "Height",
	DEF_MARKER_HEIGHT, Blt_Offset(WindowMarker, reqHeight),
	BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_BOOLEAN, "-hide", "hide", "Hide",
	DEF_MARKER_HIDE, Blt_Offset(WindowMarker, base.object.hidden),
	BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_CUSTOM, "-mapx", "mapX", "MapX", DEF_MARKER_MAP_X, 
	Blt_Offset(WindowMarker, base.axes.x), 0, &bltXAxisOption},
    {BLT_CONFIG_CUSTOM, "-mapy", "mapY", "MapY", DEF_MARKER_MAP_Y, 
	Blt_Offset(WindowMarker, base.axes.y), 0, &bltYAxisOption},
    {BLT_CONFIG_STRING, "-name", (char *)NULL, (char *)NULL,
	DEF_MARKER_NAME, Blt_Offset(WindowMarker, base.object.name), 
	BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_STATE, "-state", "state", "State", DEF_MARKER_STATE, 
	Blt_Offset(WindowMarker, base.state), BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_BOOLEAN, "-under", "under", "Under", DEF_MARKER_UNDER, 
	Blt_Offset(WindowMarker, base.drawUnder), 
	BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_PIXELS_POSITIVE, "-width", "width", "Width",
	DEF_MARKER_WIDTH, Blt_Offset(WindowMarker, reqWidth),
	BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_STRING, "-window", "window", "Window",
	DEF_MARKER_WINDOW, Blt_Offset(WindowMarker, pathName),
	BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_PIXELS_ANY, "-xoffset", "xOffset", "XOffset",
	DEF_MARKER_X_OFFSET, Blt_Offset(WindowMarker, base.xOffset),
	BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_PIXELS_ANY, "-yoffset", "yOffset", "YOffset",
	DEF_MARKER_Y_OFFSET, Blt_Offset(WindowMarker, base.yOffset),
	BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_END, NULL, NULL, NULL, NULL, 0, 0}
};

/*
 * -------------------------------------------------------------------
 *
 * BitmapMarker --
 *
 * -------------------------------------------------------------------
 */
typedef struct {
    Marker base;

    /* Fields specific to bitmap markers. */

    Pixmap srcBitmap;		/* Original bitmap. May be further
				 * scaled or rotated. */
    double reqAngle;		/* Requested rotation of the bitmap */
    double angle;		/* Normalized rotation (0..360
				 * degrees) */
    Tk_Anchor anchor;		/* If only one X-Y coordinate is
				 * given, indicates how to translate
				 * the given marker position.  Otherwise,
				 * if there are two X-Y coordinates, then
				 * this value is ignored. */
    Point2D anchorPt;		/* Translated anchor point. */

    XColor *outlineColor;	/* Foreground color */
    XColor *fillColor;		/* Background color */

    GC gc;			/* Private graphic context */
    GC fillGC;			/* Shared graphic context */
    Pixmap destBitmap;		/* Bitmap to be drawn. */
    int destWidth, destHeight;	/* Dimensions of the final bitmap */

    Point2D outline[MAX_OUTLINE_POINTS]; 
				/* Polygon representing the background
				 * of the bitmap. */
    int nOutlinePts;
} BitmapMarker;

static Blt_ConfigSpec bitmapConfigSpecs[] =
{
    {BLT_CONFIG_ANCHOR, "-anchor", "anchor", "Anchor",
	DEF_MARKER_ANCHOR, Blt_Offset(BitmapMarker, anchor), 0},
    {BLT_CONFIG_COLOR, "-background", "background", "Background",
	DEF_MARKER_BACKGROUND, Blt_Offset(BitmapMarker, fillColor),
	BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_SYNONYM, "-bg", "background", (char *)NULL, (char *)NULL, 0, 0},
    {BLT_CONFIG_LIST, "-bindtags", "bindTags", "BindTags", 
	DEF_MARKER_BITMAP_TAGS, Blt_Offset(BitmapMarker, base.object.tags), 
        BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_BITMAP, "-bitmap", "bitmap", "Bitmap", DEF_MARKER_BITMAP, 
	Blt_Offset(BitmapMarker, srcBitmap), BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_CUSTOM, "-coords", "coords", "Coords", DEF_MARKER_COORDS, 
	Blt_Offset(BitmapMarker, base.worldPts), BLT_CONFIG_NULL_OK, 
	&coordsOption},
    {BLT_CONFIG_STRING, "-element", "element", "Element", DEF_MARKER_ELEMENT, 
	Blt_Offset(BitmapMarker, base.elemName), BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_SYNONYM, "-fg", "foreground", (char *)NULL, (char *)NULL, 0, 0},
    {BLT_CONFIG_SYNONYM, "-fill", "background", (char *)NULL, (char *)NULL, 
	0, 0},
    {BLT_CONFIG_COLOR, "-foreground", "foreground", "Foreground",
	DEF_MARKER_FOREGROUND, Blt_Offset(BitmapMarker, outlineColor),
	BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_BOOLEAN, "-hide", "hide", "Hide", DEF_MARKER_HIDE, 
	Blt_Offset(BitmapMarker, base.object.hidden), 
	BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_CUSTOM, "-mapx", "mapX", "MapX", DEF_MARKER_MAP_X, 
	Blt_Offset(BitmapMarker, base.axes.x), 0, &bltXAxisOption},
    {BLT_CONFIG_CUSTOM, "-mapy", "mapY", "MapY", DEF_MARKER_MAP_Y, 
	Blt_Offset(BitmapMarker, base.axes.y), 0, &bltYAxisOption},
    {BLT_CONFIG_STRING, "-name", (char *)NULL, (char *)NULL, DEF_MARKER_NAME, 
	Blt_Offset(BitmapMarker, base.object.name), BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_SYNONYM, "-outline", "foreground", (char *)NULL, (char *)NULL, 
	0, 0},
    {BLT_CONFIG_DOUBLE, "-rotate", "rotate", "Rotate", DEF_MARKER_ANGLE, 
	Blt_Offset(BitmapMarker, reqAngle), BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_STATE, "-state", "state", "State", DEF_MARKER_STATE, 
	Blt_Offset(BitmapMarker, base.state), BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_BOOLEAN, "-under", "under", "Under", DEF_MARKER_UNDER, 
	Blt_Offset(BitmapMarker, base.drawUnder), 
	BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_PIXELS_ANY, "-xoffset", "xOffset", "XOffset", 
	DEF_MARKER_X_OFFSET, Blt_Offset(BitmapMarker, base.xOffset), 
	BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_PIXELS_ANY, "-yoffset", "yOffset", "YOffset",
	DEF_MARKER_Y_OFFSET, Blt_Offset(BitmapMarker, base.yOffset),
	BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_END, NULL, NULL, NULL, NULL, 0, 0}
};


/*
 * -------------------------------------------------------------------
 *
 * ImageMarker --
 *
 * -------------------------------------------------------------------
 */
typedef struct {
    Marker base;

    char *imageName;		/* Name of image to be displayed. */
    Tk_Image image;		/* Tk image to be displayed. */
    Tk_Anchor anchor;		/* Indicates how to translate the given
				 * marker position. */
    Point2D anchorPt;		/* Translated anchor point. */
    int width, height;		/* Dimensions of the possibly scaled image. */
    Blt_Painter painter;
    Blt_Picture original;
    int pictX, pictY;		/*  */
    Blt_Picture scaled;	/* Pixmap containing the scaled image */
    GC gc;

} ImageMarker;

static Blt_ConfigSpec imageConfigSpecs[] =
{
    {BLT_CONFIG_ANCHOR, "-anchor", "anchor", "Anchor",
	DEF_MARKER_ANCHOR, Blt_Offset(ImageMarker, anchor), 0},
    {BLT_CONFIG_LIST, "-bindtags", "bindTags", "BindTags",
	DEF_MARKER_IMAGE_TAGS, Blt_Offset(BitmapMarker, base.object.tags),
	BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_CUSTOM, "-coords", "coords", "Coords", DEF_MARKER_COORDS, 
	Blt_Offset(BitmapMarker, base.worldPts), BLT_CONFIG_NULL_OK, 
	&coordsOption},
    {BLT_CONFIG_STRING, "-element", "element", "Element", DEF_MARKER_ELEMENT, 
	Blt_Offset(BitmapMarker, base.elemName), BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_BOOLEAN, "-hide", "hide", "Hide", DEF_MARKER_HIDE,	
	Blt_Offset(BitmapMarker, base.object.hidden), 
	BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_STRING, "-image", "image", "Image",
	(char *)NULL, Blt_Offset(ImageMarker, imageName), BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_CUSTOM, "-mapx", "mapX", "MapX", DEF_MARKER_MAP_X, 
	Blt_Offset(BitmapMarker, base.axes.x), 0, &bltXAxisOption},
    {BLT_CONFIG_CUSTOM, "-mapy", "mapY", "MapY", DEF_MARKER_MAP_Y, 
	Blt_Offset(BitmapMarker, base.axes.y), 0, &bltYAxisOption},
    {BLT_CONFIG_STRING, "-name", (char *)NULL, (char *)NULL, DEF_MARKER_NAME, 
	Blt_Offset(BitmapMarker, base.object.name), BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_STATE, "-state", "state", "State", DEF_MARKER_STATE, 
	Blt_Offset(BitmapMarker, base.state), BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_BOOLEAN, "-under", "under", "Under", DEF_MARKER_UNDER, 
	Blt_Offset(BitmapMarker, base.drawUnder), 
	BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_PIXELS_ANY, "-xoffset", "xOffset", "XOffset", 
	DEF_MARKER_X_OFFSET, Blt_Offset(BitmapMarker, base.xOffset),
	BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_PIXELS_ANY, "-yoffset", "yOffset", "YOffset",
	DEF_MARKER_Y_OFFSET, Blt_Offset(BitmapMarker, base.yOffset),
	BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_END, NULL, NULL, NULL, NULL, 0, 0}
};

/*
 * -------------------------------------------------------------------
 *
 * LineMarker --
 *
 * -------------------------------------------------------------------
 */
typedef struct {
    Marker base;

    XColor *fillColor;
    XColor *outlineColor;	/* Foreground and background colors */

    int lineWidth;		/* Line width. */
    int capStyle;		/* Cap style. */
    int joinStyle;		/* Join style.*/
    Blt_Dashes dashes;		/* Dash list values (max 11) */

    GC gc;			/* Private graphic context */

    Segment2D *segments;	/* Malloc'ed array of points.
				 * Represents individual line segments
				 * (2 points per segment) comprising
				 * the mapped line.  The segments may
				 * not necessarily be connected after
				 * clipping. */
    int nSegments;		/* # segments in the above array. */

    int xor;
    int xorState;		/* State of the XOR drawing. Indicates
				 * if the marker is currently drawn. */
} LineMarker;

static Blt_ConfigSpec lineConfigSpecs[] =
{
    {BLT_CONFIG_LIST, "-bindtags", "bindTags", "BindTags",
	DEF_MARKER_LINE_TAGS, Blt_Offset(ImageMarker, base.object.tags),
	BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_CAP_STYLE, "-cap", "cap", "Cap",
	DEF_MARKER_CAP_STYLE, Blt_Offset(LineMarker, capStyle),
	BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_CUSTOM, "-coords", "coords", "Coords", DEF_MARKER_COORDS, 
	Blt_Offset(ImageMarker, base.worldPts), BLT_CONFIG_NULL_OK, 
	&coordsOption},
    {BLT_CONFIG_DASHES, "-dashes", "dashes", "Dashes",
	DEF_MARKER_DASHES, Blt_Offset(LineMarker, dashes), BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_PIXELS, "-dashoffset", "dashOffset", "DashOffset",
	DEF_MARKER_DASH_OFFSET, Blt_Offset(LineMarker, dashes.offset),
	BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_STRING, "-element", "element", "Element",
	DEF_MARKER_ELEMENT, Blt_Offset(ImageMarker, base.elemName), 
	BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_COLOR, "-fill", "fill", "Fill",
	(char *)NULL, Blt_Offset(LineMarker, fillColor), BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_JOIN_STYLE, "-join", "join", "Join",
	DEF_MARKER_JOIN_STYLE, Blt_Offset(LineMarker, joinStyle),
	BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_PIXELS, "-linewidth", "lineWidth", "LineWidth",
	DEF_MARKER_LINE_WIDTH, Blt_Offset(LineMarker, lineWidth),
	BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_BOOLEAN, "-hide", "hide", "Hide", DEF_MARKER_HIDE, 
	Blt_Offset(ImageMarker, base.object.hidden), 
	BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_CUSTOM, "-mapx", "mapX", "MapX", DEF_MARKER_MAP_X, 
	Blt_Offset(ImageMarker, base.axes.x), 0, &bltXAxisOption},
    {BLT_CONFIG_CUSTOM, "-mapy", "mapY", "MapY", DEF_MARKER_MAP_Y, 
	Blt_Offset(ImageMarker, base.axes.y), 0, &bltYAxisOption},
    {BLT_CONFIG_STRING, "-name", (char *)NULL, (char *)NULL, DEF_MARKER_NAME, 
	Blt_Offset(ImageMarker, base.object.name), BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_COLOR, "-outline", "outline", "Outline",
	DEF_MARKER_OUTLINE_COLOR, Blt_Offset(LineMarker, outlineColor),
	BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_STATE, "-state", "state", "State", DEF_MARKER_STATE, 
	Blt_Offset(ImageMarker, base.state), BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_BOOLEAN, "-under", "under", "Under", DEF_MARKER_UNDER, 
	Blt_Offset(ImageMarker, base.drawUnder), 
	BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_PIXELS_ANY, "-xoffset", "xOffset", "XOffset", 
	DEF_MARKER_X_OFFSET, Blt_Offset(ImageMarker, base.xOffset),
	BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_BOOLEAN, "-xor", "xor", "Xor", DEF_MARKER_XOR, 
	Blt_Offset(LineMarker, xor), BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_PIXELS_ANY, "-yoffset", "yOffset", "YOffset", 
	DEF_MARKER_Y_OFFSET, Blt_Offset(ImageMarker, base.yOffset),
	BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_END, NULL, NULL, NULL, NULL, 0, 0}
};

/*
 * -------------------------------------------------------------------
 *
 * PolygonMarker --
 *
 * -------------------------------------------------------------------
 */
typedef struct {
    Marker base;

    Point2D *screenPts;		/* Array of points representing the
				 * polygon in screen coordinates. It's
				 * not used for drawing, but to
				 * generate the outlinePts and fillPts
				 * arrays that are the coordinates of
				 * the possibly clipped outline and
				 * filled polygon. */

    ColorPair outline;
    ColorPair fill;

    Pixmap stipple;		/* Stipple pattern to fill the
				 * polygon. */
    int lineWidth;		/* Width of polygon outline. */
    int capStyle;
    int joinStyle;
    Blt_Dashes dashes;		/* List of dash values.  Indicates how
				 * draw the dashed line.  If no dash
				 * values are provided, or the first
				 * value is zero, then the line is
				 * drawn solid. */

    GC outlineGC;		/* Graphics context to draw the
				 * outline of the polygon. */
    GC fillGC;			/* Graphics context to draw the filled
				 * polygon. */

    Point2D *fillPts;		/* Malloc'ed array of points used to
				 * draw the filled polygon. These
				 * points may form a degenerate
				 * polygon after clipping.
				 */

    int nFillPts;		/* # points in the above array. */

    Segment2D *outlinePts;	/* Malloc'ed array of points.
				 * Represents individual line segments
				 * (2 points per segment) comprising
				 * the outline of the polygon.  The
				 * segments may not necessarily be
				 * closed or connected after
				 * clipping. */

    int nOutlinePts;		/* # points in the above array. */

    int xor;
    int xorState;		/* State of the XOR drawing. Indicates
				 * if the marker is visible. We have
				 * to drawn it again to erase it. */
} PolygonMarker;

static Blt_ConfigSpec polygonConfigSpecs[] =
{
    {BLT_CONFIG_LIST, "-bindtags", "bindTags", "BindTags",
	DEF_MARKER_POLYGON_TAGS, Blt_Offset(PolygonMarker, base.object.tags),
        BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_CAP_STYLE, "-cap", "cap", "Cap",
	DEF_MARKER_CAP_STYLE, Blt_Offset(PolygonMarker, capStyle),
	BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_CUSTOM, "-coords", "coords", "Coords", DEF_MARKER_COORDS, 
	Blt_Offset(PolygonMarker, base.worldPts), BLT_CONFIG_NULL_OK, 
	&coordsOption},
    {BLT_CONFIG_DASHES, "-dashes", "dashes", "Dashes", DEF_MARKER_DASHES, 
	Blt_Offset(PolygonMarker, dashes), BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_STRING, "-element", "element", "Element", DEF_MARKER_ELEMENT, 
	Blt_Offset(PolygonMarker, base.elemName), BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_CUSTOM, "-fill", "fill", "Fill", DEF_MARKER_FILL_COLOR, 
	Blt_Offset(PolygonMarker, fill), BLT_CONFIG_NULL_OK, 
	&colorPairOption},
    {BLT_CONFIG_JOIN_STYLE, "-join", "join", "Join", DEF_MARKER_JOIN_STYLE, 
	Blt_Offset(PolygonMarker, joinStyle), BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_PIXELS, "-linewidth", "lineWidth", "LineWidth",
	DEF_MARKER_LINE_WIDTH, Blt_Offset(PolygonMarker, lineWidth),
	BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_BOOLEAN, "-hide", "hide", "Hide", DEF_MARKER_HIDE, 
	Blt_Offset(PolygonMarker, base.object.hidden), 
	BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_CUSTOM, "-mapx", "mapX", "MapX", DEF_MARKER_MAP_X, 
	Blt_Offset(PolygonMarker, base.axes.x), 0, &bltXAxisOption},
    {BLT_CONFIG_CUSTOM, "-mapy", "mapY", "MapY", DEF_MARKER_MAP_Y, 
	Blt_Offset(PolygonMarker, base.axes.y), 0, &bltYAxisOption},
    {BLT_CONFIG_STRING, "-name", (char *)NULL, (char *)NULL, DEF_MARKER_NAME, 
	Blt_Offset(PolygonMarker, base.object.name), BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_CUSTOM, "-outline", "outline", "Outline", 
	DEF_MARKER_OUTLINE_COLOR, Blt_Offset(PolygonMarker, outline),
	BLT_CONFIG_NULL_OK, &colorPairOption},
    {BLT_CONFIG_STATE, "-state", "state", "State", DEF_MARKER_STATE, 
	Blt_Offset(PolygonMarker, base.state), BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_BITMAP, "-stipple", "stipple", "Stipple", DEF_MARKER_STIPPLE, 
	Blt_Offset(PolygonMarker, stipple), BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_BOOLEAN, "-under", "under", "Under", DEF_MARKER_UNDER, 
	Blt_Offset(PolygonMarker, base.drawUnder), 
	BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_PIXELS_ANY, "-xoffset", "xOffset", "XOffset", 
	DEF_MARKER_X_OFFSET, Blt_Offset(PolygonMarker, base.xOffset),
	BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_BOOLEAN, "-xor", "xor", "Xor", DEF_MARKER_XOR, 
	Blt_Offset(PolygonMarker, xor), BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_PIXELS_ANY, "-yoffset", "yOffset", "YOffset",
	DEF_MARKER_Y_OFFSET, Blt_Offset(PolygonMarker, base.yOffset),
	BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_END, NULL, NULL, NULL, NULL, 0, 0}
};

static MarkerCreateProc CreateBitmapMarker, CreateLineMarker, CreateImageMarker,
	CreatePolygonMarker, CreateTextMarker, CreateWindowMarker;

static MarkerDrawProc DrawBitmapMarker, DrawLineMarker, DrawImageMarker,
	DrawPolygonMarker, DrawTextMarker, DrawWindowMarker;

static MarkerFreeProc FreeBitmapMarker, FreeLineMarker, FreeImageMarker, 
	FreePolygonMarker, FreeTextMarker, FreeWindowMarker;

static MarkerConfigProc ConfigureBitmapMarker, ConfigureLineMarker, 
	ConfigureImageMarker, ConfigurePolygonMarker, ConfigureTextMarker, 
	ConfigureWindowMarker;

static MarkerMapProc MapBitmapMarker, MapLineMarker, MapImageMarker, 
	MapPolygonMarker, MapTextMarker, MapWindowMarker;

static MarkerPostScriptProc BitmapMarkerToPostScript, LineMarkerToPostScript, 
	ImageMarkerToPostScript, PolygonMarkerToPostScript, 
	TextMarkerToPostScript, WindowMarkerToPostScript;

static MarkerPointProc PointInBitmapMarker, PointInLineMarker, 
	PointInImageMarker, PointInPolygonMarker, PointInTextMarker, 
	PointInWindowMarker;

static MarkerRegionProc RegionInBitmapMarker, RegionInLineMarker, 
	RegionInImageMarker, RegionInPolygonMarker, RegionInTextMarker, 
	RegionInWindowMarker;

static Tk_ImageChangedProc ImageChangedProc;

static MarkerClass bitmapMarkerClass = {
    bitmapConfigSpecs,
    ConfigureBitmapMarker,
    DrawBitmapMarker,
    FreeBitmapMarker,
    MapBitmapMarker,
    PointInBitmapMarker,
    RegionInBitmapMarker,
    BitmapMarkerToPostScript,
};

static MarkerClass imageMarkerClass = {
    imageConfigSpecs,
    ConfigureImageMarker,
    DrawImageMarker,
    FreeImageMarker,
    MapImageMarker,
    PointInImageMarker,
    RegionInImageMarker,
    ImageMarkerToPostScript,
};

static MarkerClass lineMarkerClass = {
    lineConfigSpecs,
    ConfigureLineMarker,
    DrawLineMarker,
    FreeLineMarker,
    MapLineMarker,
    PointInLineMarker,
    RegionInLineMarker,
    LineMarkerToPostScript,
};

static MarkerClass polygonMarkerClass = {
    polygonConfigSpecs,
    ConfigurePolygonMarker,
    DrawPolygonMarker,
    FreePolygonMarker,
    MapPolygonMarker,
    PointInPolygonMarker,
    RegionInPolygonMarker,
    PolygonMarkerToPostScript,
};

static MarkerClass textMarkerClass = {
    textConfigSpecs,
    ConfigureTextMarker,
    DrawTextMarker,
    FreeTextMarker,
    MapTextMarker,
    PointInTextMarker,
    RegionInTextMarker,
    TextMarkerToPostScript,
};

static MarkerClass windowMarkerClass = {
    windowConfigSpecs,
    ConfigureWindowMarker,
    DrawWindowMarker,
    FreeWindowMarker,
    MapWindowMarker,
    PointInWindowMarker,
    RegionInWindowMarker,
    WindowMarkerToPostScript,
};

#ifdef notdef
static MarkerClass rectangleMarkerClass = {
    rectangleConfigSpecs,
    ConfigureRectangleMarker,
    DrawRectangleMarker,
    FreeRectangleMarker,
    MapRectangleMarker,
    PointInRectangleMarker,
    RegionInRectangleMarker,
    RectangleMarkerToPostScript,
};

static MarkerClass ovalMarkerClass = {
    ovalConfigSpecs,
    ConfigureOvalMarker,
    DrawOvalMarker,
    FreeOvalMarker,
    MapOvalMarker,
    PointInOvalMarker,
    RegionInOvalMarker,
    OvalMarkerToPostScript,
};
#endif

/*
 * ----------------------------------------------------------------------
 *
 * BoxesDontOverlap --
 *
 *	Tests if the bounding box of a marker overlaps the plotting
 *	area in any way.  If so, the marker will be drawn.  Just do a
 *	min/max test on the extents of both boxes.
 *
 *	Note: It's assumed that the extents of the bounding box lie 
 *	      within the area.  So for a 10x10 rectangle, bottom and
 *	      left would be 9.
 *
 * Results:
 *	Returns 0 is the marker is visible in the plotting area, and
 *	1 otherwise (marker is clipped).
 *
 * ----------------------------------------------------------------------
 */
static int
BoxesDontOverlap(Graph *graphPtr, Region2D *extsPtr)
{
    assert(extsPtr->right >= extsPtr->left);
    assert(extsPtr->bottom >= extsPtr->top);
    assert(graphPtr->right >= graphPtr->left);
    assert(graphPtr->bottom >= graphPtr->top);

    return (((double)graphPtr->right < extsPtr->left) ||
	    ((double)graphPtr->bottom < extsPtr->top) ||
	    (extsPtr->right < (double)graphPtr->left) ||
	    (extsPtr->bottom < (double)graphPtr->top));
}


/*
 * ----------------------------------------------------------------------
 *
 * GetCoordinate --
 *
 * 	Convert the expression string into a floating point value. The
 *	only reason we use this routine instead of Blt_ExprDouble is to
 *	handle "elastic" bounds.  That is, convert the strings "-Inf",
 *	"Inf" into -(DBL_MAX) and DBL_MAX respectively.
 *
 * Results:
 *	The return value is a standard Tcl result.  The value of the
 * 	expression is passed back via valuePtr.
 *
 * ----------------------------------------------------------------------
 */
static int
GetCoordinate(
    Tcl_Interp *interp,		/* Interpreter to send results back to */
    Tcl_Obj *objPtr,			/* Numeric expression string to parse */
    double *valuePtr)		/* Real-valued result of expression */
{
    char c;
    char *expr;
    
    expr = Tcl_GetString(objPtr);
    c = expr[0];
    if ((c == 'I') && (strcmp(expr, "Inf") == 0)) {
	*valuePtr = DBL_MAX;	/* Elastic upper bound */
    } else if ((c == '-') && (expr[1] == 'I') && (strcmp(expr, "-Inf") == 0)) {
	*valuePtr = -DBL_MAX;	/* Elastic lower bound */
    } else if ((c == '+') && (expr[1] == 'I') && (strcmp(expr, "+Inf") == 0)) {
	*valuePtr = DBL_MAX;	/* Elastic upper bound */
    } else if (Blt_ExprDoubleFromObj(interp, objPtr, valuePtr) != TCL_OK) {
	return TCL_ERROR;
    }
    return TCL_OK;
}


/*
 * ----------------------------------------------------------------------
 *
 * PrintCoordinate --
 *
 * 	Convert the floating point value into its string
 * 	representation.  The only reason this routine is used in
 * 	instead of sprintf, is to handle the "elastic" bounds.  That
 * 	is, convert the values DBL_MAX and -(DBL_MAX) into "+Inf" and
 * 	"-Inf" respectively.
 *
 * Results:
 *	The return value is a standard Tcl result.  The string of the
 * 	expression is passed back via string.
 *
 * ---------------------------------------------------------------------- */
static Tcl_Obj *
PrintCoordinate(x)
    double x;			/* Numeric value */
{
    if (x == DBL_MAX) {
	return Tcl_NewStringObj("+Inf", -1);
    } else if (x == -DBL_MAX) {
	return Tcl_NewStringObj("-Inf", -1);
    } else {
	return Tcl_NewDoubleObj(x);
    }
}

/*
 * ----------------------------------------------------------------------
 *
 * ParseCoordinates --
 *
 *	The Tcl coordinate list is converted to their floating point
 *	values. It will then replace the current marker coordinates.
 *
 *	Since different marker types require different number of
 *	coordinates this must be checked here.
 *
 * Results:
 *	The return value is a standard Tcl result.
 *
 * Side effects:
 *	If the marker coordinates are reset, the graph is eventually
 *	redrawn with at the new marker coordinates.
 *
 * ----------------------------------------------------------------------
 */
static int
ParseCoordinates(
    Tcl_Interp *interp,
    Marker *markerPtr,
    int objc,
    Tcl_Obj *CONST *objv)
{
    int nWorldPts;
    int minArgs, maxArgs;
    Point2D *worldPts;
    int i;

    if (objc == 0) {
	return TCL_OK;
    }
    if (objc & 1) {
	Tcl_AppendResult(interp, "odd number of marker coordinates specified",
	    (char *)NULL);
	return TCL_ERROR;
    }
    switch (markerPtr->object.classId) {
    case OBJECT_CLASS_LINE_MARKER:
	minArgs = 4, maxArgs = 0;
	break;
    case OBJECT_CLASS_POLYGON_MARKER:
	minArgs = 6, maxArgs = 0;
	break;
    case OBJECT_CLASS_WINDOW_MARKER:
    case OBJECT_CLASS_TEXT_MARKER:
	minArgs = 2, maxArgs = 2;
	break;
    case OBJECT_CLASS_IMAGE_MARKER:
    case OBJECT_CLASS_BITMAP_MARKER:
	minArgs = 2, maxArgs = 4;
	break;
    default:
	Tcl_AppendResult(interp, "unknown marker type", (char *)NULL);
	return TCL_ERROR;
    }

    if (objc < minArgs) {
	Tcl_AppendResult(interp, "too few marker coordinates specified",
	    (char *)NULL);
	return TCL_ERROR;
    }
    if ((maxArgs > 0) && (objc > maxArgs)) {
	Tcl_AppendResult(interp, "too many marker coordinates specified",
	    (char *)NULL);
	return TCL_ERROR;
    }
    nWorldPts = objc / 2;
    worldPts = Blt_Malloc(nWorldPts * sizeof(Point2D));
    if (worldPts == NULL) {
	Tcl_AppendResult(interp, "can't allocate new coordinate array",
	    (char *)NULL);
	return TCL_ERROR;
    }

    {
	Point2D *pp;

	pp = worldPts;
	for (i = 0; i < objc; i += 2) {
	    double x, y;
	    
	    if ((GetCoordinate(interp, objv[i], &x) != TCL_OK) ||
		(GetCoordinate(interp, objv[i + 1], &y) != TCL_OK)) {
		Blt_Free(worldPts);
		return TCL_ERROR;
	    }
	    pp->x = x, pp->y = y, pp++;
	}
    }
    /* Don't free the old coordinate array until we've parsed the new
     * coordinates without errors.  */
    if (markerPtr->worldPts != NULL) {
	Blt_Free(markerPtr->worldPts);
    }
    markerPtr->worldPts = worldPts;
    markerPtr->nWorldPts = nWorldPts;
    markerPtr->flags |= MAP_ITEM;
    return TCL_OK;
}

/*ARGSUSED*/
static void
FreeCoordinates(
    ClientData clientData,
    Display *display,		/* Not used. */
    char *widgRec,
    int offset)
{
    Marker *markerPtr = (Marker *)widgRec;
    Point2D **pointsPtr = (Point2D **)(widgRec + offset);

    if (*pointsPtr != NULL) {
	Blt_Free(*pointsPtr);
	*pointsPtr = NULL;
    }
    markerPtr->nWorldPts = 0;
}

/*
 * ----------------------------------------------------------------------
 *
 * ObjToCoordinates --
 *
 *	Given a Tcl list of numeric expression representing the
 *	element values, convert into an array of floating point
 *	values. In addition, the minimum and maximum values are saved.
 *	Since elastic values are allow (values which translate to the
 *	min/max of the graph), we must try to get the non-elastic
 *	minimum and maximum.
 *
 * Results:
 *	The return value is a standard Tcl result.  The vector is
 *	passed back via the vecPtr.
 *
 * ---------------------------------------------------------------------- 
 */
/*ARGSUSED*/
static int
ObjToCoordinates(
    ClientData clientData,	/* Not used. */
    Tcl_Interp *interp,		/* Interpreter to send results back to */
    Tk_Window tkwin,		/* Not used. */
    Tcl_Obj *objPtr,		/* Tcl list of numeric expressions */
    char *widgRec,		/* Marker record */
    int offset,			/* Offset to field in structure */
    int flags)	
{
    Marker *markerPtr = (Marker *)widgRec;
    int objc;
    Tcl_Obj **objv;

    if (Tcl_ListObjGetElements(interp, objPtr, &objc, &objv) != TCL_OK) {
	return TCL_ERROR;
    }
    if (objc == 0) {
	return TCL_OK;
    }
    return ParseCoordinates(interp, markerPtr, objc, objv);
}

/*
 * ----------------------------------------------------------------------
 *
 * CoordinatesToString --
 *
 *	Convert the vector of floating point values into a Tcl list.
 *
 * Results:
 *	The string representation of the vector is returned.
 *
 * ----------------------------------------------------------------------
 */
/*ARGSUSED*/
static Tcl_Obj *
CoordinatesToObj(
    ClientData clientData,	/* Not used. */
    Tcl_Interp *interp,
    Tk_Window tkwin,		/* Not used. */
    char *widgRec,		/* Marker record */
    int offset,			/* Offset to field in structure */
    int flags)	
{
    Marker *markerPtr = (Marker *)widgRec;
    Tcl_Obj *listObjPtr;
    Point2D *pp, *pend;

    listObjPtr = Tcl_NewListObj(0, (Tcl_Obj **)NULL);
    for (pp = markerPtr->worldPts, pend = pp + markerPtr->nWorldPts; pp < pend;
	 pp++) {
	Tcl_ListObjAppendElement(interp, listObjPtr, PrintCoordinate(pp->x));
	Tcl_ListObjAppendElement(interp, listObjPtr, PrintCoordinate(pp->y));
    }
    return listObjPtr;
}

/*LINTLIBRARY*/
static int
GetColorPair(
    Tcl_Interp *interp,
    Tk_Window tkwin,
    Tcl_Obj *fgObjPtr, Tcl_Obj *bgObjPtr,
    ColorPair *pairPtr,
    int allowDefault)
{
    XColor *fgColor, *bgColor;
    char *string;
    size_t length;

    fgColor = bgColor = NULL;
    if (fgObjPtr != NULL) {
	string = Tcl_GetString(fgObjPtr);
	length = strlen(string);
	if (string[0] == '\0') {
	    fgColor = NULL;
	} else if ((allowDefault) && (string[0] == 'd') &&
		   (strncmp(string, "defcolor", length) == 0)) {
	    fgColor = COLOR_DEFAULT;
	} else {
	    fgColor = Tk_AllocColorFromObj(interp, tkwin, fgObjPtr);
	    if (fgColor == NULL) {
		return TCL_ERROR;
	    }
	}
    }
    if (bgObjPtr != NULL) {
	string = Tcl_GetString(bgObjPtr);
	length = strlen(string);
	if (string[0] == '\0') {
	    bgColor = NULL;
	} else if ((allowDefault) && (string[0] == 'd') &&
		   (strncmp(string, "defcolor", length) == 0)) {
	    bgColor = COLOR_DEFAULT;
	} else {
	    bgColor = Tk_AllocColorFromObj(interp, tkwin, bgObjPtr);
	    if (bgColor == NULL) {
		return TCL_ERROR;
	    }
	}
    }
    if (pairPtr->fgColor != NULL) {
	Tk_FreeColor(pairPtr->fgColor);
    }
    if (pairPtr->bgColor != NULL) {
	Tk_FreeColor(pairPtr->bgColor);
    }
    pairPtr->fgColor = fgColor;
    pairPtr->bgColor = bgColor;
    return TCL_OK;
}

void
Blt_FreeColorPair(pairPtr)
    ColorPair *pairPtr;
{
    if ((pairPtr->bgColor != NULL) && (pairPtr->bgColor != COLOR_DEFAULT)) {
	Tk_FreeColor(pairPtr->bgColor);
    }
    if ((pairPtr->fgColor != NULL) && (pairPtr->fgColor != COLOR_DEFAULT)) {
	Tk_FreeColor(pairPtr->fgColor);
    }
    pairPtr->bgColor = pairPtr->fgColor = NULL;
}

static void
FreeColorPair(
    ClientData clientData,
    Display *display,		/* Not used. */
    char *widgRec,
    int offset)
{
    ColorPair *pairPtr = (ColorPair *)(widgRec + offset);

    Blt_FreeColorPair(pairPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * ObjToColorPair --
 *
 *	Convert the color names into pair of XColor pointers.
 *
 * Results:
 *	A standard Tcl result.  The color pointer is written into the
 *	widget record.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
ObjToColorPair(
    ClientData clientData,	/* Not used. */
    Tcl_Interp *interp,		/* Interpreter to send results back to */
    Tk_Window tkwin,		/* Not used. */
    Tcl_Obj *objPtr,		/* String representing color */
    char *widgRec,		/* Widget record */
    int offset,			/* Offset to field in structure */
    int flags)	
{
    ColorPair *pairPtr = (ColorPair *)(widgRec + offset);
    long longValue = (long)clientData;
    int bool;
    int objc;
    Tcl_Obj **objv;

    if (Tcl_ListObjGetElements(interp, objPtr, &objc, &objv) != TCL_OK) {
	return TCL_ERROR;
    }
    if (objc > 2) {
	Tcl_AppendResult(interp, "too many names in colors list", 
		(char *)NULL);
	return TCL_ERROR;
    }
    if (objc == 0) {
	Blt_FreeColorPair(pairPtr);
	return TCL_OK;
    }
    bool = (int)longValue;
    if (objc == 1) {
	if (GetColorPair(interp, tkwin, objv[0], NULL, pairPtr, bool) 
	    != TCL_OK) {
	    return TCL_ERROR;
	}
    } else {
	if (GetColorPair(interp, tkwin, objv[0], objv[1], pairPtr, bool) 
	    != TCL_OK) {
	    return TCL_ERROR;
	}
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * NameOfColor --
 *
 *	Convert the color option value into a string.
 *
 * Results:
 *	The static string representing the color option is returned.
 *
 *----------------------------------------------------------------------
 */
static char *
NameOfColor(colorPtr)
    XColor *colorPtr;
{
    if (colorPtr == NULL) {
	return "";
    } else if (colorPtr == COLOR_DEFAULT) {
	return "defcolor";
    } else {
	return Tk_NameOfColor(colorPtr);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * ColorPairToObj --
 *
 *	Convert the color pairs into color names.
 *
 * Results:
 *	The string representing the symbol color is returned.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static Tcl_Obj *
ColorPairToObj(
    ClientData clientData,	/* Not used. */
    Tcl_Interp *interp,
    Tk_Window tkwin,		/* Not used. */
    char *widgRec,		/* Element information record */
    int offset,			/* Offset to field in structure */
    int flags)	
{
    ColorPair *pairPtr = (ColorPair *)(widgRec + offset);
    Tcl_Obj *listObjPtr;

    listObjPtr = Tcl_NewListObj(0, (Tcl_Obj **)NULL);
    Tcl_ListObjAppendElement(interp, listObjPtr, 
		Tcl_NewStringObj(NameOfColor(pairPtr->fgColor), -1));
    Tcl_ListObjAppendElement(interp, listObjPtr,
		Tcl_NewStringObj(NameOfColor(pairPtr->bgColor), -1));
    return listObjPtr;
}

/*
 * ----------------------------------------------------------------------
 *
 * HMap --
 *
 *	Map the given graph coordinate value to its axis, returning a
 *	window position.
 *
 * Results:
 *	Returns a floating point number representing the window
 *	coordinate position on the given axis.
 *
 * ---------------------------------------------------------------------- 
 */
static double
HMap(Graph *graphPtr, Axis *axisPtr, double x)
{
    double norm;

    if (x == DBL_MAX) {
	norm = 1.0;
    } else if (x == -DBL_MAX) {
	norm = 0.0;
    } else {
	if (axisPtr->logScale) {
	    if (x > 0.0) {
		x = log10(x);
	    } else if (x < 0.0) {
		x = 0.0;
	    }
	}
	norm = NORMALIZE(axisPtr, x);
    }
    if (axisPtr->descending) {
	norm = 1.0 - norm;
    }
    /* Horizontal transformation */
    return ((norm * graphPtr->hRange) + graphPtr->hOffset);
}

/*
 * ----------------------------------------------------------------------
 *
 * VMap --
 *
 *	Map the given graph coordinate value to its axis, returning a
 *	window position.
 *
 * Results:
 *	Returns a double precision number representing the window
 *	coordinate position on the given axis.
 *
 * ----------------------------------------------------------------------
 */
static double
VMap(Graph *graphPtr, Axis *axisPtr, double y)
{
    double norm;

    if (y == DBL_MAX) {
	norm = 1.0;
    } else if (y == -DBL_MAX) {
	norm = 0.0;
    } else {
	if (axisPtr->logScale) {
	    if (y > 0.0) {
		y = log10(y);
	    } else if (y < 0.0) {
		y = 0.0;
	    }
	}
	norm = NORMALIZE(axisPtr, y);
    }
    if (axisPtr->descending) {
	norm = 1.0 - norm;
    }
    /* Vertical transformation */
    return (((1.0 - norm) * graphPtr->vRange) + graphPtr->vOffset);
}

/*
 * ----------------------------------------------------------------------
 *
 * MapPoint --
 *
 *	Maps the given graph x,y coordinate values to a window position.
 *
 * Results:
 *	Returns a XPoint structure containing the window coordinates
 *	of the given graph x,y coordinate.
 *
 * ----------------------------------------------------------------------
 */
static Point2D
MapPoint(
    Graph *graphPtr,
    Point2D *pointPtr,		/* Graph X-Y coordinate. */
    Axis2D *axesPtr)		/* Specifies which axes to use */
{
    Point2D result;

    if (graphPtr->inverted) {
	result.x = HMap(graphPtr, axesPtr->y, pointPtr->y);
	result.y = VMap(graphPtr, axesPtr->x, pointPtr->x);
    } else {
	result.x = HMap(graphPtr, axesPtr->x, pointPtr->x);
	result.y = VMap(graphPtr, axesPtr->y, pointPtr->y);
    }
    return result;		/* Result is screen coordinate. */
}

static Marker *
CreateMarker(
    Graph *graphPtr,
    char *name,
    int classId)
{    
    Marker *markerPtr;

    /* Create the new marker based upon the given type */
    switch (classId) {
    case OBJECT_CLASS_BITMAP_MARKER:
	markerPtr = CreateBitmapMarker(); /* bitmap */
	break;
    case OBJECT_CLASS_LINE_MARKER:
	markerPtr = CreateLineMarker(); /* line */
	break;
    case OBJECT_CLASS_IMAGE_MARKER:
	markerPtr = CreateImageMarker(); /* image */
	break;
    case OBJECT_CLASS_TEXT_MARKER:
	markerPtr = CreateTextMarker(); /* text */
	break;
    case OBJECT_CLASS_POLYGON_MARKER:
	markerPtr = CreatePolygonMarker(); /* polygon */
	break;
    case OBJECT_CLASS_WINDOW_MARKER:
	markerPtr = CreateWindowMarker(); /* window */
	break;
    default:
	return NULL;
    }
    assert(markerPtr);
    markerPtr->object.graphPtr = graphPtr;
    markerPtr->object.hidden = markerPtr->drawUnder = FALSE;
    markerPtr->flags |= MAP_ITEM;
    markerPtr->object.name = Blt_Strdup(name);
    Blt_GraphSetObjectClass(&markerPtr->object, classId);
    return markerPtr;
}


static void
DestroyMarker(Marker *markerPtr)
{
    Graph *graphPtr = markerPtr->object.graphPtr;

    if (markerPtr->drawUnder) {
	/* If the marker to be deleted is currently displayed below
	 * the elements, then backing store needs to be repaired. */
	graphPtr->flags |= REDRAW_BACKING_STORE;
    }
    /* 
     * Call the marker's type-specific deallocation routine. We do it
     * first while all the marker fields are still valid.
     */
    (*markerPtr->classPtr->freeProc) (graphPtr, markerPtr);

    /* Dump any bindings that might be registered for the marker. */
    Blt_DeleteBindings(graphPtr->bindTable, markerPtr);

    /* Release all the X resources associated with the marker. */
    Blt_FreeOptions(markerPtr->classPtr->configSpecs, (char *)markerPtr,
	graphPtr->display, 0);

    if (markerPtr->hashPtr != NULL) {
	Blt_DeleteHashEntry(&graphPtr->markers.table, 
			    markerPtr->hashPtr);
    }
    if (markerPtr->linkPtr != NULL) {
	Blt_ChainDeleteLink(graphPtr->markers.displayList, markerPtr->linkPtr);
    }
    if (markerPtr->object.name != NULL) {
	Blt_Free(markerPtr->object.name);
    }
    Blt_Free(markerPtr);
}

/*
 * ----------------------------------------------------------------------
 *
 * ConfigureBitmapMarker --
 *
 *	This procedure is called to process an objv/objc list, plus
 *	the Tk option database, in order to configure (or reconfigure)
 *	a bitmap marker.
 *
 * Results:
 *	A standard Tcl result.  If TCL_ERROR is returned, then
 *	interp->result contains an error message.
 *
 * Side effects:
 *	Configuration information, such as bitmap pixmap, colors,
 *	rotation, etc. get set for markerPtr; old resources get freed,
 *	if there were any.  The marker is eventually redisplayed.
 *
 * ----------------------------------------------------------------------
 */
/* ARGSUSED */
static int
ConfigureBitmapMarker(Marker *markerPtr)
{
    Graph *graphPtr = markerPtr->object.graphPtr;
    BitmapMarker *bmPtr = (BitmapMarker *)markerPtr;
    GC newGC;
    XGCValues gcValues;
    unsigned long gcMask;

    if (bmPtr->srcBitmap == None) {
	return TCL_OK;
    }
    bmPtr->angle = FMOD(bmPtr->reqAngle, 360.0);
    if (bmPtr->angle < 0.0) {
	bmPtr->angle += 360.0;
    }
    gcMask = 0;

    if (bmPtr->outlineColor != NULL) {
	gcMask |= GCForeground;
	gcValues.foreground = bmPtr->outlineColor->pixel;
    }

    if (bmPtr->fillColor != NULL) {
	/* Opaque bitmap: both foreground and background (fill) colors
	 * are used. */
	gcValues.background = bmPtr->fillColor->pixel;
	gcMask |= GCBackground;
    } else {
	/* Transparent bitmap: set the clip mask to the current bitmap. */
	gcValues.clip_mask = bmPtr->srcBitmap;
	gcMask |= GCClipMask;
    }

    /* 
     * This is technically a shared GC, but we're going to set/change
     * the clip origin anyways before we draw the bitmap.  This relies
     * on the fact that no other client will be allocated this GC with
     * the GCClipMask set to this particular bitmap.
     */

    newGC = Tk_GetGC(graphPtr->tkwin, gcMask, &gcValues);
    if (bmPtr->gc != NULL) {
	Tk_FreeGC(graphPtr->display, bmPtr->gc);
    }
    bmPtr->gc = newGC;

    /* Create the background GC containing the fill color. */

    if (bmPtr->fillColor != NULL) {
	gcValues.foreground = bmPtr->fillColor->pixel;
	newGC = Tk_GetGC(graphPtr->tkwin, gcMask, &gcValues);
	if (bmPtr->fillGC != NULL) {
	    Tk_FreeGC(graphPtr->display, bmPtr->fillGC);
	}
	bmPtr->fillGC = newGC;
    }

    markerPtr->flags |= MAP_ITEM;
    if (markerPtr->drawUnder) {
	graphPtr->flags |= REDRAW_BACKING_STORE;
    }

    Blt_EventuallyRedrawGraph(graphPtr);
    return TCL_OK;
}

#ifdef notdef
static void
PrintPolyPoint(char *mesg, Point2D *points, int nPoints)
{
    int i;

    fprintf(stderr, "%s:\t\tpoint[0]=%g,%g\n", mesg, points[0].x, points[0].y);
    for (i = 1; i < nPoints; i++) {
	fprintf(stderr, "\t\tpoint[%d]=%g,%g\n", i, points[i].x, points[i].y);
    }
}	
#endif

/*
 * ----------------------------------------------------------------------
 *
 * MapBitmapMarker --
 *
 * 	This procedure gets called each time the layout of the graph
 *	changes.  The x, y window coordinates of the bitmap marker are
 *	saved in the marker structure.
 *
 *	Additionly, if no background color was specified, the
 *	GCTileStipXOrigin and GCTileStipYOrigin attributes are set in
 *	the private GC.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Window coordinates are saved and if no background color was
 * 	set, the GC stipple origins are changed to calculated window
 *	coordinates.
 *
 * ----------------------------------------------------------------------
 */
static void
MapBitmapMarker(markerPtr)
    Marker *markerPtr;
{
    BitmapMarker *bmPtr = (BitmapMarker *)markerPtr;
    Region2D extents;
    Graph *graphPtr = markerPtr->object.graphPtr;
    Point2D anchorPt;
    Point2D corner1, corner2;
    int destWidth, destHeight;
    int srcWidth, srcHeight;
    int i;

    if (bmPtr->srcBitmap == None) {
	return;
    }
    if (bmPtr->destBitmap != None) {
	Tk_FreePixmap(graphPtr->display, bmPtr->destBitmap);
	bmPtr->destBitmap = None;
    }
    /* 
     * Collect the coordinates.  The number of coordinates will determine
     * the calculations to be made.
     * 
     *	   x1 y1	A single pair of X-Y coordinates.  They represent
     *			the anchor position of the bitmap.  
     *
     *	x1 y1 x2 y2	Two pairs of X-Y coordinates.  They represent
     *			two opposite corners of a bounding rectangle. The
     *			bitmap is possibly rotated and scaled to fit into
     *			this box.
     *
     */   
    Tk_SizeOfBitmap(graphPtr->display, bmPtr->srcBitmap, &srcWidth, 
		    &srcHeight);
    corner1 = MapPoint(graphPtr, markerPtr->worldPts, &markerPtr->axes);
    if (markerPtr->nWorldPts > 1) {
	double hold;

	corner2 = MapPoint(graphPtr, markerPtr->worldPts + 1, &markerPtr->axes);
	/* Flip the corners if necessary */
	if (corner1.x > corner2.x) {
	    hold = corner1.x, corner1.x = corner2.x, corner2.x = hold;
	}
	if (corner1.y > corner2.y) {
	    hold = corner1.y, corner1.y = corner2.y, corner2.y = hold;
	}
    } else {
	corner2.x = corner1.x + srcWidth - 1;
	corner2.y = corner1.y + srcHeight - 1;
    }
    destWidth = (int)(corner2.x - corner1.x) + 1;
    destHeight = (int)(corner2.y - corner1.y) + 1;

    if (markerPtr->nWorldPts == 1) {
	anchorPt = Blt_AnchorPoint(corner1.x, corner1.y, (double)destWidth, 
		(double)destHeight, bmPtr->anchor);
    } else {
	anchorPt = corner1;
    }
    anchorPt.x += markerPtr->xOffset;
    anchorPt.y += markerPtr->yOffset;

    /* Check if the bitmap sits at least partially in the plot area. */
    extents.left   = anchorPt.x;
    extents.top    = anchorPt.y;
    extents.right  = anchorPt.x + destWidth - 1;
    extents.bottom = anchorPt.y + destHeight - 1;
    markerPtr->clipped = BoxesDontOverlap(graphPtr, &extents);
    if (markerPtr->clipped) {
	return;			/* Bitmap is offscreen. Don't generate
				 * rotated or scaled bitmaps. */
    }

    /*  
     * Scale the bitmap if necessary. It's a little tricky because we
     * only want to scale what's visible on the screen, not the entire
     * bitmap.  
     */
    if ((bmPtr->angle != 0.0) || (destWidth != srcWidth) || 
	(destHeight != srcHeight)) {
	int regionX, regionY, regionWidth, regionHeight; 
	double left, right, top, bottom;

	/*
	 * Ignore parts of the bitmap outside of the plot area.
	 */
	left   = MAX(graphPtr->left, extents.left);
	right  = MIN(graphPtr->right, extents.right);
	top    = MAX(graphPtr->top, extents.top);
	bottom = MIN(graphPtr->bottom, extents.bottom);

	/* 
	 * Determine the portion of the scaled bitmap to display. 
	 */
	regionX = regionY = 0;
	if (graphPtr->left > extents.left) {
	    regionX = (int)(graphPtr->left - extents.left);
	}
	if (graphPtr->top > extents.top) {
	    regionY = (int)(graphPtr->top - extents.top);
	}	    
	regionWidth = (int)(right - left) + 1;
	regionHeight = (int)(bottom - top) + 1;
	
	anchorPt.x = left;
	anchorPt.y = top;
	bmPtr->destBitmap = Blt_ScaleRotateBitmapArea(graphPtr->tkwin, 
		bmPtr->srcBitmap, srcWidth, srcHeight, regionX, regionY, 
		regionWidth, regionHeight, destWidth, destHeight, bmPtr->angle);
	bmPtr->destWidth = regionWidth;
	bmPtr->destHeight = regionHeight;
    } else {
	bmPtr->destWidth = srcWidth;
	bmPtr->destHeight = srcHeight;
	bmPtr->destBitmap = None;
    }
    bmPtr->anchorPt = anchorPt;
    {
	double xScale, yScale;
	double tx, ty;
	double rotWidth, rotHeight;
	Point2D polygon[5];
	int n;

	/* 
	 * Compute a polygon to represent the background area of the bitmap.  
	 * This is needed for backgrounds of arbitrarily rotated bitmaps.  
	 * We also use it to print a background in PostScript. 
	 */
	Blt_GetBoundingBox(srcWidth, srcHeight, bmPtr->angle, &rotWidth, 
			   &rotHeight, polygon);
	xScale = (double)destWidth / rotWidth;
	yScale = (double)destHeight / rotHeight;
	
	/* 
	 * Adjust each point of the polygon. Both scale it to the new size
	 * and translate it to the actual screen position of the bitmap.
	 */
	tx = extents.left + destWidth * 0.5;
	ty = extents.top + destHeight * 0.5;
	for (i = 0; i < 4; i++) {
	    polygon[i].x = (polygon[i].x * xScale) + tx;
	    polygon[i].y = (polygon[i].y * yScale) + ty;
	}
	Blt_GraphExtents(graphPtr, &extents);
	n = Blt_PolyRectClip(&extents, polygon, 4, bmPtr->outline); 
	assert(n <= MAX_OUTLINE_POINTS);
	if (n < 3) { 
	    memcpy(&bmPtr->outline, polygon, sizeof(Point2D) * 4);
	    bmPtr->nOutlinePts = 4;
	} else {
	    bmPtr->nOutlinePts = n;
	}
    }
}

/*
 * ----------------------------------------------------------------------
 *
 * PointInBitmapMarker --
 *
 *	Indicates if the given point is over the bitmap marker.  The
 *	area of the bitmap is the rectangle.
 *
 * Results:
 *	Returns 1 is the point is over the bitmap marker, 0 otherwise.
 *
 * ----------------------------------------------------------------------
 */
static int
PointInBitmapMarker(markerPtr, samplePtr)
    Marker *markerPtr;
    Point2D *samplePtr;
{
    BitmapMarker *bmPtr = (BitmapMarker *)markerPtr;

    if (bmPtr->srcBitmap == None) {
	return 0;
    }
    if (bmPtr->angle != 0.0) {
	Point2D points[MAX_OUTLINE_POINTS];
	int i;

	/*  
	 * Generate the bounding polygon (isolateral) for the bitmap
	 * and see if the point is inside of it.  
	 */
	for (i = 0; i < bmPtr->nOutlinePts; i++) {
	    points[i].x = bmPtr->outline[i].x + bmPtr->anchorPt.x;
	    points[i].y = bmPtr->outline[i].y + bmPtr->anchorPt.y;
	}
	return Blt_PointInPolygon(samplePtr, points, bmPtr->nOutlinePts);
    }
    return ((samplePtr->x >= bmPtr->anchorPt.x) && 
	    (samplePtr->x < (bmPtr->anchorPt.x + bmPtr->destWidth)) &&
	    (samplePtr->y >= bmPtr->anchorPt.y) && 
	    (samplePtr->y < (bmPtr->anchorPt.y + bmPtr->destHeight)));
}


/*
 * ----------------------------------------------------------------------
 *
 * RegionInBitmapMarker --
 *
 * ----------------------------------------------------------------------
 */
static int
RegionInBitmapMarker(Marker *markerPtr, Region2D *extsPtr, int enclosed)
{
    BitmapMarker *bmPtr = (BitmapMarker *)markerPtr;

    if (markerPtr->nWorldPts < 1) {
	return FALSE;
    }
    if (bmPtr->angle != 0.0) {
	Point2D points[MAX_OUTLINE_POINTS];
	int i;
	
	/*  
	 * Generate the bounding polygon (isolateral) for the bitmap
	 * and see if the point is inside of it.  
	 */
	for (i = 0; i < bmPtr->nOutlinePts; i++) {
	    points[i].x = bmPtr->outline[i].x + bmPtr->anchorPt.x;
	    points[i].y = bmPtr->outline[i].y + bmPtr->anchorPt.y;
	}
	return Blt_RegionInPolygon(extsPtr, points, bmPtr->nOutlinePts, 
		   enclosed);
    }
    if (enclosed) {
	return ((bmPtr->anchorPt.x >= extsPtr->left) &&
		(bmPtr->anchorPt.y >= extsPtr->top) && 
		((bmPtr->anchorPt.x + bmPtr->destWidth) <= extsPtr->right) &&
		((bmPtr->anchorPt.y + bmPtr->destHeight) <= extsPtr->bottom));
    }
    return !((bmPtr->anchorPt.x >= extsPtr->right) ||
	     (bmPtr->anchorPt.y >= extsPtr->bottom) ||
	     ((bmPtr->anchorPt.x + bmPtr->destWidth) <= extsPtr->left) ||
	     ((bmPtr->anchorPt.y + bmPtr->destHeight) <= extsPtr->top));
}

/*
 * ----------------------------------------------------------------------
 *
 * DrawBitmapMarker --
 *
 *	Draws the bitmap marker that have a transparent of filled
 *	background.  
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	GC stipple origins are changed to current window coordinates.
 *	Commands are output to X to draw the marker in its current
 *	mode.
 *
 * ----------------------------------------------------------------------
 */
static void
DrawBitmapMarker(Marker *markerPtr, Drawable drawable)
{
    Graph *graphPtr = markerPtr->object.graphPtr;
    BitmapMarker *bmPtr = (BitmapMarker *)markerPtr;
    double rangle;
    Pixmap bitmap;

    bitmap = GETBITMAP(bmPtr);
    if ((bitmap == None) || (bmPtr->destWidth < 1) || (bmPtr->destHeight < 1)) {
	return;
    }
    rangle = FMOD(bmPtr->angle, (double)90.0);
    if ((bmPtr->fillColor == NULL) || (rangle != 0.0)) {

	/* 
	 * If the bitmap is rotated and a filled background is
	 * required, then a filled polygon is drawn before the
	 * bitmap. 
	 */

	if (bmPtr->fillColor != NULL) {
	    int i;
	    XPoint polygon[MAX_OUTLINE_POINTS];

	    for (i = 0; i < bmPtr->nOutlinePts; i++) {
		polygon[i].x = (short int)bmPtr->outline[i].x;
		polygon[i].y = (short int)bmPtr->outline[i].y;
	    }
	    XFillPolygon(graphPtr->display, drawable, bmPtr->fillGC,
		 polygon, bmPtr->nOutlinePts, Convex, CoordModeOrigin);
	}
	XSetClipMask(graphPtr->display, bmPtr->gc, bitmap);
	XSetClipOrigin(graphPtr->display, bmPtr->gc, (int)bmPtr->anchorPt.x, 
	       (int)bmPtr->anchorPt.y);
    } else {
	XSetClipMask(graphPtr->display, bmPtr->gc, None);
	XSetClipOrigin(graphPtr->display, bmPtr->gc, 0, 0);
    }
    XCopyPlane(graphPtr->display, bitmap, drawable, bmPtr->gc, 0, 0,
	bmPtr->destWidth, bmPtr->destHeight, (int)bmPtr->anchorPt.x, 
	(int)bmPtr->anchorPt.y, 1);
}

/*
 * ----------------------------------------------------------------------
 *
 * BitmapMarkerToPostScript --
 *
 *	Generates PostScript to print a bitmap marker.
 *
 * Results:
 *	None.
 *
 * ----------------------------------------------------------------------
 */
static void
BitmapMarkerToPostScript(Marker *markerPtr, Blt_PostScript ps)
{
    Graph *graphPtr = markerPtr->object.graphPtr;
    BitmapMarker *bmPtr = (BitmapMarker *)markerPtr;
    Pixmap bitmap;

    bitmap = GETBITMAP(bmPtr);
    if ((bitmap == None) || (bmPtr->destWidth < 1) || (bmPtr->destHeight < 1)) {
	return;			/* No bitmap to display. */
    }
    if (bmPtr->fillColor != NULL) {
	Blt_BackgroundToPostScript(ps, bmPtr->fillColor);
	Blt_PolygonToPostScript(ps, bmPtr->outline, 4);
    }
    Blt_ForegroundToPostScript(ps, bmPtr->outlineColor);

    Blt_FormatToPostScript(ps,
	"  gsave\n    %g %g translate\n    %d %d scale\n", 
	   bmPtr->anchorPt.x, bmPtr->anchorPt.y + bmPtr->destHeight, 
	   bmPtr->destWidth, -bmPtr->destHeight);
    Blt_FormatToPostScript(ps, "    %d %d true [%d 0 0 %d 0 %d] {",
	bmPtr->destWidth, bmPtr->destHeight, bmPtr->destWidth, 
	-bmPtr->destHeight, bmPtr->destHeight);
    Blt_BitmapDataToPostScript(ps, graphPtr->display, bitmap,
	bmPtr->destWidth, bmPtr->destHeight);
    Blt_AppendToPostScript(ps, "    } imagemask\n",
	"grestore\n", (char *)NULL);
}

/*
 * ----------------------------------------------------------------------
 *
 * FreeBitmapMarker --
 *
 *	Releases the memory and attributes of the bitmap marker.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Bitmap attributes (GCs, colors, bitmap, etc) get destroyed.
 *	Memory is released, X resources are freed, and the graph is
 *	redrawn.
 *
 * ----------------------------------------------------------------------
 */
static void
FreeBitmapMarker(Graph *graphPtr, Marker *markerPtr)
{
    BitmapMarker *bmPtr = (BitmapMarker *)markerPtr;

    if (bmPtr->gc != NULL) {
	Tk_FreeGC(graphPtr->display, bmPtr->gc);
    }
    if (bmPtr->fillGC != NULL) {
	Tk_FreeGC(graphPtr->display, bmPtr->fillGC);
    }
    if (bmPtr->destBitmap != None) {
	Tk_FreePixmap(graphPtr->display, bmPtr->destBitmap);
    }
}

/*
 * ----------------------------------------------------------------------
 *
 * CreateBitmapMarker --
 *
 *	Allocate memory and initialize methods for the new bitmap marker.
 *
 * Results:
 *	The pointer to the newly allocated marker structure is returned.
 *
 * Side effects:
 *	Memory is allocated for the bitmap marker structure.
 *
 * ----------------------------------------------------------------------
 */
static Marker *
CreateBitmapMarker()
{
    BitmapMarker *bmPtr;

    bmPtr = Blt_Calloc(1, sizeof(BitmapMarker));
    if (bmPtr != NULL) {
	bmPtr->base.classPtr = &bitmapMarkerClass;
    }
    return (Marker *)bmPtr;
}

/*
 *----------------------------------------------------------------------
 *
 * ImageChangedProc
 *
 *
 * Results:
 *	None.
 *
 *----------------------------------------------------------------------
 */
/* ARGSUSED */
static void
ImageChangedProc(
    ClientData clientData,
    int x, int y,		/* Not used. */
    int width, int height,	/* Not used. */
    int imageWidth, int imageHeight) /* Not used. */
{
    Blt_Picture picture;
    Graph *graphPtr;
    ImageMarker *imPtr = clientData;
    Tk_PhotoHandle photo;

    graphPtr = imPtr->base.object.graphPtr;
    photo = Tk_FindPhoto(graphPtr->interp, imPtr->imageName);
    if (photo == NULL) {
	picture = Blt_PhotoToPicture(photo);
    } else if (Blt_GetPicture(graphPtr->interp, imPtr->imageName, &picture) 
	       != TCL_OK) {
	return;			/* It's neither a photo nor a picture. */
    }
    if (imPtr->original != NULL) {
	Blt_FreePicture(imPtr->original);
    }
    imPtr->original = picture;
    graphPtr->flags |= REDRAW_BACKING_STORE;
    imPtr->base.flags |= MAP_ITEM;
    Blt_EventuallyRedrawGraph(graphPtr);
}

/*
 * ----------------------------------------------------------------------
 *
 * ConfigureImageMarker --
 *
 *	This procedure is called to process an objv/objc list, plus
 *	the Tk option database, in order to configure (or reconfigure)
 *	a image marker.
 *
 * Results:
 *	A standard Tcl result.  If TCL_ERROR is returned, then
 *	interp->result contains an error message.
 *
 * Side effects:
 *	Configuration information, such as image pixmap, colors,
 *	rotation, etc. get set for markerPtr; old resources get freed,
 *	if there were any.  The marker is eventually redisplayed.
 *
 * ----------------------------------------------------------------------
 */
static int
ConfigureImageMarker(Marker *markerPtr)
{
    ImageMarker *imPtr = (ImageMarker *)markerPtr;
    Graph *graphPtr = markerPtr->object.graphPtr;
    Blt_Painter painter;

    if (Blt_ConfigModified(markerPtr->classPtr->configSpecs, "-image", 
			   (char *)NULL)) {
	Tcl_Interp *interp = graphPtr->interp;

	if (imPtr->image != NULL) {
	    Tk_FreeImage(imPtr->image);
	    imPtr->image = NULL;
	}
	if (imPtr->imageName != NULL) {
	    GC newGC;
	    Tk_PhotoHandle photo;
	    Blt_Picture picture;
	    Tk_Image image;

	    image = Tk_GetImage(interp, graphPtr->tkwin, imPtr->imageName, 
		ImageChangedProc, imPtr);
	    if (image == NULL) {
		Blt_Free(imPtr->imageName);
		imPtr->imageName = NULL;
		return TCL_ERROR;
	    }
	    imPtr->image = image;
	    photo = Tk_FindPhoto(interp, imPtr->imageName);
	    if (photo != NULL) {
		picture = Blt_PhotoToPicture(photo);
	    } else if (Blt_GetPicture(interp, imPtr->imageName, &picture) 
		       != TCL_OK) {
		return TCL_ERROR;
	    }
	    if (imPtr->original != NULL) {
		Blt_FreePicture(imPtr->original);
	    }
	    /* Convert the photo into a picture */
	    imPtr->original = picture;

	    newGC = Tk_GetGC(graphPtr->tkwin, 0L, (XGCValues *)NULL);
	    if (imPtr->gc != NULL) {
		Tk_FreeGC(graphPtr->display, imPtr->gc);
	    }
	    imPtr->gc = newGC;
	}
    }
    painter = Blt_GetPainter(graphPtr->tkwin, 1.0);
    if (imPtr->painter != NULL) {
	Blt_FreePainter(painter);
    }
    imPtr->painter = painter;
    markerPtr->flags |= MAP_ITEM;
    if (markerPtr->drawUnder) {
	graphPtr->flags |= REDRAW_BACKING_STORE;
    }
    Blt_EventuallyRedrawGraph(graphPtr);
    return TCL_OK;
}

/*
 * ----------------------------------------------------------------------
 *
 * MapImageMarker --
 *
 * 	This procedure gets called each time the layout of the graph
 *	changes.  The x, y window coordinates of the image marker are
 *	saved in the marker structure.
 *
 *	In addition, if no background color was specified, the
 *	GCTileStipXOrigin and GCTileStipYOrigin attributes will not
 *	set in the private GC.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Window coordinates are saved and if no background color was 
 *	set, the GC stipple origins are changed to calculated window
 *	coordinates.
 *
 * ----------------------------------------------------------------------
 */
static void
MapImageMarker(Marker *markerPtr)
{
    Region2D extents;
    Graph *graphPtr;
    ImageMarker *imPtr;
    Point2D anchorPt;
    Point2D c1, c2;
    int newWidth, newHeight;
    int srcWidth, srcHeight;
    int x, y, width, height;
    int left, right, top, bottom;

    imPtr = (ImageMarker *)markerPtr;
    if ((imPtr->image == NULL) || (imPtr->original == NULL)) {
	return;
    }
    if (imPtr->scaled != NULL) { 
	Blt_FreePicture(imPtr->scaled);
	imPtr->scaled = NULL;
    }
    graphPtr = markerPtr->object.graphPtr;
    c1 = MapPoint(graphPtr, markerPtr->worldPts, &markerPtr->axes);

    imPtr->width = srcWidth = Blt_PictureWidth(imPtr->original);
    imPtr->height = srcHeight = Blt_PictureHeight(imPtr->original);

    if ((srcWidth == 0) || (srcHeight == 0)) {
	markerPtr->clipped = TRUE;
	return;			/* Empty image. */
    }
    if (markerPtr->nWorldPts > 1) {
	double hold;

	c2 = MapPoint(graphPtr, markerPtr->worldPts + 1, &markerPtr->axes);
	/* Flip the corners if necessary */
	if (c1.x > c2.x) {
	    hold = c1.x, c1.x = c2.x, c2.x = hold;
	}
	if (c1.y > c2.y) {
	    hold = c1.y, c1.y = c2.y, c2.y = hold;
	}
    } else {
	c2.x = c1.x + srcWidth - 1;
	c2.y = c1.y + srcHeight - 1;
    }
    newWidth = (int)(c2.x - c1.x) + 1;
    newHeight = (int)(c2.y - c1.y) + 1;

    if (markerPtr->nWorldPts == 1) {
	anchorPt = Blt_AnchorPoint(c1.x, c1.y, (double)newWidth, 
		(double)newHeight, imPtr->anchor);
    } else {
	anchorPt = c1;
    }
    anchorPt.x += markerPtr->xOffset;
    anchorPt.y += markerPtr->yOffset;

    /* Check if the image sits at least partially in the plot area. */
    extents.left   = anchorPt.x;
    extents.top    = anchorPt.y;
    extents.right  = anchorPt.x + newWidth - 1;
    extents.bottom = anchorPt.y + newHeight - 1;

    markerPtr->clipped = BoxesDontOverlap(graphPtr, &extents);
    if (markerPtr->clipped) {
	return;			/* Image is offscreen. Don't generate
				 * rotated or scaled images. */
    }

    /* Determine the extents of the subimage inside of the destination
     * image. */
    left =   MAX((int)extents.left, graphPtr->left);
    top =    MAX((int)extents.top, graphPtr->top);
    right =  MIN((int)extents.right, graphPtr->right);
    bottom = MIN((int)extents.bottom, graphPtr->bottom);
    
    /* Reset image location and coordinates to that of the region */
    anchorPt.x = left;
    anchorPt.y = top;
    
    x = y = 0;
    if (graphPtr->left > (int)extents.left) {
	x = graphPtr->left - (int)extents.left;
    } 
    if (graphPtr->top > (int)extents.top) {
	y = graphPtr->top - (int)extents.top;
    } 
    width  = (int)(right - left + 1);
    height = (int)(bottom - top + 1);
    
    if (markerPtr->nWorldPts > 1) {
	Blt_Picture scaled;

	scaled = Blt_ScalePictureArea(imPtr->original, x, y, width, height, 
		newWidth, newHeight);
	imPtr->scaled = scaled;
	imPtr->pictX = 0;
	imPtr->pictY = 0;
    } else {
	imPtr->pictX = x;
	imPtr->pictY = y;
    }
    imPtr->width = width;
    imPtr->height = height;
    imPtr->anchorPt = anchorPt;
}

/*
 * ----------------------------------------------------------------------
 *
 * PointInWindowMarker --
 *
 *	Indicates if the given point is over the window marker.  The
 *	area of the window is the rectangle.
 *
 * Results:
 *	Returns 1 is the point is over the window marker, 0 otherwise.
 *
 * ----------------------------------------------------------------------
 */
static int
PointInImageMarker(Marker *markerPtr, Point2D *samplePtr)
{
    ImageMarker *imPtr = (ImageMarker *)markerPtr;
    float left, right, top, bottom;
    
    left = imPtr->anchorPt.x;
    right = imPtr->anchorPt.x + imPtr->width;
    top = imPtr->anchorPt.y;
    bottom = imPtr->anchorPt.y + imPtr->height;

    return ((samplePtr->x >= left) && (samplePtr->x < right) &&
	    (samplePtr->y >= top) && (samplePtr->y < bottom));
}

/*
 * ----------------------------------------------------------------------
 *
 * RegionInImageMarker --
 *
 * ----------------------------------------------------------------------
 */
static int
RegionInImageMarker(Marker *markerPtr, Region2D *regPtr, int enclosed)
{
    ImageMarker *imPtr = (ImageMarker *)markerPtr;

    if (markerPtr->nWorldPts > 0) {
	float left, right, top, bottom;

	left = imPtr->anchorPt.x;
	right = imPtr->anchorPt.x + imPtr->width;
	top = imPtr->anchorPt.y;
	bottom = imPtr->anchorPt.y + imPtr->height;
	if (enclosed) {
	    return ((left >= regPtr->left) && (top >= regPtr->top) && 
		    (right <= regPtr->right) && (bottom <= regPtr->bottom));
	} 
	return !((left >= regPtr->right) || (top >= regPtr->bottom) ||
		 (right <= regPtr->left) || (bottom <= regPtr->top));
    }
    return FALSE;
}

/*
 * ----------------------------------------------------------------------
 *
 * DrawImageMarker --
 *
 *	This procedure is invoked to draw a image marker.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	GC stipple origins are changed to current window coordinates.
 *	Commands are output to X to draw the marker in its current
 *	mode.
 *
 * ----------------------------------------------------------------------
 */
static void
DrawImageMarker(Marker *markerPtr, Drawable drawable)
{
    ImageMarker *imPtr = (ImageMarker *)markerPtr;
    Blt_Picture pict;

    if ((imPtr->image == NULL) || (Tk_ImageIsDeleted(imPtr->image))) {
	return;
    }
    pict = (imPtr->scaled != NULL) ? imPtr->scaled : imPtr->original;
    if (pict != NULL) {
	Blt_PaintPictureWithBlend(imPtr->painter, drawable, pict, 
		imPtr->pictX, imPtr->pictY, imPtr->width, imPtr->height, 
		(int)imPtr->anchorPt.x, (int)imPtr->anchorPt.y, 0, 0.4);
    }
}

/*
 * ----------------------------------------------------------------------
 *
 * ImageMarkerToPostScript --
 *
 *	This procedure is invoked to print a image marker.
 *
 * Results:
 *	None.
 *
 * ----------------------------------------------------------------------
 */
static void
ImageMarkerToPostScript(Marker *markerPtr, Blt_PostScript ps)
{
    ImageMarker *imPtr = (ImageMarker *)markerPtr;
    Blt_Picture pict;

    if ((imPtr->image == NULL) || (Tk_ImageIsDeleted(imPtr->image))) {
	return;			/* Image doesn't exist anymore */
    }
    pict = (imPtr->scaled != NULL) ? imPtr->scaled : imPtr->original;
    if (pict != NULL) {
	Blt_PictureToPostScript(ps, pict, imPtr->anchorPt.x, imPtr->anchorPt.y);
    }
}

/*
 * ----------------------------------------------------------------------
 *
 * FreeImageMarker --
 *
 *	Destroys the structure containing the attributes of the image
 * 	marker.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Image attributes (GCs, colors, image, etc) get destroyed.
 *	Memory is released, X resources are freed, and the graph is
 *	redrawn.
 *
 * ----------------------------------------------------------------------
 */
static void
FreeImageMarker(Graph *graphPtr, Marker *markerPtr)
{
    ImageMarker *imPtr = (ImageMarker *)markerPtr;

    if (imPtr->image != NULL) {
	Tk_FreeImage(imPtr->image);
    }
    if (imPtr->painter != NULL) {
	Blt_FreePainter(imPtr->painter);
    }
    if (imPtr->original != NULL) {
	Blt_FreePicture(imPtr->original);
    }
    if (imPtr->scaled != NULL) {
	Blt_FreePicture(imPtr->scaled);
    }
    if (imPtr->gc != NULL) {
	Tk_FreeGC(graphPtr->display, imPtr->gc);
    }
}

/*
 * ----------------------------------------------------------------------
 *
 * CreateImageMarker --
 *
 *	Allocate memory and initialize methods for the new image marker.
 *
 * Results:
 *	The pointer to the newly allocated marker structure is returned.
 *
 * Side effects:
 *	Memory is allocated for the image marker structure.
 *
 * ----------------------------------------------------------------------
 */
static Marker *
CreateImageMarker()
{
    ImageMarker *imPtr;

    imPtr = Blt_Calloc(1, sizeof(ImageMarker));
    if (imPtr != NULL) {
	imPtr->base.classPtr = &imageMarkerClass;
    }
    return (Marker *)imPtr;
}

/*
 * ----------------------------------------------------------------------
 *
 * ConfigureTextMarker --
 *
 *	This procedure is called to process an objv/objc list, plus
 *	the Tk option database, in order to configure (or reconfigure)
 *	a text marker.
 *
 * Results:
 *	A standard Tcl result.  If TCL_ERROR is returned, then
 *	interp->result contains an error message.
 *
 * Side effects:
 *	Configuration information, such as text string, colors, font,
 *	etc. get set for markerPtr; old resources get freed, if there
 *	were any.  The marker is eventually redisplayed.
 *
 * ----------------------------------------------------------------------
 */
static int
ConfigureTextMarker(Marker *markerPtr)
{
    Graph *graphPtr = markerPtr->object.graphPtr;
    TextMarker *tmPtr = (TextMarker *)markerPtr;
    GC newGC;
    XGCValues gcValues;
    unsigned long gcMask;

    tmPtr->style.angle = (float)FMOD(tmPtr->style.angle, 360.0);
    if (tmPtr->style.angle < 0.0) {
	tmPtr->style.angle += 360.0;
    }
    newGC = NULL;
    if (tmPtr->fillColor != NULL) {
	gcMask = GCForeground;
	gcValues.foreground = tmPtr->fillColor->pixel;
	newGC = Tk_GetGC(graphPtr->tkwin, gcMask, &gcValues);
    }
    if (tmPtr->fillGC != NULL) {
	Tk_FreeGC(graphPtr->display, tmPtr->fillGC);
    }
    tmPtr->fillGC = newGC;

    if (Blt_ConfigModified(markerPtr->classPtr->configSpecs, "-text", 
	(char *)NULL)) {
	if (tmPtr->textPtr != NULL) {
	    Blt_Free(tmPtr->textPtr);
	    tmPtr->textPtr = NULL;
	}
	tmPtr->width = tmPtr->height = 0;
	if (tmPtr->string != NULL) {
	    int i;
	    double rotWidth, rotHeight;

	    tmPtr->textPtr = Blt_TextCreateLayout(tmPtr->string, &tmPtr->style);
	    Blt_GetBoundingBox(tmPtr->textPtr->width, tmPtr->textPtr->height, 
	       tmPtr->style.angle, &rotWidth, &rotHeight, tmPtr->outline);
	    tmPtr->width = ROUND(rotWidth);
	    tmPtr->height = ROUND(rotHeight);
	    for (i = 0; i < 4; i++) {
		tmPtr->outline[i].x += ROUND(rotWidth * 0.5);
		tmPtr->outline[i].y += ROUND(rotHeight * 0.5);
	    }
	    tmPtr->outline[4].x = tmPtr->outline[0].x;
	    tmPtr->outline[4].y = tmPtr->outline[0].y;
	}
    }
    markerPtr->flags |= MAP_ITEM;
    if (markerPtr->drawUnder) {
	graphPtr->flags |= REDRAW_BACKING_STORE;
    }
    Blt_EventuallyRedrawGraph(graphPtr);
    return TCL_OK;
}

/*
 * ----------------------------------------------------------------------
 *
 * MapTextMarker --
 *
 *	Calculate the layout position for a text marker.  Positional
 *	information is saved in the marker.  If the text is rotated, a
 *	bitmap containing the text is created.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	If no background color has been specified, the GC stipple
 *	origins are changed to current window coordinates. For both
 *	rotated and non-rotated text, if any old bitmap is leftover,
 *	it is freed.
 *
 * ----------------------------------------------------------------------
 */
static void
MapTextMarker(Marker *markerPtr)
{
    Graph *graphPtr = markerPtr->object.graphPtr;
    TextMarker *tmPtr = (TextMarker *)markerPtr;
    Region2D extents;
    Point2D anchorPt;

    if (tmPtr->string == NULL) {
	return;
    }
    anchorPt = MapPoint(graphPtr, markerPtr->worldPts, &markerPtr->axes);
    anchorPt = Blt_AnchorPoint(anchorPt.x, anchorPt.y, (double)(tmPtr->width), 
	(double)(tmPtr->height), tmPtr->anchor);
    anchorPt.x += markerPtr->xOffset;
    anchorPt.y += markerPtr->yOffset;
    /*
     * Determine the bounding box of the text and test to see if it is
     * at least partially contained within the plotting area.
     */
    extents.left = anchorPt.x;
    extents.top = anchorPt.y;
    extents.right = anchorPt.x + tmPtr->width - 1;
    extents.bottom = anchorPt.y + tmPtr->height - 1;
    markerPtr->clipped = BoxesDontOverlap(graphPtr, &extents);
    tmPtr->anchorPt = anchorPt;

}

static int
PointInTextMarker(Marker *markerPtr, Point2D *samplePtr)
{
    TextMarker *tmPtr = (TextMarker *)markerPtr;

    if (tmPtr->string == NULL) {
	return 0;
    }
    if (tmPtr->style.angle != 0.0) {
	Point2D points[5];
	int i;

	/* 
	 * Figure out the bounding polygon (isolateral) for the text
	 * and see if the point is inside of it.
	 */
	for (i = 0; i < 5; i++) {
	    points[i].x = tmPtr->outline[i].x + tmPtr->anchorPt.x;
	    points[i].y = tmPtr->outline[i].y + tmPtr->anchorPt.y;
	}
	return Blt_PointInPolygon(samplePtr, points, 5);
    } 
    return ((samplePtr->x >= tmPtr->anchorPt.x) && 
	    (samplePtr->x < (tmPtr->anchorPt.x + tmPtr->width)) &&
	    (samplePtr->y >= tmPtr->anchorPt.y) && 
	    (samplePtr->y < (tmPtr->anchorPt.y + tmPtr->height)));
}

/*
 * ----------------------------------------------------------------------
 *
 * RegionInTextMarker --
 *
 * ----------------------------------------------------------------------
 */
static int
RegionInTextMarker(Marker *markerPtr, Region2D *extsPtr, int enclosed)
{
    TextMarker *tmPtr = (TextMarker *)markerPtr;

    if (markerPtr->nWorldPts < 1) {
	return FALSE;
    }
    if (tmPtr->style.angle != 0.0) {
	Point2D points[5];
	int i;
	
	/*  
	 * Generate the bounding polygon (isolateral) for the bitmap
	 * and see if the point is inside of it.  
	 */
	for (i = 0; i < 4; i++) {
	    points[i].x = tmPtr->outline[i].x + tmPtr->anchorPt.x;
	    points[i].y = tmPtr->outline[i].y + tmPtr->anchorPt.y;
	}
	return Blt_RegionInPolygon(extsPtr, points, 4, enclosed);
    } 
    if (enclosed) {
	return ((tmPtr->anchorPt.x >= extsPtr->left) &&
		(tmPtr->anchorPt.y >= extsPtr->top) && 
		((tmPtr->anchorPt.x + tmPtr->width) <= extsPtr->right) &&
		((tmPtr->anchorPt.y + tmPtr->height) <= extsPtr->bottom));
    }
    return !((tmPtr->anchorPt.x >= extsPtr->right) ||
	     (tmPtr->anchorPt.y >= extsPtr->bottom) ||
	     ((tmPtr->anchorPt.x + tmPtr->width) <= extsPtr->left) ||
	     ((tmPtr->anchorPt.y + tmPtr->height) <= extsPtr->top));
}

/*
 * ----------------------------------------------------------------------
 *
 * DrawTextMarker --
 *
 *	Draws the text marker on the graph.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Commands are output to X to draw the marker in its current
 *	mode.
 *
 * ----------------------------------------------------------------------
 */
static void
DrawTextMarker(Marker *markerPtr, Drawable drawable) 
{
    TextMarker *tmPtr = (TextMarker *)markerPtr;
    Graph *graphPtr = markerPtr->object.graphPtr;

    if (tmPtr->string == NULL) {
	return;
    }
    if (tmPtr->fillGC != NULL) {
	XPoint pointArr[4];
	int i;

	/*
	 * Simulate the rotated background of the bitmap by filling a
	 * bounding polygon with the background color.
	 */
	for (i = 0; i < 4; i++) {
	    pointArr[i].x = (short int)
		(tmPtr->outline[i].x + tmPtr->anchorPt.x);
	    pointArr[i].y = (short int)
		(tmPtr->outline[i].y + tmPtr->anchorPt.y);
	}
	XFillPolygon(graphPtr->display, drawable, tmPtr->fillGC, pointArr, 4,
	    Convex, CoordModeOrigin);
    }
    if (tmPtr->style.color != NULL) {
	Blt_TextDrawLayout(graphPtr->tkwin, drawable, tmPtr->textPtr,
	    &tmPtr->style, (int)tmPtr->anchorPt.x, (int)tmPtr->anchorPt.y, -1);
    }
}

/*
 * ----------------------------------------------------------------------
 *
 * TextMarkerToPostScript --
 *
 *	Outputs PostScript commands to draw a text marker at a given
 *	x,y coordinate, rotation, anchor, and font.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	PostScript font and color settings are changed.
 *
 * ----------------------------------------------------------------------
 */
static void
TextMarkerToPostScript(Marker *markerPtr, Blt_PostScript ps)
{
    TextMarker *tmPtr = (TextMarker *)markerPtr;

    if (tmPtr->string == NULL) {
	return;
    }
    if (tmPtr->fillGC != NULL) {
	Point2D polygon[4];
	int i;

	/*
	 * Simulate the rotated background of the bitmap by filling a
	 * bounding polygon with the background color.
	 */
	for (i = 0; i < 4; i++) {
	    polygon[i].x = tmPtr->outline[i].x + tmPtr->anchorPt.x;
	    polygon[i].y = tmPtr->outline[i].y + tmPtr->anchorPt.y;
	}
	Blt_BackgroundToPostScript(ps, tmPtr->fillColor);
	Blt_PolygonToPostScript(ps, polygon, 4);
    }
    Blt_TextToPostScript(ps, tmPtr->string, &tmPtr->style, tmPtr->anchorPt.x, 
	tmPtr->anchorPt.y);
}

/*
 * ----------------------------------------------------------------------
 *
 * FreeTextMarker --
 *
 *	Destroys the structure containing the attributes of the text
 * 	marker.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Text attributes (GCs, colors, stipple, font, etc) get
 *	destroyed.  Memory is released, X resources are freed, and the
 *	graph is redrawn.
 *
 * ----------------------------------------------------------------------
 */
static void
FreeTextMarker(Graph *graphPtr, Marker *markerPtr)
{
    TextMarker *tmPtr = (TextMarker *)markerPtr;

    Blt_TextFreeStyle(graphPtr->display, &tmPtr->style);
    if (tmPtr->textPtr != NULL) {
	Blt_Free(tmPtr->textPtr);
    }
}

/*
 * ----------------------------------------------------------------------
 *
 * CreateTextMarker --
 *
 *	Allocate memory and initialize methods for the new text marker.
 *
 * Results:
 *	The pointer to the newly allocated marker structure is returned.
 *
 * Side effects:
 *	Memory is allocated for the text marker structure.
 *
 * ----------------------------------------------------------------------
 */
static Marker *
CreateTextMarker()
{
    TextMarker *tmPtr;

    tmPtr = Blt_Calloc(1, sizeof(TextMarker));
    assert(tmPtr);

    tmPtr->base.classPtr = &textMarkerClass;
    Blt_TextInitStyle(tmPtr->style);
    tmPtr->style.anchor = TK_ANCHOR_NW;
    tmPtr->style.padLeft = tmPtr->style.padRight = 4;
    tmPtr->style.padTop = tmPtr->style.padBottom = 4;

    return (Marker *)tmPtr;
}

static Tk_EventProc ChildEventProc;
static Tk_GeomRequestProc ChildGeometryProc;
static Tk_GeomLostSlaveProc ChildCustodyProc;
static Tk_GeomMgr winMarkerMgrInfo =
{
    "graph",			/* Name of geometry manager used by winfo */
    ChildGeometryProc,		/* Procedure to for new geometry requests */
    ChildCustodyProc,		/* Procedure when window is taken away */
};

/*
 * ----------------------------------------------------------------------
 *
 * ConfigureWindowMarker --
 *
 *	This procedure is called to process an objv/objc list, plus
 *	the Tk option database, in order to configure (or reconfigure)
 *	a window marker.
 *
 * Results:
 *	A standard Tcl result.  If TCL_ERROR is returned, then
 *	interp->result contains an error message.
 *
 * Side effects:
 *	Configuration information, such as window pathname, placement,
 *	etc. get set for markerPtr; old resources get freed, if there
 *	were any.  The marker is eventually redisplayed.
 *
 * ----------------------------------------------------------------------
 */
static int
ConfigureWindowMarker(markerPtr)
    Marker *markerPtr;
{
    Graph *graphPtr = markerPtr->object.graphPtr;
    WindowMarker *wmPtr = (WindowMarker *)markerPtr;
    Tk_Window tkwin;

    if (wmPtr->pathName == NULL) {
	return TCL_OK;
    }
    tkwin = Tk_NameToWindow(graphPtr->interp, wmPtr->pathName, 
	    graphPtr->tkwin);
    if (tkwin == NULL) {
	return TCL_ERROR;
    }
    if (Tk_Parent(tkwin) != graphPtr->tkwin) {
	Tcl_AppendResult(graphPtr->interp, "\"", wmPtr->pathName,
	    "\" is not a child of \"", Tk_PathName(graphPtr->tkwin), "\"",
	    (char *)NULL);
	return TCL_ERROR;
    }
    if (tkwin != wmPtr->tkwin) {
	if (wmPtr->tkwin != NULL) {
	    Tk_DeleteEventHandler(wmPtr->tkwin, StructureNotifyMask,
		ChildEventProc, wmPtr);
	    Tk_ManageGeometry(wmPtr->tkwin, (Tk_GeomMgr *) 0, (ClientData)0);
	    Tk_UnmapWindow(wmPtr->tkwin);
	}
	Tk_CreateEventHandler(tkwin, StructureNotifyMask, ChildEventProc, 
		wmPtr);
	Tk_ManageGeometry(tkwin, &winMarkerMgrInfo, wmPtr);
    }
    wmPtr->tkwin = tkwin;
    markerPtr->flags |= MAP_ITEM;
    if (markerPtr->drawUnder) {
	graphPtr->flags |= REDRAW_BACKING_STORE;
    }
    Blt_EventuallyRedrawGraph(graphPtr);
    return TCL_OK;
}

/*
 * ----------------------------------------------------------------------
 *
 * MapWindowMarker --
 *
 *	Calculate the layout position for a window marker.  Positional
 *	information is saved in the marker.
 *
 * Results:
 *	None.
 *
 * ----------------------------------------------------------------------
 */
static void
MapWindowMarker(Marker *markerPtr)
{
    WindowMarker *wmPtr = (WindowMarker *)markerPtr;
    Graph *graphPtr = markerPtr->object.graphPtr;
    Point2D anchorPt;
    Region2D extents;
    int width, height;

    if (wmPtr->tkwin == (Tk_Window)NULL) {
	return;
    }
    anchorPt = MapPoint(graphPtr, markerPtr->worldPts, &markerPtr->axes);

    width = Tk_ReqWidth(wmPtr->tkwin);
    height = Tk_ReqHeight(wmPtr->tkwin);
    if (wmPtr->reqWidth > 0) {
	width = wmPtr->reqWidth;
    }
    if (wmPtr->reqHeight > 0) {
	height = wmPtr->reqHeight;
    }
    wmPtr->anchorPt = Blt_AnchorPoint(anchorPt.x, anchorPt.y, (double)width, 
	(double)height, wmPtr->anchor);
    wmPtr->anchorPt.x += markerPtr->xOffset;
    wmPtr->anchorPt.y += markerPtr->yOffset;
    wmPtr->width = width;
    wmPtr->height = height;

    /*
     * Determine the bounding box of the window and test to see if it
     * is at least partially contained within the plotting area.
     */
    extents.left = wmPtr->anchorPt.x;
    extents.top = wmPtr->anchorPt.y;
    extents.right = wmPtr->anchorPt.x + wmPtr->width - 1;
    extents.bottom = wmPtr->anchorPt.y + wmPtr->height - 1;
    markerPtr->clipped = BoxesDontOverlap(graphPtr, &extents);
}

/*
 * ----------------------------------------------------------------------
 *
 * PointInWindowMarker --
 *
 * ----------------------------------------------------------------------
 */
static int
PointInWindowMarker(Marker *markerPtr, Point2D *samplePtr)
{
    WindowMarker *wmPtr = (WindowMarker *)markerPtr;

    return ((samplePtr->x >= wmPtr->anchorPt.x) && 
	    (samplePtr->x < (wmPtr->anchorPt.x + wmPtr->width)) &&
	    (samplePtr->y >= wmPtr->anchorPt.y) && 
	    (samplePtr->y < (wmPtr->anchorPt.y + wmPtr->height)));
}

/*
 * ----------------------------------------------------------------------
 *
 * RegionInWindowMarker --
 *
 * ----------------------------------------------------------------------
 */
static int
RegionInWindowMarker(Marker *markerPtr, Region2D *extsPtr, int enclosed)
{
    WindowMarker *wmPtr = (WindowMarker *)markerPtr;

    if (markerPtr->nWorldPts < 1) {
	return FALSE;
    }
    if (enclosed) {
	return ((wmPtr->anchorPt.x >= extsPtr->left) &&
		(wmPtr->anchorPt.y >= extsPtr->top) && 
		((wmPtr->anchorPt.x + wmPtr->width) <= extsPtr->right) &&
		((wmPtr->anchorPt.y + wmPtr->height) <= extsPtr->bottom));
    }
    return !((wmPtr->anchorPt.x >= extsPtr->right) ||
	     (wmPtr->anchorPt.y >= extsPtr->bottom) ||
	     ((wmPtr->anchorPt.x + wmPtr->width) <= extsPtr->left) ||
	     ((wmPtr->anchorPt.y + wmPtr->height) <= extsPtr->top));
}

/*
 * ----------------------------------------------------------------------
 *
 * DrawWindowMarker --
 *
 * ----------------------------------------------------------------------
 */
/*ARGSUSED*/
static void
DrawWindowMarker(markerPtr, drawable)
    Marker *markerPtr;
    Drawable drawable;		/* Pixmap or window to draw into */
{
    WindowMarker *wmPtr = (WindowMarker *)markerPtr;

    if (wmPtr->tkwin == NULL) {
	return;
    }
    if ((wmPtr->height != Tk_Height(wmPtr->tkwin)) ||
	(wmPtr->width != Tk_Width(wmPtr->tkwin)) ||
	((int)wmPtr->anchorPt.x != Tk_X(wmPtr->tkwin)) ||
	((int)wmPtr->anchorPt.y != Tk_Y(wmPtr->tkwin))) {
	Tk_MoveResizeWindow(wmPtr->tkwin, (int)wmPtr->anchorPt.x, 
	    (int)wmPtr->anchorPt.y, wmPtr->width, wmPtr->height);
    }
    if (!Tk_IsMapped(wmPtr->tkwin)) {
	Tk_MapWindow(wmPtr->tkwin);
    }
}

/*
 * ----------------------------------------------------------------------
 *
 * WindowMarkerToPostScript --
 *
 * ----------------------------------------------------------------------
 */
static void
WindowMarkerToPostScript(Marker *markerPtr, Blt_PostScript ps)
{
    WindowMarker *wmPtr = (WindowMarker *)markerPtr;

    if (wmPtr->tkwin == NULL) {
	return;
    }
    if (Tk_IsMapped(wmPtr->tkwin)) {
	Blt_WindowToPostScript(ps, wmPtr->tkwin, wmPtr->anchorPt.x, 
		wmPtr->anchorPt.y);
    }
}

/*
 * ----------------------------------------------------------------------
 *
 * FreeWindowMarker --
 *
 *	Destroys the structure containing the attributes of the window
 *      marker.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Window is destroyed and removed from the screen.
 *
 * ----------------------------------------------------------------------
 */
/*ARGSUSED*/
static void
FreeWindowMarker(graphPtr, markerPtr)
    Graph *graphPtr;
    Marker *markerPtr;
{
    WindowMarker *wmPtr = (WindowMarker *)markerPtr;

    if (wmPtr->tkwin != NULL) {
	Tk_DeleteEventHandler(wmPtr->tkwin, StructureNotifyMask,
	    ChildEventProc, wmPtr);
	Tk_ManageGeometry(wmPtr->tkwin, (Tk_GeomMgr *) 0, (ClientData)0);
	Tk_DestroyWindow(wmPtr->tkwin);
    }
}

/*
 * ----------------------------------------------------------------------
 *
 * CreateWindowMarker --
 *
 *	Allocate memory and initialize methods for the new window marker.
 *
 * Results:
 *	The pointer to the newly allocated marker structure is returned.
 *
 * Side effects:
 *	Memory is allocated for the window marker structure.
 *
 * ----------------------------------------------------------------------
 */
static Marker *
CreateWindowMarker()
{
    WindowMarker *wmPtr;

    wmPtr = Blt_Calloc(1, sizeof(WindowMarker));
    if (wmPtr != NULL) {
	wmPtr->base.classPtr = &windowMarkerClass;
    }
    return (Marker *)wmPtr;
}

/*
 * ----------------------------------------------------------------------
 *
 * ChildEventProc --
 *
 *	This procedure is invoked whenever StructureNotify events
 *	occur for a window that's managed as part of a graph window
 *	marker. This procedure's only purpose is to clean up when
 *	windows are deleted.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The window is disassociated from the window item when it is
 *	deleted.
 *
 * ----------------------------------------------------------------------
 */
static void
ChildEventProc(clientData, eventPtr)
    ClientData clientData;	/* Pointer to record describing window item. */
    XEvent *eventPtr;		/* Describes what just happened. */
{
    WindowMarker *wmPtr = clientData;

    if (eventPtr->type == DestroyNotify) {
	wmPtr->tkwin = NULL;
    }
}

/*
 * ----------------------------------------------------------------------
 *
 * ChildGeometryProc --
 *
 *	This procedure is invoked whenever a window that's associated
 *	with a window item changes its requested dimensions.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The size and location on the window of the window may change,
 *	depending on the options specified for the window item.
 *
 * ----------------------------------------------------------------------
 */
/* ARGSUSED */
static void
ChildGeometryProc(clientData, tkwin)
    ClientData clientData;	/* Pointer to record for window item. */
    Tk_Window tkwin;		/* Window that changed its desired size. */
{
    WindowMarker *wmPtr = clientData;

    if (wmPtr->reqWidth == 0) {
	wmPtr->width = Tk_ReqWidth(tkwin);
    }
    if (wmPtr->reqHeight == 0) {
	wmPtr->height = Tk_ReqHeight(tkwin);
    }
}

/*
 * ----------------------------------------------------------------------
 *
 * ChildCustodyProc --
 *
 *	This procedure is invoked when an embedded window has been
 *	stolen by another geometry manager.  The information and
 *	memory associated with the widget is released.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Arranges for the graph to be redrawn without the embedded
 *	widget at the next idle point.
 *
 * ----------------------------------------------------------------------
 */
 /* ARGSUSED */
static void
ChildCustodyProc(clientData, tkwin)
    ClientData clientData;	/* Window marker to be destroyed. */
    Tk_Window tkwin;		/* Not used. */
{
    Marker *markerPtr = clientData;
    Graph *graphPtr;

    graphPtr = markerPtr->object.graphPtr;
    DestroyMarker(markerPtr);
    /*
     * Not really needed. We should get an Expose event when the
     * child window is unmapped.
     */
    Blt_EventuallyRedrawGraph(graphPtr);
}

/*
 * ----------------------------------------------------------------------
 *
 * MapLineMarker --
 *
 *	Calculate the layout position for a line marker.  Positional
 *	information is saved in the marker.  The line positions are
 *	stored in an array of points (malloc'ed).
 *
 * Results:
 *	None.
 *
 * ----------------------------------------------------------------------
 */
static void
MapLineMarker(markerPtr)
    Marker *markerPtr;
{
    Graph *graphPtr = markerPtr->object.graphPtr;
    LineMarker *lmPtr = (LineMarker *)markerPtr;
    Point2D *srcPtr, *pend;
    Segment2D *segments, *segPtr;
    Point2D p, q;
    Region2D extents;

    lmPtr->nSegments = 0;
    if (lmPtr->segments != NULL) {
	Blt_Free(lmPtr->segments);
    }
    if (markerPtr->nWorldPts < 2) {
	return;			/* Too few points */
    }
    Blt_GraphExtents(graphPtr, &extents);

    /* 
     * Allow twice the number of world coordinates. The line will
     * represented as series of line segments, not one continous
     * polyline.  This is because clipping against the plot area may
     * chop the line into several disconnected segments.
     */
    segments = Blt_Malloc(markerPtr->nWorldPts * sizeof(Segment2D));
    srcPtr = markerPtr->worldPts;
    p = MapPoint(graphPtr, srcPtr, &markerPtr->axes);
    p.x += markerPtr->xOffset;
    p.y += markerPtr->yOffset;

    segPtr = segments;
    for (srcPtr++, pend = markerPtr->worldPts + markerPtr->nWorldPts; 
	 srcPtr < pend; srcPtr++) {
	Point2D next;

	next = MapPoint(graphPtr, srcPtr, &markerPtr->axes);
	next.x += markerPtr->xOffset;
	next.y += markerPtr->yOffset;
	q = next;
	if (Blt_LineRectClip(&extents, &p, &q)) {
	    segPtr->p = p;
	    segPtr->q = q;
	    segPtr++;
	}
	p = next;
    }
    lmPtr->nSegments = segPtr - segments;
    lmPtr->segments = segments;
    markerPtr->clipped = (lmPtr->nSegments == 0);
}

static int
PointInLineMarker(markerPtr, samplePtr)
    Marker *markerPtr;
    Point2D *samplePtr;
{
    LineMarker *lmPtr = (LineMarker *)markerPtr;

    return Blt_PointInSegments(samplePtr, lmPtr->segments, lmPtr->nSegments, 
	   (double)markerPtr->object.graphPtr->halo);
}

/*
 * ----------------------------------------------------------------------
 *
 * RegionInLineMarker --
 *
 * ----------------------------------------------------------------------
 */
static int
RegionInLineMarker(markerPtr, extsPtr, enclosed)
    Marker *markerPtr;
    Region2D *extsPtr;
    int enclosed;
{
    if (markerPtr->nWorldPts < 2) {
	return FALSE;
    }
    if (enclosed) {
	Point2D *pp, *pend;

	for (pp = markerPtr->worldPts, pend = pp + markerPtr->nWorldPts; 
	     pp < pend; pp++) {
	    Point2D p;

	    p = MapPoint(markerPtr->object.graphPtr, pp, &markerPtr->axes);
	    if ((p.x < extsPtr->left) && (p.x > extsPtr->right) &&
		(p.y < extsPtr->top) && (p.y > extsPtr->bottom)) {
		return FALSE;
	    }
	}
	return TRUE;		/* All points inside bounding box. */
    } else {
	int count;
	Point2D *pp, *pend;

	count = 0;
	for (pp = markerPtr->worldPts, pend = pp + (markerPtr->nWorldPts - 1); 
		pp < pend; pp++) {
	    Point2D p, q;

	    p = MapPoint(markerPtr->object.graphPtr, pp, &markerPtr->axes);
	    q = MapPoint(markerPtr->object.graphPtr, pp + 1, &markerPtr->axes);
	    if (Blt_LineRectClip(extsPtr, &p, &q)) {
		count++;
	    }
	}
	return (count > 0);	/* At least 1 segment passes through region. */
    }
}

/*
 * ----------------------------------------------------------------------
 *
 * DrawLineMarker --
 *
 * ----------------------------------------------------------------------
 */
static void
DrawLineMarker(markerPtr, drawable)
    Marker *markerPtr;
    Drawable drawable;		/* Pixmap or window to draw into */
{
    LineMarker *lmPtr = (LineMarker *)markerPtr;

    if (lmPtr->nSegments > 0) {
	Graph *graphPtr = markerPtr->object.graphPtr;

	Blt_Draw2DSegments(graphPtr->display, drawable, lmPtr->gc, 
		lmPtr->segments, lmPtr->nSegments);
	if (lmPtr->xor) {	/* Toggle the drawing state */
	    lmPtr->xorState = (lmPtr->xorState == 0);
	}
    }
}

/*
 * ----------------------------------------------------------------------
 *
 * ConfigureLineMarker --
 *
 *	This procedure is called to process an objv/objc list, plus
 *	the Tk option database, in order to configure (or reconfigure)
 *	a line marker.
 *
 * Results:
 *	A standard Tcl result.  If TCL_ERROR is returned, then
 *	interp->result contains an error message.
 *
 * Side effects:
 *	Configuration information, such as line width, colors, dashes,
 *	etc. get set for markerPtr; old resources get freed, if there
 *	were any.  The marker is eventually redisplayed.
 *
 * ----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
ConfigureLineMarker(markerPtr)
    Marker *markerPtr;
{
    Graph *graphPtr = markerPtr->object.graphPtr;
    LineMarker *lmPtr = (LineMarker *)markerPtr;
    GC newGC;
    XGCValues gcValues;
    unsigned long gcMask;
    Drawable drawable;

    drawable = Tk_WindowId(graphPtr->tkwin);
    gcMask = (GCLineWidth | GCLineStyle | GCCapStyle | GCJoinStyle);
    if (lmPtr->outlineColor != NULL) {
	gcMask |= GCForeground;
	gcValues.foreground = lmPtr->outlineColor->pixel;
    }
    if (lmPtr->fillColor != NULL) {
	gcMask |= GCBackground;
	gcValues.background = lmPtr->fillColor->pixel;
    }
    gcValues.cap_style = lmPtr->capStyle;
    gcValues.join_style = lmPtr->joinStyle;
    gcValues.line_width = LineWidth(lmPtr->lineWidth);
    gcValues.line_style = LineSolid;
    if (LineIsDashed(lmPtr->dashes)) {
	gcValues.line_style = 
	    (gcMask & GCBackground) ? LineDoubleDash : LineOnOffDash;
    }
    if (lmPtr->xor) {
	unsigned long pixel;
	gcValues.function = GXxor;

	gcMask |= GCFunction;
	if (graphPtr->plotBgColor == NULL) {
	    pixel = WhitePixelOfScreen(Tk_Screen(graphPtr->tkwin));
	} else {
	    pixel = graphPtr->plotBgColor->pixel;
	}
	if (gcMask & GCBackground) {
	    gcValues.background ^= pixel;
	}
	gcValues.foreground ^= pixel;
	if (drawable != None) {
	    DrawLineMarker(markerPtr, drawable);
	}
    }
    newGC = Blt_GetPrivateGC(graphPtr->tkwin, gcMask, &gcValues);
    if (lmPtr->gc != NULL) {
	Blt_FreePrivateGC(graphPtr->display, lmPtr->gc);
    }
    if (LineIsDashed(lmPtr->dashes)) {
	Blt_SetDashes(graphPtr->display, newGC, &lmPtr->dashes);
    }
    lmPtr->gc = newGC;
    if (lmPtr->xor) {
	if (drawable != None) {
	    MapLineMarker(markerPtr);
	    DrawLineMarker(markerPtr, drawable);
	}
	return TCL_OK;
    }
    markerPtr->flags |= MAP_ITEM;
    if (markerPtr->drawUnder) {
	graphPtr->flags |= REDRAW_BACKING_STORE;
    }
    Blt_EventuallyRedrawGraph(graphPtr);
    return TCL_OK;
}

/*
 * ----------------------------------------------------------------------
 *
 * LineMarkerToPostScript --
 *
 *	Prints postscript commands to display the connect line.
 *	Dashed lines need to be handled specially, especially if a
 *	background color is designated.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	PostScript output commands are saved in the interpreter
 *	(infoPtr->interp) result field.
 *
 * ----------------------------------------------------------------------
 */
static void
LineMarkerToPostScript(Marker *markerPtr, Blt_PostScript ps)
{
    LineMarker *lmPtr = (LineMarker *)markerPtr;

    if (lmPtr->nSegments > 0) {
	Blt_LineAttributesToPostScript(ps, lmPtr->outlineColor, 
		lmPtr->lineWidth, &lmPtr->dashes, lmPtr->capStyle,
		lmPtr->joinStyle);
	if ((LineIsDashed(lmPtr->dashes)) && (lmPtr->fillColor != NULL)) {
	    Blt_AppendToPostScript(ps, "/DashesProc {\n  gsave\n    ", 
		(char *)NULL);
	    Blt_BackgroundToPostScript(ps, lmPtr->fillColor);
	    Blt_AppendToPostScript(ps, "    ", (char *)NULL);
	    Blt_LineDashesToPostScript(ps, (Blt_Dashes *)NULL);
	    Blt_AppendToPostScript(ps,
		"stroke\n",
		"  grestore\n",
		"} def\n", (char *)NULL);
	} else {
	    Blt_AppendToPostScript(ps, "/DashesProc {} def\n", (char *)NULL);
	}
	Blt_2DSegmentsToPostScript(ps, lmPtr->segments, lmPtr->nSegments);
    }
}

/*
 * ----------------------------------------------------------------------
 *
 * FreeLineMarker --
 *
 *	Destroys the structure and attributes of a line marker.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Line attributes (GCs, colors, stipple, etc) get released.
 *	Memory is deallocated, X resources are freed.
 *
 * ----------------------------------------------------------------------
 */
static void
FreeLineMarker(graphPtr, markerPtr)
    Graph *graphPtr;
    Marker *markerPtr;
{
    LineMarker *lmPtr = (LineMarker *)markerPtr;

    if (lmPtr->gc != NULL) {
	Blt_FreePrivateGC(graphPtr->display, lmPtr->gc);
    }
    if (lmPtr->segments != NULL) {
	Blt_Free(lmPtr->segments);
    }
}


/*
 * ----------------------------------------------------------------------
 *
 * CreateLineMarker --
 *
 *	Allocate memory and initialize methods for a new line marker.
 *
 * Results:
 *	The pointer to the newly allocated marker structure is returned.
 *
 * Side effects:
 *	Memory is allocated for the line marker structure.
 *
 * ----------------------------------------------------------------------
 */
static Marker *
CreateLineMarker()
{
    LineMarker *lmPtr;

    lmPtr = Blt_Calloc(1, sizeof(LineMarker));
    if (lmPtr != NULL) {
	lmPtr->base.classPtr = &lineMarkerClass;
	lmPtr->xor = FALSE;
	lmPtr->capStyle = CapButt;
	lmPtr->joinStyle = JoinMiter;
    }
    return (Marker *)lmPtr;
}

/*
 * ----------------------------------------------------------------------
 *
 * MapPolygonMarker --
 *
 *	Calculate the layout position for a polygon marker.  Positional
 *	information is saved in the polygon in an array of points
 *	(malloc'ed).
 *
 * Results:
 *	None.
 *
 * ----------------------------------------------------------------------
 */
static void
MapPolygonMarker(markerPtr)
    Marker *markerPtr;
{
    Graph *graphPtr = markerPtr->object.graphPtr;
    PolygonMarker *pmPtr = (PolygonMarker *)markerPtr;
    Point2D *screenPts;
    Region2D extents;
    int nScreenPts;

    if (pmPtr->outlinePts != NULL) {
	Blt_Free(pmPtr->outlinePts);
	pmPtr->outlinePts = NULL;
	pmPtr->nOutlinePts = 0;
    }
    if (pmPtr->fillPts != NULL) {
	Blt_Free(pmPtr->fillPts);
	pmPtr->fillPts = NULL;
	pmPtr->nFillPts = 0;
    }
    if (pmPtr->screenPts != NULL) {
	Blt_Free(pmPtr->screenPts);
	pmPtr->screenPts = NULL;
    }
    if (markerPtr->nWorldPts < 3) {
	return;			/* Too few points */
    }

    /* 
     * Allocate and fill a temporary array to hold the screen
     * coordinates of the polygon. 
     */
    nScreenPts = markerPtr->nWorldPts + 1;
    screenPts = Blt_Malloc((nScreenPts + 1) * sizeof(Point2D));
    {
	Point2D *sp, *dp, *send;

	dp = screenPts;
	for (sp = markerPtr->worldPts, send = sp + markerPtr->nWorldPts; 
	     sp < send; sp++) {
	    *dp = MapPoint(graphPtr, sp, &markerPtr->axes);
	    dp->x += markerPtr->xOffset;
	    dp->y += markerPtr->yOffset;
	    dp++;
	}
	*dp = screenPts[0];
    }
    Blt_GraphExtents(graphPtr, &extents);
    markerPtr->clipped = TRUE;
    if (pmPtr->fill.fgColor != NULL) { /* Polygon fill required. */
	Point2D *fillPts;
	int n;

	fillPts = Blt_Malloc(sizeof(Point2D) * nScreenPts * 3);
	assert(fillPts);
	n = Blt_PolyRectClip(&extents, screenPts, markerPtr->nWorldPts, fillPts);
	if (n < 3) { 
	    Blt_Free(fillPts);
	} else {
	    pmPtr->nFillPts = n;
	    pmPtr->fillPts = fillPts;
	    markerPtr->clipped = FALSE;
	}
    }
    if ((pmPtr->outline.fgColor != NULL) && (pmPtr->lineWidth > 0)) { 
	Segment2D *outlinePts;
	Segment2D *segPtr;
	Point2D *sp, *send;

	/* 
	 * Generate line segments representing the polygon outline.
	 * The resulting outline may or may not be closed from
	 * viewport clipping.  
	 */
	outlinePts = Blt_Malloc(nScreenPts * sizeof(Segment2D));
	if (outlinePts == NULL) {
	    return;		/* Can't allocate point array */
	}
	/* 
	 * Note that this assumes that the point array contains an
	 * extra point that closes the polygon. 
	 */
	segPtr = outlinePts;
	for (sp = screenPts, send = sp + (nScreenPts - 1); sp < send; sp++) {
	    segPtr->p = sp[0];
	    segPtr->q = sp[1];
	    if (Blt_LineRectClip(&extents, &segPtr->p, &segPtr->q)) {
		segPtr++;
	    }
	}
	pmPtr->nOutlinePts = segPtr - outlinePts;
	pmPtr->outlinePts = outlinePts;
	if (pmPtr->nOutlinePts > 0) {
	    markerPtr->clipped = FALSE;
	}
    }
    pmPtr->screenPts = screenPts;
}

static int
PointInPolygonMarker(markerPtr, samplePtr)
    Marker *markerPtr;
    Point2D *samplePtr;
{
    PolygonMarker *pmPtr = (PolygonMarker *)markerPtr;

    if ((markerPtr->nWorldPts >= 3) && (pmPtr->screenPts != NULL)) {
	return Blt_PointInPolygon(samplePtr, pmPtr->screenPts, 
		markerPtr->nWorldPts + 1);
    }
    return FALSE;
}

/*
 * ----------------------------------------------------------------------
 *
 * RegionInPolygonMarker --
 *
 * ----------------------------------------------------------------------
 */
static int
RegionInPolygonMarker(markerPtr, extsPtr, enclosed)
    Marker *markerPtr;
    Region2D *extsPtr;
    int enclosed;
{
    PolygonMarker *pmPtr = (PolygonMarker *)markerPtr;
    
    if ((markerPtr->nWorldPts >= 3) && (pmPtr->screenPts != NULL)) {
	return Blt_RegionInPolygon(extsPtr, pmPtr->screenPts, 
		markerPtr->nWorldPts, enclosed);
    }
    return FALSE;
}

static void
DrawPolygonMarker(markerPtr, drawable)
    Marker *markerPtr;
    Drawable drawable;		/* Pixmap or window to draw into */
{
    Graph *graphPtr = markerPtr->object.graphPtr;
    PolygonMarker *pmPtr = (PolygonMarker *)markerPtr;

    /* Draw polygon fill region */
    if ((pmPtr->nFillPts > 0) && (pmPtr->fill.fgColor != NULL)) {
	XPoint *dp, *points;
	Point2D *sp, *send;
	
	points = Blt_Malloc(pmPtr->nFillPts * sizeof(XPoint));
	if (points == NULL) {
	    return;
	}
	dp = points;
	for (sp = pmPtr->fillPts, send = sp + pmPtr->nFillPts; sp < send; 
	     sp++) {
	    dp->x = (short int)sp->x;
	    dp->y = (short int)sp->y;
	    dp++;
	}
	XFillPolygon(graphPtr->display, drawable, pmPtr->fillGC, points, 
		pmPtr->nFillPts, Complex, CoordModeOrigin);
	Blt_Free(points);
    }
    /* and then the outline */
    if ((pmPtr->nOutlinePts > 0) && (pmPtr->lineWidth > 0) && 
	(pmPtr->outline.fgColor != NULL)) {
	Blt_Draw2DSegments(graphPtr->display, drawable, pmPtr->outlineGC,
	    pmPtr->outlinePts, pmPtr->nOutlinePts);
    }
}


static void
PolygonMarkerToPostScript(Marker *markerPtr, Blt_PostScript ps)
{
    Graph *graphPtr = markerPtr->object.graphPtr;
    PolygonMarker *pmPtr = (PolygonMarker *)markerPtr;

    if (pmPtr->fill.fgColor != NULL) {

	/*
	 * Options:  fg bg
	 *			Draw outline only.
	 *	     x          Draw solid or stipple.
	 *	     x  x       Draw solid or stipple.
	 */

	/* Create a path to use for both the polygon and its outline. */
	Blt_PathToPostScript(ps, pmPtr->fillPts, pmPtr->nFillPts);
	Blt_AppendToPostScript(ps, "closepath\n", (char *)NULL);

	/* If the background fill color was specified, draw the
	 * polygon in a solid fashion with that color.  */
	if (pmPtr->fill.bgColor != NULL) {
	    Blt_BackgroundToPostScript(ps, pmPtr->fill.bgColor);
	    Blt_AppendToPostScript(ps, "Fill\n", (char *)NULL);
	}
	Blt_ForegroundToPostScript(ps, pmPtr->fill.fgColor);
	if (pmPtr->stipple != None) {
	    /* Draw the stipple in the foreground color. */
	    Blt_StippleToPostScript(ps, graphPtr->display, pmPtr->stipple);
	} else {
	    Blt_AppendToPostScript(ps, "Fill\n", (char *)NULL);
	}
    }

    /* Draw the outline in the foreground color.  */
    if ((pmPtr->lineWidth > 0) && (pmPtr->outline.fgColor != NULL)) {

	/*  Set up the line attributes.  */
	Blt_LineAttributesToPostScript(ps, pmPtr->outline.fgColor,
	    pmPtr->lineWidth, &pmPtr->dashes, pmPtr->capStyle,
	    pmPtr->joinStyle);

	/*  
	 * Define on-the-fly a PostScript macro "DashesProc" that
	 * will be executed for each call to the Polygon drawing
	 * routine.  If the line isn't dashed, simply make this an
	 * empty definition.  
	 */
	if ((pmPtr->outline.bgColor != NULL) && (LineIsDashed(pmPtr->dashes))) {
	    Blt_AppendToPostScript(ps,
		"/DashesProc {\n",
		"gsave\n    ", (char *)NULL);
	    Blt_BackgroundToPostScript(ps, pmPtr->outline.bgColor);
	    Blt_AppendToPostScript(ps, "    ", (char *)NULL);
	    Blt_LineDashesToPostScript(ps, (Blt_Dashes *)NULL);
	    Blt_AppendToPostScript(ps,
		"stroke\n",
		"  grestore\n",
		"} def\n", (char *)NULL);
	} else {
	    Blt_AppendToPostScript(ps, "/DashesProc {} def\n", (char *)NULL);
	}
	Blt_2DSegmentsToPostScript(ps, pmPtr->outlinePts, pmPtr->nOutlinePts);
    }
}

/*
 * ----------------------------------------------------------------------
 *
 * ConfigurePolygonMarker --
 *
 *	This procedure is called to process an objv/objc list, plus
 *	the Tk option database, in order to configure (or reconfigure)
 *	a polygon marker.
 *
 * Results:
 *	A standard Tcl result.  If TCL_ERROR is returned, then
 *	interp->result contains an error message.
 *
 * Side effects:
 *	Configuration information, such as polygon color, dashes,
 *	fillstyle, etc. get set for markerPtr; old resources get
 *	freed, if there were any.  The marker is eventually
 *	redisplayed.
 *
 * ---------------------------------------------------------------------- 
 */
/*ARGSUSED*/
static int
ConfigurePolygonMarker(markerPtr)
    Marker *markerPtr;
{
    Graph *graphPtr = markerPtr->object.graphPtr;
    PolygonMarker *pmPtr = (PolygonMarker *)markerPtr;
    GC newGC;
    XGCValues gcValues;
    unsigned long gcMask;
    Drawable drawable;

    drawable = Tk_WindowId(graphPtr->tkwin);
    gcMask = (GCLineWidth | GCLineStyle);
    if (pmPtr->outline.fgColor != NULL) {
	gcMask |= GCForeground;
	gcValues.foreground = pmPtr->outline.fgColor->pixel;
    }
    if (pmPtr->outline.bgColor != NULL) {
	gcMask |= GCBackground;
	gcValues.background = pmPtr->outline.bgColor->pixel;
    }
    gcMask |= (GCCapStyle | GCJoinStyle);
    gcValues.cap_style = pmPtr->capStyle;
    gcValues.join_style = pmPtr->joinStyle;
    gcValues.line_style = LineSolid;
    gcValues.dash_offset = 0;
    gcValues.line_width = LineWidth(pmPtr->lineWidth);
    if (LineIsDashed(pmPtr->dashes)) {
	gcValues.line_style = (pmPtr->outline.bgColor == NULL)
	    ? LineOnOffDash : LineDoubleDash;
    }
    if (pmPtr->xor) {
	unsigned long pixel;
	gcValues.function = GXxor;

	gcMask |= GCFunction;
	if (graphPtr->plotBgColor == NULL) {
	    /* The graph's color option may not have been set yet */
	    pixel = WhitePixelOfScreen(Tk_Screen(graphPtr->tkwin));
	} else {
	    pixel = graphPtr->plotBgColor->pixel;
	}
	if (gcMask & GCBackground) {
	    gcValues.background ^= pixel;
	}
	gcValues.foreground ^= pixel;
	if (drawable != None) {
	    DrawPolygonMarker(markerPtr, drawable);
	}
    }
    newGC = Blt_GetPrivateGC(graphPtr->tkwin, gcMask, &gcValues);
    if (LineIsDashed(pmPtr->dashes)) {
	Blt_SetDashes(graphPtr->display, newGC, &pmPtr->dashes);
    }
    if (pmPtr->outlineGC != NULL) {
	Blt_FreePrivateGC(graphPtr->display, pmPtr->outlineGC);
    }
    pmPtr->outlineGC = newGC;

    gcMask = 0;
    if (pmPtr->fill.fgColor != NULL) {
	gcMask |= GCForeground;
	gcValues.foreground = pmPtr->fill.fgColor->pixel;
    }
    if (pmPtr->fill.bgColor != NULL) {
	gcMask |= GCBackground;
	gcValues.background = pmPtr->fill.bgColor->pixel;
    }
    if (pmPtr->stipple != None) {
	gcValues.stipple = pmPtr->stipple;
	gcValues.fill_style = (pmPtr->fill.bgColor != NULL)
	    ? FillOpaqueStippled : FillStippled;
	gcMask |= (GCStipple | GCFillStyle);
    }
    newGC = Tk_GetGC(graphPtr->tkwin, gcMask, &gcValues);
    if (pmPtr->fillGC != NULL) {
	Tk_FreeGC(graphPtr->display, pmPtr->fillGC);
    }
    pmPtr->fillGC = newGC;

    if ((gcMask == 0) && !(graphPtr->flags & RESET_AXES) && (pmPtr->xor)) {
	if (drawable != None) {
	    MapPolygonMarker(markerPtr);
	    DrawPolygonMarker(markerPtr, drawable);
	}
	return TCL_OK;
    }
    markerPtr->flags |= MAP_ITEM;
    if (markerPtr->drawUnder) {
	graphPtr->flags |= REDRAW_BACKING_STORE;
    }
    graphPtr->flags |= RESET_WORLD;
    Blt_EventuallyRedrawGraph(graphPtr);
    return TCL_OK;
}

/*
 * ----------------------------------------------------------------------
 *
 * FreePolygonMarker --
 *
 *	Release memory and resources allocated for the polygon element.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Everything associated with the polygon element is freed up.
 *
 * ----------------------------------------------------------------------
 */
static void
FreePolygonMarker(graphPtr, markerPtr)
    Graph *graphPtr;
    Marker *markerPtr;
{
    PolygonMarker *pmPtr = (PolygonMarker *)markerPtr;

    if (pmPtr->fillGC != NULL) {
	Tk_FreeGC(graphPtr->display, pmPtr->fillGC);
    }
    if (pmPtr->outlineGC != NULL) {
	Blt_FreePrivateGC(graphPtr->display, pmPtr->outlineGC);
    }
    if (pmPtr->fillPts != NULL) {
	Blt_Free(pmPtr->fillPts);
    }
    if (pmPtr->outlinePts != NULL) {
	Blt_Free(pmPtr->outlinePts);
    }
    if (pmPtr->screenPts != NULL) {
	Blt_Free(pmPtr->screenPts);
    }
}

/*
 * ----------------------------------------------------------------------
 *
 * CreatePolygonMarker --
 *
 *	Allocate memory and initialize methods for the new polygon
 *	marker.
 *
 * Results:
 *	The pointer to the newly allocated marker structure is
 *	returned.
 *
 * Side effects:
 *	Memory is allocated for the polygon marker structure.
 *
 * ---------------------------------------------------------------------- 
 */
static Marker *
CreatePolygonMarker()
{
    PolygonMarker *pmPtr;

    pmPtr = Blt_Calloc(1, sizeof(PolygonMarker));
    if (pmPtr != NULL) {
	pmPtr->base.classPtr = &polygonMarkerClass;
	pmPtr->capStyle = CapButt;
	pmPtr->joinStyle = JoinMiter;

    }
    return (Marker *)pmPtr;
}

static int
GetMarkerFromObj(
    Tcl_Interp *interp,
    Graph *graphPtr,
    Tcl_Obj *objPtr,
    Marker **markerPtrPtr)
{
    Blt_HashEntry *hPtr;
    char *string;

    string = Tcl_GetString(objPtr);
    hPtr = Blt_FindHashEntry(&graphPtr->markers.table, string);
    if (hPtr != NULL) {
	*markerPtrPtr = (Marker *)Blt_GetHashValue(hPtr);
	return TCL_OK;
    }
    if (interp != NULL) {
	Tcl_AppendResult(interp, "can't find marker \"", string, 
	     "\" in \"", Tk_PathName(graphPtr->tkwin), (char *)NULL);
    }
    return TCL_ERROR;
}


static int
RenameMarker(
    Graph *graphPtr,
    Marker *markerPtr,
    char *oldName, 
    char *newName)
{
    int isNew;
    Blt_HashEntry *hPtr;

    /* Rename the marker only if no marker already exists by that name */
    hPtr = Blt_CreateHashEntry(&graphPtr->markers.table, newName, &isNew);
    if (!isNew) {
	Tcl_AppendResult(graphPtr->interp, "can't rename marker: \"", newName,
	    "\" already exists", (char *)NULL);
	return TCL_ERROR;
    }
    markerPtr->object.name = Blt_Strdup(newName);
    markerPtr->hashPtr = hPtr;
    Blt_SetHashValue(hPtr, (char *)markerPtr);

    /* Delete the old hash entry */
    hPtr = Blt_FindHashEntry(&graphPtr->markers.table, oldName);
    Blt_DeleteHashEntry(&graphPtr->markers.table, hPtr);
    if (oldName != NULL) {
	Blt_Free(oldName);
    }
    return TCL_OK;
}

/*
 * ----------------------------------------------------------------------
 *
 * NamesOp --
 *
 *	Returns a list of marker identifiers in interp->result;
 *
 * Results:
 *	The return value is a standard Tcl result.
 *
 * ----------------------------------------------------------------------
 */
static int
NamesOp(
    Graph *graphPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Tcl_Obj *listObjPtr;

    listObjPtr = Tcl_NewListObj(0, (Tcl_Obj **)NULL);
    if (objc == 3) {
	Blt_ChainLink *lp;

	for (lp = Blt_ChainFirstLink(graphPtr->markers.displayList); lp != NULL;
	     lp = Blt_ChainNextLink(lp)) {
	    Marker *markerPtr;

	    markerPtr = Blt_ChainGetValue(lp);
	    Tcl_ListObjAppendElement(interp, listObjPtr,
		Tcl_NewStringObj(markerPtr->object.name, -1));
	}
    } else {
	Blt_ChainLink *lp;

	for (lp = Blt_ChainFirstLink(graphPtr->markers.displayList); lp != NULL;
	     lp = Blt_ChainNextLink(lp)) {
	    Marker *markerPtr;
	    int i;

	    markerPtr = Blt_ChainGetValue(lp);
	    for (i = 3; i < objc; i++) {
		char *pattern;

		pattern = Tcl_GetString(objv[i]);
		if (Tcl_StringMatch(markerPtr->object.name, pattern)) {
		    Tcl_ListObjAppendElement(interp, listObjPtr,
			Tcl_NewStringObj(markerPtr->object.name, -1));
		    break;
		}
	    }
	}
    }
    Tcl_SetObjResult(interp, listObjPtr);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * BindOp --
 *
 *	.g element bind elemName sequence command
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
BindOp(
    Graph *graphPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    if (objc == 3) {
	Blt_HashEntry *hp;
	Blt_HashSearch cursor;
	Tcl_Obj *listObjPtr;

	listObjPtr = Tcl_NewListObj(0, (Tcl_Obj **)NULL);
	for (hp = Blt_FirstHashEntry(&graphPtr->markers.tagTable, &cursor);
	    hp != NULL; hp = Blt_NextHashEntry(&cursor)) {
	    char *tag;
	    Tcl_Obj *objPtr;

	    tag = Blt_GetHashKey(&graphPtr->markers.tagTable, hp);
	    objPtr = Tcl_NewStringObj(tag, -1);
	    Tcl_ListObjAppendElement(interp, listObjPtr, objPtr);
	}
	Tcl_SetObjResult(interp, listObjPtr);
	return TCL_OK;
    }
    return Blt_ConfigureBindingsFromObj(interp, graphPtr->bindTable,
	Blt_MakeMarkerTag(graphPtr, Tcl_GetString(objv[3])),
	objc - 4, objv + 4);
}

/*
 * ----------------------------------------------------------------------
 *
 * CgetOp --
 *
 * ----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
CgetOp(
    Graph *graphPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Marker *markerPtr;

    if (GetMarkerFromObj(interp, graphPtr, objv[3], &markerPtr) != TCL_OK) {
	return TCL_ERROR;
    }
    if (Blt_ConfigureValueFromObj(interp, graphPtr->tkwin, 
	markerPtr->classPtr->configSpecs, (char *)markerPtr, objv[4], 0) 
	!= TCL_OK) {
	return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 * ----------------------------------------------------------------------
 *
 * ConfigureOp --
 *
 * Results:
 *	The return value is a standard Tcl result.
 *
 * Side Effects:
 *
 * ----------------------------------------------------------------------
 */
static int
ConfigureOp(
    Graph *graphPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Marker *markerPtr;
    Tcl_Obj *CONST *options;
    char *oldName;
    char *string;
    int flags = BLT_CONFIG_OBJV_ONLY;
    int nNames, nOpts;
    int i;
    int under;

    /* Figure out where the option value pairs begin */
    objc -= 3;
    objv += 3;
    for (i = 0; i < objc; i++) {
	string = Tcl_GetString(objv[i]);
	if (string[0] == '-') {
	    break;
	}
	if (GetMarkerFromObj(interp, graphPtr, objv[i], &markerPtr) != TCL_OK) {
	    return TCL_ERROR;
	}
    }
    nNames = i;			/* Number of element names specified */
    nOpts = objc - i;		/* Number of options specified */
    options = objv + nNames;	/* Start of options in objv  */
    
    for (i = 0; i < nNames; i++) {
	GetMarkerFromObj(interp, graphPtr, objv[i], &markerPtr);
	if (nOpts == 0) {
	    return Blt_ConfigureInfoFromObj(interp, graphPtr->tkwin, 
		markerPtr->classPtr->configSpecs, (char *)markerPtr, 
		(Tcl_Obj *)NULL, flags);
	} else if (nOpts == 1) {
	    return Blt_ConfigureInfoFromObj(interp, graphPtr->tkwin,
		markerPtr->classPtr->configSpecs, (char *)markerPtr, 
		options[0], flags);
	}
	/* Save the old marker name. */
	oldName = markerPtr->object.name;
	under = markerPtr->drawUnder;
	if (Blt_ConfigureWidgetFromObj(interp, graphPtr->tkwin, 
		markerPtr->classPtr->configSpecs, nOpts, options, 
		(char *)markerPtr, flags) != TCL_OK) {
	    return TCL_ERROR;
	}
	if (oldName != markerPtr->object.name) {
	    if (RenameMarker(graphPtr, markerPtr, oldName, 
			     markerPtr->object.name) != TCL_OK) {
		markerPtr->object.name = oldName;
		return TCL_ERROR;
	    }
	}
	if ((*markerPtr->classPtr->configProc) (markerPtr) != TCL_OK) {

	    return TCL_ERROR;
	}
	if (markerPtr->drawUnder != under) {
	    graphPtr->flags |= REDRAW_BACKING_STORE;
	}
    }
    return TCL_OK;
}

/*
 * ----------------------------------------------------------------------
 *
 * CreateOp --
 *
 *	This procedure creates and initializes a new marker.
 *
 * Results:
 *	The return value is a pointer to a structure describing
 *	the new element.  If an error occurred, then the return
 *	value is NULL and an error message is left in interp->result.
 *
 * Side effects:
 *	Memory is allocated, etc.
 *
 * ----------------------------------------------------------------------
 */
static int
CreateOp(
    Graph *graphPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Marker *markerPtr;
    Blt_HashEntry *hPtr;
    int isNew;
    int classId;
    int i;
    char *name;
    char ident[200];
    char *string;
    size_t length;
    char c;

    string = Tcl_GetString(objv[3]);
    c = string[0];
    /* Create the new marker based upon the given type */
    if ((c == 't') && (strcmp(string, "text") == 0)) {
	classId = OBJECT_CLASS_TEXT_MARKER;
    } else if ((c == 'l') && (strcmp(string, "line") == 0)) {
	classId = OBJECT_CLASS_LINE_MARKER;
    } else if ((c == 'p') && (strcmp(string, "polygon") == 0)) {
	classId = OBJECT_CLASS_POLYGON_MARKER;
    } else if ((c == 'i') && (strcmp(string, "image") == 0)) {
	classId = OBJECT_CLASS_IMAGE_MARKER;
    } else if ((c == 'b') && (strcmp(string, "bitmap") == 0)) {
	classId = OBJECT_CLASS_BITMAP_MARKER;
    } else if ((c == 'w') && (strcmp(string, "window") == 0)) {
	classId = OBJECT_CLASS_WINDOW_MARKER;
    } else {
	Tcl_AppendResult(interp, "unknown marker type \"", string,
    "\": should be \"text\", \"line\", \"polygon\", \"bitmap\", \"image\", or \
\"window\"", (char *)NULL);
	return TCL_ERROR;
    }
    /* Scan for "-name" option. We need it for the component name */
    name = NULL;
    for (i = 4; i < objc; i += 2) {
	string = Tcl_GetString(objv[i]);
	length = strlen(string);
	if ((length > 1) && (strncmp(string, "-name", length) == 0)) {
	    name = Tcl_GetString(objv[i + 1]);
	    break;
	}
    }
    /* If no name was given for the marker, make up one. */
    if (name == NULL) {
	sprintf(ident, "marker%d", graphPtr->nextMarkerId++);
	name = ident;
    } else if (name[0] == '-') {
	Tcl_AppendResult(interp, "name of marker \"", name, 
		"\" can't start with a '-'", (char *)NULL);
	return TCL_ERROR;
    }
    markerPtr = CreateMarker(graphPtr, name, classId);
    if (Blt_ConfigureComponentFromObj(interp, graphPtr->tkwin, name, 
		markerPtr->object.className, markerPtr->classPtr->configSpecs, 
		objc - 4, objv + 4, (char *)markerPtr, 0) != TCL_OK) {
	DestroyMarker(markerPtr);
	return TCL_ERROR;
    }
    if ((*markerPtr->classPtr->configProc) (markerPtr) != TCL_OK) {
	DestroyMarker(markerPtr);
	return TCL_ERROR;
    }
    hPtr = Blt_CreateHashEntry(&graphPtr->markers.table, name, &isNew);
    if (!isNew) {
	Marker *oldPtr;
	/*
	 * Marker by the same name already exists.  Delete the old
	 * marker and it's list entry.  But save the hash entry.
	 */
	oldPtr = (Marker *)Blt_GetHashValue(hPtr);
	oldPtr->hashPtr = NULL;
	DestroyMarker(oldPtr);
    }
    Blt_SetHashValue(hPtr, markerPtr);
    markerPtr->hashPtr = hPtr;
    markerPtr->linkPtr = 
	Blt_ChainAppend(graphPtr->markers.displayList, markerPtr);
    if (markerPtr->drawUnder) {
	graphPtr->flags |= REDRAW_BACKING_STORE;
    }
    Blt_EventuallyRedrawGraph(graphPtr);
    Tcl_SetStringObj(Tcl_GetObjResult(interp), name, -1);
    return TCL_OK;
}

/*
 * ----------------------------------------------------------------------
 *
 * DeleteOp --
 *
 *	Deletes the marker given by markerId.
 *
 * Results:
 *	The return value is a standard Tcl result.
 *
 * Side Effects:
 *	Graph will be redrawn to reflect the new display list.
 *
 * ----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
DeleteOp(
    Graph *graphPtr,
    Tcl_Interp *interp,		/* Not used. */
    int objc,
    Tcl_Obj *CONST *objv)
{
    int i;

    for (i = 3; i < objc; i++) {
	Marker *markerPtr;

	if (GetMarkerFromObj((Tcl_Interp *)NULL, graphPtr, objv[i], 
		     &markerPtr) == TCL_OK) {
	    DestroyMarker(markerPtr);
	}
    }
    Blt_EventuallyRedrawGraph(graphPtr);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * GetOp --
 *
 * 	Find the legend entry from the given argument.  The argument
 *	can be either a screen position "@x,y" or the name of an
 *	element.
 *
 *	I don't know how useful it is to test with the name of an
 *	element.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side Effects:
 *	Graph will be redrawn to reflect the new legend attributes.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
GetOp(
    Graph *graphPtr,
    Tcl_Interp *interp,
    int objc,			/* Not used. */
    Tcl_Obj *CONST *objv)
{
    Marker *markerPtr;
    char *string;

    string = Tcl_GetString(objv[3]);
    if ((string[0] == 'c') && (strcmp(string, "current") == 0)) {
	markerPtr = (Marker *)Blt_GetCurrentItem(graphPtr->bindTable);
	if (markerPtr == NULL) {
	    return TCL_OK;	/* Report only on markers. */

	}
	if (markerPtr->object.classId & OBJECT_CLASS_MARKER) {
	    Tcl_SetStringObj(Tcl_GetObjResult(interp), 
		markerPtr->object.name, -1);
	}
    }
    return TCL_OK;
}

/*
 * ----------------------------------------------------------------------
 *
 * RelinkOp --
 *
 *	Reorders the marker (given by the first name) before/after
 *	the another marker (given by the second name) in the
 *	marker display list.  If no second name is given, the
 *	marker is placed at the beginning/end of the list.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side Effects:
 *	Graph will be redrawn to reflect the new display list.
 *
 * ----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
RelinkOp(
    Graph *graphPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_ChainLink *linkPtr, *placePtr;
    Marker *markerPtr;
    char *string;

    /* Find the marker to be raised or lowered. */
    if (GetMarkerFromObj(interp, graphPtr, objv[3], &markerPtr) != TCL_OK) {
	return TCL_ERROR;
    }
    /* Right now it's assumed that all markers are always in the
       display list. */
    linkPtr = markerPtr->linkPtr;
    Blt_ChainUnlinkLink(graphPtr->markers.displayList, markerPtr->linkPtr);

    placePtr = NULL;
    if (objc == 5) {
	if (GetMarkerFromObj(interp, graphPtr, objv[4], &markerPtr) != TCL_OK) {
	    return TCL_ERROR;
	}
	placePtr = markerPtr->linkPtr;
    }

    /* Link the marker at its new position. */
    string = Tcl_GetString(objv[2]);
    if (string[0] == 'a') {
	Blt_ChainLinkAfter(graphPtr->markers.displayList, linkPtr, placePtr);
    } else {
	Blt_ChainLinkBefore(graphPtr->markers.displayList, linkPtr, placePtr);
    }
    if (markerPtr->drawUnder) {
	graphPtr->flags |= REDRAW_BACKING_STORE;
    }
    Blt_EventuallyRedrawGraph(graphPtr);
    return TCL_OK;
}


/*
 * ----------------------------------------------------------------------
 *
 * FindOp --
 *
 *	Returns if marker by a given ID currently exists.
 *
 * Results:
 *	A standard Tcl result.
 *
 * ----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
FindOp(
    Graph *graphPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_ChainLink *linkPtr;
    Region2D extents;
    Marker *markerPtr;
    char *string;
    int enclosed;
    int left, right, top, bottom;
    int mode;

#define FIND_ENCLOSED	 (1<<0)
#define FIND_OVERLAPPING (1<<1)
    string = Tcl_GetString(objv[3]);
    if (strcmp(string, "enclosed") == 0) {
	mode = FIND_ENCLOSED;
    } else if (strcmp(string, "overlapping") == 0) {
	mode = FIND_OVERLAPPING;
    } else {
	Tcl_AppendResult(interp, "bad search type \"", string, 
		": should be \"enclosed\", or \"overlapping\"", (char *)NULL);
	return TCL_ERROR;
    }

    if ((Tcl_GetIntFromObj(interp, objv[4], &left) != TCL_OK) ||
	(Tcl_GetIntFromObj(interp, objv[5], &top) != TCL_OK) ||
	(Tcl_GetIntFromObj(interp, objv[6], &right) != TCL_OK) ||
	(Tcl_GetIntFromObj(interp, objv[7], &bottom) != TCL_OK)) {
	return TCL_ERROR;
    }
    if (left < right) {
	extents.left = (double)left;
	extents.right = (double)right;
    } else {
	extents.left = (double)right;
	extents.right = (double)left;
    }
    if (top < bottom) {
	extents.top = (double)top;
	extents.bottom = (double)bottom;
    } else {
	extents.top = (double)bottom;
	extents.bottom = (double)top;
    }
    enclosed = (mode == FIND_ENCLOSED);
    for (linkPtr = Blt_ChainFirstLink(graphPtr->markers.displayList);
	 linkPtr != NULL; linkPtr = Blt_ChainNextLink(linkPtr)) {
	markerPtr = Blt_ChainGetValue(linkPtr);
	if (markerPtr->object.hidden) {
	    continue;
	}
	if (markerPtr->elemName != NULL) {
	    Blt_HashEntry *hPtr;
	    
	    hPtr = Blt_FindHashEntry(&graphPtr->elements.table, 
				     markerPtr->elemName);
	    if (hPtr != NULL) {
		Element *elemPtr;
		
		elemPtr = (Element *)Blt_GetHashValue(hPtr);
		if (elemPtr->object.hidden) {
		    continue;
		}
	    }
	}
	if ((*markerPtr->classPtr->regionProc)(markerPtr, &extents, enclosed)) {
	    Tcl_SetStringObj(Tcl_GetObjResult(interp), 
			     markerPtr->object.name, -1);
	    return TCL_OK;
	}
    }
    Tcl_SetStringObj(Tcl_GetObjResult(interp), "", -1);
    return TCL_OK;
}

/*
 * ----------------------------------------------------------------------
 *
 * ExistsOp --
 *
 *	Returns if marker by a given ID currently exists.
 *
 * Results:
 *	A standard Tcl result.
 *
 * ----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
ExistsOp(
    Graph *graphPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_HashEntry *hPtr;

    hPtr = Blt_FindHashEntry(&graphPtr->markers.table, Tcl_GetString(objv[3]));
    Tcl_SetBooleanObj(Tcl_GetObjResult(interp), (hPtr != NULL));
    return TCL_OK;
}

/*
 * ----------------------------------------------------------------------
 *
 * TypeOp --
 *
 *	Returns a symbolic name for the type of the marker whose ID is
 *	given.
 *
 * Results:
 *	A standard Tcl result. interp->result will contain the symbolic
 *	type of the marker.
 *
 * ----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
TypeOp(
    Graph *graphPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Marker *markerPtr;

    if (GetMarkerFromObj(interp, graphPtr, objv[3], &markerPtr) != TCL_OK) {
	return TCL_ERROR;
    }
    Tcl_SetStringObj(Tcl_GetObjResult(interp), markerPtr->object.className, -1);
    return TCL_OK;
}

/* Public routines */

/*
 * ----------------------------------------------------------------------
 *
 * Blt_MarkerOp --
 *
 *	This procedure is invoked to process the Tcl command
 *	that corresponds to a widget managed by this module.
 *	See the user documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 * ----------------------------------------------------------------------
 */

static Blt_OpSpec markerOps[] =
{
    {"after", 1, (Blt_Op)RelinkOp, 4, 5, "marker ?afterMarker?",},
    {"before", 2, (Blt_Op)RelinkOp, 4, 5, "marker ?beforeMarker?",},
    {"bind", 2, (Blt_Op)BindOp, 3, 6, "marker sequence command",},
    {"cget", 2, (Blt_Op)CgetOp, 5, 5, "marker option",},
    {"configure", 2, (Blt_Op)ConfigureOp, 4, 0,
	"marker ?marker?... ?option value?...",},
    {"create", 2, (Blt_Op)CreateOp, 4, 0,
	"type ?option value?...",},
    {"delete", 1, (Blt_Op)DeleteOp, 3, 0, "?marker?...",},
    {"exists", 1, (Blt_Op)ExistsOp, 4, 4, "marker",},
    {"find", 1, (Blt_Op)FindOp, 8, 8, "enclosed|overlapping x1 y1 x2 y2",},
    {"get", 1, (Blt_Op)GetOp, 4, 4, "name",},
    {"names", 1, (Blt_Op)NamesOp, 3, 0, "?pattern?...",},
    {"type", 1, (Blt_Op)TypeOp, 4, 4, "marker",},
};
static int nMarkerOps = sizeof(markerOps) / sizeof(Blt_OpSpec);

/*ARGSUSED*/
int
Blt_MarkerOp(
    Graph *graphPtr,
    Tcl_Interp *interp,		/* Not used. */
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_Op proc;
    int result;

    proc = Blt_GetOpFromObj(interp, nMarkerOps, markerOps, BLT_OP_ARG2, 
	objc, objv,0);
    if (proc == NULL) {
	return TCL_ERROR;
    }
    result = (*proc) (graphPtr, interp, objc, objv);
    return result;
}

/*
 * -------------------------------------------------------------------------
 *
 * Blt_MarkersToPostScript --
 *
 * -------------------------------------------------------------------------
 */
void
Blt_MarkersToPostScript(Graph *graphPtr, Blt_PostScript ps, int under)
{
    Blt_ChainLink *lp;

    for (lp = Blt_ChainFirstLink(graphPtr->markers.displayList); lp != NULL; 
	 lp = Blt_ChainNextLink(lp)) {
	Marker *markerPtr;

	markerPtr = Blt_ChainGetValue(lp);
	if ((markerPtr->classPtr->postscriptProc == NULL) || 
	    (markerPtr->nWorldPts == 0)) {
	    continue;
	}
	if (markerPtr->drawUnder != under) {
	    continue;
	}
	if (markerPtr->object.hidden) {
	    continue;
	}
	if (markerPtr->elemName != NULL) {
	    Blt_HashEntry *hPtr;

	    hPtr = Blt_FindHashEntry(&graphPtr->elements.table, 
			     markerPtr->elemName);
	    if (hPtr != NULL) {
		Element *elemPtr;

		elemPtr = (Element *)Blt_GetHashValue(hPtr);
		if (elemPtr->object.hidden) {
		    continue;
		}
	    }
	}
	Blt_AppendToPostScript(ps, "\n% Marker \"", markerPtr->object.name, 
		"\" is a ", markerPtr->object.className, ".\n", (char *)NULL);
	(*markerPtr->classPtr->postscriptProc) (markerPtr, ps);
    }
}

/*
 * -------------------------------------------------------------------------
 *
 * Blt_DrawMarkers --
 *
 *	Calls the individual drawing routines (based on marker type)
 *	for each marker in the display list.
 *
 *	A marker will not be drawn if
 *
 *	1) An element linked to the marker (indicated by elemName) 
 *	   is currently hidden.
 *
 *	2) No coordinates have been specified for the marker.
 *
 *	3) The marker is requesting to be drawn at a different level
 *	   (above/below the elements) from the current mode.
 *
 *	4) The marker is configured as hidden (-hide option).
 *
 *	5) The marker isn't visible in the current viewport
 *	   (i.e. clipped).
 *
 * Results:
 *	None
 *
 * Side Effects:
 *	Markers are drawn into the drawable (pixmap) which will eventually
 *	be displayed in the graph window.
 *
 * -------------------------------------------------------------------------
 */
void
Blt_DrawMarkers(
    Graph *graphPtr,
    Drawable drawable,		/* Pixmap or window to draw into */
    int under)
{
    Blt_ChainLink *lp;

    for (lp = Blt_ChainFirstLink(graphPtr->markers.displayList); lp != NULL; 
	 lp = Blt_ChainNextLink(lp)) {
	Marker *markerPtr;

	markerPtr = Blt_ChainGetValue(lp);

	if ((markerPtr->nWorldPts == 0) || 
	    (markerPtr->drawUnder != under) ||
	    (markerPtr->object.hidden) || 
	    (markerPtr->clipped)) {
	    continue;
	}
	if (markerPtr->elemName != NULL) {
	    Blt_HashEntry *hPtr;

	    /* Look up the named element and see if it's hidden */
	    hPtr = Blt_FindHashEntry(&graphPtr->elements.table, 
				     markerPtr->elemName);
	    if (hPtr != NULL) {
		Element *elemPtr;

		elemPtr = (Element *)Blt_GetHashValue(hPtr);
		if (elemPtr->object.hidden) {
		    continue;
		}
	    }
	}

	(*markerPtr->classPtr->drawProc) (markerPtr, drawable);
    }
}

void
Blt_MapMarkers(Graph *graphPtr)
{
    Blt_ChainLink *lp;

    for (lp = Blt_ChainFirstLink(graphPtr->markers.displayList); lp != NULL; 
	 lp = Blt_ChainNextLink(lp)) {
	Marker *markerPtr;

	markerPtr = Blt_ChainGetValue(lp);
	if ((markerPtr->nWorldPts == 0) || (markerPtr->object.hidden)) {
	    continue;
	}
	if ((graphPtr->flags & MAP_ALL) || 
	    (markerPtr->flags & MAP_ITEM)) {
	    (*markerPtr->classPtr->mapProc) (markerPtr);
	    markerPtr->flags &= ~MAP_ITEM;
	}
    }
}


void
Blt_DestroyMarkers(Graph *graphPtr)
{
    Blt_HashEntry *hp;
    Blt_HashSearch cursor;

    for (hp = Blt_FirstHashEntry(&graphPtr->markers.table, &cursor); hp != NULL;
	 hp = Blt_NextHashEntry(&cursor)) {
	Marker *markerPtr;

	markerPtr = (Marker *)Blt_GetHashValue(hp);
	/*
	 * Dereferencing the pointer to the hash table prevents the
	 * hash table entry from being automatically deleted.
	 */
	markerPtr->hashPtr = NULL;
	DestroyMarker(markerPtr);
    }
    Blt_DeleteHashTable(&graphPtr->markers.table);
    Blt_DeleteHashTable(&graphPtr->markers.tagTable);
    Blt_ChainDestroy(graphPtr->markers.displayList);
}

Marker *
Blt_NearestMarker(
    Graph *graphPtr,
    int x, int y,			/* Screen coordinates */
    int under)
{
    Blt_ChainLink *lp;
    Point2D point;

    point.x = (double)x;
    point.y = (double)y;
    for (lp = Blt_ChainLastLink(graphPtr->markers.displayList);
	lp != NULL; lp = Blt_ChainPrevLink(lp)) {
	Marker *markerPtr;

	markerPtr = Blt_ChainGetValue(lp);
	/* 
	 * Don't consider markers that are pending to be mapped. Even
	 * if the marker has already been mapped, the coordinates
	 * could be invalid now.  Better to pick no marker than the
	 * wrong marker.
	 */
	if ((markerPtr->drawUnder == under) && (markerPtr->nWorldPts > 0) && 
	    ((markerPtr->flags & MAP_ITEM) == 0) && 
	    (!markerPtr->object.hidden) && (markerPtr->state == STATE_NORMAL)) {
	    if ((*markerPtr->classPtr->pointProc) (markerPtr, &point)) {
		return markerPtr;
	    }
	}
    }
    return NULL;
}

ClientData
Blt_MakeMarkerTag(Graph *graphPtr, CONST char *tagName)
{
    Blt_HashEntry *hPtr;
    int isNew;

    hPtr = Blt_CreateHashEntry(&graphPtr->markers.tagTable, tagName, &isNew);
    assert(hPtr);
    return Blt_GetHashKey(&graphPtr->markers.tagTable, hPtr);
}

