
/*
 * bltPictureTif.c --
 *
 * This module implements TIF file format conversion routines for
 * the picture image type in the BLT toolkit.
 *
 *	Copyright 2003-2005 George A Howlett.
 *
 *	Permission is hereby granted, free of charge, to any person
 *	obtaining a copy of this software and associated documentation
 *	files (the "Software"), to deal in the Software without
 *	restriction, including without limitation the rights to use,
 *	copy, modify, merge, publish, distribute, sublicense, and/or
 *	sell copies of the Software, and to permit persons to whom the
 *	Software is furnished to do so, subject to the following
 *	conditions:
 *
 *	The above copyright notice and this permission notice shall be
 *	included in all copies or substantial portions of the
 *	Software.
 *
 *	THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY
 *	KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 *	WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 *	PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS
 *	OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
 *	OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 *	OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 *	SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 */

#include "bltInt.h"

#ifdef HAVE_LIBTIFF

#include <bltSink.h>
#include <bltHash.h>
#include "bltPicture.h"
#include "bltPictureFormats.h"

#include <X11/Xutil.h>

#if defined(WIN32) || defined(MACOSX)
#include <setjmp.h>
#endif

typedef struct Blt_PictureStruct Picture;

#include <tiffio.h>

static tsize_t
TifRead(thandle_t handle, tdata_t out, tsize_t nWanted)
{
    Blt_DataSink data = (Blt_DataSink)handle;

    if (Blt_SinkBytesLeft(data) < nWanted) {
	nWanted = Blt_SinkBytesLeft(data);
    }
    if (nWanted > 0) {
	memcpy(out, Blt_SinkPointer(data), nWanted);
	Blt_SinkIncrCursor(data, nWanted);
    }
    return nWanted;
}

static toff_t
TifSeek(thandle_t handle, toff_t offset, int whence)
{
    Blt_DataSink data = (Blt_DataSink)handle;

    if (whence == SEEK_CUR) {
	offset += Blt_SinkCursor(data);
    } else if (whence == SEEK_END) {
	offset += Blt_SinkMark(data);
    }
    if (offset < 0) {
	return -1;
    }
    if (offset > Blt_SinkLength(data)) {
	/* Attempting to seek past the end of the current
	 * buffer. Resize the buffer */
	Blt_SinkSetLength(data, offset);
    }
    Blt_SinkSetCursor(data, offset);
    return (toff_t) Blt_SinkCursor(data);
}

static toff_t
TifSize(thandle_t handle)
{
    Blt_DataSink data = (Blt_DataSink)handle;

    return Blt_SinkMark(data);
}

static int
TifMapFile(thandle_t handle, tdata_t *data, toff_t *offsetPtr)
{
    return 0;
}

static void
TifUnmapFile(thandle_t handle, tdata_t data, toff_t offset)
{
}

static int
TifClose(thandle_t handle)
{
    return 0;
}


static tsize_t
TifWrite(thandle_t handle, tdata_t out, tsize_t nBytes)
{
    Blt_DataSink data = (Blt_DataSink)handle;
    unsigned int needed;

    needed = Blt_SinkCursor(data) + nBytes;
    if (needed > Blt_SinkLength(data)) {
	if (!Blt_SinkSetLength(data, needed)) {
	    return -1;
	}
    }
    memcpy(Blt_SinkPointer(data), out, nBytes);
    Blt_SinkIncrCursor(data, nBytes);
    if (Blt_SinkCursor(data) > Blt_SinkMark(data)) {
	Blt_SinkSetMark(data, Blt_SinkCursor(data));
    }
    return nBytes;
}

typedef struct {
    Tcl_DString errors;
    Tcl_DString warnings;
    int nWarnings, nErrors;
} TifMessage;

static TifMessage *tifMessagePtr;

static void 
TifError(const char *routine, const char *fmt, va_list ap)
{
    char string[BUFSIZ+4];
    int length;

    length = vsnprintf(string, BUFSIZ, fmt, ap);
    if (length > BUFSIZ) {
	strcat(string, "...");
    }
    Tcl_DStringAppend(&tifMessagePtr->errors, string, -1);
    tifMessagePtr->nErrors++;
}

/* warnings are not processed in Tcl */
static void
TifWarning(const char *routine, const char *fmt, va_list ap)
{
    char string[BUFSIZ+4];
    int length;

    length = vsnprintf(string, BUFSIZ, fmt, ap);
    if (length > BUFSIZ) {
	strcat(string, "...");
    }
    Tcl_DStringAppend(&tifMessagePtr->warnings, string, -1);
    Tcl_DStringAppend(&tifMessagePtr->warnings, "\n", -1);
    tifMessagePtr->nWarnings++;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_IsTif --
 *
 *      Attempts to parse a TIF file header.
 *
 * Results:
 *      Returns 1 is the header is TIF and 0 otherwise.  Note that
 *      the validity of the header contents is not checked here.  That's
 *	done in Blt_TifToPicture.
 *
 *----------------------------------------------------------------------
 */
int
Blt_IsTif(Blt_DataSink data)
{
    unsigned char *bp;
    int bool;

    Blt_SinkResetCursor(data);
    bp = Blt_SinkPointer(data);
    bool = (((bp[0] == 'M') && (bp[1] == 'M')) || 
	    ((bp[0] == 'I') && (bp[1] == 'I')));
    return bool;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_TifToPicture --
 *
 *      Reads a TIFF file and converts it into a picture.
 *
 * Results:
 *      The picture is returned.  If an error occured, such
 *	as the designated file could not be opened, NULL is returned.
 *
 *----------------------------------------------------------------------
 */
Blt_Picture
Blt_TifToPicture(
    Tcl_Interp *interp, 
    char *fileName,
    Blt_DataSink data)
{
    Picture *destPtr;
    Pix32 *destRowPtr;
    TIFF *tifPtr;
    TIFFErrorHandler oldErrorHandler, oldWarningHandler;
    TifMessage message;
    int nPixels, width, height;
    int y;
    uint32 *srcBuffer, *sp;

    Tcl_SetErrorCode(interp, "", (char *)NULL);

    message.nWarnings = message.nErrors = 0;
    Tcl_DStringInit(&message.errors);
    Tcl_DStringInit(&message.warnings);
    Tcl_DStringAppend(&message.errors, "error reading \"", -1);
    Tcl_DStringAppend(&message.errors, fileName, -1);
    Tcl_DStringAppend(&message.errors, "\": ", -1);
    tifMessagePtr = &message;

    oldErrorHandler = TIFFSetErrorHandler(TifError);
    oldWarningHandler = TIFFSetWarningHandler(TifWarning);

    srcBuffer = NULL;
    destPtr = NULL;
    tifPtr = TIFFClientOpen(fileName, "r", (thandle_t)data,
	TifRead,		/* TIFFReadWriteProc */
	TifWrite, 		/* TIFFReadWriteProc */
	TifSeek, 		/* TIFFSeekProc */
	TifClose, 		/* TIFFCloseProc */
	TifSize, 		/* TIFFSizeProc */
	TifMapFile, 		/* TIFFMapFileProc */
	TifUnmapFile);		/* TIFFUnmapFileProc */
    if (tifPtr == NULL) {
	goto bad;
    }
    TIFFGetField(tifPtr, TIFFTAG_IMAGEWIDTH, &width);
    TIFFGetField(tifPtr, TIFFTAG_IMAGELENGTH, &height);
    nPixels = width * height;
    srcBuffer = _TIFFmalloc(sizeof(uint32) * nPixels);
    if (srcBuffer == NULL) {
	Tcl_AppendResult(interp, "can't allocate ", Blt_Itoa(nPixels), 
		" buffer for TIF image", (char *)NULL);
	goto bad;
    }
    if (!TIFFReadRGBAImage(tifPtr, width, height, srcBuffer, 
	/*stopOnError*/0)) {
	goto bad;
    }	
    destPtr = Blt_CreatePicture(width, height);
    destRowPtr = destPtr->bits + (destPtr->pixelsPerRow * (height - 1));
    sp  = srcBuffer;
    for (y = height - 1; y >= 0; y--) {
	Pix32 *dp;
	int x;

	dp = destRowPtr;
	for (x = 0; x < width; x++) {
	    dp->Red = TIFFGetR(*sp);
	    dp->Green = TIFFGetG(*sp);
	    dp->Blue = TIFFGetB(*sp);
	    dp->Alpha = TIFFGetA(*sp);
	    dp++, sp++;
	}
	destRowPtr -= destPtr->pixelsPerRow;
    }
 bad:
    if (srcBuffer != NULL) {
	_TIFFfree(srcBuffer);
    }
    if (tifPtr != NULL) {
	TIFFClose(tifPtr);
    }
    TIFFSetErrorHandler(oldErrorHandler);
    TIFFSetWarningHandler(oldWarningHandler);
    if (message.nWarnings > 0) {
	Tcl_SetErrorCode(interp, "PICTURE", "TIF_READ_WARNINGS", 
		Tcl_DStringValue(&message.warnings), (char *)NULL);
    }
    Tcl_DStringFree(&message.warnings);
    if (message.nErrors > 0) {
	Tcl_AppendResult(interp, Tcl_DStringValue(&message.errors), 
			 (char *)NULL); 
    }
    Tcl_DStringFree(&message.errors);
    return destPtr;
}

static int tifCompressionSchemes[] = {
    COMPRESSION_NONE,

#ifdef LZW_SUPPORT
    COMPRESSION_LZW,		/* Lempel-Ziv & Welch */
#else 
    COMPRESSION_NONE,
#endif

#ifdef OJPEG_SUPPORT
    COMPRESSION_OJPEG,		/* !6.0 JPEG */
#else 
    COMPRESSION_NONE,
#endif /* OJPEG_SUPPORT */

#ifdef JPEG_SUPPORT
    COMPRESSION_JPEG,		/* %JPEG DCT compression */
#else 
    COMPRESSION_NONE,
#endif /* JPEG_SUPPORT */

#ifdef NEXT_SUPPORT
    COMPRESSION_NEXT,		/* NeXT 2-bit RLE */
#else 
    COMPRESSION_NONE,
#endif /* NEXT_SUPPORT */

#ifdef PACKBITS_SUPPORT
    COMPRESSION_PACKBITS,	/* Macintosh RLE */
#else 
    COMPRESSION_NONE,
#endif /* PACKBITS_SUPPORT */

#ifdef THUNDER_SUPPORT
    COMPRESSION_THUNDERSCAN,	/* ThunderScan RLE */
#else 
    COMPRESSION_NONE,
#endif /* THUNDER_SUPPORT */

    COMPRESSION_PIXARFILM,	/* Pixar 10-bit LZW */

#ifdef PIXARLOG_SUPPORT
    COMPRESSION_PIXARLOG,	/* Pixar 11-bit ZIP */
#else 
    COMPRESSION_NONE,
#endif /* PIXARLOG_SUPPORT */

#ifdef ZIP_SUPPORT
    COMPRESSION_DEFLATE,	/* Deflate compression */
#else 
    COMPRESSION_NONE,
#endif /* ZIP_SUPPORT */

#ifdef ADOBE_SUPPORT
    COMPRESSION_ADOBE_DEFLATE,	/* Adobe's deflate */
#else
    COMPRESSION_NONE,
#endif /* ADOBE_SUPPORT */
    COMPRESSION_DCS,		/* Kodak DCS encoding */

#ifdef SGILOG_SUPPORT
    COMPRESSION_SGILOG,		/* SGI Log Luminance RLE */
    COMPRESSION_SGILOG24,	/* SGI Log 24-bit packed */
#else 
    COMPRESSION_NONE,
    COMPRESSION_NONE,
#endif /* SGILOG_SUPPORT */

};

/*
 *----------------------------------------------------------------------
 *
 * Blt_PictureToTif --
 *
 *      Writes a TIFF format image to the provided data buffer.
 *
 * Results:
 *      A standard Tcl result.  If an error occured, TCL_ERROR is
 *	returned and an error message will be place in the interpreter
 *	result. Otherwise, the data sink will contain the binary
 *	output of the image.
 *
 * Side Effects:
 *	Memory is allocated for the data sink.
 *
 *----------------------------------------------------------------------
 */
int 
Blt_PictureToTif(
    Tcl_Interp *interp, 
    Blt_Picture picture,
    Blt_DataSink dest,
    int compress)
{
    TIFF *tifPtr;
    TIFFErrorHandler oldErrorHandler, oldWarningHandler;
    TifMessage message;
    int photometric, samplesPerPixel;
    int result, nColors;
    Picture *srcPtr;

    Tcl_SetErrorCode(interp, "", (char *)NULL);

    compress = tifCompressionSchemes[compress];
    if (compress == COMPRESSION_NONE) {
	fprintf(stderr, "not compressing TIFF output\n");
    }
#ifdef notdef
    if (!TIFFIsCODECConfigured((unsigned short int)compress)) {
	compress = COMPRESSION_NONE;
    }	
#endif
    srcPtr = picture;

    Tcl_DStringInit(&message.errors);
    Tcl_DStringInit(&message.warnings);
    Tcl_DStringAppend(&message.errors, "error writing TIF output: ", -1);
    tifMessagePtr = &message;
    message.nErrors = message.nWarnings = 0;

    oldErrorHandler = TIFFSetErrorHandler(TifError);
    oldWarningHandler = TIFFSetWarningHandler(TifWarning);

    tifPtr = TIFFClientOpen("data buffer", "w", (thandle_t)dest,
	TifRead,		/* TIFFReadWriteProc */
	TifWrite, 		/* TIFFReadWriteProc */
	TifSeek, 		/* TIFFSeekProc */
	TifClose, 		/* TIFFCloseProc */
	TifSize, 		/* TIFFSizeProc */
	TifMapFile, 		/* TIFFMapFileProc */
	TifUnmapFile);		/* TIFFUnmapFileProc */
    if (tifPtr == NULL) {
	Tcl_AppendResult(interp, "can't register TIF procs: ", (char *)NULL);
	return TCL_ERROR;
    }
    nColors = Blt_QueryColors(srcPtr, (Blt_HashTable *)NULL);
    if (Blt_PictureIsColor(srcPtr)) {
	samplesPerPixel = (Blt_PictureIsOpaque(srcPtr)) ? 3 : 4;
	photometric = PHOTOMETRIC_RGB;
    } else {
	if (!Blt_PictureIsOpaque(srcPtr)) {
	    Blt_Picture background;
	    Pix32 white;
	    /* Blend picture with solid color background. */
	    background = Blt_CreatePicture(srcPtr->width, srcPtr->height);
	    white.color = 0xFFFFFFFF;
	    Blt_BlankPicture(background, &white); /* White background. */
	    Blt_BlendPictures(background, srcPtr);
	    srcPtr = background;
	}
	samplesPerPixel = 1;
	photometric = PHOTOMETRIC_MINISBLACK;
    }
    TIFFSetField(tifPtr, TIFFTAG_BITSPERSAMPLE,    8);
    TIFFSetField(tifPtr, TIFFTAG_COMPRESSION, (unsigned short int)compress);
    TIFFSetField(tifPtr, TIFFTAG_IMAGELENGTH,      srcPtr->height);
    TIFFSetField(tifPtr, TIFFTAG_IMAGEWIDTH,       srcPtr->width);
    TIFFSetField(tifPtr, TIFFTAG_ORIENTATION,      ORIENTATION_TOPLEFT);
    TIFFSetField(tifPtr, TIFFTAG_PHOTOMETRIC,      photometric);
    TIFFSetField(tifPtr, TIFFTAG_PLANARCONFIG,     PLANARCONFIG_CONTIG);
    TIFFSetField(tifPtr, TIFFTAG_RESOLUTIONUNIT,   2);
    TIFFSetField(tifPtr, TIFFTAG_ROWSPERSTRIP,     srcPtr->height);
    TIFFSetField(tifPtr, TIFFTAG_SAMPLESPERPIXEL,  samplesPerPixel);
    TIFFSetField(tifPtr, TIFFTAG_SOFTWARE,         TIFFGetVersion());
    TIFFSetField(tifPtr, TIFFTAG_XRESOLUTION,      300.0f);
    TIFFSetField(tifPtr, TIFFTAG_YRESOLUTION,      300.0f);
#ifdef WORD_BIGENDIAN
    TIFFSetField(tifPtr, TIFFTAG_FILLORDER,        FILLORDER_MSB2LSB);
#else
    TIFFSetField(tifPtr, TIFFTAG_FILLORDER,        FILLORDER_LSB2MSB);
#endif
    result = -1;
    {
	Pix32 *srcRowPtr;
	int destBufferSize;
	int y;
	unsigned char *destBuffer;
	unsigned char *dp;

	destBufferSize = srcPtr->width * srcPtr->height * sizeof(uint32);
	destBuffer = (unsigned char *)_TIFFmalloc(destBufferSize);

	if (destBuffer == NULL) {
	    TIFFError("tiff", "can't allocate space for TIF buffer", 
		(char *)NULL);
	    TIFFClose(tifPtr);
	    return TCL_ERROR;
	}
	dp = destBuffer;
	srcRowPtr = srcPtr->bits;
	switch (samplesPerPixel) {
	case 4:
	    for (y = 0; y < srcPtr->height; y++) {
		Pix32 *sp;
		int x;
		
		sp = srcRowPtr;
		for (x = 0; x < srcPtr->width; x++) {
		    dp[0] = sp->Red;
		    dp[1] = sp->Green;
		    dp[2] = sp->Blue;
		    dp[3] = sp->Alpha;
		    dp += 4, sp++;
		}
		srcRowPtr += srcPtr->pixelsPerRow;
	    }
	    break;

	case 3:			/* RGB, 100% opaque image. */
	    for (y = 0; y < srcPtr->height; y++) {
		Pix32 *sp;
		int x;
		
		sp = srcRowPtr;
		for (x = 0; x < srcPtr->width; x++) {
		    dp[0] = sp->Red;
		    dp[1] = sp->Green;
		    dp[2] = sp->Blue;
		    dp += 3, sp++;
		}
		srcRowPtr += srcPtr->pixelsPerRow;
	    }
	    break;

	case 1:
	    for (y = 0; y < srcPtr->height; y++) {
		Pix32 *sp;
		int x;
		
		sp = srcRowPtr;
		for (x = 0; x < srcPtr->width; x++) {
		    *dp++ = sp->Red;
		    sp++;
		}
		srcRowPtr += srcPtr->pixelsPerRow;
	    }
	    break;

	}
	result = TIFFWriteEncodedStrip(tifPtr, 0, destBuffer, destBufferSize);
	if (result < 0) {
	    Tcl_AppendResult(interp, "error writing TIFF encoded strip",
			     (char *)NULL);
	}
	_TIFFfree(destBuffer);
    }
    TIFFClose(tifPtr);
    if (result == -1) {
	Blt_SinkFree(dest);
    }
    TIFFSetErrorHandler(oldErrorHandler);
    TIFFSetWarningHandler(oldWarningHandler);
    if (message.nWarnings > 0) {
	Tcl_SetErrorCode(interp, "PICTURE", "TIF_WRITE_WARNINGS", 
		Tcl_DStringValue(&message.warnings), (char *)NULL);
    }
    Tcl_DStringFree(&message.warnings);
    if (message.nErrors > 0) {
	Tcl_DStringResult(interp, &message.errors);
    }
    Tcl_DStringFree(&message.errors);
    if (srcPtr != picture) {
	Blt_FreePicture(srcPtr);
    }
    return (result == -1) ? TCL_ERROR : TCL_OK;
}

#endif /* HAVE_LIBTIFF */
