
/*
 * bltTree.c --
 *
 *	Copyright 1998-2004 George A Howlett.
 *
 *	Permission is hereby granted, free of charge, to any person
 *	obtaining a copy of this software and associated documentation
 *	files (the "Software"), to deal in the Software without
 *	restriction, including without limitation the rights to use,
 *	copy, modify, merge, publish, distribute, sublicense, and/or
 *	sell copies of the Software, and to permit persons to whom the
 *	Software is furnished to do so, subject to the following
 *	conditions:
 *
 *	The above copyright notice and this permission notice shall be
 *	included in all copies or substantial portions of the
 *	Software.
 *
 *	THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY
 *	KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 *	WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 *	PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS
 *	OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
 *	OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 *	OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 *	SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include "bltInt.h"

/* TODO:
 *	traces and notifiers should be in one list in tree object.
 *	notifier is always fired.
 *	incorporate first/next tag routines ?
 */


#ifndef NO_TREE

#include "bltTree.h"
#include "bltNsUtil.h"

static Tcl_InterpDeleteProc TreeInterpDeleteProc;
static Blt_TreeApplyProc SizeApplyProc;
static Tcl_IdleProc NotifyIdleProc;

#define TREE_THREAD_KEY		"BLT Tree Data"
#define TREE_MAGIC		((unsigned int) 0x46170277)
#define TREE_DESTROYED		(1<<0)

#define TREE_NODE_REBUILD_SIZE	3U

typedef struct Blt_TreeNodeStruct Node;
typedef struct Blt_TreeClientStruct TreeClient;
typedef struct Blt_TreeObjectStruct TreeObject;
typedef struct Blt_TreeValueStruct Value;

/*
 * Blt_TreeValue --
 *
 *	A tree node may have zero or more data fields or values that
 *	are represented by these container structures.  Each data
 *	field has both the name of the field (Blt_TreeKey) and its
 *	data (Tcl_Obj).  Values are private or public.  Private values
 *	are only be seen by the tree client that created the field.
 * 
 *	Values are organized in two ways. They are stored in a linked
 *	list in order that they were created.  In addition, they may
 *	be placed into a hash table when the number of values reaches
 *	a high-water mark.  
 *
 */
struct Blt_TreeValueStruct {
    Blt_TreeKey key;		/* String identifying the data field */
    Tcl_Obj *objPtr;		/* Data representation. */
    Blt_Tree owner;		/* Non-NULL if privately owned. */
    Blt_TreeValue next;		/* Next value in the chain. */
    Blt_TreeValue hnext;	/* Next value in hash table. */
};

typedef struct {
    unsigned int flags;
    Node *rootPtr;
    Blt_HashTable idTable;
    int nLines;
} RestoreInfo;

#include <stdio.h>
#include <string.h>
/* The following header is required for LP64 compilation */
#include <stdlib.h>

#include "bltHash.h"

static void TreeDestroyValues(Blt_TreeNode node);

static Value *TreeFindValue(Blt_TreeNode node, Blt_TreeKey key);
static Value *TreeCreateValue(Blt_TreeNode node, Blt_TreeKey key, int *newPtr);

static int TreeDeleteValue(Blt_TreeNode node, Blt_TreeValue value);

static Value *TreeFirstValue(Blt_TreeNode, Blt_TreeKeySearch *searchPtr);

static Value *TreeNextValue(Blt_TreeKeySearch *srchPtr);

/*
 * When there are this many entries per bucket, on average, rebuild
 * the hash table to make it larger.
 */
#define REBUILD_MULTIPLIER  3
#define START_LOGSIZE       5	/* Initial hash table size is 32. */
#define HASH_HIGH_WATER	    20	/* Start a hash table when a node has
				 * this many values or child nodes. */
#define HASH_LOW_WATER	    (HASH_HIGH_WATER << 1)

#if (SIZEOF_VOID_P == 8)
#define RANDOM_INDEX(i)		HashOneWord(mask, downshift, i)
static Blt_Hash HashOneWord _ANSI_ARGS_((uint64_t mask, unsigned int downshift,
	CONST void *key));
#define BITSPERWORD		64
#else 

/*
 * The following macro takes a preliminary integer hash value and
 * produces an index into a hash tables bucket list.  The idea is
 * to make it so that preliminary values that are arbitrarily similar
 * will end up in different buckets.  The hash function was taken
 * from a random-number generator.
 */
#define RANDOM_INDEX(i) \
    (((((long) (i))*1103515245) >> downshift) & mask)
#define BITSPERWORD		32
#endif /* SIZEOF_VOID_P == 8 */

#define DOWNSHIFT_START		(BITSPERWORD - 2) 

/*
 * The hash table below is used to keep track of all the Blt_TreeKeys
 * created so far.
 */
static Blt_HashTable keyTable;
static int keyTableInitialized = 0;

typedef struct {
    Blt_HashTable treeTable;	/* Table of trees. */
    Tcl_Interp *interp;
    unsigned int nextId;
} TreeInterpData;

typedef struct {
    Tcl_Interp *interp;
    ClientData clientData;
    Blt_TreeKey key;
    Blt_TreeNotifyEventProc *proc;
    Blt_TreeNotifyEvent event;
    unsigned int mask;
    int notifyPending;
} EventHandler;

typedef struct {
    ClientData clientData;
    char *keyPattern;
    char *withTag;
    Node *nodePtr;
    Blt_TreeTraceProc *proc;
    TreeClient *clientPtr;
    Blt_ChainLink *linkPtr;
    unsigned int mask;
} TraceHandler;

/*
 * --------------------------------------------------------------
 *
 * GetTreeInterpData --
 *
 *	Creates or retrieves data associated with tree data objects
 *	for a particular thread.  We're using Tcl_GetAssocData rather
 *	than the Tcl thread routines so BLT can work with pre-8.0 
 *	Tcl versions that don't have thread support.
 *
 * Results:
 *	Returns a pointer to the tree interpreter data.
 *
 * -------------------------------------------------------------- 
 */
static TreeInterpData *
GetTreeInterpData(Tcl_Interp *interp)
{
    Tcl_InterpDeleteProc *proc;
    TreeInterpData *dataPtr;

    dataPtr = (TreeInterpData *)
	Tcl_GetAssocData(interp, TREE_THREAD_KEY, &proc);
    if (dataPtr == NULL) {
	dataPtr = Blt_Malloc(sizeof(TreeInterpData));
	assert(dataPtr);
	dataPtr->interp = interp;
	Tcl_SetAssocData(interp, TREE_THREAD_KEY, TreeInterpDeleteProc,
		 dataPtr);
	Blt_InitHashTable(&dataPtr->treeTable, BLT_STRING_KEYS);
    }
    return dataPtr;
}

char *
Blt_TreeNodeIdAscii(Node *nodePtr)
{
    static char stringRep[200];

    sprintf(stringRep, "%ld", nodePtr->inode);
    return stringRep;
}


/*
 * --------------------------------------------------------------
 *
 * NewNode --
 *
 *	Creates a new node in the tree without installing it.  The
 *	number of nodes in the tree is incremented and a unique serial
 *	number is generated for the node. 
 *
 *	Also, all nodes have a label.  If no label was provided (name
 *	is NULL) then automatically generate one in the form "nodeN"
 *	where N is the serial number of the node.
 *
 * Results:
 *	Returns a pointer to the new node.
 *
 * -------------------------------------------------------------- 
 */
static Node *
NewNode(TreeObject *treeObjPtr, CONST char *name, long inode)
{
    Node *np;

    /* Create the node structure */
    np = Blt_PoolAllocItem(treeObjPtr->nodePool, sizeof(Node));
    np->inode = inode;
    np->treeObject = treeObjPtr;
    np->parent = NULL;
    np->depth = 0;
    np->flags = 0;
    np->next = np->prev = NULL;
    np->first = np->last = NULL;
    np->nChildren = 0;
    np->values = NULL;     
    np->valueTable = NULL;     
    np->valueTableSize2 = 0;
    np->nValues = 0;
    np->nodeTable = NULL;
    np->nodeTableSize2 = 0;
    np->hnext = NULL;

    np->label = NULL;
    if (name != NULL) {
	np->label = Blt_TreeGetKey(name);
    }
    treeObjPtr->nNodes++;
    return np;
}

/*
 *----------------------------------------------------------------------
 *
 * ReleaseTagTable --
 *
 *---------------------------------------------------------------------- 
 */
static void
ReleaseTagTable(Blt_TreeTagTable *tablePtr)
{
    tablePtr->refCount--;
    if (tablePtr->refCount <= 0) {
	Blt_HashEntry *hPtr;
	Blt_HashSearch cursor;

	for(hPtr = Blt_FirstHashEntry(&tablePtr->tagTable, &cursor); 
	    hPtr != NULL; hPtr = Blt_NextHashEntry(&cursor)) {
	    Blt_TreeTagEntry *tePtr;

	    tePtr = Blt_GetHashValue(hPtr);
	    Blt_DeleteHashTable(&tePtr->nodeTable);
	    Blt_Free(tePtr);
	}
	Blt_DeleteHashTable(&tablePtr->tagTable);
	Blt_Free(tablePtr);
    }
}

/*
 * ----------------------------------------------------------------------
 *
 * ResetDepths --
 *
 *	Called after moving a node, resets the depths of each node
 *	for the entire branch (node and it's decendants).  
 *
 * Results: 
 *	None.
 *
 * ---------------------------------------------------------------------- 
 */
static void
ResetDepths(
    Node *branchPtr,		/* Root node of subtree. */
    long depth)			/* Depth of the node. */
{
    Node *np;

    branchPtr->depth = depth;

    /* Also reset the depth for each descendant node. */
    for (np = branchPtr->first; np != NULL; np = np->next) {
	ResetDepths(np, depth + 1);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * RebuildNodeTable --
 *
 *	This procedure is invoked when the ratio of entries to hash
 *	buckets becomes too large.  It creates a new table with a
 *	larger bucket array and moves all of the entries into the
 *	new table.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Memory gets reallocated and entries get re-hashed to new
 *	buckets.
 *
 *----------------------------------------------------------------------
 */
static void
RebuildNodeTable(Node *parentPtr) /* Table to enlarge. */
{
    Node **bp, **bend;
    unsigned int downshift;
    unsigned long mask;
    Node **buckets;
    size_t nBuckets;

    nBuckets = (1 << parentPtr->nodeTableSize2);
    bend = parentPtr->nodeTable + nBuckets;

    /*
     * Allocate and initialize the new bucket array, and set up
     * hashing constants for new array size.
     */
    parentPtr->nodeTableSize2 += 2;
    nBuckets = (1 << parentPtr->nodeTableSize2);
    buckets = Blt_Calloc(nBuckets, sizeof(Node *));
    assert(buckets);
    /*
     * Move all of the existing entries into the new bucket array,
     * based on their new hash values.
     */
    mask = nBuckets - 1;
    downshift = DOWNSHIFT_START - parentPtr->nodeTableSize2;
    for (bp = parentPtr->nodeTable; bp < bend; bp++) {
	Node *np, *nextPtr;

	for (np = *bp; np != NULL; np = nextPtr) {
	    Node **bucketPtr;
    
	    nextPtr = np->hnext;
	    bucketPtr = buckets + RANDOM_INDEX(np->label);
	    np->hnext = *bucketPtr;
	    *bucketPtr = np;
	}
    }
    Blt_Free(parentPtr->nodeTable);
    parentPtr->nodeTable = buckets;
}

/*
 *----------------------------------------------------------------------
 *
 * MakeNodeTable --
 *
 *	Generates a hash table from the nodes list of children.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Hash table for child nodes is created.
 *
 *----------------------------------------------------------------------
 */
static void
MakeNodeTable(Node *parentPtr)
{
    Node **buckets;
    Node *np, *nextPtr;
    int downshift;
    unsigned int mask;
    unsigned int nBuckets;

    assert(parentPtr->nodeTable == NULL);
    parentPtr->nodeTableSize2 = START_LOGSIZE;
    nBuckets = 1 << parentPtr->nodeTableSize2;
    buckets = Blt_Calloc(nBuckets, sizeof(Node *));
    mask = nBuckets - 1;
    downshift = DOWNSHIFT_START - parentPtr->nodeTableSize2;
    for (np = parentPtr->first; np != NULL; np = nextPtr) {
	Node **bucketPtr;

	nextPtr = np->next;
	bucketPtr = buckets + RANDOM_INDEX(np->label);
	np->hnext = *bucketPtr;
	*bucketPtr = np;
    }
    parentPtr->nodeTable = buckets;
}

/*
 *----------------------------------------------------------------------
 *
 * LinkBefore --
 *
 *	Inserts a link preceding a given link.
 *
 * Results:
 *	None.
 *
 *----------------------------------------------------------------------
 */
static void
LinkBefore(
    Node *parentPtr,	/* Parent to hold the new entry. */
    Node *nodePtr,	/* New node to be inserted. */
    Node *beforePtr)	/* Node to link before. */
{
    if (parentPtr->first == NULL) {
	parentPtr->last = parentPtr->first = nodePtr;
    } else if (beforePtr == NULL) { /* Append onto the end of the chain */
	nodePtr->next = NULL;
	nodePtr->prev = parentPtr->last;
	parentPtr->last->next = nodePtr;
	parentPtr->last = nodePtr;
    } else {
	nodePtr->prev = beforePtr->prev;
	nodePtr->next = beforePtr;
	if (beforePtr == parentPtr->first) {
	    parentPtr->first = nodePtr;
	} else {
	    beforePtr->prev->next = nodePtr;
	}
	beforePtr->prev = nodePtr;
    }
    parentPtr->nChildren++;
    nodePtr->parent = parentPtr;

    /* 
     * Check if there as so many children that an addition hash table
     * should be created.
     */
    if (parentPtr->nodeTable == NULL) {
	if (parentPtr->nChildren > HASH_HIGH_WATER) {
	    MakeNodeTable(parentPtr);
	}
    } else {
	Node **bucketPtr;
	size_t nBuckets;
	unsigned int downshift;
	unsigned long mask;

	nBuckets = (1 << parentPtr->nodeTableSize2);
	mask = nBuckets - 1;
	downshift = DOWNSHIFT_START - parentPtr->nodeTableSize2;
	bucketPtr = parentPtr->nodeTable + RANDOM_INDEX(nodePtr->label);
	nodePtr->hnext = *bucketPtr;
	*bucketPtr = nodePtr;
	/*
	 * If the table has exceeded a decent size, rebuild it with many
	 * more buckets.
	 */
	if (parentPtr->nChildren >= (nBuckets * TREE_NODE_REBUILD_SIZE)) {
	    RebuildNodeTable(parentPtr);
	}
    } 
}


/*
 *----------------------------------------------------------------------
 *
 * UnlinkNode --
 *
 *	Unlinks a link from the chain. The link is not deallocated, 
 *	but only removed from the chain.
 *
 * Results:
 *	None.
 *
 *----------------------------------------------------------------------
 */
static void
UnlinkNode(Node *nodePtr)
{
    Node *parentPtr;
    int unlinked;		/* Indicates if the link is actually
				 * removed from the chain. */
    parentPtr = nodePtr->parent;
    unlinked = FALSE;
    if (parentPtr->first == nodePtr) {
	parentPtr->first = nodePtr->next;
	unlinked = TRUE;
    }
    if (parentPtr->last == nodePtr) {
	parentPtr->last = nodePtr->prev;
	unlinked = TRUE;
    }
    if (nodePtr->next != NULL) {
	nodePtr->next->prev = nodePtr->prev;
	unlinked = TRUE;
    }
    if (nodePtr->prev != NULL) {
	nodePtr->prev->next = nodePtr->next;
	unlinked = TRUE;
    }
    if (unlinked) {
	parentPtr->nChildren--;
    }
    nodePtr->prev = nodePtr->next = NULL;
    if (parentPtr->nodeTable != NULL) {
	Node **bucketPtr;
	unsigned int downshift;
	unsigned long mask;

	mask = (1 << parentPtr->nodeTableSize2) - 1;
	downshift = DOWNSHIFT_START - parentPtr->nodeTableSize2;
	bucketPtr = parentPtr->nodeTable + RANDOM_INDEX(nodePtr->label);
	if (*bucketPtr == nodePtr) {
	    *bucketPtr = nodePtr->hnext;
	} else {
	    Node *np;

	    for (np = *bucketPtr; /*empty*/; np = np->hnext) {
		if (np == NULL) {
		    return;	/* Can't find node in hash bucket. */
		}
		if (np->hnext == nodePtr) {
		    np->hnext = nodePtr->hnext;
		    break;
		}
	    }
	}
    } 
    nodePtr->hnext = NULL;
    if (parentPtr->nChildren < HASH_LOW_WATER) {
	Blt_Free(parentPtr->nodeTable);
	parentPtr->nodeTable = NULL;
    }
}

/*
 * --------------------------------------------------------------
 *
 * FreeNode --
 *
 *	Unlinks a given node from the tree, removes its data, and
 *	frees memory allocated to the node.
 *
 * Results:
 *	None.
 *
 * -------------------------------------------------------------- 
 */
static void
FreeNode(TreeObject *treeObjPtr, Node *nodePtr)
{
    Blt_HashEntry *hPtr;

    /*
     * Destroy any data fields associated with this node.
     */
    if (nodePtr->values != NULL) { 
	TreeDestroyValues(nodePtr);
    }
    if (nodePtr->nodeTable != NULL) {
	Blt_Free(nodePtr->nodeTable);
    } 
    UnlinkNode(nodePtr);
    treeObjPtr->nNodes--;
    hPtr = Blt_FindHashEntry(&treeObjPtr->nodeTable, (char *)nodePtr->inode);
    assert(hPtr);
    Blt_DeleteHashEntry(&treeObjPtr->nodeTable, hPtr);
    Blt_PoolFreeItem(treeObjPtr->nodePool, (char *)nodePtr);
}

/*
 * --------------------------------------------------------------
 *
 * NewTreeObject --
 *
 *	Creates and initializes a new tree object. Trees always
 *	contain a root node, so one is allocated here.
 *
 * Results:
 *	Returns a pointer to the new tree object is successful, NULL
 *	otherwise.  If a tree can't be generated, interp->result will
 *	contain an error message.
 *
 * -------------------------------------------------------------- */
static TreeObject *
NewTreeObject(
    TreeInterpData *dataPtr, 
    Tcl_Interp *interp, 
    CONST char *treeName)
{
    TreeObject *treeObjPtr;
    int isNew;
    Blt_HashEntry *hPtr;

    treeObjPtr = Blt_Calloc(1, sizeof(TreeObject));
    if (treeObjPtr == NULL) {
	Tcl_AppendResult(interp, "can't allocate tree", (char *)NULL);
	return NULL;
    }
    treeObjPtr->name = Blt_Strdup(treeName);
    treeObjPtr->interp = interp;
    treeObjPtr->valuePool = Blt_PoolCreate(BLT_FIXED_SIZE_ITEMS);
    treeObjPtr->nodePool = Blt_PoolCreate(BLT_FIXED_SIZE_ITEMS);
    treeObjPtr->clients = Blt_ChainCreate();
    treeObjPtr->depth = 1;
    treeObjPtr->notifyFlags = 0;
    Blt_InitHashTableWithPool(&treeObjPtr->nodeTable, BLT_ONE_WORD_KEYS);

    hPtr = Blt_CreateHashEntry(&treeObjPtr->nodeTable, (char *)0, &isNew);
    treeObjPtr->root = NewNode(treeObjPtr, treeName, 0);
    Blt_SetHashValue(hPtr, treeObjPtr->root);

    treeObjPtr->tablePtr = &dataPtr->treeTable;
    treeObjPtr->hashPtr = Blt_CreateHashEntry(treeObjPtr->tablePtr, treeName, 
	&isNew);
    Blt_SetHashValue(treeObjPtr->hashPtr, treeObjPtr);

    return treeObjPtr;
}

static TreeObject *
FindTreeInNamespace(
    TreeInterpData *dataPtr,	/* Interpreter-specific data. */
    Blt_ObjectName *objNamePtr)
{
    Tcl_DString ds;
    char *name;
    Blt_HashEntry *hPtr;

    name = Blt_MakeQualifiedName(objNamePtr, &ds);
    hPtr = Blt_FindHashEntry(&dataPtr->treeTable, name);
    Tcl_DStringFree(&ds);
    if (hPtr != NULL) {
	return Blt_GetHashValue(hPtr);
    }
    return NULL;
}

/*
 * ----------------------------------------------------------------------
 *
 * GetTreeObject --
 *
 *	Searches for the tree object associated by the name given.
 *
 * Results:
 *	Returns a pointer to the tree if found, otherwise NULL.
 *
 * ----------------------------------------------------------------------
 */
static TreeObject *
GetTreeObject(Tcl_Interp *interp, CONST char *name, int flags)
{
    Blt_ObjectName objName;
    TreeInterpData *dataPtr;	/* Interpreter-specific data. */
    TreeObject *treeObjPtr;

    treeObjPtr = NULL;
    if (!Blt_ParseObjectName(interp, name, &objName, BLT_NO_DEFAULT_NS)) {
	return NULL;
    }
    dataPtr = GetTreeInterpData(interp);
    if (objName.nsPtr != NULL) { 
	treeObjPtr = FindTreeInNamespace(dataPtr, &objName);
    } else { 
	if (flags & NS_SEARCH_CURRENT) {
	    /* Look first in the current namespace. */
	    objName.nsPtr = Tcl_GetCurrentNamespace(interp);
	    treeObjPtr = FindTreeInNamespace(dataPtr, &objName);
	}
	if ((treeObjPtr == NULL) && (flags & NS_SEARCH_GLOBAL)) {
	    objName.nsPtr = Tcl_GetGlobalNamespace(interp);
	    treeObjPtr = FindTreeInNamespace(dataPtr, &objName);
	}
    }
    return treeObjPtr;
}

/*
 * ----------------------------------------------------------------------
 *
 * TeardownTree --
 *
 *	Destroys an entire branch.  This is a special purpose routine
 *	used to speed up the final clean up of the tree.
 *
 * Results: 
 *	None.
 *
 * ---------------------------------------------------------------------- 
 */
static void
TeardownTree(TreeObject *treeObjPtr, Node *branchPtr)
{
    Node *np, *nextPtr;
    
    if (branchPtr->nodeTable != NULL) {
	Blt_Free(branchPtr->nodeTable);
	branchPtr->nodeTable = NULL;
    } 
    if (branchPtr->values != NULL) {
	TreeDestroyValues(branchPtr);
    }
    for (np = branchPtr->first; np != NULL; np = nextPtr) {
	nextPtr = np->next;
	TeardownTree(treeObjPtr, np);
    }
    Blt_PoolFreeItem(treeObjPtr->nodePool, (char *)branchPtr);
}

static void
DestroyTreeObject(TreeObject *treeObjPtr)
{
    Blt_ChainLink *linkPtr;

    treeObjPtr->flags |= TREE_DESTROYED;
    treeObjPtr->nNodes = 0;

    /* Remove the remaining clients. */
    for (linkPtr = Blt_ChainFirstLink(treeObjPtr->clients); linkPtr != NULL;
	linkPtr = Blt_ChainNextLink(linkPtr)) {
	TreeClient *clientPtr;

	clientPtr = Blt_ChainGetValue(linkPtr);
	Blt_ChainDestroy(clientPtr->events);
	Blt_ChainDestroy(clientPtr->traces);
	Blt_Free(clientPtr);
    }
    Blt_ChainDestroy(treeObjPtr->clients);

    TeardownTree(treeObjPtr, treeObjPtr->root);
    Blt_PoolDestroy(treeObjPtr->nodePool);
    Blt_PoolDestroy(treeObjPtr->valuePool);
    Blt_DeleteHashTable(&treeObjPtr->nodeTable);

    if (treeObjPtr->hashPtr != NULL) {
	/* Remove the entry from the global tree table. */
	Blt_DeleteHashEntry(treeObjPtr->tablePtr, treeObjPtr->hashPtr); 
	if ((treeObjPtr->tablePtr->numEntries == 0) && (keyTableInitialized)) {
	    keyTableInitialized = FALSE;
	    Blt_DeleteHashTable(&keyTable);
	}
    }
    if (treeObjPtr->name != NULL) {
	Blt_Free(treeObjPtr->name);
    }
    Blt_Free(treeObjPtr);
}

/*
 * -----------------------------------------------------------------------
 *
 * TreeInterpDeleteProc --
 *
 *	This is called when the interpreter hosting the tree object
 *	is deleted from the interpreter.  
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Destroys all remaining trees and removes the hash table
 *	used to register tree names.
 *
 * ------------------------------------------------------------------------
 */
/* ARGSUSED */
static void
TreeInterpDeleteProc(
    ClientData clientData,	/* Interpreter-specific data. */
    Tcl_Interp *interp)
{
    TreeInterpData *dataPtr = clientData;
    Blt_HashEntry *hPtr;
    Blt_HashSearch cursor;
    
    for (hPtr = Blt_FirstHashEntry(&dataPtr->treeTable, &cursor);
	 hPtr != NULL; hPtr = Blt_NextHashEntry(&cursor)) {
	TreeObject *treeObjPtr;

	treeObjPtr = (TreeObject *)Blt_GetHashValue(hPtr);
	treeObjPtr->hashPtr = NULL;
	DestroyTreeObject(treeObjPtr);
    }
    if (keyTableInitialized) {
	keyTableInitialized = FALSE;
	Blt_DeleteHashTable(&keyTable);
    }
    Blt_DeleteHashTable(&dataPtr->treeTable);
    Tcl_DeleteAssocData(interp, TREE_THREAD_KEY);
    Blt_Free(dataPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * NotifyIdleProc --
 *
 *	Used to invoke event handler routines at some idle point.
 *	This routine is called from the Tcl event loop.  Errors
 *	generated by the event handler routines are backgrounded.
 *	
 * Results:
 *	None.
 *
 *---------------------------------------------------------------------- 
 */
static void
NotifyIdleProc(ClientData clientData)
{
    EventHandler *notifyPtr = clientData;
    int result;

    notifyPtr->notifyPending = FALSE;
    notifyPtr->mask |= TREE_NOTIFY_ACTIVE;
    result = (*notifyPtr->proc)(notifyPtr->clientData, &notifyPtr->event);
    notifyPtr->mask &= ~TREE_NOTIFY_ACTIVE;
    if (result != TCL_OK) {
	Tcl_BackgroundError(notifyPtr->interp);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * CheckEventHandlers --
 *
 *	Traverses the list of client event callbacks and checks
 *	if one matches the given event.  A client may trigger an
 *	action that causes the tree to notify it.  The can be
 *	prevented by setting the TREE_NOTIFY_FOREIGN_ONLY bit in
 *	the event handler.
 *
 *	If a matching handler is found, a callback may be called either
 *	immediately or at the next idle time depending upon the
 *	TREE_NOTIFY_WHENIDLE bit.  
 *
 *	Since a handler routine may trigger yet another call to
 *	itself, callbacks are ignored while the event handler is
 *	executing.
 *	
 * Results:
 *	None.
 *
 *---------------------------------------------------------------------- 
 */
static void
CheckEventHandlers(
    TreeClient *clientPtr,
    int isSource,		/* Indicates if the client is the source
				 * of the event. */
    Blt_TreeNotifyEvent *eventPtr)
{
    Blt_ChainLink *linkPtr, *nextPtr;

    eventPtr->tree = clientPtr;
    for (linkPtr = Blt_ChainFirstLink(clientPtr->events); linkPtr != NULL; 
	linkPtr = nextPtr) {
	EventHandler *notifyPtr;

	nextPtr = Blt_ChainNextLink(linkPtr);
	notifyPtr = Blt_ChainGetValue(linkPtr);
	if ((notifyPtr->mask & TREE_NOTIFY_ACTIVE) ||
	    (notifyPtr->mask & eventPtr->type) == 0) {
	    continue;		/* Ignore callbacks that are generated
				 * inside of a notify handler routine. */
	}
	if ((isSource) && (notifyPtr->mask & TREE_NOTIFY_FOREIGN_ONLY)) {
	    continue;		/* Don't notify yourself. */
	}
	if (notifyPtr->mask & TREE_NOTIFY_WHENIDLE) {
	    if (!notifyPtr->notifyPending) {
		notifyPtr->notifyPending = TRUE;
		notifyPtr->event = *eventPtr;
		Tcl_DoWhenIdle(NotifyIdleProc, notifyPtr);
	    }
	} else {
	    int result;

	    notifyPtr->mask |= TREE_NOTIFY_ACTIVE;
	    result = (*notifyPtr->proc) (notifyPtr->clientData, eventPtr);
	    notifyPtr->mask &= ~TREE_NOTIFY_ACTIVE;
	    if (result != TCL_OK) {
		Tcl_BackgroundError(notifyPtr->interp);
	    }
	}
    }
}

/*
 *----------------------------------------------------------------------
 *
 * NotifyClients --
 *
 *	Traverses the list of clients for a particular tree and
 *	notifies each client that an event occurred.  Clients 
 *	indicate interest in a particular event through a bit
 *	flag.  
 *
 *---------------------------------------------------------------------- 
 */
static void
NotifyClients(
    TreeClient *sourcePtr,
    TreeObject *treeObjPtr,
    Node *nodePtr,
    int eventFlag)
{
    Blt_ChainLink *linkPtr;
    Blt_TreeNotifyEvent event;

    event.type = eventFlag;
    event.inode = nodePtr->inode;

    /* 
     * Issue callbacks to each client indicating that a new node has
     * been created.
     */
    for (linkPtr = Blt_ChainFirstLink(treeObjPtr->clients); linkPtr != NULL; 
	linkPtr = Blt_ChainNextLink(linkPtr)) {
	TreeClient *clientPtr;
	int isSource;

	clientPtr = Blt_ChainGetValue(linkPtr);
	isSource = (clientPtr == sourcePtr);
	CheckEventHandlers(clientPtr, isSource, &event);
    }
}

static void
FreeValue(Node *nodePtr, Value *valuePtr)
{
    if (valuePtr->objPtr != NULL) {
	Tcl_DecrRefCount(valuePtr->objPtr);
    }
    Blt_PoolFreeItem(nodePtr->treeObject->valuePool, valuePtr);
}



#if (SIZEOF_VOID_P == 8)
/*
 *----------------------------------------------------------------------
 *
 * HashOneWord --
 *
 *	Compute a one-word hash value of a 64-bit word, which then can
 *	be used to generate a hash index.
 *
 *	From Knuth, it's a multiplicative hash.  Multiplies an unsigned
 *	64-bit value with the golden ratio (sqrt(5) - 1) / 2.  The
 *	downshift value is 64 - n, when n is the log2 of the size of
 *	the hash table.
 *		
 * Results:
 *	The return value is a one-word summary of the information in
 *	64 bit word.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */
static Blt_Hash
HashOneWord(
    uint64_t mask,
    unsigned int downshift,	        
    CONST void *key)
{
    uint64_t a0, a1;
    uint64_t y0, y1;
    uint64_t y2, y3; 
    uint64_t p1, p2;
    uint64_t result;

    /* Compute key * GOLDEN_RATIO in 128-bit arithmetic */
    a0 = (uint64_t)key & 0x00000000FFFFFFFF; 
    a1 = (uint64_t)key >> 32;
    
    y0 = a0 * 0x000000007f4a7c13;
    y1 = a0 * 0x000000009e3779b9; 
    y2 = a1 * 0x000000007f4a7c13;
    y3 = a1 * 0x000000009e3779b9; 
    y1 += y0 >> 32;		/* Can't carry */ 
    y1 += y2;			/* Might carry */
    if (y1 < y2) {
	y3 += (1LL << 32);	/* Propagate */ 
    }

    /* 128-bit product: p1 = loword, p2 = hiword */
    p1 = ((y1 & 0x00000000FFFFFFFF) << 32) + (y0 & 0x00000000FFFFFFFF);
    p2 = y3 + (y1 >> 32);
    
    /* Left shift the value downward by the size of the table */
    if (downshift > 0) { 
	if (downshift < 64) { 
	    result = ((p2 << (64 - downshift)) | (p1 >> (downshift & 63))); 
	} else { 
	    result = p2 >> (downshift & 63); 
	} 
    } else { 
	result = p1;
    } 
    /* Finally mask off the high bits */
    return (Blt_Hash)(result & mask);
}

#endif /* SIZEOF_VOID_P == 8 */

/*
 *----------------------------------------------------------------------
 *
 * RebuildTable --
 *
 *	This procedure is invoked when the ratio of entries to hash
 *	buckets becomes too large.  It creates a new table with a
 *	larger bucket array and moves all of the entries into the
 *	new table.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Memory gets reallocated and entries get re-hashed to new
 *	buckets.
 *
 *----------------------------------------------------------------------
 */
static void
RebuildValueTable(Node *nodePtr)		/* Table to enlarge. */
{
    Value **bp, **bend, **buckets, **oldBuckets;
    size_t nBuckets;
    unsigned int downshift;
    unsigned long mask;

    oldBuckets = nodePtr->valueTable;
    nBuckets = (1 << nodePtr->valueTableSize2);
    bend = oldBuckets + nBuckets;

    /*
     * Allocate and initialize the new bucket array, and set up
     * hashing constants for new array size.
     */
    nodePtr->valueTableSize2 += 2;
    nBuckets = (1 << nodePtr->valueTableSize2);
    buckets = Blt_Calloc(nBuckets, sizeof(Value *));

    /*
     * Move all of the existing entries into the new bucket array,
     * based on their hash values.  
     */
    mask = nBuckets - 1;
    downshift = DOWNSHIFT_START - nodePtr->valueTableSize2;
    for (bp = oldBuckets; bp < bend; bp++) {
	Value *vp, *nextPtr;

	for (vp = *bp; vp != NULL; vp = nextPtr) {
	    Value **bucketPtr;

	    nextPtr = vp->hnext;
	    bucketPtr = buckets + RANDOM_INDEX(vp->key);
	    vp->hnext = *bucketPtr;
	    *bucketPtr = vp;
	}
    }
    nodePtr->valueTable = buckets;
    Blt_Free(oldBuckets);
}

static void
MakeValueTable(Node *nodePtr)
{
    unsigned int nBuckets;
    Value **buckets;
    unsigned int mask;
    int downshift;
    Value *vp, *nextPtr;

    assert(nodePtr->valueTable == NULL);
    /*
     * Generate hash table from list of values.
     */
    nodePtr->valueTableSize2 = START_LOGSIZE;
    nBuckets = 1 << nodePtr->valueTableSize2;
    buckets = Blt_Calloc(nBuckets, sizeof(Value *));
    mask = nBuckets - 1;
    downshift = DOWNSHIFT_START - nodePtr->valueTableSize2;
    for (vp = nodePtr->values; vp != NULL; vp = nextPtr) {
	Value **bucketPtr;

	nextPtr = vp->next;
	bucketPtr = buckets + RANDOM_INDEX(vp->key);
	vp->hnext = *bucketPtr;
	*bucketPtr = vp;
    }
    nodePtr->valueTable = buckets;
}

/*
 *----------------------------------------------------------------------
 *
 * TreeDeleteValue --
 *
 *	Remove a single entry from a hash table.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The entry given by entryPtr is deleted from its table and
 *	should never again be used by the caller.  It is up to the
 *	caller to free the clientData field of the entry, if that
 *	is relevant.
 *
 *----------------------------------------------------------------------
 */
static int
TreeDeleteValue(Node *nodePtr, Blt_TreeValue value)
{
    Value *vp, *prevPtr;
    
    if (nodePtr->valueTable != NULL) {
	Value **bucketPtr;
	unsigned int downshift;
	unsigned long mask;

	mask = (1 << nodePtr->valueTableSize2) - 1;
	downshift = DOWNSHIFT_START - nodePtr->valueTableSize2;
	bucketPtr = nodePtr->valueTable + RANDOM_INDEX(((Value *)value)->key);
	if (*bucketPtr == value) {
	    *bucketPtr = ((Value *)value)->hnext;
	} else {
	    Value *pp;

	    for (pp = *bucketPtr; /*empty*/; pp = pp->hnext) {
		if (pp == NULL) {
		    return TCL_ERROR; /* Can't find value in hash bucket. */
		}
		if (pp->hnext == value) {
		    pp->hnext = ((Value *)value)->hnext;
		    break;
		}
	    }
	}
    } 
    prevPtr = NULL;
    for (vp = nodePtr->values; vp != NULL; vp = vp->next) {
	if (vp == value) {
	    break;
	}
	prevPtr = vp;
    }
    if (vp == NULL) {
	return TCL_ERROR;	/* Can't find value in list. */
    }
    if (prevPtr == NULL) {
	nodePtr->values = vp->next;
    } else {
	prevPtr->next = vp->next;
    }
    nodePtr->nValues--;
    FreeValue(nodePtr, value);
    if (nodePtr->nValues < HASH_LOW_WATER) {
	Blt_Free(nodePtr->valueTable);
	nodePtr->valueTable = NULL;
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * TreeDestroyValues --
 *
 *	Free up everything associated with a hash table except for
 *	the record for the table itself.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The hash table is no longer useable.
 *
 *----------------------------------------------------------------------
 */
static void
TreeDestroyValues(Node *nodePtr)
{
    Value *vp;
    Value *nextPtr;

    /* Free value hash table. */
    if (nodePtr->valueTable != NULL) {
	Blt_Free(nodePtr->valueTable);
    } 

    /* Free all the entries in the value list. */
    for (vp = nodePtr->values; vp != NULL; vp = nextPtr) {
	nextPtr = vp->next;
	FreeValue(nodePtr, vp);
    }
    nodePtr->values = NULL;
    nodePtr->valueTable = NULL;
    nodePtr->nValues = 0;
    nodePtr->valueTableSize2 = 0;
}

/*
 *----------------------------------------------------------------------
 *
 * TreeFirstValue --
 *
 *	Locate the first entry in a hash table and set up a record
 *	that can be used to step through all the remaining entries
 *	of the table.
 *
 * Results:
 *	The return value is a pointer to the first value in tablePtr,
 *	or NULL if tablePtr has no entries in it.  The memory at
 *	*searchPtr is initialized so that subsequent calls to
 *	Blt_TreeNextValue will return all of the values in the table,
 *	one at a time.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */
static Value *
TreeFirstValue(
    Node *nodePtr,
    Blt_TreeKeySearch *searchPtr) /* Place to store information about
				   * progress through the table. */
{
    searchPtr->node = nodePtr;
    searchPtr->nextIndex = 0;
    searchPtr->nextValue = nodePtr->values;
    return TreeNextValue(searchPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * TreeNextValue --
 *
 *	Once a hash table enumeration has been initiated by calling
 *	Blt_TreeFirstValue, this procedure may be called to return
 *	successive elements of the table.
 *
 * Results:
 *	The return value is the next entry in the hash table being
 *	enumerated, or NULL if the end of the table is reached.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */
static Value *
TreeNextValue(
    Blt_TreeKeySearch *searchPtr) /* Place to store information about
				   * progress through the table.  Must
				   * have been initialized by calling
				   * Blt_TreeFirstValue. */
{
    Value *valuePtr;

    valuePtr = searchPtr->nextValue;
    if (valuePtr != NULL) {
	searchPtr->nextValue = valuePtr->next;
    }
    return valuePtr;
}

/*
 *----------------------------------------------------------------------
 *
 * TreeFindValue --
 *
 *	Given a hash table with one-word keys, and a one-word key, find
 *	the entry with a matching key.
 *
 * Results:
 *	The return value is a token for the matching entry in the
 *	hash table, or NULL if there was no matching entry.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */
static Value *
TreeFindValue(
    Node *nodePtr,
    Blt_TreeKey key)		/* Key to use to find matching entry. */
{
    Value *vp;

    if (nodePtr->valueTable != NULL) {
	unsigned int downshift;
	unsigned long mask;
	Value *bucket;

	mask = (1 << nodePtr->valueTableSize2) - 1;
	downshift = DOWNSHIFT_START - nodePtr->valueTableSize2;
	bucket = nodePtr->valueTable[RANDOM_INDEX(key)];
	/*
	 * Search all of the entries in the appropriate bucket.
	 */
	for (vp = bucket; (vp != NULL) && (vp->key != key); vp = vp->hnext) {
	    /* empty */;
	}
    } else {
	for (vp = nodePtr->values; (vp != NULL) && (vp->key != key); 
	     vp = vp->next) {
	    /* empty */;
	}
    }
    return vp;
}

/*
 *----------------------------------------------------------------------
 *
 * TreeCreateValue --
 *
 *	Find the value with a matching key.  If there is no matching 
 *	value, then create a new one.
 *
 * Results:
 *	The return value is a pointer to the matching value.  If this
 *	is a newly-created value, then *newPtr will be set to a non-zero
 *	value;  otherwise *newPtr will be set to 0.  
 *
 * Side effects:
 *	A new value may be added to the hash table.
 *
 *----------------------------------------------------------------------
 */
static Value *
TreeCreateValue(
    Node *nodePtr,
    Blt_TreeKey key,		/* Key to use to find or create matching
				 * entry. */
    int *isNewPtr)		/* (out) If non-zero, indicates a new
				 * hash entry was created. */
{
    Value *vp, *prevPtr;
    
    prevPtr = NULL;
    *isNewPtr = FALSE;
    for (vp = nodePtr->values; vp != NULL; vp = vp->next) {
	if (vp->key == key) {
	    return vp;
	}
	prevPtr = vp;
    }
    /* Value not found. Add a new value to the list. */
    *isNewPtr = TRUE;
    vp = Blt_PoolAllocItem(nodePtr->treeObject->valuePool, sizeof(Value));
    vp->key = key;
    vp->owner = NULL;
    vp->next = NULL;
    vp->hnext = NULL;
    vp->objPtr = NULL;
    if (prevPtr == NULL) {
	nodePtr->values = vp;
    } else {
	prevPtr->next = vp;
    }
    nodePtr->nValues++;

    if (nodePtr->valueTable == NULL) {
	/* 
	 * If we reach a threshold number of values, create a hash
	 * table of values.
	 */
	if (nodePtr->nValues > HASH_HIGH_WATER) {
	    MakeValueTable(nodePtr);
	}
    } else {
	Value **bucketPtr;
	size_t nBuckets;
	unsigned int downshift;
	unsigned long mask;

	nBuckets = (1 << nodePtr->valueTableSize2);
	mask = nBuckets - 1;
	downshift = DOWNSHIFT_START - nodePtr->valueTableSize2;
	bucketPtr = nodePtr->valueTable + RANDOM_INDEX((void *)key);
	vp->hnext = *bucketPtr;
	*bucketPtr = vp;
	/*
	 * If the table has exceeded a decent size, rebuild it with many
	 * more buckets.
	 */
	if ((unsigned int)nodePtr->nValues >= (nBuckets * 3)) {
	    RebuildValueTable(nodePtr);
	}
    } 
    return vp;
}


/*
 *----------------------------------------------------------------------
 *
 * ParseDumpRecord --
 *
 *	Gets the next full record in the dump string, returning the
 *	record as a list. Blank lines and comments are ignored.
 *
 * Results: 
 *	TCL_RETURN	The end of the string is reached.
 *	TCL_ERROR	An error occurred and an error message 
 *			is left in the interpreter result.  
 *	TCL_OK		The next record has been successfully parsed.
 *
 *----------------------------------------------------------------------
 */
static int
ParseDumpRecord(
    Tcl_Interp *interp,
    char **stringPtr,		/* (in/out) points to current location
				 * in in dump string. Updated after
				 * parsing record. */
    int *argcPtr,		/* (out) Will contain the length of
				 * the record's list. */
    char ***argvPtr,		/* (out) Will contain the list representing
				 * the dump record of the node. */
    RestoreInfo *restorePtr)
{
    char *entry, *eol;
    char saved;
    int result;

    entry = *stringPtr;
    /* Get first line, ignoring blank lines and comments. */
    for (;;) {
	char *first;

	first = NULL;
	restorePtr->nLines++;
	/* Find the end of the first line. */
	for (eol = entry; (*eol != '\n') && (*eol != '\0'); eol++) {
	    if ((first == NULL) && (!isspace(UCHAR(*eol)))) {
		first = eol;	/* Track first non-whitespace
				 * character. */
	    }
	}
	if (first == NULL) {
	    if (*eol == '\0') {
		return TCL_RETURN;
	    }
	} else if (*first != '#') {
	    break;		/* Not a comment or blank line. */
	}
	entry = eol + 1;
    }
    saved = *eol;
    *eol = '\0';
    while (!Tcl_CommandComplete(entry)) {
	*eol = saved;
	if (*eol == '\0') {
	    Tcl_AppendResult(interp, "incomplete dump record: \"", entry, 
		"\"", (char *)NULL);
	    return TCL_ERROR;		/* Found EOF (incomplete
					 * entry) or error. */
	}
	/* Get the next line. */
	for (eol = eol + 1; (*eol != '\n') && (*eol != '\0'); eol++) {
	    /*empty*/
	}
	restorePtr->nLines++;
	saved = *eol;
	*eol = '\0';
    }
    if (entry == eol) {
	return TCL_RETURN;
    }
    result = Tcl_SplitList(interp, entry, argcPtr, argvPtr);
    *eol = saved;
    *stringPtr = eol + 1;
    return result;
}

/*
 *----------------------------------------------------------------------
 *
 * ReadDumpRecord --
 *
 *	Reads the next full record from the given channel, returning the
 *	record as a list. Blank lines and comments are ignored.
 *
 * Results: 
 *	TCL_RETURN	The end of the file has been reached.
 *	TCL_ERROR	A read error has occurred and an error message 
 *			is left in the interpreter result.  
 *	TCL_OK		The next record has been successfully parsed.
 *
 *----------------------------------------------------------------------
 */
static int
ReadDumpRecord(
    Tcl_Interp *interp,
    Tcl_Channel channel,	/* Channel from which to read the next
				 * record. */
    int *argcPtr,		/* (out) Will contain the length of
				 * the record's list. */
    char ***argvPtr,		/* (out) Will contain the list representing
				 * the dump record of the node. */
    RestoreInfo *restorePtr)
{
    int result;
    Tcl_DString ds;

    Tcl_DStringInit(&ds);
    /* Get first line, ignoring blank lines and comments. */
    for (;;) {
	char *cp;
	int nBytes;

	Tcl_DStringSetLength(&ds, 0);
	nBytes = Tcl_Gets(channel, &ds);
	if (nBytes < 0) {
	    if (Tcl_Eof(channel)) {
		return TCL_RETURN;
	    }
	    return TCL_ERROR;
	}
	restorePtr->nLines++;
	for (cp = Tcl_DStringValue(&ds); *cp != '\0'; cp++) {
	    if (!isspace(UCHAR(*cp))) {
		break;
	    }
	}
	if ((*cp != '\0') && (*cp != '#')) {
	    break;		/* Not a comment or blank line. */
	}
    }

    Tcl_DStringAppend(&ds, "\n", 1);
    while (!Tcl_CommandComplete(Tcl_DStringValue(&ds))) {
	int nBytes;

	/* Process additional lines if needed */
	nBytes = Tcl_Gets(channel, &ds);
	if (nBytes < 0) {
	    Tcl_AppendResult(interp, "error reading file: ", 
			     Tcl_PosixError(interp), (char *)NULL);
	    Tcl_DStringFree(&ds);
	    return TCL_ERROR;		/* Found EOF (incomplete
					 * entry) or error. */
	}
	restorePtr->nLines++;
	Tcl_DStringAppend(&ds, "\n", 1);
    }
    result = Tcl_SplitList(interp, Tcl_DStringValue(&ds), argcPtr, argvPtr);
    Tcl_DStringFree(&ds);
    return result;
}


static int
RestoreValues(
    Tcl_Interp *interp, 
    TreeClient *clientPtr, 
    Node *nodePtr, 
    int nValues, 
    char **values)
{
    int i;

    for (i = 0; i < nValues; i += 2) {
	Tcl_Obj *valueObjPtr;
	int result;

	if ((i + 1) < nValues) {
	    valueObjPtr = Tcl_NewStringObj(values[i + 1], -1);
	} else {
	    valueObjPtr = bltEmptyStringObjPtr;
	}
	Tcl_IncrRefCount(valueObjPtr);
	result = Blt_TreeSetValue(interp, clientPtr, nodePtr, values[i], 
		valueObjPtr);
	Tcl_DecrRefCount(valueObjPtr);
	if (result != TCL_OK) {
	    return TCL_ERROR;
	}
    }
    return TCL_OK;
}

static int 
RestoreTags(
    TreeClient *clientPtr, 
    Node *nodePtr, 
    int nTags, 
    char **tags) 
{
    int i;

    for (i = 0; i < nTags; i++) {
	Blt_TreeAddTag(clientPtr, nodePtr, tags[i]);
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * RestoreNode5 --
 *
 *	Parses and creates a node based upon the first 3 fields of
 *	a five field entry.  This is the new restore file format.
 *
 *	   parentId nodeId pathList dataList tagList ?attrList?
 *
 *	The purpose is to attempt to save and restore the node ids
 *	embedded in the restore file information.  The old format
 *	could not distinquish between two sibling nodes with the same
 *	label unless they were both leaves.  I'm trying to avoid
 *	dependencies upon labels.  
 *
 *	If you're starting from an empty tree, this obviously should
 *	work without a hitch.  We only need to map the file's root id
 *	to 0.  It's a little more complicated when adding node to an
 *	already full tree.  
 *
 *	First see if the node id isn't already in use.  Otherwise, map
 *	the node id (via a hashtable) to the real node. We'll need it
 *	later when subsequent entries refer to their parent id.
 *
 *	If a parent id is unknown (the restore file may be out of
 *	order), then follow plan B and use its path.
 *	
 *---------------------------------------------------------------------- 
 */
static int
RestoreNode5(
    Tcl_Interp *interp,
    TreeClient *clientPtr, 
    int argc, 
    char **argv, 
    RestoreInfo *restorePtr)
{
    Blt_HashEntry *hPtr;
    Node *nodePtr, *parentPtr;
    int isNew;
    long pid, id;
    char **attrs, **tags, **values, **names;
    int nTags, nValues, nNames;

    /* 
     * The second and first fields respectively are the ids of the
     * node and its parent.  The parent id of the root node is always
     * -1.  
     */

    if ((Tcl_GetLong(interp, argv[0], &pid) != TCL_OK) ||
	(Tcl_GetLong(interp, argv[1], &id) != TCL_OK)) {
	return TCL_ERROR;
    }
    names = values = tags = attrs = NULL;
    nodePtr = NULL;

    /* 
     * The third, fourth, and fifth fields respectively are the list
     * of component names representing the path to the node including
     * the name of the node, a key-value list of data values, and a
     * list of tag names.
     */     

    if ((Tcl_SplitList(interp, argv[2], &nNames, &names) != TCL_OK) ||
	(Tcl_SplitList(interp, argv[3], &nValues, &values) != TCL_OK)  || 
	(Tcl_SplitList(interp, argv[4], &nTags, &tags) != TCL_OK)) {
	goto error;
    }    

    /* Get the parent of the node. */

    if (pid == -1) {	/* Map -1 id to the root node of the subtree. */
	nodePtr = restorePtr->rootPtr;
	hPtr = Blt_CreateHashEntry(&restorePtr->idTable, (char *)id, &isNew);
	Blt_SetHashValue(hPtr, nodePtr);
	Blt_TreeRelabelNode(clientPtr, nodePtr, names[0]);
    } else {

	/* 
	 * Check if the parent has been mapped to another id in the
	 * tree.  This can happen when there's a id collision with an
	 * existing node.
	 */

	hPtr = Blt_FindHashEntry(&restorePtr->idTable, (char *)pid);
	if (hPtr != NULL) {
	    parentPtr = Blt_GetHashValue(hPtr);
	} else {
	    parentPtr = Blt_TreeGetNode(clientPtr, pid);
	    if (parentPtr == NULL) {
		/* 
		 * Normally the parent node should already exist in
		 * the tree, but in a partial restore it might not.
		 * "Plan B" is to use the list of path components to
		 * create the missing components, including the
		 * parent.
		 */
		if (nNames == 0) {
		    parentPtr = restorePtr->rootPtr;
		} else {
		    int i;

		    for (i = 1; i < (nNames - 2); i++) {
			nodePtr = Blt_TreeFindChild(parentPtr, names[i]);
			if (nodePtr == NULL) {
			    nodePtr = Blt_TreeCreateNode(clientPtr, parentPtr, 
				names[i], -1);
			}
			parentPtr = nodePtr;
		    }
		    /* 
		     * If there's a node with the same label as the
		     * parent, we'll use that node. Otherwise, try to
		     * create a new node with the desired parent id.
		     */
		    nodePtr = Blt_TreeFindChild(parentPtr, names[nNames - 2]);
		    if (nodePtr == NULL) {
			nodePtr = Blt_TreeCreateNodeWithId(clientPtr, parentPtr,
				names[nNames - 2], pid, -1);
			if (nodePtr == NULL) {
			    goto error;
			}
		    }
		    parentPtr = nodePtr;
		}
	    }
	} 

	/* 
	 * It's an error if the desired id has already been remapped.
	 * That means there were two nodes in the dump with the same
	 * id.
	 */
	hPtr = Blt_FindHashEntry(&restorePtr->idTable, (char *)id);
 	if (hPtr != NULL) {
	    Tcl_AppendResult(interp, "node \"", Blt_Ltoa(id), 
		"\" has already been restored", (char *)NULL);
	    goto error;
	}


	if (restorePtr->flags & TREE_RESTORE_OVERWRITE) {
	    /* Can you find the child by name. */
	    nodePtr = Blt_TreeFindChild(parentPtr, names[nNames - 1]);
	    if (nodePtr != NULL) {
		hPtr = Blt_CreateHashEntry(&restorePtr->idTable, (char *)id,
					   &isNew);
		Blt_SetHashValue(hPtr, nodePtr);
	    }
	}

	if (nodePtr == NULL) {
	    nodePtr = Blt_TreeGetNode(clientPtr, id);
	    if (nodePtr == NULL) {
		nodePtr = Blt_TreeCreateNodeWithId(clientPtr, parentPtr, 
			names[nNames - 1], id, -1);
	    } else {
		nodePtr = Blt_TreeCreateNode(clientPtr, parentPtr, 
			names[nNames - 1], -1);
		hPtr = Blt_CreateHashEntry(&restorePtr->idTable, (char *)id,
			&isNew);
		Blt_SetHashValue(hPtr, nodePtr);
	    }
	}
    } 
	
    if (nodePtr == NULL) {
	goto error;		/* Couldn't create a node with the
				 * requested id. */
    }
    Blt_Free(names);
    names = NULL;

    /* Values */
    if (RestoreValues(interp, clientPtr, nodePtr, nValues, values) != TCL_OK) {
	goto error;
    }
    Blt_Free(values);
    values = NULL;

    /* Tags */
    if (!(restorePtr->flags & TREE_RESTORE_NO_TAGS)) {
	RestoreTags(clientPtr, nodePtr, nTags, tags);
    }
    Blt_Free(tags);
    tags = NULL;
    return TCL_OK;

 error:
    if (attrs != NULL) {
	Blt_Free(attrs);
    }
    if (tags != NULL) {
	Blt_Free(tags);
    }
    if (values != NULL) {
	Blt_Free(values);
    }
    if (names != NULL) {
	Blt_Free(names);
    }
    if (nodePtr != NULL) {
	Blt_TreeDeleteNode(clientPtr, nodePtr);
    }
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * RestoreNode3 --
 *
 *	Parses and creates a node based upon the first field of
 *	a three field entry.  This is the old restore file format.
 *
 *		pathList dataList tagList
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
RestoreNode3(
    Tcl_Interp *interp, 
    TreeClient *clientPtr, 
    int argc,			/* Not used. */
    char **argv, 
    RestoreInfo *restorePtr)
{
    Node *nodePtr, *parentPtr;
    int i;
    char **names, **values, **tags;
    int nNames, nValues, nTags;

    /* The first field is a list of component names representing the
     * path to the node, including the name of the node. */

    if (Tcl_SplitList(interp, argv[0], &nNames, &names) != TCL_OK) {
	return TCL_ERROR;
    }
    nodePtr = parentPtr = restorePtr->rootPtr;

    /* Automatically create ancestor nodes as needed. */
    for (i = 0; i < (nNames - 1); i++) {
	nodePtr = Blt_TreeFindChild(parentPtr, names[i]);
	if (nodePtr == NULL) {
	    nodePtr = Blt_TreeCreateNode(clientPtr, parentPtr, names[i], -1);
	}
	parentPtr = nodePtr;
    }
    if (nNames > 0) {

	/* 
	 * By default duplicate nodes (two sibling nodes with the same
	 * label) unless the -overwrite switch was given.
	 */

	nodePtr = NULL;
	if (restorePtr->flags & TREE_RESTORE_OVERWRITE) {
	    nodePtr = Blt_TreeFindChild(parentPtr, names[i]);
	}
	if (nodePtr == NULL) {
	    nodePtr = Blt_TreeCreateNode(clientPtr, parentPtr, names[i], -1);
	}
    }
    Blt_Free(names);

    /* The second field is a key-value list of the node's values. */

    if (Tcl_SplitList(interp, argv[1], &nValues, &values) != TCL_OK) {
	return TCL_ERROR;
    }
    if (RestoreValues(interp, clientPtr, nodePtr, nValues, values) != TCL_OK) {
	goto error;
    }
    Blt_Free(values);

    /* The third field is a list of tags. */

    if (!(restorePtr->flags & TREE_RESTORE_NO_TAGS)) {
	/* Parse the tag list. */
	if (Tcl_SplitList(interp, argv[2], &nTags, &tags) != TCL_OK) {
	    goto error;
	}
	RestoreTags(clientPtr, nodePtr, nTags, tags);
	Blt_Free(tags);
    }
    return TCL_OK;

 error:
    Blt_Free(argv);
    Blt_TreeDeleteNode(clientPtr, nodePtr);
    return TCL_ERROR;
}

/* Public Routines */
/*
 *----------------------------------------------------------------------
 *
 * Blt_TreeGetKey --
 *
 *	Given a string, returns a unique identifier for the string.
 *
 *----------------------------------------------------------------------
 */
Blt_TreeKey
Blt_TreeGetKey(CONST char *string) /* String to convert. */
{
    Blt_HashEntry *hPtr;
    int isNew;

    if (!keyTableInitialized) {
	Blt_InitHashTable(&keyTable, BLT_STRING_KEYS);
	keyTableInitialized = 1;
    }
    hPtr = Blt_CreateHashEntry(&keyTable, string, &isNew);
    return (Blt_TreeKey)Blt_GetHashKey(&keyTable, hPtr);
}


/*
 *----------------------------------------------------------------------
 *
 * Blt_TreeCreateNode --
 *
 *	Creates a new node in the given parent node.  The name and
 *	position in the parent are also provided.
 *
 *----------------------------------------------------------------------
 */
Blt_TreeNode
Blt_TreeCreateNode(
    TreeClient *clientPtr,	/* The tree client that is creating
				 * this node.  If NULL, indicates to
				 * trigger notify events on behalf of
				 * the initiating client also. */
    Node *parentPtr,		/* Parent node where the new node will
				 * be inserted. */
    CONST char *name,		/* Name of node. */
    long position)		/* Position in the parent's list of children
				 * where to insert the new node. */
{
    Blt_HashEntry *hPtr;
    Node *beforePtr;
    Node *nodePtr;	/* Node to be inserted. */
    TreeObject *treeObjPtr;
    long inode;
    int isNew;

    treeObjPtr = parentPtr->treeObject;

    /* Generate an unique serial number for this node.  */
    do {
	inode = treeObjPtr->nextInode++;
	hPtr = Blt_CreateHashEntry(&treeObjPtr->nodeTable,(char *)inode, 
		   &isNew);
    } while (!isNew);
    nodePtr = NewNode(treeObjPtr, name, inode);
    Blt_SetHashValue(hPtr, nodePtr);

    if ((position == -1) || ((size_t)position >= parentPtr->nChildren)) {
	beforePtr = NULL;
    } else {
	beforePtr = parentPtr->first;
	while ((position > 0) && (beforePtr != NULL)) {
	    position--;
	    beforePtr = beforePtr->next;
	}
    }
    LinkBefore(parentPtr, nodePtr, beforePtr);
    nodePtr->depth = parentPtr->depth + 1;
    /* 
     * Issue callbacks to each client indicating that a new node has
     * been created.
     */
    NotifyClients(clientPtr, treeObjPtr, nodePtr, TREE_NOTIFY_CREATE);
    return nodePtr;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_TreeCreateNodeWithId --
 *
 *	Like Blt_TreeCreateNode, but provides a specific id to use
 *	for the node.  If the tree already contains a node by that
 *	id, NULL is returned.
 *
 *----------------------------------------------------------------------
 */
Blt_TreeNode
Blt_TreeCreateNodeWithId(
    TreeClient *clientPtr,
    Node *parentPtr,		/* Parent node where the new node will
				 * be inserted. */
    CONST char *name,		/* Name of node. */
    long inode,			/* Requested id of the new node. If a
				 * node by this id already exists in the
				 * tree, no node is created. */
    long position)		/* Position in the parent's list of children
				 * where to insert the new node. */
{
    Blt_HashEntry *hPtr;
    Node *beforePtr;
    Node *nodePtr;	/* Node to be inserted. */
    TreeObject *treeObjPtr;
    int isNew;

    treeObjPtr = parentPtr->treeObject;
    hPtr = Blt_CreateHashEntry(&treeObjPtr->nodeTable, (char *)inode, &isNew);
    if (!isNew) {
	return NULL;
    }
    nodePtr = NewNode(treeObjPtr, name, inode);
    Blt_SetHashValue(hPtr, nodePtr);

    if ((position == -1) || ((size_t)position >= parentPtr->nChildren)) {
	beforePtr = NULL;
    } else {
	beforePtr = parentPtr->first;
	while ((position > 0) && (beforePtr != NULL)) {
	    position--;
	    beforePtr = beforePtr->next;
	}
    }
    LinkBefore(parentPtr, nodePtr, beforePtr);
    nodePtr->depth = parentPtr->depth + 1;
    /* 
     * Issue callbacks to each client indicating that a new node has
     * been created.
     */
    NotifyClients(clientPtr, treeObjPtr, nodePtr, TREE_NOTIFY_CREATE);
    return nodePtr;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_TreeMoveNode --
 *
 *	Move an entry into a new location in the hierarchy.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
int
Blt_TreeMoveNode(
    TreeClient *clientPtr,
    Node *nodePtr, 
    Node *parentPtr, 
    Node *beforePtr)
{
    TreeObject *treeObjPtr = nodePtr->treeObject;
    size_t newDepth;

    if (nodePtr == beforePtr) {
	return TCL_ERROR;
    }
    if ((beforePtr != NULL) && (beforePtr->parent != parentPtr)) {
	return TCL_ERROR;
    }
    if (nodePtr->parent == NULL) {
	return TCL_ERROR;	/* Can't move root. */
    }
    /* Verify that the node isn't an ancestor of the new parent. */
    if (Blt_TreeIsAncestor(nodePtr, parentPtr)) {
	return TCL_ERROR;
    }
    UnlinkNode(nodePtr);

    /* 
     * Relink the node as a child of the new parent.
     */
    LinkBefore(parentPtr, nodePtr, beforePtr);
    newDepth = parentPtr->depth + 1;
    if (nodePtr->depth != newDepth) { 
	/* Recursively reset the depths of all descendant nodes. */
	ResetDepths(nodePtr, newDepth);
    }

    /* 
     * Issue callbacks to each client indicating that a node has
     * been moved.
     */
    NotifyClients(clientPtr, treeObjPtr, nodePtr, TREE_NOTIFY_MOVE);
    return TCL_OK;
}

int
Blt_TreeDeleteNode(TreeClient *clientPtr, Node *nodePtr)
{
    TreeObject *treeObjPtr = nodePtr->treeObject;
    Node *np, *nextPtr;

    /* In depth-first order, delete each descendant node. */
    for (np = nodePtr->first; np != NULL; np = nextPtr) {
	nextPtr = np->next;
	Blt_TreeDeleteNode(clientPtr, np);
    }
    /* 
     * Issue callbacks to each client indicating that the node can
     * no longer be used.  
     */
    NotifyClients(clientPtr, treeObjPtr, nodePtr, TREE_NOTIFY_DELETE);

    /* Now remove the actual node. */
    FreeNode(treeObjPtr, nodePtr);
    return TCL_OK;
}

Blt_TreeNode
Blt_TreeGetNode(TreeClient *clientPtr, long inode)
{
    TreeObject *treeObjPtr = clientPtr->treeObject;
    Blt_HashEntry *hPtr;

    hPtr = Blt_FindHashEntry(&treeObjPtr->nodeTable, (char *)inode);
    if (hPtr != NULL) {
	return Blt_GetHashValue(hPtr);
    }
    return NULL;
}

Blt_TreeTrace
Blt_TreeCreateTrace(
    TreeClient *clientPtr,
    Node *nodePtr,
    CONST char *keyPattern,
    CONST char *tagName,
    unsigned int mask,
    Blt_TreeTraceProc *proc,
    ClientData clientData)
{
    TraceHandler *tracePtr;

    tracePtr = Blt_Calloc(1, sizeof (TraceHandler));
    assert(tracePtr);
    tracePtr->linkPtr = Blt_ChainAppend(clientPtr->traces, tracePtr);
    if (keyPattern != NULL) {
	tracePtr->keyPattern = Blt_Strdup(keyPattern);
    }
    if (tagName != NULL) {
	tracePtr->withTag = Blt_Strdup(tagName);
    }
    tracePtr->clientPtr = clientPtr;
    tracePtr->proc = proc;
    tracePtr->clientData = clientData;
    tracePtr->mask = mask;
    tracePtr->nodePtr = nodePtr;
    return (Blt_TreeTrace)tracePtr;
}

void
Blt_TreeDeleteTrace(Blt_TreeTrace trace)
{
    TraceHandler *tracePtr = (TraceHandler *)trace;

    Blt_ChainDeleteLink(tracePtr->clientPtr->traces, tracePtr->linkPtr);
    if (tracePtr->keyPattern != NULL) {
	Blt_Free(tracePtr->keyPattern);
    }
    if (tracePtr->withTag != NULL) {
	Blt_Free(tracePtr->withTag);
    }
    Blt_Free(tracePtr);
}

void
Blt_TreeRelabelNodeWithoutNotify(Node *nodePtr, CONST char *string)
{
    Blt_TreeKey oldLabel;
    Node **bucketPtr;
    Node *parentPtr;
    Node *np;
    unsigned int downshift;
    unsigned long mask;

    oldLabel = nodePtr->label;
    nodePtr->label = Blt_TreeGetKey(string);
    parentPtr = nodePtr->parent;
    if ((parentPtr == NULL) || (parentPtr->nodeTable == NULL)) {
	return;			/* Root node. */
    }
    /* Changing the node's name requires that we rehash the node in
     * the parent's table of children. */
    mask = (1 << parentPtr->nodeTableSize2) - 1;
    downshift = DOWNSHIFT_START - parentPtr->nodeTableSize2;
    bucketPtr = parentPtr->nodeTable + RANDOM_INDEX(oldLabel);
    if (*bucketPtr == nodePtr) {
	*bucketPtr = nodePtr->hnext;
    } else {
	for (np = *bucketPtr; /*empty*/; np = np->hnext) {
	    if (np == NULL) {
		return;	/* Can't find node in hash bucket. */
	    }
	    if (np->hnext == nodePtr) {
		np->hnext = nodePtr->hnext;
		break;
	    }
	}
    }
    bucketPtr = parentPtr->nodeTable + RANDOM_INDEX(nodePtr->label);
    nodePtr->hnext = *bucketPtr;
    *bucketPtr = nodePtr;
} 

void
Blt_TreeRelabelNode(TreeClient *clientPtr, Node *nodePtr, CONST char *string)
{
    Blt_TreeRelabelNodeWithoutNotify(nodePtr, string);
    NotifyClients(clientPtr, clientPtr->treeObject, nodePtr, 
	TREE_NOTIFY_RELABEL);
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_TreeFindChild --
 *
 *	Searches for the named node in a parent's chain of siblings.  
 *
 *
 * Results:
 *	If found, the child node is returned, otherwise NULL.
 *
 *----------------------------------------------------------------------
 */
Blt_TreeNode
Blt_TreeFindChild(Node *parentPtr, CONST char *string)
{
    Blt_TreeKey key;
    Node *np;
    
    key = Blt_TreeGetKey(string);
    if (parentPtr->nodeTable != NULL) {
	unsigned int downshift;
	unsigned long mask;
	Node *bucket;

	mask = (1 << parentPtr->nodeTableSize2) - 1;
	downshift = DOWNSHIFT_START - parentPtr->nodeTableSize2;
	bucket = parentPtr->nodeTable[RANDOM_INDEX(key)];
	/*
	 * Search all of the entries in the appropriate bucket.
	 */
	for (np = bucket; np != NULL; np = np->hnext) {
	    if (key == np->label) {
		return np;
	    }
	}
    } else {
	for (np = parentPtr->first; np != NULL; np = np->next) {
	    if (key == np->label) {
		return np;
	    }
	}
    }
    return NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_TreeNodePosition --
 *
 *	Returns the position of the node in its parent's list of
 *	children.  The root's position is 0.
 *
 *----------------------------------------------------------------------
 */
long
Blt_TreeNodePosition(Node *nodePtr)
{
    Node *parentPtr;
    long count;

    count = 0;
    parentPtr = nodePtr->parent;
    if (parentPtr != NULL) {
	Node *np;

	for (np = parentPtr->first; np != NULL; np = np->next) {
	    if (nodePtr == np) {
		break;
	    }
	    count++;
	}
    }
    return count;
}

Blt_TreeNode 
Blt_TreeFirstChild(Node *parentPtr)
{
    return parentPtr->first;
}

Blt_TreeNode 
Blt_TreeLastChild(Node *parentPtr)
{
    return parentPtr->last;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_TreePrevNode --
 *
 *	Returns the "previous" node in the tree.  This node (in 
 *	depth-first order) is its parent, if the node has no siblings
 *	that are previous to it.  Otherwise it is the last descendant 
 *	of the last sibling.  In this case, descend the sibling's
 *	hierarchy, using the last child at any ancestor, with we
 *	we find a leaf.
 *
 *----------------------------------------------------------------------
 */
Blt_TreeNode
Blt_TreePrevNode(
    Node *rootPtr,		/* Root of subtree. If NULL, indicates
				* the tree's root. */
    Node *nodePtr)		/* Current node in subtree. */
{
    Node *prevPtr;

    if (rootPtr == NULL) {
	rootPtr = nodePtr->treeObject->root;
    }
    if (nodePtr == rootPtr) {
	return NULL;		/* The root is the first node. */
    }
    prevPtr = nodePtr->prev;
    if (prevPtr == NULL) {
	/* There are no siblings previous to this one, so pick the parent. */
	return nodePtr->parent;
    }
    /*
     * Traverse down the right-most thread, in order to select the
     * next entry.  Stop when we reach a leaf.
     */
    nodePtr = prevPtr;
    while ((prevPtr = nodePtr->last) != NULL) {
	nodePtr = prevPtr;
    }
    return nodePtr;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_TreeNextNode --
 *
 *	Returns the "next" node in relation to the given node.  
 *	The next node (in depth-first order) is either the first 
 *	child of the given node the next sibling if the node has
 *	no children (the node is a leaf).  If the given node is the 
 *	last sibling, then try it's parent next sibling.  Continue
 *	until we either find a next sibling for some ancestor or 
 *	we reach the root node.  In this case the current node is 
 *	the last node in the tree.
 *
 *----------------------------------------------------------------------
 */
Blt_TreeNode
Blt_TreeNextNode(
    Node *rootPtr,		/* Root of subtree. If NULL, indicates
				 * the tree's root. */
    Node *nodePtr)		/* Current node in subtree. */
{
    Node *nextPtr;

    /* Pick the first sub-node. */
    nextPtr = nodePtr->first;
    if (nextPtr != NULL) {
	return nextPtr;
    }
    /* 
     * Back up until we can find a level where we can pick a 
     * "next sibling".  For the last entry we'll thread our 
     * way back to the root.  
     */
    if (rootPtr == NULL) {
	rootPtr = nodePtr->treeObject->root;
    }
    while (nodePtr != rootPtr) {
	nextPtr = nodePtr->next;
	if (nextPtr != NULL) {
	    return nextPtr;
	}
	nodePtr = nodePtr->parent;
    }
    return NULL;		/* At root, no next node. */
}


int
Blt_TreeIsBefore(Node *n1Ptr, Node *n2Ptr)
{
    long depth;
    long i;
    Node *nodePtr;

    if (n1Ptr == n2Ptr) {
	return FALSE;
    }
    depth = MIN(n1Ptr->depth, n2Ptr->depth);
    if (depth == 0) {		/* One of the nodes is root. */
	return (n1Ptr->parent == NULL);
    }
    /* 
     * Traverse back from the deepest node, until both nodes are at
     * the same depth.  Check if this ancestor node is the same for
     * both nodes.
     */
    for (i = n1Ptr->depth; i > depth; i--) {
	n1Ptr = n1Ptr->parent;
    }
    if (n1Ptr == n2Ptr) {
	return FALSE;
    }
    for (i = n2Ptr->depth; i > depth; i--) {
	n2Ptr = n2Ptr->parent;
    }
    if (n2Ptr == n1Ptr) {
	return TRUE;
    }

    /* 
     * First find the mutual ancestor of both nodes.  Look at each
     * preceding ancestor level-by-level for both nodes.  Eventually
     * we'll find a node that's the parent of both ancestors.  Then
     * find the first ancestor in the parent's list of subnodes.  
     */
    for (i = depth; i > 0; i--) {
	if (n1Ptr->parent == n2Ptr->parent) {
	    break;
	}
	n1Ptr = n1Ptr->parent;
	n2Ptr = n2Ptr->parent;
    }
    for (nodePtr = n1Ptr->parent->first; nodePtr != NULL; 
	 nodePtr = nodePtr->next) {
	if (nodePtr == n1Ptr) {
	    return TRUE;
	} else if (nodePtr == n2Ptr) {
	    return FALSE;
	}
    }
    return FALSE;
}

static void
CallTraces(
    Tcl_Interp *interp,
    TreeClient *sourcePtr,	/* Client holding a reference to the
				 * tree.  If NULL, indicates to
				 * execute all handlers, including
				 * those of the caller. */
    TreeObject *treeObjPtr,	/* Tree that was changed. */
    Node *nodePtr,		/* Node that received the event. */
    Blt_TreeKey key,
    unsigned int flags)
{
    Blt_ChainLink *l1Ptr;

    for(l1Ptr = Blt_ChainFirstLink(treeObjPtr->clients); l1Ptr != NULL; 
	l1Ptr = Blt_ChainNextLink(l1Ptr)) {
	Blt_ChainLink *l2Ptr;
	TreeClient *clientPtr;

	clientPtr = Blt_ChainGetValue(l1Ptr);
	for(l2Ptr = Blt_ChainFirstLink(clientPtr->traces); l2Ptr != NULL; 
	    l2Ptr = Blt_ChainNextLink(l2Ptr)) {
	    TraceHandler *tracePtr;	

	    tracePtr = Blt_ChainGetValue(l2Ptr);
	    if ((tracePtr->keyPattern != NULL) && 
		(!Tcl_StringMatch(key, tracePtr->keyPattern))) {
		continue;	/* Key pattern doesn't match. */
	    }
	    if ((tracePtr->withTag != NULL) && 
		(!Blt_TreeHasTag(clientPtr, nodePtr, tracePtr->withTag))) {
		continue;	/* Doesn't have the tag. */
	    }
	    if ((tracePtr->mask & flags) == 0) {
		continue;	/* Flags don't match. */
	    }
	    if ((clientPtr == sourcePtr) && 
		(tracePtr->mask & TREE_TRACE_FOREIGN_ONLY)) {
		continue;	/* This client initiated the trace. */
	    }
	    if ((tracePtr->nodePtr != NULL) && (tracePtr->nodePtr != nodePtr)) {
		continue;	/* Nodes don't match. */
	    }
	    nodePtr->flags |= TREE_TRACE_ACTIVE;
	    if ((*tracePtr->proc) (tracePtr->clientData, treeObjPtr->interp, 
		nodePtr, key, flags) != TCL_OK) {
		if (interp != NULL) {
		    Tcl_BackgroundError(interp);
		}
	    }
	    nodePtr->flags &= ~TREE_TRACE_ACTIVE;
	}
    }
}

static Value *
GetTreeValue(
    Tcl_Interp *interp,
    TreeClient *clientPtr,
    Node *nodePtr,
    Blt_TreeKey key)
{
    Value *valuePtr;

    valuePtr = TreeFindValue(nodePtr, key); 
    if (valuePtr == NULL) {
	if (interp != NULL) {
	    Tcl_AppendResult(interp, "can't find field \"", key, "\"", 
			     (char *)NULL);
	}
	return NULL;
    }	
    if ((valuePtr->owner != NULL) && (valuePtr->owner != clientPtr)) {
	if (interp != NULL) {
	    Tcl_AppendResult(interp, "can't access private field \"", 
			     key, "\"", (char *)NULL);
	}
	return NULL;
    }
    return valuePtr;
}

int
Blt_TreePrivateValue(
    Tcl_Interp *interp,
    TreeClient *clientPtr,
    Node *nodePtr,
    Blt_TreeKey key)
{
    Value *valuePtr;

    valuePtr = TreeFindValue(nodePtr, key); 
    if (valuePtr == NULL) {
	if (interp != NULL) {
	    Tcl_AppendResult(interp, "can't find field \"", key, "\".", 
			     (char *)NULL);
	}
	return TCL_ERROR;
    }
    valuePtr->owner = clientPtr;
    return TCL_OK;
}

int
Blt_TreePublicValue(
    Tcl_Interp *interp,
    TreeClient *clientPtr,
    Node *nodePtr,
    Blt_TreeKey key)
{
    Value *valuePtr;

    valuePtr = TreeFindValue(nodePtr, key); 
    if (valuePtr == NULL) {
	if (interp != NULL) {
	    Tcl_AppendResult(interp, "can't find field \"", key, "\"", 
			     (char *)NULL);
	}
	return TCL_ERROR;
    }
    if (valuePtr->owner != clientPtr) {
	if (interp != NULL) {
	    Tcl_AppendResult(interp, "not the owner of \"", key, "\"", 
		     (char *)NULL);
	}
	return TCL_ERROR;
    }
    valuePtr->owner = NULL;
    return TCL_OK;
}

int
Blt_TreeValueExistsByKey(
    TreeClient *clientPtr,
    Node *nodePtr,
    Blt_TreeKey key)
{
    Value *valuePtr;

    valuePtr = GetTreeValue((Tcl_Interp *)NULL, clientPtr, nodePtr, key);
    if (valuePtr == NULL) {
	return FALSE;
    }
    return TRUE;
}

int
Blt_TreeGetValueByKey(
    Tcl_Interp *interp,
    TreeClient *clientPtr,
    Node *nodePtr,
    Blt_TreeKey key,
    Tcl_Obj **objPtrPtr)
{
    Value *valuePtr;
    TreeObject *treeObjPtr = nodePtr->treeObject;

    valuePtr = GetTreeValue(interp, clientPtr, nodePtr, key);
    if (valuePtr == NULL) {
	return TCL_ERROR;
    }
    *objPtrPtr = valuePtr->objPtr;
    if (!(nodePtr->flags & TREE_TRACE_ACTIVE)) {
	CallTraces(interp, clientPtr, treeObjPtr, nodePtr, key, 
		   TREE_TRACE_READ);
    }
    return TCL_OK;
}

int
Blt_TreeSetValueByKey(
    Tcl_Interp *interp,
    TreeClient *clientPtr,
    Node *nodePtr,		/* Node to be updated. */
    Blt_TreeKey key,		/* Identifies the field key. */
    Tcl_Obj *objPtr)		/* New value of field. */
{
    TreeObject *treeObjPtr = nodePtr->treeObject;
    Value *valuePtr;
    unsigned int flags;
    int isNew;

    assert(objPtr != NULL);
    valuePtr = TreeCreateValue(nodePtr, key, &isNew);
    if ((valuePtr->owner != NULL) && (valuePtr->owner != clientPtr)) {
	if (interp != NULL) {
	    Tcl_AppendResult(interp, "can't set private field \"", 
			     key, "\"", (char *)NULL);
	}
	return TCL_ERROR;
    }
    if (objPtr != valuePtr->objPtr) {
	Tcl_IncrRefCount(objPtr);
	if (valuePtr->objPtr != NULL) {
	    Tcl_DecrRefCount(valuePtr->objPtr);
	}
	valuePtr->objPtr = objPtr;
    }
    flags = TREE_TRACE_WRITE;
    if (isNew) {
	flags |= TREE_TRACE_CREATE;
    }
    if (!(nodePtr->flags & TREE_TRACE_ACTIVE)) {
	CallTraces(interp, clientPtr, treeObjPtr, nodePtr, valuePtr->key, 
		flags);
    }
    return TCL_OK;
}

int
Blt_TreeUnsetValueByKey(
    Tcl_Interp *interp,
    TreeClient *clientPtr,
    Node *nodePtr,		/* Node to be updated. */
    Blt_TreeKey key)		/* Name of field in node. */
{
    TreeObject *treeObjPtr = nodePtr->treeObject;
    Value *valuePtr;

    valuePtr = TreeFindValue(nodePtr, key);
    if (valuePtr == NULL) {
	return TCL_OK;		/* It's okay to unset values that don't
				 * exist in the node. */
    }
    if ((valuePtr->owner != NULL) && (valuePtr->owner != clientPtr)) {
	if (interp != NULL) {
	    Tcl_AppendResult(interp, "can't unset private field \"", 
			     key, "\"", (char *)NULL);
	}
	return TCL_ERROR;
    }
    TreeDeleteValue(nodePtr, valuePtr);
    CallTraces(interp, clientPtr, treeObjPtr, nodePtr, key, TREE_TRACE_UNSET);
    return TCL_OK;
}

static int
ParseParentheses(
    Tcl_Interp *interp,
    CONST char *string,
    char **leftPtr, 
    char **rightPtr)
{
    char *p;
    char *left, *right;

    left = right = NULL;
    for (p = (char *)string; *p != '\0'; p++) {
	if (*p == '(') {
	    left = p;
	} else if (*p == ')') {
	    right = p;
	}
    }
    if (left != right) {
	if (((left != NULL) && (right == NULL)) ||
	    ((left == NULL) && (right != NULL)) ||
	    (left > right) || (right != (p - 1))) {
	    if (interp != NULL) {
		Tcl_AppendResult(interp, "bad array specification \"", string,
			     "\"", (char *)NULL);
	    }
	    return TCL_ERROR;
	}
    }
    *leftPtr = left;
    *rightPtr = right;
    return TCL_OK;
}

int
Blt_TreeGetValue(
    Tcl_Interp *interp,
    TreeClient *clientPtr,
    Node *nodePtr,
    CONST char *string,		/* String identifying the field in node. */
    Tcl_Obj **objPtrPtr)
{
    char *left, *right;
    int result;

    if (ParseParentheses(interp, string, &left, &right) != TCL_OK) {
	return TCL_ERROR;
    }
    if (left != NULL) {
	*left = *right = '\0';
	result = Blt_TreeGetArrayValue(interp, clientPtr, nodePtr, string, 
		left + 1, objPtrPtr);
	*left = '(', *right = ')';
    } else {
	result = Blt_TreeGetValueByKey(interp, clientPtr, nodePtr, 
		Blt_TreeGetKey(string), objPtrPtr);
    }
    return result;
}

int
Blt_TreeSetValue(
    Tcl_Interp *interp,
    TreeClient *clientPtr,
    Node *nodePtr,		/* Node to be updated. */
    CONST char *string,		/* String identifying the field in node. */
    Tcl_Obj *valueObjPtr)	/* New value of field. If NULL, field
				 * is deleted. */
{
    char *left, *right;
    int result;

    if (ParseParentheses(interp, string, &left, &right) != TCL_OK) {
	return TCL_ERROR;
    }
    if (left != NULL) {
	*left = *right = '\0';
	result = Blt_TreeSetArrayValue(interp, clientPtr, nodePtr, string, 
		left + 1, valueObjPtr);
	*left = '(', *right = ')';
    } else {
	result = Blt_TreeSetValueByKey(interp, clientPtr, nodePtr, 
			       Blt_TreeGetKey(string), valueObjPtr);
    }
    return result;
}

int
Blt_TreeUnsetValue(
    Tcl_Interp *interp,
    TreeClient *clientPtr,
    Node *nodePtr,		/* Node to be updated. */
    CONST char *string)		/* String identifying the field in node. */
{
    char *left, *right;
    int result;

    if (ParseParentheses(interp, string, &left, &right) != TCL_OK) {
	return TCL_ERROR;
    }
    if (left != NULL) {
	*left = *right = '\0';
	result = Blt_TreeUnsetArrayValue(interp, clientPtr, nodePtr, string, 
		left + 1);
	*left = '(', *right = ')';
    } else {
	result = Blt_TreeUnsetValueByKey(interp, clientPtr, nodePtr, 
		Blt_TreeGetKey(string));
    }
    return result;
}

int
Blt_TreeValueExists(
    TreeClient *clientPtr, 
    Node *nodePtr, 
    CONST char *string)
{
    char *left, *right;
    int result;

    if (ParseParentheses((Tcl_Interp *)NULL, string, &left, &right) != TCL_OK) {
	return FALSE;
    }
    if (left != NULL) {
	*left = *right = '\0';
	result = Blt_TreeArrayValueExists(clientPtr, nodePtr, string, left + 1);
	*left = '(', *right = ')';
    } else {
	result = Blt_TreeValueExistsByKey(clientPtr, nodePtr, 
		Blt_TreeGetKey(string));
    }
    return result;
}

Blt_TreeKey
Blt_TreeFirstKey(
    TreeClient *clientPtr, 
    Node *nodePtr, 
    Blt_TreeKeySearch *iterPtr)
{
    Value *valuePtr;
    
    valuePtr = TreeFirstValue(nodePtr, iterPtr);
    if (valuePtr == NULL) {
	return NULL;
    }
    while ((valuePtr->owner != NULL) && (valuePtr->owner != clientPtr)) {
	valuePtr = TreeNextValue(iterPtr);
	if (valuePtr == NULL) {
	    return NULL;
	}
    }
    return valuePtr->key;
}

Blt_TreeKey
Blt_TreeNextKey(
    TreeClient *clientPtr, 
    Blt_TreeKeySearch *iterPtr)
{
    Value *valuePtr;

    valuePtr = TreeNextValue(iterPtr);
    if (valuePtr == NULL) {
	return NULL;
    }
    while ((valuePtr->owner != NULL) && (valuePtr->owner != clientPtr)) {
	valuePtr = TreeNextValue(iterPtr);
	if (valuePtr == NULL) {
	    return NULL;
	}
    }
    return valuePtr->key;
}

int
Blt_TreeIsAncestor(Node *n1Ptr, Node *n2Ptr)
{
    if (n2Ptr != NULL) {
	n2Ptr = n2Ptr->parent;
	while (n2Ptr != NULL) {
	    if (n2Ptr == n1Ptr) {
		return TRUE;
	    }
	    n2Ptr = n2Ptr->parent;
	}
    }
    return FALSE;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_TreeSortNode --
 *
 *	Sorts the subnodes at a given node.
 *
 * Results:
 *	Always returns TCL_OK.
 *
 *----------------------------------------------------------------------
 */
int
Blt_TreeSortNode(
    TreeClient *clientPtr,
    Node *nodePtr,
    Blt_TreeCompareNodesProc *proc)
{
    Node **nodes;
    int nNodes;
    Node *np, **npp;

    nNodes = nodePtr->nChildren;
    if (nNodes < 2) {
	return TCL_OK;
    }
    nodes = Blt_Malloc((nNodes + 1) * sizeof(Node *));
    if (nodes == NULL) {
	return TCL_ERROR;	/* Out of memory. */
    }
    for (npp = nodes, np = nodePtr->first; np != NULL; np = np->next, npp++) {
	*npp = np;
    }
    *npp = NULL;

    qsort((char *)nodes, nNodes, sizeof(Node *), (QSortCompareProc *)proc);
    for (npp = nodes; *npp != NULL; npp++) {
	UnlinkNode(*npp);
	LinkBefore(nodePtr, *npp, (Blt_TreeNode)NULL);
    }
    Blt_Free(nodes);
    NotifyClients(clientPtr, nodePtr->treeObject, nodePtr, TREE_NOTIFY_SORT);
    return TCL_OK;
}

#define TEST_RESULT(result) \
	switch (result) { \
	case TCL_CONTINUE: \
	    return TCL_OK; \
	case TCL_OK: \
	    break; \
	default: \
	    return (result); \
	}

int
Blt_TreeApply(
    Node *branchPtr,		/* Root node of subtree. */
    Blt_TreeApplyProc *proc,	/* Procedure to call for each node. */
    ClientData clientData)	/* One-word of data passed when calling
				 * proc. */
{
    Node *np, *nextPtr;

    for (np = branchPtr->first; np != NULL; np = nextPtr) {
	int result;

	/* 
	 * Get the next link in the chain before calling Blt_TreeApply
	 * recursively.  This is because the apply callback may delete
	 * the node and its link.  
	 */
	nextPtr = np->next;

	result = Blt_TreeApply(np, proc, clientData);
	TEST_RESULT(result);
    }
    return (*proc) (branchPtr, clientData, TREE_POSTORDER);
}

int
Blt_TreeApplyDFS(
    Node *branchPtr,		/* Root node of subtree. */
    Blt_TreeApplyProc *proc,	/* Procedure to call for each node. */
    ClientData clientData,	/* One-word of data passed when calling
				 * proc. */
    int order)			/* Order of traversal. */
{
    Node *nodePtr, *nextPtr;
    int result;

    if (order & TREE_PREORDER) {
	result = (*proc) (branchPtr, clientData, TREE_PREORDER);
	TEST_RESULT(result);
    }
    nodePtr = branchPtr->first;
    if (order & TREE_INORDER) {
	if (nodePtr != NULL) {
	    result = Blt_TreeApplyDFS(nodePtr, proc, clientData, order);
	    TEST_RESULT(result);
	    nodePtr = nodePtr->next;
	}
	result = (*proc) (branchPtr, clientData, TREE_INORDER);
	TEST_RESULT(result);
    }
    for (/* empty */; nodePtr != NULL; nodePtr = nextPtr) {
	/* 
	 * Get the next link in the chain before calling
	 * Blt_TreeApply recursively.  This is because the 
	 * apply callback may delete the node and its link.  
	 */
	nextPtr = nodePtr->next;
	result = Blt_TreeApplyDFS(nodePtr, proc, clientData, order);
	TEST_RESULT(result);
    }
    if (order & TREE_POSTORDER) {
	return (*proc) (branchPtr, clientData, TREE_POSTORDER);
    }
    return TCL_OK;
}

int
Blt_TreeApplyBFS(
    Node *branchPtr,		/* Root node of subtree. */
    Blt_TreeApplyProc *proc,	/* Procedure to call for each node. */
    ClientData clientData)	/* One-word of data passed when calling
				 * proc. */
{
    Blt_Chain *queuePtr;
    Blt_ChainLink *linkPtr, *nextPtr;
    Node *nodePtr;
    int result;

    queuePtr = Blt_ChainCreate();
    linkPtr = Blt_ChainAppend(queuePtr, branchPtr);
    while (linkPtr != NULL) {
	Node *np;

	nodePtr = Blt_ChainGetValue(linkPtr);
	/* Add the children to the queue. */
	for (np = nodePtr->first; np != NULL; np = np->next) {
	    Blt_ChainAppend(queuePtr, np);
	}
	/* Process the node. */
	result = (*proc) (nodePtr, clientData, TREE_BREADTHFIRST);
	switch (result) { 
	case TCL_CONTINUE: 
	    Blt_ChainDestroy(queuePtr);
	    return TCL_OK; 
	case TCL_OK: 
	    break; 
	default: 
	    Blt_ChainDestroy(queuePtr);
	    return result; 
	}
	/* Remove the node from the queue. */
	nextPtr = Blt_ChainNextLink(linkPtr);
	Blt_ChainDeleteLink(queuePtr, linkPtr);
	linkPtr = nextPtr;
    }
    Blt_ChainDestroy(queuePtr);
    return TCL_OK;
}

static TreeClient *
NewTreeClient(TreeObject *treeObjPtr)
{
    TreeClient *clientPtr;

    clientPtr = Blt_Calloc(1, sizeof(TreeClient));
    if (clientPtr != NULL) {
	Blt_TreeTagTable *tablePtr;

	clientPtr->magic = TREE_MAGIC;
	clientPtr->linkPtr = Blt_ChainAppend(treeObjPtr->clients, clientPtr);
	clientPtr->events = Blt_ChainCreate();
	clientPtr->traces = Blt_ChainCreate();
	clientPtr->treeObject = treeObjPtr;
	clientPtr->root = treeObjPtr->root;
	tablePtr = Blt_Malloc(sizeof(Blt_TreeTagTable));
	Blt_InitHashTable(&tablePtr->tagTable, BLT_STRING_KEYS);
	tablePtr->refCount = 1;
	clientPtr->tagTablePtr = tablePtr;
    }
    return clientPtr;
}

int
Blt_TreeCreate(
    Tcl_Interp *interp,		/* Interpreter to report errors back to. */
    CONST char *name,		/* Name of tree in namespace.  Tree
				 * must not already exist. */
    TreeClient **clientPtrPtr)	/* (out) Client token of newly created
				 * tree.  Releasing the token will
				 * free the tree.  If NULL, no token
				 * is generated. */
{
    Tcl_DString ds;
    TreeInterpData *dataPtr;
    TreeObject *treeObjPtr;
    Blt_ObjectName objName;
    char string[200];

    dataPtr = GetTreeInterpData(interp);
    if (name != NULL) {
	/* Check if this tree already exists the current namespace. */
	treeObjPtr = GetTreeObject(interp, name, NS_SEARCH_CURRENT);
	if (treeObjPtr != NULL) {
	    Tcl_AppendResult(interp, "a tree object \"", name,
			     "\" already exists", (char *)NULL);
	    return TCL_ERROR;
	}
    } else {
	/* Generate a unique tree name in the current namespace. */
	do  {
	    sprintf(string, "tree%d", dataPtr->nextId++);
	} while (GetTreeObject(interp, name, NS_SEARCH_CURRENT) != NULL);
	name = string;
    } 
    /* 
     * Tear apart and put back together the namespace-qualified name 
     * of the tree. This is to ensure that naming is consistent.
     */ 
    if (!Blt_ParseObjectName(interp, name, &objName, 0)) {
	return TCL_ERROR;
    }
    name = Blt_MakeQualifiedName(&objName, &ds);
    treeObjPtr = NewTreeObject(dataPtr, interp, name);
    if (treeObjPtr == NULL) {
	Tcl_AppendResult(interp, "can't allocate tree \"", name, "\"", 
		(char *)NULL);
	Tcl_DStringFree(&ds);
	return TCL_ERROR;
    }
    Tcl_DStringFree(&ds);
    if (clientPtrPtr != NULL) {
	TreeClient *clientPtr;
	
	clientPtr = NewTreeClient(treeObjPtr);
	if (clientPtr == NULL) {
	    Tcl_AppendResult(interp, "can't allocate tree token",(char *)NULL);
	    return TCL_ERROR;
	}
	*clientPtrPtr = clientPtr;
    }
    return TCL_OK;
}

int
Blt_TreeGetToken(
    Tcl_Interp *interp,		/* Interpreter to report errors back to. */
    CONST char *name,		/* Name of tree in namespace. */
    TreeClient **clientPtrPtr)
{
    TreeClient *clientPtr;
    TreeObject *treeObjPtr;

    treeObjPtr = GetTreeObject(interp, name, NS_SEARCH_BOTH);
    if (treeObjPtr == NULL) {
	Tcl_AppendResult(interp, "can't find a tree object \"", name, "\"", 
			 (char *)NULL);
	return TCL_ERROR;
    }
    clientPtr = NewTreeClient(treeObjPtr);
    if (clientPtr == NULL) {
	Tcl_AppendResult(interp, "can't allocate token for tree \"", 
			 name, "\"", (char *)NULL);
	return TCL_ERROR;
    }
    *clientPtrPtr = clientPtr;
    return TCL_OK;
}

void
Blt_TreeReleaseToken(TreeClient *clientPtr)
{
    TreeObject *treeObjPtr;
    Blt_ChainLink *linkPtr;

    if (clientPtr->magic != TREE_MAGIC) {
	fprintf(stderr, "invalid tree object token 0x%lx\n", 
		(unsigned long)clientPtr);
	return;
    }
    /* Remove any traces that may be set. */
    for (linkPtr = Blt_ChainFirstLink(clientPtr->traces); linkPtr != NULL;
	 linkPtr = Blt_ChainNextLink(linkPtr)) {
	TraceHandler *tracePtr;

	tracePtr = Blt_ChainGetValue(linkPtr);
	if (tracePtr->keyPattern != NULL) {
	    Blt_Free(tracePtr->keyPattern);
	}
	Blt_Free(tracePtr);
    }
    Blt_ChainDestroy(clientPtr->traces);
    /* And any event handlers. */
    for(linkPtr = Blt_ChainFirstLink(clientPtr->events); 
	linkPtr != NULL; linkPtr = Blt_ChainNextLink(linkPtr)) {
	EventHandler *notifyPtr;

	notifyPtr = Blt_ChainGetValue(linkPtr);
	if (notifyPtr->notifyPending) {
	    Tcl_CancelIdleCall(NotifyIdleProc, notifyPtr);
	}
	Blt_Free(notifyPtr);
    }
    if (clientPtr->tagTablePtr != NULL) {
	ReleaseTagTable(clientPtr->tagTablePtr);
    }
    Blt_ChainDestroy(clientPtr->events);
    treeObjPtr = clientPtr->treeObject;
    if (treeObjPtr != NULL) {
	/* Remove the client from the server's list */
	Blt_ChainDeleteLink(treeObjPtr->clients, clientPtr->linkPtr);
	if (Blt_ChainGetLength(treeObjPtr->clients) == 0) {
	    DestroyTreeObject(treeObjPtr);
	}
    }
    clientPtr->magic = 0;
    Blt_Free(clientPtr);
}

int
Blt_TreeExists(
    Tcl_Interp *interp,		/* Interpreter to report errors back to. */
    CONST char *name)		/* Namespace-qualifed name of tree. */
{
    TreeObject *treeObjPtr;

    treeObjPtr = GetTreeObject(interp, name, NS_SEARCH_BOTH);
    if (treeObjPtr == NULL) {
	Tcl_ResetResult(interp);
	return 0;
    }
    return 1;
}

/*ARGSUSED*/
static int
SizeApplyProc(
    Node *nodePtr,		/* Not used. */
    ClientData clientData,
    int order)			/* Not used. */
{
    int *sumPtr = clientData;
    *sumPtr = *sumPtr + 1;
    return TCL_OK;
}

int
Blt_TreeSize(Node *nodePtr)
{
    int sum;

    sum = 0;
    Blt_TreeApply(nodePtr, SizeApplyProc, &sum);
    return sum;
}


void
Blt_TreeCreateEventHandler(
    TreeClient *clientPtr,
    unsigned int mask,
    Blt_TreeNotifyEventProc *proc,
    ClientData clientData)
{
    Blt_ChainLink *linkPtr;
    EventHandler *notifyPtr;

    notifyPtr = NULL;		/* Suppress compiler warning. */

    /* Check if the event is already handled. */
    for(linkPtr = Blt_ChainFirstLink(clientPtr->events); 
	linkPtr != NULL; linkPtr = Blt_ChainNextLink(linkPtr)) {
	notifyPtr = Blt_ChainGetValue(linkPtr);
	if ((notifyPtr->proc == proc) && 
	    (notifyPtr->mask == mask) &&
	    (notifyPtr->clientData == clientData)) {
	    break;
	}
    }
    if (linkPtr == NULL) {
	notifyPtr = Blt_Malloc(sizeof (EventHandler));
	assert(notifyPtr);
	linkPtr = Blt_ChainAppend(clientPtr->events, notifyPtr);
    }
    if (proc == NULL) {
	Blt_ChainDeleteLink(clientPtr->events, linkPtr);
	Blt_Free(notifyPtr);
    } else {
	notifyPtr->proc = proc;
	notifyPtr->clientData = clientData;
	notifyPtr->mask = mask;
	notifyPtr->notifyPending = FALSE;
	notifyPtr->interp = clientPtr->treeObject->interp;
    }
}

void
Blt_TreeDeleteEventHandler(
    TreeClient *clientPtr,
    unsigned int mask,
    Blt_TreeNotifyEventProc *proc,
    ClientData clientData)
{
    Blt_ChainLink *linkPtr;
    EventHandler *notifyPtr;

    for(linkPtr = Blt_ChainFirstLink(clientPtr->events); 
	linkPtr != NULL; linkPtr = Blt_ChainNextLink(linkPtr)) {
	notifyPtr = Blt_ChainGetValue(linkPtr);
	if ((notifyPtr->proc == proc) && (notifyPtr->mask == mask) &&
	    (notifyPtr->clientData == clientData)) {
	    if (notifyPtr->notifyPending) {
		Tcl_CancelIdleCall(NotifyIdleProc, notifyPtr);
	    }
	    Blt_ChainDeleteLink(clientPtr->events, linkPtr);
	    Blt_Free(notifyPtr);
	    return;
	}
    }
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_TreePathFromNode --
 *
 *---------------------------------------------------------------------- 
 */
char *
Blt_TreeNodeRelativePath(
    Node *rootPtr,		/* Root of subtree. */
    Node *nodePtr,		/* Node whose path is to be
				 * returned. */
    CONST char *separator,	/* Character string to separator elements. */
    unsigned int flags,		/* Indicates how to print the path. */
    Tcl_DString *dsPtr)	/* (out) Contains the path of the
				 * node. */
{
    char **names;		/* Used to stack the component names. */
    char *staticSpace[64];
    long i;
    long nLevels;

    if (rootPtr == NULL) {
	rootPtr = nodePtr->treeObject->root;
    }
    nLevels = Blt_TreeNodeDepth(nodePtr) - Blt_TreeNodeDepth(rootPtr);
    if (flags & TREE_INCLUDE_ROOT) {
	nLevels++;
    }
    if (nLevels > 64) {
	names = Blt_Malloc(nLevels * sizeof(char *));
	assert(names);
    } else {
	names = staticSpace;
    }
    for (i = nLevels; i > 0; i--) {
	/* Save the name of each ancestor in the name array.  Note
	 * that we ignore the root. */
	names[i - 1] = nodePtr->label;
	nodePtr = nodePtr->parent;
    }
    /* Append each the names in the array. */
    if ((nLevels > 0) && (separator != NULL)) {
	Tcl_DStringAppend(dsPtr, names[0], -1);
	for (i = 1; i < nLevels; i++) {
	    Tcl_DStringAppend(dsPtr, separator, -1);
	    Tcl_DStringAppend(dsPtr, names[i], -1);
	}
    } else {
	for (i = 0; i < nLevels; i++) {
	    Tcl_DStringAppendElement(dsPtr, names[i]);
	}
    }
    if (names != staticSpace) {
	Blt_Free(names);
    }
    return Tcl_DStringValue(dsPtr);
}
    
/*
 *----------------------------------------------------------------------
 *
 * Blt_TreeNodePath --
 *
 *---------------------------------------------------------------------- 
 */
char *
Blt_TreeNodePath(Node *nodePtr, Tcl_DString *dsPtr)
{
    Blt_TreeNode root;

    root = nodePtr->treeObject->root;
    return Blt_TreeNodeRelativePath(root, nodePtr, NULL, 0, dsPtr);
}

int
Blt_TreeArrayValueExists(
    TreeClient *clientPtr,
    Node *nodePtr,
    CONST char *arrayName, 
    CONST char *elemName)
{
    Blt_TreeKey key;
    Blt_HashEntry *hPtr;
    Blt_HashTable *tablePtr;
    Value *valuePtr;

    key = Blt_TreeGetKey(arrayName);
    valuePtr = GetTreeValue((Tcl_Interp *)NULL, clientPtr, nodePtr, key);
    if (valuePtr == NULL) {
	return FALSE;
    }
    if (Tcl_IsShared(valuePtr->objPtr)) {
	Tcl_DecrRefCount(valuePtr->objPtr);
	valuePtr->objPtr = Tcl_DuplicateObj(valuePtr->objPtr);
	Tcl_IncrRefCount(valuePtr->objPtr);
    }
    if (Blt_GetArrayFromObj((Tcl_Interp *)NULL, valuePtr->objPtr, &tablePtr) 
	!= TCL_OK) {
	return FALSE;
    }
    hPtr = Blt_FindHashEntry(tablePtr, elemName);
    return (hPtr != NULL);
}

int
Blt_TreeGetArrayValue(
    Tcl_Interp *interp,
    TreeClient *clientPtr,
    Node *nodePtr,
    CONST char *arrayName,
    CONST char *elemName,
    Tcl_Obj **valueObjPtrPtr)
{
    Blt_TreeKey key;
    Blt_HashEntry *hPtr;
    Blt_HashTable *tablePtr;
    Value *valuePtr;

    key = Blt_TreeGetKey(arrayName);
    valuePtr = GetTreeValue(interp, clientPtr, nodePtr, key);
    if (valuePtr == NULL) {
	return TCL_ERROR;
    }
    if (Tcl_IsShared(valuePtr->objPtr)) {
	Tcl_DecrRefCount(valuePtr->objPtr);
	valuePtr->objPtr = Tcl_DuplicateObj(valuePtr->objPtr);
	Tcl_IncrRefCount(valuePtr->objPtr);
    }
    if (Blt_GetArrayFromObj(interp, valuePtr->objPtr, &tablePtr) != TCL_OK) {
	return TCL_ERROR;
    }
    hPtr = Blt_FindHashEntry(tablePtr, elemName);
    if (hPtr == NULL) {
	if (interp != NULL) {
	    Tcl_AppendResult(interp, "can't find \"", arrayName, "(",
			     elemName, ")\"", (char *)NULL);
	}
	return TCL_ERROR;
    }
    *valueObjPtrPtr = (Tcl_Obj *)Blt_GetHashValue(hPtr);

    /* Reading any element of the array can cause a trace to fire. */
    if (!(nodePtr->flags & TREE_TRACE_ACTIVE)) {
	CallTraces(interp, clientPtr, nodePtr->treeObject, nodePtr, key, 
		   TREE_TRACE_READ);
    }
    return TCL_OK;
}

int
Blt_TreeSetArrayValue(
    Tcl_Interp *interp,
    TreeClient *clientPtr,
    Node *nodePtr,		/* Node to be updated. */
    CONST char *arrayName,
    CONST char *elemName,
    Tcl_Obj *valueObjPtr)	/* New value of element. */
{
    Blt_TreeKey key;
    Blt_HashEntry *hPtr;
    Blt_HashTable *tablePtr;
    Value *valuePtr;
    unsigned int flags;
    int isNew;

    assert(valueObjPtr != NULL);

    /* 
     * Search for the array in the list of data fields.  If one
     * doesn't exist, create it.
     */
    key = Blt_TreeGetKey(arrayName);
    valuePtr = TreeCreateValue(nodePtr, key, &isNew);
    if ((valuePtr->owner != NULL) && (valuePtr->owner != clientPtr)) {
	if (interp != NULL) {
	    Tcl_AppendResult(interp, "can't set private field \"", 
			     key, "\"", (char *)NULL);
	}
	return TCL_ERROR;
    }
    flags = TREE_TRACE_WRITE;
    if (isNew) {
	valuePtr->objPtr = Blt_NewArrayObj(0, (Tcl_Obj **)NULL);
	Tcl_IncrRefCount(valuePtr->objPtr);
	flags |= TREE_TRACE_CREATE;
    } else if (Tcl_IsShared(valuePtr->objPtr)) {
	Tcl_DecrRefCount(valuePtr->objPtr);
	valuePtr->objPtr = Tcl_DuplicateObj(valuePtr->objPtr);
	Tcl_IncrRefCount(valuePtr->objPtr);
    }
    if (Blt_GetArrayFromObj(interp, valuePtr->objPtr, &tablePtr) != TCL_OK) {
	return TCL_ERROR;
    }
    Tcl_InvalidateStringRep(valuePtr->objPtr);
    hPtr = Blt_CreateHashEntry(tablePtr, elemName, &isNew);
    assert(hPtr);

    Tcl_IncrRefCount(valueObjPtr);
    if (!isNew) {
	Tcl_Obj *oldValueObjPtr;

	/* An element by the same name already exists. Decrement the
	 * reference count of the old value. */

	oldValueObjPtr = (Tcl_Obj *)Blt_GetHashValue(hPtr);
	if (oldValueObjPtr != NULL) {
	    Tcl_DecrRefCount(oldValueObjPtr);
	}
    }
    Blt_SetHashValue(hPtr, valueObjPtr);

    /*
     * We don't handle traces on a per array element basis.  Setting
     * any element can fire traces for the value.
     */
    if (!(nodePtr->flags & TREE_TRACE_ACTIVE)) {
	CallTraces(interp, clientPtr, nodePtr->treeObject, nodePtr, 
		valuePtr->key, flags);
    }
    return TCL_OK;
}

int
Blt_TreeUnsetArrayValue(
    Tcl_Interp *interp,
    TreeClient *clientPtr,
    Node *nodePtr,		/* Node to be updated. */
    CONST char *arrayName,
    CONST char *elemName)
{
    Blt_TreeKey key;		/* Name of field in node. */
    Blt_HashEntry *hPtr;
    Blt_HashTable *tablePtr;
    Tcl_Obj *valueObjPtr;
    Value *valuePtr;

    key = Blt_TreeGetKey(arrayName);
    valuePtr = TreeFindValue(nodePtr, key);
    if (valuePtr == NULL) {
	return TCL_OK;
    }
    if ((valuePtr->owner != NULL) && (valuePtr->owner != clientPtr)) {
	if (interp != NULL) {
	    Tcl_AppendResult(interp, "can't unset private field \"", 
			     key, "\"", (char *)NULL);
	}
	return TCL_ERROR;
    }
    if (Tcl_IsShared(valuePtr->objPtr)) {
	Tcl_DecrRefCount(valuePtr->objPtr);
	valuePtr->objPtr = Tcl_DuplicateObj(valuePtr->objPtr);
	Tcl_IncrRefCount(valuePtr->objPtr);
    }
    if (Blt_GetArrayFromObj(interp, valuePtr->objPtr, &tablePtr) != TCL_OK) {
	return TCL_ERROR;
    }
    hPtr = Blt_FindHashEntry(tablePtr, elemName);
    if (hPtr == NULL) {
	return TCL_OK;		/* Element doesn't exist, Ok. */
    }
    valueObjPtr = (Tcl_Obj *)Blt_GetHashValue(hPtr);
    Tcl_DecrRefCount(valueObjPtr);
    Blt_DeleteHashEntry(tablePtr, hPtr);

    /*
     * Un-setting any element in the array can cause the trace on the value
     * to fire.
     */
    if (!(nodePtr->flags & TREE_TRACE_ACTIVE)) {
	CallTraces(interp, clientPtr, nodePtr->treeObject, nodePtr, 
		valuePtr->key, TREE_TRACE_WRITE);
    }
    return TCL_OK;
}

int
Blt_TreeArrayNames(
    Tcl_Interp *interp,
    TreeClient *clientPtr,
    Node *nodePtr,
    CONST char *arrayName,
    Tcl_Obj *listObjPtr)
{
    Blt_HashEntry *hPtr;
    Blt_HashSearch cursor;
    Blt_HashTable *tablePtr;
    Value *valuePtr;
    char *key;

    key = Blt_TreeGetKey(arrayName);
    valuePtr = GetTreeValue(interp, clientPtr, nodePtr, key);
    if (valuePtr == NULL) {
	return TCL_ERROR;
    }
    if (Tcl_IsShared(valuePtr->objPtr)) {
	Tcl_DecrRefCount(valuePtr->objPtr);
	valuePtr->objPtr = Tcl_DuplicateObj(valuePtr->objPtr);
	Tcl_IncrRefCount(valuePtr->objPtr);
    }
    if (Blt_GetArrayFromObj(interp, valuePtr->objPtr, &tablePtr) != TCL_OK) {
	return TCL_ERROR;
    }
    tablePtr = (Blt_HashTable *)valuePtr->objPtr;
    for (hPtr = Blt_FirstHashEntry(tablePtr, &cursor); hPtr != NULL; 
	 hPtr = Blt_NextHashEntry(&cursor)) {
	Tcl_Obj *objPtr;

	objPtr = Tcl_NewStringObj(Blt_GetHashKey(tablePtr, hPtr), -1);
	Tcl_ListObjAppendElement(interp, listObjPtr, objPtr);
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_TreeShareTagTable --
 *
 *---------------------------------------------------------------------- 
 */
int
Blt_TreeShareTagTable(
    TreeClient *sourcePtr,
    TreeClient *targetPtr)
{
    sourcePtr->tagTablePtr->refCount++;
    if (targetPtr->tagTablePtr != NULL) {
	ReleaseTagTable(targetPtr->tagTablePtr);
    }
    targetPtr->tagTablePtr = sourcePtr->tagTablePtr;
    return TCL_OK;
}

int
Blt_TreeTagTableIsShared(TreeClient *clientPtr)
{
    return (clientPtr->tagTablePtr->refCount > 1);
}   

void
Blt_TreeClearTags(TreeClient *clientPtr, Node *nodePtr)
{
    Blt_HashEntry *hPtr;
    Blt_HashSearch cursor;
    
    for (hPtr = Blt_FirstHashEntry(&clientPtr->tagTablePtr->tagTable, &cursor); 
	hPtr != NULL; hPtr = Blt_NextHashEntry(&cursor)) {
	Blt_TreeTagEntry *tePtr;
	Blt_HashEntry *h2Ptr;

	tePtr = Blt_GetHashValue(hPtr);
	h2Ptr = Blt_FindHashEntry(&tePtr->nodeTable, (char *)nodePtr);
	if (h2Ptr != NULL) {
	    Blt_DeleteHashEntry(&tePtr->nodeTable, h2Ptr);
	}
    }
}

Blt_TreeTagEntry *
Blt_TreeRememberTag(TreeClient *clientPtr, CONST char *tagName)
{
    int isNew;
    Blt_HashEntry *hPtr;
    Blt_TreeTagEntry *tePtr;
    Blt_HashTable *tablePtr;

    tablePtr = &clientPtr->tagTablePtr->tagTable;
    hPtr = Blt_CreateHashEntry(tablePtr, tagName, &isNew);
    assert(hPtr);
    if (isNew) {
	tePtr = Blt_Malloc(sizeof(Blt_TreeTagEntry));
	Blt_InitHashTable(&tePtr->nodeTable, BLT_ONE_WORD_KEYS);
	Blt_SetHashValue(hPtr, tePtr);
	tePtr->hashPtr = hPtr;
	tePtr->tagName = Blt_GetHashKey(tablePtr, hPtr);
    } else {
	tePtr = Blt_GetHashValue(hPtr);
    }
    return tePtr;
}

void
Blt_TreeForgetTag(TreeClient *clientPtr, CONST char *tagName)
{
    if ((strcmp(tagName, "all") != 0) && (strcmp(tagName, "root") != 0)) {
	Blt_HashEntry *hPtr;
	
	hPtr = Blt_FindHashEntry(&clientPtr->tagTablePtr->tagTable, tagName);
	if (hPtr != NULL) {
	    Blt_TreeTagEntry *tePtr;
	    
	    Blt_DeleteHashEntry(&clientPtr->tagTablePtr->tagTable, hPtr);
	    tePtr = Blt_GetHashValue(hPtr);
	    Blt_DeleteHashTable(&tePtr->nodeTable);
	    Blt_Free(tePtr);
	}
    }
}

int
Blt_TreeHasTag(
    TreeClient *clientPtr,
    Node *nodePtr,
    CONST char *tagName)
{
    Blt_HashEntry *hPtr;
    Blt_TreeTagEntry *tePtr;

    if (strcmp(tagName, "all") == 0) {
	return TRUE;
    }
    if ((strcmp(tagName, "root") == 0) && 
	(nodePtr == Blt_TreeRootNode(clientPtr))) {
	return TRUE;
    }
    hPtr = Blt_FindHashEntry(&clientPtr->tagTablePtr->tagTable, tagName);
    if (hPtr == NULL) {
	return FALSE;
    }
    tePtr = Blt_GetHashValue(hPtr);
    hPtr = Blt_FindHashEntry(&tePtr->nodeTable, (char *)nodePtr);
    if (hPtr == NULL) {
	return FALSE;
    }
    return TRUE;
}

void
Blt_TreeAddTag(
    TreeClient *clientPtr,
    Node *nodePtr,
    CONST char *tagName)
{
    Blt_TreeTagEntry *tePtr;

    if ((strcmp(tagName, "all") == 0) || (strcmp(tagName, "root") == 0)) {
	return;
    }
    tePtr = Blt_TreeRememberTag(clientPtr, tagName);
    if (nodePtr != NULL) {
	Blt_HashEntry *hPtr;
	int isNew;

	hPtr = Blt_CreateHashEntry(&tePtr->nodeTable, (char *)nodePtr, &isNew);
	assert(hPtr);
	if (isNew) {
	    Blt_SetHashValue(hPtr, nodePtr);
	}
    }
}

void
Blt_TreeRemoveTag(
    TreeClient *clientPtr,
    Node *nodePtr,
    CONST char *tagName)
{
    Blt_HashEntry *hPtr;
    Blt_TreeTagEntry *tePtr;
    
    if (strcmp(tagName, "all") == 0) {
	return;			/* Can't remove tag "all". */
    }
    if ((strcmp(tagName, "root") == 0) && 
	(nodePtr == Blt_TreeRootNode(clientPtr))) {
	return;			/* Can't remove tag "root" from root node. */
    }
    hPtr = Blt_FindHashEntry(&clientPtr->tagTablePtr->tagTable, tagName);
    if (hPtr == NULL) {
	return;			/* No such tag. */
    }
    tePtr = Blt_GetHashValue(hPtr);
    hPtr = Blt_FindHashEntry(&tePtr->nodeTable, (char *)nodePtr);
    if (hPtr == NULL) {
	return;			/* Node isn't tagged. */
    }
    Blt_DeleteHashEntry(&tePtr->nodeTable, hPtr);
    return;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_TreeTagHashTable --
 *
 *---------------------------------------------------------------------- 
 */
Blt_HashTable *
Blt_TreeTagHashTable(TreeClient *clientPtr, CONST char *tagName)
{
    Blt_HashEntry *hPtr;
   
    hPtr = Blt_FindHashEntry(&clientPtr->tagTablePtr->tagTable, tagName);
    if (hPtr != NULL) {
	Blt_TreeTagEntry *tePtr;
	
	tePtr = Blt_GetHashValue(hPtr);
	return &tePtr->nodeTable;
    }
    return NULL;
}

Blt_HashEntry *
Blt_TreeFirstTag(TreeClient *clientPtr, Blt_HashSearch *cursorPtr)
{
    return Blt_FirstHashEntry(&clientPtr->tagTablePtr->tagTable, cursorPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_TreeDumpNode --
 *
 *---------------------------------------------------------------------- 
 */
void
Blt_TreeDumpNode(
    TreeClient *clientPtr,
    Node *rootPtr,		/* Root node of subtree. */
    Node *nodePtr, 
    Tcl_DString *dsPtr)
{
    if (nodePtr == rootPtr) {
	Tcl_DStringAppendElement(dsPtr, "-1");
    } else {
	Tcl_DStringAppendElement(dsPtr, 
		Blt_TreeNodeIdAscii(nodePtr->parent));
    }	
    Tcl_DStringAppendElement(dsPtr, Blt_TreeNodeIdAscii(nodePtr));

    Tcl_DStringStartSublist(dsPtr);
    Blt_TreeNodeRelativePath(rootPtr, nodePtr, NULL, TREE_INCLUDE_ROOT, dsPtr);
    Tcl_DStringEndSublist(dsPtr);

    Tcl_DStringStartSublist(dsPtr);
    {
	Blt_TreeKeySearch keyIter;
	Blt_TreeKey key;

	/* Add list of key-value pairs. */
	for (key = Blt_TreeFirstKey(clientPtr, nodePtr, &keyIter); key != NULL; 
	     key = Blt_TreeNextKey(clientPtr, &keyIter)) {
	    Tcl_Obj *valueObjPtr;

	    if (Blt_TreeGetValueByKey((Tcl_Interp *)NULL, clientPtr, nodePtr, 
			key, &valueObjPtr) == TCL_OK) {
		Tcl_DStringAppendElement(dsPtr, key);
		Tcl_DStringAppendElement(dsPtr, Tcl_GetString(valueObjPtr));
	    }
	}	    
    }
    Tcl_DStringEndSublist(dsPtr);
    Tcl_DStringStartSublist(dsPtr);
    {
	Blt_HashEntry *hPtr;
	Blt_HashSearch cursor;

	/* Add list of tags. */
	for (hPtr = Blt_TreeFirstTag(clientPtr, &cursor); hPtr != NULL; 
	     hPtr = Blt_NextHashEntry(&cursor)) {
	    Blt_TreeTagEntry *tePtr;

	    tePtr = Blt_GetHashValue(hPtr);
	    if (Blt_FindHashEntry(&tePtr->nodeTable, (char *)nodePtr) != NULL) {
		Tcl_DStringAppendElement(dsPtr, tePtr->tagName);
	    }
	}
    }
    Tcl_DStringEndSublist(dsPtr);
    Tcl_DStringAppend(dsPtr, "\n", -1);
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_TreeDump --
 *
 *	Dumps node information recursively from the given tree based
 *	starting at *rootPtr*. The dump information is written to the
 *	Tcl dynamic string provided. It the caller's responsibility to
 * 	initialize and free the dynamic string.
 *	
 * Results:
 *	Always returns TCL_OK.
 *
 * Side Effects:
 *	Dump information is written to the dynamic string provided.
 *
 *---------------------------------------------------------------------- 
 */
int
Blt_TreeDump(
    TreeClient *clientPtr,
    Node *rootPtr,		/* Root node of sub-tree. */
    Tcl_DString *dsPtr)
{
    Node *np;

    for (np = rootPtr; np != NULL; np = Blt_TreeNextNode(rootPtr, np)) {
	Blt_TreeDumpNode(clientPtr, rootPtr, np, dsPtr);
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_TreeDumpToFile --
 *
 *	Dumps node information recursively from the given tree based
 *	starting at *rootPtr*. The dump information is written to the
 *	file named. If the file name starts with an '@', then it is
 *	the name of an already opened channel to be used. 
 *	
 * Results:
 *	A standard Tcl result.  If the dump was successful, TCL_OK
 *	is returned.  Otherwise, TCL_ERROR is returned and an error
 *	message is left in the interpreter result.
 *
 * Side Effects:
 *	Dump information is written to the named file.
 *
 *---------------------------------------------------------------------- 
 */
int
Blt_TreeDumpToFile(
    Tcl_Interp *interp,
    TreeClient *clientPtr,
    Node *rootPtr,		/* Root node of subtree. */
    char *fileName)
{
    Tcl_Channel channel;
    Node *np;
    Tcl_DString ds;
    int closeChannel;
    
    closeChannel = TRUE;
    if ((fileName[0] == '@') && (fileName[1] != '\0')) {
	int mode;
	
	channel = Tcl_GetChannel(interp, fileName+1, &mode);
	if (channel == NULL) {
	    return TCL_ERROR;
	}
	if ((mode & TCL_WRITABLE) == 0) {
	    Tcl_AppendResult(interp, "channel \"", fileName, 
		"\" not opened for writing", (char *)NULL);
	    return TCL_ERROR;
	}
	closeChannel = FALSE;
    } else {
	channel = Tcl_OpenFileChannel(interp, fileName, "w", 0666);
	if (channel == NULL) {
	    return TCL_ERROR;
	}
    }
    Tcl_DStringInit(&ds);
    for (np = rootPtr; np != NULL; np = Blt_TreeNextNode(rootPtr, np)) {
	int nWritten, length;

	Tcl_DStringSetLength(&ds, 0);
	Blt_TreeDumpNode(clientPtr, rootPtr, np, &ds);
	length = Tcl_DStringLength(&ds);
#if HAVE_UTF
	nWritten = Tcl_WriteChars(channel, Tcl_DStringValue(&ds), length);
#else
	nWritten = Tcl_Write(channel, Tcl_DStringValue(&ds), length);
#endif
	if (nWritten < 0) {
	    Tcl_AppendResult(interp, fileName, ": write error:", 
			     Tcl_PosixError(interp), (char *)NULL);
	    Tcl_DStringFree(&ds);
	    if (closeChannel) {
		Tcl_Close(interp, channel);
	    }
	    return TCL_ERROR;
	}
    }
    Tcl_DStringFree(&ds);
    if (closeChannel) {
	Tcl_Close(interp, channel);
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_TreeRestoreFromFile --
 *
 *	Restores nodes to the given tree based upon the dump file
 *	provided. The dump file should have been generated by
 *	Blt_TreeDump or Blt_TreeDumpToFile.  If the file name starts
 *	with an '@', then it is the name of an already opened channel
 *	to be used. Nodes are added relative to the node *rootPtr* as
 *	the root of the sub-tree.  Two bit flags may be set.
 *	
 *	TREE_RESTORE_NO_TAGS	Don't restore tag information.
 *	TREE_RESTORE_OVERWRITE	Look for nodes with the same label.
 *				Overwrite if necessary.
 *
 * Results:
 *	A standard Tcl result.  If the restore was successful, TCL_OK
 *	is returned.  Otherwise, TCL_ERROR is returned and an error
 *	message is left in the interpreter result.
 *
 * Side Effects:
 *	New nodes are created in the tree and may possibly generate
 *	notify callbacks.
 *
 *---------------------------------------------------------------------- 
 */
int
Blt_TreeRestoreFromFile(
    Tcl_Interp *interp,
    TreeClient *clientPtr,
    Node *rootPtr,		/* Root node of branch to be restored. */
    char *fileName,
    unsigned int flags)
{
    Tcl_Channel channel;
    RestoreInfo restore;
    int argc;
    char **argv;
    int closeChannel;
    int result;

    closeChannel = TRUE;
    if ((fileName[0] == '@') && (fileName[1] != '\0')) {
	int mode;
	
	channel = Tcl_GetChannel(interp, fileName+1, &mode);
	if (channel == NULL) {
	    return TCL_ERROR;
	}
	if ((mode & TCL_READABLE) == 0) {
	    Tcl_AppendResult(interp, "channel \"", fileName, 
		"\" not opened for reading", (char *)NULL);
	    return TCL_ERROR;
	}
	closeChannel = FALSE;
    } else {
	channel = Tcl_OpenFileChannel(interp, fileName, "r", 0);
	if (channel == NULL) {
	    return TCL_ERROR;	/* Can't open dump file. */
	}
    }
    memset((char *)&restore, 0, sizeof(restore));
    Blt_InitHashTable(&restore.idTable, BLT_ONE_WORD_KEYS);
    restore.rootPtr = rootPtr;
    restore.flags = flags;

    argv = NULL;
    result = TCL_ERROR;		
    for (;;) {
	result = ReadDumpRecord(interp, channel, &argc, &argv, &restore);
	if (result != TCL_OK) {
	    break;		/* Found error or EOF */
	}
	if (argc == 0) {
	    result = TCL_OK; /* Do nothing. */
	} else if (argc == 3) {
	    result = RestoreNode3(interp, clientPtr, argc, argv, &restore);
	} else if ((argc == 5) || (argc == 6)) {
	    result = RestoreNode5(interp, clientPtr, argc, argv, &restore);
	} else {
	    Tcl_AppendResult(interp, "line #", Blt_Itoa(restore.nLines), 
		": wrong # elements in restore entry", (char *)NULL);
	    result = TCL_ERROR;
	}
	Blt_Free(argv);
	if (result != TCL_OK) {
	    break;
	}
    } 
    if (closeChannel) {
	Tcl_Close(interp, channel);
    }
    Blt_DeleteHashTable(&restore.idTable);
    if (result == TCL_ERROR) {
	return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_TreeRestore --
 *
 *	Restores nodes to the given tree based upon the dump string.
 *	The dump string should have been generated by Blt_TreeDump.
 *	Nodes are added relative to the node *rootPtr* as the root of
 *	the sub-tree.  Two bit flags may be set.
 *	
 *	TREE_RESTORE_NO_TAGS	Don't restore tag information.
 *	TREE_RESTORE_OVERWRITE	Look for nodes with the same label.
 *				Overwrite if necessary.
 *
 * Results:
 *	A standard Tcl result.  If the restore was successful, TCL_OK
 *	is returned.  Otherwise, TCL_ERROR is returned and an error
 *	message is left in the interpreter result.
 *
 * Side Effects:
 *	New nodes are created in the tree and may possibly generate
 *	notify callbacks.
 *
 *---------------------------------------------------------------------- 
 */
int
Blt_TreeRestore(
    Tcl_Interp *interp,
    TreeClient *clientPtr,
    Node *rootPtr,		/* Root node of branch to be restored. */
    char *string,
    unsigned int flags)
{
    RestoreInfo restore;
    int result;
	
    memset((char *)&restore, 0, sizeof(restore));
    Blt_InitHashTable(&restore.idTable, BLT_ONE_WORD_KEYS);
    restore.rootPtr = rootPtr;
    restore.flags = flags;
    result = TCL_ERROR;
    for (;;) {
	char **argv;
	int argc;

	result = ParseDumpRecord(interp, &string, &argc, &argv, &restore);
	if (result != TCL_OK) {
	    break;		/* Found error or EOF */
	}
	if (argc == 0) {
	    result = TCL_OK; /* Do nothing. */
	} else if (argc == 3) {
	    result = RestoreNode3(interp, clientPtr, argc, argv, &restore);
	} else if ((argc == 5) || (argc == 6)) {
	    result = RestoreNode5(interp, clientPtr, argc, argv, &restore);
	} else {
	    Tcl_AppendResult(interp, "line #", Blt_Itoa(restore.nLines), 
		": wrong # elements in restore entry", (char *)NULL);
	    result = TCL_ERROR;
	}
	Blt_Free(argv);
	if (result != TCL_OK) {
	    break;
	}
    } 
    Blt_DeleteHashTable(&restore.idTable);

    /* result will be TCL_RETURN if successful, TCL_ERROR otherwise. */
    if (result == TCL_ERROR) {
	return TCL_ERROR;
    }
    return TCL_OK;
}

#ifdef HAVE_LIBEXPAT

#ifdef HAVE_EXPAT_H
#include <expat.h>
#endif /* HAVE_EXPAT_H */

typedef struct {
    Blt_Tree tree;
    Blt_TreeNode parent;
    Tcl_Interp *interp;
    int flags;
} ImportData;

static void
GetXmlCharacterData(void *userData, const XML_Char *string, int length) 
{
    ImportData *dataPtr = userData;
    Tcl_Obj *objPtr;
    
    objPtr = Tcl_NewStringObj(string, length);
    Blt_TreeSetValue(dataPtr->interp, dataPtr->tree, dataPtr->parent, 
	"Character Data", objPtr);
}

static void
StartXmlTag(void *userData, const char *element, const char **attr) 
{
    Blt_TreeNode node;
    ImportData *dataPtr = userData;
    const char **p;

    node = NULL;
    if (dataPtr->flags & TREE_RESTORE_OVERWRITE) {
	node = Blt_TreeFindChild(dataPtr->parent, element);
    }
    if (node == NULL) {
	node = Blt_TreeCreateNode(dataPtr->tree, dataPtr->parent, element, -1);
    }
    for (p = attr; *p != NULL; p += 2) {
	Tcl_Obj *objPtr;

	objPtr = Tcl_NewStringObj(*(p+1), strlen(*(p+1)));
	Blt_TreeSetValue(dataPtr->interp, dataPtr->tree, node, *p, objPtr);
    }
    dataPtr->parent = node;	/* Increase depth.  */
}

static void
EndXmlTag(void *userData, const char *element) 
{
    ImportData *dataPtr = userData;

    dataPtr->parent = Blt_TreeNodeParent(dataPtr->parent);
}

static int
ReadXmlFromFile(
    Tcl_Interp *interp, 
    XML_Parser parser, 
    const char *fileName)
{
    int closeChannel;
    int done;
    Tcl_Channel channel;

    closeChannel = TRUE;
    if ((fileName[0] == '@') && (fileName[1] != '\0')) {
	int mode;
	
	channel = Tcl_GetChannel(interp, fileName+1, &mode);
	if (channel == NULL) {
	    return FALSE;
	}
	if ((mode & TCL_READABLE) == 0) {
	    Tcl_AppendResult(interp, "channel \"", fileName, 
		"\" not opened for reading", (char *)NULL);
	    return FALSE;
	}
	closeChannel = FALSE;
    } else {
	channel = Tcl_OpenFileChannel(interp, fileName, "r", 0);
	if (channel == NULL) {
	    return FALSE;	/* Can't open dump file. */
	}
    }
    done = FALSE;
    while (!done) {
	int length;
#define BUFFSIZE	8192
	char buffer[BUFFSIZE];
	
	length = Tcl_Read(channel, buffer, sizeof(char) * BUFFSIZE);
	if (length < 0) {
	    Tcl_AppendResult(interp, "\nread error: ", Tcl_PosixError(interp),
			     (char *)NULL);
	    if (closeChannel) {
		Tcl_Close(interp, channel);
	    }
	    return FALSE;
	}
	done = Tcl_Eof(channel);
	if (!XML_Parse(parser, buffer, length, done)) {
	    Tcl_AppendResult(interp, "\n", fileName, ":",
			Blt_Itoa(XML_GetCurrentLineNumber(parser)), ": ",
			"error: ", 
			XML_ErrorString(XML_GetErrorCode(parser)), 
			(char *)NULL);
	    if (closeChannel) {
		Tcl_Close(interp, channel);
	    }
	    return FALSE;
	}
    }
    if (closeChannel) {
	Tcl_Close(interp, channel);
    }
    return TRUE;
}


static int
GetXmlExternalEntityRef(
    XML_Parser parser, 
    const XML_Char *context,
    const XML_Char *base,
    const XML_Char *systemId,
    const XML_Char *publicId)
{
    ImportData *dataPtr;
    Tcl_DString dString;
    Tcl_Interp *interp;
    XML_Parser newParser;
    int result;

    dataPtr = XML_GetUserData(parser);
    assert(dataPtr != NULL);
    interp = dataPtr->interp;
    Tcl_DStringInit(&dString);
    if ((base != NULL) && (Tcl_GetPathType(systemId) == TCL_PATH_RELATIVE)) {
	char **argv;
	char **baseArr, **sysIdArr;
	int argc;
	int i, j;
	int nBase, nSysId;

	Tcl_SplitPath(base, &nBase, &baseArr);
	Tcl_SplitPath(systemId, &nSysId, &sysIdArr);
	argc = nBase + nSysId;
	argv = Blt_Malloc(sizeof(char *) * (argc + 1));
	if (argv == NULL) {
	    return FALSE;
	}
	for (i = 0; i < nBase; i++) {
	    argv[i] = baseArr[i];
	}
	for (j = 0; j < nSysId; j++, i++) {
	    argv[i] = sysIdArr[j];
	}
	argv[i] = NULL;
	Tcl_JoinPath(argc, argv, &dString);
	Blt_Free(baseArr);
	Blt_Free(sysIdArr);
	Blt_Free(argv);
    } else {
	Tcl_DStringAppend(&dString, systemId, -1);
    }
    newParser = XML_ExternalEntityParserCreate(parser, context, NULL);
    if (newParser == NULL) {
	Tcl_AppendResult(interp, "can't create external entity ref parser", 
			 (char *)NULL);
	return FALSE;
    }
    result = ReadXmlFromFile(interp, newParser, Tcl_DStringValue(&dString));
    Tcl_DStringFree(&dString);
    XML_ParserFree(newParser);
    return result;
}

int
Blt_TreeImportFile(
    Tcl_Interp *interp, 
    Blt_Tree tree, 
    Blt_TreeNode parent, 
    Tcl_Obj *objPtr,
    unsigned int flags) 
{
    ImportData data;
    XML_Parser parser;
    int result;
    char *fileName;

    parser = XML_ParserCreate(NULL);
    if (parser == NULL) {
	Tcl_AppendResult(interp, "can't create XML parser", (char *)NULL);
	return TCL_ERROR;
    }
    data.tree = tree;
    data.parent = parent;
    data.interp = interp;
    data.flags = flags;
    XML_SetUserData(parser, &data);

    fileName = Tcl_GetString(objPtr);
    {
	Tcl_DString dString;
	int argc;
	char **argv;

	Tcl_DStringInit(&dString);
	Tcl_SplitPath(fileName, &argc, &argv);
	Tcl_JoinPath(argc - 1, argv, &dString);
	XML_SetBase(parser, Tcl_DStringValue(&dString));
	Blt_Free(argv);
	Tcl_DStringFree(&dString);
    }
    XML_SetElementHandler(parser, StartXmlTag, EndXmlTag);
    XML_SetCharacterDataHandler(parser, GetXmlCharacterData);
    XML_SetExternalEntityRefHandler(parser, GetXmlExternalEntityRef);
    result = ReadXmlFromFile(interp, parser, fileName);
    XML_ParserFree(parser);
    return (result) ? TCL_OK : TCL_ERROR;
} 


int
Blt_TreeImportData(
    Tcl_Interp *interp, 
    Blt_Tree tree, 
    Blt_TreeNode parent, 
    Tcl_Obj *dataObjPtr,
    unsigned int flags) 
{
    ImportData data;
    XML_Parser parser;
    char *string;
    int length;

    data.tree = tree;
    data.parent = parent;
    data.interp = interp;
    data.flags = flags;

    parser = XML_ParserCreate(NULL);
    if (parser == NULL) {
	Tcl_AppendResult(interp, "can't create parser", (char *)NULL);
	return TCL_ERROR;
    }
    XML_SetUserData(parser, &data);
    XML_SetElementHandler(parser, StartXmlTag, EndXmlTag);
    XML_SetCharacterDataHandler(parser, GetXmlCharacterData);
    string = Tcl_GetStringFromObj(dataObjPtr, &length);
    if (!XML_Parse(parser, string, length, 1)) {
	Tcl_AppendResult(interp, "\nparse error at line ",
			 Blt_Itoa(XML_GetCurrentLineNumber(parser)), ":  ",
			 XML_ErrorString(XML_GetErrorCode(parser)),
			 (char *)NULL);
	XML_ParserFree(parser);
	return TCL_ERROR;
    }
    XML_ParserFree(parser);
    return TCL_OK;
} 

#endif /* HAVE_LIBEXPAT */

#endif /* NO_TREE */
