
/*
 * bltUnixPainter.c --
 *
 * This module implements X11-specific image processing procedures for
 * the BLT toolkit.
 *
 *	Copyright 1998-2004 George A Howlett.
 *
 *	Permission is hereby granted, free of charge, to any person
 *	obtaining a copy of this software and associated documentation
 *	files (the "Software"), to deal in the Software without
 *	restriction, including without limitation the rights to use,
 *	copy, modify, merge, publish, distribute, sublicense, and/or
 *	sell copies of the Software, and to permit persons to whom the
 *	Software is furnished to do so, subject to the following
 *	conditions:
 *
 *	The above copyright notice and this permission notice shall be
 *	included in all copies or substantial portions of the
 *	Software.
 *
 *	THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY
 *	KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 *	WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 *	PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS
 *	OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
 *	OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 *	OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 *	SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * The color allocation routines are adapted from tkImgPhoto.c of the
 * Tk library distrubution.  The photo image type was designed and
 * implemented by Paul Mackerras.
 *
 *	Copyright (c) 1987-1993 The Regents of the University of
 *	California.
 *
 *	Copyright (c) 19941998 Sun Microsystems, Inc.
 * 
 */

#include "bltInt.h"
#include "bltHash.h"
#include "bltPicture.h"
#include "bltPainterInt.h"
#include "bltPainter.h"
#include <X11/Xutil.h>
#include <X11/Xproto.h>

typedef struct Blt_PictureStruct Picture;

#define MAXIMAGESIZE(dpy)	(XMaxRequestSize(dpy) << 2) - 24

/*
 *----------------------------------------------------------------------
 *
 * PaintXImage --
 *
 *	Draw the given XImage. If the size of the image exceeds the
 *	maximum request size of the X11 protocol, the image is drawn
 *	using XPutImage in multiples of rows that fit within the
 *	limit.
 *
 *----------------------------------------------------------------------
 */
static void
PaintXImage(
    Painter *p,
    Drawable drawable, 
    XImage *imgPtr, 
    int srcX, int srcY,
    int width, int height,
    int destX, int destY)
{
    int y;
    int nLines;
    long maxPixels;

    maxPixels = Blt_MaxRequestSize(p->display, sizeof(Pix32));
    nLines = (maxPixels + width - 1) / width;
    if (nLines < 1) {
	nLines = 1;
    } 
    if (nLines > height ) {
	nLines = height;
    }
    for (y = 0; y < height; y += nLines) {
	if ((y + nLines) > height) {
	    nLines = height - y;
	}
	XPutImage(p->display, drawable, p->gc, imgPtr, srcX, srcY + y, destX, 
		destY + y, width, nLines);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * XGetImageErrorProc --
 *
 *	Error handling routine for the XGetImage request below. Sets
 *	the flag passed via *clientData* to TCL_ERROR indicating an
 *	error occurred.
 *
 *----------------------------------------------------------------------
 */
/* ARGSUSED */
static int
XGetImageErrorProc(
    ClientData clientData, 
    XErrorEvent *errEventPtr)	/* Not used. */
{
    int *errorPtr = clientData;

    *errorPtr = TCL_ERROR;
    return 0;
}

/*
 *----------------------------------------------------------------------
 *
 * DrawableToXImage --
 *
 *	Attempts to snap the image from the drawable into an XImage
 *	structure (using XGetImage).  This may fail is the coordinates
 *	of the region in the drawable are obscured.
 *
 * Results:
 *	Returns a pointer to the XImage if successful. Otherwise NULL
 *	is returned.
 *
 *----------------------------------------------------------------------
 */
static XImage *
DrawableToXImage(
    Display *display,
    Drawable drawable, 
    int x, int y, int width, int height)
{
    XImage *imgPtr;
    Tk_ErrorHandler errHandler;
    int result;

    result = TCL_OK;
    errHandler = Tk_CreateErrorHandler(display, BadMatch, X_GetImage, -1, 
	XGetImageErrorProc, &result);
    imgPtr = XGetImage(display, drawable, x, y, width, height, AllPlanes, 
	ZPixmap);
    Tk_DeleteErrorHandler(errHandler);
    XSync(display, False);
    if (result != TCL_OK) {
#ifdef notdef
	fprintf(stderr, "can't snap picture\n");
#endif
	return NULL;
    }
    return imgPtr;
}

/*
 *----------------------------------------------------------------------
 *
 * DrawableToPicture --
 *
 *      Takes a snapshot of an X drawable (pixmap or window) and
 *	converts it to a picture.
 *
 * Results:
 *      Returns a picture of the drawable.  If an error occurred (a
 *      portion of the region specified is obscured), then NULL is
 *      returned.
 *
 *----------------------------------------------------------------------
 */
static Picture *
DrawableToPicture(
    Painter *p,
    Drawable drawable,
    int x, int y,		/* Offset of image from the drawable's
				 * origin. */
    int width, int height)	/* Dimension of the image.  Image must
				 * be completely contained by the
				 * drawable. */
{
    Picture *destPtr;
    Pix32 *destRowPtr;
    Pix32 palette[256];
    XImage *imgPtr;
    int shift[4];
    unsigned char *srcRowPtr;

    imgPtr = DrawableToXImage(p->display, drawable, x, y, width, height);
    if (imgPtr == NULL) {
	int drawX, drawY, drawWidth, drawHeight;

	if (Blt_GetWindowRegion(p->display, drawable, &drawX, &drawY, 
		&drawWidth, &drawHeight) == TCL_OK) {
	    if ((width > drawWidth) || (height > drawHeight)) {
		width = MIN(drawWidth, width);
		height = MIN(drawHeight, height);
		imgPtr = DrawableToXImage(p->display, drawable, x, y, width, 
			height);
	    }
	}
    }
    if (imgPtr == NULL) {
	return NULL;
    }

    /* Allocate a picture to hold the screen snapshot. */
    destPtr = Blt_CreatePicture(width, height);

    /* Get the palette of the current painter/window */
    Blt_QueryPalette(p, palette);

    switch (p->visualPtr->class) {
    case TrueColor:
    case DirectColor:
	if (imgPtr->byte_order == MSBFirst) {
	    shift[0] = 24, shift[1] = 16, shift[2] = 8, shift[3] = 0;
	} else {
	    switch (imgPtr->bits_per_pixel) {
	    case 32:
		shift[0] = 0, shift[1] = 8, shift[2] = 16, shift[3] = 24;
		break;
	    case 24:
		shift[1] = 0, shift[2] = 8, shift[3] = 16;
		break;
	    case 16:
		shift[2] = 0, shift[3] = 8;
		break;
	    case 8:
		shift[3] = 0;
		break;
	    }
	}
	srcRowPtr = imgPtr->data;
	destRowPtr = destPtr->bits;
	for (y = 0; y < height; y++) {
	    unsigned char *sp;
	    Pix32 *dp;

	    sp = srcRowPtr, dp = destRowPtr;

	    switch (imgPtr->bits_per_pixel) {
	    case 32:
		for (x = 0; x < width; x++) {
		    int r, g, b;
		    unsigned long pixel;
		    
		    /* Get the next pixel from the image. */
		    pixel = ((sp[0] << shift[0]) | (sp[1] << shift[1]) |
			     (sp[2] << shift[2]) | (sp[3] << shift[3]));

		    /* Convert the pixel to RGB, correcting for input gamma. */
		    r = ((pixel & p->rMask) >> p->rShift);
		    g = ((pixel & p->gMask) >> p->gShift);
		    b = ((pixel & p->bMask) >> p->bShift);
		    dp->Red = palette[r].Red;
		    dp->Green = palette[g].Green;
		    dp->Blue = palette[b].Blue;
		    dp->Alpha = ALPHA_OPAQUE;
		    dp++, sp += 4;
		}
		break;

	    case 24:
		for (x = 0; x < width; x++) {
		    int r, g, b;
		    unsigned long pixel;
		    
		    /* Get the next pixel from the image. */
		    pixel = ((sp[0] << shift[1]) | (sp[1] << shift[2]) |
			     (sp[2] << shift[3]));

		    /* Convert the pixel to RGB, correcting for input gamma. */
		    r = ((pixel & p->rMask) >> p->rShift);
		    g = ((pixel & p->gMask) >> p->gShift);
		    b = ((pixel & p->bMask) >> p->bShift);
		    dp->Red = palette[r].Red;
		    dp->Green = palette[g].Green;
		    dp->Blue = palette[b].Blue;
		    dp->Alpha = ALPHA_OPAQUE;
		    dp++, sp += 3;
		}
		break;
		
	    case 16:
		for (x = 0; x < width; x++) {
		    int r, g, b;
		    unsigned long pixel;
		    
		    /* Get the next pixel from the image. */
		    pixel = ((sp[0] << shift[2]) | (sp[1] << shift[3]));

		    /* Convert the pixel to RGB, correcting for input gamma. */
		    r = ((pixel & p->rMask) >> p->rShift);
		    g = ((pixel & p->gMask) >> p->gShift);
		    b = ((pixel & p->bMask) >> p->bShift);
		    dp->Red = palette[r].Red;
		    dp->Green = palette[g].Green;
		    dp->Blue = palette[b].Blue;
		    dp->Alpha = ALPHA_OPAQUE;
		    dp++, sp += 2;
		}
		break;

	    case 8:
		for (x = 0; x < width; x++) {
		    int r, g, b;
		    unsigned long pixel;
		    
		    /* Get the next pixel from the image. */
		    pixel = (*sp++ << shift[3]);

		    /* Convert the pixel to RGB, correcting for input gamma. */
		    r = ((pixel & p->rMask) >> p->rShift);
		    g = ((pixel & p->gMask) >> p->gShift);
		    b = ((pixel & p->bMask) >> p->bShift);
		    dp->Red = palette[r].Red;
		    dp->Green = palette[g].Green;
		    dp->Blue = palette[b].Blue;
		    dp->Alpha = ALPHA_OPAQUE;
		    dp++;
		}
		break;
	    }
	    destRowPtr += destPtr->pixelsPerRow;
	    srcRowPtr += imgPtr->bytes_per_line;
	}
	break;

    case PseudoColor:
    case StaticColor:
    case GrayScale:
    case StaticGray:
	if ((imgPtr->bits_per_pixel != 8) && (imgPtr->bits_per_pixel != 4)) {
	    return NULL;	/* Can only handle 4 or 8 bit pixels.  */
	}
	srcRowPtr = imgPtr->data;
	destRowPtr = destPtr->bits;
	for (y = 0; y < height; y++) {
	    unsigned char *sp;
	    Pix32 *dp;

	    sp = srcRowPtr, dp = destRowPtr;
	    for (x = 0; x < width; x++) {
		unsigned long pixel;

		if (imgPtr->bits_per_pixel == 8) {
		    pixel = *sp++;
		} else {
		    if (x & 1) { /* Odd: pixel is high nybble. */
			pixel = (*sp & 0xF0) >> 4;
			sp++;
		    } else {	/* Even: pixel is low nybble. */
			pixel = (*sp & 0x0F);
		    }
		} 
		/* Convert the pixel to RGB, correcting for input gamma. */
		dp->Red = palette[pixel].Red;
		dp->Green = palette[pixel].Green;
		dp->Blue = palette[pixel].Blue;
		dp->Alpha = ALPHA_OPAQUE;
		dp++;
	    }
	    srcRowPtr += imgPtr->bytes_per_line;
	    destRowPtr += destPtr->pixelsPerRow;
	}
	break;
    default:
	break;
    }
    XDestroyImage(imgPtr);
    return destPtr;
}


/*
 *----------------------------------------------------------------------
 *
 * PaintPicture --
 *
 *	Paints the picture to the given drawable. The region of
 *	the picture is specified and the coordinates where in the 
 *	destination drawable is the image to be displayed.
 *
 *	The image may be dithered depending upon the bit set in
 *	the flags parameter: 0 no dithering, 1 for dithering.
 * 
 * Results:
 *	Returns TRUE is the picture was successfully display,
 *	Otherwise FALSE is returned if the particular combination
 *	visual and image depth is not handled.
 *
 *----------------------------------------------------------------------
 */
static int
PaintPicture(
    Painter *p,
    Drawable drawable,
    Picture *srcPtr,
    int srcX, int srcY,		/* Coordinates of region in the
				 * picture. */
    int width, int height,	/* Dimension of the region.  Area
				 * cannot extend beyond the end of the
				 * picture. */
    int destX, int destY,	/* Coordinates of region in the
				 * drawable.  */
    unsigned int flags)
{
#ifdef WORD_BIGENDIAN
    static int nativeByteOrder = MSBFirst;
#else
    static int nativeByteOrder = LSBFirst;
#endif
    Picture *ditherPtr;
    Pix32 *srcRowPtr;
    XImage *imgPtr;
    int x, y;
    unsigned char *destRowPtr;

    ditherPtr = NULL;
    if (flags & BLT_PAINTER_DITHER) {
	ditherPtr = Blt_DitherPicture(srcPtr, p->palette);
	if (ditherPtr != NULL) {
	    srcPtr = ditherPtr;
	}
    }
    imgPtr = XCreateImage(p->display, p->visualPtr, p->depth, ZPixmap, 0, 
	(char *)NULL, width, height, 32, 0);
    assert(imgPtr);

    /* 
     * Set the byte order to the platform's native byte order. We'll
     * let Xlib handle byte swapping.
     */
    imgPtr->byte_order = nativeByteOrder;
    imgPtr->data = Blt_Malloc(sizeof(Pix32) * width * height);
    assert(imgPtr->data);

    switch (p->visualPtr->class) {
    case TrueColor:

	/* Directly compute the pixel 8, 16, 24, or 32 bit values from
	 * the RGB components. */

	srcRowPtr = srcPtr->bits + ((srcY * srcPtr->pixelsPerRow) + srcX);
	destRowPtr = imgPtr->data;
	for (y = 0; y < height; y++) {
	    Pix32 *sp;
	    unsigned char *dp;

	    sp = srcRowPtr, dp = destRowPtr;
	    switch (imgPtr->bits_per_pixel) {
	    case 32:
		for (x = 0; x < width; ++x) {
		    unsigned int r, g, b;
		    
		    r = (p->igammaTable[sp->Red] >> p->rAdjust) << p->rShift;
		    g = (p->igammaTable[sp->Green] >> p->gAdjust) << p->gShift;
		    b = (p->igammaTable[sp->Blue] >> p->bAdjust) << p->bShift;
		    sp++;
		    *(unsigned int *)dp = r | g | b;
		    dp += 4;
		}
		break;

	    case 24:
		for (x = 0; x < width; x++) {
		    unsigned long pixel;
		    unsigned int r, g, b;
		    
		    r = (p->igammaTable[sp->Red] >> p->rAdjust) << p->rShift;
		    g = (p->igammaTable[sp->Green] >> p->gAdjust) << p->gShift;
		    b = (p->igammaTable[sp->Blue] >> p->bAdjust) << p->bShift;
		    sp++;
		    pixel = r | g | b;
		    
		    *dp++ = pixel & 0xFF;
		    *dp++ = (pixel >> 8) & 0xFF;
		    *dp++ = (pixel >> 16) & 0xFF;
		}
		break;

	    case 16:
		for (x = 0; x < width; ++x) {
		    unsigned long pixel;
		    unsigned int r, g, b;
		    
		    r = (p->igammaTable[sp->Red] >> p->rAdjust) << p->rShift;
		    g = (p->igammaTable[sp->Green] >> p->gAdjust) << p->gShift;
		    b = (p->igammaTable[sp->Blue] >> p->bAdjust) << p->bShift;
		    sp++;
		    pixel = r | g | b;
		    *(unsigned short *)dp = pixel;
		    dp += 2;
		}
		break;

	    case 8:
		for (x = 0; x < width; ++x) {
		    unsigned long pixel;
		    unsigned int r, g, b;
		    
		    r = (p->igammaTable[sp->Red] >> p->rAdjust) << p->rShift;
		    g = (p->igammaTable[sp->Green] >> p->gAdjust) << p->gShift;
		    b = (p->igammaTable[sp->Blue] >> p->bAdjust) << p->bShift;
		    sp++;
		    pixel = r | g | b;

		    *dp++ = pixel & 0xFF;
		}
		break;
	    }
	    destRowPtr += imgPtr->bytes_per_line;
	    srcRowPtr += srcPtr->pixelsPerRow;
	}
	break;

    case DirectColor:

	/* Translate the RGB components to 8, 16, 24, or 32-bit pixel
	 * values. */

	srcRowPtr = srcPtr->bits + ((srcY * srcPtr->pixelsPerRow) + srcX);
	destRowPtr = imgPtr->data;
	for (y = 0; y < height; y++) {
	    Pix32 *sp;
	    unsigned char *dp;

	    sp = srcRowPtr, dp = destRowPtr;
	    switch (imgPtr->bits_per_pixel) {
	    case 32:
		for (x = 0; x < width; ++x) {
		    unsigned long pixel;
		    
		    pixel = (p->rBits[sp->Red] | p->gBits[sp->Green] |
			     p->bBits[sp->Blue]);
		    sp++;

		    *(unsigned int *)dp = pixel;
		    dp += 4;
		}
		break;

	    case 24:
		for (x = 0; x < width; x++) {
		    unsigned long pixel;
		    
		    pixel = (p->rBits[sp->Red] | p->gBits[sp->Green] |
			     p->bBits[sp->Blue]);
		    sp++;
		    
		    *dp++ = pixel & 0xFF;
		    *dp++ = (pixel >> 8) & 0xFF;
		    *dp++ = (pixel >> 16) & 0xFF;
		}
		break;
	    case 16:
		for (x = 0; x < width; ++x) {
		    unsigned long pixel;
		    
		    pixel = (p->rBits[sp->Red] | p->gBits[sp->Green] |
			     p->bBits[sp->Blue]);
		    sp++;
		    
		    *(unsigned short *)dp = pixel;
		    dp += 2;
		}
		break;
	    case 8:
		for (x = 0; x < width; ++x) {
		    unsigned long pixel;
		    
		    pixel = (p->rBits[sp->Red] | p->gBits[sp->Green] |
			     p->bBits[sp->Blue]);
		    sp++;
		    *dp++ = pixel & 0xFF;
		}
		break;
	    }
	    destRowPtr += imgPtr->bytes_per_line;
	    srcRowPtr += srcPtr->pixelsPerRow;
	}
	break;

    case PseudoColor:
    case StaticColor:
    case GrayScale:
    case StaticGray:

	/* Translate RGB components to the correct 8-bit or 4-bit
	 * pixel values. */

	srcRowPtr = srcPtr->bits + ((srcY * srcPtr->pixelsPerRow) + srcX);
	destRowPtr = imgPtr->data;
	if (imgPtr->bits_per_pixel == 8) {
	    for (y = 0; y < height; y++) {
		Pix32 *sp;
		unsigned char *dp;

		dp = destRowPtr, sp = srcRowPtr;
		for (x = 0; x < width; x++) {
		    unsigned long pixel;

		    pixel = (p->rBits[sp->Red] + p->gBits[sp->Green] +
			     p->bBits[sp->Blue]);
		    pixel = p->pixels[pixel];
		    *dp++ = (pixel & 0xFF);
		    sp++;
		}
		destRowPtr += imgPtr->bytes_per_line;
		srcRowPtr += srcPtr->pixelsPerRow;
	    }
	} else {
	    for (y = 0; y < height; y++) {
		Pix32 *sp;
		unsigned char *dp;

		dp = destRowPtr, sp = srcRowPtr;
		for (x = 0; x < width; x++) {
		    unsigned long pixel;

		    pixel = (p->rBits[sp->Red] + p->gBits[sp->Green] +
			     p->bBits[sp->Blue]);
		    pixel = p->pixels[pixel];
		    sp++;
		    if (x & 1) {	
			*dp |= (pixel & 0x0F) << 4;
			/* Move to the next address after odd nybbles. */
			dp++;
		    } else {
			*dp = (pixel & 0x0F);
		    }
		}
		destRowPtr += imgPtr->bytes_per_line;
		srcRowPtr += srcPtr->width;
	    }
	}
	break;

    default:
	return FALSE;
    }
    PaintXImage(p, drawable, imgPtr, 0, 0, width, height, destX, destY);

    if (ditherPtr != NULL) {
	Blt_FreePicture(ditherPtr);
    }
    XDestroyImage(imgPtr);
    return TRUE;
}

/*
 *----------------------------------------------------------------------
 *
 * PaintPictureWithBlend --
 *
 *	Blends and paints the picture with the given drawable. The
 *	region of the picture is specified and the coordinates where
 *	in the destination drawable is the image to be displayed.
 *
 *	The background is snapped from the drawable and converted into
 *	a picture.  This picture is then blended with the current
 *	picture (the background always assumed to be 100% opaque).
 * 
 * Results:
 *	Returns TRUE is the picture was successfully display,
 *	Otherwise FALSE is returned.  This may happen if the
 *	background can not be obtained from the drawable.
 *
 *----------------------------------------------------------------------
 */
static int
PaintPictureWithBlend(
    Painter *p,
    Drawable drawable,
    Blt_Picture fg,
    int x, int y,		/* Coordinates of region in the
				 * picture. */
    int width, int height,	/* Dimension of the region.  Area
				 * cannot extend beyond the end of the
				 * picture. */
    int destX, int destY,	/* Coordinates of region in the
				 * drawable.  */
    unsigned int flags, 
    int alpha)
{
    Blt_Picture bg;

    if (destX < 0) {
	width += destX;
	destX = 0;
    } 
    if (destY < 0) {
	height += destY;
	destY = 0;
    }
    if ((width < 0) || (height < 0)) {
	return FALSE;
    }
    bg = DrawableToPicture(p, drawable, destX, destY, width, height);
    if (bg == NULL) {
	return FALSE;
    }
#ifdef notdef
    Blt_FadePicture(bgPtr, fg, x, y, width, height, 0, 0, alpha);
#else
    Blt_BlendPictureArea(bg, fg, x, y, width, height, 0, 0);
#endif
    PaintPicture(p, drawable, bg, 0, 0, width, height, destX, destY, flags);
    Blt_FreePicture(bg);
    return TRUE;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_DrawableToPicture --
 *
 *      Takes a snapshot of an X drawable (pixmap or window) and
 *	converts it to a picture.
 *
 * Results:
 *      Returns a picture of the drawable.  If an error occurred,
 *	NULL is returned.
 *
 *----------------------------------------------------------------------
 */
Picture *
Blt_DrawableToPicture(
    Tk_Window tkwin,
    Drawable drawable,
    int x, int y,		/* Offset of image from the drawable's
				 * origin. */
    int width, int height,	/* Dimension of the image.  Image must
				 * be completely contained by the
				 * drawable. */
    double gamma)
{
    Blt_Painter painter;
    Blt_Picture picture;

    painter = Blt_GetPainter(tkwin, gamma);
    picture =  DrawableToPicture(painter, drawable, x, y, width, height);
    Blt_FreePainter(painter);
    return picture;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_WindowToPicture --
 *
 *      Takes a snapshot of an X drawable (pixmap or window) and
 *	converts it to a picture.
 *
 *	This routine is used to snap foreign (non-Tk) windows. For
 *	pixmaps and Tk windows, Blt_DrawableToPicture is preferred.
 *
 * Results:
 *      Returns a picture of the drawable.  If an error occurred,
 *	NULL is returned.
 *
 *----------------------------------------------------------------------
 */
Picture *
Blt_WindowToPicture(
    Display *display,
    Drawable drawable,
    int x, int y,		/* Offset of image from the drawable's
				 * origin. */
    int width, int height,	/* Dimension of the image.  Image must
				 * be completely contained by the
				 * drawable. */
    double gamma)
{
    Blt_Painter painter;
    Blt_Picture picture;

    painter = Blt_GetPainterFromDrawable(display, drawable, gamma);
    picture =  DrawableToPicture(painter, drawable, x, y, width, height);
    Blt_FreePainter(painter);
    return picture;
}

int
Blt_PaintPicture(
    Blt_Painter painter,
    Drawable drawable,
    Blt_Picture picture,
    int areaX, int areaY,	/* Starting coordinates of subregion in the
				 * picture to be painted. */
    int areaWidth, int areaHeight, /* Dimension of the subregion.  */
    int x, int y,		/* Coordinates of region in the
				 * drawable.  */
    unsigned int flags)
{
    /* 
     * Nothing to draw. The region offset starts beyond the end of
     * the picture. 
     *
     *  +---------------+    	
     *  |               |    	
     *  |               |    	
     *  |               | ax,ay   
     *  |               |   +---------+
     *  |               |   |         |
     *  +---------------+   |         |
     *			    |         |
     *                      +---------+
     */			        
    if ((picture == NULL) || 
	(areaX >= Blt_PictureWidth(picture)) || 
	(areaY >= Blt_PictureHeight(picture))) {
	return TRUE;	
    }
    /* 
     * Correct the dimensions if the origin starts before the picture
     * (i.e. coordinate is negative).  Reset the coordinate the 0. 
     *
     * ax,ay		       
     *   +---------+ 	          ax,ay		       
     *   |  +------|--------+       +------+--------+
     *   |  |      |        |       |      |        |
     *   |  |      |        |       |      |        |
     *   +--|------+        |       +------+        |
     *      |               |       |               |
     *      |               |       |               |
     *      +---------------+       +---------------+ 
     *
     */
    if (areaX < 0) {		
        areaWidth += areaX;
        areaX = 0;
    }
    if (areaY < 0) {
        areaHeight += areaY;
        areaY = 0;
    }
    /* 
     * Check that the given area does not extend beyond the end of the
     * picture.
     * 
     *    +-----------------+	     +-----------------+	  	
     *    |                 |	     |                 |	  	
     *    |      ax,ay      |	     |      ax,ay      |	  	
     *    |         +---------+      |         +-------+
     *    |         |       | |      |         |       |
     *    |         |       | |      |         |       |
     *    +---------|-------+ |      +---------+-------+
     * 	            +---------+   	           
     *                                                    
     * Clip the end of the area if it's too big.
     */
    if ((areaWidth + areaX) > Blt_PictureWidth(picture)) {
	areaWidth = Blt_PictureWidth(picture) - areaX;
    }
    if ((areaHeight + areaY) > Blt_PictureHeight(picture)) {
	areaHeight = Blt_PictureHeight(picture) - areaY;
    }
    /* Check that there's still something to paint. */
    if ((areaWidth <= 0) || (areaHeight <= 0)) {
	return TRUE;
    }
    if (Blt_PictureIsOpaque(picture)) {
	return PaintPicture(painter, drawable, picture, areaX, areaY, 
		areaWidth, areaHeight, x, y, flags);
    } else {
	int alpha = 128;

	return PaintPictureWithBlend(painter, drawable, picture, areaX, areaY, 
		areaWidth, areaHeight, x, y, flags, alpha);
    }
}

int
Blt_PaintPictureWithBlend(
    Blt_Painter painter,
    Drawable drawable,
    Blt_Picture picture,
    int areaX, int areaY,	/* Coordinates of region in the
				 * picture. */
    int areaWidth, int areaHeight, /* Dimension of the region.  Area
				 * cannot extend beyond the end of the
				 * picture. */
    int x, int y,		/* Coordinates of region in the
				 * drawable.  */
    unsigned int flags,		/* Indicates whether to dither the
				 * picture before displaying. */
    double falpha)
{
    int alpha;

    alpha = (int)(falpha * 255.0 + 0.5);

    /* 
     * Nothing to draw. The region offset starts beyond the end of
     * the picture. 
     *
     *  +---------------+    	
     *  |               |    	
     *  |               |    	
     *  |               | ax,ay   
     *  |               |   +---------+
     *  |               |   |         |
     *  +---------------+   |         |
     *			    |         |
     *                      +---------+
     */			        
    if ((picture == NULL) || 
	(areaX >= Blt_PictureWidth(picture)) || 
	(areaY >= Blt_PictureHeight(picture))) {
	return TRUE;	
    }
    /* 
     * Correct the dimensions if the origin starts before the picture
     * (i.e. coordinate is negative).  Reset the coordinate the 0. 
     *
     * ax,ay		       
     *   +---------+ 	          ax,ay		       
     *   |  +------|--------+       +------+--------+
     *   |  |      |        |       |      |        |
     *   |  |      |        |       |      |        |
     *   +--|------+        |       +------+        |
     *      |               |       |               |
     *      |               |       |               |
     *      +---------------+       +---------------+ 
     *
     */
    if (areaX < 0) {		
        areaWidth += areaX;
        areaX = 0;
    }
    if (areaY < 0) {
        areaHeight += areaY;
        areaY = 0;
    }
    /* 
     * Check that the given area does not extend beyond the end of the
     * picture.
     * 
     *    +-----------------+	     +-----------------+	  	
     *    |                 |	     |                 |	  	
     *    |      ax,ay      |	     |      ax,ay      |	  	
     *    |         +---------+      |         +-------+
     *    |         |       | |      |         |       |
     *    |         |       | |      |         |       |
     *    +---------|-------+ |      +---------+-------+
     * 	            +---------+   	           
     *                                                    
     * Clip the end of the area if it's too big.
     */
    if ((areaWidth + areaX) > Blt_PictureWidth(picture)) {
	areaWidth = Blt_PictureWidth(picture) - areaX;
    }
    if ((areaHeight + areaY) > Blt_PictureHeight(picture)) {
	areaHeight = Blt_PictureHeight(picture) - areaY;
    }
    /* Check that there's still something to paint. */
    if ((areaWidth <= 0) || (areaHeight <= 0)) {
	return TRUE;
    }
    return PaintPictureWithBlend(painter, drawable, picture, areaX, areaY,
	areaWidth, areaHeight, x, y, flags, alpha);
}

