###########################################################################
#                                                                         #
# Procedures to deal with events that come back from tgraph and cause     #
# display changes inside the tged editor.                                 #
#                                                                         #
# This file is part of the tged package.                                  #
#                                                                         #
###########################################################################

# Copyright (c) 2000-2003, JYL Software Inc.
# 
# Permission is hereby granted, free of charge, to any person obtaining
# a copy of this software and associated documentation files (the
# "Software"), to deal in the Software without restriction, including
# without limitation the rights to use, copy, modify, merge, publish,
# distribute, sublicense, and/or sell copies of the Software, and to
# permit persons to whom the Software is furnished to do so, subject to
# the following conditions:
# 
# The above copyright notice and this permission notice shall be
# included in all copies or substantial portions of the Software.
# 
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
# MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
# LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
# OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
# WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

# Install event handlers for the events of interest to the tged editor:

proc ::tged::initCallbacks {tree text storage} {
    variable data

    if {[string equal "" $storage]} {
	return
    }

    set data(i,$tree,addvertexcb) \
	[$storage callback add vertex add "::tged::addVertexCB $tree $text"]
    set data(i,$tree,detvertexcb) \
	[$storage callback add vertex det "::tged::detVertexCB $tree $text"]
    set data(i,$tree,modvertexcb) \
	[$storage callback add vertex mod "::tged::modVertexCB $tree $text"]
    set data(i,$tree,chgstoragecb) \
	[$storage callback add storage change \
		  "::tged::chgStorageCB $tree $tree"]
}

# Remove installed event handlers for a storage:

proc ::tged::removeCallbacks {tree} {
    variable data

    set storage ""
    catch {set storage $data(i,$tree,storage)}

    if {$storage == ""} {
	return
    }
    if {[info exists data(i,$tree,addvertexcb)]} {
	$storage callback del $data(i,$tree,addvertexcb)
	unset data(i,$tree,addvertexcb)
    }
    if {[info exists data(i,$tree,modvertexcb)]} {
	$storage callback del $data(i,$tree,modvertexcb)
	unset data(i,$tree,modvertexcb)
    }
    if {[info exists data(i,$tree,detvertexcb)]} {
	$storage callback del $data(i,$tree,detvertexcb)
	unset data(i,$tree,detvertexcb)
    }
    if {[info exist data(i,$tree,chgstoragecb)]} {
	$storage callback del $data(i,$tree,chgstoragecb)
	unset data(i,$tree,chgstoragecb)
    }
}

# This procedure is called when a vertex is added to a storage displayed
# in a tged editor.

proc ::tged::addVertexCB {tree text vertex} {
    # If the vertex is detached, we don't currently display it.

    if {[$vertex isdetached]} {
	return
    }

    # Retrieve the vertex ID and its type:

    set id [$vertex id]
    set type [$vertex type]

    # Cause a display change depending on the type of the value.

    if {$type == "node"} {
	addVertexToDisplayedNode $tree [$vertex node] $vertex
    } else {
	set node [$vertex node]
	if {[isSelectedNode $tree [$node id]]} {
	    refreshTextPane $text $node
	}
    }
}

# This procedure is called when a vertex is detached from a node in a
# storage being displayed in a tged editor.

proc ::tged::detVertexCB {tree text vertex} {
    variable data

    # Retrieve the vertex ID and its type:

    set id [$vertex id]
    set type [$vertex type]

    # Drop it from the cache:

    forgetCachedVertex $tree $id

    # If the vertex's value is a node, potentially remove the vertex from
    # the tree display. Otherwise potentially update the value display to
    # no longer display this value.

    if {$type == "node"} {
	removeVertexFromTreeDisplay $tree [[$vertex get] id]
    } else {

	# Scalar value. Determine if the vertex is being displayed now. If it
	# is then refresh the text pane so it will no longer be displayed.

	if {[isDisplayedScalarVertex $text $vertex]} {
	    set storage $data(i,$tree,storage)
	    refreshTextPane $text \
			    [$storage get node $data(i,$tree,selectednode)]
	}
    }
}

# This procedure is called when a vertex is modified in a storage displayed
# in a tged editor.

proc ::tged::modVertexCB {tree text vertex} {
    # Retrieve the vertex ID and its type:

    set id [$vertex id]
    set type [$vertex type]

    # If the vertex is now detached, remove it from the cache.

    if {[$vertex isdetached]} {
	forgetCachedVertex $tree $id
    }

    # If the type of the vertex is scalar, the potentially refresh the
    # text display.

    if {$type != "node"} {
	# Scalar value. Determine if the vertex is being displayed now. If it
	# is then refresh the text pane so it will be updated.

	if {[isDisplayedScalarVertex $text $vertex]} {
	    refreshTextPane $text [$vertex node]
	}
    } else {
	puts "oops! What to do with $vertex being detached? :)"
    }
}

# This procedure is called when the storage stability is changed from stable
# to unstable and vice-versa.

proc ::tged::chgStorageCB {tree text storage} {
    displayModified $tree
}
