/*
 * tkCanvSticker.c --
 *
 *	This file implements sticker items for canvas widgets.
 *	Much of the code is from tkCanvText.c and tkRectOval.c
 *	because a sticker item is basically text inside an rectangle item,
 *	but only that many characters will be drawn that fit into the bbox.
 *
 * Copyright (c) 1994-1995 Heribert Dahms (dahms@ifk20.mach.uni-karlsruhe.de)
 * All rights reserved. The following UCB regulations shall apply accordingly.
 *
 * Copyright (c) 1991-1993 The Regents of the University of California.
 * All rights reserved.
 *
 * Modified for tk8.0 16 October 1997 by Peter Fletcher, pfl@fmsc.com.au
 * Modified for tk8.0 - tk8.4  Feb 2001 by CRRW crrw@gmx.de
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE UNIVERSITY OF
 * CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND THE UNIVERSITY OF CALIFORNIA HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 * RCS: @(#) $Id: tkCanvSticker.c,v 1.4 2003/07/09 01:48:38 hobbs Exp $
 */

#include "pkg.h"

#if (defined(__WIN32__) || defined(MAC_TCL) || defined(MAC_OSX_TK)) && !defined(XDrawPoint)
#define XDrawPoint(display, d, gc, x, y) XDrawLine(display, d, gc, x, y, x, y)
#endif

/*
 * The structure below is used for rotating text.
 */

typedef struct Rotate_Stuff {
    int maxHeight;		/* Height of Bitmap */
    int maxWidth;		/* Width of Bitmap */
    Pixmap hBitmap;		/* To draw text invisibly here. */
    GC clearGC;			/* To clear Bitmap. */
    GC writeGC;			/* To write Bitmap. */
    Display *display;		/* Where it's valid. */
} Rotate_Stuff;

/*
 * Set sizes for initial allocation to some reasonable binary power.
 * Note: A single static, permanent buffer is nonreentrant, but efficient.
 */

static Rotate_Stuff RotateStuff = { 32, 128, None, None, None, NULL };

/*
 * The structure below defines the record for each sticker item.
 */

typedef struct StickerItem {
    Tk_Item header;		/* Generic stuff that's the same for all
				 * types.  MUST BE FIRST IN STRUCTURE. */
    double bbox[4];		/* Coordinates of bounding box for sticker's
				 * rectangle (x1, y1, x2, y2).  Item includes
				 * x1 and x2 but not y1 and y2. */
    int width;			/* Width of outline. */
    XColor *outlineColor;	/* Color for outline. */
    XColor *fillColor;		/* Color for filling sticker. */
    XColor *barColor;		/* Additional color for filling bar. */
    Pixmap fillStipple;		/* Stipple bitmap for filling item. */
    GC outlineGC;		/* Graphics context for outline. */
    GC fillGC;			/* Graphics context for filling item. */
    GC barGC;			/* Graphics context for filling bar. */
    double relheight, relwidth, /* Relative geometry of bar inside sticker. */
	relx, rely;
    char *text;			/* Text for item (malloc-ed). */
    int numChars;		/* Number of non-NULL characters in text. */
    char *start;		/* Start of visible portion of text. */
    int visChars;		/* Number of visible characters of text. */
    int space;			/* Space pixels between repeated text. */
    int repeats;		/* How many times text is repeated. */
    int horSize;		/* Horizontal size of text+space in Pixels. */
    Tk_Anchor anchor;		/* Where to anchor text relative to sticker. */
    int lefttrunc;		/* Non-zero means truncate text on the left. */
    Tk_Font fontPtr;		/* Font for drawing text. */
    XColor *textColor;		/* Color for text. */
    GC textGC;			/* Graphics context for drawing text. */
    int x, y;			/* Start coordinates of the text. */
    Tk_Uid orientUid;		/* Orientation for text ("vertical" or
				 * "horizontal"). */
    int vertical;		/* Non-zero means vertical orientation
				 * requested, zero means horizontal. */
    int numPixels;		/* Used text width in Pixels. */
    int minWidth, minHeight,	/* Size constraints in Pixels. */
	maxWidth, maxHeight;
} StickerItem;

/*
 * Information used for parsing configuration specs:
 */

static Tk_ConfigSpec configSpecs[] = {
    {TK_CONFIG_ANCHOR, "-anchor", (char *) NULL, (char *) NULL,
	"center", Tk_Offset(StickerItem, anchor), TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_COLOR, "-bar", (char *) NULL, (char *) NULL,
	"black", Tk_Offset(StickerItem, barColor), TK_CONFIG_NULL_OK},
    {TK_CONFIG_COLOR, "-color", (char *) NULL, (char *) NULL,
	"black", Tk_Offset(StickerItem, textColor), TK_CONFIG_NULL_OK},
    {TK_CONFIG_COLOR, "-fill", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(StickerItem, fillColor), TK_CONFIG_NULL_OK},
    {TK_CONFIG_FONT, "-font", (char *) NULL, (char *) NULL,
	DEF_FONT, Tk_Offset(StickerItem, fontPtr), 0},
    {TK_CONFIG_BOOLEAN, "-lefttrunc", (char *) NULL, (char *) NULL,
	"0", Tk_Offset(StickerItem, lefttrunc), 0},
    {TK_CONFIG_COLOR, "-outline", (char *) NULL, (char *) NULL,
	"black", Tk_Offset(StickerItem, outlineColor), TK_CONFIG_NULL_OK},
    {TK_CONFIG_DOUBLE, "-relheight", (char *) NULL, (char *) NULL,
	"0", Tk_Offset(StickerItem, relheight), 0},
    {TK_CONFIG_DOUBLE, "-relwidth", (char *) NULL, (char *) NULL,
	"0", Tk_Offset(StickerItem, relwidth), 0},
    {TK_CONFIG_DOUBLE, "-relx", (char *) NULL, (char *) NULL,
	"0", Tk_Offset(StickerItem, relx), 0},
    {TK_CONFIG_DOUBLE, "-rely", (char *) NULL, (char *) NULL,
	"0", Tk_Offset(StickerItem, rely), 0},
    {TK_CONFIG_PIXELS, "-space", (char *) NULL, (char *) NULL,
	"0", Tk_Offset(StickerItem, space), 0},
    {TK_CONFIG_BITMAP, "-stipple", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(StickerItem, fillStipple), TK_CONFIG_NULL_OK},
    {TK_CONFIG_CUSTOM, "-tags", (char *) NULL, (char *) NULL,
	(char *) NULL, 0, TK_CONFIG_NULL_OK, &customTagsOption},
    {TK_CONFIG_STRING, "-text", (char *) NULL, (char *) NULL,
	"", Tk_Offset(StickerItem, text), 0},
    {TK_CONFIG_PIXELS, "-width", (char *) NULL, (char *) NULL,
	"0", Tk_Offset(StickerItem, width), TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_UID, "-orient", (char *) NULL, (char *) NULL,
	"horizontal", Tk_Offset(StickerItem, orientUid), 0},
    {TK_CONFIG_PIXELS, "-minwidth", (char *) NULL, (char *) NULL,
	"0", Tk_Offset(StickerItem, minWidth), TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_PIXELS, "-minheight", (char *) NULL, (char *) NULL,
	"0", Tk_Offset(StickerItem, minHeight), TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_PIXELS, "-maxwidth", (char *) NULL, (char *) NULL,
	"32767", Tk_Offset(StickerItem, maxWidth), TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_PIXELS, "-maxheight", (char *) NULL, (char *) NULL,
	"32767", Tk_Offset(StickerItem, maxHeight), TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_END, (char *) NULL, (char *) NULL, (char *) NULL,
	(char *) NULL, 0, 0}
};

/*
 * Prototypes for procedures defined in this file:
 */

int			TkSticker_Init _ANSI_ARGS_((Tcl_Interp *interp));
int			Tksticker_Init _ANSI_ARGS_((Tcl_Interp *interp));
int			Tksticker_SaveInit _ANSI_ARGS_((Tcl_Interp *interp));
static void             ComputeStickerBbox _ANSI_ARGS_((Tk_Canvas canvas,
			    StickerItem *stickerPtr));
static int		ConfigureSticker _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Canvas canvas, Tk_Item *itemPtr, int objc,
			    Tcl_Obj *CONST objv[], int flags));
static int		CreateSticker _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Canvas canvas, struct Tk_Item *itemPtr,
				int objc, Tcl_Obj *CONST objv[]));
static void		DeleteSticker _ANSI_ARGS_((Tk_Canvas canvas,
			    Tk_Item *itemPtr, Display *display));
static void		DisplaySticker _ANSI_ARGS_((Tk_Canvas canvas,
			    Tk_Item *itemPtr, Display *display, Drawable dst,
			    int x, int y, int width, int height));
static void		LineToPostscript _ANSI_ARGS_((Tcl_Interp *interp,
			    char *string, int numChars));
static void		ScaleSticker _ANSI_ARGS_((Tk_Canvas canvas,
			    Tk_Item *itemPtr, double originX, double originY,
			    double scaleX, double scaleY));
static int		StickerCoords _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Canvas canvas, Tk_Item *itemPtr,
			    int objc, Tcl_Obj *CONST objv[]));
static int		StickerToArea _ANSI_ARGS_((Tk_Canvas canvas,
			    Tk_Item *itemPtr, double *rectPtr));
static double		StickerToPoint _ANSI_ARGS_((Tk_Canvas canvas,
			    Tk_Item *itemPtr, double *pointPtr));
static int		StickerToPostscript _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Canvas canvas, Tk_Item *itemPtr, int prepass));
static void		TranslateSticker _ANSI_ARGS_((Tk_Canvas canvas,
			    Tk_Item *itemPtr, double deltaX, double deltaY));

/*
 * The structures below defines the sticker item type
 * by means of procedures that can be invoked by generic item code.
 */

Tk_ItemType TkCanvStickerType = {
    "sticker",				/* name */
    sizeof(StickerItem),		/* itemSize */
    CreateSticker,			/* createProc */
    configSpecs,			/* configSpecs */
    ConfigureSticker,			/* configureProc */
    StickerCoords,			/* coordProc */
    DeleteSticker,			/* deleteProc */
    DisplaySticker,			/* displayProc */
#ifdef ORIG /*CRRW*/
    0,					/* alwaysRedraw */
#else
    TK_CONFIG_OBJS,			/* flags */
#endif
    StickerToPoint,			/* pointProc */
    StickerToArea,			/* areaProc */
    StickerToPostscript,		/* postscriptProc */
    ScaleSticker,			/* scaleProc */
    TranslateSticker,			/* translateProc */
    (Tk_ItemIndexProc *) NULL,		/* indexProc */
    (Tk_ItemCursorProc *) NULL,		/* icursorProc */
    (Tk_ItemSelectionProc *) NULL,	/* selectionProc */
    (Tk_ItemInsertProc *) NULL,	        /* insertProc */
    (Tk_ItemDCharsProc *) NULL,	        /* dTextProc */
    (Tk_ItemType *) NULL		/* nextPtr */
};

/*
 *--------------------------------------------------------------
 *
 * CreateSticker --
 *
 *	This procedure is invoked to create a new sticker item
 *	in a canvas.
 *
 * Results:
 *	A standard Tcl return value.  If an error occurred in
 *	creating the item then an error message is left in
 *	interp->result;  in this case itemPtr is left uninitialized,
 *	so it can be safely freed by the caller.
 *      which seems weird, but I think it's the best solution.
 *
 * Side effects:
 *	A new sticker item is created.
 *
 *--------------------------------------------------------------
 */

static int
CreateSticker(interp, canvas, itemPtr, objc, objv)
    Tcl_Interp *interp;			/* For error reporting. */
    Tk_Canvas canvas;			/* Canvas to hold new item. */
    Tk_Item *itemPtr;			/* Record to hold new item;  header
					 * has been initialized by caller. */
    int objc;				/* Number of arguments in objv. */
    Tcl_Obj *CONST objv[];		/* Arguments describing sticker. */
{
    StickerItem *stickerPtr = (StickerItem *) itemPtr;

    if (objc < 4) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		Tk_PathName(Tk_CanvasTkwin(canvas)), "\" create ",
		itemPtr->typePtr->name, " x1 y1 x2 y2 ?options?",
		(char *) NULL);
	return TCL_ERROR;
    }

    /*
     * Carry out initialization that is needed in order to clean
     * up after errors during the the remainder of this procedure.
     */

    stickerPtr->width		= 0;
    stickerPtr->outlineColor	= NULL;
    stickerPtr->fillColor	= NULL;
    stickerPtr->barColor	= NULL;
    stickerPtr->fillStipple	= None;
    stickerPtr->outlineGC	= None;
    stickerPtr->fillGC		= None;
    stickerPtr->barGC		= None;
    stickerPtr->relheight	= 0;
    stickerPtr->relwidth	= 0;
    stickerPtr->relx		= 0;
    stickerPtr->rely		= 0;
    stickerPtr->text		= NULL;
    stickerPtr->numChars	= 0;
    stickerPtr->start		= NULL;
    stickerPtr->visChars	= 0;
    stickerPtr->space		= 0;
    stickerPtr->repeats		= 1;
    stickerPtr->horSize		= 1;
    stickerPtr->anchor		= TK_ANCHOR_CENTER;
    stickerPtr->lefttrunc	= 0;
    stickerPtr->fontPtr		= NULL;
    stickerPtr->textColor	= NULL;
    stickerPtr->textGC		= None;
    stickerPtr->x		= 0;
    stickerPtr->y		= 0;
    stickerPtr->orientUid	= NULL;
    stickerPtr->vertical	= 0;
    stickerPtr->numPixels	= 0;
    stickerPtr->minWidth	= 0;
    stickerPtr->minHeight	= 0;
    stickerPtr->maxWidth	= 32767;
    stickerPtr->maxHeight	= 32767;

    /*
     * Process the arguments to fill in the item record.
     */

    if (StickerCoords(interp, canvas, itemPtr, 4, objv) != TCL_OK) {
	goto error;
    }

    if (ConfigureSticker(interp, canvas, itemPtr, objc - 4, objv + 4, 0)
	    == TCL_OK) {
	return TCL_OK;
    }

    error:
    DeleteSticker(canvas, itemPtr, Tk_Display(Tk_CanvasTkwin(canvas)));
    return TCL_ERROR;
}

/*
 *--------------------------------------------------------------
 *
 * StickerCoords --
 *
 *	This procedure is invoked to process the "coords" widget
 *	command on sticker items.  See the user documentation for
 *	details on what it does.
 *
 * Results:
 *	Returns TCL_OK or TCL_ERROR, and sets interp->result.
 *
 * Side effects:
 *	The coordinates for the given item may be changed.
 *
 *--------------------------------------------------------------
 */

static int
StickerCoords(interp, canvas, itemPtr, objc, objv)
    Tcl_Interp *interp;			/* Used for error reporting. */
    Tk_Canvas canvas;			/* Canvas containing item. */
    Tk_Item *itemPtr;			/* Item whose coordinates are to be
					 * read or modified. */
    int objc;				/* Number of coordinates supplied in
					 * objv. */
    Tcl_Obj *CONST objv[];		/* Array of coordinates: x1, y1,
					 * x2, y2, ... */
{
    StickerItem *stickerPtr = (StickerItem *) itemPtr;
    int i;

    if (objc == 0) {
	Tcl_Obj *objPtr = Tcl_NewObj();
	for (i = 0; i < 4; i++) {
	    Tcl_ListObjAppendElement(interp, objPtr,
		    Tcl_NewDoubleObj(stickerPtr->bbox[i]));
	}
	Tcl_SetObjResult(interp, objPtr);
    } else if (objc == 4) {
	for (i = 0; i < 4; i++) {
	    if (Tk_CanvasGetCoordFromObj(interp, canvas, objv[i],
		    &stickerPtr->bbox[i]) != TCL_OK) {
		return TCL_ERROR;
	    }
	}

	ComputeStickerBbox(canvas, stickerPtr);
    } else {
	char buf[64 + TCL_INTEGER_SPACE];
	
	sprintf(buf, "wrong # coordinates: expected 0 or 4, got %d", objc);
	Tcl_SetResult(interp, buf, TCL_VOLATILE);
	return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * ConfigureSticker --
 *
 *	This procedure is invoked to configure various aspects
 *	of a sticker item, such as its border and background colors.
 *
 * Results:
 *	A standard Tcl result code.  If an error occurs, then
 *	an error message is left in interp->result.
 *
 * Side effects:
 *	Configuration information, such as colors and stipple
 *	patterns, may be set for itemPtr.
 *
 *--------------------------------------------------------------
 */

static int
ConfigureSticker(interp, canvas, itemPtr, objc, objv, flags)
    Tcl_Interp *interp;		/* Used for error reporting. */
    Tk_Canvas canvas;		/* Canvas containing itemPtr. */
    Tk_Item *itemPtr;		/* Sticker item to reconfigure. */
    int objc;			/* Number of elements in objv.  */
    Tcl_Obj *CONST objv[];   	/* Arguments describing things to configure. */
    int flags;			/* Flags to pass to Tk_ConfigureWidget. */
{
    StickerItem *stickerPtr = (StickerItem *) itemPtr;
    XGCValues gcValues;
    GC newGC;
    unsigned long mask;
    size_t length;
    Tk_Window tkwin;

    tkwin = Tk_CanvasTkwin(canvas);
    if (Tk_ConfigureWidget(interp, tkwin, configSpecs, objc,
	    (CONST84 char **) objv,
	    (char *) stickerPtr, flags|TK_CONFIG_OBJS) != TCL_OK) {
	return TCL_ERROR;
    }

    /*
     * A few of the options require additional processing, such as
     * graphics contexts.
     */

    stickerPtr->numChars = strlen(stickerPtr->text);
    newGC = None;
    if ((stickerPtr->textColor != NULL) && (stickerPtr->fontPtr != NULL)) {
	gcValues.foreground = stickerPtr->textColor->pixel;
	gcValues.font = Tk_FontId(stickerPtr->fontPtr);
	mask = GCForeground|GCFont;
	newGC = Tk_GetGC(tkwin, mask, &gcValues);
    }
    if (stickerPtr->textGC != None) {
	Tk_FreeGC(Tk_Display(tkwin), stickerPtr->textGC);
    }
    stickerPtr->textGC = newGC;

    if (stickerPtr->width < 0) {
	stickerPtr->width = 1;
    }
    if (stickerPtr->outlineColor == NULL) {
	newGC = None;
    } else {
	gcValues.foreground = stickerPtr->outlineColor->pixel;
	gcValues.cap_style = CapProjecting;
	gcValues.line_width = stickerPtr->width;
	mask = GCForeground|GCCapStyle|GCLineWidth;
	newGC = Tk_GetGC(tkwin, mask, &gcValues);
    }
    if (stickerPtr->outlineGC != None) {
	Tk_FreeGC(Tk_Display(tkwin), stickerPtr->outlineGC);
    }
    stickerPtr->outlineGC = newGC;

    if (stickerPtr->fillColor == NULL) {
	newGC = None;
    } else {
	gcValues.foreground = stickerPtr->fillColor->pixel;
	if (stickerPtr->fillStipple != None) {
	    gcValues.stipple = stickerPtr->fillStipple;
	    gcValues.fill_style = FillStippled;
	    mask = GCForeground|GCStipple|GCFillStyle;
	} else {
	    mask = GCForeground;
	}
	newGC = Tk_GetGC(tkwin, mask, &gcValues);
    }
    if (stickerPtr->fillGC != None) {
	Tk_FreeGC(Tk_Display(tkwin), stickerPtr->fillGC);
    }
    stickerPtr->fillGC = newGC;

    if (stickerPtr->barColor == NULL) {
	newGC = None;
    } else {
	gcValues.foreground = stickerPtr->barColor->pixel;
	if (stickerPtr->fillStipple != None) {
	    gcValues.stipple = stickerPtr->fillStipple;
	    gcValues.fill_style = FillStippled;
	    mask = GCForeground|GCStipple|GCFillStyle;
	} else {
	    mask = GCForeground;
	}
	newGC = Tk_GetGC(tkwin, mask, &gcValues);
    }
    if (stickerPtr->barGC != None) {
	Tk_FreeGC(Tk_Display(tkwin), stickerPtr->barGC);
    }
    stickerPtr->barGC = newGC;

    length = strlen(stickerPtr->orientUid);
    if (strncmp(stickerPtr->orientUid, "horizontal", length) == 0) {
	stickerPtr->vertical = 0;
    } else if (strncmp(stickerPtr->orientUid, "vertical", length) == 0) {
	stickerPtr->vertical = 1;
    } else {
	Tcl_AppendResult(interp, "bad orientation \"", stickerPtr->orientUid,
		"\": must be vertical or horizontal", (char *) NULL);
	return TCL_ERROR;
    }

    ComputeStickerBbox(canvas, stickerPtr);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * DeleteSticker --
 *
 *	This procedure is called to clean up the data structure
 *	associated with a sticker item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Resources associated with itemPtr are released.
 *
 *--------------------------------------------------------------
 */

static void
DeleteSticker(canvas, itemPtr, display)
    Tk_Canvas canvas;			/* Info about overall canvas widget. */
    Tk_Item *itemPtr;			/* Item that is being deleted. */
    Display *display;			/* Display containing window for
					 * canvas. */
{
    StickerItem *stickerPtr = (StickerItem *) itemPtr;

    if (stickerPtr->text != NULL) {
	ckfree(stickerPtr->text);
    }
    if (stickerPtr->fontPtr != NULL) {
	Tk_FreeFont(stickerPtr->fontPtr);
    }
    if (stickerPtr->textColor != NULL) {
	Tk_FreeColor(stickerPtr->textColor);
    }
    if (stickerPtr->fillColor != NULL) {
	Tk_FreeColor(stickerPtr->fillColor);
    }
    if (stickerPtr->barColor != NULL) {
	Tk_FreeColor(stickerPtr->barColor);
    }
    if (stickerPtr->outlineColor != NULL) {
	Tk_FreeColor(stickerPtr->outlineColor);
    }
    if (stickerPtr->fillStipple != None) {
	Tk_FreeBitmap(display, stickerPtr->fillStipple);
    }
    if (stickerPtr->textGC != None) {
	Tk_FreeGC(display, stickerPtr->textGC);
    }
    if (stickerPtr->fillGC != None) {
	Tk_FreeGC(display, stickerPtr->fillGC);
    }
    if (stickerPtr->barGC != None) {
	Tk_FreeGC(display, stickerPtr->barGC);
    }
    if (stickerPtr->outlineGC != None) {
	Tk_FreeGC(display, stickerPtr->outlineGC);
    }
}

/*
 *--------------------------------------------------------------
 *
 * ComputeStickerBbox --
 *
 *	This procedure is invoked to compute the bounding box of
 *	all the pixels that may be drawn as part of a sticker item.
 *	In addition, it recomputes all of the geometry information
 *	used to display a sticker item or check for mouse hits.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The fields x1, y1, x2, and y2 are updated in the header
 *	for itemPtr.
 *
 *--------------------------------------------------------------
 */

static void
ComputeStickerBbox(canvas, stickerPtr)
    Tk_Canvas canvas;			/* Canvas that contains item. */
    StickerItem *stickerPtr;		/* Item whose bbox is to be
					 * recomputed. */
{
    int maxPixels, leftX, topY;
    int lineHeight, maxHeight;
    int bloat;
    double factor;
    Tk_FontMetrics fmp;

    if (stickerPtr->fontPtr == NULL) {
	/*
	 * This only occurs when the sticker hasn't been properly
	 * configured at least once.
	 */
	return;
    }

    /*
     * Make sure that the first coordinates are the lowest ones.
     */

    if (stickerPtr->bbox[1] > stickerPtr->bbox[3]) {
	double tmp;
	tmp = stickerPtr->bbox[3];
	stickerPtr->bbox[3] = stickerPtr->bbox[1];
       	stickerPtr->bbox[1] = tmp;
    }
    if (stickerPtr->bbox[0] > stickerPtr->bbox[2]) {
	double tmp;
	tmp = stickerPtr->bbox[2];
	stickerPtr->bbox[2] = stickerPtr->bbox[0];
	stickerPtr->bbox[0] = tmp;
    }

    if (stickerPtr->outlineColor == NULL) {
	bloat = 0;
    } else {
	bloat = (stickerPtr->width+1)/2;
    }
    stickerPtr->header.x1 = stickerPtr->bbox[0] - bloat;
    stickerPtr->header.y1 = stickerPtr->bbox[1] - bloat;
    stickerPtr->header.x2 = stickerPtr->bbox[2] + bloat + 1;
    stickerPtr->header.y2 = stickerPtr->bbox[3] + bloat + 1;

    /*
     * Work through the text computing the starting point, number of
     * visible characters, and number of pixels.
     */

    if (stickerPtr->vertical) {
	maxPixels = stickerPtr->bbox[3] - stickerPtr->bbox[1] - 1 - 2*bloat;
	maxHeight = stickerPtr->bbox[2] - stickerPtr->bbox[0] - 1 - 2*bloat;
    } else {
	maxPixels = stickerPtr->bbox[2] - stickerPtr->bbox[0] - 1 - 2*bloat;
	maxHeight = stickerPtr->bbox[3] - stickerPtr->bbox[1] - 1 - 2*bloat;
    }

    if (!stickerPtr->lefttrunc) {
	stickerPtr->visChars = Tk_MeasureChars(stickerPtr->fontPtr,
		stickerPtr->text,
		stickerPtr->numChars,
		maxPixels, 0, &stickerPtr->numPixels);
    } else {
	char *ptr;
	int i, j;
	char buf[100];

	ptr = (stickerPtr->numChars <= sizeof(buf)) ? buf :
		ckalloc((size_t) stickerPtr->numChars);

	/*
	 * Reverse all characters of the text.
	 */

	for (i = stickerPtr->numChars - 1, j = 0; i >= 0; --i, j++) {
		ptr[i] = stickerPtr->text[j];
	}

	stickerPtr->visChars = Tk_MeasureChars(stickerPtr->fontPtr,
                ptr,
                stickerPtr->numChars,
                maxPixels, 0, &stickerPtr->numPixels);
	if (ptr != buf) ckfree(ptr);
    }


    stickerPtr->start = stickerPtr->text + (stickerPtr->lefttrunc ?
	(stickerPtr->numChars - stickerPtr->visChars) : 0);

    /*
     * Compute repeat count and spacing information, if desired.
     */

    if (stickerPtr->space > 0) {
	stickerPtr->horSize = stickerPtr->numPixels + stickerPtr->space;
	stickerPtr->repeats = (maxPixels + stickerPtr->space) /
				stickerPtr->horSize;
	stickerPtr->numPixels = stickerPtr->repeats * stickerPtr->horSize -
			stickerPtr->space;
    } else {
	stickerPtr->repeats = 1;
    }

    /*
     * Use overall geometry information to compute the top-left corner
     * of the bounding box for the text of the sticker item.
     * If rotated (vertical), the roles of x and y exchange temporarily,
     * so the anchor coordinate system is always relative to the text.
     */

    if (stickerPtr->vertical) {
	leftX = stickerPtr->bbox[3] - 0.5 - bloat;
	topY = stickerPtr->bbox[0] + 0.5 + bloat;
    } else {
	leftX = stickerPtr->bbox[0] + 0.5 + bloat;
	topY = stickerPtr->bbox[1] + 0.5 + bloat;
    }
    Tk_GetFontMetrics(stickerPtr->fontPtr, &fmp);
    lineHeight = fmp.ascent + fmp.descent;
    factor = 0;		/* Only to prevent compiler warnings */
    switch (stickerPtr->anchor) {
	case TK_ANCHOR_NW:
	case TK_ANCHOR_N:
	case TK_ANCHOR_NE:
	    factor = 0;
	    break;

	case TK_ANCHOR_W:
	case TK_ANCHOR_CENTER:
	case TK_ANCHOR_E:
	    factor = 0.5;
	    break;

	case TK_ANCHOR_SW:
	case TK_ANCHOR_S:
	case TK_ANCHOR_SE:
	    factor = 1;
	    break;
    }
    topY += (maxHeight - lineHeight) * factor;
    switch (stickerPtr->anchor) {
	case TK_ANCHOR_NW:
	case TK_ANCHOR_W:
	case TK_ANCHOR_SW:
	    factor = 0;
	    break;

	case TK_ANCHOR_N:
	case TK_ANCHOR_CENTER:
	case TK_ANCHOR_S:
	    factor = 0.5;
	    break;

	case TK_ANCHOR_NE:
	case TK_ANCHOR_E:
	case TK_ANCHOR_SE:
	    factor = 1;
	    break;
    }

    if (stickerPtr->vertical) {
	leftX -= (maxPixels - stickerPtr->numPixels) * factor;
	stickerPtr->y = leftX;
	stickerPtr->x = topY;
    } else {
	leftX += (maxPixels - stickerPtr->numPixels) * factor;
	stickerPtr->x = leftX;
	stickerPtr->y = topY;
    }
    if (lineHeight > maxHeight) stickerPtr->visChars = 0;
}

/*
 *--------------------------------------------------------------
 *
 * DisplaySticker --
 *
 *	This procedure is invoked to draw a sticker item in a given
 *	drawable.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	ItemPtr is drawn in drawable using the transformation
 *	information in canvas.
 *
 *--------------------------------------------------------------
 */

static void
DisplaySticker(canvas, itemPtr, display, drawable, x, y, width, height)
    Tk_Canvas canvas;			/* Canvas that contains item. */
    Tk_Item *itemPtr;			/* Item to be displayed. */
    Display *display;			/* Display on which to draw item. */
    Drawable drawable;			/* Pixmap or window in which to draw
					 * item. */
     int x, y, width, height;		/* Describes region of canvas that
					 * must be redisplayed (not used). */
{
    StickerItem *stickerPtr = (StickerItem *) itemPtr;
    short x1, y1, x2, y2;
    short bx1, by1, bx2, by2;
    int i, w;
    short drawableX, drawableY;
    Tk_FontMetrics fmp;

    /*
     * Compute the screen coordinates of the bounding box for the item.
     * Make sure that the bbox is at least one pixel large, since some
     * X servers will die if it isn't.
     */

    Tk_CanvasDrawableCoords(canvas, stickerPtr->bbox[0], stickerPtr->bbox[1],
	&x1, &y1);
    Tk_CanvasDrawableCoords(canvas, stickerPtr->bbox[2], stickerPtr->bbox[3],
	&x2, &y2);
    if (x2 <= x1) {
	x2 = x1+1;
    }
    if (y2 <= y1) {
	y2 = y1+1;
    }

    /*
     * Suppress all drawing if size is out of the given range.
     */

     if (x2 - x1 < stickerPtr->minWidth  || x2 - x1 > stickerPtr->maxWidth ||
	 y2 - y1 < stickerPtr->minHeight || y2 - y1 > stickerPtr->maxHeight)
	return;

    /*
     * A fat outline produces a smaller remaining inner rectangle, which
     * is then partitioned in a relative manner inspired by the Tk Placer.
     */

    w = stickerPtr->width;
    bx1 = x1 + w/2 + stickerPtr->relx * (x2 - x1 - w);
    by1 = y1 + w/2 + stickerPtr->rely * (y2 - y1 - w);
    bx2 = bx1 + stickerPtr->relwidth * (x2 - x1 - w);
    by2 = by1 + stickerPtr->relheight * (y2 - y1 - w);

    /*
     * Clip to bbox.
     */

    if (bx1 < x1) bx1 = x1; else if (bx1 > x2) bx1 = x2;
    if (by1 < y1) by1 = y1; else if (by1 > y2) by1 = y2;
    if (bx2 < x1) bx2 = x1; else if (bx2 > x2) bx2 = x2;
    if (by2 < y1) by2 = y1; else if (by2 > y2) by2 = y2;

    /*
     * Display filled parts first (if wanted), then outline.  If we're
     * stippling, then modify the stipple offset in the GC.  Be sure to
     * reset the offset when done, since the GC is supposed to be
     * read-only.
     */

    if (stickerPtr->fillGC != None) {
	if (stickerPtr->fillStipple != None) {
	    Tk_CanvasSetStippleOrigin(canvas, stickerPtr->fillGC);
	}
	XFillRectangle(display, drawable, stickerPtr->fillGC,
		x1, y1, (unsigned int) (x2-x1), (unsigned int) (y2-y1));
	if (stickerPtr->fillStipple != None) {
	    XSetTSOrigin(display, stickerPtr->fillGC, 0, 0);
	}
    }

    if (stickerPtr->barGC != None && bx1 < bx2 && by1 < by2) {
        if (stickerPtr->fillStipple != None) {
	    Tk_CanvasSetStippleOrigin(canvas, stickerPtr->barGC);
        }
        XFillRectangle(display, drawable, stickerPtr->barGC,
                bx1, by1, (unsigned int) (bx2-bx1), (unsigned int) (by2-by1));
        if (stickerPtr->fillStipple != None) {
            XSetTSOrigin(display, stickerPtr->barGC, 0, 0);
        }
    }

    if (stickerPtr->outlineGC != None) {
	XDrawRectangle(display, drawable, stickerPtr->outlineGC,
		x1, y1, (unsigned int) (x2-x1-1), (unsigned int) (y2-y1-1));
    }

    /*
     * Finally display the text, optionally repeated and/or rotated.
     */

    if (stickerPtr->textGC != None && stickerPtr->visChars > 0) {
	int w, h;
	int lineHeight;
	XImage *hImage;

        Tk_GetFontMetrics(stickerPtr->fontPtr, &fmp);
	lineHeight = fmp.ascent + fmp.descent;
	w = stickerPtr->numPixels;	/* More is useless */
	h = lineHeight;

	/*
	 * Check for enough resources if rotated text is desired.
	 * Also ensure cached resources are valid for current display.
	 */

	if (stickerPtr->vertical && (RotateStuff.display != display ||
		w > RotateStuff.maxWidth ||
		h > RotateStuff.maxHeight)) {
	    if (RotateStuff.hBitmap != None) {
		Tk_FreePixmap(RotateStuff.display, RotateStuff.hBitmap);
	    }

	    /* Double sizes until fit. */
	    while (w > RotateStuff.maxWidth)
		RotateStuff.maxWidth <<= 1;
	    while (h > RotateStuff.maxHeight)
		RotateStuff.maxHeight <<= 1;

	    RotateStuff.hBitmap = Tk_GetPixmap(display, drawable,
		RotateStuff.maxWidth, RotateStuff.maxHeight, 1);

	    if (RotateStuff.display != display) {
	        XGCValues gcValues;
	        unsigned long mask;

		if (RotateStuff.clearGC != None) {
		    Tk_FreeXId(display,
			(XID) XGContextFromGC(RotateStuff.clearGC));
		    XFreeGC(RotateStuff.display, RotateStuff.clearGC);
		    Tk_FreeXId(display,
			(XID) XGContextFromGC(RotateStuff.writeGC));
		    XFreeGC(RotateStuff.display, RotateStuff.writeGC);
		}
		gcValues.foreground = gcValues.background = 0;
		mask = GCForeground | GCBackground;
		RotateStuff.clearGC = XCreateGC(display, RotateStuff.hBitmap,
					mask, &gcValues);

		gcValues.foreground = 1;
		mask = GCForeground | GCBackground;
		RotateStuff.writeGC = XCreateGC(display, RotateStuff.hBitmap,
					mask, &gcValues);
		RotateStuff.display = display;
	    }
	}

	hImage = None;
	Tk_CanvasDrawableCoords(canvas, (double) stickerPtr->x,
		(double) stickerPtr->y, &drawableX, &drawableY);
	for (i = 0; i < stickerPtr->repeats; i++) {
	    if (stickerPtr->vertical) {
#if !defined(__WIN32__) && !defined(MAC_TCL) && !defined(MAC_OSX_TK)
		/* XGetImage is only fully implemented on Unix */
		int x, y;

		if (i == 0) {
		    XFillRectangle(display, RotateStuff.hBitmap,
			    RotateStuff.clearGC, 0, 0,
			    (unsigned int) w, (unsigned int) h);
		    XSetFont(display, RotateStuff.writeGC,
			    Tk_FontId(stickerPtr->fontPtr));

		    Tk_DrawChars(display, RotateStuff.hBitmap,
			    RotateStuff.writeGC, stickerPtr->fontPtr,
			    stickerPtr->start, stickerPtr->visChars,
			    0, fmp.ascent);
		    hImage = XGetImage(display, RotateStuff.hBitmap, 0, 0,
			    (unsigned int) w, (unsigned int) h, 1, ZPixmap);
		}

		for (x = 0; x < w; x++) {
		    for (y = 0; y < h; y++) {
			unsigned pixel;

			pixel = XGetPixel(hImage, x, y);
			if (pixel) {
			    XDrawPoint(display, drawable, stickerPtr->textGC,
				drawableX + y,
				drawableY - (i * stickerPtr->horSize) - x);
			}
		    }
		}
#endif
	    } else {
		Tk_DrawChars(display, drawable,
		    stickerPtr->textGC, stickerPtr->fontPtr,
		    stickerPtr->start, stickerPtr->visChars,
		    drawableX + (i * stickerPtr->horSize),
		    drawableY + fmp.ascent);
	    }
	}
	if (hImage != None) XDestroyImage(hImage);
    }
}

/*
 *--------------------------------------------------------------
 *
 * StickerToPoint --
 *
 *	Computes the distance from a given point to a given
 *	sticker item, in canvas units.
 *
 * Results:
 *	The return value is 0 if the point whose x and y coordinates
 *	are pointPtr[0] and pointPtr[1] is inside the sticker. If the
 *	point isn't inside the sticker then the return value is the
 *	distance from the point to the sticker.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

	/* ARGSUSED */
static double
StickerToPoint(canvas, itemPtr, pointPtr)
    Tk_Canvas canvas;		/* Canvas containing itemPtr. */
    Tk_Item *itemPtr;		/* Item to check against point. */
    double *pointPtr;		/* Pointer to x and y coordinates. */
{
    StickerItem *stickerPtr = (StickerItem *) itemPtr;
    double x1, x2, y1, y2, xDiff, yDiff;

    x1 = stickerPtr->header.x1;
    x2 = stickerPtr->header.x2;
    y1 = stickerPtr->header.y1;
    y2 = stickerPtr->header.y2;

    /*
     * Point is outside rectangle.
     */

    if (pointPtr[0] < x1) {
	xDiff = x1 - pointPtr[0];
    } else if (pointPtr[0] >= x2)  {
	xDiff = pointPtr[0] + 1 - x2;
    } else {
	xDiff = 0;
    }

    if (pointPtr[1] < y1) {
	yDiff = y1 - pointPtr[1];
    } else if (pointPtr[1] >= y2)  {
	yDiff = pointPtr[1] + 1 - y2;
    } else {
	yDiff = 0;
    }

    return hypot(xDiff, yDiff);
}

/*
 *--------------------------------------------------------------
 *
 * StickerToArea --
 *
 *	This procedure is called to determine whether an item
 *	lies entirely inside, entirely outside, or overlapping
 *	a given rectangle.
 *
 * Results:
 *	-1 is returned if the item is entirely outside the area
 *	given by rectPtr, 0 if it overlaps, and 1 if it is entirely
 *	inside the given area.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

	/* ARGSUSED */
static int
StickerToArea(canvas, itemPtr, rectPtr)
    Tk_Canvas canvas;		/* Canvas containing itemPtr. */
    Tk_Item *itemPtr;		/* Item to check against rectangle. */
    double *rectPtr;		/* Pointer to array of four coordinates
				 * (x1, y1, x2, y2) describing rectangular
				 * area.  */
{
    StickerItem *stickerPtr = (StickerItem *) itemPtr;

    if ((rectPtr[2] <= stickerPtr->header.x1)
            || (rectPtr[0] >= stickerPtr->header.x2)
            || (rectPtr[3] <= stickerPtr->header.y1)
            || (rectPtr[1] >= stickerPtr->header.y2)) {
        return -1;
    }
    if ((rectPtr[0] <= stickerPtr->header.x1)
            && (rectPtr[1] <= stickerPtr->header.y1)
            && (rectPtr[2] >= stickerPtr->header.x2)
            && (rectPtr[3] >= stickerPtr->header.y2)) {
        return 1;
    }
    return 0;
}

/*
 *--------------------------------------------------------------
 *
 * ScaleSticker --
 *
 *	This procedure is invoked to rescale a sticker item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Scales the position and size of the sticker, but not the size
 *	of the font for the text.
 *
 *--------------------------------------------------------------
 */

	/* ARGSUSED */
static void
ScaleSticker(canvas, itemPtr, originX, originY, scaleX, scaleY)
    Tk_Canvas canvas;			/* Canvas containing sticker. */
    Tk_Item *itemPtr;			/* Sticker item to be scaled. */
    double originX, originY;		/* Origin about which to scale item. */
    double scaleX;			/* Amount to scale in X direction. */
    double scaleY;			/* Amount to scale in Y direction. */
{
    StickerItem *stickerPtr = (StickerItem *) itemPtr;

    stickerPtr->bbox[0] = originX + scaleX*(stickerPtr->bbox[0] - originX);
    stickerPtr->bbox[1] = originY + scaleY*(stickerPtr->bbox[1] - originY);
    stickerPtr->bbox[2] = originX + scaleX*(stickerPtr->bbox[2] - originX);
    stickerPtr->bbox[3] = originY + scaleY*(stickerPtr->bbox[3] - originY);
    ComputeStickerBbox(canvas, stickerPtr);
    return;
}

/*
 *--------------------------------------------------------------
 *
 * TranslateSticker --
 *
 *	This procedure is called to move a sticker item by a
 *	given amount.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The position of the sticker item is offset by (xDelta, yDelta),
 *	and the bounding box is updated in the generic part of the
 *	item structure.
 *
 *--------------------------------------------------------------
 */

static void
TranslateSticker(canvas, itemPtr, deltaX, deltaY)
    Tk_Canvas canvas;			/* Canvas containing item. */
    Tk_Item *itemPtr;			/* Item that is being moved. */
    double deltaX, deltaY;		/* Amount by which item is to be
					 * moved. */
{
    StickerItem *stickerPtr = (StickerItem *) itemPtr;

    stickerPtr->bbox[0] += deltaX;
    stickerPtr->bbox[1] += deltaY;
    stickerPtr->bbox[2] += deltaX;
    stickerPtr->bbox[3] += deltaY;
    ComputeStickerBbox(canvas, stickerPtr);
}

/*
 *--------------------------------------------------------------
 *
 * StickerToPostscript --
 *
 *	This procedure is called to generate Postscript for
 *	sticker items.
 *
 * Results:
 *	The return value is a standard Tcl result.  If an error
 *	occurs in generating Postscript then an error message is
 *	left in interp->result, replacing whatever used to be there.
 *	If no error occurs, then Postscript for the sticker is
 *	appended to the result.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static int
StickerToPostscript(interp, canvas, itemPtr, prepass)
    Tcl_Interp *interp;			/* Interpreter for error reporting. */
    Tk_Canvas canvas;			/* Information about overall canvas. */
    Tk_Item *itemPtr;			/* Item for which Postscript is
					 * wanted. */
    int prepass;			/* 1 means this is a prepass to
					 * collect font information;  0 means
					 * final Postscript is being created. */
{
    StickerItem *stickerPtr = (StickerItem *) itemPtr;
    char buffer[500];
    double x1, x2, y1, y2;
    Tk_FontMetrics fmp;
    int i;

    /*
     * Suppress all drawing if size is out of the given range.
     * The check uses the X11 coordinate system to avoid rounding problems.
     */

    {
      short x1, y1, x2, y2;

      Tk_CanvasDrawableCoords(canvas, stickerPtr->bbox[0], stickerPtr->bbox[1],
	&x1, &y1);
      Tk_CanvasDrawableCoords(canvas, stickerPtr->bbox[2], stickerPtr->bbox[3],
	&x2, &y2);
      if (x2 <= x1) {
	  x2 = x1+1;
      }
      if (y2 <= y1) {
	  y2 = y1+1;
      }

       if (x2 - x1 < stickerPtr->minWidth  || x2 - x1 > stickerPtr->maxWidth ||
	   y2 - y1 < stickerPtr->minHeight || y2 - y1 > stickerPtr->maxHeight)
	return TCL_OK;
    }

    if (prepass) goto font_only;

    x1 = stickerPtr->bbox[0];
    y1 = Tk_CanvasPsY(canvas, stickerPtr->bbox[1]);
    x2 = stickerPtr->bbox[2];
    y2 = Tk_CanvasPsY(canvas, stickerPtr->bbox[3]);


    /*
     * First draw the filled areas of the rectangle.
     */

    sprintf(buffer, "%.15g %.15g moveto %.15g 0 rlineto 0 %.15g rlineto \
%.15g 0 rlineto closepath\n",
		x1, y1,
		x2-x1, y2-y1,
		x1-x2);

    if (stickerPtr->fillColor != NULL) {
	Tcl_AppendResult(interp, buffer, (char *) NULL);
	if (Tk_CanvasPsColor(interp, canvas, stickerPtr->fillColor) != TCL_OK) {
	    return TCL_ERROR;
	}
	if (stickerPtr->fillStipple != None) {
	    Tcl_AppendResult(interp, "clip ", (char *) NULL);
	    if (Tk_CanvasPsStipple(interp, canvas, stickerPtr->fillStipple)
			!= TCL_OK) {
		return TCL_ERROR;
	    }
	    if (stickerPtr->outlineColor != NULL) {
		Tcl_AppendResult(interp, "grestore gsave\n", (char *) NULL);
	    }
	} else {
	    Tcl_AppendResult(interp, "fill\n", (char *) NULL);
	}
    }

    if (stickerPtr->barColor != NULL) {
	char buffer[500];
	double y1, y2;
	double bx1, by1, bx2, by2;
	int w;

	y1 = stickerPtr->bbox[1];
	y2 = stickerPtr->bbox[3];

	w = stickerPtr->width;
	bx1 = x1 + w/2 + stickerPtr->relx * (x2 - x1 - w);
	if (bx1 < x1) bx1 = x1; else if (bx1 > x2) bx1 = x2;
	by1 = y1 + w/2 + stickerPtr->rely * (y2 - y1 - w);
	if (by1 < y1) by1 = y1; else if (by1 > y2) by1 = y2;
	bx2 = bx1 + stickerPtr->relwidth * (x2 - x1 - w);
	if (bx2 < x1) bx2 = x1; else if (bx2 > x2) bx2 = x2;
	by2 = by1 + stickerPtr->relheight * (y2 - y1 - w);
	if (by2 < y1) by2 = y1; else if (by2 > y2) by2 = y2;

	by1 = Tk_CanvasPsY(canvas, by1);
	by2 = Tk_CanvasPsY(canvas, by2);

	sprintf(buffer, "%.15g %.15g moveto %.15g 0 rlineto 0 %.15g rlineto \
%.15g 0 rlineto closepath\n",
		bx1, by1,
		bx2-bx1, by2-by1,
		bx1-bx2);

	Tcl_AppendResult(interp, buffer, (char *) NULL);
	if (Tk_CanvasPsColor(interp, canvas, stickerPtr->barColor) != TCL_OK) {
	    return TCL_ERROR;
	}
	if (stickerPtr->fillStipple != None) {
	    Tcl_AppendResult(interp, "clip ", (char *) NULL);
	    if (Tk_CanvasPsStipple(interp, canvas, stickerPtr->fillStipple)
			!= TCL_OK) {
		return TCL_ERROR;
	    }
	    if (stickerPtr->outlineColor != NULL) {
		Tcl_AppendResult(interp, "grestore gsave\n", (char *) NULL);
	    }
	} else {
	    Tcl_AppendResult(interp, "fill\n", (char *) NULL);
	}
    }

    /*
     * Now draw the outline, if there is one.
     */

    if (stickerPtr->outlineColor != NULL) {
        Tcl_AppendResult(interp, buffer, (char *) NULL);
        sprintf(buffer, "%d setlinewidth", stickerPtr->width);
        Tcl_AppendResult(interp, buffer,
                " 0 setlinejoin 2 setlinecap\n", (char *) NULL);
        if (Tk_CanvasPsColor(interp, canvas, stickerPtr->outlineColor)
                != TCL_OK) {
            return TCL_ERROR;
        }
        Tcl_AppendResult(interp, "stroke\n", (char *) NULL);
    }

    /*
     * Finally draw the text, if there is one, optionally repeated.
     */

font_only:
    if (stickerPtr->textColor != NULL && stickerPtr->visChars > 0) {
	if (Tk_CanvasPsFont(interp, canvas, stickerPtr->fontPtr) != TCL_OK) {
	    return TCL_ERROR;
	}
	if (prepass) return TCL_OK;
	if (Tk_CanvasPsColor(interp, canvas, stickerPtr->textColor) != TCL_OK) {
	    return TCL_ERROR;
	}
        Tk_GetFontMetrics(stickerPtr->fontPtr, &fmp);
	for (i = 0; i < stickerPtr->repeats; i++) {
	    static char* PSrotate[] = { "", "translate 90 rotate 0 0" };
	    sprintf(buffer, "gsave %.15g %.15g %s [\n",
		(double) stickerPtr->x + (stickerPtr->vertical ?
			fmp.descent :
				i * stickerPtr->horSize),
		Tk_CanvasPsY(canvas, (double) stickerPtr->y -
			(stickerPtr->vertical ? i * stickerPtr->horSize :
				- fmp.descent)),
		PSrotate[stickerPtr->vertical]);
	    Tcl_AppendResult(interp, buffer, (char *) NULL);
	    Tcl_AppendResult(interp, "    ", (char *) NULL);
	    LineToPostscript(interp, stickerPtr->start,
		stickerPtr->visChars);
	    Tcl_AppendResult(interp, "\n", (char *) NULL);
#ifdef ORIG /*CRRW: the following line was really a BUG  */
	    sprintf(buffer, "] %d %d 0 0 0 false DrawText grestore\n",
		stickerPtr->horSize,
		fmp.ascent + fmp.descent);
#else
	    sprintf(buffer, "] %d 0 0 0 false DrawText grestore\n",
		stickerPtr->horSize);
#endif
	    Tcl_AppendResult(interp, buffer, (char *) NULL);
	}
    }
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * LineToPostscript --
 *
 *	This procedure generates a parenthesized Postscript string
 *	describing one line of text.
 *
 * Results:
 *	None. The parenthesized string is appended to
 *	interp->result.  It generates proper backslash notation so
 *	that Postscript can interpret the string correctly.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static void
LineToPostscript(interp, string, numChars)
    Tcl_Interp *interp;		/* Interp whose result is to be appended to. */
    char *string;		/* String to Postscript-ify. */
    int numChars;		/* Number of characters in the string. */
{
#define BUFFER_SIZE 100
    char buffer[BUFFER_SIZE+5];
    int used, c;

    buffer[0] = '(';
    used = 1;

    for ( ; numChars > 0; string++, numChars--) {
	c = (*string) & 0xff;
	if ((c == '(') || (c == ')') || (c == '\\') || (c < 0x20)
		|| (c >= 0x7f)) {
	    /*
	     * Tricky point:  the "03" is necessary in the sprintf below,
	     * so that a full three digits of octal are always generated.
	     * Without the "03", a number following this sequence could
	     * be interpreted by Postscript as part of this sequence.
	     */
#ifndef ORIG
	    /*CRRW
             * mask out blanks ; adjust Characters to make the Umlauts
             * "working" for ISO FONTS ONLY
             */
	    if ( c == 0xc3 ) continue;
/*printf("CRRW: %x %o \n",c,c);*/
	     c= c + 0x40;
/*	     if (c >0xe7) c= c - 0x40;*/
/*printf("CRRW2: %x %o %c \n",c,c,c);*/
#endif
	    sprintf(buffer+used, "\\%03o", c);
	    used += strlen(buffer+used );

	} else {
	    buffer[used] = c;
	    used++;
	}
	if (used >= BUFFER_SIZE) {
	    buffer[used] = 0;
	    Tcl_AppendResult(interp, buffer, (char *) NULL);
	    used = 0;
	}
    }
    buffer[used] = ')';
    buffer[used+1] = 0;
    Tcl_AppendResult(interp, buffer, (char *) NULL);
}
