/* $XConsortium: TextAction.c,v 1.44 92/03/18 12:03:30 rws Exp $ */

/***********************************************************
Copyright 1989 by the Massachusetts Institute of Technology,
Cambridge, Massachusetts.

                        All Rights Reserved

Permission to use, copy, modify, and distribute this software and its 
documentation for any purpose and without fee is hereby granted, 
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in 
supporting documentation, and that the names of Digital or MIT not be
used in advertising or publicity pertaining to distribution of the
software without specific, written prior permission.  

DIGITAL DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
DIGITAL BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.

******************************************************************/

/*
 * NOTE:  There are some ASCII Dependancies on '\n' and '\0' that I
 *        am not too thrilled with.  There is also the implicit assumption
 *        that the individual characters will fit inside a "char".
 *        It would be nice if we could generalize this a but more.  If
 *        anyone out there comes up with an implementation of this stuff
 *        that has no dependency on ASCII, please send the code back to us.
 *
 *						Chris D. Peterson     
 *						MIT X Consortium 
 */

#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/Xatom.h>
#include <X11/Xmu/Misc.h>
#include <X11/Xaw3d/TextP.h>
#include <stdio.h>
#include <ctype.h>

#define SrcScan                XawTextSourceScan
#define SrcSearch              XawTextSourceSearch
#define FindDist               XawTextSinkFindDistance
#define FindPos                XawTextSinkFindPosition

/*
 * These are defined in TextPop.c
 */

void _XawTextWriteFileAction(), _XawTextWriteFile();
void _XawTextInsertFileAction(), _XawTextInsertFile(), _XawTextSearch();
void _XawTextSearch(), _XawTextDoSearchAction(), _XawTextDoReplaceAction();
void _XawTextSetField(), _XawTextPopdownSearchAction();

/*
 * These are defined in Text.c
 */

char * _XawTextGetText();
void _XawTextBuildLineTable(), _XawTextAlterSelection(), _XawTextVScroll();
void _XawTextSetSelection(), _XawTextCheckResize(), _XawTextExecuteUpdate();
void _XawTextSetScrollBars(), _XawTextClearAndCenterDisplay();
void _XawTextSaltAwaySelection(), _XawTextPrepareToUpdate();
Atom * _XawTextSelectionList();
int _XawTextReplace();

extern XtTranslations DefaultTrans, ViTrans, InsertTrans, FindTrans;


/*
 * These are defined in TextAction.c
 */
static int GetLine();


/*
 * This eventually calls the SrcScan macro, but it first saves a record
 * of cursor movement to be used in vi mode by the dot command to repeat
 * those movements later.
 */
XawTextPosition
ViSrcScan(ctx, position, type, dir, count, include)
TextWidget            ctx;
XawTextPosition       position;
XawTextScanType       type;
XawTextScanDirection  dir;
int                   count;
Boolean               include;
{
  if ((ctx->text.dot_mode == TRUE) && (ctx->text.dot_data.insert == FALSE))
  {
    ctx->text.dot_data.MovePtr->next = (struct Move_rec *)
				XtMalloc(sizeof(struct Move_rec));
    ctx->text.dot_data.MovePtr = ctx->text.dot_data.MovePtr->next;
    ctx->text.dot_data.MovePtr->search = 0;
    ctx->text.dot_data.MovePtr->type = type;
    ctx->text.dot_data.MovePtr->dir = dir;
    ctx->text.dot_data.MovePtr->count = count;
    ctx->text.dot_data.MovePtr->include = include;
    ctx->text.dot_data.MovePtr->next = NULL;
  }
  return(SrcScan(ctx->text.source, position, type, dir, count, include));
}

/*
 * This eventually calls the SrcSearch macro, but it first saves a record
 * of cursor movement to be used in vi mode by the dot command to repeat
 * those movements later.
 */
XawTextPosition
ViSrcSearch(ctx, position, dir, text)
TextWidget            ctx;
XawTextPosition       position;
XawTextScanDirection  dir;
XawTextBlock		*text;
{
  if ((ctx->text.dot_mode == TRUE) && (ctx->text.dot_data.insert == FALSE))
  {
    ctx->text.dot_data.MovePtr->next = (struct Move_rec *)
				XtMalloc(sizeof(struct Move_rec));
    ctx->text.dot_data.MovePtr = ctx->text.dot_data.MovePtr->next;
    ctx->text.dot_data.MovePtr->search = 1;
    ctx->text.dot_data.MovePtr->text = (char *)XtMalloc(text->length + 1);
    strncpy(ctx->text.dot_data.MovePtr->text, text->ptr, text->length);
    ctx->text.dot_data.MovePtr->text[text->length] = '\0';
    ctx->text.dot_data.MovePtr->dir = dir;
    ctx->text.dot_data.MovePtr->count = 1;
    ctx->text.dot_data.MovePtr->next = NULL;
  }
  return(SrcSearch(ctx->text.source, position, dir, text));
}

static void
StartAction(ctx, event)
TextWidget ctx;
XEvent *event;
{
  _XawTextPrepareToUpdate(ctx);
  if (event != NULL) {
    switch (event->type) {
    case ButtonPress:
    case ButtonRelease:
      ctx->text.time = event->xbutton.time;
      break;
    case KeyPress:
    case KeyRelease:
      ctx->text.time = event->xkey.time;
      break;
    case MotionNotify:
      ctx->text.time = event->xmotion.time;
      break;
    case EnterNotify:
    case LeaveNotify:
      ctx->text.time = event->xcrossing.time;
    }
  }
}

static void
NotePosition(ctx, event)
TextWidget ctx;
XEvent *event;
{
  switch (event->type) {
  case ButtonPress:
  case ButtonRelease:
    ctx->text.ev_x = event->xbutton.x;
    ctx->text.ev_y = event->xbutton.y;
    break;
  case KeyPress:
  case KeyRelease:
    {
	XRectangle cursor;
	XawTextSinkGetCursorBounds(ctx->text.sink, &cursor);
	ctx->text.ev_x = cursor.x + cursor.width / 2;;
	ctx->text.ev_y = cursor.y + cursor.height / 2;;
    }
    break;
  case MotionNotify:
    ctx->text.ev_x = event->xmotion.x;
    ctx->text.ev_y = event->xmotion.y;
    break;
  case EnterNotify:
  case LeaveNotify:
    ctx->text.ev_x = event->xcrossing.x;
    ctx->text.ev_y = event->xcrossing.y;
  }
}

static void
EndAction(ctx)
TextWidget ctx;
{
  _XawTextCheckResize(ctx);
  _XawTextExecuteUpdate(ctx);
  ctx->text.mult = 1;
  ctx->text.dyc_mode = 0;
  ctx->text.num_mode = FALSE;
  ctx->text.cut_buf = 1;
  ctx->text.dot_mode = FALSE;
}

/*
 * This function will completely clear the dot record used by vi mode.
 * The dot record stores all information necessary to completely repeat
 * any vi command.  This function should be called at the beginning
 * of any delete, change, yank, or put command.
 */
/*ARGSUSED*/
static void 
FreeDot(w)
Widget w;
{
	TextWidget ctx = (TextWidget)w;
	struct Move_rec *mptr1;
	struct Move_rec *mptr2;

	ctx->text.dot_data.buf = 1;
	ctx->text.dot_data.dyc_mode = 0;
	ctx->text.dot_data.nomove = 0;
	if (ctx->text.dot_data.insert == TRUE)
	{
		XtFree(ctx->text.dot_data.insert_text);
	}
	ctx->text.dot_data.insert = FALSE;
	ctx->text.dot_data.SetupMove.count = 0;
	mptr1 = ctx->text.dot_data.MoveList.next;
	while (mptr1 != NULL)
	{
		mptr2 = mptr1->next;
		if (mptr1->search)
		{
			XtFree(mptr1->text);
		}
		XtFree((char *)mptr1);
		mptr1 = mptr2;
	}
	ctx->text.dot_data.MoveList.next = NULL;
	ctx->text.dot_data.MovePtr = &(ctx->text.dot_data.MoveList);
}

/*
 * This function will undo the last action that resulted in the deletion
 * insertion, or change of any text.  Two undos in sequence leave the file
 * unchanged.
 */
/*ARGSUSED*/
static void 
UndoFunc(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
	TextWidget ctx = (TextWidget)w;
	char *ptr;
	XawTextPosition from, to;
	XawTextPosition new;
	XawTextPosition junk;
	int from_line, to_line, garbage;

	if ((ctx->text.undo_from != ctx->text.undo_to)||
	    (ctx->text.undo_text.length != 0))
	{
		StartAction(ctx, event);
		from_line = GetLine(ctx, ctx->text.undo_from);
		to_line = GetLine(ctx, ctx->text.undo_to);
		ptr = _XawTextGetText(ctx, ctx->text.undo_from,
				ctx->text.undo_to);
		if (_XawTextReplace(ctx, ctx->text.undo_from, ctx->text.undo_to,
				&(ctx->text.undo_text)))
		{
			XtFree(ptr);
			XBell(XtDisplay(ctx), 50);
			return;
		}
		if (ctx->text.undo_text.length != 0)
		{
			ctx->text.undo_to = SrcScan(ctx->text.source,
				ctx->text.undo_from, XawstPositions, 
				XawsdRight, ctx->text.undo_text.length, TRUE);
		}
		else
		{
			ctx->text.undo_to = ctx->text.undo_from;
		}
		ctx->text.undo_text.firstPos = 0;
		ctx->text.undo_text.format = FMT8BIT;
		if (ctx->text.undo_text.ptr != NULL)
		{
			XtFree(ctx->text.undo_text.ptr);
			ctx->text.undo_text.ptr = NULL;
		}
		ctx->text.undo_text.ptr = ptr;
		ctx->text.undo_text.length = strlen(ptr);
		ctx->text.insertPos = ctx->text.undo_from;
		ctx->text.showposition = TRUE; 
		new = SrcScan(ctx->text.source, ctx->text.insertPos,
			XawstEOL, XawsdLeft, 1, FALSE);
		FindDist(ctx->text.sink, new, ctx->text.margin.left,
			ctx->text.insertPos, &(ctx->text.from_left),
			&junk, &garbage);
		if ((from_line > (ctx->text.lt.lines - 2))||
		    (to_line > (ctx->text.lt.lines - 2)))
		{
			XawTextDisplay(w);
		}
		EndAction(ctx);
	}
	else
	{
		XBell(XtDisplay(ctx), 50);
	}
}


struct _SelectionList {
    String *params;
    Cardinal count;
    Time time;
};

static void GetSelection();

/* ARGSUSED */
static void 
_SelectionReceived(w, client_data, selection, type, value, length, format)
Widget w;
XtPointer client_data;
Atom *selection, *type;
XtPointer value;
unsigned long *length;
int *format;
{
  TextWidget ctx = (TextWidget)w;
  XawTextBlock text;
  
#ifdef MOTOROLA_BROKEN_CC
  if (*type == 0 || *length == 0) {
#else
  if (*type == 0 /*XT_CONVERT_FAIL*/ || *length == 0) {
#endif
    struct _SelectionList* list = (struct _SelectionList*)client_data;
    if (list != NULL) {
      GetSelection(w, list->time, list->params, list->count);
      XtFree(client_data);
    }
    return;
  }
  
  StartAction(ctx, (XEvent *)NULL);
  text.ptr = (char*)value;
  text.firstPos = 0;
  text.length = *length;
  text.format = FMT8BIT;
  if (_XawTextReplace(ctx, ctx->text.insertPos, ctx->text.insertPos, &text)) {
    XBell(XtDisplay(ctx), 0);
    return;
  }
  ctx->text.insertPos = SrcScan(ctx->text.source, 
				(XawTextPosition) ctx->text.insertPos, 
				XawstPositions, XawsdRight, text.length, TRUE);

  EndAction(ctx);
  _XawTextSetScrollBars(ctx);
  XtFree(client_data);
  XFree(value);		/* the selection value should be freed with XFree */
}

static void 
GetSelection(w, time, params, num_params)
Widget w;
Time time;
String *params;			/* selections in precedence order */
Cardinal num_params;
{
    Atom selection;
    int buffer;

    selection = XInternAtom(XtDisplay(w), *params, False);
    switch (selection) {
      case XA_CUT_BUFFER0: buffer = 0; break;
      case XA_CUT_BUFFER1: buffer = 1; break;
      case XA_CUT_BUFFER2: buffer = 2; break;
      case XA_CUT_BUFFER3: buffer = 3; break;
      case XA_CUT_BUFFER4: buffer = 4; break;
      case XA_CUT_BUFFER5: buffer = 5; break;
      case XA_CUT_BUFFER6: buffer = 6; break;
      case XA_CUT_BUFFER7: buffer = 7; break;
      default:	       buffer = -1;
    }
    if (buffer >= 0) {
	int nbytes;
	unsigned long length;
	int fmt8 = 8;
	Atom type = XA_STRING;
	char *line = XFetchBuffer(XtDisplay(w), &nbytes, buffer);
	if ((length = nbytes))
	    _SelectionReceived(w, (XtPointer) NULL, &selection, &type, 
						   (caddr_t)line, &length, &fmt8);
	else if (num_params > 1)
	    GetSelection(w, time, params+1, num_params-1);
    } else {
	struct _SelectionList* list;
	if (--num_params) {
	    list = XtNew(struct _SelectionList);
	    list->params = params + 1;
	    list->count = num_params;
	    list->time = time;
	} else list = NULL;
	XtGetSelectionValue(w, selection, XA_STRING, _SelectionReceived,
			    (XtPointer)list, time);
    }
}

static void 
InsertSelection(w, event, params, num_params)
Widget w;
XEvent *event;
String *params;		/* precedence list of selections to try */
Cardinal *num_params;
{
  StartAction((TextWidget)w, event); /* Get Time. */
  GetSelection(w, ((TextWidget)w)->text.time, params, *num_params);
  EndAction((TextWidget)w);
}

/*
 * These static strings are needed because the XmuInternStrings call just takes
 * the pointer to a string passed to it, and does not make its own local
 * copy.
 */
static char *BufNames[] = {
	"CUT_BUFFER0",
	"CUT_BUFFER1",
	"CUT_BUFFER2",
	"CUT_BUFFER3",
	"CUT_BUFFER4",
	"CUT_BUFFER5",
	"CUT_BUFFER6",
	"CUT_BUFFER7"
};

/*
 * A variation of InsertSelection that is used by the vi mode.
 * It behave the same as insert selection except for two differences.
 * It inserts from whatever cut buffer is in ctx->text.cut_buf, instead of
 * always from cut buffer 1, and it does insertions differently depending on
 * whether or not the inserted text has a <CR> in it.
 *
 * This particular function will insert text with a <CR> before the current
 * line.
 */
static void 
InsertChosenSelectionBefore(w, event, params, num_params)
Widget w;
XEvent *event;
String *params;		/* precedence list of selections to try */
Cardinal *num_params;
{
  TextWidget ctx = (TextWidget)w;
  XawTextPosition new;
  char *bptr;
  Atom selection;
  int i, isline, buffer, nbytes;
  char *text;

  StartAction((TextWidget)w, event); /* Get Time. */
  if ((ctx->text.cut_buf >= 0)&&(ctx->text.cut_buf < 8))
  {
	bptr = BufNames[ctx->text.cut_buf];
  }
  else
  {
	bptr = BufNames[1];
  }
  ctx->text.undo_text.firstPos = 0;
  ctx->text.undo_text.format = FMT8BIT;
  if (ctx->text.undo_text.ptr != NULL)
  {
    XtFree(ctx->text.undo_text.ptr);
    ctx->text.undo_text.ptr = NULL;
  }
  ctx->text.undo_text.length = 0;
  FreeDot(w);
  ctx->text.dot_data.insert = TRUE;
  ctx->text.dot_mode = TRUE;

  selection = XInternAtom(XtDisplay(w), bptr, False);
  switch (selection) {
      case XA_CUT_BUFFER0: buffer = 0; break;
      case XA_CUT_BUFFER1: buffer = 1; break;
      case XA_CUT_BUFFER2: buffer = 2; break;
      case XA_CUT_BUFFER3: buffer = 3; break;
      case XA_CUT_BUFFER4: buffer = 4; break;
      case XA_CUT_BUFFER5: buffer = 5; break;
      case XA_CUT_BUFFER6: buffer = 6; break;
      case XA_CUT_BUFFER7: buffer = 7; break;
      default:	       buffer = -1;
  }
  text = XFetchBuffer(XtDisplay(w), &nbytes, buffer);
  isline = 0;
  for (i=0; i<nbytes; i++)
  {
    if (text[i] == '\n')
    {
	isline = 1;
	break;
    }
  }
  if (isline)
  {
    ctx->text.insertPos = SrcScan(ctx->text.source, ctx->text.insertPos,
				XawstEOL, XawsdLeft, 1, FALSE);
    ctx->text.dot_data.SetupMove.search = 0;
    ctx->text.dot_data.SetupMove.type = XawstEOL;
    ctx->text.dot_data.SetupMove.dir = XawsdLeft;
    ctx->text.dot_data.SetupMove.count = 1;
    ctx->text.dot_data.SetupMove.include = FALSE;
    ctx->text.dot_data.SetupMove.next = NULL;
    ctx->text.dot_data.nomove = 2;
  }
  ctx->text.undo_from = ctx->text.insertPos;
  GetSelection(w, ((TextWidget)w)->text.time, &bptr, 1);
  ctx->text.undo_to = ctx->text.insertPos;
  EndAction((TextWidget)w);
  /*
   * We need to start a new action here so we can move the cursor to the
   * beginning of the inserted text, and have its position update properly.
   */
  StartAction((TextWidget)w, event); /* Get Time. */
  if (ctx->text.undo_from < ctx->text.undo_to)
  {
    ctx->text.dot_data.insert_text = _XawTextGetText(ctx, ctx->text.undo_from,
					ctx->text.undo_to);
  }
  else
  {
    ctx->text.dot_data.insert = FALSE;
  }
  if (isline)
  {
    ctx->text.insertPos = ctx->text.undo_from;
    ctx->text.from_left = 0;
  }
  else
  {
    XawTextPosition junk;
    int garbage;

    new = SrcScan(ctx->text.source, ctx->text.insertPos,
			XawstEOL, XawsdLeft, 1, FALSE);
    FindDist(ctx->text.sink, new, ctx->text.margin.left,
		ctx->text.insertPos, &(ctx->text.from_left), &junk, &garbage);
  }
  EndAction((TextWidget)w);
  _XawTextSetScrollBars(ctx);
}

/*
 * Functions identical to InsertChosenSelectionBefore(), except if the text to
 * be inserted has a <CR> it inserts it after the current line.
 */
static void 
InsertChosenSelectionAfter(w, event, params, num_params)
Widget w;
XEvent *event;
String *params;		/* precedence list of selections to try */
Cardinal *num_params;
{
  TextWidget ctx = (TextWidget)w;
  XawTextPosition new;
  char *bptr;
  Atom selection;
  int i, isline, buffer, nbytes;
  char *text;

  StartAction((TextWidget)w, event); /* Get Time. */
  if ((ctx->text.cut_buf >= 0)&&(ctx->text.cut_buf < 8))
  {
	bptr = BufNames[ctx->text.cut_buf];
  }
  else
  {
	bptr = BufNames[1];
  }
  ctx->text.undo_text.firstPos = 0;
  ctx->text.undo_text.format = FMT8BIT;
  if (ctx->text.undo_text.ptr != NULL)
  {
    XtFree(ctx->text.undo_text.ptr);
    ctx->text.undo_text.ptr = NULL;
  }
  ctx->text.undo_text.length = 0;
  FreeDot(w);
  ctx->text.dot_data.insert = TRUE;
  ctx->text.dot_mode = TRUE;

  selection = XInternAtom(XtDisplay(w), bptr, False);
  switch (selection) {
      case XA_CUT_BUFFER0: buffer = 0; break;
      case XA_CUT_BUFFER1: buffer = 1; break;
      case XA_CUT_BUFFER2: buffer = 2; break;
      case XA_CUT_BUFFER3: buffer = 3; break;
      case XA_CUT_BUFFER4: buffer = 4; break;
      case XA_CUT_BUFFER5: buffer = 5; break;
      case XA_CUT_BUFFER6: buffer = 6; break;
      case XA_CUT_BUFFER7: buffer = 7; break;
      default:	       buffer = -1;
  }
  text = XFetchBuffer(XtDisplay(w), &nbytes, buffer);
  isline = 0;
  for (i=0; i<nbytes; i++)
  {
    if (text[i] == '\n')
    {
	isline = 1;
	break;
    }
  }
  if (isline)
  {
    ctx->text.insertPos = SrcScan(ctx->text.source, ctx->text.insertPos,
				XawstEOL, XawsdRight, 1, TRUE);
    ctx->text.dot_data.SetupMove.search = 0;
    ctx->text.dot_data.SetupMove.type = XawstEOL;
    ctx->text.dot_data.SetupMove.dir = XawsdRight;
    ctx->text.dot_data.SetupMove.count = 1;
    ctx->text.dot_data.SetupMove.include = TRUE;
    ctx->text.dot_data.SetupMove.next = NULL;
    ctx->text.dot_data.nomove = 2;
  }
  ctx->text.undo_from = ctx->text.insertPos;
  GetSelection(w, ((TextWidget)w)->text.time, &bptr, 1);
  ctx->text.undo_to = ctx->text.insertPos;
  EndAction((TextWidget)w);
  /*
   * We need to start a new action here so we can move the cursor to the
   * beginning of the inserted text, and have its position update properly.
   */
  StartAction((TextWidget)w, event); /* Get Time. */
  if (ctx->text.undo_from < ctx->text.undo_to)
  {
    ctx->text.dot_data.insert_text = _XawTextGetText(ctx, ctx->text.undo_from,
					ctx->text.undo_to);
  }
  else
  {
    ctx->text.dot_data.insert = FALSE;
  }
  if (isline)
  {
    ctx->text.insertPos = ctx->text.undo_from;
    ctx->text.from_left = 0;
  }
  else
  {
    XawTextPosition junk;
    int garbage;

    new = SrcScan(ctx->text.source, ctx->text.insertPos,
			XawstEOL, XawsdLeft, 1, FALSE);
    FindDist(ctx->text.sink, new, ctx->text.margin.left,
		ctx->text.insertPos, &(ctx->text.from_left), &junk, &garbage);
  }
  EndAction((TextWidget)w);
  _XawTextSetScrollBars(ctx);
}

/************************************************************
 *
 * Routines for Moving Around.
 *
 ************************************************************/

static void
Move(ctx, event, dir, type, include)
TextWidget ctx;
XEvent *event;
XawTextScanDirection dir;
XawTextScanType type;
Boolean include;
{
  StartAction(ctx, event);
  ctx->text.insertPos = SrcScan(ctx->text.source, ctx->text.insertPos,
				type, dir, ctx->text.mult, include);
  EndAction(ctx);
}

/*ARGSUSED*/
static void 
MoveForwardChar(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
   Move((TextWidget) w, event, XawsdRight, XawstPositions, TRUE);
}

/*ARGSUSED*/
static void 
MoveBackwardChar(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  Move((TextWidget) w, event, XawsdLeft, XawstPositions, TRUE);
}

/*ARGSUSED*/
static void 
MoveForwardWord(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  Move((TextWidget) w, event, XawsdRight, XawstWhiteSpace, FALSE);
}

/*ARGSUSED*/
static void 
MoveBackwardWord(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  Move((TextWidget) w, event, XawsdLeft, XawstWhiteSpace, FALSE);
}

/*ARGSUSED*/
static void MoveForwardParagraph(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  Move((TextWidget) w, event, XawsdRight, XawstParagraph, FALSE);
}

/*ARGSUSED*/
static void MoveBackwardParagraph(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  Move((TextWidget) w, event, XawsdLeft, XawstParagraph, FALSE);
}

/*ARGSUSED*/
static void 
MoveToLineEnd(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  Move((TextWidget) w, event, XawsdRight, XawstEOL, FALSE);
}

/*ARGSUSED*/
static void 
MoveToLineStart(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  Move((TextWidget) w, event, XawsdLeft, XawstEOL, FALSE);
}


static void
MoveLine(ctx, event, dir)
TextWidget ctx;
XEvent *event;
XawTextScanDirection dir;
{
  XawTextPosition new, next_line, junk;
  int from_left, garbage;

  StartAction(ctx, event);

  if (dir == XawsdLeft)
    ctx->text.mult++;

  new = SrcScan(ctx->text.source, ctx->text.insertPos,
		XawstEOL, XawsdLeft, 1, FALSE);

  FindDist(ctx->text.sink, new, ctx->text.margin.left, ctx->text.insertPos,
	   &from_left, &junk, &garbage);

  new = SrcScan(ctx->text.source, ctx->text.insertPos, XawstEOL, dir,
		ctx->text.mult, (dir == XawsdRight));

  next_line = SrcScan(ctx->text.source, new, XawstEOL, XawsdRight, 1, FALSE);

  FindPos(ctx->text.sink, new, ctx->text.margin.left, from_left, FALSE,
	  &(ctx->text.insertPos), &garbage, &garbage);
  
  if (ctx->text.insertPos > next_line)
    ctx->text.insertPos = next_line;

  EndAction(ctx);
}

/*ARGSUSED*/
static void 
MoveNextLine(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  MoveLine( (TextWidget) w, event, XawsdRight);
}

/*ARGSUSED*/
static void 
MovePreviousLine(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  MoveLine( (TextWidget) w, event, XawsdLeft);
}

/*ARGSUSED*/
static void
MoveBeginningOfFile(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  Move((TextWidget) w, event, XawsdLeft, XawstAll, TRUE);
}

/*ARGSUSED*/
static void 
MoveEndOfFile(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  Move((TextWidget) w, event, XawsdRight, XawstAll, TRUE);
}

static void 
Scroll(ctx, event, dir)
TextWidget ctx;
XEvent *event;
XawTextScanDirection dir;
{
  StartAction(ctx, event);

  if (dir == XawsdLeft)
    _XawTextVScroll(ctx, ctx->text.mult);
  else
    _XawTextVScroll(ctx, -ctx->text.mult);

  EndAction(ctx);
}

/*ARGSUSED*/
static void 
ScrollOneLineUp(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  Scroll( (TextWidget) w, event, XawsdLeft);
}

/*ARGSUSED*/
static void 
ScrollOneLineDown(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  Scroll( (TextWidget) w, event, XawsdRight);
}

static void 
MovePage(ctx, event, dir)
TextWidget ctx;
XEvent *event;
XawTextScanDirection dir;
{
  int scroll_val = Max(1, ctx->text.lt.lines - 2);

  if (dir == XawsdLeft)
    scroll_val = -scroll_val;

  StartAction(ctx, event);
  _XawTextVScroll(ctx, scroll_val);
  ctx->text.insertPos = ctx->text.lt.top;
  EndAction(ctx);
}

/*ARGSUSED*/
static void 
MoveNextPage(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  MovePage((TextWidget) w, event, XawsdRight);
}

/*ARGSUSED*/
static void 
MovePreviousPage(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  MovePage((TextWidget) w, event, XawsdLeft);
}

static void 
MoveNextHalfPage(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  TextWidget ctx = (TextWidget)w;
  int scroll_val = Max(1, ctx->text.lt.lines - 2) / 2;

  StartAction(ctx, event);
  _XawTextVScroll(ctx, scroll_val);
  ctx->text.insertPos = ctx->text.lt.top;
  EndAction(ctx);
}

static void 
MovePreviousHalfPage(w, event, p, n)
TextWidget w;
XEvent *event;
String *p;
Cardinal *n;
{
  TextWidget ctx = (TextWidget)w;
  int scroll_val = Max(1, ctx->text.lt.lines - 2) / 2;

  scroll_val = -scroll_val;
  StartAction(ctx, event);
  _XawTextVScroll(ctx, scroll_val);
  ctx->text.insertPos = ctx->text.lt.top;
  EndAction(ctx);
}

/************************************************************
 *
 * Vi routines for Moving Around.
 *
 ************************************************************/

/*
 * This special routine is called after every movement command.  It is used
 * to do necessary functions depending on whether the command is a delete,
 * yank, change, or just a normal movement command.
 */
static void
ProcessExtraFlags(ctx, pos, dir)
TextWidget ctx;
XawTextPosition pos;
XawTextScanDirection dir;
{
	int garbage;
	XawTextPosition new, junk;
	XawTextBlock text;
	char *ptr;

	if ((ctx->text.dot_mode == TRUE)&&(ctx->text.dot_data.insert == FALSE))
	{
		ctx->text.dot_data.buf = ctx->text.cut_buf;
	}
	if (ctx->text.dyc_mode & (VI_DELETE|VI_CHANGE))
	{
		if (dir == XawsdLeft)
		{
			ctx->text.del_to = ctx->text.del_from;
			ctx->text.del_from = pos;
		}
		else 
		{
			ctx->text.del_to = pos;
		}
		ctx->text.undo_from = ctx->text.del_from;
		ctx->text.undo_to = ctx->text.del_from;
		ctx->text.undo_text.firstPos = 0;
		ctx->text.undo_text.format = FMT8BIT;
		if (ctx->text.undo_text.ptr != NULL)
		{
			XtFree(ctx->text.undo_text.ptr);
			ctx->text.undo_text.ptr = NULL;
		}
		ctx->text.undo_text.length = 0;
		if (ctx->text.del_from < ctx->text.del_to)
		{
			ptr = _XawTextGetText(ctx, ctx->text.del_from,
				ctx->text.del_to);
			XStoreBuffer(XtDisplay(ctx), ptr, strlen(ptr),
					ctx->text.cut_buf);
			ctx->text.undo_text.ptr = ptr;
			ctx->text.undo_text.length = strlen(ptr);
		}
		text.length = 0;
		text.firstPos = 0;
		if (_XawTextReplace(ctx, ctx->text.del_from,
				ctx->text.del_to, &text))
		{
			XBell(XtDisplay(ctx), 50);
			return;
		}
		ctx->text.insertPos = ctx->text.del_from;
		ctx->text.showposition = TRUE; 
		_XawTextSetScrollBars(ctx);
		if (ctx->text.dyc_mode & VI_CHANGE)
		{
			ctx->text.replace_cnt = 0;
			if (ctx->text.dot_mode == TRUE)
			{
				ctx->text.dot_data.insert = TRUE;
			}
			XtOverrideTranslations((Widget)ctx, InsertTrans);
			ctx->text.last_trans = InsertTrans;
		}
		else
		{
			ctx->text.undo_to = ctx->text.del_from;
		}
		ctx->text.dyc_mode = 0;
	}
	else if (ctx->text.dyc_mode & VI_YANK)
	{
		ctx->text.insertPos = ctx->text.del_from;
		if (dir == XawsdLeft)
		{
			ctx->text.del_to = ctx->text.del_from;
			ctx->text.del_from = pos;
		}
		else 
		{
			ctx->text.del_to = pos;
		}
		if (ctx->text.del_from < ctx->text.del_to)
		{
			ptr = _XawTextGetText(ctx, ctx->text.del_from,
				ctx->text.del_to);
			XStoreBuffer(XtDisplay(ctx), ptr, strlen(ptr),
					ctx->text.cut_buf);
			XtFree(ptr);
		}
		ctx->text.dyc_mode = 0;
	}
	/*
	 * This remembers how far from the left margin the cursor is.
	 */
	new = SrcScan(ctx->text.source, ctx->text.insertPos,
			XawstEOL, XawsdLeft, 1, FALSE);
	FindDist(ctx->text.sink, new, ctx->text.margin.left,
		ctx->text.insertPos, &(ctx->text.from_left), &junk, &garbage);
}

static void
ViMove(ctx, event, dir, type, include)
TextWidget ctx;
XEvent *event;
XawTextScanDirection dir;
XawTextScanType type;
Boolean include;
{
  StartAction(ctx, event);
  ctx->text.insertPos = ViSrcScan(ctx, ctx->text.insertPos,
				type, dir, ctx->text.mult, include);
  ProcessExtraFlags(ctx, ctx->text.insertPos, dir);
  EndAction(ctx);
}

/*ARGSUSED*/
static void 
ViForwardChar(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  ViMove((TextWidget) w, event, XawsdRight, XawstPositions, TRUE);
}

/*ARGSUSED*/
static void 
ViBackwardChar(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  ViMove((TextWidget) w, event, XawsdLeft, XawstPositions, TRUE);
}

/*ARGSUSED*/
static void 
ViEndWord(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  ViMove((TextWidget) w, event, XawsdRight, XawstWhiteSpace, FALSE);
}

/*
 * vi's idea of forward a word requires some special treatment here.
 * the dot command looks at the direction of the first move to determine
 * the overall direction of this command.  The overall direction of a
 * forward word command is right, but the first move is left.  So we add
 * an extra move right of 1 character position at the beginning just so
 * dot won't get confused.
 *
 * Also, the vi 'cw' and 'dw' commands affect different amounts of text
 * so we give them seperate treatment here.
 */
/*ARGSUSED*/
static void 
ViForwardWord(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  TextWidget ctx = (TextWidget)w;

  if (ctx->text.dyc_mode & VI_CHANGE)
  {
    ViMove((TextWidget) w, event, XawsdRight, XawstWhiteSpace, FALSE);
  }
  else
  {
    StartAction(ctx, event);
    ctx->text.insertPos = ViSrcScan(ctx, ctx->text.insertPos,
				XawstPositions, XawsdRight,
				1, TRUE);
    ctx->text.insertPos = ViSrcScan(ctx, ctx->text.insertPos,
				XawstWhiteSpace, XawsdLeft,
				1, FALSE);
    ctx->text.insertPos = ViSrcScan(ctx, ctx->text.insertPos,
				XawstWhiteSpace, XawsdRight,
				ctx->text.mult + 1, FALSE);
    ctx->text.insertPos = ViSrcScan(ctx, ctx->text.insertPos,
				XawstWhiteSpace, XawsdLeft,
				1, FALSE);
    ProcessExtraFlags(ctx, ctx->text.insertPos, XawsdRight);
    EndAction(ctx);
  }
}

/*ARGSUSED*/
static void 
ViBackwardWord(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  ViMove((TextWidget) w, event, XawsdLeft, XawstWhiteSpace, FALSE);
}

/*ARGSUSED*/
static void
ViForwardParagraph(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  ViMove((TextWidget) w, event, XawsdRight, XawstParagraph, FALSE);
}

/*ARGSUSED*/
static void
ViBackwardParagraph(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  ViMove((TextWidget) w, event, XawsdLeft, XawstParagraph, FALSE);
}

/*
 * In vi end-of-line is a special location.  We kludge this by setting
 * from_left to an abitrarily large value so we will always be at the
 * end of any line.
/*ARGSUSED*/
static void 
ViToLineEnd(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  TextWidget ctx = (TextWidget)w;
  ViMove((TextWidget) w, event, XawsdRight, XawstEOL, FALSE);
  ctx->text.from_left = 999999;
}

/*ARGSUSED*/
static void 
ViToLineStart(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  ViMove((TextWidget) w, event, XawsdLeft, XawstEOL, FALSE);
}


static void
ViMoveLine(ctx, event, dir)
TextWidget ctx;
XEvent *event;
XawTextScanDirection dir;
{
  XawTextPosition new, next_line, junk;
  int from_left, garbage;

  StartAction(ctx, event);

  if (dir == XawsdLeft)
    ctx->text.mult++;

  from_left = ctx->text.from_left;

  new = ViSrcScan(ctx, ctx->text.insertPos, XawstEOL, dir,
		ctx->text.mult, (dir == XawsdRight));

  next_line = SrcScan(ctx->text.source, new, XawstEOL, XawsdRight, 1, FALSE);

  FindPos(ctx->text.sink, new, ctx->text.margin.left, from_left, FALSE,
	  &(ctx->text.insertPos), &garbage, &garbage);
  
  if (ctx->text.insertPos > next_line)
    ctx->text.insertPos = next_line;

  ProcessExtraFlags(ctx, ctx->text.insertPos, dir);
  ctx->text.from_left = from_left;
  EndAction(ctx);
}

/*ARGSUSED*/
static void 
ViNextLine(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  ViMoveLine( (TextWidget) w, event, XawsdRight);
}

/*ARGSUSED*/
static void 
ViPreviousLine(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  ViMoveLine( (TextWidget) w, event, XawsdLeft);
}

/*
 * This is a special vi function that goes to the beginning of the first
 * word on the next line.
 */
/*ARGSUSED*/
static void 
ViNextLineWord(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  TextWidget ctx = (TextWidget)w;
  XawTextPosition new, next_line;

  StartAction(ctx, event);
  new = ViSrcScan(ctx, ctx->text.insertPos,
		XawstEOL, XawsdRight, ctx->text.mult, TRUE);
  next_line = SrcScan(ctx->text.source, new,
		XawstEOL, XawsdRight, 1, FALSE);
  ctx->text.insertPos = ViSrcScan(ctx, new,
		XawstWhiteSpace, XawsdRight, 1, FALSE);
  ctx->text.insertPos = ViSrcScan(ctx, ctx->text.insertPos,
		XawstWhiteSpace, XawsdLeft, 1, FALSE);
  if ((ctx->text.insertPos > next_line)||(ctx->text.insertPos < new))
  {
    ctx->text.insertPos = new;
  }
  ProcessExtraFlags(ctx, ctx->text.insertPos, XawsdRight);
  EndAction(ctx);
}

/*
 * This is a special vi function that goes to the beginning of the first
 * word on the previous line.
 */
/*ARGSUSED*/
static void 
ViPreviousLineWord(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  TextWidget ctx = (TextWidget)w;
  XawTextPosition new, next_line;

  StartAction(ctx, event);
  new = ViSrcScan(ctx, ctx->text.insertPos,
		XawstEOL, XawsdLeft, ctx->text.mult + 1, FALSE);
  next_line = SrcScan(ctx->text.source, new,
		XawstEOL, XawsdRight, 1, FALSE);
  ctx->text.insertPos = ViSrcScan(ctx, new,
		XawstWhiteSpace, XawsdRight, 1, FALSE);
  ctx->text.insertPos = ViSrcScan(ctx, ctx->text.insertPos,
		XawstWhiteSpace, XawsdLeft, 1, FALSE);
  if ((ctx->text.insertPos > next_line)||(ctx->text.insertPos < new))
  {
    ctx->text.insertPos = new;
  }
  ProcessExtraFlags(ctx, ctx->text.insertPos, XawsdLeft);
  EndAction(ctx);
}

/*ARGSUSED*/
static void
ViBeginningOfFile(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  ViMove((TextWidget) w, event, XawsdLeft, XawstAll, TRUE);
}

/*ARGSUSED*/
static void 
ViEndOfFile(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  ViMove((TextWidget) w, event, XawsdRight, XawstAll, TRUE);
}

/*
 * Special code to allow the G commant to take an arbitrary count,
 * while haveing it default to end-of-file
 * This function also puts the cursor at the first word on the line
 * moved to.
 * This function also adds 1 character movements to the beginnign of the
 * movement list to not confuse the dot command.
 */
/*ARGSUSED*/
static void 
ViLineInFile(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  TextWidget ctx = (TextWidget)w;
  int count;
  XawTextPosition new, next_line, junk;
  XawTextPosition old;
  int from_left, garbage;
  struct Move_rec *mptr;

  if (ctx->text.num_mode == TRUE)
  {
    StartAction(ctx, event);
    old = ctx->text.insertPos;
    ctx->text.insertPos = ViSrcScan(ctx, ctx->text.insertPos,
				XawstAll, XawsdLeft, 1, TRUE);
    new = ctx->text.insertPos;
    if (ctx->text.mult > 1)
    {
      new = ViSrcScan(ctx, ctx->text.insertPos,
		XawstEOL, XawsdRight, ctx->text.mult - 1, TRUE);
    }
    next_line = SrcScan(ctx->text.source, new,
		XawstEOL, XawsdRight, 1, FALSE);
    ctx->text.insertPos = ViSrcScan(ctx, new,
		XawstWhiteSpace, XawsdRight, 1, FALSE);
    ctx->text.insertPos = ViSrcScan(ctx, ctx->text.insertPos,
		XawstWhiteSpace, XawsdLeft, 1, FALSE);
    if (ctx->text.insertPos > next_line)
    {
      ctx->text.insertPos = new;
    }

    if (ctx->text.insertPos > old)
    {
      if (ctx->text.dot_mode == TRUE)
      {
	mptr = (struct Move_rec *)XtMalloc(sizeof(struct Move_rec));
	mptr->type = XawstPositions;
	mptr->dir = XawsdRight;
	mptr->count = 1;
	mptr->include = TRUE;
	mptr->next = ctx->text.dot_data.MoveList.next;
	ctx->text.dot_data.MoveList.next = mptr;
      }
      ProcessExtraFlags(ctx, ctx->text.insertPos, XawsdRight);
    }
    else
    {
      if (ctx->text.dot_mode == TRUE)
      {
	mptr = (struct Move_rec *)XtMalloc(sizeof(struct Move_rec));
	mptr->type = XawstPositions;
	mptr->dir = XawsdLeft;
	mptr->count = 1;
	mptr->include = TRUE;
	mptr->next = ctx->text.dot_data.MoveList.next;
	ctx->text.dot_data.MoveList.next = mptr;
      }
      ProcessExtraFlags(ctx, ctx->text.insertPos, XawsdLeft);
    }
    EndAction(ctx);
  }
  else
  {
    ViEndOfFile(w, event, p, n);
  }
}


/************************************************************
 *
 * Delete Routines.
 *
 ************************************************************/

static void 
_DeleteOrKill(ctx, from, to, kill)
TextWidget ctx;
XawTextPosition from, to;
Boolean	kill;
{
  XawTextBlock text;
  char *ptr;
  
  if (kill && from < to) {
    ptr = _XawTextGetText(ctx, from, to);
    XStoreBuffer(XtDisplay(ctx), ptr, strlen(ptr), ctx->text.cut_buf);
    XtFree(ptr);
  }
  text.length = 0;
  text.firstPos = 0;
  if (_XawTextReplace(ctx, from, to, &text)) {
    XBell(XtDisplay(ctx), 50);
    return;
  }
  ctx->text.insertPos = from;
  ctx->text.showposition = TRUE; 
}

static void
DeleteOrKill(ctx, event, dir, type, include, kill)
TextWidget	   ctx;
XEvent *event;
XawTextScanDirection dir;
XawTextScanType type;
Boolean	   include, kill;
{
  XawTextPosition from, to;
  
  StartAction(ctx, event);
  to = SrcScan(ctx->text.source, ctx->text.insertPos,
	       type, dir, ctx->text.mult, include);

/*
 * If no movement actually happened, then bump the count and try again. 
 * This causes the character position at the very beginning and end of 
 * a boundry to act correctly. 
 */

  if (to == ctx->text.insertPos)
      to = SrcScan(ctx->text.source, ctx->text.insertPos,
		   type, dir, ctx->text.mult + 1, include);

  if (dir == XawsdLeft) {
    from = to;
    to = ctx->text.insertPos;
  }
  else 
    from = ctx->text.insertPos;

  _DeleteOrKill(ctx, from, to, kill);
  _XawTextSetScrollBars(ctx);
  EndAction(ctx);
}

/*ARGSUSED*/
static void 
DeleteForwardChar(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  DeleteOrKill((TextWidget) w, event, XawsdRight, XawstPositions, TRUE, FALSE);
}

/*ARGSUSED*/
static void
DeleteBackwardChar(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  DeleteOrKill((TextWidget) w, event, XawsdLeft, XawstPositions, TRUE, FALSE);
}

/*ARGSUSED*/
static void 
DeleteForwardWord(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  DeleteOrKill((TextWidget) w, event,
	       XawsdRight, XawstWhiteSpace, FALSE, FALSE);
}

/*ARGSUSED*/
static void 
DeleteBackwardWord(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  DeleteOrKill((TextWidget) w, event,
	       XawsdLeft, XawstWhiteSpace, FALSE, FALSE);
}

/*ARGSUSED*/
static void 
KillForwardChar(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  DeleteOrKill((TextWidget) w, event, XawsdRight, XawstPositions, TRUE, TRUE);
}

/*ARGSUSED*/
static void
KillBackwardChar(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  DeleteOrKill((TextWidget) w, event, XawsdLeft, XawstPositions, TRUE, TRUE);
}

/*ARGSUSED*/
static void 
KillForwardWord(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  DeleteOrKill((TextWidget) w, event, 
	       XawsdRight, XawstWhiteSpace, FALSE, TRUE);
}

/*ARGSUSED*/
static void 
KillBackwardWord(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  DeleteOrKill((TextWidget) w, event,
	       XawsdLeft, XawstWhiteSpace, FALSE, TRUE);
}

/*ARGSUSED*/
static void
KillToEndOfLine(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  TextWidget ctx = (TextWidget) w;
  XawTextPosition end_of_line;

  StartAction(ctx, event);
  end_of_line = SrcScan(ctx->text.source, ctx->text.insertPos, XawstEOL, 
			XawsdRight, ctx->text.mult, FALSE);
  if (end_of_line == ctx->text.insertPos)
    end_of_line = SrcScan(ctx->text.source, ctx->text.insertPos, XawstEOL, 
			  XawsdRight, ctx->text.mult, TRUE);

  _DeleteOrKill(ctx, ctx->text.insertPos, end_of_line, TRUE);
  EndAction(ctx);
  _XawTextSetScrollBars(ctx);
}

/*ARGSUSED*/
static void 
KillToEndOfParagraph(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  DeleteOrKill((TextWidget) w, event, XawsdRight, XawstParagraph, FALSE, TRUE);
}

void 
_XawTextZapSelection(ctx, event, kill)
TextWidget ctx;
XEvent *event;
Boolean kill;
{
   StartAction(ctx, event);
   _DeleteOrKill(ctx, ctx->text.s.left, ctx->text.s.right, kill);
   EndAction(ctx);
  _XawTextSetScrollBars(ctx);
}

/*ARGSUSED*/
static void 
KillCurrentSelection(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  _XawTextZapSelection( (TextWidget) w, event, TRUE);
}

/*ARGSUSED*/
static void 
DeleteCurrentSelection(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  _XawTextZapSelection( (TextWidget) w, event, FALSE);
}

/************************************************************
 *
 * Insertion Routines.
 *
 ************************************************************/

static int 
InsertNewLineAndBackupInternal(ctx)
TextWidget ctx;
{
  int count, error = XawEditDone;
  XawTextBlock text;
  char *buf, *ptr;

  ptr = buf = XtMalloc((unsigned) sizeof(char) * ctx->text.mult);
  for (count = 0; count < ctx->text.mult; count++, ptr++)
    ptr[0] = '\n';

  text.length = ctx->text.mult;
  text.ptr = buf;
  text.firstPos = 0;
  text.format = FMT8BIT;

  if (_XawTextReplace(ctx, ctx->text.insertPos, ctx->text.insertPos, &text)) {
    XBell( XtDisplay(ctx), 50);
    error = XawEditError;
  }
  else 
    ctx->text.showposition = TRUE;

  XtFree(buf);
  return(error);
}

/*ARGSUSED*/
static void 
InsertNewLineAndBackup(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  StartAction( (TextWidget) w, event );
  (void) InsertNewLineAndBackupInternal( (TextWidget) w );
  EndAction( (TextWidget) w );
  _XawTextSetScrollBars( (TextWidget) w);
}

static int
LocalInsertNewLine(ctx, event)
TextWidget ctx;
XEvent *event;
{
  StartAction(ctx, event);
  if (InsertNewLineAndBackupInternal(ctx) == XawEditError)
    return(XawEditError);
  ctx->text.insertPos = SrcScan(ctx->text.source, ctx->text.insertPos, 
			     XawstPositions, XawsdRight, ctx->text.mult, TRUE);
  EndAction(ctx);
  _XawTextSetScrollBars(ctx);
  return(XawEditDone);
}

/*ARGSUSED*/
static void
InsertNewLine(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  (void) LocalInsertNewLine( (TextWidget) w, event);
}

/*ARGSUSED*/
static void 
InsertNewLineAndIndent(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  XawTextBlock text;
  XawTextPosition pos1;
  register char *ptr;
  register int length;
  TextWidget ctx = (TextWidget) w;

  StartAction(ctx, event);
  pos1 = SrcScan(ctx->text.source, ctx->text.insertPos, 
		 XawstEOL, XawsdLeft, 1, FALSE);

  /* Hacked because Ascii Source Object Scan method is permanently broken. */
  text.ptr = _XawTextGetText(ctx, pos1, ctx->text.insertPos);
  length = strlen(text.ptr);
  for (ptr=text.ptr; length && isspace(*ptr); ptr++, length--)
      ;
  *ptr = '\0';

  text.length = strlen(text.ptr);
  if (LocalInsertNewLine(ctx, event)) {
      XtFree(text.ptr);
      return;
  }
  text.firstPos = 0;
  if (_XawTextReplace(ctx,ctx->text.insertPos, ctx->text.insertPos, &text)) {
    XBell(XtDisplay(ctx), 50);
    XtFree(text.ptr);
    EndAction(ctx);
    return;
  }
  ctx->text.insertPos = SrcScan(ctx->text.source, ctx->text.insertPos,
				XawstPositions, XawsdRight, text.length, TRUE);
  XtFree(text.ptr);
  EndAction(ctx);
  _XawTextSetScrollBars(ctx);
}

/************************************************************
 *
 * Selection Routines.
 *
 *************************************************************/

static void 
SelectWord(w, event, params, num_params)
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
{
  TextWidget ctx = (TextWidget) w;
  XawTextPosition l, r;

  StartAction(ctx, event);
  l = SrcScan(ctx->text.source, ctx->text.insertPos, 
	      XawstWhiteSpace, XawsdLeft, 1, FALSE);
  r = SrcScan(ctx->text.source, l, XawstWhiteSpace, XawsdRight, 1, FALSE);
  _XawTextSetSelection(ctx, l, r, params, *num_params);
  EndAction(ctx);
}

static void 
SelectAll(w, event, params, num_params)
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
{
  TextWidget ctx = (TextWidget) w;

  StartAction(ctx, event);
  _XawTextSetSelection(ctx,zeroPosition,ctx->text.lastPos,params,*num_params);
  EndAction(ctx);
}

static void
ModifySelection(ctx, event, mode, action, params, num_params)
TextWidget ctx;
XEvent *event;
XawTextSelectionMode mode;
XawTextSelectionAction action;
String *params;		/* unused */
Cardinal *num_params;	/* unused */
{
  StartAction(ctx, event);
  NotePosition(ctx, event);
  _XawTextAlterSelection(ctx, mode, action, params, num_params);
  EndAction(ctx);
}
		
/* ARGSUSED */
static void 
SelectStart(w, event, params, num_params)
Widget w;
XEvent *event;
String *params;		/* unused */
Cardinal *num_params;	/* unused */
{
  ModifySelection((TextWidget) w, event, 
		  XawsmTextSelect, XawactionStart, params, num_params);
}

/* ARGSUSED */
static void 
SelectAdjust(w, event, params, num_params)
Widget w;
XEvent *event;
String *params;		/* unused */
Cardinal *num_params;	/* unused */
{
  ModifySelection((TextWidget) w, event, 
		  XawsmTextSelect, XawactionAdjust, params, num_params);
}

static void 
SelectEnd(w, event, params, num_params)
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
{
  ModifySelection((TextWidget) w, event, 
		  XawsmTextSelect, XawactionEnd, params, num_params);
}

/* ARGSUSED */
static void 
ExtendStart(w, event, params, num_params)
Widget w;
XEvent *event;
String *params;		/* unused */
Cardinal *num_params;	/* unused */
{
  ModifySelection((TextWidget) w, event, 
		  XawsmTextExtend, XawactionStart, params, num_params);
}

/* ARGSUSED */
static void 
ExtendAdjust(w, event, params, num_params)
Widget w;
XEvent *event;
String *params;		/* unused */
Cardinal *num_params;	/* unused */
{
  ModifySelection((TextWidget) w, event, 
		  XawsmTextExtend, XawactionAdjust, params, num_params);
}

static void 
ExtendEnd(w, event, params, num_params)
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
{
  ModifySelection((TextWidget) w, event, 
		  XawsmTextExtend, XawactionEnd, params, num_params);
}

static void
SelectSave(w, event, params, num_params)
Widget  w;
XEvent *event;
String *params;
Cardinal *num_params;
{
    int	    num_atoms;
    Atom*   sel;
    Display* dpy = XtDisplay(w);
    Atom    selections[256];

    StartAction ((TextWidget)w, event);
    num_atoms = *num_params;
    if (num_atoms > 256)
	num_atoms = 256;
    for (sel=selections; --num_atoms >= 0; sel++, params++)
	*sel = XInternAtom(dpy, *params, False);
    num_atoms = *num_params;
    _XawTextSaltAwaySelection ((TextWidget)w, selections, num_atoms);
    EndAction ((TextWidget)w);
}

/************************************************************
 *
 * Misc. Routines.
 *
 ************************************************************/

/*
 * Function used by vi to find what line of the currently visible screen
 * you are on.
 */
static int
GetLine (ctx, position)
TextWidget ctx;
XawTextPosition position;
{
  int line;

  for (line = 0; line < ctx->text.lt.lines; line++)
    if (position < ctx->text.lt.info[line + 1].position)
      break;
  return(line);
}

/*
 * Just redraw text, nothing more.
 */
/* ARGSUSED */
static void 
RedrawScreen(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  StartAction( (TextWidget) w, event);
  XawTextDisplay(w);
  EndAction( (TextWidget) w);
}

/*
 * redraw text and move cursor line to center of screen
 */
/* ARGSUSED */
static void 
RedrawScreenAndCenter(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  TextWidget ctx = (TextWidget)w;
  int insert_line = GetLine(ctx, ctx->text.insertPos);
  int scroll_by = insert_line - ctx->text.lt.lines/2;

  StartAction( (TextWidget) w, event);
  _XawTextVScroll(ctx, scroll_by);
  XawTextDisplay(w);
  EndAction( (TextWidget) w);
}

/*
 * redraw text and move cursor line to top of screen
 */
/* ARGSUSED */
static void 
RedrawScreenAndTop(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  TextWidget ctx = (TextWidget)w;
  int insert_line = GetLine(ctx, ctx->text.insertPos);
  int scroll_by = insert_line;

  StartAction( (TextWidget) w, event);
  _XawTextVScroll(ctx, scroll_by);
  XawTextDisplay(w);
  EndAction( (TextWidget) w);
}

/*
 * redraw text and move cursor line to bottom of screen
 */
/* ARGSUSED */
static void 
RedrawScreenAndBottom(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  TextWidget ctx = (TextWidget)w;
  int insert_line = GetLine(ctx, ctx->text.insertPos);
  int scroll_by = insert_line - ctx->text.lt.lines + 1;

  StartAction( (TextWidget) w, event);
  _XawTextVScroll(ctx, scroll_by);
  XawTextDisplay(w);
  EndAction( (TextWidget) w);
}

/* ARGSUSED */
static void 
RedrawDisplay(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  StartAction( (TextWidget) w, event);
  _XawTextClearAndCenterDisplay((TextWidget) w);
  EndAction( (TextWidget) w);
}

/*ARGSUSED*/
static void
TextFocusIn (w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  TextWidget ctx = (TextWidget) w;

  ctx->text.hasfocus = TRUE; 
}

/*ARGSUSED*/
static void
TextFocusOut(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  TextWidget ctx = (TextWidget) w;

  ctx->text.hasfocus = FALSE;
}

static XComposeStatus compose_status = {NULL, 0};

/*	Function Name: AutoFill
 *	Description: Breaks the line at the previous word boundry when
 *                   called inside InsertChar.
 *	Arguments: ctx - The text widget.
 *	Returns: none
 */

static void
AutoFill(ctx)
TextWidget ctx;
{
  int width, height, x, line_num, max_width;
  XawTextPosition ret_pos;
  XawTextBlock text;

  if ( !((ctx->text.auto_fill) && (ctx->text.mult == 1)) )
    return;

  for ( line_num = 0; line_num < ctx->text.lt.lines ; line_num++)
    if ( ctx->text.lt.info[line_num].position >= ctx->text.insertPos )
      break;
  line_num--;			/* backup a line. */

  max_width = Max(0, (int)(ctx->core.width - HMargins(ctx)));

  x = ctx->text.margin.left;
  XawTextSinkFindPosition( ctx->text.sink,ctx->text.lt.info[line_num].position,
			  x, max_width, TRUE, &ret_pos, &width, &height);
  
  if ( ret_pos >= ctx->text.insertPos )
    return;
  
  text.ptr = "\n";
  text.length = 1;
  text.firstPos = 0;
  text.format = FMT8BIT;

  if (_XawTextReplace(ctx, ret_pos - 1, ret_pos, &text))
    XBell(XtDisplay((Widget) ctx), 0);	/* Unable to edit, complain. */

}

/*
 * This function sets delete mode so that 'd' followed by any movement command
 * will delete that area of text moved over.
 * It has a special check for 'dd' which deletes all of the current line.
 */
/*ARGSUSED*/
static void
SetDelete(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  TextWidget ctx = (TextWidget) w;
  char strbuf[BUFSIZ];
  int length;
  KeySym keysym;

  length = XLookupString (&event->xkey, strbuf, BUFSIZ,
			       &keysym, &compose_status);
  if ((ctx->text.dyc_mode & VI_DELETE)&&(length != 0)&&(strbuf[0] == 'd'))
  {
    ctx->text.insertPos = SrcScan(ctx->text.source, ctx->text.insertPos,
				XawstEOL, XawsdLeft, 1, FALSE);
    if (ctx->text.dot_mode == TRUE)
    {
	ctx->text.dot_data.SetupMove.search = 0;
	ctx->text.dot_data.SetupMove.type = XawstEOL;
	ctx->text.dot_data.SetupMove.dir = XawsdLeft;
	ctx->text.dot_data.SetupMove.count = 1;
	ctx->text.dot_data.SetupMove.include = FALSE;
	ctx->text.dot_data.SetupMove.next = NULL;
    }
    ctx->text.del_from = ctx->text.insertPos;
    ViMove((TextWidget) w, event, XawsdRight, XawstEOL, TRUE);
  }
  else
  {
    ctx->text.dyc_mode = VI_DELETE;
    ctx->text.del_from = ctx->text.insertPos;

    if (ctx->text.dot_mode == FALSE)
    {
	FreeDot(w);
	ctx->text.dot_data.dyc_mode = VI_DELETE;
	ctx->text.dot_mode = TRUE;
    }
  }
}

/*
 * Function used to yank all of the current line.
 * called by 'yy' and 'Y'
 */
/*ARGSUSED*/
static void
YankLine(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  TextWidget ctx = (TextWidget) w;
  XawTextPosition old_pos;
  int from_left;

  ctx->text.dyc_mode = VI_YANK;
  ctx->text.del_from = ctx->text.insertPos;

  if (ctx->text.dot_mode == FALSE)
  {
	FreeDot(w);
	ctx->text.dot_data.dyc_mode = VI_YANK;
	ctx->text.dot_mode = TRUE;
  }
  StartAction(ctx, event);
  old_pos = ctx->text.insertPos;
  from_left = ctx->text.from_left;
  ctx->text.insertPos = SrcScan(ctx->text.source, ctx->text.insertPos,
				XawstEOL, XawsdLeft, 1, FALSE);
  if (ctx->text.dot_mode == TRUE)
  {
	ctx->text.dot_data.SetupMove.search = 0;
	ctx->text.dot_data.SetupMove.type = XawstEOL;
	ctx->text.dot_data.SetupMove.dir = XawsdLeft;
	ctx->text.dot_data.SetupMove.count = 1;
	ctx->text.dot_data.SetupMove.include = FALSE;
	ctx->text.dot_data.SetupMove.next = NULL;
	ctx->text.dot_data.nomove = 1;
  }
  ctx->text.del_from = ctx->text.insertPos;
  ctx->text.insertPos = ViSrcScan(ctx, ctx->text.insertPos,
				XawstEOL, XawsdRight, ctx->text.mult, TRUE);
  ProcessExtraFlags(ctx, ctx->text.insertPos, XawsdRight);
  ctx->text.insertPos = old_pos;
  ctx->text.from_left = from_left;
  EndAction(ctx);
}

/*
 * set yank mode so that 'y' followed by movement yanks the text moved over.
 * also a special test for 'yy' yo yank an entire line.
 */
/*ARGSUSED*/
static void
SetYank(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  TextWidget ctx = (TextWidget) w;
  char strbuf[BUFSIZ];
  int length;
  KeySym keysym;

  length = XLookupString (&event->xkey, strbuf, BUFSIZ,
			       &keysym, &compose_status);
  if ((ctx->text.dyc_mode & VI_YANK)&&(length != 0)&&(strbuf[0] == 'y'))
  {
	YankLine(w, event, p, n);
  }
  else
  {
    ctx->text.dyc_mode = VI_YANK;
    ctx->text.del_from = ctx->text.insertPos;

    if (ctx->text.dot_mode == FALSE)
    {
	FreeDot(w);
	ctx->text.dot_data.dyc_mode = VI_YANK;
	ctx->text.dot_mode = TRUE;
    }
  }
}

/*
 * function to change the text of an entire line.  called by 'S' and 'cc'
 */
/*ARGSUSED*/
static void
ChangeLine(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  TextWidget ctx = (TextWidget) w;

  ctx->text.dyc_mode = VI_CHANGE;
  ctx->text.del_from = ctx->text.insertPos;

  if (ctx->text.dot_mode == FALSE)
  {
	FreeDot(w);
	ctx->text.dot_data.dyc_mode = VI_CHANGE;
	ctx->text.dot_mode = TRUE;
  }
  StartAction(ctx, event);
  ctx->text.insertPos = SrcScan(ctx->text.source, ctx->text.insertPos,
				XawstEOL, XawsdLeft, 1, FALSE);
  if (ctx->text.dot_mode == TRUE)
  {
	ctx->text.dot_data.SetupMove.search = 0;
	ctx->text.dot_data.SetupMove.type = XawstEOL;
	ctx->text.dot_data.SetupMove.dir = XawsdLeft;
	ctx->text.dot_data.SetupMove.count = 1;
	ctx->text.dot_data.SetupMove.include = FALSE;
	ctx->text.dot_data.SetupMove.next = NULL;
  }
  ctx->text.del_from = ctx->text.insertPos;
  ctx->text.insertPos = ViSrcScan(ctx, ctx->text.insertPos,
				XawstEOL, XawsdRight, ctx->text.mult, TRUE);
  if (ctx->text.dot_mode == TRUE)
  {
	ViSrcScan(ctx, ctx->text.insertPos, XawstPositions,
		XawsdLeft, 1, TRUE);
  }
  ProcessExtraFlags(ctx, ctx->text.insertPos, XawsdRight);
  if (ctx->text.undo_text.length > 0)
  {
	ctx->text.undo_text.length--;
  }
  InsertNewLineAndBackupInternal(ctx);
  EndAction(ctx);
  _XawTextSetScrollBars(ctx);
}

/*
 * set change mode so 'c' followed by any move changes the text moved over.
 * also special check for 'cc' to change an entire line.
 */
/*ARGSUSED*/
static void
SetChange(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  TextWidget ctx = (TextWidget) w;
  char strbuf[BUFSIZ];
  int length;
  KeySym keysym;

  length = XLookupString (&event->xkey, strbuf, BUFSIZ,
			       &keysym, &compose_status);
  if ((ctx->text.dyc_mode & VI_CHANGE)&&(length != 0)&&(strbuf[0] == 'c'))
  {
	ChangeLine(w, event, p, n);
  }
  else
  {
    ctx->text.dyc_mode = VI_CHANGE;
    ctx->text.del_from = ctx->text.insertPos;

    if (ctx->text.dot_mode == FALSE)
    {
	FreeDot(w);
	ctx->text.dot_data.dyc_mode = VI_CHANGE;
	ctx->text.dot_mode = TRUE;
    }
  }
}

/*
 * Execute the dot record to repeat the last delete, yank, change, insert,
 * or put function.
 */
/*ARGSUSED*/
static void
DotFunc(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
	TextWidget ctx = (TextWidget)w;
	XawTextScanDirection thedir;
	XawTextPosition old_pos;
	XawTextPosition new;
	int from_left;

	StartAction(ctx, event);
	if (ctx->text.dot_data.nomove == 1)
	{
		old_pos = ctx->text.insertPos;
		from_left = ctx->text.from_left;
	}
	if (ctx->text.dot_data.SetupMove.count != 0)
	{
		ctx->text.insertPos = SrcScan(ctx->text.source,
			ctx->text.insertPos, ctx->text.dot_data.SetupMove.type,
			ctx->text.dot_data.SetupMove.dir,
			ctx->text.dot_data.SetupMove.count,
			ctx->text.dot_data.SetupMove.include);
	}

	if (ctx->text.dot_data.MoveList.next != NULL)
	{
		thedir = ctx->text.dot_data.MoveList.next->dir;
	}
	else
	{
		thedir = XawsdRight;
	}
	ctx->text.dot_mode = TRUE;
	if (ctx->text.dot_data.dyc_mode & VI_DELETE)
	{
		SetDelete(w, event, p, n);
	}
	else if (ctx->text.dot_data.dyc_mode & VI_YANK)
	{
		SetYank(w, event, p, n);
	}
	else if (ctx->text.dot_data.dyc_mode & VI_CHANGE)
	{
		SetChange(w, event, p, n);
	}
	else
	{
		ctx->text.undo_text.firstPos = 0;
		ctx->text.undo_text.format = FMT8BIT;
		if (ctx->text.undo_text.ptr != NULL)
		{
			XtFree(ctx->text.undo_text.ptr);
			ctx->text.undo_text.ptr = NULL;
		}
		ctx->text.undo_text.length = 0;
	}
	ctx->text.dot_mode = FALSE;
	ctx->text.dot_data.MovePtr = &(ctx->text.dot_data.MoveList);
	while (ctx->text.dot_data.MovePtr->next != NULL)
	{
		ctx->text.dot_data.MovePtr = ctx->text.dot_data.MovePtr->next;
		if (ctx->text.dot_data.MovePtr->search)
		{
			XawTextBlock text;
			XawTextPosition new_pos;
			text.firstPos = 0;
			text.format = FMT8BIT;
			text.ptr = ctx->text.dot_data.MovePtr->text;
			text.length = strlen(ctx->text.dot_data.MovePtr->text);
			new_pos = SrcSearch(ctx->text.source,
				ctx->text.insertPos,
				ctx->text.dot_data.MovePtr->dir, &text);
			if (new_pos == XawTextSearchError)
			{
				XBell(XtDisplay((Widget)ctx), 50);
			}
			else
			{
				ctx->text.insertPos = new_pos;
			}
		}
		else
		{
			ctx->text.insertPos = SrcScan(ctx->text.source,
				ctx->text.insertPos,
				ctx->text.dot_data.MovePtr->type,
				ctx->text.dot_data.MovePtr->dir,
				ctx->text.dot_data.MovePtr->count,
				ctx->text.dot_data.MovePtr->include);
		}
	}
	ctx->text.cut_buf = ctx->text.dot_data.buf;
	if (ctx->text.dot_data.dyc_mode & VI_CHANGE)
	{
		ctx->text.dyc_mode = VI_DELETE;
	}
	ProcessExtraFlags(ctx, ctx->text.insertPos, thedir);
	if (ctx->text.dot_data.insert == TRUE)
	{
		XawTextBlock text;

		ctx->text.undo_from = ctx->text.insertPos;
		text.firstPos = 0;
		text.format = FMT8BIT;
		text.ptr = ctx->text.dot_data.insert_text;
		text.length = strlen(ctx->text.dot_data.insert_text);
		if (_XawTextReplace(ctx, ctx->text.insertPos,
			ctx->text.insertPos, &text))
		{
			XBell(XtDisplay(ctx), 50);
		}
		ctx->text.insertPos = SrcScan(ctx->text.source,
			ctx->text.insertPos, XawstPositions, XawsdRight,
			text.length, TRUE);
		ctx->text.undo_to = ctx->text.insertPos;
	}
	else
	{
		ctx->text.undo_from = ctx->text.insertPos;
		ctx->text.undo_to = ctx->text.insertPos;
	}
	/* nomove == 1 means move to location before execution of dot command */
	if (ctx->text.dot_data.nomove == 1)
	{
		ctx->text.insertPos = old_pos;
		ctx->text.from_left = from_left;
	}
	/* nomove == 2 means move to beginning of inserted text */
	else if (ctx->text.dot_data.nomove == 2)
	{
		ctx->text.insertPos = ctx->text.undo_from;
		ctx->text.from_left = 0;
	}
	if ((ctx->text.dot_data.insert == TRUE)&&
	    (ctx->text.dot_data.nomove != 2))
	{
		XawTextPosition junk;
		int garbage;

		new = SrcScan(ctx->text.source, ctx->text.insertPos,
			XawstEOL, XawsdLeft, 1, FALSE);
		FindDist(ctx->text.sink, new, ctx->text.margin.left,
			ctx->text.insertPos, &(ctx->text.from_left),
			&junk, &garbage);
	}
	EndAction(ctx);
	_XawTextSetScrollBars(ctx);
}

/*
 * Join 2 lines.  Done by moving to the end of the currcnt line, and deleting
 * the <CR> character that seperates them
 */
/*ARGSUSED*/
static void
JoinLine(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  TextWidget ctx = (TextWidget) w;
  StartAction(ctx, event);
  ctx->text.insertPos = SrcScan(ctx->text.source, ctx->text.insertPos,
				XawstEOL, XawsdRight, 1, FALSE);
  FreeDot(w);
  ctx->text.dot_data.dyc_mode = VI_DELETE;
  ctx->text.dot_mode = TRUE;
  ctx->text.dot_data.SetupMove.search = 0;
  ctx->text.dot_data.SetupMove.type = XawstEOL;
  ctx->text.dot_data.SetupMove.dir = XawsdRight;
  ctx->text.dot_data.SetupMove.count = 1;
  ctx->text.dot_data.SetupMove.include = FALSE;
  ctx->text.dot_data.SetupMove.next = NULL;
  SetDelete(w, event, p, n);
  ctx->text.insertPos = ViSrcScan(ctx, ctx->text.insertPos,
				XawstPositions, XawsdRight, 1, TRUE);
  ProcessExtraFlags(ctx, ctx->text.insertPos, XawsdRight);
  EndAction(ctx);
}

/*
 * Chnage translations to emacs-like mode.
 * the start and end actions forces a clearing of all state variables.
 *
 * NOTE:  This function was used when I used to allow the user to
 * dynamically switch between emacs and vi modes as they were typing.
 * It turned out this introduced lots of bugs, so now you can only
 * chose your mode at startup time with the startMode resource.
 */
/*ARGSUSED*/
static void
DefaultMode(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  TextWidget ctx = (TextWidget) w;
  StartAction(ctx, event);
/*
  XtUninstallTranslations(w);
*/
  XtOverrideTranslations(w, DefaultTrans);
  EndAction(ctx);
}

/*
 * Return to vi-mode.
 * different behaviour depending on if you are returning from emacs mode
 * or returning form vi-insert mode
 *
 * Note:  You can no longer transfer from emacs mode to vi mode.  See note
 * at beginning of DefaultMode() function.
 */
/*ARGSUSED*/
static void
ViMode(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  TextWidget ctx = (TextWidget) w;

  StartAction(ctx, event);
  if (ctx->text.insert_extra != 0)
  {
    ctx->text.insertPos = SrcScan(ctx->text.source, ctx->text.insertPos,
				XawstPositions, XawsdRight,
				ctx->text.insert_extra, TRUE);
    ctx->text.insert_extra = 0;
  }
  ctx->text.undo_to = ctx->text.insertPos;
  if (ctx->text.dot_data.insert == TRUE)
  {
    if (ctx->text.undo_from < ctx->text.undo_to)
    {
      ctx->text.dot_data.insert_text = _XawTextGetText(ctx, ctx->text.undo_from,
					ctx->text.undo_to);
    }
    else
    {
      ctx->text.dot_data.insert = FALSE;
    }
  }
  if (ctx->text.last_trans != NULL)
  {
    _XtUnmergeTranslations(w, ctx->text.last_trans);
    ctx->text.last_trans = NULL;
  }
  EndAction(ctx);
}

/*
 * Enter special mode used by the f, F, t, T commands so search forward an back
 * for specific characters.
 */
/*ARGSUSED*/
static void
FindMode(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  TextWidget ctx = (TextWidget) w;

  if (*n != 2) {
    XtAppError(XtWidgetToApplicationContext(w), 
	       "The find-mode action takes exactly two arguments.");
    XBell(XtDisplay(w), 0);
    return;
  }
  if ((p[0][0] == 'R')||(p[0][0] == 'r'))
  {
    ctx->text.find_dir = XawsdRight;
  }
  else if ((p[0][0] == 'L')||(p[0][0] == 'l'))
  {
    ctx->text.find_dir = XawsdLeft;
  }
  else
  {
    char buf[BUFSIZ];
    sprintf(buf, "%s %s", "Text Widget: The find-mode action's first argument",
	    "must be either 'RIGHT' or 'LEFT'.");
    XtAppError(XtWidgetToApplicationContext(w), buf);
    XBell(XtDisplay(w), 0);
    return;
  }
  if ((p[1][0] == 'T')||(p[1][0] == 't'))
  {
    ctx->text.find_include = TRUE;
  }
  else if ((p[1][0] == 'F')||(p[1][0] == 'f'))
  {
    ctx->text.find_include = FALSE;
  }
  else
  {
    char buf[BUFSIZ];
    sprintf(buf, "%s %s", "Text Widget: The find-mode action's second argument",
	    "must be either 'TRUE' or 'FALSE'.");
    XtAppError(XtWidgetToApplicationContext(w), buf);
    XBell(XtDisplay(w), 0);
    return;
  }

  XtOverrideTranslations(w, FindTrans);
  ctx->text.last_trans = FindTrans;
}

/*
 * function called in insert mode to count replaced characters if you are in
 * replace mode and want to return to vi mode automatically when you have
 * replaced enough characters.
 */
/*ARGSUSED*/
static void
ReplaceTick(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  TextWidget ctx = (TextWidget)w;
  char strbuf[BUFSIZ];
  KeySym keysym;
  int keylen;

  keylen = XLookupString (&event->xkey, strbuf, BUFSIZ,
			       &keysym, &compose_status);
  if ((ctx->text.replace_cnt != 0)&&(keylen != 0))
  {
    if (ctx->text.replace_cnt > 0)
    {
	ctx->text.replace_cnt--;
    }
    if (ctx->text.replace_cnt == 0)
    {
	  TextWidget ctx = (TextWidget)w;
	  ctx->text.undo_to = ctx->text.insertPos;
	  if (ctx->text.dot_data.insert == TRUE)
	  {
	    if (ctx->text.undo_from < ctx->text.undo_to)
	    {
	      ctx->text.dot_data.insert_text =
			_XawTextGetText(ctx, ctx->text.undo_from,
						ctx->text.undo_to);
	    }
	    else
	    {
	      ctx->text.dot_data.insert = FALSE;
	    }
	  }
	  if (ctx->text.last_trans != NULL)
	  {
		_XtUnmergeTranslations(w, ctx->text.last_trans);
		ctx->text.last_trans = NULL;
	  }
    }
  }
}

/*
 * Function used in insert mode to delete characters before inserting new
 * ones if you are actually in replace mode.
 */
/*ARGSUSED*/
static void
MayDelete(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  TextWidget ctx = (TextWidget) w;
  XawTextPosition new;
  char *ptr, strbuf[BUFSIZ];
  char *ptr2;
  KeySym keysym;
  int keylen;

  keylen = XLookupString (&event->xkey, strbuf, BUFSIZ,
			       &keysym, &compose_status);
  if ((ctx->text.replace_cnt != 0)&&(keylen != 0))
  {
    ctx->text.dot_mode = TRUE;
    ctx->text.dot_data.insert = FALSE;
    new = ViSrcScan(ctx, ctx->text.insertPos, XawstPositions,
		XawsdRight, 1, TRUE);
    ctx->text.dot_data.insert = TRUE;
    ptr = _XawTextGetText(ctx, ctx->text.insertPos, new);
    ptr2 = (char *)XtMalloc(ctx->text.undo_text.length + strlen(ptr) + 1);
    strncpy(ptr2, ctx->text.undo_text.ptr, ctx->text.undo_text.length);
    ptr2[ctx->text.undo_text.length] = '\0';
    strcat(ptr2, ptr);
    ctx->text.undo_text.length += strlen(ptr);
    ptr2[ctx->text.undo_text.length] = '\0';
    if (ctx->text.undo_text.ptr != NULL)
    {
	XtFree(ctx->text.undo_text.ptr);
	ctx->text.undo_text.ptr = NULL;
    }
    XtFree(ptr);
    ctx->text.undo_text.ptr = ptr2;
    DeleteOrKill((TextWidget)w, event, XawsdRight, XawstPositions, TRUE, TRUE);
  }
}

/*
 * enter vi-insert mode.  May also be replace mode of some number oc characters,
 * or of infinite characters.
 */
/*ARGSUSED*/
static void
InsertMode(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  TextWidget ctx = (TextWidget) w;
  int cnt;

  if (*n != 1) {
    XtAppError(XtWidgetToApplicationContext(w), 
	       "The insert action takes exactly one argument.");
    XBell(XtDisplay(w), 0);
    return;
  }

  if ((p[0][0] == 'I')||(p[0][0] == 'i'))
  {
    cnt = -1;
  }
  else
  {
    cnt = atoi(p[0]);
    cnt = cnt * ctx->text.mult;
    ctx->text.mult = 1;
  }
  if ( cnt < -1 ) {
    char buf[BUFSIZ];
    sprintf(buf, "%s %s", "Text Widget: The insert action's argument",
	    "must be a number greater than or equal to zero, or 'INF'.");
    XtAppError(XtWidgetToApplicationContext(w), buf);
    XBell(XtDisplay(w), 0);
    return;
  }
  ctx->text.undo_from = ctx->text.insertPos;
  ctx->text.undo_to = ctx->text.insertPos;
  ctx->text.undo_text.firstPos = 0;
  ctx->text.undo_text.format = FMT8BIT;
  if (ctx->text.undo_text.ptr != NULL)
  {
    XtFree(ctx->text.undo_text.ptr);
    ctx->text.undo_text.ptr = NULL;
  }
  ctx->text.undo_text.length = 0;
  if (ctx->text.dot_mode == TRUE)
  {
	ctx->text.dot_data.insert = TRUE;
  }
  else
  {
    FreeDot(w);
    if (cnt != 0)
    {
	ctx->text.dot_data.dyc_mode = VI_DELETE;
    }
    ctx->text.dot_data.insert = TRUE;
    ctx->text.dot_mode = TRUE;
  }
  XtOverrideTranslations(w, InsertTrans);
  ctx->text.last_trans = InsertTrans;

  ctx->text.replace_cnt = cnt;
}

/*
 * enter insert mode at the beginning of the current line, 'I' command
 */
/*ARGSUSED*/
static void
StartLineInsertMode(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  TextWidget ctx = (TextWidget) w;

  _XawTextPrepareToUpdate(ctx);
  ctx->text.insertPos = SrcScan(ctx->text.source, ctx->text.insertPos,
				XawstEOL, XawsdLeft, 1, FALSE);
  FreeDot(w);
  ctx->text.dot_mode = TRUE;
  ctx->text.dot_data.SetupMove.search = 0;
  ctx->text.dot_data.SetupMove.type = XawstEOL;
  ctx->text.dot_data.SetupMove.dir = XawsdLeft;
  ctx->text.dot_data.SetupMove.count = 1;
  ctx->text.dot_data.SetupMove.include = FALSE;
  ctx->text.dot_data.SetupMove.next = NULL;
  _XawTextCheckResize(ctx);
  _XawTextExecuteUpdate(ctx);
  InsertMode(w, event, p, n);
}

/*
 * enter insert mode at the end of the current line, 'A' command
 */
/*ARGSUSED*/
static void
EndLineInsertMode(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  TextWidget ctx = (TextWidget) w;

  _XawTextPrepareToUpdate(ctx);
  ctx->text.insertPos = SrcScan(ctx->text.source, ctx->text.insertPos,
				XawstEOL, XawsdRight, 1, FALSE);
  FreeDot(w);
  ctx->text.dot_mode = TRUE;
  ctx->text.dot_data.SetupMove.search = 0;
  ctx->text.dot_data.SetupMove.type = XawstEOL;
  ctx->text.dot_data.SetupMove.dir = XawsdRight;
  ctx->text.dot_data.SetupMove.count = 1;
  ctx->text.dot_data.SetupMove.include = FALSE;
  ctx->text.dot_data.SetupMove.next = NULL;
  _XawTextCheckResize(ctx);
  _XawTextExecuteUpdate(ctx);
  InsertMode(w, event, p, n);
}

/*
 * open a new line after the current line, and enter insert mode there.
 * in vi is the 'o' command.
 */
/*ARGSUSED*/
static void
OpenNextLine(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  TextWidget ctx = (TextWidget) w;

  _XawTextPrepareToUpdate(ctx);
  ctx->text.mult = 1;
  ctx->text.insertPos = SrcScan(ctx->text.source, ctx->text.insertPos,
				XawstEOL, XawsdRight, 1, FALSE);
  FreeDot(w);
  ctx->text.dot_mode = TRUE;
  ctx->text.dot_data.SetupMove.search = 0;
  ctx->text.dot_data.SetupMove.type = XawstEOL;
  ctx->text.dot_data.SetupMove.dir = XawsdRight;
  ctx->text.dot_data.SetupMove.count = 1;
  ctx->text.dot_data.SetupMove.include = FALSE;
  ctx->text.dot_data.SetupMove.next = NULL;
  InsertMode(w, event, p, n);
  InsertNewLineAndBackupInternal(ctx);
  ctx->text.insertPos = ViSrcScan(ctx, ctx->text.insertPos, 
			     XawstPositions, XawsdRight, 1, TRUE);
  _XawTextCheckResize(ctx);
  _XawTextExecuteUpdate(ctx);
  _XawTextSetScrollBars(ctx);
}

/*
 * open a new line before the current line, and enter insert mode there.
 * in vi is the 'O' command.
 */
/*ARGSUSED*/
static void
OpenPreviousLine(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  TextWidget ctx = (TextWidget) w;

  _XawTextPrepareToUpdate(ctx);
  ctx->text.mult = 1;
  ctx->text.insertPos = SrcScan(ctx->text.source, ctx->text.insertPos,
				XawstEOL, XawsdLeft, 1, FALSE);
  FreeDot(w);
  ctx->text.dot_mode = TRUE;
  ctx->text.dot_data.SetupMove.search = 0;
  ctx->text.dot_data.SetupMove.type = XawstEOL;
  ctx->text.dot_data.SetupMove.dir = XawsdLeft;
  ctx->text.dot_data.SetupMove.count = 1;
  ctx->text.dot_data.SetupMove.include = FALSE;
  ctx->text.dot_data.SetupMove.next = NULL;
  InsertMode(w, event, p, n);
  InsertNewLineAndBackupInternal(ctx);
  ctx->text.insert_extra = 1;
  _XawTextCheckResize(ctx);
  _XawTextExecuteUpdate(ctx);
  _XawTextSetScrollBars(ctx);
}

/*
 * Use find mode to search forward for the character just pressed 
 */
/*ARGSUSED*/
static void
FindForwardChar(ctx, event, text, inclusive)
TextWidget ctx;
XEvent *event;
XawTextBlock *text;
Boolean inclusive;
{
  int count;
  XawTextPosition new_pos;

  StartAction(ctx, event);
  new_pos = ctx->text.insertPos;
  for (count=0; count<(ctx->text.mult - 1); count++)
  {
    new_pos = SrcSearch(ctx->text.source, new_pos,
				XawsdRight, text);
    new_pos = SrcScan(ctx->text.source, new_pos,
				XawstPositions, XawsdRight, 1, TRUE);
  }
  new_pos = SrcSearch(ctx->text.source, new_pos,
				XawsdRight, text);
  if (new_pos == XawTextSearchError)
  {
    XBell(XtDisplay((Widget)ctx), 50);
  }
  else
  {
    for (count=0; count<(ctx->text.mult - 1); count++)
    {
      ctx->text.insertPos = ViSrcSearch(ctx, ctx->text.insertPos,
				XawsdRight, text);
      ctx->text.insertPos = ViSrcScan(ctx, ctx->text.insertPos,
				XawstPositions, XawsdRight, 1, TRUE);
    }
    ctx->text.insertPos = ViSrcSearch(ctx, ctx->text.insertPos,
				XawsdRight, text);
    if (inclusive == TRUE)
    {
      ctx->text.insertPos = ViSrcScan(ctx, ctx->text.insertPos,
				XawstPositions, XawsdRight, 1, TRUE);
    }
  }
  ProcessExtraFlags(ctx, ctx->text.insertPos, XawsdRight);
  EndAction(ctx);
}

/*
 * Use find mode to search backward for the character just pressed 
 */
/*ARGSUSED*/
static void
FindBackwardChar(ctx, event, text, inclusive)
TextWidget ctx;
XEvent *event;
XawTextBlock *text;
Boolean inclusive;
{
  int count;
  XawTextPosition new_pos;

  StartAction(ctx, event);
  new_pos = ctx->text.insertPos;
  for (count=0; count<ctx->text.mult; count++)
  {
    new_pos = SrcSearch(ctx->text.source, new_pos,
				XawsdLeft, text);
  }
  if (new_pos == XawTextSearchError)
  {
    XBell(XtDisplay((Widget)ctx), 50);
  }
  else
  {
    for (count=0; count<ctx->text.mult; count++)
    {
      ctx->text.insertPos = ViSrcSearch(ctx, ctx->text.insertPos,
				XawsdLeft, text);
    }
    if (inclusive == FALSE)
    {
      ctx->text.insertPos = ViSrcScan(ctx, ctx->text.insertPos,
				XawstPositions, XawsdRight, 1, TRUE);
    }
  }
  ProcessExtraFlags(ctx, ctx->text.insertPos, XawsdLeft);
  EndAction(ctx);
}

/*
 * called directly form find mode, will search forward or backward
 * depending on state variables
 */
/*ARGSUSED*/
static void
FindChar(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  TextWidget ctx = (TextWidget) w;
  char *ptr, strbuf[BUFSIZ];
  int count;
  KeySym keysym;
  XawTextBlock text;

  if ( (text.length = XLookupString (&event->xkey, strbuf, BUFSIZ,
			       &keysym, &compose_status)) == 0) {
    return;
  }
  text.ptr = XtMalloc(sizeof(char) * text.length);
  strncpy(text.ptr, strbuf, text.length);
  text.ptr[text.length] = '\0';
  text.firstPos = 0;
  text.format = FMT8BIT;

  if (ctx->text.last_trans != NULL)
  {
	_XtUnmergeTranslations(w, ctx->text.last_trans);
	ctx->text.last_trans = NULL;
  }

  if (ctx->text.find_dir == XawsdRight)
  {
    FindForwardChar(ctx, event, &text, ctx->text.find_include);
  }
  else
  {
    FindBackwardChar(ctx, event, &text, ctx->text.find_include);
  }
}

/*ARGSUSED*/
static void
InsertChar(w, event, p, n)
Widget w;
XEvent *event;
String *p;
Cardinal *n;
{
  TextWidget ctx = (TextWidget) w;
  char *ptr, strbuf[BUFSIZ];
  int count, error;
  KeySym keysym;
  XawTextBlock text;

  if ( (text.length = XLookupString (&event->xkey, strbuf, BUFSIZ,
			       &keysym, &compose_status)) == 0) {
    return;
  }
  
  text.ptr = ptr = 
	XtMalloc((unsigned) sizeof(char) * text.length * ctx->text.mult);
  for (count = 0 ; count < ctx->text.mult ; count++) {
    (void) strncpy(ptr, strbuf, text.length);
    ptr += text.length;
  }

  text.length = text.length * ctx->text.mult;
  text.firstPos = 0;
  text.format = FMT8BIT;
  
  StartAction(ctx, event);
  
  error = _XawTextReplace(ctx, ctx->text.insertPos,ctx->text.insertPos, &text);

  if (error == XawEditDone) {
    ctx->text.insertPos = 
      SrcScan(ctx->text.source, ctx->text.insertPos,
	      XawstPositions, XawsdRight, text.length, TRUE);
    AutoFill(ctx);
  }
  else 
    XBell(XtDisplay(ctx), 50);

  XtFree(text.ptr);
  EndAction(ctx);
  _XawTextSetScrollBars(ctx);
}

/*ARGSUSED*/
static void 
InsertString(w, event, params, num_params)
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
{
  TextWidget ctx = (TextWidget) w;
  XawTextBlock text;
  int	   i;

  text.firstPos = 0;
  StartAction(ctx, event);
  for (i = *num_params; i; i--, params++) {
    unsigned char hexval;
    if ((*params)[0] == '0' && (*params)[1] == 'x' && (*params)[2] != '\0') {
      char c, *p;
      hexval = 0;
      for (p = *params+2; (c = *p); p++) {
	hexval *= 16;
	if (c >= '0' && c <= '9')
	  hexval += c - '0';
	else if (c >= 'a' && c <= 'f')
	  hexval += c - 'a' + 10;
	else if (c >= 'A' && c <= 'F')
	  hexval += c - 'A' + 10;
	else break;
      }
      if (c == '\0') {
	text.ptr = (char*)&hexval;
	text.length = 1;
      } else text.length = strlen(text.ptr = *params);
    } else text.length = strlen(text.ptr = *params);
    if (text.length == 0) continue;
    if (_XawTextReplace(ctx, ctx->text.insertPos, 
			ctx->text.insertPos, &text)) {
      XBell(XtDisplay(ctx), 50);
      EndAction(ctx);
      return;
    }
    ctx->text.insertPos =
      SrcScan(ctx->text.source, ctx->text.insertPos,
	      XawstPositions, XawsdRight, text.length, TRUE);
  }
  EndAction(ctx);
}

static void 
DisplayCaret(w, event, params, num_params)
Widget w;
XEvent *event;		/* CrossingNotify special-cased */
String *params;		/* Off, False, No, On, True, Yes, etc. */
Cardinal *num_params;	/* 0, 1 or 2 */
{
  TextWidget ctx = (TextWidget)w;
  Boolean display_caret = True;

  if (event->type == EnterNotify || event->type == LeaveNotify) {
    /* for Crossing events, the default case is to check the focus
     * field and only change the caret when focus==True.  A second
     * argument of "always" will cause the focus field to be ignored.
     */
    Boolean check_focus = True;
    if (*num_params == 2 && strcmp(params[1], "always") == 0)
      check_focus = False;
    if (check_focus && !event->xcrossing.focus) return;
  }

  if (*num_params > 0) {	/* default arg is "True" */
    XrmValue from, to;
    from.size = strlen(from.addr = params[0]);
    XtConvert(w, XtRString, &from, XtRBoolean, &to);
    if (to.addr != NULL) display_caret = *(Boolean*)to.addr;
    if (ctx->text.display_caret == display_caret) return;
  }
  StartAction(ctx, event);
  ctx->text.display_caret = display_caret;
  EndAction(ctx);
}

/*	Function Name: Multiply
 *	Description: Multiplies the current action by the number passed.
 *	Arguments: w - the text widget.
 *                 event - ** NOT USED **.
 *                 params, num_params - The parameter list, see below.
 *	Returns: none.
 *
 * Parameter list;
 *  
 * The parameter list may contain either a number or the string 'Reset'.
 * 
 * A number will multiply the current multiplication factor by that number.
 * Many of the text widget actions will will perform n actions, where n is
 * the multiplication factor.
 *
 * The string reset will reset the mutiplication factor to 1.
 * 
 */

/* ARGSUSED */
static void 
Multiply(w, event, params, num_params)
Widget w;
XEvent *event;
String * params;
Cardinal * num_params;
{
  TextWidget ctx = (TextWidget) w;
  int mult;

  if (*num_params != 1) {
    XtAppError(XtWidgetToApplicationContext(w), 
	       "The multiply action takes exactly one argument.");
    XBell(XtDisplay(w), 0);
    return;
  }

  if ( (params[0][0] == 'r') || (params[0][0] == 'R') ) {
    XBell(XtDisplay(w), 0);
    ctx->text.mult = 1;
    return;
  }

  if ( (mult = atoi(params[0])) == 0 ) {
    char buf[BUFSIZ];
    (void) sprintf(buf, "%s %s", "Text Widget: The multiply action's argument",
	    "must be a number greater than zero, or 'Reset'.");
    XtAppError(XtWidgetToApplicationContext(w), buf);
    XBell(XtDisplay(w), 0);
    return;
  }

  ctx->text.mult *= mult;
}

/*
 * The vi format of entering a number for count by just typing the
 * digits.  If the first digit typed is a 0 with no preceeding digits,
 * it is assumed to be the move to beginning of line command.
 */
/* ARGSUSED */
static void 
EnterDigit(w, event, params, num_params)
Widget w;
XEvent *event;
String * params;
Cardinal * num_params;
{
  TextWidget ctx = (TextWidget) w;
  int mult;

  if (*num_params != 1) {
    XtAppError(XtWidgetToApplicationContext(w), 
	       "The enter-digit action takes exactly one argument.");
    XBell(XtDisplay(w), 0);
    return;
  }

  if ( (mult = atoi(params[0])) < 0 ) {
    char buf[BUFSIZ];
    sprintf(buf, "%s %s", "Text Widget: The enter-digit action's argument",
	    "must be a number greater than or equal to zero.");
    XtAppError(XtWidgetToApplicationContext(w), buf);
    XBell(XtDisplay(w), 0);
    return;
  }
  if (mult == 0)
  {
    if (ctx->text.num_mode == FALSE)
    {
	/*
	 * Zero was pressed before any other digit.
	 * so Move to beginning of line.
	 */
	ViToLineStart(w, event, params, num_params);
    }
    else
    {
	ctx->text.mult *= 10;
    }
  }
  else
  {
    if (ctx->text.num_mode == FALSE)
    {
	ctx->text.num_mode = TRUE;
	ctx->text.mult = mult;
    }
    else
    {
	ctx->text.mult *= 10;
	ctx->text.mult += mult;
    }
  }
}

/*
 * used in vi to select a named buffer.  letters a-f select cut buffers
 * 2-7.
 */
/* ARGSUSED */
static void 
SelectBuffer(w, event, params, num_params)
Widget w;
XEvent *event;
String * params;
Cardinal * num_params;
{
  TextWidget ctx = (TextWidget) w;
  int buf;

  if (*num_params != 1) {
    XtAppError(XtWidgetToApplicationContext(w), 
	       "The buffer-select action takes exactly one argument.");
    XBell(XtDisplay(w), 0);
    return;
  }

  buf = atoi(params[0]);

  if ((buf < 0)||(buf > 7)) {
    char buf[BUFSIZ];
    sprintf(buf, "%s %s", "Text Widget: The buffer-select action's argument",
	    "must be a number greater than or equal to zero and less than eight.");
    XtAppError(XtWidgetToApplicationContext(w), buf);
    XBell(XtDisplay(w), 0);
    return;
  }

  ctx->text.cut_buf = buf;
}

/*	Function Name: StripOutOldCRs
 *	Description: strips out the old carrige returns.
 *	Arguments: ctx - the text widget.
 *                 from - starting point.
 *                 to - the ending point
 *	Returns: the new ending location (we may add some characters),
 *		 or XawTextReplaceError, which indicates failure.
 */

static XawTextPosition
StripOutOldCRs(ctx, from, to)
TextWidget ctx;
XawTextPosition from, to;
{
  XawTextPosition startPos, endPos, eop_begin, eop_end, temp;
  Widget src = ctx->text.source;
  XawTextBlock text;
  char *buf;

  text.ptr= "  ";
  text.firstPos = 0;
  text.format = FMT8BIT;
   
/*
 * Strip out CR's. 
 */

  eop_begin = eop_end = startPos = endPos = from;
  while (TRUE) {
    endPos=SrcScan(src, startPos, XawstEOL, XawsdRight, 1, FALSE);

    temp=SrcScan(src, endPos, XawstWhiteSpace, XawsdLeft, 1, FALSE);
    temp=SrcScan(src, temp, XawstWhiteSpace, XawsdRight, 1, FALSE);

    if (temp > startPos)
	endPos = temp;

    if (endPos >= to)
      break;

    if (endPos >= eop_begin) {
      startPos = eop_end;
      eop_begin = SrcScan(src, startPos, XawstParagraph, XawsdRight, 1, FALSE);
      eop_end = SrcScan(src, startPos, XawstParagraph, XawsdRight, 1, TRUE);
    }
    else {
      XawTextPosition periodPos, next_word;
      int i, len;

      periodPos= SrcScan(src, endPos, XawstPositions, XawsdLeft, 1, TRUE);
      next_word = SrcScan(src, endPos, XawstWhiteSpace, XawsdRight, 1, FALSE);

      len = next_word - periodPos;

      text.length = 1;
      buf = _XawTextGetText(ctx, periodPos, next_word);
      if ( (periodPos < endPos) && (buf[0] == '.') )
	  text.length++;	/* Put in two spaces. */

      /*
       * Remove all extra spaces. 
       */

      for (i = 1 ; i < len; i++) 
	  if ( !isspace(buf[i]) || ((periodPos + i) >= to) ) {
	      break;
	  }
      
      XtFree(buf);

      to -= (i - text.length - 1);
      startPos = SrcScan(src, periodPos, XawstPositions, XawsdRight, i, TRUE);
      if (_XawTextReplace(ctx, endPos, startPos, &text) != XawEditDone)
	  return XawReplaceError;
      startPos -= i - text.length;
    }
  }
  return(to);
}

/*	Function Name: InsertNewCRs
 *	Description: Inserts the new Carrige Returns.
 *	Arguments: ctx - the text widget.
 *                 from, to - the ends of the region.
 *	Returns: none
 */

static void
InsertNewCRs(ctx, from, to)
TextWidget ctx;
XawTextPosition from, to;
{
  XawTextPosition startPos, endPos, space, eol;
  XawTextBlock text;
  int i, width, height, len;
  char * buf;

  text.ptr = "\n";
  text.length = 1;
  text.firstPos = 0;
  text.format = FMT8BIT;

  startPos = from;
  while (TRUE) {
    XawTextSinkFindPosition( ctx->text.sink, startPos, 
			    (int) ctx->text.margin.left,
			    (int) (ctx->core.width - HMargins(ctx)), 
			    TRUE, &eol, &width, &height);
    if (eol >= to)
      break;

    eol = SrcScan(ctx->text.source, eol, XawstPositions, XawsdLeft, 1, TRUE);
    space= SrcScan(ctx->text.source, eol, XawstWhiteSpace, XawsdRight, 1,TRUE);
    
    startPos = endPos = eol;
    if (eol == space) 
      return;

    len = (int) (space - eol);
    buf = _XawTextGetText(ctx, eol, space);
    for ( i = 0 ; i < len ; i++)
      if (!isspace(buf[i]))
	break;

    to -= (i - 1);
    endPos = SrcScan(ctx->text.source, endPos,
		     XawstPositions, XawsdRight, i, TRUE);
    XtFree(buf);
    
    if (_XawTextReplace(ctx, startPos, endPos, &text))
	return;
    startPos = SrcScan(ctx->text.source, startPos,
		       XawstPositions, XawsdRight, 1, TRUE);
  }
}  
  
/*	Function Name: FormRegion
 *	Description: Forms up the region specified.
 *	Arguments: ctx - the text widget.
 *                 from, to - the ends of the region.
 *	Returns: XawEditDone if successful, or XawReplaceError.
 */

static int
FormRegion(ctx, from, to)
TextWidget ctx;
XawTextPosition from, to;
{
  if (from >= to) return XawEditDone;

  if ((to = StripOutOldCRs(ctx, from, to)) == XawReplaceError)
      return XawReplaceError;
  /* insure that the insertion point is within legal bounds */
  if (ctx->text.insertPos >
      SrcScan(ctx->text.source, (XawTextPosition)0, XawstAll, XawsdRight, 1, TRUE))
      ctx->text.insertPos = to;
  InsertNewCRs(ctx, from, to);
  _XawTextBuildLineTable(ctx, ctx->text.lt.top, TRUE);
  return XawEditDone;
}

/*	Function Name: FormParagraph.
 *	Description: reforms up the current paragraph.
 *	Arguments: w - the text widget.
 *                 event - the X event.
 *                 params, num_params *** NOT USED ***.
 *	Returns: none
 */

/* ARGSUSED */
static void 
FormParagraph(w, event, params, num_params)
Widget w;
XEvent *event;
String * params;
Cardinal * num_params;
{
  TextWidget ctx = (TextWidget) w;
  XawTextPosition from, to;

  StartAction(ctx, event);

  from =  SrcScan(ctx->text.source, ctx->text.insertPos,
		  XawstParagraph, XawsdLeft, 1, FALSE);
  to  =  SrcScan(ctx->text.source, from,
		 XawstParagraph, XawsdRight, 1, FALSE);

  if (FormRegion(ctx, from, to) == XawReplaceError)
      XBell(XtDisplay(w), 0);
  EndAction(ctx);
  _XawTextSetScrollBars(ctx);
}

/*	Function Name: TransposeCharacters
 *	Description: Swaps the character to the left of the mark with
 *                   the character to the right of the mark.
 *	Arguments: w - the text widget.
 *                 event - the event that cause this action.
 *                 params, num_params *** NOT USED ***.
 *	Returns: none.
 */
	     
/* ARGSUSED */
static void 
TransposeCharacters(w, event, params, num_params)
Widget w;
XEvent *event;
String * params;
Cardinal * num_params;
{
  TextWidget ctx = (TextWidget) w;
  XawTextPosition start, end;
  XawTextBlock text;
  unsigned char * buf, c;
  int i;

  StartAction(ctx, event);

/*
 * Get bounds. 
 */

  start = SrcScan(ctx->text.source, ctx->text.insertPos, XawstPositions, 
		  XawsdLeft, 1, TRUE);
  end = SrcScan(ctx->text.source, ctx->text.insertPos, XawstPositions, 
		XawsdRight, ctx->text.mult, TRUE);

  if ( (start == ctx->text.insertPos) || (end == ctx->text.insertPos) ) 
    XBell(XtDisplay(w), 0);	/* complain. */
  else {
    ctx->text.insertPos = end;

    /*
     * Retrieve text and swap the characters. 
     */
    
    buf = (unsigned char *) _XawTextGetText(ctx, start, end);
    text.length = strlen((char *)buf);
    text.firstPos = 0;
    text.format = FMT8BIT;
    
    c = buf[0];
    for (i = 1 ; i < text.length ; i++)
      buf[i - 1] = buf[i];
    buf[i - 1] = c;
    
    /* 
     * Store new text is source.
     */
    
    text.ptr = (char *) buf;
    if (_XawTextReplace (ctx, start, end, &text))
	XBell(XtDisplay(w), 0);	/* Unable to edit, complain. */
    XtFree((char *) buf);
  }
  EndAction(ctx);
}

/*	Function Name: NoOp
 *	Description: This action performs no action, and allows
 *                   the user or application programmer to unbind a 
 *                   translation.
 *	Arguments: w - the text widget.
 *                 event - *** UNUSED ***.
 *                 parms and num_params - the action parameters.
 *	Returns: none.
 *
 * Note: If the parameter list contains the string "RingBell" then
 *       this action will ring the bell.
 */

/*ARGSUSED*/
static void
NoOp(w, event, params, num_params)
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
{
    if (*num_params != 1)
	return;

    switch(params[0][0]) {
    case 'R':
    case 'r':
	XBell(XtDisplay(w), 0);
    default:			/* Fall Through */
	break;
    }
}

/*
 * very like the noop function, but clears state variables.
 */
static void
NullAction(w, event, params, num_params)
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
{
  char strbuf[BUFSIZ];
  int length;
  KeySym keysym;

  length = XLookupString (&event->xkey, strbuf, BUFSIZ,
			&keysym, &compose_status);
  if (length != 0)
  {
    StartAction((TextWidget)w, event);
    NoOp(w, event, params, num_params);
    EndAction((TextWidget)w);
  }
}
	
/* Action Table */

XtActionsRec _XawTextActionsTable[] = {
/* motion bindings */
  {"forward-character", 	MoveForwardChar},
  {"backward-character", 	MoveBackwardChar},
  {"forward-word", 		MoveForwardWord},
  {"backward-word", 		MoveBackwardWord},
  {"forward-paragraph", 	MoveForwardParagraph},
  {"backward-paragraph", 	MoveBackwardParagraph},
  {"beginning-of-line", 	MoveToLineStart},
  {"end-of-line", 		MoveToLineEnd},
  {"next-line", 		MoveNextLine},
  {"previous-line", 		MovePreviousLine},
  {"next-page", 		MoveNextPage},
  {"previous-page", 		MovePreviousPage},
  {"beginning-of-file", 	MoveBeginningOfFile},
  {"end-of-file", 		MoveEndOfFile},
  {"scroll-one-line-up",	ScrollOneLineUp},
  {"scroll-one-line-down",	ScrollOneLineDown},
  {"next-half-page",		MoveNextHalfPage},
  {"previous-half-page", 	MovePreviousHalfPage},

/* vi motion bindings */
  {"vi-forward-character", 	ViForwardChar},
  {"vi-backward-character", 	ViBackwardChar},
  {"vi-forward-word", 		ViForwardWord},
  {"vi-end-word", 		ViEndWord},
  {"vi-backward-word", 		ViBackwardWord},
  {"vi-forward-paragraph", 	ViForwardParagraph},
  {"vi-backward-paragraph", 	ViBackwardParagraph},
  {"vi-beginning-of-line", 	ViToLineStart},
  {"vi-end-of-line", 		ViToLineEnd},
  {"vi-next-line", 		ViNextLine},
  {"vi-previous-line", 		ViPreviousLine},
  {"vi-next-line-word", 	ViNextLineWord},
  {"vi-previous-line-word", 	ViPreviousLineWord},
  {"vi-beginning-of-file", 	ViBeginningOfFile},
  {"vi-end-of-file", 		ViEndOfFile},
  {"vi-line-in-file", 		ViLineInFile},

/* delete bindings */
  {"delete-next-character", 	DeleteForwardChar},
  {"delete-previous-character", DeleteBackwardChar},
  {"delete-next-word", 		DeleteForwardWord},
  {"delete-previous-word", 	DeleteBackwardWord},
  {"delete-selection", 		DeleteCurrentSelection},
/* kill bindings */
  {"kill-next-character", 	KillForwardChar},
  {"kill-previous-character",	KillBackwardChar},
  {"kill-word", 		KillForwardWord},
  {"backward-kill-word", 	KillBackwardWord},
  {"kill-selection", 		KillCurrentSelection},
  {"kill-to-end-of-line", 	KillToEndOfLine},
  {"kill-to-end-of-paragraph", 	KillToEndOfParagraph},
/* special vi bindings */
  {"set-delete", 		SetDelete},
  {"set-change", 		SetChange},
  {"set-yank", 			SetYank},
  {"yank-line", 		YankLine},
  {"change-line", 		ChangeLine},
  {"undo-func", 		UndoFunc},
  {"dot-func",	 		DotFunc},
/* new line stuff */
  {"newline-and-indent", 	InsertNewLineAndIndent},
  {"newline-and-backup", 	InsertNewLineAndBackup},
  {"newline", 			InsertNewLine},
/* Selection stuff */
  {"select-word", 		SelectWord},
  {"select-all", 		SelectAll},
  {"select-start", 		SelectStart},
  {"select-adjust", 		SelectAdjust},
  {"select-end", 		SelectEnd},
  {"select-save",		SelectSave},
  {"extend-start", 		ExtendStart},
  {"extend-adjust", 		ExtendAdjust},
  {"extend-end", 		ExtendEnd},
  {"insert-selection",		InsertSelection},
  {"insert-chosen-selection-before",	InsertChosenSelectionBefore},
  {"insert-chosen-selection-after",	InsertChosenSelectionAfter},
/* Miscellaneous */
  {"redraw-screen", 		RedrawScreen},
  {"redraw-screen-and-center",	RedrawScreenAndCenter},
  {"redraw-screen-and-top",	RedrawScreenAndTop},
  {"redraw-screen-and-bottom",	RedrawScreenAndBottom},
  {"redraw-display", 		RedrawDisplay},
  {"insert-file", 		_XawTextInsertFile},
  {"write-file", 		_XawTextWriteFile},
  {"search",		        _XawTextSearch},
  {"select-buffer", 		SelectBuffer},
  {"default-mode", 		DefaultMode},
  {"vi-mode", 			ViMode},
  {"find-mode", 		FindMode},
  {"may-delete", 		MayDelete},
  {"replace-tick", 		ReplaceTick},
  {"insert-mode", 		InsertMode},
  {"start-line-insert-mode",	StartLineInsertMode},
  {"end-line-insert-mode",	EndLineInsertMode},
  {"open-next-line",		OpenNextLine},
  {"open-previous-line",	OpenPreviousLine},
  {"join-line", 		JoinLine},
  {"find-character", 		FindChar},
  {"insert-char", 		InsertChar},
  {"insert-string",		InsertString},
  {"focus-in", 	 	        TextFocusIn},
  {"focus-out", 		TextFocusOut},
  {"display-caret",		DisplayCaret},
  {"multiply",		        Multiply},
  {"enter-digit",		EnterDigit},
  {"form-paragraph",            FormParagraph},
  {"transpose-characters",      TransposeCharacters},
  {"no-op",                     NoOp},
  {"null-action",               NullAction},
/* Action to bind special translations for text Dialogs. */
  {"InsertFileAction",          _XawTextInsertFileAction},
  {"WriteFileAction",           _XawTextWriteFileAction},
  {"DoSearchAction",            _XawTextDoSearchAction},
  {"DoReplaceAction",           _XawTextDoReplaceAction},
  {"SetField",                  _XawTextSetField},
  {"PopdownSearchAction",       _XawTextPopdownSearchAction},
};

Cardinal _XawTextActionsTableCount = XtNumber(_XawTextActionsTable);
