/*                               -*- Mode: C -*- 
 * rdd.c
 * 
 * Description     :  Roger's Drag 'n Drop Library.
 * 					  A drag-n-drop library for Xt programs.
 * 
 * Author          : Roger Reynolds	 - 	rogerr@netcom.com
 * Created On      : Sun Apr 26 14:14:01 1992
 * 
 * Configuration Management
 * 
 * @(#)rdd.c	1.3	5/1/92
 * 
 * VERSION   SPR   MODIFIER   DATE AND TIME
 * 
 */

#include <stdio.h>
#include <X11/Xatom.h>
#include <X11/IntrinsicP.h>
#include <X11/cursorfont.h>

#include "rdd.h"

/*
 *  Private variables containing bits of current state 
 */
static Widget topLevel;
static Display *display;
static Atom rddProtocolMsg;
static Atom rddSelnAtom;
static Cursor dragCursor;
static Window getPointerWindow();

/*
 *  Is this event an rdd drop message? 
 */
#define rddIsRddMessageEvent(ev)\
	((ev).type == ClientMessage && (ev).xclient.message_type == rddProtocolMsg)

/*
 *  Debug Macro 
 */
static int dbg=0;
#define DEBUG(stmt) if(dbg) {stmt}
/* #define DEBUG(stmt) */


/*
 * Data structure describing a "drop event handler" 
 */
typedef struct 
{
	XtCallbackProc proc;
	char *data;
	Widget w;
	Boolean active;
} HandlerData;

/*
 * Internal list of all drop handlers 
 */
static HandlerData *handlerList;
static int handlerListCnt, handlerListAllocCnt;


/*
 *  rddInit - 
 * 		Initialize the rdd package.	This routine must be called before
 * 		any other rdd routines.
 * 			Initialize static data,
 * 			Create rdd protocol atoms.
 * 			Create default cursor used for drag operations.
 * 			Register rdd actions.
 */
void
rddInit(shell, context)
	Widget shell;
	XtAppContext context;
{
	XtActionsRec actions;

	DEBUG (fprintf(stderr, "rddInit entered\n"););

	topLevel		= shell;
	display			= XtDisplay(topLevel);
	rddProtocolMsg	= XInternAtom (display, "RDD_MESSAGE", FALSE);
	rddSelnAtom		= XInternAtom (display, "RDD_SELECTION", FALSE);
	dragCursor 		= XCreateFontCursor (display, XC_cross);

	actions.string	= "rddStartAction";
	actions.proc	= rddStartAction;
	XtAppAddActions (context, &actions, 1);

	actions.string	= "rddDragAction";
	actions.proc	= rddDragAction;
	XtAppAddActions (context, &actions, 1);

	actions.string	= "rddDropAction";
	actions.proc	= rddDropAction;
	XtAppAddActions (context, &actions, 1);

	DEBUG (fprintf (stderr,"rddInit: topLevel=%d display=%d\n",
					topLevel, display););
}

/*
 * rddSetDragCursor - 
 * 		Set the cursor that will be used for drag-n-drop operations.
 */
void
rddSetDragCursor (cursor)
	Cursor cursor;

{
	dragCursor = cursor;
}

/*
 *  rddInitCheck - 
 * 		Ensure that rdd has been initialized.
 */
static void
rddInitCheck()
{
	if (!topLevel)
	{
		fprintf (stderr, "rdd not initialized\n");
		exit(-1);
	}
}

/*
 *  These routines coordinate access to the rddSelection 
 */

static unsigned char *rddSelnData;
static unsigned long rddSelnLen;


static int 
rddSetSelection(data, nbytes)
	caddr_t data;
	int nbytes;
{
	int stat;
	Window window;

	rddInitCheck();

	window = XtWindow(topLevel);

	XSetSelectionOwner (display, rddSelnAtom, window, CurrentTime);

	if (XGetSelectionOwner (display, rddSelnAtom) != window)
	{
		fprintf (stderr, "XSetSelectionOwner failed\n");
	}

	if (rddSelnLen)
	{
		XtFree(rddSelnData);
		rddSelnLen = 0;
	}

	rddSelnData = (u_char *)XtMalloc(rddSelnLen=nbytes);
	bcopy (data, rddSelnData, rddSelnLen);

	stat = XChangeProperty (display, RootWindow(display,0), rddSelnAtom,
							XA_STRING,
							8,
							PropModeReplace,
							rddSelnData, (int)rddSelnLen);
	return(stat);
}

static int
rddGetSelection (data)
	u_char **data;
{
	int n;
	Atom type;
	int format, count;
	u_long leftover;

	DEBUG (fprintf (stderr, "rddGetSelection entered\n"););

	rddInitCheck();

	n =XGetWindowProperty (display, RootWindow(display,0), rddSelnAtom,
						   0L, 100000L,
						   FALSE,
						   AnyPropertyType,
						   &type, &format,
						   &rddSelnLen,
						   &leftover,
						   &rddSelnData);

	*data = rddSelnData;
	return(rddSelnLen);	
}

/*
 *  getPointerWindow - 
 * 		return the first subwindow of top which the pointer is in that
 * 		has a drop handler installed on it.
 */
static Window
getPointerWindow ()
{
	int i;
	for (i=0; i < handlerListCnt; i++)
	{
		if (handlerList[i].active)
		{
			DEBUG (fprintf(stderr, "returning handler %d %s window = %d\n",
						   i, XtName(handlerList[i].w),
						   XtWindow(handlerList[i].w)););
			return (XtWindow(handlerList[i].w));
		}
	}
	DEBUG (fprintf (stderr, "getPointerWindow returned 0\n"););
	return (0);
}

/*
 *  rddSendDropEvent - 
 * 		Send an rdd drop message to the window which contains the pointer.
 * 		The window parameter is filled in with the window id  of the widget
 * 		passed in, or with the window id of the topLevel shell if w is NULL
 * 		This is to facilitate a possible conversation between the
 * 		dropping window and its target.  Such a conversation could be 
 * 		needed to negotiate the details of a complex drop operation,
 * 		none is required to send simple messages.
 */
void
rddSendDropEvent(w)
	Widget w;
{
	static XClientMessageEvent ev;
	if (!ev.type)
	{
		rddInitCheck();
		ev.type			= ClientMessage;
		ev.display		= display;
		ev.message_type = rddProtocolMsg;
		ev.format		= 32;
		ev.data.l[0]	= XtWindow(topLevel);
	}

	ev.window	= w ? XtWindow(w) : XtWindow(topLevel);
	++ev.serial;

	ev.data.l[1] = 234;
	ev.data.l[2] = 345;

	DEBUG (fprintf(stderr, "rddSendDropEvent to win %d\n", ev.window););

	XSendEvent (display, PointerWindow, TRUE, NoEventMask, (XEvent*)&ev);
}

/*
 *  rddDropEventHandler - 
 * 		This procedure is installed to handel rdd drop events on all windows.
 * 		It dispatches to the user specified procedure after retrieving data
 * 		from the rdd selection buffer.
 */
static XtEventHandler
rddDropEventHandler(w, hd, ev)
	Widget w;
	HandlerData *hd;
	XClientMessageEvent *ev;
{
	int nbytes;
	u_char *sdata;
	RddCallbackStruct cbs;

	if (ev->type != ClientMessage || ev->message_type != rddProtocolMsg)
		return;
	
	nbytes = rddGetSelection (&sdata);

	DEBUG (fprintf (stderr,"GOT DROP MESSAGE %d bytes\n", nbytes););

	if (hd && hd->proc)
	{
		cbs.data = (caddr_t)sdata;
		cbs.len  = nbytes;
		(hd->proc)(hd->w, hd->data, &cbs);
	}
}

static void
enterLeaveProc(w, hd, ev)
	Widget w;
	HandlerData *hd;
	XCrossingEvent *ev;
{
	hd->active = (ev->type == EnterNotify);
	DEBUG (fprintf(stderr, "window %s active = %d\n",
				   XtName(hd->w), hd->active););
}

/*
 *  rddAddDropHandler - 
 * 		Add a callback routine to a widget for rdd drop events.
 * 		Call scemantics for proc:
 * 			void proc (Widget w; XtPointer data; RddCallbackStruct *cbs)
 * 		where w     is the widget for which the handler is being called
 * 			  data  is user specified data passed through to proc
 * 			  cbs	is an RddCallbackStruct filled in.
 */
void
rddAddDropHandler(w, proc, data)
	Widget w;
	XtCallbackProc proc;
	caddr_t data;
{
	int i;
	HandlerData *newHandler;

	rddInitCheck();

	if (handlerListCnt >= handlerListAllocCnt)
	{

		for (i=0; i < handlerListCnt; i++)
		{
			XtRemoveEventHandler (handlerList[i].w, 0, TRUE, 
								  rddDropEventHandler, handlerList+i);
			XtRemoveEventHandler (w, EnterWindowMask|LeaveWindowMask, FALSE, 
							   enterLeaveProc, handlerList+i);
		}

		handlerListAllocCnt += 20;
		handlerList =
			(HandlerData*) XtRealloc(handlerList,
								 sizeof(HandlerData) * handlerListAllocCnt);
		DEBUG(fprintf(stderr,"handlerList = %x\n", handlerList););

		for (i=0; i < handlerListCnt; i++)
		{
			XtAddEventHandler (handlerList[i].w, 0, TRUE, 
							   rddDropEventHandler, handlerList+i);
			XtAddEventHandler (w, EnterWindowMask|LeaveWindowMask, FALSE, 
							   enterLeaveProc, handlerList+i);
		}
	}

	newHandler = handlerList + handlerListCnt++;
	
	DEBUG (fprintf(stderr, "rddAddDropHandler: for widget %s window=%d\n",
				  XtName(w), XtWindow(w)););

	newHandler->proc	=	proc;
	newHandler->data	=	data;
	newHandler->w		=   w;
	newHandler->active	=   FALSE;
	
	XtAddEventHandler (w, 0, TRUE, rddDropEventHandler, newHandler);
	XtAddEventHandler (w, EnterWindowMask|LeaveWindowMask, FALSE, 
					   enterLeaveProc, newHandler);
}


/*
 *  rddSetDropData - 
 * 		Public function to set data into the rdd selection buffer.
 */
void
rddSetDropData (data, len)
	caddr_t data;
	int len;
{
	rddSetSelection (data, len);
}


/*
 * Action routines which changes the cursor when appropriate and begin
 * the drag-n-drop process.
 * They could be bound directly to btndown, btnmotion, and btnup events,
 * but to do anything useful, rddSetDropData must be called someplace.
 */
static int bx, by, dragflag;
void rddStartAction (w, event, args, nargs)
	Widget w;
	XButtonEvent *event;
	String *args;
	int *nargs;
{
	DEBUG (fprintf(stderr, "rddStartAction\n"););
	dragflag = 0;
	bx = event->x;
	by = event->y;
}

void rddDragAction (w, event, args, nargs)
	Widget w;
	XButtonEvent *event;
	String *args;
	int *nargs;
{
	DEBUG (fprintf(stderr, "rddDragAction\n"););
	if (dragflag == 0 && ((abs(bx - event->x)) > 10 ||
						  (abs(by - event->y)) > 10))
	{
		DEBUG (fprintf(stderr, "using drag cursor %d\n", dragCursor););
		dragflag = 1;
		XDefineCursor (XtDisplay(w), XtWindow(w), dragCursor);
	}
}
	
void rddDropAction (w, event, args, nargs)
	Widget w;
	XButtonEvent *event;
	String *args;
	int *nargs;
{
	DEBUG (fprintf(stderr, "rddDropAction\n"););
	if (dragflag == 1)
	{
		XUndefineCursor (display, XtWindow(w));
		rddSendDropEvent(NULL);
	}
}

/*
 *  rddAppMainLoop - 
 * 		Replacement for XtAppMainLoop.
 */
void rddAppMainLoop(app)
	XtAppContext app;
{
    XEvent event;
	Widget w;

    for (;;)
	{
    	XtAppNextEvent(app, &event);

		/*
		 * If it is an rdd message, try to dispatch it to the
		 * currently active drop window. 
		 */
		if (rddIsRddMessageEvent(event))
		{
			DEBUG (fprintf(stderr, "clientMessage received window %d\n",
						   event.xclient.window););
			if (!(event.xclient.window = getPointerWindow()))
			{
				DEBUG (fprintf (stderr, "no window to dispatch to\n"););
				continue;
			}
		}

		if (!XtDispatchEvent(&event))
		{
			DEBUG (fprintf (stderr, "rddAppMainLoop dispatch failed\n"););
		}
	}
}





#ifdef MAIN

/* As an example of how to add drag-n-drop to an existimg program,
 * here is the hello program from chapter 2 of Dan Heller's book,
 * modified so that the button can both initiate a drag-n-drop
 * operation, and respond to something being dropped on it.
 * 
 * RDD comments denote rdd additions and changes.
 */


/* Written by Dan Heller.  Copyright 1991, O'Reilly && Associates.
 * This program is freely distributable without licensing fees and
 * is provided without guarantee or warrantee expressed or implied.
 * This program is -not- in the public domain.
 */

/* hello.c --
 * Initialize the toolkit using an application context and a toplevel
 * shell widget, then create a pushbutton that says Hello using
 * the R4 varargs interface.
 */
#include <X11/Xaw/Command.h>
#include <X11/StringDefs.h>


/*
 * RDD
 * The normal translations for a button, plus those needed to do a 
 * drag-n-drop procedure.
 */
String myTranslations =
		"<Btn1Down>:	highlight() rddStartAction()		\n\
		<Btn1Up>:		set() unhighlight() myDropAction() \n\
		<Btn1Motion>:	rddDragAction() \n\
";


void myDropAction (w, event, args, nargs)
	Widget w;
	XButtonEvent *event;
	String *args;
	int *nargs;
{
	static char *data = "my drop data";
	rddSetDropData (data, strlen(data));		/* copy data to rdd */
	rddDropAction (w, event, args, nargs);		/* then use default action */
}

XtCallbackProc dropTraceProc(w, call, cbs)
	Widget w;
	char *call;
	RddCallbackStruct *cbs;
{
	fprintf (stderr, "dropTraceProc entered w = %s call=%s len=%d data=<%s>\n",
			 XtName(w), call, cbs->len, cbs->data);
}
/* RDD */

main(argc, argv)
	int argc;
	char *argv[];
{
    Widget        toplevel, button;
    XtAppContext  app;
    void i_was_pushed();

    toplevel = XtVaAppInitialize(&app, "Hello", NULL, 0,
        (Cardinal*)&argc, argv, (String*)NULL, (String*)NULL);

	/* RDD add action myAction */
	{
		XtActionsRec actions;
		actions.string	= "myDropAction";
		actions.proc	= myDropAction;
		XtAppAddActions (app, &actions, 1);
	}

	/* RDD  initialize the rdd package */
	rddInit (toplevel, app);

    button = XtVaCreateManagedWidget("pushme",
        commandWidgetClass, toplevel,
        XtNlabel, "Push here to say hello",
        NULL);
    XtAddCallback(button, XtNcallback, i_was_pushed, NULL);


	/* RDD  set new translations on button */
	XtVaSetValues (button,
				   XtNtranslations, XtParseTranslationTable(myTranslations),
				   NULL);

	/* RDD  add drop callback to button */
	rddAddDropHandler (button, (XtCallbackProc)dropTraceProc, "no client data");


    XtRealizeWidget(toplevel);

	/* RDD  use rddAppMainLoop instead of XtAppMainLoop */
    rddAppMainLoop(app);
}

void
i_was_pushed(w, client_data, cbs)
Widget w;
XtPointer client_data;
XtPointer *cbs;
/*
XmPushButtonCallbackStruct *cbs;
*/
{
    printf("Hello Yourself!\n");
}

#endif /*MAIN*/
