.\" SCCSID = @(#)dbug.man	1.1
.\"
.TH DBUG 3L "89/07/20" "K Sys Ap 121"
.SH NAME
dbug \- a toolset for internal debugging of C programs.
.ad
.SH SYNOPSIS
Note, the following "functions" are C macros. Therefore, the description of
the parameters are not necessarily correct C. The intention is, to describe as
close as possible but shortly what type of "data" can be used for the
parameters of the macros.
.PP
#include <dbug.h>
.PP
\fIDBUG_ENTER\fP ( aName )
.RS 6
char *	aName;
.RS -6
.PP
\fIDBUG_RETURN\fP ( aValue )
.RS 6
int	aValue;
.RS -6
.PP
\fIDBUG_VOID_RETURN\fP
.PP
\fIDBUG_PRINT\fP ( aLabelString, (thePrintfParameters) )
.RS 6
char *	aLabelString;
.br
ParameterList thePrintfParameters;
.RS -6
.PP
\fIDBUG_PROCESS\fP ( aName )
.RS 6
char *	aName;
.RS -6
.PP
\fIDBUG_PUSH\fP ( aDbugOptionString )
.RS 6
char *	aDbugOptionString;
.RS -6
.PP
\fIDBUG_POP\fP ()
.PP
\fIDBUG_FILE\fP
.PP
\fIDBUG_EXECUTE\fP ( aLabelString, someCode )
.RS 6
char *	aLabelString;
.br
C-Code	someCode;
.RS -6
.PP
\fIDBUG_SETJMP\fP ( env )
.RS 6
jmp_buf	env;
.RS -6
.PP
\fIDBUG_LONGJMP\fP ( env, val )
.RS 6
jmp_buf	env;
.br
int	val;
.RS -6
.SH DESCRIPTION
\fIDbug\fP is
a macro based C internal debugging
package which has proven to be a very flexible and useful tool
for debugging, testing, and porting C programs.
.PP
Internal debugging are typically "print statements" included into the program
source. \fIDbug\fP provides for enhanced capabilities to do so. In addition
each capability can be individually enabled or disabled at the time a program
is invoked by specifying the appropriate command line arguments.
.PP
So, all of the features of the
.I dbug
package can be enabled or disabled dynamically at execution time.
This means that production programs will run normally when
debugging is not enabled, and eliminates the need to maintain two
separate versions of a program.
.PP
The following specifies a summary of the \fIdbug\fP capabilities:
.TP 3
-
Execution trace showing function level control flow in a
semi-graphically manner using indentation to indicate nesting
depth.
.TP 3
-
Output the values of all, or any subset of, key internal variables.
.TP 3
-
Limit actions to a specific set of named functions.
.TP 3
-
Limit function trace to a specified nesting depth.
.TP 3
-
Label each output line with source file name and line number.
.TP 3
-
Label each output line with name of current process.
.TP 3
-
Push or pop internal debugging state to allow execution with
built in debugging defaults.
.TP 3
-
Redirect the debug output stream to standard output (stdout)
or a named file.
The default output stream is standard error (stderr).
.PP
Many of the things easily accomplished with conventional debugging
tools, such as symbolic debuggers, are difficult or impossible with this
package, and vice versa.
Thus the
.I dbug
package should 
.I not
be thought of as a replacement or substitute for
other debugging tools, but simply as a useful
.I addition
to the
program development and maintenance environment.
.PP
The
.I dbug
package imposes only a slight speed penalty on executing
programs, typically much less than 10 percent, and a modest size
penalty, typically 10 to 20 percent.
By defining a specific C preprocessor symbol both of these
can be reduced to zero with no changes required to the
source code.
.PP
Notice, that all of the debugger functions are implemented
via preprocessor macros.
This does not detract from the readability of the code and makes disabling
all debug compilation trivial (a single preprocessor symbol, 
.B DBUG_OFF ,
forces the macro expansions to be null).
.SH "SUMMARY OF MACROS"
This summarizes the usage of all currently defined macros
in the 
.I dbug
package.
The macros definitions are found in the user include file
.B dbug.h
from the standard include directory.
.TP 15
DBUG_ENTER 
Used to tell the runtime support module the name of the function
being entered.
The argument must be of type "pointer to character".
The 
DBUG_ENTER
macro must precede all executable lines in the
function just entered, and must come after all local declarations.
Each 
DBUG_ENTER
macro must have a matching 
DBUG_RETURN 
or
DBUG_VOID_RETURN
macro 
at the function exit points.
DBUG_ENTER 
macros used without a matching 
DBUG_RETURN 
or 
DBUG_VOID_RETURN
macro 
will cause warning messages from the 
.I dbug
package runtime support module.
.br
.sp
EX:\ DBUG_ENTER\ ("main");
.TP 15
DBUG_RETURN 
Used at each exit point of a function containing a 
DBUG_ENTER 
macro
at the entry point.
The argument is the value to return.
Functions which return no value (void) should use the 
DBUG_VOID_RETURN
macro.
It 
is an error to have a 
DBUG_RETURN 
or 
DBUG_VOID_RETURN 
macro in a function
which has no matching 
DBUG_ENTER 
macro, and the compiler will complain
if the macros are actually used (expanded).
.br
.sp
EX:\ DBUG_RETURN\ (value);
.br
EX:\ DBUG_VOID_RETURN;
.TP 15
DBUG_PROCESS 
Used to name the current process being executed.
A typical argument for this macro is "argv[0]", though
it will be perfectly happy with any other string.
.br
.sp
EX:\ DBUG_PROCESS\ (argv[0]);
.TP 15
DBUG_PUSH 
Sets a new debugger state by pushing the current
.B dbug
state onto an
internal stack and setting up the new state using the debug control
string passed as the macro argument.
The most common usage is to set the state specified by a debug
control string retrieved from the argument list.
Note that the leading "\-#" in a debug control string specified
as a command line argument must
.B not
be passed as part of the macro argument.
The proper usage is to pass a pointer to the first character
.B after
the "\-#" string.
.br
.sp
EX:\ DBUG_PUSH\ (\&(argv[i][2]));
.br
EX:\ DBUG_PUSH\ ("d:t");
.br
EX:\ DBUG_PUSH\ (" ");
.TP 15
DBUG_POP 
Restores the previous debugger state by popping the state stack.
Attempting to pop more states than pushed will be ignored and no
warning will be given.
The 
DBUG_POP 
macro has no arguments.
.br
.sp
EX:\ DBUG_POP\ ();
.TP 15
DBUG_FILE 
The 
DBUG_FILE 
macro is used to do explicit I/O on the debug output
stream.
It is used in the same manner as the symbols "stdout" and "stderr"
in the standard I/O package.
.br
.sp
EX:\ fprintf\ (DBUG_FILE,\ "Doing\ my\ own\ I/O!\\n");
.TP 15
DBUG_EXECUTE 
The DBUG_EXECUTE macro is used to execute any arbitrary C code.
The first argument is the debug keyword, used to trigger execution
of the code specified as the second argument.
This macro must be used cautiously because, like the 
DBUG_PRINT 
macro,
it is automatically selected by default whenever the 'd' flag has
no argument list (I.E., a "\-#d:t" control string).
.br
.sp
EX:\ DBUG_EXECUTE\ ("abort",\ abort\ ());
.TP 15
DBUG_N
These macros, where N is in the range 2\-5, are currently obsolete
and will be removed in a future release.
Use the new DBUG_PRINT macro.
.TP 15
DBUG_PRINT
Used to do printing via the "fprintf" library function on the
current debug stream,
DBUG_FILE.
The first argument is a debug keyword, the second is a format string
and the corresponding argument list.
Note that the format string and argument list are all one macro argument
and
.B must
be enclosed in parenthesis.
.br
.sp
EX:\ DBUG_PRINT\ ("eof",\ ("end\ of\ file\ found"));
.br
EX:\ DBUG_PRINT\ ("type",\ ("type\ is\ %x",\ type));
.br
EX:\ DBUG_PRINT\ ("stp",\ ("%x\ \->\ %s",\ stp,\ stp\ \->\ name));
.TP 15
DBUG_SETJMP
Used in place of the setjmp() function to first save the current
debugger state and then execute the standard setjmp call.
This allows the debugger to restore its state when the
DBUG_LONGJMP macro is used to invoke the standard longjmp() call.
Currently all instances of DBUG_SETJMP must occur within the
same function and at the same function nesting level.
.br
.sp
EX:\ DBUG_SETJMP\ (env);
.TP 15
DBUG_LONGJMP
Used in place of the longjmp() function to first restore the
previous debugger state at the time of the last DBUG_SETJMP
and then execute the standard longjmp() call.
Note that currently all DBUG_LONGJMP macros restore the state
at the time of the last DBUG_SETJMP.
It would be possible to maintain separate DBUG_SETJMP and DBUG_LONGJMP
pairs by having the debugger runtime support module use the first
argument to differentiate the pairs.
.br
.sp
EX:\ DBUG_LONGJMP\ (env,val);
.SH "DEBUG CONTROL STRING"
The debug control string is used to control the "behaviour" of the dbug macros,
when running the application. The \fIdebug control string\fP is passed to the
\fIdbug\fP package via the \fIDBUG_PUSH()\fP macro (note, that the parameter of
this macro may be the value of a program argument; see example above).
.PP
The following list summarizes the currently available debugger options
and the flag characters which enable or disable them. Debugger options may be
concatenated to a \fIdebug control string\fP using the ":" (colon) as a
seperator (e.g., <option1>:<option2>:...:<optionN>).
.PP
Argument lists enclosed in '[' and ']' are optional.
.TP 15
d[,keywords]
Enable output from macros with specified keywords.
A null list of keywords implies that all keywords are selected.
.TP 15
D[,time]
Delay for specified time after each output line, to let output drain.
Time is given in tenths of a second (value of 10 is one second).
Default is zero.
.TP 15
f[,functions]
Limit debugger actions to the specified list of functions.
A null list of functions implies that all functions are selected.
.TP 15
F
Mark each debugger output line with the name of the source file
containing the macro causing the output.
.TP 15
g
Turn on machine independent profiling.
A profiling data collection file, named dbugmon.out, will be written
for postprocessing by the "analyze(1L)" program.
The accuracy of this feature is relatively unknown at this time.
.TP 15
i
Identify the process emitting each line of debug or trace output
with the process id for that process.
.TP 15
L
Mark each debugger output line with the source file line number of
the macro causing the output.
.TP 15
n
Mark each debugger output line with the current function nesting depth.
.TP 15
N
Sequentially number each debugger output line starting at 1.
This is useful for reference purposes when debugger output is
interspersed with program output.
.TP 15
o[,file]
Redirect the debugger output stream to the specified file.
The default output stream is stderr.
A null argument list causes output to be redirected to stdout.
.TP 15
p[,processes]
Limit debugger actions to the specified processes.
A null list implies all processes.
This is useful for processes which run child processes.
Note that each debugger output line can be marked with the name of
the current process via the 'P' flag.
The process name must match the argument passed to the
.B DBUG_PROCESS
macro.
.TP 15
P
Mark each debugger output line with the name of the current process
from argv[0].
Most useful when used with a process which runs child processes that
are also being debugged.
Note that the parent process must arrange for the debugger control
string to be passed to the child processes.
.TP 15
r
Used in conjunction with the 
.B DBUG_PUSH 
macro to reset the current
indentation level back to zero.
Most useful with 
.B DBUG_PUSH 
macros used to temporarily alter the
debugger state.
.TP 15
t[,N]
Enable function control flow tracing.
The maximum nesting depth is specified by N, and defaults to
200.
.SH USAGE
The following example shows a very simple use of the \fIdbug\fP facilities.
.PP
First, the source code including some \fIdbug\fP statements is given:
.RS 12
.sp
.nf
#include <stdio.h>
#include <dbug.h>

main (argc, argv)
int argc;
char *argv[];
{
    int i;
    int atoi();

    DBUG_ENTER ("main");
    DBUG_PROCESS (argv[0]);

    for (i = 1; i < argc && argv[i][0] == '-'; i++) {
	switch (argv[i][1]) {
	    case '#':
		DBUG_PUSH (&(argv[i][2]));
		break;
	}
    }

    for (; i < argc; i++) {
	DBUG_PRINT ("args", ("argv[%d] = %s", i, argv[i]));
	printf ("%F\\n", 1.0/atoi(argv[i]));
    }

    DBUG_RETURN (0);
}
.fi
.sp
.RS -12
.PP
The following invocation of the program "inverse" compiled from the source
above would cause the corresponding outputs:
.RS 12
.sp
.nf
$ inverse\ \ 2 4
0.5
0.25

$ inverse -#t\ \ 2 4
|
0.5
0.25
< main

inverse -#d:t\ \ 2 4
| args: argv[2] = 2
0.5
| args: argv[3] = 4
0.25
< main
.fi
.sp
.RS -12
.SH FILES
/LocalTools/lib/local/libdbug.a
/LocalTools/include/local/dbug.h
.SH "SEE ALSO"
DBUG Users Manual, analyze(1L)
.SH ENVIRONMENT
none
.SH BUGS
There still are some.
.SH COPYRIGHT
none
.SH AUTHORS
Fred Fish, Binayak Banerjee
.br
Enhanced Software Technologies, Tempe, AZ
.br
asuvax!mcdphx!estinc!fnf, seismo!bpa!sjuvax!bbanerje

