/* -*- C -*-
 *
 * Program:	ximap
 * File:        mailcap.c -- Parses mailcap file into structures.  I believe
 *                           the format is specified in RFC1343.  This file
 *                           may be out of date.
 *
 * Author:	Kevin Brock
 *	        Symbolic Systems Resources Group
 *		Stanford University
 *              MSOB x241
 *		Stanford, CA 94305
 *		Internet: brock@CAMIS.Stanford.Edu
 *
 * Date:	07 September 1992
 *
 * Copyright 1992 by The Leland Stanford Junior University.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notices appear in all copies and that both the
 * above copyright notices and this permission notice appear in supporting
 * documentation, and that the name of The Leland Stanford Junior University 
 * not be used in advertising or publicity pertaining to distribution of the 
 * software without specific, written prior permission.  This software is made 
 * available "as is", and
 * THE LELAND STANFORD JUNIOR UNIVERSITY DISCLAIMS ALL WARRANTIES, EXPRESS OR 
 * IMPLIED, WITH REGARD TO THIS SOFTWARE, INCLUDING WITHOUT LIMITATION ALL IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, AND IN NO EVENT 
 * SHALL THE LELAND STANFORD JUNIOR UNIVERSITY BE LIABLE FOR ANY SPECIAL, INDIRECT 
 * OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, 
 * DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, TORT (INCLUDING NEGLIGENCE) 
 * OR STRICT LIABILITY, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE 
 * OF THIS SOFTWARE.
 *
 */
#include <stdio.h>
#include <ctype.h>
#include <string.h>

#include <sys/file.h>

#include "Client/osdep.h"
#include "Client/misc.h"

#include "mailcap.h"

static char *MIMEtypes[] = 
{
    "text",
    "multipart",
    "message",
    "application",
    "audio",
    "image",
    "video",
    "x-token"
};

static HandlerPtr parseEntry();
static char*      getViewCommand();
static char*      MIMEtoken();
static void       dumpWhitespace();
static int        dumpLine();
static int        isMIMEtspecial();
static int        isMTEXTspecial();
static int        getType();

extern char* firstfound();

HandlerArray parseMailcap( mailcappath )
     char *mailcappath;
{
    HandlerPtr    htemp, temp;
    HandlerArray  ret;
    FILE         *ifile;
    int           i, done = 0;
    char          buffer[MAILCAPBFRSIZE];
    char         *filename = firstfound(mailcappath);
    int status;

    if(filename == NULL)
    {
	fprintf(stderr, "error: parseMailcap: could not find mailcap file\n");
	return( NULL );
    }	
    
    if((ifile = fopen(filename, "r")) == NULL)
    {
	fprintf(stderr, "error: parseMailcap: could not open file: %s\n", filename);
	return( NULL );
    }

    ret = (HandlerArray) malloc (NUMBEROFTYPES * sizeof(HandlerPtr));
    if( ret == NULL )
    {
	fprintf(stderr, "error: parseMailcap: malloc failed.\n");
	return( NULL );
    }
    
    for( i = 0; i < NUMBEROFTYPES; i++ )
	ret[i] = (HandlerPtr) NULL;

    while(!done)
    {
	status = getMailcapLine(ifile, buffer, MAILCAPBFRSIZE);
	if(status == EOF)
	{

	    done = 1;
	}
	else
	{
	    htemp = parseEntry(buffer);
	    if( htemp == NULL )
	    {
		fprintf(stderr, "error: parseMailcap: bad mailcap entry.\n");
		return(NULL);
	    }

	    if( ret[htemp->contentType] != NULL )
	    {
		temp = ret[htemp->contentType];
		if( htemp->contentType == TYPEOTHER )
		{
		    do
		    {
			if( !strcasecmp(temp->asciiType,htemp->asciiType))
			{
			    if( (temp->subType == NULL &&  htemp->subType == NULL)
			       || (temp->subType != NULL 
				   &&  htemp->subType != NULL
				   && !strcasecmp(temp->subType,htemp->subType)))
			    {
				free(htemp->view);
				free(htemp->subType);
				free(htemp);
				htemp = NULL;
				fprintf(stderr, "warning: parseMailcap: two entries for same type.\n");
				break;
			    }
			}
		    } while( temp = temp->next);
		}
		else
		{
		    if ( htemp->subType != NULL )
		    {
			Handler *ht = NULL;
			do
			{
			    if( !strcasecmp(temp->subType,htemp->subType))
			    {
				if(ht)
				{
				    ht->next = htemp;
				    htemp->next = temp->next;
				}
				else
				{
				    htemp->next = ret[htemp->contentType];
				    ret[htemp->contentType] = htemp;
				}
/*				free(htemp->view);
				free(htemp->subType);
				free(htemp);
				htemp = NULL;
				fprintf(stderr, "warning: parseMailcap: two entries for same type.\n");
*/
				htemp = NULL;
				break;
			    }
			    ht = temp;
			} while( temp = temp->next);
		    }
		}
	    }
	
	    
	    if(htemp)
	    {
		htemp->next = ret[htemp->contentType];
		ret[htemp->contentType] = htemp;
	    }
	}
    }
    return( ret );
}

static HandlerPtr parseEntry(buffer)
     char *buffer;
{
    HandlerPtr ret;
    int type = -1;
    char *subtype = NULL;
    char *viewCommand = NULL;
    char *temp = buffer;

    ret = (HandlerPtr)malloc(sizeof(Handler));
    
    dumpWhitespace(&temp);

    type = getType(&temp, &ret->asciiType);
    if( type < 0 || type > NUMBEROFTYPES-1 )
	return(NULL);
	
    if( *temp == '/' )
    {
	temp++;
	subtype = MIMEtoken(&temp);
	if( subtype == NULL )
	    return(NULL);
    }

    dumpWhitespace(&temp);
    if( *temp++ != ';' )
	return(NULL);
    dumpWhitespace(&temp);
    
    viewCommand = getViewCommand(&temp);
    if(viewCommand == NULL)
	return(NULL);

    ret->contentType = type;
    if(subtype)
	ret->subType = subtype;
    else
	ret->subType = NULL;
    ret->view = viewCommand;

    return(ret);
}

static int getType(bufferPtr, asciiType)
     char **bufferPtr;
     char **asciiType;
{
    int i=0, found = -1;

    *asciiType = MIMEtoken(bufferPtr);

    if(*asciiType)
    {
	while(i < NUMBEROFTYPES-1 && found < 0)
	{
	    if( !strcasecmp( *asciiType, MIMEtypes[i] ))
		found = i;
	    else
		i++;
	}

	if(found < 0)
	    if(!strncasecmp(X_TOKENPREFIX,*asciiType, 2))
		found = i;

    }
    return(found);
}

static void dumpWhitespace(buffer)
     char** buffer;
{
    while(**buffer == ' ' || **buffer == '\t')
	(*buffer)++;
}

static int dumpLine(ifile)
    FILE *ifile;
{
    char c = '\0';
    
    while((c = getc(ifile)) != '\n'
	  && c != EOF)
	;
}

static char *MIMEtoken(bufferPtr)
     char **bufferPtr;
{
    char token[MAILCAPTOKSIZE];
    char *temp = token;

    dumpWhitespace(bufferPtr);
    while(!iscntrl(**bufferPtr)
	  && !isMIMEtspecial(**bufferPtr)
	  && **bufferPtr != ' ')
	*temp++ = *(*bufferPtr)++;

    *temp = '\0';
    temp = (char*) malloc (1 + strlen(token));
    strcpy(temp,token);
    return(temp);
}

static int isMIMEtspecial(c)
     char c;
{
    return((int)strchr(MIMEtspecials, c));
}


static int isMTEXTspecial(c)
     char c;
{
    return((int)strchr(MTEXTspecials, c));
}


static char *getViewCommand(bufferPtr)
     char** bufferPtr;
{
    int done = 0;
    char  mtoken[MAILCAPBFRSIZE];
    char *temp = mtoken;

    while(!done)
    {
	while(!iscntrl(**bufferPtr)
	      && !isMTEXTspecial(**bufferPtr))
	    *temp++ =  *(*bufferPtr)++;

	if(**bufferPtr == '\\')
	{
	    *temp++ = *++*bufferPtr;
	    (*bufferPtr)++;
	}
	else
	{
	    done = 1;
	}
    }
    *temp = '\0';
    temp = (char*) malloc ( 1 + strlen(mtoken));
    strcpy(temp,mtoken);
    return(temp);
}


/*
 *  Takes a colon separated list of alternative files, returns
 *  the first one which actually exists
 * 
 */
char *firstfound( path )
     char *path;
{
    char part[256];
    register char *ptemp;

    if(!path)
	return(NULL);
	
    while( *path )
    {
	ptemp = part;
	*ptemp =  '\0';
	while( *path && *path != ':' )
	    *ptemp++ = *path++;
	path++;
	*ptemp =  '\0';
	if(part[0] && !access(part, F_OK))
	{
	    char* temp = cpystr(part);
	    return(temp);
	}
    }
    
    return(NULL);

}

int getMailcapLine(ifile, buffer, size)
     FILE* ifile;
     char* buffer;
     int size;
{
    char btemp[MAILCAPBFRSIZE];    
    int status;
    char c;

    while( 1 )
    {
	if((status = fscanf(ifile,"%[^\n]", buffer)) == EOF)
	{
	    return(EOF);
	}
	else
	{
	    /* There's an empty line -- discard newline and continue. */
	    if(status == 0)
	    {
		c = getc(ifile);
		if(c != '\n')
		    ;/* there should be an error message here... */

		continue;
	    }
	}

	if(*buffer == '#')
	{
	    /* It's a comment, get rid of it */
	    status = dumpLine(ifile);
	    if( status == EOF )
		return(status);
	}
	else
	{
	    int l = strlen(buffer);
	    if(buffer[l - 1] == CONTINUELINE)
	    {
		buffer[l-1] = '\0';
		status = getMailcapLine(ifile, btemp, MAILCAPBFRSIZE);
		if( status == EOF )
		    return(status);
		strcat(buffer, btemp);
	    }
	    
	    c = getc(ifile);
	    if( c != '\n' )
		; /* error message */
	    
	    return(c);
	}
    }
}
