/* -*- C -*-
 *
 * Program:	ximap
 * File:        mboxlist.c -- Creates a window to get a mailbox name.  There's
 *                            a browser of the .mailboxlist file, and a text 
 *                            entry field.
 *
 * Author:	Kevin Brock
 *	        Symbolic Systems Resources Group
 *		Stanford University
 *              MSOB x241
 *		Stanford, CA 94305
 *		Internet: brock@CAMIS.Stanford.Edu
 *
 * Date:	07 September 1992
 *
 * Idosynchrasies - This uses the Xb library and XtNdoubleClick.
 *    The simulation of double-click is done by always calling
 *    the single-click callback first followed by the double-click
 *    callback if appropriate. 
 *
 * Copyright 1992 by The Leland Stanford Junior University.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notices appear in all copies and that both the
 * above copyright notices and this permission notice appear in supporting
 * documentation, and that the name of The Leland Stanford Junior University 
 * not be used in advertising or publicity pertaining to distribution of the 
 * software without specific, written prior permission.  This software is made 
 * available "as is", and
 * THE LELAND STANFORD JUNIOR UNIVERSITY DISCLAIMS ALL WARRANTIES, EXPRESS OR 
 * IMPLIED, WITH REGARD TO THIS SOFTWARE, INCLUDING WITHOUT LIMITATION ALL IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, AND IN NO EVENT 
 * SHALL THE LELAND STANFORD JUNIOR UNIVERSITY BE LIABLE FOR ANY SPECIAL, INDIRECT 
 * OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, 
 * DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, TORT (INCLUDING NEGLIGENCE) 
 * OR STRICT LIABILITY, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE 
 * OF THIS SOFTWARE.
 *
 */
#include <Client/osdep.h>
#include <Client/mail.h>
#include <Client/misc.h>
 
#include <netdb.h>
#include <signal.h>
#include <ctype.h>

#include <stdio.h>
#include <string.h>
#include <memory.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>

#include <X11/Xaw/Paned.h>
#include <X11/Xaw/Box.h>

#ifdef MOTIF
#include <X11/Xm/Text.h>
#else
#include <X11/Xaw/AsciiText.h>
#endif

#include <X11/Xaw/Label.h>

#include <Xb/Browser.h>

#include "structures.h"

#include "message.h"
#include "globals.h"
#include "buttons.h"
#include "resources.h"
#include "textutil.h"
#include "util.h"
#include "ximap.h"
#include "mailbox.h"
#include "mboxlist.h"
#include "browserutil.h"

typedef struct _Mboxlist
{
  Widget shell;
  Widget buttons;
  Widget panes;
  Widget mailboxlist;
  Widget text;

  char   *host;
  char   *mailbox;

  char   **msmailbox;
  char   *mailboxfile;
  int no_boxes;
  StateList *states;
} MBXLIST;

static void mbxlQuit(/*Widget, MBXLIST*, XEvent* */);
static void mbxlOk(/*Widget, MBXLIST*, XEvent* */);
static void mbxlDelete(/*Widget, MBXLIST*, XEvent* */);

void setMbxlSize();

extern Widget telemetry;
extern XtAppContext app_context;
extern AppResources res;

MBXLIST MBoxList;

ButtonStruct MbxlButtons[] = 
{
  { "Quit",  mbxlQuit,   NULL,  0 },
  { "Ok", mbxlOk,        NULL,  0 },
  { "Delete", mbxlDelete,NULL,  0 },
  NULL,
};

  static void
mbxlOkAct ( w, e )
  Widget w;
  XEvent* e;
{
  mbxlOk( w, &MBoxList, NULL );
}

  static void
mbxlQuitAct ( w, e )
  Widget w;
  XEvent* e;
{
  mbxlQuit( w, &MBoxList, e );
}

  static void
mbxlDefault( w, e )
  Widget w;
  XEvent e;
{
  ;
}

static XtActionsRec mbxlActs[] =
{
  { "mbxlOkAct",   mbxlOkAct },
  { "mbxlQuitAct" ,mbxlQuitAct },
  { "mbxlDefault", mbxlDefault }
};

void mailboxList( w, ms, e )
     Widget w;
     MailBox* ms;
     XEvent *e;
{
    Arg warg[16];
    int n = 0;
    
    XbListSelectionPtr header_list = NULL;
    
    char *home;
    
    XFontStruct *font;
    
    MBXLIST *mbxl = &MBoxList;
    
    static XbListSelectionPtr getMailboxListMap(/* MBXLIST*  */);
    void swapText(/* Widget, MBXLIST*, XbListSelectionPtr */);
    
    home = getHomeDir();
    mbxl->mailboxfile = XtMalloc (strlen(home) + strlen("/.mailboxlist") + 1);
    strcat( strcpy(mbxl->mailboxfile, home), "/.mailboxlist" );
    
    XtAppAddActions( app_context, mbxlActs, XtNumber(mbxlActs));
    
    mbxl->msmailbox = &(ms->mailboxName);
    if(res.default_mailbox)
    {
	parseMailboxName(res.default_mailbox, 
			 &mbxl->host,
			 &mbxl->mailbox);

	if(mbxl->host == NULL)
	{
	    if(res.default_host)
		mbxl->host = cpystr(res.default_host);
	}
    }
    else
    {
	mbxl->mailbox = cpystr("");
    }
    
    mbxl->shell = XtCreatePopupShell("mailboxlist_shell", topLevelShellWidgetClass,
				     w, warg, n); n = 0;
    
    mbxl->panes = XtCreateManagedWidget("mbxl_panes", panedWidgetClass,
				      mbxl->shell, warg, n); n = 0;
    
    XtSetArg(warg[n], XtNshowGrip, FALSE); n++;
    mbxl->buttons = XtCreateManagedWidget("mbxl_buttons", boxWidgetClass,
					  mbxl->panes,  warg, n); n = 0;
    
    if( header_list = getMailboxListMap( mbxl ) )
    {
	XtSetArg(warg[n], XtNinitialViewHt, 10); n++;
	XtSetArg(warg[n], XtNlistPosition, 0); n++;
	XtSetArg(warg[n], XtNlist, header_list); n++;
	XtSetArg(warg[n], XtNshowGrip, FALSE); n++;
    
	mbxl->mailboxlist = XtCreateManagedWidget("mbxl_list", 
						  XbbrowserWidgetClass,
						  mbxl->panes, warg, n); n = 0;
	XtAddCallback( mbxl->mailboxlist, XtNsingleClick, swapText, mbxl );
	XtAddCallback( mbxl->mailboxlist, XtNdoubleClick, mbxlOk, mbxl );
    }

    XtSetArg(warg[n], XtNeditType, XawtextEdit); n++;
    XtSetArg(warg[n], XtNshowGrip, FALSE); n++;
    XtSetArg(warg[n], XtNskipAdjust, TRUE); n++;
    mbxl->text = XtCreateManagedWidget("mbxl_text", asciiTextWidgetClass, 
				       mbxl->panes, warg, n); n = 0;	
    XtSetArg(warg[0], XtNfont, &font); n++ ;
    XtGetValues(mbxl->text, warg, 1); n = 0;
    XtSetArg(warg[0], XtNheight,
	     (font->max_bounds.ascent + font->max_bounds.descent) + 6);n++ ;
    XtSetValues(mbxl->text, warg, 1); n = 0;
    
    if( mbxl->mailbox)
    {
	char *temp = NULL;
	if(mbxl->host)
	{
	    temp = (char*)fs_get( 1 + strlen(mbxl->host) + 1 + strlen(mbxl->mailbox) + 1 );
	    sprintf(temp,"{%s}%s",mbxl->host, mbxl->mailbox);
	}
	else
	{
	    temp = (char*)fs_get( strlen(mbxl->mailbox) + 1 );
	    sprintf(temp,"%s",mbxl->mailbox);
	}

	XbTextAppend(mbxl->text, temp);
	fs_give(&temp);
    }
    
    createButtons(mbxl->buttons, 
		  mbxl,
		  MbxlButtons);

    setMbxlSize(mbxl);
    XtPopup(mbxl->shell, XtGrabNone);
    
    XtSetKeyboardFocus( mbxl->shell, mbxl->text );
}

void setMbxlSize( mbxl )
     MBXLIST *mbxl;
{
    Arg warg[ARGLISTSIZE];
    int n = 0;
    
    Dimension box_height = 0;
    Dimension list_height = 0;
    Dimension int_border_width = 0;
    Dimension text_height = 0;

    Dimension height = 0;
    Dimension width = 0;
    Dimension fixed_height = 0;

    XFontStruct *font;

    XtWidgetGeometry intended, result;

    XtSetArg(warg[n], XtNfont, &font); n++ ;
    XtGetValues(mbxl->text, warg, n); n = 0;
    
    text_height = font->max_bounds.ascent + font->max_bounds.descent + 6;

    XtSetArg(warg[n], XtNheight, &list_height); n++;
    XtSetArg(warg[n], XtNwidth, &width); n++;
    XtGetValues( mbxl->mailboxlist, warg, n ); n = 0;

    intended.request_mode = CWWidth | XtCWQueryOnly;
    intended.width = width;

    XtQueryGeometry( mbxl->buttons, &intended, &result);

    box_height = result.height;

    XtSetArg(warg[n], XtNinternalBorderWidth, &int_border_width); n++;
    XtGetValues(mbxl->panes,warg,n); n = 0;
    
    /* the user can't change the fixed_height... */
    fixed_height = box_height + text_height + 2*(int_border_width);

    height = list_height + fixed_height;

    XtSetArg(warg[n], XtNheight, list_height); n++;
    XtSetValues( mbxl->mailboxlist, warg, n ); n = 0;

    XtSetArg(warg[n], XtNwidth, width); n++;
    XtSetArg(warg[n], XtNheight, height); n++;
    XtSetValues(mbxl->panes,warg,n); n = 0;
}

static XbListSelectionPtr getMailboxListMap( mbxl )
     MBXLIST* mbxl;
{
    FILE *ifptr = NULL;
    char temp[TMPLEN];
    char *htmp[TMPLEN];
    
    XbListSelectionPtr result = NULL;

    int n = 0;
    
    memset( temp, 0, TMPLEN );
    memset( htmp, 0, TMPLEN );
    
    if ((ifptr = fopen(mbxl->mailboxfile, "r")) == NULL)
    {
	if( mbxl->mailbox )
	{
	    result = ximap_newlistmap(1);
	    result->entries[0].entry = malloc(strlen(mbxl->mailbox)+1);
	    strcpy(result->entries[0].entry,mbxl->mailbox);
	    result->entries[0].index = 0;
	}
	else
	{
	    result = ximap_newlistmap(0);
	}
    }
    else
    {
	while ( fscanf( ifptr, "%s", temp ) != EOF )
	{
	    htmp[n] = malloc(1+strlen(temp));
	    strcpy(htmp[n],temp);
	    n++;
	    memset( temp, 0, TMPLEN );
	}
	fclose(ifptr);

	result = ximap_newlistmap(n);
	while( n-- )
	{
	    result->entries[n].entry = malloc(1+strlen(htmp[n]));
	    strcpy(result->entries[n].entry,htmp[n]);
	    if(htmp[n])
		free(htmp[n]);
	    result->entries[n].index = n;
	}
    }
    mbxl->no_boxes = result->size;
    return(result);
}

static void mbxlQuit( w, mbxl, e )
     Widget w;
     MBXLIST *mbxl;
     XEvent *e;
{
    free_statelist(mbxl->states);

    if( !*mbxl->msmailbox )
	*mbxl->msmailbox = cpystr("");

    if ( mbxl && mbxl->shell )
	XtDestroyWidget( mbxl->shell );
}					

static void mbxlOk( w, mbxl, map )
     Widget w;
     MBXLIST *mbxl;
     XbListSelectionPtr map;
{
    char *temp;
    mbxl->mailbox = NULL;
    
    if (mbxl && mbxl->text)
    {
	temp = (char*) XbTextCopyBuffer( mbxl->text );
	parseMailboxName(temp, &mbxl->host, &mbxl->mailbox);
    }

    *mbxl->msmailbox = temp;
    
    mbxlQuit( w, mbxl, NULL );
}					

static void swapText( w, mbxl, map )
     Widget w;
     MBXLIST *mbxl;
     XbListSelectionPtr map;
{
    if ( map->size == 1 )
    {
	char *txt = NULL;
	XbListSelectionPtr temp = XbBrowserCurrentHighlight(mbxl->mailboxlist);
	txt = cpystr(temp->entries[0].entry);
	XbTextClearBuffer(mbxl->text);
	XbTextAppend(mbxl->text,txt);
	XbListFreeSelection(temp);
	if(txt)
	    XtFree(txt);
    }
    else
    {
	XbTextClearBuffer(mbxl->text);
	mbxl->mailbox = NULL;
    }
}

void updateMailboxList( ms )
     MailBox* ms;
{
    char temp[TMPLEN];
    int match = FALSE;
    FILE *ifptr = NULL;
    char *home;
    char *mailboxlist;
    
    memset(temp, 0, TMPLEN);
    
    fixnl(ms->mailboxName);
    
    home = getHomeDir();
    mailboxlist = XtMalloc(strlen(home) + strlen("/.mailboxlist") + 1);
    strcat( strcpy(mailboxlist, home), "/.mailboxlist" );
    
    if ((ifptr = fopen(mailboxlist, "r+")) == NULL)
    {
	if ((ifptr = fopen(mailboxlist, "w")) == NULL)
	{
	    exit(0);
	}
	else
	{
	    fprintf( ifptr, "%s\n", ms->mailboxName );
	    fclose(ifptr);
	}
    }
    else
    {
	while (fscanf( ifptr, "%s", temp ) != EOF )
	{
	    fixnl(temp);
	    if (strcmp( ms->mailboxName, temp ))
	    {
		memset(temp, 0, TMPLEN);
	    }
	    else
	    {
		match = TRUE;
		break;
	    }
	}
	if (match == FALSE)
	{
	    fseek( ifptr, 0, 2);
	    fprintf( ifptr, "%s\n", ms->mailboxName );
	}
	
	fclose(ifptr);
    }
}

static void mbxlDelete( w, mbxl, e )
     Widget w;
     MBXLIST *mbxl;
     XEvent *e;
{
    char temp[TMPLEN];
    FILE *ifptr = NULL;
    int size = mbxl->no_boxes;
    XbListSelectionPtr sel = NULL;
    XbListSelectionPtr store = NULL;
    char   **toBeDeleted;
    int count = 0;
    int current = 0;
    
    memset(temp, 0, TMPLEN);
    
    sel = XbBrowserCurrentHighlight(mbxl->mailboxlist);
    store = ximap_newlistmap(size);
    if ( sel->size != 0 ) 
    {
	count = 0;
	toBeDeleted = (char**) XtMalloc (sel->size*sizeof(char*));
	
	while ( count < sel->size )
	{      
	    toBeDeleted[count] = cpystr(sel->entries[count].entry);
	    ++count;
	}

	count = 0;
	if ((ifptr = fopen(mbxl->mailboxfile, "r+")) == NULL)
	{
	    exit(0);
	}
	else
	{
	    while (fscanf( ifptr, "%s", temp ) != EOF && size != 0 )
	    {
		if ( count >= sel->size 
		    || strcmp( toBeDeleted[count], temp ) )
		{
		    store->entries[current].entry = cpystr(temp);
		    store->entries[current].index = current;
		    current++;
		}
		else
		{
		    --mbxl->no_boxes;
		    ++count;
		}
		memset(temp, 0, TMPLEN);
		size--;
	    }

	    store->size = current;
	    XbBrowserChangeItems(mbxl->mailboxlist, store);
	    
	    freopen( mbxl->mailboxfile, "w+", ifptr );
	    for( count = 0; count < mbxl->no_boxes ; ++count )
		fprintf( ifptr, "%s\n", store->entries[count].entry );
	    
	    fclose(ifptr);
	    XtFree((char*) toBeDeleted);
	}
    }
    XbListFreeSelection(sel);
}




