/* -*- C -*-
 *
 * Program:	ximap
 * File:        message.c -- Fns dealing with MessageNodes.
 *
 * Author:	Kevin Brock
 *	        Symbolic Systems Resources Group
 *		Stanford University
 *              MSOB x241
 *		Stanford, CA 94305
 *		Internet: brock@CAMIS.Stanford.Edu
 *
 * Date:	07 September 1992
 *
 * Copyright 1992 by The Leland Stanford Junior University.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notices appear in all copies and that both the
 * above copyright notices and this permission notice appear in supporting
 * documentation, and that the name of The Leland Stanford Junior University 
 * not be used in advertising or publicity pertaining to distribution of the 
 * software without specific, written prior permission.  This software is made 
 * available "as is", and
 * THE LELAND STANFORD JUNIOR UNIVERSITY DISCLAIMS ALL WARRANTIES, EXPRESS OR 
 * IMPLIED, WITH REGARD TO THIS SOFTWARE, INCLUDING WITHOUT LIMITATION ALL IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, AND IN NO EVENT 
 * SHALL THE LELAND STANFORD JUNIOR UNIVERSITY BE LIABLE FOR ANY SPECIAL, INDIRECT 
 * OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, 
 * DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, TORT (INCLUDING NEGLIGENCE) 
 * OR STRICT LIABILITY, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE 
 * OF THIS SOFTWARE.
 *
 */
#include <string.h>

#include "Client/osdep.h"
#include "Client/mail.h"
#include "Client/misc.h"
#include "Client/rfc822.h"

#include <X11/Intrinsic.h>

#include "structures.h"
#include "message.h"
#include "globals.h"

char *typeAcronyms[] =
{
    "TXT",
    "MLT",
    "MSG",
    "APP",
    "AUD",
    "IMG",
    "VID",
    "OTH",
    NULL

};

void parseBodyContents();

void free_node(node)
     MessageNode* node;
{
    if (node
	&& node->locked <= 0)
    {
	/* 
	  these should all be okay, because XtFree
	  is supposed to be able to handle NULL
	  as an argument...
	  */
	XtFree((char*)node->text);
	XtFree((char*)node->header);
	XtFree((char*)node->filtered_head);
	XtFree((char*)node->shortheader);
	XtFree((char*)node->header_line);

	XtFree((char*)node);
    }
    else if (node)
    {
	node->status |= DEADNODE;
    }
}

MessageNode* ximap_newnode()
{
    MessageNode* new = (MessageNode*) XtMalloc (sizeof(MessageNode));

    if (new)
    {
	new->text = 
	    new->header = new->filtered_head = new->shortheader = NULL;

	new->header_line = NULL;
	new->elt = NULL;
	new->body = NULL;
	new->env = NULL;
	new->msgno = -1;
	new->status = 0;
	new->locked = 0;
	new->visible = 0;
	new->insert = NULL;
	new->toMe = FALSE;
	new->changed = 0;
    }

    return new;
}

NodeList* ximap_newnodelist( nodes, number )
     MessageNode* *nodes;
     int          number;
{
    NodeList* retval = (NodeList*)XtMalloc(sizeof(NodeList));

    retval->nodes = (MessageNode**)XtMalloc(number*sizeof(MessageNode*));
    retval->length = number;

    while(number--)
	retval->nodes[number] = nodes[number];

    return(retval);
}

/*
 * Creates a formatted header line in text, for display in the 
 *  header browser:
 *     <Flags> <Date> <From> <Keywords> <Subject> (<Size>)
 *
 *  -> text must have enough space for the header line
 *
 */
void  headerline (text, stream, node)
     char *text;
     MAILSTREAM *stream;
     MessageNode *node;
{
    Boolean empty_line(/* char* line */);
    long i;
    int msgno;
    char from[FROMLEN+1];
    char sub[TMPLEN];
    char mnum[MNLENGTH+1];
    char flgs[FLAGSLEN+1];
    char date[MAILTMPLEN];
    char type[4];

    if (!(stream && node->elt)) 	/* want a prototype? */
    {
	strcpy (mnum, MNUMBER);   
	strcpy (flgs,FLAGS);	
	strcpy (date,DATE);	
	strcpy(type, typeAcronyms[0]);
	
	strcpy(from, FROM);		/* dummy from */
	
	strcpy (sub,SUBJECT);
	
	i = 999999;			/*  humongous size */
    }
    else 
    {			
	if(node->body)
	{
	    strcpy(type, typeAcronyms[node->body->type]);
	}
	else
	{
	    strcpy(type, typeAcronyms[0]);
	}

	msgno = node->msgno;		/*  set up message number */
	
	sprintf( mnum, "%4d", msgno );
	
	/*  if cache not loaded, load it now */
	flgs[0] = node->elt->recent ? (node->elt->seen ? 'R' : 'N'):
	                              (node->elt->seen ? ' ' : 'U');
	flgs[1] = node->elt->flagged ? 'F' : ' ';
	flgs[2] = node->elt->answered ? 'A' : ' ';
	flgs[3] = node->elt->deleted ? 'D' : ' ';
	
	/*  only use "dd-mmm" from date */
	mail_date(date, node->elt);
	date[6] = '\0';

	mail_fetchfrom(from, stream, node->msgno, FROMLEN);

	if (i = node->elt->user_flags) /*first stash user flags into subject */
	{	
	    sub[0] = '{';	/*  open brace for keywords */
	    sub[1] = '\0';	/*  tie off so mstrct starts off right */
	    
	    while (i) /*  output each keyword */
	    {		
		strcat (sub,stream->user_flags[find_rightmost_bit (&i)]);
		
		/*  followed by space if still more keywords */
		if (i) 
		    strcat (sub," ");
	    }
	    strcat (sub,"} ");	/*  close brace and space before subject */
	}
	else 
	{
	    sub[0] = '\0';	/*  no user flags in front of subject */
	}

	/*  now append the subject */
	mail_fetchsubject(sub+strlen(sub), stream, node->msgno, (SUBJECTLEN-strlen(sub)));

	i = node->elt->rfc822_size;	/*  get message size */
	/*  tie off strings */
	flgs[FLAGSLEN-1] = '\0';
    }
    

    /*  output what we got */

    sprintf (text,"%s %s%s %s %s %s (%d chars)",mnum, flgs,date,from,type,sub,i);

}

/*
 * Returns true if the line has no printable chars
 *
 */
Boolean empty_line(line)
     register char* line;
{
    while (*line && ( *line == ' ' || *line == '\t'))
	line++;

    return (*line == '\0' || *line == CRCHAR || *line == LFCHAR);	   
}

/*
 * Some versions of xrn save News messages with a bad
 * format.  If the IMAP server can't find a "From," this
 * checks for a ">From."  Yes, it's a hack.
 *
 */
char* parse_news_from(stream, msgno, length)
     MAILSTREAM *stream;
     int msgno;
     int length;
{
    char temp[TMPLEN];

    char *rettmp = NULL;
    char *ret = NULL;
    char *retval = NULL;

    char *htmp = temp;
    char *rfc822_header;
    
    int i = 0;
    
    rfc822_header = mail_fetchheader(stream, msgno);
    
    if ((rfc822_header = (char*) strstr(rfc822_header, ">From:")) != NULL)
    {
	while (*rfc822_header
	       && *rfc822_header != CRCHAR
	       && *rfc822_header != LFCHAR)
	{   
	    *htmp++ = *rfc822_header++;
	}
	*htmp = *rfc822_header;
	
	/* we won't need more than (length*sizeof(char)) bytes.. */
	retval = ret = XtMalloc (length +1);
	
	/* 
	  if there's a personal name in the FROM line
	  copy it and ignore the rest of the stuff...
	  */
	if ((rettmp = strchr(temp, '(')) != NULL)
	{
	    rettmp++;
	    while (*rettmp == ' '
		   || *rettmp == '\t')
		rettmp++;

	    while(   *rettmp != '\0' 
		  && *rettmp != ')'
		  && i < length)
	    {
		*retval++ = *rettmp++;
		i++;
	    }
	}
	else
	{
	    /* 7 == strlen(">From: ") */
	    rettmp = temp + 7;

	    while (*rettmp == ' '
		   || *rettmp == '\t')
		rettmp++;

	    while(   *rettmp != '\0'
		  && *rettmp != CRCHAR
		  && *rettmp != LFCHAR
		  && i < length)
	    {
		*retval++ = *rettmp++;
		i++;
	    }
	}
	
	for( ; i < length; i++)
	{
	    *retval++ = ' ';
	}
	*retval = '\0';
    }
    
    return(ret);
}

/*
 * If no "To" field, checks for a "Newsgroups" field (saved News messages).
 *
 */
char* parse_news_newsgroups(stream, msgno, length)
     MAILSTREAM *stream;
     int msgno;
     int length;
{
    char temp[TMPLEN];

    char *rettmp = NULL;
    char *ret = NULL;
    char *retval = NULL;

    char *htmp = temp;
    char *rfc822_header;
    
    int i = 0;
    
    rfc822_header = mail_fetchheader(stream, msgno);
    
    if ((rfc822_header = (char*) strstr(rfc822_header, "Newsgroups:")) != NULL)
    {
	while (*rfc822_header
	       && *rfc822_header != CRCHAR
	       && *rfc822_header != LFCHAR)
	{   
	    *htmp++ = *rfc822_header++;
	}
	*htmp = *rfc822_header;
	
	/* we won't need more than (length*sizeof(char)) bytes.. */
	retval = ret = XtMalloc (length +1);
	
	rettmp = temp;

	while(   *rettmp != '\0'
	      && *rettmp != CRCHAR
	      && *rettmp != LFCHAR
	      && i < length)
	{
	    *retval++ = *rettmp++;
	    i++;
	}

	
	for( ; i < length; i++)
	{
	    *retval++ = ' ';
	}
	*retval = '\0';
    }
    
    return(ret);
}


void parse_multimedia( stream, node, host )
     MAILSTREAM  *stream;
     MessageNode *node;
     char        *host;
{
    char temp[2*TMPLEN];

    char *text, *header;

    text = mail_fetchtext( stream, node->msgno );
    header = mail_fetchheader( stream, node->msgno );

    rfc822_parse_msg(&node->env, &node->body,
		     header, strlen(header),
		     text, strlen(text),
		     host, temp);

    node->header = header;
    parseBodyContents(node->body, text);
    free(text);
}


void parse_message(ms, node)
     MailBox     *ms;
     MessageNode *node;
{
    char temp[2*TMPLEN];

    char *text, *header;

    text = mail_fetchtext( ms->mailbox, node->msgno );
    header = mail_fetchheader( ms->mailbox, node->msgno );

    rfc822_parse_msg(&node->env, &node->body,
		     header, strlen(header),
		     text, strlen(text),
		     ms->host, temp);

    node->header = header;
    parseBodyContents(node->body, text);
    free(text);
}

void parseBodyContents(body, text)
     BODY *body;
     char *text;
{
    PART *pptr;

    switch(body->type)
    {
      case TYPETEXT:
	body->contents.text = (unsigned char *)fs_get(body->size.bytes + 1);
	bcopy(text, body->contents.text, body->size.bytes);
	*(body->contents.text+body->size.bytes) = '\0';
	break;
      case TYPEMULTIPART:
	pptr = body->contents.part;
	while(pptr)
	{
	    parseBodyContents(&(pptr->body), text+pptr->offset);
	    pptr = pptr->next;
	}
	break;
      case TYPEMESSAGE:
	break;
      case TYPEAPPLICATION:
      case TYPEAUDIO:
      case TYPEIMAGE:
      case TYPEVIDEO:
	body->contents.binary = (void *) malloc(body->size.bytes*sizeof(void*));
	bcopy(text, body->contents.binary, body->size.bytes);
	break;
      case TYPEOTHER:
      default:
	break;
    };
}



