/* -*- C -*-
 *
 * Program:	ximap
 * File:        util.c -- Whatever didn't end up somewhere else :-)
 *
 * Author:	Kevin Brock
 *	        Symbolic Systems Resources Group
 *		Stanford University
 *              MSOB x241
 *		Stanford, CA 94305
 *		Internet: brock@CAMIS.Stanford.Edu
 *
 * Date:	07 September 1992
 *
 * Copyright 1992 by The Leland Stanford Junior University.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notices appear in all copies and that both the
 * above copyright notices and this permission notice appear in supporting
 * documentation, and that the name of The Leland Stanford Junior University 
 * not be used in advertising or publicity pertaining to distribution of the 
 * software without specific, written prior permission.  This software is made 
 * available "as is", and
 * THE LELAND STANFORD JUNIOR UNIVERSITY DISCLAIMS ALL WARRANTIES, EXPRESS OR 
 * IMPLIED, WITH REGARD TO THIS SOFTWARE, INCLUDING WITHOUT LIMITATION ALL IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, AND IN NO EVENT 
 * SHALL THE LELAND STANFORD JUNIOR UNIVERSITY BE LIABLE FOR ANY SPECIAL, INDIRECT 
 * OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, 
 * DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, TORT (INCLUDING NEGLIGENCE) 
 * OR STRICT LIABILITY, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE 
 * OF THIS SOFTWARE.
 *
 */
#include <netdb.h>
#include <signal.h>
#include <ctype.h>

#include <stdio.h>
#include <memory.h>
#include <string.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/file.h>

#include <Client/osdep.h>
#include <Client/mail.h>
#include <Client/misc.h>
#include <Client/smtp.h>
#include <Client/rfc822.h>
 
#include <X11/Xlib.h>
#include <X11/Xutil.h>

#include <X11/Intrinsic.h> 
#include <X11/StringDefs.h>
#include <X11/Shell.h>

#include <X11/Xaw/AsciiText.h>
#include <X11/Xaw/Dialog.h>
#include <X11/Xaw/Paned.h>

#include <X11/Xmu/Converters.h>

#include <X11/Xaw/Text.h>

#include "Xb/Browser.h"

#include "structures.h"

#include "globals.h"
#include "resources.h"
#include "buttons.h"
#include "message.h"
#include "textutil.h"
#include "browserutil.h"
#include "addrutil.h"
#include "util.h"
#include "ximap.h"
#include "mailbox.h"
#include "readwin.h"
#include "compose.h"

#define WAITING -1
#define OK      1 
#define ABORT   0

#ifndef S_IRWXU
#define S_IRWXU 0000700
#endif

FILE* displayMultiMedia();

static void insert_reply_info(/* ENVELOPE*, ENVELOPE* */);

static void PositionShell();
static void MoveShell();

extern XtAppContext app_context;
extern AppResources res;

extern char *typeAcronyms[];

/*
 *  Finds the Shell ancestor of a widget.
 *    -> it assumes that there *is* a shell somewhere
 *      up the widget tree.
 */
Widget getShell(w)
     Widget w;
{
    register Widget temp = w;

    while( !XtIsShell( temp ) )
	temp = XtParent(temp);

    return( temp );
}

/*
 *  Turns a NodeList into an IMAP format sequence
 *  of message numbers (e.g., "1,2:47,49,123:124,207")
 *
 */
char* updateSequence( nodelist )
     NodeList* nodelist;
{
    char *returnval;
    
    MessageNode* *tmp = nodelist->nodes;
    int         number = nodelist->length;
    
    int i = 0;
    int begin = 0;
    int end = 0;
    int msgno = tmp[i++]->msgno;
    
    char str[TMPLEN];
    char *sequence = str;
    
    *sequence = '\0';		
    
    sprintf (sequence,"%d", msgno);
    sequence += strlen(sequence);	  
    end = begin = msgno;	
    
    for ( ; i < number; i++ ) 
    {
	msgno = tmp[i]->msgno;
    	if(msgno != end + 1)
	{
	    if (end != begin) 
		sprintf (sequence,":%d,%d",end,msgno);
	    else 
		sprintf (sequence,",%d",msgno);

	    begin = msgno;	
	}
	end = msgno;
	sequence += strlen(sequence);	
    }

    if (end != begin)
	sprintf (sequence,":%d",end);
    
    returnval = cpystr( str );
    return (returnval);			
}

/***********************************************************

  String input functions


************************************************************/

static char **_dg_dest;

static void dialogOk(w, s, e)
     Widget w;
     char **s;
     XEvent *e;
{
    char *c = NULL;
    char *t1, *t2;
    
    c = XawDialogGetValueString(XtParent(w));
    t1 = c;
    
    *s = (char*) XtMalloc ( strlen (c) + 1 );
    t2 = *s;
    
    while ( *(t2)++ = *t1++ )
	; /* VOID LOOP */
}

static void dialogQuit(w, s, e)
     Widget w;
     char **s;
     XEvent *e;
{
    *s = "";
}

static void dialogOkAct ( w, e )
     Widget w;
     XEvent* e;
{
    dialogOk( w, _dg_dest, e );
}

static void dialogQuitAct ( w, e )
     Widget w;
     XEvent* e;
{
    dialogQuit( w, _dg_dest, e );
}

static void dialogDefault( w, e )
     Widget w;
     XEvent e;
{
    /* Empty Function */;
}

static XtActionsRec dialogActs[] =
{{ "dialogOkAct",   dialogOkAct },
 { "dialogQuitAct" ,dialogQuitAct },
 { "dialogDefault", dialogDefault }};

void inputString(parent, prompt, destination)
     Widget parent;
     char *prompt;
     char **destination;
{  
    Arg warg[4];
    int n = 0;
    
    Widget dialogShell, dialogBox;
    
    XFontStruct *font;
    XEvent       event;
    int          width = 0;
    char        *initialstring = "";

    _dg_dest = destination;

    XtAppAddActions( app_context, dialogActs, XtNumber(dialogActs));
  
    XtSetArg(warg[n], XtNtransientFor, parent); n++;
    dialogShell = XtCreatePopupShell("dialog_shell", 
				     transientShellWidgetClass,
				     parent, warg, n); n = 0;
    
    if (*destination != NULL)
    {
	initialstring = *destination;
	*destination = NULL;
    }
    
    XtSetArg(warg[n], XtNlabel, prompt);n++;
    XtSetArg(warg[n], XtNvalue, initialstring); n++;
    dialogBox = XtCreateManagedWidget( "dialog", dialogWidgetClass,
				      dialogShell, warg, n); n = 0;
    
    XawDialogAddButton( dialogBox, "OK",     dialogOk, destination );
    XawDialogAddButton( dialogBox, "Cancel", dialogQuit, destination );
    
    XtSetArg(warg[n], XtNfont, &font); n++ ;
    XtGetValues((XtNameToWidget(dialogBox, "value")), warg, n); n = 0;
    
    if ( strlen(initialstring) > 40 )
	width = strlen(initialstring) + 5;
    else
	width = 30;
    
    width *= (font->max_bounds.width);
    
    XtSetArg(warg[n], XtNwidth,  width); n++;
    XtSetValues(dialogBox, warg, n); n = 0;

    XtRealizeWidget(dialogShell);
    PositionPopupShell(dialogShell, NULL, dialogShell);

    XtPopup( dialogShell, XtGrabNone );
    
    while ( *destination == NULL )
    {
	XtAppNextEvent(app_context, &event);
	XtDispatchEvent( &event );
    }
    
    XtDestroyWidget( dialogShell );
}

/************************************************

  "do" Functions

  these functions actually do the message manipulation set
  up in the individual source files -- for instance, 
  both the Mailbox.c and ReadWindow.c will call 
  doForward, doCopy and doMove, but they will each set up the call
  differently on their end.

************************************************/

void doPrint(ms, nodelist)
     MailBox*   ms;
     NodeList*  nodelist;
{
    int i = 0;
    int number = -1;
    FILE *fout = NULL;
    char *commandstring1 = NULL;
    char *printCommand = NULL;
    char *filename = tmpnam((char*) NULL);
    
    if ((fout = fopen( filename, "a+")) != 0)
    {
	chmod( filename, S_IRWXU );

	for ( i = 0; i < nodelist->length; i++ )
	{
	    number = nodelist->nodes[i]->msgno;
	    fprintf( fout, "%s", mail_fetchheader( ms->mailbox, number));
	    fprintf( fout, "%c%c", '\n', '\n' );
	    fprintf( fout, "%s", mail_fetchtext( ms->mailbox, number));
	    fprintf( fout, "\f");
	}
	fclose( fout);
	
	printCommand = cpystr(res.printCommand);

	commandstring1 = XtMalloc( strlen(printCommand) 
				  +strlen(filename)
				  +2);

	sprintf(commandstring1,"%s %s", printCommand, filename);
	system( commandstring1);
    }
    XtFree(printCommand);
    unlink(filename);
    XtFree(commandstring1);
}

void doCopy(ms, nodelist)
     MailBox*     ms;
     NodeList*    nodelist;
{
    char *sequence = NULL;
    char *save = NULL;

    fetch_unfetched(ms, nodelist);
    
    if (ms->lastcopy)
    {
	save = (char*) XtMalloc(strlen( ms->lastcopy) + 1);
	strcpy(save, ms->lastcopy);
    }
    
    if ( nodelist->length > 0 ) 
	sequence = updateSequence( nodelist );
    
    inputString(ms->mailWindow, "Copy to: ", &save );
    
    if ( sequence != NULL && *save != '\0' )
    {
	if ( !(ms->mailbox->lock))
	{
	    mail_copy( ms->mailbox,sequence, save );
	    updateHeaders( ms, nodelist );
	    ms->lastcopy = save;
	}
    }
}

void doMove(ms, nodelist)
     MailBox*    ms;
     NodeList*   nodelist;
{
    char *sequence = NULL;
    char *save = NULL;

    fetch_unfetched(ms, nodelist);
    
    if (ms->lastcopy != NULL)
    {
	save = (char*) XtMalloc(strlen( ms->lastcopy) + 1);
	strcpy(save, ms->lastcopy);
    }
    
    if ( nodelist->length > 0 ) 
	sequence = updateSequence( nodelist );
    
    inputString(ms->mailWindow, "Move to: ", &save );
    
    if (    sequence != NULL 
	&& *save != '\0'
	&& !ms->mailbox->lock)
    {
	mail_move( ms->mailbox,sequence, save);
	updateHeaders( ms, nodelist );
	ms->lastcopy = save;
    }
}

void addComposeWindowToMailbox(rwin, ms, env, body, status)
     READWINDOW * rwin;
     MailBox*    ms;
     ENVELOPE   *env;
     BODY       *body;
     unsigned long status;
{
    ComposeWindow *temp = ComposeMessage( rwin, ms, env, body, status );
    if(temp)
    {
	temp->next = ms->compList;
	ms->compList = temp;
    }
}

void doSetFlag(ms, nodelist, flag)
     MailBox*   ms;
     NodeList*  nodelist;
     char     *flag;
{ 
    char *sequence = NULL;

    if(nodelist->length > 0)
	sequence = updateSequence( nodelist );
    
    if ( sequence != NULL
	&& !ms->mailbox->lock)
    {
	mail_setflag( ms->mailbox, sequence, flag );
	updateHeaders( ms, nodelist );
    }
}

void doClearFlag(ms, nodelist, flag)
     MailBox*   ms;
     NodeList*  nodelist;
     char     *flag;
{ 
    char *sequence = NULL;

    if( nodelist->length > 0 ) 
	sequence = updateSequence( nodelist );
    
    if ( sequence != NULL
	&& !ms->mailbox->lock)
    {
	mail_clearflag( ms->mailbox, sequence, flag );
	updateHeaders( ms, nodelist );
    }
}

/*
  most of the information here is set up just as though there
  were only one message (the first one) being forwarded.  The
  message text, however, is treated very differently when there
  are multiple messages...
*/
void doForward(ms, nodelist)
     MailBox*   ms;
     NodeList*  nodelist;
{
    MessageNode* *nodes = nodelist->nodes;
    int i = 0;
    
    char *scratch;
    char temp[TMPLEN];
    char frm[TMPLEN];
    ENVELOPE *new = NULL;
    ENVELOPE *old = NULL;
    BODY     *newbody = NULL;
    char *txtemp = NULL;
    char *hdrtemp = NULL;
    char *ftmp = NULL;
    int length = 0;
    int divlength = 0;

    memset(frm, '\0', TMPLEN);

    divlength = strlen(FWDMULTIPLEFRM);
    if (nodelist->length > 1)
	length += divlength;

    fetch_unfetched(ms, nodelist);

    while ( i < nodelist->length )
    {
	if (new == NULL)
	{
	    old = nodes[i]->env; 
	    new = set_up_envelope(nodes[i], ms->host);
	    nodes[i]->status &= ~INRESPONSE;
	    nodes[i]->status |= FORWARDMSG;
	    ms->replyto = nodes[i];
	}
	
	if (nodelist->length > 1)
	{
	    nodes[i]->shortheader = shortheader(ms->mailbox, nodes[i], i+1);
	    length += (strlen(nodes[i]->shortheader) + 4);
	}
	
	nodes[i]->text = 
	    gettext_for_display(mail_fetchtext(ms->mailbox, 
					       nodes[i]->elt->msgno));
	length += (strlen(nodes[i]->text) + 4);
	
	nodes[i]->header = 
	    gettext_for_display(mail_fetchheader(ms->mailbox, 
						 nodes[i]->elt->msgno));
	length += (strlen(nodes[i]->header) + 4);
	length += divlength;
	
	i += 1;
    }
    
    hdrtemp = XtMalloc(divlength + 4);
    txtemp = XtMalloc(length);
    
    if (nodelist->length > 1)
    {
	strcpy(txtemp, GENERICDIVIDER);
	for (i = 0; i < nodelist->length; ++i)
	{
	    strcat(txtemp, "\n");
	    strcat(txtemp, nodes[i]->shortheader);
	    XtFree((char*)nodes[i]->shortheader);
	}
    }
    else
    {
	strcpy(txtemp, FWDHDR);
    }
    for (i = 0; i < nodelist->length; ++i)
    {
	strcat(txtemp, "\n\n");
	if (nodelist->length > 1)
	{
	    sprintf(hdrtemp,FWDMULTIPLEFRM, i+1);
	    strcat(txtemp, hdrtemp);
	}
	strcat(txtemp, nodes[i]->header);
	strcat(txtemp,"\n");
	strcat(txtemp, nodes[i]->text);
	hdrtemp[0] = '\0';
    }


    newbody = mail_newbody();

    newbody->contents.text = (unsigned char*) XtMalloc (1+strlen(txtemp));
    strcpy(newbody->contents.text,txtemp);

/*    new->text = cpystr(txtemp); */

    XtFree((char*)txtemp);
    XtFree((char*)hdrtemp);

    scratch = address_to_text(nodes[0]->env->from, 999);
    strncpy(frm, scratch, FROMLEN);
    ftmp = frm + (strlen(frm) - 1);
    while(isspace(*ftmp))
	*ftmp-- = '\0';
    sprintf(temp, "[%s: %s]", frm, (old->subject?
				    old->subject: "(forwarded message)"));
    new->subject = cpystr(temp);
	
    /* address for replies... */
    
    /* 
      if it's specified as a resource, you have
      to have both name and host...
    */
    if (res.mail_name
	&& res.mail_host)
    {
	new->reply_to = mail_newaddr();
	new->reply_to->mailbox = cpystr(res.mail_name);
	new->reply_to->host = cpystr(res.mail_host);
    }
    else
    {
	copy_address(new->from);
    }

    /* error responses */
    new->return_path = copy_address(new->from);
    
    /* setting these to NULL untill I find out more 
       about the conventions... */
    new->sender = NULL;
    
    addComposeWindowToMailbox( NULL, ms, new, newbody, FORWARDMSG );
    ms->replyto = NULL;
    XtFree(newbody->contents.text);
}

void doReply( rwin, ms, nodelist)
     READWINDOW * rwin;
     MailBox*   ms;
     NodeList*  nodelist;
{
    ADDRESS  *tmp = NULL;
    ENVELOPE *new = NULL;
    ENVELOPE *old = NULL;
    MessageNode* node = nodelist->nodes[0];
    char *temp = NULL;

    old = nodelist->nodes[0]->env; 
    new = set_up_envelope(nodelist->nodes[0], ms->host);
    ms->replyto = nodelist->nodes[0];
    
    fetch_unfetched(ms, nodelist);
    insert_reply_info(new, old);

    if( !node->text )
    {
	temp = mail_fetchtext(ms->mailbox, node->msgno);
	node->text = gettext_for_display(temp);
    }

    node->insert = gettext_for_insert(rwin,node->text, res.indentPrefix );

    /* set to return addess of old or by user */
    new->to = copy_address(old->reply_to);

    if (nodelist->nodes[0]->status & REPLYALL)
    {
	add_address_to_list(copy_address(old->cc), &(new->cc));
	add_address_to_list(old->to, &(new->cc));

	if( new->cc )
	{
	    remove_addr_from_list(new->from, &(new->cc));
	    remove_addr_from_list(new->to, &(new->cc));
	    remove_addr_from_list(new->bcc, &(new->cc));
	}
	
	if(*res.default_cc)
	{
	    tmp = rfc822_parse_address(res.default_cc, ms->host);
	    add_address_to_list(tmp, &(new->cc));
	}
    }

    addComposeWindowToMailbox(rwin, ms, new, NULL,nodelist->nodes[0]->status);
    ms->replyto = NULL;
}

void doRemail(ms, nodelist)
     MailBox*   ms;
     NodeList*  nodelist;
{
    MessageNode* node = nodelist->nodes[0];

    MAILSTREAM *stream = ms->mailbox;
    ENVELOPE *new = NULL;
    int msgno = BAD_MESSAGE;

    fetch_unfetched(ms, nodelist);

    ms->replyto = node;
    
    node->status &= ~INRESPONSE;
    node->status |= REMAILMSG;
    
    msgno = node->elt->msgno;

    new = set_up_envelope(node, ms->host);
    new->remail = cpystr(mail_fetchheader(stream, msgno));

    parse_message(ms,node);
    /* Make sure we step on CRLF 
     *    Bill Yeager 23-Oct-92  */
    if (*node->body->contents.text)
      fixnl(node->body->contents.text);
    /* User cannot edit remailed text. This is here not to be
     * moralistic, but rather to fix a bug */
    addComposeWindowToMailbox( NULL, ms, new, node->body,  
			      REMAILMSG);
    ms->replyto = NULL;
}

void doComposeNew(ms)
     MailBox*  ms;
{
  ENVELOPE *new;
 
  new = set_up_envelope(NULL, ms->host);
  
  if (res.mail_name
      && res.mail_host)
  {
      new->reply_to = mail_newaddr();
      new->reply_to->mailbox = cpystr(res.mail_name);
      new->reply_to->host = cpystr(res.mail_host);
  }
  else
  {
      mail_free_address(&new->reply_to);
      new->reply_to = copy_address(new->from);
  }
  
  /* error responses */
  new->return_path = copy_address(new->from);
  
  addComposeWindowToMailbox( NULL, ms, new, NULL, 0 );
}
 
static void insert_reply_info(new, old)
     ENVELOPE *new;
     ENVELOPE *old;
{
    char temp[TMPLEN];
    char frm[TMPLEN];

    /* keep it personal if possible...*/
    if (!old->from) 
	strcpy (frm,"(unknown)'s");
    else if (old->from->personal) 
	sprintf (frm,"%s's",old->from->personal);
    else 
	sprintf (frm,"%s@%s's",old->from->mailbox,old->from->host);
    
    /*
      ...but also be specific. there should just about always
      be a message id of some sort, but just in case...
    */
    if (old->message_id)
	sprintf(temp,"%s message of %s: %s", frm, old->date, old->message_id);
    else
	sprintf(temp,"%s message of %s", frm, old->date);
    
    new->in_reply_to = cpystr(temp);

    if (old->subject 
	&& (strlen(old->subject) > 0))
    {
	strncpy(temp, old->subject, 3);
	temp[3] = '\0';
	if (areDifferentStrings(temp, "re:"))
	    sprintf(temp, "RE: %s", old->subject);
	else
	    strcpy(temp, old->subject);
    }
    else
    {
	sprintf(temp, "In reply to your message of %s", old->date);
    }
    new->subject = cpystr(temp);
    
    /* 
      address for replies...

      if it's specified as a resource, you have
      to have both name and host...
    */
    if (res.mail_name
	&& res.mail_host)
    {
	new->reply_to = mail_newaddr();
	new->reply_to->mailbox = cpystr(res.mail_name);
	new->reply_to->host = cpystr(res.mail_host);
    }
    else
    {
	copy_address(new->from);
    }

    /* error responses */
    new->return_path = copy_address(new->from);
}

char* shortheader(stream, node, num)
     MAILSTREAM   *stream;
     MessageNode* node;
     int         num;
{
    char from[FROMLEN+1];
    char sub[SUBJECTLEN + 1];
    char date[DATELEN];
    char tmpbuf[TMPLEN];
    
    /*  only use "dd-mmm" from date */
    strncpy (date, mail_date(tmpbuf, node->elt), DATELEN-1);
    
    mail_fetchfrom(from, stream, node->msgno, FROMLEN);
    mail_fetchsubject(sub, stream, node->msgno, SUBJECTLEN);
    
    /*  output what we got */
    sprintf (tmpbuf," %d) %s %s %s",num,date,from,sub);

    XtFree(from);
    return( cpystr(tmpbuf) );
}


/* ARGSUSED */
void PositionPopupShell(w, event, shell)
     Widget w;
     XEvent * event;
     Widget shell;
{ 
    PositionShell(w, NULL);

    return;
}  

static void PositionShell(w, location)
     Widget w;
     XPoint * location;
{
    XPoint t_point;

    Arg warg[8];
    int n = 0;

    Dimension wi = 0;
    Dimension he = 0;

    static void MoveShell();

    if (location == NULL) 
    {
	Window junk1, junk2;
	int root_x, root_y, junkX, junkY;
	unsigned int junkM;
	
	location = &t_point;
	if (XQueryPointer(XtDisplay(w), XtWindow(w), &junk1, &junk2, 
			  &root_x, &root_y, &junkX, &junkY, &junkM) == FALSE) {
	    char error_buf[BUFSIZ];
	    sprintf(error_buf, "%s %s", "Position Shell:",
		    "Could not find location of mouse pointer");
	    XtAppWarning(XtWidgetToApplicationContext(w), error_buf);
	    return;
	}
	location->x = (short) root_x;
	location->y = (short) root_y;
    }
    
    /*
     * The width will not be correct unless it is realized.
     */
    
    XtRealizeWidget(w);

    XtSetArg(warg[n], XtNheight, &he); n++;
    XtSetArg(warg[n], XtNwidth, &wi);n++;
    XtGetValues(w, warg, n); n = 0;
 
    location->x -= (Position) wi/2;
    location->y -= (Position) he/2;
    
    MoveShell(w, (Position) location->x, (Position) location->y);
}

static void MoveShell(w, x, y)
     Widget w;
     Position x, y;
{
    Arg warg[8];
    int n = 0;

    Dimension wi = 0;
    Dimension he = 0;
    Dimension bw = 0;

    int width = 0;
    int height = 0;

    XtSetArg(warg[n], XtNborderWidth, &bw); n++;
    XtSetArg(warg[n], XtNheight, &he); n++;
    XtSetArg(warg[n], XtNwidth, &wi);n++;
    XtGetValues(w, warg, n); n = 0;
 
    width = wi + 2 * bw;
    height = he + 2 * bw;
    
    if (x < 0) 
    {
	x = 0;
    }
    else 
    {
	int scr_width = WidthOfScreen(XtScreen(w));
	if (x + width > scr_width)
	    x = scr_width - width;
    }
    
    if (y < 0)
    {
	y = 0;
    }
    else 
    {
	int scr_height = HeightOfScreen(XtScreen(w));
	if (y + height > scr_height)
	    y = scr_height - height;
    }

    XtSetArg(warg[n], XtNx, x); n++;
    XtSetArg(warg[n], XtNy, y); n++;
    XtSetValues(w, warg, n); n = 0;
}





/*
 * resolve path takes two arguments:
 *    path - a colon separated list of directories to search
 *    name - the name of the file
 * 
 * it returns either the full name of the file
 * or NULL if the file is not found
 *
 */
char *resolvepath( path, name )
     char *path;
     char *name;
{
    char part[1024];
    char fullname[1024];
    char *ptemp = part;
    char *temp = path;

    if(!path || !name)
	return(NULL);
	
    *ptemp =  '\0';
    while( *temp )
    {
	while( *temp && *temp != ':' )
	    *ptemp++ = *temp++;
	temp++;
	*ptemp =  '\0';
	if(part[0])
	{
	    strcpy(fullname, part);
	    strcat(fullname,"/");
	    strcat(fullname, name);
	    if(!access(fullname, F_OK))
		return(cpystr(fullname));
	}

	ptemp = part;
	*ptemp =  '\0';
    }
    
    return(NULL);

}

FILE *displayMultiMedia( body )
     BODY *body;
{
    char        temp[TMPLEN];
    unsigned long leng = 0;
    HandlerPtr   hptr;
    char        *vcom = NULL;
    FILE        *outfile = NULL;
    FILE        *ftemp = NULL;
    void        *rawdata = NULL;
    char        *filename;

    if( body->type == TYPEMULTIPART )
	return(NULL);

    hptr = getHandler(body);

    if(hptr)
    {
	vcom = cpystr(hptr->view);
    }
    else
    {
	mm_log("Unknown message format", ERROR);
	return(NULL);
    }

    /* If it's encoded in BASE64, decode it... */
    if( body->encoding == ENCBASE64 )
    {
	rawdata = rfc822_base64(body->contents.text, body->size.bytes, &leng);
    }
    else 
    {
	if(body->type == TYPETEXT)
	{
	    rawdata = body->contents.text;
	}
	else
	{
	    mm_log("Bad encoding\n", ERROR);
	    close(outfile);
	    return(NULL);
	}
    }
	
    if(strstr(vcom, "%s"))
    {
	filename = tmpnam((char*) NULL);
	if ((ftemp = fopen( filename, "w")) != 0)
	{
	    int numberWritten = 0;
	    chmod( filename, S_IRWXU );
	    if(body->type != TYPETEXT)
	    {
		do
		{
		    numberWritten += fwrite(rawdata, 1, leng, ftemp);
		} while (numberWritten < leng);
	    }
	    else
	    {
		do
		{
		    numberWritten += fwrite(rawdata, 1, strlen(rawdata), ftemp);
		} while (numberWritten < leng);
	    }
	    fclose(ftemp);
	    
	    sprintf(temp, vcom, filename);
	}
	outfile = popen(temp, "w");
    }
    else
    {
	outfile = popen(vcom, "w");

	if(outfile == NULL)
	    mm_log("Could not open pipe for display", ERROR);
	else if (rawdata)
	    write(outfile, rawdata, leng);
    }

    return(outfile);
}     

Handler *getHandler(body)
     BODY *body;
{
    HandlerArray h;
    HandlerPtr   hptr;
    Handler     *defaultHandler = NULL;

    /* Multiparts are handled at the application level */
    if( body->type == TYPEMULTIPART )
	return(NULL);

    /* Get handlers parsed from mailcap file */
    if(h = getHandlers())
    {
	/* get the head of the list of handlers appropriate for that body type... */
	if(hptr = h[body->type])
	{
	    while(hptr)
	    {
		if(!hptr->subType && !body->subtype)
		{
		    return(hptr);
		}
		else 
		{
		    if(hptr->subType && body->subtype)
		    {
			if(!strcmp(hptr->subType,"*"))
			    defaultHandler = hptr;
			
			if(!strcasecmp(hptr->subType, body->subtype))
			    return(hptr);
		    }
		}
		
		hptr = hptr->next;
	    }
	}
    }	
    return(defaultHandler);
}

/*
   Reads a file into memory, returns a BinaryBuffer structure 

   The basic structure was taken from Pine...
*/
BinaryBuffer *readFileToMem(ifp)
     FILE *ifp;
{
    long          totalbytes = 0;
    long          nbytes = 0;
    struct stat   sb;
    BinaryBuffer *bb;
    char         *temp;

    fstat(fileno(ifp), &sb);

    bb = (BinaryBuffer*) malloc (sizeof(BinaryBuffer));
    bb->bufferContents = malloc(sb.st_size);
    bb->bufferSize = sb.st_size;
    temp = bb->bufferContents;

    do
    {
	nbytes = fread(temp, 1, (sb.st_size-totalbytes), ifp);
	totalbytes += nbytes;
	temp += nbytes;
    } while(nbytes && totalbytes < sb.st_size);

    if(totalbytes != sb.st_size) 
    {
	free(bb->bufferContents);
	free(bb);
        bb = NULL;
    }

    return(bb);
}

#define BROWSERLINE 81
XbListSelection* makeMultiPartHeaderList(part)
     PART * part;
{
    XbListSelection *list = NULL;
    int i = 0;
    int residue;

    char browserline[2*BROWSERLINE];	/* L'excess est tres important */

    PART* temp = part;
    while(temp)
    {
	i++;
	temp = temp->next;
    }
    
    list = ximap_newlistmap(i);

    i = 0;
    temp =  part;
    while(temp)
    {
#define SUBTYPEFIELD 10
#define SUBJECTLINE  62
        char ctmp[2*SUBJECTLINE], *cp;
	int ctmp_len;
	/*
	 * make a line:
	 *  msg# TYPE/SUBTYPE <content-description> */
	sprintf(browserline, " %2d %s/", i+1, typeAcronyms[temp->body.type]);
	cp = (char *)((temp->body.subtype) ? temp->body.subtype :
		      "NO-SUBTYPE");
	sprintf(ctmp,"%s", cp);
	if ((ctmp_len = strlen(ctmp)) < SUBTYPEFIELD) {
	  /* Faites "SUBTYPE " with max length of SUBTYPEFIELD  bytes */
	  cp = ctmp + ctmp_len;
	  while (ctmp_len++ < SUBTYPEFIELD)
	    *cp++ = ' ';
	  *cp++ = ' ';
	  *cp = '\0';
	} else {			/* Set the limits */
	  ctmp[SUBTYPEFIELD] = ' ';
	  ctmp[SUBTYPEFIELD+1] = '\0';
	}
	strcat(browserline, ctmp);
	residue = BROWSERLINE - strlen(browserline);
	if (residue > 0 && temp->body.description) {
	    strncat(browserline, temp->body.description, residue);
	    strcat(browserline, " ");	/* Elle est tres jolie maintenant */
	  }
	list->entries[i].entry = cpystr(browserline);
	list->entries[i].index = i+1;
	
	i++;
	temp = temp->next;
    }

    return(list);
}

void parseMailboxName(name, host, mailbox)
     char *name;
     char **host;
     char **mailbox;
{
    char *start = name;
    char *end = NULL;
    char save = '\0';

    if(*start == '{')
    {
	start++;
	end = strchr(start,'}');
	save = *end;
	*end = '\0';
	*host = cpystr(start);
	*mailbox = cpystr(end+1);
	*end = save;
    }
    else
    {
	*host = NULL;
	*mailbox = cpystr(name);
    }
}

void
getQualifiedHostname(hn)
     char *hn;
{
  int hlen;

  gethostname(hn, HNAMELEN);
  /* Just in case it isn't NULL terminated */
  hn[HNAMELEN - 1] = '\0';
}

extern FILE *fichier;
extern int imp_log;

#ifdef XIMAPDEBUG

void
ximap_log(str)
     char *str;
{
  if (!imp_log)
    return;
  if (fichier) {
    fwrite(str, 1, strlen(str), fichier);
    fflush(fichier);
    fsync(fileno(fichier));
  }
}

void
always_log(str)
     char *str;
{
  if (fichier) {
    fwrite(str, 1, strlen(str), fichier);
    fflush(fichier);
    fsync(fileno(fichier));
  }
}

#else
void
ximap_log(str)
     char *str;
{
  return;
}

void
always_log(str)
     char *str;
{
  return;
}

#endif
