
/*
 * lock.c -- locking routines
 *
 * $Id$
 */

#include <h/mh.h>
#include <h/signals.h>
#include <mts.h>

#ifdef HAVE_ERRNO_H
# include <errno.h>
#endif

#ifdef MMDFONLY
# include <mmdfonly.h>
# include <mts.h>
# include <lockonly.h>
#endif /* MMDFONLY */

#ifdef HAVE_FCNTL_H
# include <fcntl.h>
#else
# include <sys/file.h>
#endif

#if defined(LOCKF_LOCKING) || defined(FLOCK_LOCKING)
# include <sys/file.h>
#endif

#include <signal.h>

extern int errno;

#ifdef LOCKDIR
char *lockdir = LOCKDIR;
#endif

/* Are we using any kernel locking? */
#if defined (FLOCK_LOCKING) || defined(LOCKF_LOCKING) || defined(FCNTL_LOCKING)
# define KERNEL_LOCKING
#endif

#ifdef DOT_LOCKING

/* struct for getting name of lock file to create */
struct lockinfo {
    char curlock[BUFSIZ];
    char tmplock[BUFSIZ];
};

/*
 * amount of time to wait before
 * updating ctime of lock file
 */
#define	NSECS 20

/*
 * how old does a lock file need to be
 * before we remove it
 */
#define RSECS 60

/* struct for recording and updating locks */
struct lock {
    int	l_fd;
    char *l_lock;
    struct lock *l_next;
};

/* top of list containing all open locks */
static struct lock *l_top = NULL;
#endif /* DOT_LOCKING */

/*
 * static prototypes
 */
#ifdef KERNEL_LOCKING
static int lkopen_kernel (char *, int);
#endif

#ifdef DOT_LOCKING
static int lkopen_dot (char *, int);
static int lockit (struct lockinfo *);
static void lockname (char *, struct lockinfo *, int);
static void timerON (char *, int);
static void timerOFF (int);
static RETSIGTYPE alrmser (int);
#endif


/*
 * Base routine to open and lock a file,
 * and return a file descriptor.
 */

int
lkopen (char *file, int access)
{
#ifdef KERNEL_LOCKING
    return lkopen_kernel(file, access);
#endif

#ifdef DOT_LOCKING
    return lkopen_dot(file, access);
#endif
}


/*
 * Base routine to close and unlock a file,
 * given a file descriptor.
 */

int
lkclose (int fd, char *file)
{
#ifdef FCNTL_LOCKING
    struct flock buf;
#endif

#ifdef DOT_LOCKING
    struct lockinfo lkinfo;
#endif

    if (fd == NOTOK)
	return OK;

#ifdef FCNTL_LOCKING
    buf.l_type   = F_UNLCK;
    buf.l_whence = SEEK_SET;
    buf.l_start  = 0;
    buf.l_len    = 0;
    fcntl(fd, F_SETLK, &buf);
#endif

#ifdef FLOCK_LOCKING
    flock (fd, LOCK_UN);
#endif

#ifdef LOCKF_LOCKING
    /* make sure we unlock the whole thing */
    lseek (fd, (off_t) 0, SEEK_SET);
    lockf (fd, F_ULOCK, 0L);
#endif	

#ifdef DOT_LOCKING
    lockname (file, &lkinfo, 0);
    unlink (lkinfo.curlock);
    timerOFF (fd);
#endif

    return (close (fd));
}


/*
 * Base routine to open and lock a file,
 * and return a FILE pointer
 */

FILE *
lkfopen (char *file, char *mode)
{
    int fd;
    FILE *fp;

    if ((fd = lkopen (file, strcmp (mode, "r") ? 2 : 0)) == NOTOK)
	return NULL;

    if ((fp = fdopen (fd, mode)) == NULL) {
	close (fd);
	return NULL;
    }

    return fp;
}


/*
 * Base routine to close and unlock a file,
 * given a FILE pointer
 */

int
lkfclose (FILE *fp, char *file)
{
#ifdef FCNTL_LOCKING
    struct flock buf;
#endif

#ifdef DOT_LOCKING
    struct lockinfo lkinfo;
#endif

    if (fp == NULL)
	return OK;

#ifdef FCNTL_LOCKING
    buf.l_type   = F_UNLCK;
    buf.l_whence = SEEK_SET;
    buf.l_start  = 0;
    buf.l_len    = 0;
    fcntl(fileno(fp), F_SETLK, &buf);
#endif

#ifdef FLOCK_LOCKING
    flock (fileno(fp), LOCK_UN);
#endif

#ifdef LOCKF_LOCKING
    /* make sure we unlock the whole thing */
    fseek (fp, 0L, 0);
    lockf (fileno(fp), F_ULOCK, 0L);
#endif

#ifdef DOT_LOCKING
    lockname (file, &lkinfo, 0);
    unlink (lkinfo.curlock);
    timerOFF (fileno(fp));
#endif

    return (fclose (fp));
}


#ifdef KERNEL_LOCKING

/*
 * open and lock a file, using kernel locking
 */

static int
lkopen_kernel (char *file, int access)
{
    int fd, i, j;

# ifdef FCNTL_LOCKING
    struct flock buf;
# endif /* FCNTL_LOCKING */

    for (i = 0; i < 5; i++) {

# if defined(LOCKF_LOCKING) || defined(FCNTL_LOCKING)
	j = access;
	/* make sure we open at the beginning */
	access &= ~O_APPEND;
	if ((access & 03) == O_RDONLY) {
	    /*
	     * We MUST have write permission or
	     * lockf/fcntl() won't work
	     */
	    access &= ~O_RDONLY;
	    access |= O_RDWR;
	}
# endif /* LOCKF_LOCKING || FCNTL_LOCKING */

	if ((fd = open (file, access | O_NDELAY)) == NOTOK)
	    return NOTOK;

# ifdef FCNTL_LOCKING
	buf.l_type   = F_WRLCK;
	buf.l_whence = SEEK_SET;
	buf.l_start  = 0;
	buf.l_len    = 0;
	if (fcntl (fd, F_SETLK, &buf) != NOTOK)
	    return fd;
# endif

# ifdef FLOCK_LOCKING
	if (flock (fd, LOCK_EX | LOCK_NB) != NOTOK)
	    return fd;
# endif

# ifdef LOCKF_LOCKING
	if (lockf (fd, F_TLOCK, 0L) != NOTOK) {
	    /* see if we should be at the end */
	    if (j & O_APPEND)
		lseek (fd, (off_t) 0, SEEK_END);
	    return fd;
	}
# endif

	j = errno;
	close (fd);
	sleep (5);
    }

    close (fd);
    errno = j;
    return NOTOK;
}

#endif /* KERNEL_LOCKING */


#ifdef DOT_LOCKING

/*
 * open and lock a file, using dot locking
 */

static int
lkopen_dot (char *file, int access)
{
    int i, fd;
    time_t curtime;
    struct lockinfo lkinfo;
    struct stat st;

    if ((fd = open (file, access)) == NOTOK)
	return NOTOK;
    lockname (file, &lkinfo, 1);

    for (i = 0;;) {
	if (lockit (&lkinfo) == OK) {
	    timerON (lkinfo.curlock, fd);
	    return fd;
	} else {
	    if (stat (lkinfo.curlock, &st) == NOTOK) {
		if (i++ > 5)
		    return NOTOK;
		sleep (5);
	    } else {
		i = 0;
		time (&curtime);
		if (curtime < st.st_ctime + RSECS)
		    sleep (5);
		else
		    unlink (lkinfo.curlock);
	    }
	}
    }
}

/*
 * Routine that actually tries to create
 * the lock file.
 */

static int
lockit (struct lockinfo *li)
{
    int fd;
    char *curlock, *tmplock;

#if 0
    char buffer[128];
#endif

    curlock = li->curlock;
    tmplock = li->tmplock;
    if ((fd = creat(tmplock, 0600)) == NOTOK)
	return NOTOK;

#if 0
    /* write our process id into lock file */
    sprintf (buffer, "nmh lock: pid %d\n", (int) getpid());
    write(fd, buffer, strlen(buffer) + 1);
#endif

    close (fd);

    /* now try to create the real lock file */
    fd = link(tmplock, curlock);
    unlink(tmplock);

    return (fd != NOTOK ? OK : NOTOK);
}

/*
 * Get name of lock file, and temporary lock file
 */

static void
lockname (char *file, struct lockinfo *li, int isnewlock)
{
    char *bp, *cp;

#if 0
    struct stat st;
#endif

    bp = li->curlock;
    if ((cp = strrchr(file, '/')) == NULL || *++cp == 0)
	cp = file;

#ifdef LOCKDIR
    sprintf (bp, "%s/", lockdir);
    bp += strlen (bp);
#else
    if (cp != file) {
	sprintf (bp, "%.*s", cp - file, file);
	bp += strlen (bp);
    }
#endif

#if 0
    /*
     * mmdf style dot locking.  Currently not supported.
     * If we start supporting mmdf style dot locking,
     * we will need to change the return value of lockname
     */
    if (stat (file, &st) == NOTOK)
	return NOTOK;

    sprintf (bp, "LCK%05d.%05d", st.st_dev, st.st_ino);
#endif

    sprintf (bp, "%s.lock", cp);

    /*
     * If this is for a new lock, create a name for
     * the temporary lock file for lockit()
     */
    if (isnewlock) {
	if ((cp = strrchr(li->curlock, '/')) == NULL || *++cp == 0)
	    strcpy (li->tmplock, ",LCK.XXXXXX");
	else
	    sprintf (li->tmplock, "%.*s,LCK.XXXXXX",
		     cp - li->curlock, li->curlock);
	unlink (mktemp (li->tmplock));
    }
}


/*
 * Add new lockfile to the list of open
 * lockfiles.
 */

static void
timerON (char *curlock, int fd)
{
    struct lock *lp;
    size_t len;

    if (!(lp = (struct lock *) malloc (sizeof(*lp))))
	return;

    len = strlen(curlock) + 1;
    lp->l_fd = fd;
    if (!(lp->l_lock = malloc (len))) {
	free ((char *) lp);
	return;
    }
    memcpy (lp->l_lock, curlock, len);
    lp->l_next = l_top;

    if (!l_top) {
	/* perhaps SIGT{STP,TIN,TOU} ? */
	SIGNAL (SIGALRM, alrmser);
	alarm (NSECS);
    }

    l_top = lp;
}


/*
 * Search through the list of lockfiles for the
 * current lockfile, and remove it from the list.
 */

static void
timerOFF (int fd)
{
    struct lock *pp, *lp;

    alarm(0);

    if (l_top) {
	for (pp = lp = l_top; lp; pp = lp, lp = lp->l_next) {
	    if (lp->l_fd == fd)
		break;
	}
	if (lp) {
	    if (lp == l_top)
		l_top = lp->l_next;
	    else
		pp->l_next = lp->l_next;

	    free (lp->l_lock);
	    free (lp);
	}
    }

    /* if there are locks left, restart timer */
    if (l_top)
	alarm (NSECS);
}


/*
 * If timer goes off, we update the ctime of all open
 * lockfiles, so another command doesn't remove them.
 */

static RETSIGTYPE
alrmser (int sig)
{
    int j;
    char *lockfile;
    struct lock *lp;

#ifndef	RELIABLE_SIGNALS
    SIGNAL (SIGALRM, alrmser);
#endif

    /* update the ctime of all the lock files */
    for (lp = l_top; lp; lp = lp->l_next) {
	lockfile = lp->l_lock;
	if (*lockfile && (j = creat (lockfile, 0600)) != NOTOK)
	    close (j);
    }

    /* restart the alarm */
    alarm (NSECS);
}

#endif /* DOT_LOCKING */
