/*
 *	Copyright 1989 by Rayan S. Zachariassen, all rights reserved.
 *	This will be free software, but only when it is finished.
 */

/*
 * Trap script and signal management, and the builtin "trap" command.
 */

#include <stdio.h>
#include "hostenv.h"
#include <ctype.h>
#include <sys/stat.h>
#include <signal.h>
#include "flags.h"
#include "malloc.h"
#include "listutils.h"
#include "io.h"			/* redefines stdio routines */
#include "shconfig.h"

#ifndef SIGCHLD
#define SIGCHLD SIGCLD
#endif  /* SIGCHLD */

/*
 * The script to execute for a particular trap is stored as a string in
 * malloc()'ed storage, with a pointer to it in the traps[] array.
 */

char *traps[NSIG];

/*
 * The effect of a signal is to increment a count of seen but unprocessed
 * (as in the trap script hasn't been run) signals in the spring[] array,
 * one counter per signal.  They had better start out as 0.
 */

STATIC int spring[NSIG];	/* pending trap counts */

/*
 * As a cheap way of testing if there are pending unprocessed signals, the
 * sprung variable is used as a flag to that effect.  It is a cheap test
 * elsewhere in the shell (in interpreter loop and input function).
 */

int sprung;			/* there are pending traps */

/*
 * In order to interrupt builtin function execution (as opposed to builtin
 * functions), set a flag whenever we see an interrupt that doesn't have
 * a trap handler.  This flag should be tested in the interpreter loop, and
 * anywhere else the shell might be spending a lot of time (e.g. the filename
 * expansion routines).
 */

int interrupted;		/* XX: we saw an interrupt */

/*
 * To maintain sh semantics, we need to know what the original signal handler
 * values are.  These are retrieved once at startup time (from main) and
 * stored in the orig_handler[] array.
 */

SIGNAL_TYPE (*orig_handler[NSIG])();

/*
 * Indeed, that is what the trapsnap() function does.
 */

void
trapsnap()
{
	int i;

	for (i = 1; i < NSIG; ++i)
		orig_handler[i] = signal(i, SIG_DFL);
	/* is there a vulnerability here due to SIG_DFL instead of SIG_IGN ? */
	for (i = 1; i < NSIG; ++i)
		if (orig_handler[i] != SIG_DFL && i != SIGCHLD)
			(void) signal(i, orig_handler[i]);
}

/*
 * This is the generic signal handler that is set whenever a trap is laid.
 */

void
trap_handler(sig)
	int sig;
{
	if (sig > 0 && sig < NSIG) {
		(void) signal(sig, trap_handler);
		spring[sig] += 1;
		sprung = 1;
	}
	if (sig == SIGINT && traps[sig] == NULL)
		interrupted = 1;
}

/*
 * Evaluate the contents of the buffer and invoke the interpreter.
 * This function really should not be here!
 */

#define CFSUFFIX	".cf"
#define FCSUFFIX	".fc"
#define FCSUBDIR	"fc"

char *
makefc(path)
	char *path;
{
	register char *cp;
	char *buf;
	struct stat stbuf;

	int plen = strlen(path);

	if (plen <= sizeof CFSUFFIX)
		return NULL;
	if (strcmp(path+plen-(sizeof FCSUFFIX - 1), FCSUFFIX) == 0) {
		buf = emalloc(strlen(path)+1);
		strcpy(buf, path);
		return buf;
	}
	if (strcmp(path+plen-(sizeof CFSUFFIX - 1), CFSUFFIX) != 0)
		return NULL;

	buf = emalloc(strlen(path)+sizeof FCSUFFIX + sizeof FCSUBDIR + 2);

	(void) strcpy(buf, path);
	if ((cp = strrchr(buf, '/')) != NULL)
		++cp;
	else
		cp = buf;
	strcpy(cp, FCSUBDIR);
	if (stat(buf, &stbuf) == 0 && (stbuf.st_mode & S_IFDIR)) {
		(void) sprintf(buf + strlen(buf), "/%s", path+(cp-buf));
		(void) strcpy(buf + strlen(buf) - (sizeof CFSUFFIX - 1), FCSUFFIX);
	} else {
		(void) strcpy(buf, path);
		(void) strcpy(buf+plen-(sizeof CFSUFFIX - 1), FCSUFFIX);
	}
	return buf;
}

extern struct osCmd *globalcaller, avcmd;
extern struct codedesc *interpret();

int
eval(script, scriptname, savefile)
	char *script, *scriptname, *savefile;
{
	int status;
	u_char *table, *eotable;
	FILE *fp;
	char *fcfile;
	extern int magic_number;
	extern char *progname;
	extern u_char *SslWalker(), *optimize();

	commandline = s_pushstack(commandline, script);
	table = SslWalker(scriptname, stdout, &eotable);
	status = 0;
	if (table != NULL) {
		if (isset('n')) {
			(void) free((char *)table);
			return 0;
		}
		fcfile = NULL;
		if (isset('O')) {
			table = optimize(0, table, &eotable);
			if (isset('V'))
				table = optimize(1, table, &eotable);
			if (savefile != NULL
			    && (fcfile = makefc(savefile)) != NULL
			    && (fp = fopen(fcfile, "w")) != NULL) {
				fprintf(fp, "#!zsh -l%d\n",
						magic_number);
				std_fwrite(table, sizeof *table,
						  eotable - table, fp);
				if (fclose(fp) == EOF) {
					fprintf(stderr,
						"%s: write to %s failed\n",
						progname, fcfile);
					(void) unlink(fcfile);
				}
			}
		}
		if (fcfile)
			free(fcfile);
		(void) interpret(table, eotable, (u_char *)NULL,
					globalcaller == NULL ? &avcmd
							     : globalcaller,
					&status, (struct codedesc *)NULL);
	}
	return status;
}

int
loadeval(fcfd, path, srcstbufp)
	int fcfd;
	char *path;
	struct stat *srcstbufp;
{
	int status;
	char *fcfile;
	extern int leaux();

	if ((fcfile = makefc(path)) == NULL)
		return -1;
	status = leaux(fcfd, fcfile, srcstbufp);
	free(fcfile);
	return status;
}

int
leaux(fcfd, path, srcstbufp)
	int fcfd;
	char *path;
	struct stat *srcstbufp;
{
	FILE *fp;
	int status, len, checknum;
	u_char *table;
	struct stat objstbuf;
	char buf[80];
	extern char *progname;
	extern int magic_number;
#if 0
	extern struct osCmd *globalcaller, avcmd;
	extern struct codedesc *interpret();
#endif

	if ((fp = fopen(path, "r")) == NULL)
		return -1;
	if (std_fgets(buf, sizeof buf, fp) == NULL) {
		(void) fclose(fp);
		fprintf(stderr, "%s: cannot get first line of %s\n",
				progname, path);
		return -1;
	}
	if (sscanf(buf, "#!zsh -l%d", &checknum) != 1) {
		(void) fclose(fp);
		fprintf(stderr, "%s: %s is not a precompiled script\n",
				progname, path);
		return -1;
	}
	if (checknum != magic_number) {
		(void) fclose(fp);
		fprintf(stderr, "%s: %s has wrong magic value %d, want %d\n",
				progname, path, checknum, magic_number);
		return -1;
	}
	if (fstat(fileno(fp), &objstbuf) < 0) {
		(void) fclose(fp);
		fprintf(stderr, "%s: fstat failed on %s\n",
				progname, path);
		return -1;
	}
	if (srcstbufp != NULL && srcstbufp->st_mtime > objstbuf.st_mtime) {
		(void) fclose(fp);
		(void) unlink(path);
		return -1;
	}
	len = objstbuf.st_size - ftell(fp);
	table = (u_char *)emalloc(len);
	if (std_fread(table, sizeof *table, len / sizeof *table, fp) != len / sizeof *table) {
		(void) fclose(fp);
		fprintf(stderr, "%s: read of %d failed on %s\n",
				progname, len, path);
		return -1;
	}
	(void) fclose(fp);
	(void) close(fcfd);
	status = 0;
	(void) interpret(table, table + len, (u_char *)NULL,
				globalcaller == NULL ? &avcmd
						     : globalcaller,
				&status, (struct codedesc *)NULL);
	return status;
}

/*
 * If unprocessed signals are pending (and the sprung flag set), we call
 * this function to do the processing.  It will deal with pending signals
 * in numerical as opposed to chronological order.
 */

void
trapped()
{
	int i;
	static int intrap = 0;

	if (!sprung)
		return;
	/*
	 * We must reset the sprung flag before calling aval(), or we will
	 * almost certainly get a recursive invocation of this routine from
	 * the interpreter.
	 */
	sprung = 0;

	/*
	 * What about this scenario: interpreter calls trapped.  trapped calls
	 * eval which calls interpreter.  a signal is delivered and sprung
	 * gets set.  the interpreter calls trapped again.  some traps will
	 * then maybe get run twice.  Ergo we need some semaphore here.
	 */
	if (intrap) {
		/* more signals have arrived, be sure not to miss them */
		++intrap;
		return;
	}
	for (++intrap; intrap > 0; --intrap) {
		for (i = 1; i < NSIG; ++i)
			while (spring[i] > 0) {
				if (traps[i] != NULL)
					(void) eval(traps[i],
						    "trap", (char *)NULL);
				--spring[i];
			}
	}
}


/*
 * This is the exit routine used when one wants a "trap 0" to be honoured.
 */

void
trapexit(n)
	int n;
{
	if (traps[0] != NULL) {
		char *cmd = traps[0];
		traps[0] = NULL;
		(void) eval(cmd, "exit trap", (char *)NULL);
	}
#ifdef	MALLOC_TRACE
	mal_dumpleaktrace(stderr);
	/* mal_heapdump(&_iob[2]); */
#endif	/* MALLOC_TRACE */
	exit(n);
	/* NOTREACHED */
}

/*
 * The builtin "trap" function is implemented here.
 */

int
sh_trap(argc, argv)
	int argc;
	char *argv[];
{
	int i;
	char *av0, *script;
	
	if (argc == 1) {
		/* just print the known traps */
		for (i = 0; i < NSIG; ++i) {
			if (traps[i] != NULL)
				printf("%d: %s\n", i, traps[i]);
		}
		return 0;
	}
	av0 = argv[0];
	--argc, ++argv;
	if (**argv == '\0') {
		/* ignore the specified signals */
		script = *argv;
		--argc, ++argv;
	} else if (isascii(**argv) && isdigit(**argv)) {
		/* reset the signal handlers to original value */
		script = NULL;
	} else {
		/* stash the script away for later execution by a trap */
		script = *argv;
		--argc, ++argv;
		/* don't bother guarding argc > 0, sh doesn't */
	}
	while (argc-- > 0) {
		if (!isascii(**argv) || !isdigit(**argv)) {
			fprintf(stderr, "%s: bad number: '%s'\n", av0, *argv);
			++argv;
			continue;
		}
		i = atoi(*argv++);
		if (i < 0 || i >= NSIG) {
			fprintf(stderr, "%s: %s: %s\n",
					av0, BAD_TRAP, *(argv-1));
			continue;
		}
		if (traps[i] != NULL)
			(void) free(traps[i]);
		if (script != NULL && *script != '\0') {
			traps[i] = emalloc(strlen(script)+1);
			(void) strcpy(traps[i], script);
			/* enable that signal */
			if (i > 0 && orig_handler[i] != SIG_IGN)
				(void) signal(i, trap_handler);
		} else if (script != NULL) {
			traps[i] = NULL;
			/* disable that signal */
			if (i > 0)
				(void) signal(i, SIG_IGN);
		} else {
			traps[i] = NULL;
			if (i > 0)
				(void) signal(i, orig_handler[i]);
		}
	}
	return 0;
}
