
/* Copyright (c) 1992 Vincent Cate
 * All Rights Reserved.
 *
 * Permission to use and modify this software and its documentation 
 * is hereby granted, provided that both the copyright notice and this 
 * permission notice appear in all copies of the software, derivative works 
 * or modified versions, and any portions thereof, and that both notices 
 * appear in supporting documentation.  This software or any derivate works
 * may not be sold or distributed without prior written approval from 
 * Vincent Cate.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND VINCENT CATE DISCLAIMS ALL 
 * WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS.  IN NO EVENT SHALL 
 * VINCENT CATE BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR 
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS 
 * OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE 
 * OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE 
 * OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Users of this software agree to return to Vincent Cate any improvements 
 * or extensions that they make and grant Vincent Cate the rights to 
 * redistribute these changes.
 * 
 */

#include "alexincs.h"
#include "alex.h"

#define MAXSECSINCONSISTENT  3600*24*7

#define REMOVEALEXDIR 1
#define NOREMOVEALEXDIR 0


#define MAXHOSTS 50000

#define MAXPARTSINHOSTNAME 6
/* I only know of 2 hosts with 6 part names.  One is:
 * USUHS.UCC.USUHS.NNMC.NAVY.MIL:131.158.4.3 CPU=VAX-8350 OS=VMS 5.0
 */



struct HostEntry *HostsTable[MAXHOSTS];
int  HostsInTable=0;

#if(defined(MAXUID) && (MAXUID < 100000)) /* have fears of 4,000,000 as MAXUID */                         
#define MAXUIDNUM MAXUID               /* but if not too big we should use it */
#else 
#define MAXUIDNUM 65535                /* 16 bits is the usual limit I think */
#endif

char *UidStrTable[MAXUIDNUM];   


char CMUMACHINE[100];          /* At CMU, "ls -L" has the oposite meaning of other places, */
int LENCMUMACHINE;             /* so games are played to do things different here          */




/*   Input  Path="/foo/bar/baz"
 *   Output part="foo"
 *   returns pointer to "/bar/baz"
 */
char *GetNextPartPath(Path, part)
char *Path, *part;
{
    part[0]=0;

    if ((Path==NULL) || (*Path != '/')) {
        return(NULL);
    }
    Path++;
    while ((*Path != '/') && (*Path != 0)) {
        *part++ = *Path++;
    }
    return(Path);
}





/*  We used to load in the The table now is just a cache, so there is not really any
 *  need to load it from a file.  
 *
 *  We return AFAIL if we can not load and AOK if we can but expect 
 *  that nobody really cares.
 */
int LoadUidStrTable()
{
    FILE *UidStrFile;
    char line[MAXPATH], token[MAXPATH];
    int Uid, i, NumUids;

    for (i=0; i<MAXUIDNUM; i++) {
        UidStrTable[i]=NULL;
    }

    if (streql(UIDSTR, "")) {
        return(AFAIL);
    }

    NumUids=0;
    UidStrFile=AFOPEN(UIDSTR,"r");
    if (UidStrFile==NULL) {
          ToLog(DBERROR, "No UidStr file %s\n", UIDSTR);
          ToLog(DBMAJOR, "LoadUidStrTable done loaded %d\n", NumUids);
          (void) AFCLOSE(UidStrFile);
          return(AFAIL);
    }

    while (fgets(line, MAXPATH , UidStrFile) != NULL) {
        if (strlen(line)>2) {
            (void) GetNthToken(line, 0, token);
            Uid= -1;
            if (sscanf(token, "%d", &Uid) !=1) {
                ToLog(DBERROR, "LoadUidStrTable ERROR expected to sscanf a number %s\n", token);
            } else {
                if ((0 <= Uid) && (Uid <= MAXUIDNUM)) {
                    (void) GetNthToken(line, 1, token);
                    UidStrTable[Uid]=strsave(token);
                    NumUids++;
                }
            }
        }
    }

    (void) AFCLOSE(UidStrFile);

    ToLog(DBMAJOR, "LoadUidStrTable done loaded %d\n", NumUids);
    return(AOK);
}


/*  format of file (only first 2 implemented XXXX)
 *
 *      Path          type    sec-connect-time     avg-kb/sec      time-last-used
 *  /edu/berkeley     UNIX     totcsec, numcon     totkb, totsec      0.0
 *
 *  Any line that does not start with a '/' is treated as a comment
 */
void LoadHostsTable()
{
    FILE *HostsListFile;
    char line[MAXPATH], HostsListNew[MAXPATH], HostsList[MAXPATH];
    char Tokens[MAXTOKENS][MAXTOKENLEN];
    int  Type, NumTokens, Status, i, NeedToRemoveHostsListNew, BadEntry;
    struct stat StatBuf;

    Log("LoadHostsTable starting");


    (void) strcpy(HostsListNew, HOSTSLIST);
    (void) strcat(HostsListNew, ".new");

    Status = lstat(HostsListNew, &StatBuf);
    if (Status == 0) {
        (void) strcpy(HostsList, HostsListNew);
        NeedToRemoveHostsListNew=1;
    } else {
        (void) strcpy(HostsList, HOSTSLIST);
        NeedToRemoveHostsListNew=0;
    }
 
    Log2("LoadHostsTable using ", HostsList);
    HostsListFile=AFOPEN(HostsList,"r");
    if (HostsListFile==NULL) {
          error2("No HostsList file", HOSTSLIST);
          exit(-1);
    }

    for (i=0; i<MAXHOSTS; i++) {
         HostsTable[HostsInTable]=NULL;
    }
    HostsInTable=0;
    while ((HostsInTable<MAXHOSTS) && (fgets(line, MAXPATH , HostsListFile) != NULL)) {
        if (strlen(line)>2) {
            line[strlen(line)-1]=NULL;                           /* get rid of New Line */
            NumTokens=LineToTokens(line, Tokens);
            BadEntry=0;
            if (((NumTokens == 5) || (NumTokens == 6)) && (Tokens[0][0] == '/')) {
                HostsTable[HostsInTable]= (struct HostEntry *) malloc(sizeof(struct HostEntry));
                HostsTable[HostsInTable]->Path=strsave(Tokens[0]); 

                HostsTable[HostsInTable]->PartsInHostName = CountOccurances(Tokens[0], '/');
                HostsTable[HostsInTable]->PathLen = strlen(Tokens[0]);

                Type=StringToAType(Tokens[1]);
                if ((Type != UNKNOWN) && ((Type < MinHostType) || (MaxHostType < Type ))) {
                    ToLog(DBERROR, "LoadHostsTable ERROR BUG Type= %s\n", ATypeToString(Type));
                    Type=UNKNOWN;
                }

                HostsTable[HostsInTable]->Type=Type;

                HostsTable[HostsInTable]->TimeZone = atoi(Tokens[2]);

                HostsTable[HostsInTable]->TimeLastCheck = atoi(Tokens[3]);  

           
                if (strlen(Tokens[NumTokens-1]) > 0) {
                    HostsTable[HostsInTable]->SymLink=strsave(Tokens[NumTokens-1]);
                    if (Type != HOSTALIAS) {
                        ToLog(DBERROR, "LoadHostsTable ERROR EXTRA HOSTALIAS %s\n", Tokens[NumTokens-1]);
                        BadEntry=1;
                    }
                } else {
                    HostsTable[HostsInTable]->SymLink=NULL;
                    if (Type == HOSTALIAS) {
                        ToLog(DBERROR, "LoadHostsTable ERROR missing HOSTALIAS %s\n", 
                                                  HostsTable[HostsInTable]->Path);
                        BadEntry=1;
                    }
                }

                HostsTable[HostsInTable]->RespTime =0.0;
                HostsTable[HostsInTable]->KBPSec =0.0;

                if (!BadEntry) {
                    HostsInTable++;
                }
            }
        }
    }

    if (HostsInTable == MAXHOSTS) {
        Log("ERROR can not read in all of the hosts");           /* just don't even start */
        exit(-1);
    }

    LogN("LoadHostsTable done.  Found ", HostsInTable);
    (void) AFCLOSE(HostsListFile);
    if (NeedToRemoveHostsListNew) {
        (void) unlink(HostsListNew);
    }
}


/*  Add a host to HostsTable array
 */
int AddNewHost(HostPath, Type, AliasPath, PtrPtrHostEntry)
char *HostPath;
int Type;
char *AliasPath;
struct HostEntry **PtrPtrHostEntry;                           /* return value if AOK */
{
    int i, InsertLoc, Status;

    *PtrPtrHostEntry = NULL;

    if (HostPath[0] != '/') {
        ToLog(DBERROR, "AddNewHost ERROR BUG bad argument %s\n", HostPath);
        return(AFAIL);
    }

    if (HostsInTable+1 >= MAXHOSTS) {
        Log2("AddNewHost did not have room in table for ", HostPath);
        return(AFAIL);
    }

    Log2("AddNewHost is adding ", HostPath);

    InsertLoc=0;
    for(i=HostsInTable-1; (InsertLoc==0) && (i>= 0); i--) {
        Status=strcmp(HostsTable[i]->Path, HostPath);
        if (Status > 0) {
            HostsTable[i+1]=HostsTable[i];
        } else {
            InsertLoc= i+1;
            if (Status==0) {
                ToLog(DBERROR, "AddNewHost ERROR BUG added host already in table %s\n", HostPath);
            }
        }
    }

    HostsTable[InsertLoc]=(struct HostEntry *) malloc(sizeof(struct HostEntry));

    HostsTable[InsertLoc]->Path=strsave(HostPath);

    HostsTable[InsertLoc]->PathLen = strlen(HostPath);
    HostsTable[InsertLoc]->PartsInHostName = CountOccurances(HostPath, '/');

    if (Type == AOK) {
        Type=UNKNOWN;
    }
    HostsTable[InsertLoc]->Type = Type;
    HostsTable[InsertLoc]->RespTime =0.0;
    HostsTable[InsertLoc]->KBPSec =0.0;
    HostsTable[InsertLoc]->NumErrorsSinceOk =0;
    HostsTable[InsertLoc]->SymLink=strsave(AliasPath);
    HostsInTable++;


    LogN("AddNewHost added at ", InsertLoc);

    *PtrPtrHostEntry = HostsTable[InsertLoc];

    return(AOK);
}

/* Should be called when we get a NOSUCHHOST */
int RemoveHost(HostPtr)
struct HostEntry *HostPtr; 
{
    int Index;

    if (HostPtr == NULL) {
        ToLog(DBERROR, "RemoveHost got a NULL\n");
        return(AFAIL);
    }

    for (Index = 0; Index < HostsInTable; Index++ ){
        if (HostsTable[Index] == HostPtr) {
            while (Index < HostsInTable) {
                HostsTable[Index] = HostsTable[Index+1];
                Index++;
            }
            HostsInTable--;
        }
    }
    return(AOK);
}


/* Hope is that we know some more types by now
 */
int SaveHostsTable()
{
    FILE *HostsListFile;
    int i, Status;
    char EmptyString[10], OldName[MAXPATH], TmpName[MAXPATH], *SL;
    char *Path;

    EmptyString[0]=0;

    Log("SaveHostsTable");
    flushLog();

    (void) strcpy(OldName, HOSTSLIST);
    (void) strcat(OldName, ".old");

    (void) strcpy(TmpName, HOSTSLIST);
    (void) strcat(TmpName, ".tmp");


    HostsListFile=AFOPEN(TmpName,"w");
    if (HostsListFile==NULL) {
          error2("SaveHostsTable can not open HOSTSLIST for write ", HOSTSLIST);
          return(AFAIL);
    }

    for (i=0; i<HostsInTable; i++) {
        if (HostsTable[i]->SymLink == NULL) {
            SL=EmptyString;
        } else {
            SL=HostsTable[i]->SymLink;
        }

        Path=HostsTable[i]->Path;
        if (((HostsTable[i]->Type == ADOMAIN) &&
                   (HostsTable[i+1] != NULL) &&
                   (HostsTable[i+1]->Path != NULL) &&
                  !strncmp(Path, HostsTable[i+1]->Path, strlen(Path)))    ||
            ((HostsTable[i]->Type != ADOMAIN) && (HostsTable[i]->Type != NOSUCHHOST))) {

            (void) fprintf(HostsListFile, "\"%s\" \t%s \t%d \t%d \t\"%s\"\n", 
                        HostsTable[i]->Path, ATypeToString(HostsTable[i]->Type), 
                        HostsTable[i]->TimeZone, HostsTable[i]->TimeLastCheck, SL);
        } else {
            Log2("SaveHostsTable not saving ", HostsTable[i]->Path);
        }
    }

    (void) AFCLOSE(HostsListFile);

    (void) unlink(OldName);                /* rm HostsList.old           */
    Status = link(HOSTSLIST, OldName);     /* ln HostsList HostsList.old */
    if (Status != 0) {
        error1("SaveHostsTable ERROR could not rename ");
    }
    Status = rename(TmpName, HOSTSLIST); /* mv HostsList.tmp HostsList -  rename guarantees atomic) */
    if (Status != 0) {
        error1("SaveHostsTable ERROR could not rename ");
    }

    LogN("SaveHostsTable finished ok", i);
    flushLog();
    return(AOK);
}


/* 
 *  Seems like this should not really be needed any more...              XXXXXXXXXX
 *
 *  Algorithm:
 *      Set WorkPath to include the last directory we make
 *      Recursively call CheckIntermediateDirs on directory above WorkPath
 *      Then mkdir WorkPath
 *
 *  Might need to check that there is a .alex.info in working order...
 *  Alexfsck.c should catch this sort of thing.  But...                  XXXXXXXXXX
 */

int CheckIntermediateDirs(ArgPath, AndThisDir)
char *ArgPath;
int  AndThisDir;
{
     static char LastWorkPath[MAXPATH]="";
     static int LastStatus;
     char WorkPath[MAXPATH];
     int mode, Status;
 

     if (AndThisDir) {
         (void) strcpy(WorkPath, ArgPath);
     } else {
         PathToDir(ArgPath, WorkPath); 
     }

     Log2("CheckIntermediateDirs (and this one) ", WorkPath); 
 
     if (streql(LastWorkPath, WorkPath) && (LastStatus == AOK)) {
        Log2("CheckIntermediateDirs has done this already", WorkPath);
        return(AOK);
     }
   
     Status=LiteStat(WorkPath);
     if ((Status != ADIR) && (Status != ALINK)) {
         Log2("CheckIntermediateDirs doing something ", WorkPath); 
         mode=0755;
         Status=mkdir(WorkPath, mode);
         if (Status != 0) {
             if (CheckIntermediateDirs(WorkPath, 0) == AFAIL){
                 return(AFAIL);
             }
             Status=mkdir(WorkPath, mode);
             if (Status != 0) {
                  Log("CheckIntermediateDirs Unlinking file and making a dir in its place");
                  if (unlink(WorkPath)) error2("CheckIntermediateDirs unlink", WorkPath);             
                  Status=mkdir(WorkPath, mode);
                  if (Status != 0) {
                       Log("CheckIntermediateDirs: I tried very hard but could not make a dir");
                       Log(WorkPath);
                       return(AFAIL);
                  }
             }
        }
    } 

    /* Check for ALEXERROR?              XXXXXXXXXXXXX  */

/*
 *   if (!AndThisDir) {
 *       Status=CheckAlexInfoHigh(WorkPath, UidStr);
 *   }
 */

    (void) strcpy(LastWorkPath, WorkPath);
    return(AOK);
}

int NumSameChars(s1, s2)
char *s1, *s2;
{
    int Result;

    if ((s1==NULL) && (s2==NULL)) {
        Log("ERROR NumSameChars does not expect null strings");
        return(0);
    }

    for (Result=0; ((*s1 != 0) && (*s1++ == *s2++)); Result++) {
        ;
    }

    return(Result);
}



#define PathLTHost 0
#define PathGEHost 1
#define BESTMATCH  2
#define PathEQHost 3

/* Search HostsTable to see if we know how this one 
 * If we do return a pointer to the host entry
 *
 *    Imagine the Machines:
 *    /edu/cmu/art
 *    /edu/cmu/cs
 *    /edu/cmu/cs/aba
 *    /edu/cmu/cs/mama
 *    /edu/cmu/cs/zztop
 *    
 *    And the path we are searching with:
 *    /edu/cmu/cs/zagnuts
 *  
 
 *    PathGEHost means that we should assume strlen(path) >= strlen(HostName)
 *    PathLHost  means we want part of a host and that path is shorter
 *    BestMatch  means just return the best you can find
 *    PathEQHost means same
 *    
 *    Find where it goes binary search 
 *        If equal return that
 *        If no exact recursively try length just longer than matched (to next /)
 *        recurse if necessary
 *    If any subset of parts of path matches a machine return pointer to that host path
 *
 *    
 */

int CheckHostsTable(Path, typeofcheck, PtrPtrHostEntry) 
char *Path;
int  typeofcheck;
struct HostEntry **PtrPtrHostEntry;                        /* returned pointer */
{
   int i, cmpstat, Low, High, BestMatch, BML, cmplen;
   char recPath[MAXPATH];                
   int LowSame, HighSame, shorter, progress;

   Log2("CheckHostsTable", Path); 

   if (strlen(Path)<3) return(AFAIL);
   if (Path[0] != '/') return(AFAIL);

   BestMatch= -1;
   BML=0;
 
   Low=0;
   High=HostsInTable-1;
   progress=1;
   while (progress) {
       progress=0;
       i=(int) rint((Low+High)/2.0);
       cmpstat=strcmp(HostsTable[i]->Path, Path);
       if (cmpstat==0) {
           Log2("CheckHostsTable found 1 ", HostsTable[i]->Path);
           *PtrPtrHostEntry=HostsTable[i];
           return(AOK);
       }
       if ((cmpstat<0) && (i>Low)) {
           Low=i;
           progress=1;
       }
       if ((cmpstat>0) && (i<High)) {
           High=i;
           progress=1;
       }
   }


   if (Low < High-1) {
       ToLog(DBERROR, "CheckHostsTable ERROR BUG Low<<High Low= %d  High= %d \n", Low, High);
   }

   if (Low != High) { 
       cmpstat=strcmp(HostsTable[Low]->Path, Path);
       if (cmpstat==0) {
           Log2("CheckHostsTable found 2 ", HostsTable[Low]->Path);
           *PtrPtrHostEntry=HostsTable[i];
           return(AOK);
       }
       cmpstat=strcmp(HostsTable[High]->Path, Path);
       if (cmpstat==0) {
           Log2("CheckHostsTable found 3 ", HostsTable[High]->Path);
           *PtrPtrHostEntry=HostsTable[i];
           return(AOK);
       }
   }
       
   switch (typeofcheck) {
       case PathEQHost:   Log2("CheckHostsTable did not find exact match ", Path);
                          return(AFAIL);              /* prob would be ok if did PathGEHost */

       case PathGEHost:   cmplen=strlen(HostsTable[Low]->Path);
                          break;

       case BESTMATCH:    
       case PathLTHost:   cmplen=strlen(Path);
                          break;


       default:
           ToLog(DBERROR, "CheckHostsTable ERROR BUG bad typeofcheck %d\n", typeofcheck);
           return(AFAIL);
   }

   cmpstat=strncmp(HostsTable[Low]->Path, Path, cmplen);
   if ((cmpstat==0) && ((Path[cmplen]==0) || (Path[cmplen]=='/'))) {
       BestMatch = Low;
       BML = strlen(HostsTable[Low]->Path);
   }

   if (strlen(HostsTable[High]->Path) > BML) {
       if (typeofcheck==PathGEHost) {
           cmplen=strlen(HostsTable[High]->Path);
       }
       cmpstat=strncmp(HostsTable[High]->Path, Path, cmplen);
       if ((cmpstat==0) && ((Path[cmplen]==0) || (Path[cmplen]=='/'))) {
           BestMatch = High;
           BML = strlen(HostsTable[High]->Path);
       }
   }

   if (BML>0) {
       Log2("CheckHostsTable found 4 ", HostsTable[BestMatch]->Path); 
       *PtrPtrHostEntry=HostsTable[BestMatch];
       return(AOK);
   } else {

       LowSame=NumSameChars(Path, HostsTable[Low]->Path);
       HighSame=NumSameChars(Path, HostsTable[High]->Path);

       if (LowSame > HighSame) {
           BestMatch=Low;
           BML=LowSame;
       } else {
           BestMatch=High;
           BML=HighSame;
       }

       if ((BML <= 1) || (typeofcheck==PathLTHost)) {                 /* strlen("/")  */
           Log("CheckHostsTable about to return AFAIL"); 
           return(AFAIL);
       }

       if (typeofcheck == BESTMATCH) {
           Log2("Returning a BESTMATCH of ", HostsTable[BestMatch]->Path);
           *PtrPtrHostEntry=HostsTable[BestMatch];
           return(AOK);
       }

       shorter=0;
       (void) strcpy(recPath, Path);

       for (i=BML; i<strlen(recPath); i++) {
           if (recPath[i] == '/') {          /* terminate at end of name that matched part */
               recPath[i] = 0;
               shorter=1;                    /* making progress */
           }
       }

       if (!shorter) {
           for (i=(strlen(recPath)-1); (i>=0) && (recPath[i] != '/'); i--) {
                ;
           }
           if (i>=0) recPath[i]=0;
       }
               
       if (streql(recPath, Path)) {         /* never do same work again  XXX debug only */
           Log2("ERROR recursion should not have done this", Path);
           return(AFAIL);
       }
    
       /* Log2("CheckHosts about to recurse ", recPath); */
       return(CheckHostsTable(recPath, typeofcheck, PtrPtrHostEntry));
    }
}



/* 
 *  We may be on our way to a host.  For example at /alex/edu/cmu which is 
 *    not a host but /alex/edu/cmu/cs is.
 *
 *  Even if we did get an exact match in the host list we may still just be
 *    working with a partial HostName.  For example /alex/edu/cmu/cs/nectar
 *    will match but should not really.
 *
 *  If we have a partial then just make sure there is a directory and return(1)
 *
 *  Returns 1 if this Path is only part of some host name in HostsTable else 0
 *      i.e. there is some host name that starts with this
 *      so true for nectar.cs.cmu.edu
 */
int PartHostPath(Path)
char *Path;
{
    char locPath[MAXPATH];
    char *LongHostName;
    struct HostEntry *HostPtr;
    int Status;

    Log2("PartHostPath ", Path);

    if (Path[0] == 0) return(0);               /* we are at CACHEDIRVAR */

    if (Path[0] != '/') {
        Log2("PartHostPath ERROR ", Path);
        return(0);
    }

    (void) strcpy(locPath, Path);
    (void) strcat(locPath, "/");
    Status=CheckHostsTable(locPath, PathLTHost, &HostPtr);
    if (Status == AOK) {
        LongHostName=HostPtr->Path;
    } else {
        LongHostName=NULL;
        AlexAssert(Status == AFAIL);
    }
 
    if ((LongHostName != NULL) && (strlen(LongHostName) > strlen(Path))) {
        Log2("PartHostPath found partial host Path - ", Path);
        (void) strcpy(locPath, CACHEDIRVAR);
        (void) strcat(locPath, Path);
       /* (void) CheckIntermediateDirs(locPath, 1); */
        Log2("PartHostPath returning YES ", Path);
        return(1);
    }

    Log("PartHostPath returning NO ");
    return(0);
}





/*
 *                             2         2            2
 *   Inputs: FilePath  =  /usrX/alex/edu/berkeley/games/nettrek
 *           LinkPath  =  /fun/nettrek  
 *           HostPath  =  /edu/berkely
 *
 *   Output: LinkName  =  ../../fun/nettrek
 *
 *   Only works if /fun/nettrek works on ftp site (i.e. don't need /usr/anon/fun)
 */
int AbsLinkPathToRel(FilePath, LinkPath, HostPath)
char *FilePath, *LinkPath, *HostPath;
{
    int numdotdot, i;
    char tmpLinkPath[MAXPATH];

    Log2("AbsLinkPathToRel input ", FilePath);
    Log2("AbsLinkPathToRel input ", LinkPath);
    Log2("AbsLinkPathToRel input ", HostPath);

    numdotdot= CountOccurances(FilePath, '/') -
               CountOccurances(CACHEDIRVAR, '/') -
               CountOccurances(HostPath, '/');

    tmpLinkPath[0]=0;
    for (i=0; i<(numdotdot-1); i++) {
        (void) strcat(tmpLinkPath, "../");
    }
    i++;
    if (i < numdotdot) {
        (void) strcat(tmpLinkPath, "..");          /* LinkName is abs so starts with a slash */
    }

    (void) strcat(tmpLinkPath, LinkPath);
    (void) strcpy(LinkPath, tmpLinkPath);

    Log2("AbsLinkPathToRel returning ", LinkPath);
    return(1);
}

/*  Want to see if symbolic links go higher than /usrX/alex (really should be
 *  if higher than that machine but only higher than /usrX/alex is dangerous).
 *      
 *   Inputs:
 *          FilePath    /usrX/alex/edu/berkeley/games/nettrek
 *          LinkPath    /fun/nettrek   or ../nettrek  or  mud/nettrek
 *          HostPath    /edu/berkeley
 *   Output:
 *          AOK or AFAIL
 */
int OkSymLink(FilePath, LinkPath, HostPath)
char *FilePath, *LinkPath, *HostPath;
{
    char part[MAXPATH], *tmp;
    int   depth;
    int   Result;

    Log2("OkSymLink FilePath ", FilePath);
    Log2("OkSymLink LinkPath ", LinkPath);

    tmp=FilePath;
    for (depth=0; (tmp != NULL) && (*tmp != 0); depth++ ) {
        tmp=GetNextPartPath(tmp, part);
        if (strlen(part) < 1) {
            Log2("OkSymLink ERROR has // that should not be", FilePath);
            return(AFAIL);
        }
    }    
    
    if (*LinkPath == '/') {
        if ((strncmp(LinkPath, "/alex/", strlen("/alex/"))==0) ||  /* Special roots */
            (strcmp(LinkPath, "/alex")==0)) {
	    char tmpLinkPath[MAXPATH];

            if (HasString(LinkPath, "..")) {                   /* no .. in these symlinks */
                Log2("OkSymLink sees .. ", LinkPath);
                return(AFAIL);
            }                                                          /* leave link as is */
	
	    Log2("OkSymLink normalizing LinkPath ", LinkPath);
	    strcpy(tmpLinkPath, ALEXPATH);
	    strncat(tmpLinkPath, LinkPath + 5, MAXPATH - sizeof(ALEXPATH));
	    strcpy(LinkPath, tmpLinkPath);
	} else if ((strncmp(LinkPath, "/afs/", strlen("/afs/"))==0)   ||
            (strcmp(LinkPath, "/afs")==0))  {       
            if (HasString(LinkPath, "..")) {                   /* no .. in these symlinks */
                Log2("OkSymLink sees .. ", LinkPath);
                return(AFAIL);
            }                                                          /* leave link as is */
        } else {
            (void) AbsLinkPathToRel(FilePath, LinkPath, HostPath);     /* modifies LinkPath */
        }
    }

    tmp=LinkPath;
    while ((depth>2) && (tmp != NULL) && (*tmp != 0)) {
        tmp=GetNextPartPath(tmp, part);
        if (streql(part, "..")) {
            depth--;
        } else {
            depth++;
        }
    }

    Log2("OkSymLink LinkPath at end ", LinkPath);

    if (depth<2) {
        Result=AFAIL;                   /* at least /usr/alex or too high */
    } else {
        Result=AOK;
    }

    LogT("OkSymLink returning ", Result);

    return(Result);
}


/*  Expected to make a symlink
 */
int ProbMakeSymLink(FilePath, LinkPath, HostPath)
char *FilePath, *LinkPath, *HostPath;
{
/*    int Status; */

    Log2("ProbMakeSymLink ", LinkPath);

    if ((LinkPath != NULL) && (LinkPath[0] == 0)) {
       Log2("ERRORProbMakeSymLink did not get a LinkPath", LinkPath);
       return(AFAIL);
    }

    if (OkSymLink(FilePath, LinkPath, HostPath) != AOK) {
       Log2("ProbMakeSymLink ERROR we do not like this symlink ", LinkPath);
       return(AFAIL);
    }

    Log2("ProbMakeSymLink about to make a symlink to ", LinkPath);

    /* Status=symlink(LinkPath, FilePath); */
    Log2("ProbMakeSymLink no longer really makes symlinks ", LinkPath);
/*
 *   if (Status == 0) {
 *       Log("ProbMakeSymLink made a symlink");
 *   } else {
 *       error2("ProbMakeSymLink could not make symlink", LinkPath);
 *   }
 */

    Log2(FilePath, LinkPath);                /* if we get this far print these */
    return(AOK);
}



/*
 *    Use FTP's "dir" command to make a .alex.dir file
 *    If it fails - assume it always fails and flag this
 *    by making a zero length .alex.dir
 */
int MakeAlexDir(DirPath, HostPtr, HostDir, UidStr)
char *DirPath;
struct HostEntry *HostPtr;
char *HostDir, *UidStr;
{
    char tmpdir[10];
    int  Result, SSW;
    char AlexDirPath[MAXPATH];
    char AlexErrorPath[MAXPATH];
    char AlexInfoPath[MAXPATH];
    double ADAge, AIAge; 
  
    (void) strcpy(AlexDirPath, DirPath);
    (void) strcat(AlexDirPath, SLASHALEXDIR);

    (void) strcpy(AlexInfoPath, DirPath);
    (void) strcat(AlexInfoPath, SLASHALEXINFO);

    Log2("MakeAlexDir AlexDirPath ", AlexDirPath);
    Log2("MakeAlexDir HostPtr->Path         ", HostPtr->Path);    /* spaced to lign up */

    (void) strcpy(AlexErrorPath, DirPath);
    (void) strcat(AlexErrorPath, SLASHALEXERROR);

    SSW=SecsSinceWrite(AlexErrorPath);
    if ((SSW > 0) && (SSW < SECSTILLRETRY)) {         /* if exists and is recent then fail */
        return(AFAIL);
    } 

    Result=AOK;


    AIAge = SecsSinceWrite(AlexInfoPath);
    ADAge=SecsSinceWrite(AlexDirPath);
    if (!InTransit(AlexDirPath) &&                /* give DoAnFtp chance to remove .alex.error */
        (ADAge>0) && 
        ((ADAge < 60) || ((ADAge < AIAge) && (ADAge < SECSTILLRETRY))) &&      
        (SizeOfCachedFile(AlexDirPath) > 5)) {
          Log2("MakeAlexDir has a recent .alex.dir ", AlexDirPath);
          Result=AOK;

    } else {

        if (HostDir == NULL) {
            Log("MakeAlexDir hates null pointers");
            HostDir = &tmpdir[0];
            (void) strcpy(HostDir,"/");
        } 

        if (HostDir[0]==0) {
            (void) strcat(HostDir,"/");
        }
        Log2("MakeAlexDir ", HostDir);

        if (Result==AOK) {
            Log2("About to send an ftp dir command", AlexDirPath);
            Result=DoAnFtp(HostPtr, "dir", HostDir, AlexDirPath, UidStr);
            if (Result==NOSUCHHOST) {
                (void) RemoveHost(HostPtr);
                Result=AOLD;                    /* could be ADOMAIN now so don't RecursiveRm */
                                                /* if need be dirtoinfo will remove */
            } else if (Result==AOLD)  {  
                (void) RecursiveRm(DirPath, __FILE__, __LINE__, DONOTWAIT);
            }
        }
    }

    LogT("MakeAlexDir is returning", Result);
    return(Result);
}



/*  Higher powers have decided to make a new .alex.info and .alex.dir
 */
int MakeAlexInfo(DirPath, HostPtr, RemotePath, UidStr)
char *DirPath;
struct HostEntry *HostPtr;
char *RemotePath, *UidStr;
{
    int Result;
    char LastRmed[MAXPATH];

    Result=MakeAlexDir(DirPath, HostPtr, RemotePath, UidStr);

    LogT("MakeAlexInfo finished with MakeAlexDir ", Result);

    if (Result != AWORKING) {
        Result=DirToInfo(DirPath, HostPtr, UidStr);       /* call even if no .alex.dir    */
        if (Result==AOLD){                                /* means no such thing any more */
              Log2("MakeAlexInfo going to do some rming", DirPath);
              if (streql(LastRmed, DirPath)) {
                  Result=AFAIL;
              }
              (void) strcpy(LastRmed, DirPath);
              (void) RecursiveRm(DirPath, __FILE__, __LINE__, DONOTWAIT);
        } else if (Result==AOK) {
              Log2("MakeAlexInfo thinks everything is good ", DirPath);
        } else {
            LogT("MakeAlexInfo thinks DirToInfo did not work", Result);
        }
    }

    LogT("MakeAlexInfo returning ", Result);
    return(Result);
}    


/*   Make the .alex.info file from .alex.dir file
 *   For any new place we will have to make .alex.dir as well
 */
int CheckAlexInfo(DirPath, HostPtr, RemotePath, UidStr)
char *DirPath;
struct HostEntry *HostPtr;
char *RemotePath, *UidStr;
{
    int Status, EStatus;
    char AlexInfoPath[MAXPATH], AlexErrorPath[MAXPATH];
    struct stat StatBuf, EStatBuf;

    (void) strcpy(AlexInfoPath, DirPath);
    (void) strcat(AlexInfoPath, SLASHALEXINFO);

    Status=AlexStat(AlexInfoPath, &StatBuf);

    (void) strcpy(AlexErrorPath, DirPath);
    (void) strcat(AlexErrorPath, SLASHALEXERROR);
    EStatus=AlexStat(AlexErrorPath, &EStatBuf);        /* Would be faster if we could just      */
                                                       /* work off what is in .alex.info XXXXX  */

    if (Status != AFILE) {
        Status=MakeAlexInfo(DirPath, HostPtr, RemotePath, UidStr);
    } else {
        if (((TimeInSeconds() - StatBuf.st_mtime) > MAXSECSINCONSISTENT) ||
            (StatBuf.st_mtime < MINTIMEOKALEXINFO) || 
            ((EStatus == AFILE) && ((TimeInSeconds() - EStatBuf.st_mtime) > SECSTILLRETRY))) {
            Status=UpdateAlexInfo(AlexInfoPath, UidStr, REMOVEALEXDIR, 0);
            if ((Status != AWORKING) && (Status != AOK)) {
                Log2("CheckAlexInfo ERROR from UpdateAlexInfo ", ATypeToString(Status));
                Status=AlexStat(AlexInfoPath, &StatBuf);        /* hope we have old one still */
            }
        }
    }

    if ((Status == AFILE) || (Status == AOK)) {
        Status=AOK;                                   
    } else {
        if (Status != AWORKING) {
            Log2("ERROR CheckAlexInfo could not make .alex.info in", DirPath);
        }
    }

    LogT("CheckAlexInfo returning ", Status);
    return(Status);
}


/* only called when we are high enough up in the /alex tree that we are only
 * amongst domains, so there is no chance of making a .alex.dir 
 *  Argument should be a *directory* that is only domains but may have .alex.info XXX
 *  Argument to DirToInfo must be a directory.
 */
int CheckAlexInfoHigh(Path, UidStr)
char *Path, *UidStr;
{
    int Status, Status2;
    char DirPath[MAXPATH], AlexInfoPath[MAXPATH];
    char LastOfPath[100];
    struct stat StatBuf;

    Log2("CheckAlexInfoHigh", Path);

    PathToFileName(Path, LastOfPath);
    if (!streql(LastOfPath, ALEXINFO)) {
        (void) strcpy(DirPath, Path);
    } else {
        PathToDir(Path, DirPath);
    }

    (void) strcpy(AlexInfoPath, DirPath);
    (void) strcat(AlexInfoPath, SLASHALEXINFO);

    Status=AlexStat(AlexInfoPath, &StatBuf);

    LogT("CheckAlexInfoHigh AlexStat returned ", Status);

    if ((Status != AFILE) ||
        ((TimeInSeconds() - StatBuf.st_mtime) > MAXSECSINCONSISTENT) ||
        (StatBuf.st_mtime < MINTIMEOKALEXINFO) ||
        (StatBuf.st_size  < 20 )) {
        Status2=CheckIntermediateDirs(Path, 1);
        if (Status2 != AOK) {
            LogT("CheckAlexInfoHigh could not check intermediate dirs ", Status);
            return(AFAIL);
        }
            Status=DirToInfo(DirPath, (struct HostEntry *) NULL, UidStr);
    } else {
        Status=AOK;
    }


    LogT("CheckAlexInfoHigh returning ", Status);
    return(Status);
}


/*
 *   Input:
 *      Localpathname
 *      Host
 *      Path on Host
 *
 *   Output:  AFAIL, AFILE, ADIR, AHOST, ALINK, NOSUCHFILE, 
 *            AOLD, AWORKING, UNKNOWN, AUPDATE, AERRORMESSAGE
 *
 *   Algorithm:
 *       Check that AlexInfo exists then do a lookup
 */

int RemFileType(LocalPath, UidStr)
char *LocalPath, *UidStr;
{
    int Result;
    struct ParsedDir Current;

    Result=AlexInfoLStat(LocalPath, &Current, UidStr, NORECURSION);  

    if (Result == AOK) {
        Result=SimpleType(Current.Type);                           /*  had problems with updating dirs XXXX */
        if ((Current.Stale) && (Result==AFILE)) { 
            Log2("RemFileType looking at stale file ", LocalPath);
            Result=AUPDATE;
        }
    } 

    LogT("RemFileType done with ", Result);
    return(Result);
}


/* copies at most num parts of a path from from to to 
 *  num==1  then copies / or /foo
 *  num==2  then copies /foo/bar
 *
*int CopyNParts(from, to, num)
*char *from, *to;
*int num;
*{
*    int numcopied;
*   
*    *to= 0; 
*    if (from==NULL) return(AFAIL);
*    if (to==NULL) return(AFAIL);
*    if (*from != '/') return(AFAIL);
* 
*    for (numcopied=0; (numcopied<num) && (*from != 0); ) {
*        *to++ = *from++;    
*        if ((*from == 0) || (*from == '/')) { 
*            numcopied++;
*        }
*    }
*    *to= 0;
*
*    return(AOK);
*}
*/

/*
 *   On an input of  /foo/bar/baz  or /foo/bar/baz/
 *   This returns   3
 *   Note that on / it returns 0
 *
* int CountPathParts(path)
* char *path;
* {
*     int Result;
*     
*     for(Result=0; *path != NULL; path++) {
*        if (*path == '/') {  
*            Result++;
*        }
*     }
*     --path;
* 
*     if (*path == '/') {
*        Result--;
*     }
*     return(Result);
* }
*/


/*  Copies file TmpPath to path.
 *  Returns AOK if it works.
 *
*int CopyFile(TmpPath, path)
*char *TmpPath, *path;
*{
*    FILE *from, *to;
*
*    from=AFOPEN(TmpPath, "r");
*    to=AFOPEN(path, "w");
*    if ((from !=NULL) && (to != NULL)) {
*        if(ffilecopy(from,to) != 0) {
*            error2("Could not copy tmp to:", path);
*            return(AFAIL);
*        }
*    } else {
*       error2("Could not open files for copy to:", path);
*       return(AFAIL);
*    }
*    (void) AFCLOSE(from);
*    (void) AFCLOSE(to);
*    return(AOK);
*}
*/

/* for each component in the name we may need to make a directory.
 *
 * Since we FTPed the file ok we know the intermediate directories do exist
 *    on the remote host.
 *
 *
*int mkdirscopy(TmpPath, LocalPathName)
*char *TmpPath, *LocalPathName;
*{
*        int Result;
*
*        Log("mkdirscopy");
*        Log(LocalPathName);
*
*        Result=CheckIntermediateDirs(LocalPathName, 0);
*        if (Result == AOK) {
*           Result=CopyFile(TmpPath, LocalPathName);
*        }
*        Log("done with mkdirscopy");
*        return(Result);
*}
*/



SleepNms(N)
int N;
{
    unsigned useconds;

    useconds=1000 * N;
    usleep(useconds);
}


/*  Upper management decided that path should be a directory.
 *  Returns AOK if it is when we are done.
 */
int ShouldBeADir(Path, HostPtr, RemotePath, UidStr)
char *Path;
struct HostEntry *HostPtr;
char *RemotePath, *UidStr;
{
    int Status;

    Log2("ShouldBeADir", Path);

    Status=LiteStat(Path);
    if ((Status != ADIR) && (Status != ALINK)) {
         if (Status!=AFAIL) {
             if (unlink(Path)) error2("unlink", Path);
             Log("ShouldBeADir did an unlink");
         }
         Status = mkdir(Path, 0777); 
         if (Status == 0) {
             Log("ShouldBeADir did a mkdir ");
         } else {
             error2("ShouldBeADir could not do a mkdir ", Path);
             return(AFAIL);     
         }
    }

    if (Status == ALINK) {
         Status=AOK;
    } else {
         Status=CheckAlexInfo(Path, HostPtr, RemotePath, UidStr);  
    }
    
    LogT("ShouldBeADir returning ", Status);
    return(Status);
}


/*  We were told in past that this was a HostAlias so it should be a symlink
 */
int ShouldBeAHostAlias(Path, HostPtr)
char *Path; 
struct HostEntry *HostPtr;
{
    int Status, Result, Type;
    char Name[100];
    char AliasPath[100];

    Log2("In ShouldBeAHostAlias", Path);

    Result=AOK;

    Status=LiteStat(Path);
    if (Status != AFAIL) {
        if (RecursiveRm(Path, __FILE__, __LINE__, DOWAIT) != AOK) {
            ToLog(DBERROR, "ShouldBeAHostAlias ERROR BUG %s\n", Path);
        }
    }

    if (HostPtr->SymLink == NULL) {
        Result=AUPDATE;                                                /* need an update */
        PathToHostName(HostPtr->Path, Name);

        Type=CallGetHostByName(Name, AliasPath);         
        if (Type == HOSTALIAS) {
            HostPtr->Type = HOSTALIAS;
            HostPtr->SymLink=strsave(AliasPath);
        } else {
            Log2("ShouldBeAHostAlias ERROR changed types", Path);
        }
    }

    Log2("ShouldBeAHostAlias done with ", ATypeToString(Result));
    return(Result);
}

int TotalFilesFTPed=0;

/*   If not a good file (ie, nothing, or directory, or shadow file)
 *      we unlink it and FTP it.
 *   Returns AOK if we have the file we want.
 */
int ShouldBeAFile(Path, HostPtr, RemotePath, UidStr)
char *Path;
struct HostEntry *HostPtr;
char *RemotePath, *UidStr;
{
    int CStatus, IStatus, Result, InTransitNow, LastNumFtps;
    struct stat CacheStat;
    struct ParsedDir Current;  

    Log("In ShouldBeAFile");
   
    IStatus=AlexInfoLStat(Path, &Current, UidStr, NORECURSION);

    CStatus=AlexStat(Path, &CacheStat);
    InTransitNow = InTransit(Path);
    if ((IStatus == AOK) && (CStatus == AFILE) && (CStatus == SimpleType(Current.Type)) &&
            (CacheStat.st_size == Current.Size) && !InTransitNow) {
                              /* note if !SizeOk AlexInfoLStat stated file */
        Log("ShouldBeAFile   Type and Size are right");
        return(AOK);             
    }


    if (CStatus == ADIR){
        if (RecursiveRm(Path, __FILE__, __LINE__, DONOTWAIT)) {
            error2("ShouldBeAFile 2 errors - had to do a RecursiveRm but could not ",Path);
            return(AFAIL);
        } else {
            Log("ShouldBeAFile error had to do a RecursiveRm (which did work)");
        }
    }

    /* dirtoinfo should remove files that have changed when it updates a directory
     * so something is wrong if we have a old file and can tell.  
     * However, regression.test will make wrongs size files so we have deal.    
     */
    if ((!InTransitNow) && (CStatus == AFILE) && ((TimeInSeconds() - CacheStat.st_mtime) > 60)) {
        ToLog(DBERROR, "ShouldBeAFile error wrong-size/stale file in cache %s\n", Path);
        (void) unlink(Path);                           
    }

    Log("About to use FTP to get a file");

    LastNumFtps = NumFinishedFTPs;
    Result=DoAnFtp(HostPtr, "get", RemotePath, Path, UidStr);

    if (Result == AOK) {
        if (LastNumFtps != NumFinishedFTPs) {
            TotalFilesFTPed++;
        }
        CStatus=AlexStat(Path, &CacheStat);
        if ((CStatus == AFILE) && (CStatus == SimpleType(Current.Type)) && 
                                  (CacheStat.st_size == Current.Size)) {
                                  
            Log("ShouldBeAFile   Type and Size are right");
        } else {
            if (HostPtr->Type == UNIX) {          /* generalize XXXXX */
               Log("ShouldBeAFile   Type and Size look odd - reporting back");
               Result=AUPDATE;
           } else {
               LogN("ShouldBeAFile   Type and Size look odd - keeping quiet", (int) CacheStat.st_size);
           }
        }
    }
 

    LogT("ShouldBeAFile returing", Result);
    return(Result);
}


/*  Really only asking for .alex.dir but we will make .alex.info too
 */
int ProbablyNeedAlexDir(AlexDirPath, HostPtr, RemotePath, UidStr)
char *AlexDirPath;
struct HostEntry *HostPtr;
char *RemotePath, *UidStr;
{
    char RemoteDir[MAXPATH], LocalDir[MAXPATH];
    int Status;

    Log2("ProbablyNeedAlexDir ", AlexDirPath);

    PathToDir(RemotePath, RemoteDir);
    PathToDir(AlexDirPath, LocalDir);

    Status=CheckAlexInfo(LocalDir, HostPtr, RemoteDir, UidStr);

    Log2("ProbablyNeedAlexDir ", ATypeToString(Status));
    return(Status);
}



int NumSlashes(Path)
char *Path;
{
    int Result;

    for(Result=0; (Path != NULL) && (*Path != 0) ; Path++) {
        if (*Path == '/') {
            Result++;
        }
    }
    
    return(Result);
}


/* returns true if there are no upper case letters */
int IsAllLower(s)
char *s;
{
    while (*s != 0) {
        if (isupper(*s)) return(0);
        s++;
    }
    return(1);
}


#define MaxFC 100

struct failtype 
     {
         char Name[100];
         int  Type;
         double Time;
     } GHBNFailureCache[MaxFC];

int  NextFailureCache=0;

int InFailureCache(s)
char *s;
{
    int i;

    for (i=0; i<MaxFC; i++) {
        if (streql(GHBNFailureCache[i].Name, s)) {
            if (GHBNFailureCache[i].Type == AFAIL) {
                return(1);
            } else if (GHBNFailureCache[i].Type == SOFTERROR) {
                if ((TimeInSeconds() - GHBNFailureCache[i].Time) < 5*60.0) {
                    return(1);
                } else {
                    GHBNFailureCache[i].Name[0]=0;        
                }
            }
        }
    }
    return(0);
}


AddToFailureCache(s, type)
char *s;
int type;
{
    NextFailureCache++;
    if (NextFailureCache >= MaxFC) NextFailureCache=0;

    (void) strcpy(GHBNFailureCache[NextFailureCache].Name, s);
    GHBNFailureCache[NextFailureCache].Type=type;
    GHBNFailureCache[NextFailureCache].Time=TimeInSeconds();
}



/*  
 *  Input   foo.bar.edu
 *  Output:
 *        AFAIL   - no matches at all
 *        AOK     - yes, this is a hostname
 *      ADOMAIN   - this is a domain name like cmu.edu
 *    HOSTALIAS   - just an alias 
 *      AWORKING  -
 */
int CallGetHostByName(PosbHostName, AliasPath)
char *PosbHostName;
char *AliasPath;
{
    struct hostent *h;             /* gethostbyname has its own static area */
    char LocPosbHostName[MAXPATH];
    char Name[MAXDNAME];
    char Tmp[MAXPATH];
    int Result;

    Log2("CallGetHostByName   ", PosbHostName);


    if (InFailureCache(PosbHostName)) {
        Log2("CallGetHostByName found string in GHBNFailureCache ", PosbHostName);
        return(AFAIL);
    }

    if (!IsAllLower(PosbHostName)) {             
        return(AFAIL);
    }

    TotalGHBNCalls++;               /* performance stuff */

    (void) strcpy(LocPosbHostName, PosbHostName);
    (void) strcat(LocPosbHostName, ".");                    /* no local aliases */
#ifdef USINGRESOLVER
    _res.options &= ~RES_DEFNAMES;    /* so top level domains work, do not append local domain */
#endif
    h = gethostbyname(LocPosbHostName);
#ifdef USINGRESOLVER
    _res.options |= RES_DEFNAMES;
#endif

    Result=AFAIL;
    if (h != NULL) {
        Log2("CallGetHostByName non null ", h->h_name);
        (void) strcpy(Name, h->h_name);
        ToLower(Name);                          /* all host paths in Alex will be lower */
        if (streql(PosbHostName, Name)) {
            Result=AOK;
        } else {                                         /* this is an alias */
            if (StrLastNCmp(Name, PosbHostName, strlen(PosbHostName)) ==  0) {
                Log2("CallGetHostByName not making symlink to lower alias ", PosbHostName);
                Result=AOK;
            } else {
                Result=HOSTALIAS;                  /* no change to cache - only in .alex.info */
                (void) strcpy(AliasPath, ALEXPATH);
                (void) HostNameToPath(Name, Tmp);
                (void) strcat(AliasPath, Tmp);
                Log2("CallGetHostByName found a HOSTALIAS", AliasPath);
            }
        }
    } else {                                                    /* h==NULL */

        switch (h_errno) { 
            case NO_ADDRESS:                     
                 if (HasStringNoCase(PosbHostName, "apple.com") ||  /* XXX should be EndsString */
                     HasStringNoCase(PosbHostName, "att.com")   ||  /* anything matches in front of this */
                     HasStringNoCase(PosbHostName, "tmc.com")) {   
                     Result=AFAIL;                         /* screwy places */
                 } else {
                     Result=ADOMAIN;       /* probably the resolver would tell us more directly */
                 }                        /*  XXXX seem to claim domain a bit too often XXXXX  */
                 break;
            case TRY_AGAIN:
                Result=SOFTERROR;
                break;
            default:
                Result=AFAIL;
        }
    }

    if ((Result==AFAIL) || (Result==SOFTERROR)) {          /* SOFTERROR will timeout correctly */
        AddToFailureCache(PosbHostName, Result);
    }
    Log2("CallGetHostByName returning ", ATypeToString(Result));
    return(Result);
}



/*  Input is a path that does not match any host or file on a known host
 *
 *  Algorithm:  see if this is a new host
 *
 *  Returns:  NULL or *HostEntry
 *
 */
int SeeIfNewHostInPath(PosbHostPath, UidStr, PtrPtrHostEntry)
char *PosbHostPath;
char *UidStr;
struct HostEntry **PtrPtrHostEntry;
{
    int  Result, Status, NumParts;
    char PosbHostName[MAXPATH], Path[MAXPATH], AliasPath[100];
    struct HostEntry *PtrHostEntry;

    Log2("SeeIfNewHostInPath with ", PosbHostPath);

    PtrHostEntry=NULL;
    *PtrPtrHostEntry=NULL;

    if (CountOccurances(PosbHostPath, '.') > 0) {
        Log2("SeeIfNewHostInPath file name has dots so below host name level", PosbHostPath);
        return(AFAIL);
    }

    NumParts=NumSlashes(PosbHostPath);
    if (NumParts > MAXPARTSINHOSTNAME) {
        Log2("SeeIfNewHostInPath found something too deap so must not be host", PosbHostPath);
        return(AFAIL);
    }

    Status=AFAIL;
    Result=AFAIL;

    /* we assume that CheckHostsTable has been called already and its not there */

    PathToHostName(PosbHostPath, PosbHostName);

    AliasPath[0]=0;
    Status=CallGetHostByName(PosbHostName, AliasPath);
    if ((Status==AOK) || (Status==ADOMAIN) || (Status==HOSTALIAS)) {
        Result=AddNewHost(PosbHostPath, Status, AliasPath, &PtrHostEntry);
        if (Result == AOK) {
            (void) strcpy(Path, CACHEDIRVAR);
            (void) strcat(Path, PosbHostPath);
            Status=UpdateAlexInfo(Path, UidStr, NOREMOVEALEXDIR, 0);  /* Handle will check Path */
            if (Status != AOK) {
                Log2("SeeIfNewHostInPath ERROR from UpdateAlexInfo going on anyway", PosbHostPath);
            }
        }
    } else {
        Result=Status;
    }

    if (Result == AOK) {
        Log2("SeeIfNewHostInPath found ", PtrHostEntry->Path);
        *PtrPtrHostEntry=PtrHostEntry;
    } else {
        LogT("SeeIfNewHostInPath did not find one", Result);
    }

    return(Result);
}


/*
 *   Returns a HostEntry if it can find a host in StartOfMachine or if
 *      all of StartOfMachine is just a domain, otherwise it returns NULL
 */
int LocateHostEntry(StartOfMachine, PartsInHostName, UidStr, PtrPtrHostEntry)
char *StartOfMachine;
int  PartsInHostName;
char *UidStr;
struct HostEntry **PtrPtrHostEntry;
{
    struct HostEntry *PtrHostEntry;
    char Machine[MAXPATH], c;
    int Result, PartsDone, i, Done;

    Log2("LocateHostEntry ", StartOfMachine);

    if (PartsInHostName > 0) {
        PartsDone= -1;
        Done=0;
        for (i=0; !Done;  i++) {
            c=StartOfMachine[i];
            Machine[i]=c;
            if (c== '/') {
                PartsDone++;
            }
            if (PartsDone == PartsInHostName) {
                Done=1;
            }
            if (c == 0) {
                Done=1;
            }
        }
        Machine[i-1]=0;
        Result=CheckHostsTable(Machine, PathEQHost, &PtrHostEntry);
    } else {
        Result=CheckHostsTable(StartOfMachine, PathGEHost, &PtrHostEntry);
    }

    if ((Result != AOK) ||
       ((PtrHostEntry->Type==ADOMAIN) && !streql(StartOfMachine, PtrHostEntry->Path))) { 
         Log("LocateHostEntry could not so calling SeeIfNewHostInPath");
         Result=SeeIfNewHostInPath(StartOfMachine, UidStr, &PtrHostEntry);
    }

    if (Result == AOK) {
         Log2("LocateHostEntry found ", PtrHostEntry->Path);
         *PtrPtrHostEntry = PtrHostEntry;
    } else {
         Log2("LocateHostEntry did not get AOK on ", StartOfMachine);
    }
   
    LogT("LocateHostEntry returning ", Result);
    return(Result);
}


typedef struct KnownUser {
    char *HostName;
    char *UidStr;
    struct KnownUser *Next;
} KnownUser;

struct KnownUser *KnownUsers[MAXUIDNUM];
int NumKnownUsers=0;

int NumDiffKnownUids()
{
    int Result, i;

    Result=0;
    for (i=0; i<MAXUIDNUM; i++) {
        if (KnownUsers[i] != NULL) {
            Result++;
        }
    }
    LogN("NumDiffKnownUids returning ", Result);

    return(Result);
}

char *CheckKnownUser(Uid, HostName)
int Uid;
char *HostName;
{
    struct KnownUser *KU;

    KU = KnownUsers[Uid % MAXUIDNUM];

    while ((KU != NULL) && (!streql(KU->HostName, HostName))) {
        KU = KU->Next;
    }

    if (KU != NULL) {
        return(KU->UidStr);
    } else {
        ToLog(DBALL, "CheckKnownUser did not find %d %s \n", Uid, HostName);
        return(NULL);
    }
}

char *AddKnownUser(Uid, HostName, UidStr)
int Uid;
char *HostName;
char *UidStr;
{
    struct KnownUser *NU;

    if ((NU = (KnownUser *) malloc((unsigned) sizeof(KnownUser))) == NULL) {
            MallocDeath("fh_find 1");
    }
    NU->HostName = strsave(HostName);
    NU->UidStr = strsave(UidStr);

    if (KnownUsers[Uid % MAXUIDNUM] != NULL) {
        NU->Next = KnownUsers[Uid % MAXUIDNUM];
    } else {
        NU->Next = NULL;
    }

    KnownUsers[Uid % MAXUIDNUM]=NU;

    NumKnownUsers++;
    return(NU->UidStr);
}

char *UidToUidStr(Uid, HostName)
int Uid;
char *HostName;
{
    char LowerHostName[MAXPATH];
    struct passwd  *pwinfo;                                 /* associated password structure */
    char *UidStr, TmpUidStr[MAXPATH];

    UidStr = CheckKnownUser(Uid, HostName);
    if (UidStr != NULL) {
        Log2("UidToUidStr is reusing UidStr ", UidStr);
        return(UidStr);
    }

    if (!HasStringNoCase(HostName, LOCALUSERDOMAIN)) {
        (void) sprintf(TmpUidStr, "-User%d", Uid);              /* we don't know him personally */
    } else {
        if ((UidStrTable[Uid] == NULL)) {
            if ((pwinfo = getpwuid(Uid)) != NULL) {         /* get and cache another user name */
                UidStrTable[Uid]=strsave(pwinfo->pw_name);
                (void) sprintf(TmpUidStr,"-%s", UidStrTable[Uid]);
            } else {
                (void) sprintf(TmpUidStr, "-User%d", Uid);      /* should not need this code */
            }
        } else {
            (void) sprintf(TmpUidStr,"-%s", UidStrTable[Uid]);
        }
    }
                                                            /* Have a Login name */

    if ((streql(TmpUidStr, "rfsd")) ||
        (streql(TmpUidStr, "anon")) ||
        (strlen(TmpUidStr) < 2)) {
            Log2("UidToUidStr had TmpUidStr ", TmpUidStr);
            (void) strcpy(TmpUidStr, UidStrTable[ALEXUIDVAR]);
    }


    if (CountOccurances(HostName, '.') > 0) {          
        strcat(TmpUidStr, "@");                                /*  foo@bar.baz */
        (void) strcpy(LowerHostName, HostName);
        ToLower(LowerHostName);
        strcat(TmpUidStr, LowerHostName);
    } else {
        strcat(TmpUidStr, "%");                                /*  foo%bar@server.host.name */
        (void) strcpy(LowerHostName, HostName);
        ToLower(LowerHostName);
        strcat(TmpUidStr, LowerHostName);
        strcat(TmpUidStr, "@");
        strcat(TmpUidStr, SERVERHOSTNAME);
    }

    UidStr=AddKnownUser(Uid, HostName, TmpUidStr);

    Log2("UidToUidStr is using UidStr ", UidStr);
    return(UidStr);
}


/*  Incoming Path may end in something like /.alex.update
 *  So if the Type==AFILE we remove the last part
 */
int NeedToUpdateDirectory(Path, StartOfMachine, PartsInHostName, UidStr, Type)
char *Path;
char *StartOfMachine;
int  PartsInHostName;
char *UidStr;
int Type;
{
    char DirPath[MAXPATH], AlexInfoPath[MAXPATH], RemotePath[MAXPATH];
    struct HostEntry *HostPtr;
    int Result, RemoteType;

    if ((Type == ADIR) || (Type == AHOST)) {
        (void) strcpy(DirPath, Path);
    } else {
        (void) PathToDir(Path, DirPath);               /* take off /.alex.update */
    }

    (void) strcpy(AlexInfoPath, DirPath);
    (void) strcat(AlexInfoPath, SLASHALEXINFO);    /* add /.alex.info        */

    Log2("NeedToUpdateDirectory ", AlexInfoPath); 


    if (PartsInHostName > 0) {
        ToLog(DBRPC, "NeedToUpdateDirectory updating %s directly \n", DirPath);
        Result = LocateHostEntry(StartOfMachine, PartsInHostName, UidStr, &HostPtr);
        if (Result != AOK) {
            return(Result);
        }
        RemoteType = MakeRemotes(DirPath, HostPtr->PathLen, RemotePath, UidStr);
        if ((RemoteType == AOLD) || (RemoteType == AWORKING) || (RemoteType == AFAIL)) {
            return(RemoteType);
        }

        Result= MakeAlexInfo(DirPath, HostPtr, RemotePath, UidStr);

    } else {
        ToLog(DBRPC, "NeedToUpdateDirectory using recursive method on %s \n", DirPath);
                                                  /* at most once every 100 seconds */
        Result=UpdateAlexInfo(AlexInfoPath, UidStr, REMOVEALEXDIR, 100);
        if ((Result != AOK) && (Result != AWORKING)) {
            Log2("NeedToUpdateDirectory ERROR having a hard time trying again", Path); 
            Result=UpdateAlexInfo(AlexInfoPath, UidStr, REMOVEALEXDIR, 100);
        }
    } 

    LogT("NeedToUpdateDirectory returning ", Result);
    return(Result);
}

/****************************************************************************
 *  If a user asks, we will drop all requests from him for a few seconds
 */
static char DropAllUidStr[MAXPATH];
static double DropAllTime;
#define DropAllInterval 30
#define ALEXDROPALL  ".alex.dropall"

int UserWantsDropAll(UidStr)
char *UidStr;
{
    Log2("UserWantsDropAll will fail all from ", UidStr);

    (void) strcpy(DropAllUidStr, UidStr);    
    DropAllTime=TimeInSeconds();

    return(AFAIL);
}

int DropAllStill(UidStr)
char *UidStr;
{
    int Result;

    Result = 0;

    if (streql(DropAllUidStr, UidStr)) {
        if ((TimeInSeconds() - DropAllTime) < DropAllInterval) {
            Result=1;
        } else {
            DropAllUidStr[0]=0;            /* times up */
        }
    }
    return(Result);
}


/*  Sets RemotePath and returns a Type
 */
int MakeRemotes(LocalPathName, HostPtrPathLen, RemotePath, UidStr)
char *LocalPathName;
int HostPtrPathLen;
char *RemotePath;
char *UidStr;
{
    int RemoteType;
                                                         /* copy after machine name */
    if ((CACHEDIRLEN+HostPtrPathLen) < strlen(LocalPathName)) {
        (void) strcpy(RemotePath, &LocalPathName[CACHEDIRLEN + HostPtrPathLen]);
        RemoteType=RemFileType(LocalPathName, UidStr);
    } else {
        RemotePath[0]='/';
        RemotePath[1]=0;
        RemoteType=ADIR;
        Log("I think we are right at edge of host name so using / as RemotePath");
    }

    Return(RemoteType);
}

int FakeUpdateNoPerf=0;
int NumAlexUpdates=0;

PerfStatsAlexUpdate(LocalPathName)
char *LocalPathName;
{
    char LastAlexUpdatePath[MAXPATH];

    if (!FakeUpdateNoPerf && !streql(LastAlexUpdatePath, LocalPathName)) {
        NumAlexUpdates++;
        strcpy(LastAlexUpdatePath, LocalPathName);
    }
}
    

/* End of DropAll stuff                                                     */
/****************************************************************************/

/*  This takes a local pathname and makes sure that it is in the cache.
 *  To do this it needs to figure out the split of hostname/remote-file-name
 *  After doing this we may call:
 *       
 *  CheckAlexInfoHigh   - if we are in the hostname part of the name
 *  ShouldBeADir        - if this should be a directory
 *  ShouldBeAFile       - if this should be a file
 *  UserWantsAnUpdate   - if the user has asked us to update a directory
 *  ShouldBeAHostAlias  - if this should be a host alias
 *  ProbablyNeedAlexDir - if asking about either .alex.info or .alex.dir
 *
 *  Can return many error codes.  Main ones are:
 *        AOK - yes the cache is good - go ahead and use what is there
 *      AFAIL - we had some type of error
 *   AWORKING - there is an FTP in progress but the file is not all here yet
 *    AUPDATE - you are asking about old files - update your info
 *
 *  Future work:
 *  This may call things which end up calling HandleOneFile so we can go recursive.
 *  If we separated out the splitting of the local name this might not be needed.
 *  Such a split function would have to return lots of things though:
 *       SplitPathToRemotes(LocalPath, UidStr, &Type, RemoteHost, RemotePath, &Current, &StatBuf)
 */

int HandleOneFile(LocalPathNameArg, UidStr)
char *LocalPathNameArg, *UidStr;
{
    char LocalPathName[MAXPATH], RemotePath[MAXPATH], HostName[MAXPATH], LocalHostPath[MAXPATH];
    char *HostPath, LocalDirPath[MAXPATH];
    char StartOfMachine[MAXPATH];
    char LastOfPath[MAXPATH];
    int  LocalType, RemoteType, WasADir, Status;
    int  IsPartHostPath, Result, IStatus, HostPtrStatus, PartsInHostName, NeedUpdate;
    struct ParsedDir Current;
    struct HostEntry *HostPtr, *PtrHostEntry;

    Log2("HandleOneFile ", LocalPathNameArg);
    strcpy(LocalPathName, LocalPathNameArg);

    PartsInHostName= -1;

    WasADir=SimplifyPath(LocalPathName);
    if (WasADir == AFAIL) {
        Log2("HandleOneFile ERROR SimplifyPath failed ", LocalPathName);
        return(AFAIL);                     
    }

    if (strncmp(LocalPathName, CACHEDIRVAR, CACHEDIRLEN) != 0) {
        Log2("HandleOneFile did not start with CACHEDIRVAR", LocalPathName);
        return(AFAIL);                     
    }

    (void) strcpy(StartOfMachine, &LocalPathName[CACHEDIRLEN]); /* skip CACHEDIRVAR */

    if (StartOfMachine[0] != '/') {
        Log2("HandleOneFile nothing left after simplification ", LocalPathName);
        Status=CheckAlexInfoHigh(LocalPathName, UidStr);      /* top level directory */
        return(Status);
    }

    (void) PathToFileName(LocalPathName, LastOfPath);
    (void) PathToDir(LocalPathName, LocalDirPath);

    if (streql(LastOfPath, ALEXDROPALL)) {
        Result=UserWantsDropAll(UidStr);
        return(Result);
    }

    NeedUpdate=0;
    if (streql(LastOfPath, ALEXUPDATE)) {
        PerfStatsAlexUpdate(LocalPathName);
        NeedUpdate=1;
    }

    IStatus=AlexInfoLStat(LocalPathName, &Current, UidStr, NORECURSION);
    if ((IStatus == AOK) || (IStatus == NOSUCHFILE)) {
        PartsInHostName = Current.PartsInHostName;                  /* for the .alex.info */
        if ((IStatus == AOK) && (SimpleType(Current.Type) == AFILE) && Current.Stale) {
            NeedUpdate=1;
        }
    }

    if (NeedUpdate) {
        Status=NeedToUpdateDirectory(LocalPathName, StartOfMachine, PartsInHostName, UidStr, AFILE);
        if (streql(LastOfPath, ALEXUPDATE)) {
            return(Status);             /* we were asked about this file */
        } else if (Status == AOK) {
            IStatus=AlexInfoLStat(LocalPathName, &Current, UidStr, NORECURSION); /* go on */
        } else {
            return(Status);             /* usually AWORKING */
        }
    } else {
        if (((IStatus == AOK) && (Current.Type == AERRORMESSAGE)) || HasString(LocalPathName, " ALEX ")) {
            Log2("HandleOneFile ignoring what looks like an error message ", LocalPathName);
            return(AOK);                /* is an ok AERRORMESSAGE */
        }
    }


    LocalType=LiteStat(LocalPathName);

    HostPtr=NULL;
    HostPtrStatus = LocateHostEntry(StartOfMachine, -1, UidStr, &HostPtr);
    if (HostPtrStatus == AWORKING) {
        return(HostPtrStatus);
    }


    IsPartHostPath=PartHostPath(StartOfMachine); 


    if ((HostPtr == NULL) && (!IsPartHostPath)) {
        Log2("HandleOneFile does not find a hostname in this", StartOfMachine);
        if (HasString(StartOfMachine, "ALEX:")) {
            Log2("HandleOneFile removing file that should not be here", LocalPathName);
            if (unlink(LocalPathName)) error2("HandleOneFile unlink", LocalPathName);
            return(NOSUCHFILE);
        }
        return(NOSUCHFILE);
    }

    if (HostPtr != NULL) {
        HostPath=HostPtr->Path;
    } else {
        HostPath="/";
    }

    if (HostPath[0] != '/') {
         ToLog(DBERROR, "HandleOneFile ERROR BUG  HostPath needs a slash %s\n", HostPath);
         return(AFAIL);
    }
                                                          /* if whole path is only a domain */
    if (((HostPtr!=NULL) && (HostPtr->Type==ADOMAIN) && streql(StartOfMachine, HostPath)) || 
                                                  /* or whole path is domain and longer than host */
        (IsPartHostPath &&  (strlen(StartOfMachine) > strlen(HostPath)))) { 
        Log2("HandleOneFile decided that this was only part of a HostName", StartOfMachine);
        Status=CheckAlexInfoHigh(LocalPathName, UidStr);   
        return(Status);
    } 

    if (HostPtr == NULL) {
        Log2("HandleOneFile really does not find a hostname in this", StartOfMachine);
        return(NOSUCHFILE);
    }

    if (HostPtr->Type==ADOMAIN) {
        Log2("HandleOneFile ERROR this is longer than a domain but not one", StartOfMachine);
        Status=CheckAlexInfoHigh(LocalDirPath, UidStr);   
        return(NOSUCHFILE);
    }

    if (HostPtr->Type==HOSTALIAS) {
        if (!streql(StartOfMachine, HostPath)) {
            ToLog(DBERROR, "HandleOneFile ERROR BUG something below a symlink?!?! ", HostPath);
            return(AFAIL);
        } else {
            Result=ShouldBeAHostAlias(LocalPathName, HostPtr); 
            Log2("HandleOneFile found HOSTALIAS ", ATypeToString(Result));
            if (Result == AOK) {
                if ((IStatus != AOK) || (Current.Type != HOSTALIAS)) {
                    Result=AUPDATE;
                }
            }
            Log2("HandleOneFile found HOSTALIAS returning ", ATypeToString(Result));
            return(Result);
        }
    }

    PathToHostName(HostPath, HostName); 

    (void) strcpy(LocalHostPath, CACHEDIRVAR);
    (void) strcat(LocalHostPath, HostPath);

    RemoteType = MakeRemotes(LocalPathName, HostPtr->PathLen, RemotePath, UidStr);
    if ((RemoteType == AOLD) || (RemoteType == AWORKING) || (RemoteType == AUPDATE)) {
        return(RemoteType);
    }


           
    if ((RemoteType == AHOST) ||                  /* if another host not file on this host */
        ((IStatus == AOK) && (Current.PartsInHostName > HostPtr->PartsInHostName))) {
        Status = SeeIfNewHostInPath(StartOfMachine, UidStr, &PtrHostEntry);
        if (Status == AOK) {
            ToLog(DBMAJOR, "HandleOneFile found a new host going to try again %s\n", StartOfMachine);
            return(HandleOneFile(LocalPathName, UidStr));   
        } else if (Status == AWORKING) {                     /* XXXX */
            return(AWORKING);
        }
    }


    Log2("HandleOneFile thinks HostName is:                  ", HostName);
    Log2("HandleOneFile thinks RemotePath is:                ", RemotePath);

    ToLog(8, "LocalType = %s    RemoteType = %s    WasADir = %d\n",
            ATypeToString(LocalType), ATypeToString(RemoteType),  WasADir);

    if ((RemoteType == NOSUCHFILE) && streql(RemotePath, "/")) {
       return(AUPDATE);
    }

    if (streql(LastOfPath, ALEXDIR) || streql(LastOfPath, ALEXINFO)) {
        if (LocalType == AFAIL) {
            Result=ProbablyNeedAlexDir(LocalPathName, HostPtr, RemotePath, UidStr);
            return(Result);     
        } else {
            Log("HandleOneFile thinks this is just a .alex.dir or .alex.info we already have");
            return(AOK);          
        }
    }


    if ((RemoteType==ADIR) || (RemoteType==AHOST))  {
        Result=ShouldBeADir( LocalPathName, HostPtr, RemotePath, UidStr);
    } else if (RemoteType==AFILE) {
        Result=ShouldBeAFile(LocalPathName, HostPtr, RemotePath, UidStr);
    } else if (RemoteType==ALINK) {
        Result=AOK;
    } else if ((RemoteType==AFAIL) && IsPartHostPath) {
        Log2("HandleOneFile in end decided that this is ADOMAIN", StartOfMachine);
        Result=CheckAlexInfoHigh(LocalPathName, UidStr);
    } else if (HasString(StartOfMachine, "ALEX:")) {
        Log2("HandleOneFile removing file that should not be here", LocalPathName);
        if (unlink(LocalPathName)) error2("HandleOneFile unlink", LocalPathName);
        Result=NOSUCHFILE;
    } else if (SeeIfNewHostInPath(StartOfMachine, UidStr, &PtrHostEntry) == AOK) {
        Log2("HandleOneFile found a new host going to try again", StartOfMachine);
        return(HandleOneFile(LocalPathName, UidStr));   
    } else if (LocalType != AFAIL) {
        Log2("HandleOneFile removing file that should not be here", LocalPathName);
        (void) RecursiveRm(LocalPathName, __FILE__, __LINE__, DOWAIT);
        Result=NOSUCHFILE;
    } else {
        Log2("HandleOneFile file is not and should not be there", LocalPathName);
        Result=NOSUCHFILE;
    }
            
    Log2("All done in HandleOneFile", ATypeToString(Result));
    Log("\n\n");
    return(Result);
}




/*  We have reason to belive that the .alex.info for the directory that this
 *  file "path" was in is out of date.   We make a new one.
 *  So on an input of:
 *                       /alex/edu/foo/bar/baz 
 *  We update:
 *                       /alex/edu/foo/bar/.alex.info
 * 
 *  DirToInfo will remove any files that are no longer listed in the .alex.info
 *
 *  Since we go recursive we are careful not to mess with the strings given to us
 */
int UpdateAlexInfo(Path, UidStr, RemoveAlexDir, MinTimeToRedo)
char *Path, *UidStr;
int RemoveAlexDir;
int MinTimeToRedo;
{
    char DirPath[MAXPATH];
    char AlexInfoPath[MAXPATH], AlexDirPath[MAXPATH], OldAlexInfoPath[MAXPATH];
    int handstat;
    int HaveOldAlexInfo, UseOldIfHaveIt;
    int Age;
  
    Log2("\n\n UpdateAlexInfo with", Path);

    PathToDir(Path, DirPath);                    /* was a file we want directory */
    if ((strncmp(DirPath, CACHEDIRVAR, CACHEDIRLEN)) != 0) {
         Log2("ERROR UpdateAlexInfo in something not below ROOTDIR", DirPath);
         return(AFAIL);
    }

 
    (void) strcpy(AlexInfoPath, DirPath);
    (void) strcat(AlexInfoPath, SLASHALEXINFO);

    Age = SecsSinceWrite(AlexInfoPath);
    if ((Age > 0) && (Age < MinTimeToRedo)) {
        return(AOK);
    }

    (void) strcpy(OldAlexInfoPath, AlexInfoPath);
    (void) strcat(OldAlexInfoPath, ".old");
    if (Age < 0) {
         HaveOldAlexInfo=0;
    } else if (rename(AlexInfoPath, OldAlexInfoPath)) {
         error2("UpdateAlexInfo could not rename", AlexInfoPath);
         HaveOldAlexInfo=0;
    } else {
         HaveOldAlexInfo=1;
    }

    (void) strcpy(AlexDirPath, DirPath);
    (void) strcat(AlexDirPath, SLASHALEXDIR);
    if (RemoveAlexDir) {
        if (SecsSinceWrite(AlexDirPath) > 60) {        /* need better way to remove XXXX */
            if (unlink(AlexDirPath)) {
                error2("UpdateAlexInfo ERROR unlink", AlexDirPath);  
            }
        }
    }


    handstat=HandleOneFile(DirPath, UidStr);

    UseOldIfHaveIt=0;
    switch (handstat) {
        case CONNECTING:
        case AWORKING:
            LogT("UpdateAlexInfo found ", handstat);
            UseOldIfHaveIt=1;
            break;

        case AOLD:                             /* could not do it an removed the dir */
        case AUPDATE:                          /* not within spec */
            Log2("UpdateAlexInfo is taking bold recursive step ", DirPath);
            handstat=UpdateAlexInfo(DirPath, UidStr, REMOVEALEXDIR, MinTimeToRedo);     
            break;

        case AOK:
            if (HaveOldAlexInfo) {
                if (unlink(OldAlexInfoPath)) error2("UpdateAlexInfo unlink", OldAlexInfoPath);
            }
            break;

        default:
            UseOldIfHaveIt=1;
            LogT("UpdateAlexInfo ERROR got ", handstat);
            (void) TouchFile(AlexInfoPath);       /* XXXXX will say good for longer than we want */
    }

    if (HaveOldAlexInfo && UseOldIfHaveIt) {
        if (LiteStat(AlexInfoPath) != AFILE) {
            if (rename(OldAlexInfoPath, AlexInfoPath)) {
                error2("UpdateAlexInfo could not rename back", AlexInfoPath);
            }
        }
    }

    LogT("UpdateAlexInfo returning ", handstat);
    return(handstat);
}

/* we just count the number of lines in the file PATH_RMTAB to see 
 * how many people have this server mounted */
int NumMounters()
{
    FILE *rmtab;
    char s[1000];
    int  Result;

    Result=0;
    if ((rmtab = fopen(PATH_RMTAB, "r")) != NULL) {
        while (fgets(s, 1000, rmtab) != NULL) {
            Result++;
        }
        (void) fclose(rmtab);
    }

    return(Result);
}


int    TotalGHBNCalls=0;
int    TotalRTZCalls=0;

/* Increment this when new fields added etc */
#define SendStatsHomeVersion 6

InitStatVars()
{
    TotalFilesNFSed =0;
    TotalFilesFTPed =0;
    TotalBytesNFSed =0;
    TotalBytesFTPed =0;
    TotalFTPTime    =0;
    NumFinishedFTPs =0;
    TotalStaleFiles =0;
    TotalStaleSecs  =0;
    LookupHits      =0;
    LookupMisses    =0;
    LookupDrops     =0;
    ReadHits        =0; 
    ReadMisses      =0; 
    ReadDrops       =0; 
    ReadDirHits     =0; 
    ReadDirMisses   =0; 
    ReadDirDrops    =0; 
    TotalGHBNCalls  =0;  
    TotalRTZCalls   =0;
    NumAlexUpdates  =0;
    InitTime        = TimeInSeconds();
}


extern void SendStatsHome(Type, SrcFile, SrcLine)
int      Type;
char *SrcFile;
int   SrcLine;
{
    char BigString[10000];
    char Tmp[MAXPATH], Tmp2[MAXPATH], *s;
    double Time;
    int CacheTotalBytes, CacheTotalDirs, CacheTotalFiles, CacheTotalUnknown;

    Time=TimeStampLog(DBMAJOR);
    ToLog(DBMAJOR, "SendStatsHome");
    BigString[0]=0;
    sprintf(Tmp, "SendStatsHomeVersion %10d\n", SendStatsHomeVersion); (void) strcat(BigString, Tmp);
    sprintf(Tmp, "TotalFilesNFSed   %10d\n",   TotalFilesNFSed);      (void) strcat(BigString, Tmp);
    sprintf(Tmp, "TotalFilesFTPed   %10d\n",   TotalFilesFTPed);      (void) strcat(BigString, Tmp);

    sprintf(Tmp, "TotalBytesNFSed   %10.0f\n", TotalBytesNFSed);      (void) strcat(BigString, Tmp);
    sprintf(Tmp, "TotalBytesFTPed   %10.0f\n", TotalBytesFTPed);      (void) strcat(BigString, Tmp);

    sprintf(Tmp, "TotalFTPTime      %10.0f\n", TotalFTPTime);         (void) strcat(BigString, Tmp);
    sprintf(Tmp, "NumFinishedFTPs   %10d\n",   NumFinishedFTPs);      (void) strcat(BigString, Tmp);
    sprintf(Tmp, "TotalStaleFiles   %10d\n",   TotalStaleFiles);      (void) strcat(BigString, Tmp);
    sprintf(Tmp, "TotalStaleSecs    %10d\n",   TotalStaleSecs);       (void) strcat(BigString, Tmp);
    sprintf(Tmp, "InitTime          %10.0f\n", InitTime);             (void) strcat(BigString, Tmp);
    sprintf(Tmp, "TimeInSeconds     %10.0f\n", Time);                 (void) strcat(BigString, Tmp);
    sprintf(Tmp, "UpTime            %10.0f\n", Time-InitTime);        (void) strcat(BigString, Tmp);
    sprintf(Tmp, "LookupHits        %10d\n",   LookupHits);           (void) strcat(BigString, Tmp);
    sprintf(Tmp, "LookupMisses      %10d\n",   LookupMisses);         (void) strcat(BigString, Tmp);
    sprintf(Tmp, "LookupDrops       %10d\n",   LookupDrops);          (void) strcat(BigString, Tmp);
    sprintf(Tmp, "ReadHits          %10d\n",   ReadHits);             (void) strcat(BigString, Tmp);
    sprintf(Tmp, "ReadMisses        %10d\n",   ReadMisses);           (void) strcat(BigString, Tmp);
    sprintf(Tmp, "ReadDrops         %10d\n",   ReadDrops);            (void) strcat(BigString, Tmp);
    sprintf(Tmp, "ReadDirHits       %10d\n",   ReadDirHits);          (void) strcat(BigString, Tmp);
    sprintf(Tmp, "ReadDirMisses     %10d\n",   ReadDirMisses);        (void) strcat(BigString, Tmp);
    sprintf(Tmp, "ReadDirDrops      %10d\n",   ReadDirDrops);         (void) strcat(BigString, Tmp);
    sprintf(Tmp, "NumKnownUsers     %10d\n",   NumKnownUsers);        (void) strcat(BigString, Tmp);
    sprintf(Tmp, "NumDiffKnownUids  %10d\n",   NumDiffKnownUids());   (void) strcat(BigString, Tmp);
    sprintf(Tmp, "NumMounters       %10d\n",   NumMounters());        (void) strcat(BigString, Tmp);
    sprintf(Tmp, "OKINCONSISTENCY   %10d\n",   OKINCONSISTENCY);      (void) strcat(BigString, Tmp);
    sprintf(Tmp, "TotalGHBNCalls    %10d\n",   TotalGHBNCalls);       (void) strcat(BigString, Tmp);
    sprintf(Tmp, "TotalRTZCalls     %10d\n",   TotalRTZCalls);        (void) strcat(BigString, Tmp);
    sprintf(Tmp, "NumAlexUpdates    %10d\n",   NumAlexUpdates);       (void) strcat(BigString, Tmp);

    if (StringFromFile(CACHEINFOHEAD, Tmp2) == AOK) {
       s = index(Tmp2, 'B');    /* B as in "Bytes" */
       if ((s != NULL) && 
           (4 == sscanf(s, "Bytes= %d   Dirs= %d  Files= %d  Unknown= %d\n",
                   &CacheTotalBytes, &CacheTotalDirs, &CacheTotalFiles, &CacheTotalUnknown))) {
           sprintf(Tmp, "CacheTotalBytes   %10d\n", CacheTotalBytes); (void) strcat(BigString, Tmp);
           sprintf(Tmp, "CacheTotalDirs    %10d\n", CacheTotalDirs);  (void) strcat(BigString, Tmp);
           sprintf(Tmp, "CacheTotalFiles   %10d\n", CacheTotalFiles); (void) strcat(BigString, Tmp);
           sprintf(Tmp, "CacheTotalUnknown %10d\n", CacheTotalUnknown);(void)strcat(BigString, Tmp);
       }
    }    

    sprintf(Tmp, "\n");                                              (void) strcat(BigString, Tmp);

    PhoneHome(Type, BigString, SrcFile, SrcLine);
    ToLog(DBMAJOR, "SendStatsHome finished");
}

#ifdef  PHONEHOMETESTING 
#define PHONEHOMEPERIOD 60*3                    /* 3 minute test */
#define CHECKONCEEVERY  1
#define CHECKAFTERRDS   1
#else
#define PHONEHOMEPERIOD 60*60*24*1              /* PhoneHome once a day */
#define CHECKONCEEVERY  10
#define CHECKAFTERRDS   10
#endif
PhoneHomeIfBeenTooLong()
{
    static double TimeLastPhoned=0;
    static int CallsSinceLastCheckedTime=0;    /* check time to phonehome after number of calls */
    static int LastRDT=0;                      /* or a bunch of readdirs                        */
    int RDT;

    RDT= ReadDirHits+ReadDirDrops;
    if ((CallsSinceLastCheckedTime > CHECKONCEEVERY) || (RDT > LastRDT+CHECKAFTERRDS)) {
        CallsSinceLastCheckedTime =0;
        LastRDT=RDT;
        if ((TimeInSeconds() - TimeLastPhoned) > PHONEHOMEPERIOD) {
            if (TimeLastPhoned != 0) {
                SendStatsHome(6, __FILE__, __LINE__); /* Every day - not when just start up */
                InitStatVars();                       /* start from scratch for next week */
            }
            TimeLastPhoned=TimeInSeconds();
        }
    } else {
        CallsSinceLastCheckedTime++;
    }
}




extern int CloseAndDie()
{

    ToLog(DBMAJOR, "\n\nCloseAndDie is cleaning up a bit and then going to exit\n");
    (void) TimeStampLog(DBMAJOR);
    flushLog();

    (void) signal(SIGALRM, AlarmIgnore);
    (void) signal(SIGINT,  AlarmIgnore);
    (void) signal(SIGHUP,  AlarmIgnore);
    (void) signal(SIGTERM, AlarmIgnore);
    (void) signal(SIGPIPE, AlarmIgnore);

    (void) SaveHostsTable();
    ToLog(DBMAJOR, "CloseAndDie has saved HostList\n");

    SaveInodeNext();
    SendStatsHome(2, __FILE__, __LINE__);
   
    (void) TimeStampLog(DBMAJOR);
    ToLog(DBMAJOR, "Good bye\n");
    flushLog();

    (void) TimeStampLog(DBMAJOR);
    CloseMostAllFDs();
    flushLog();

    exit(0);
}



/*  We cache the last Number Saved so we don't do unnecessary I/O.
 */
SaveInodeNext()
{
    static int LastSaved= -1;
    char tmp[100];
    int Status;

    if (LastSaved == InodeNext) {
        LogN("SaveInodeNext had already saved ", (int) InodeNext);
    } else {
        sprintf(tmp, "%u", InodeNext);
        Status=StringIntoFile(INODENEXT, tmp);

        if (Status == 0) {
            LogN("SaveInodeNext saved ", (int) InodeNext);
            LastSaved=InodeNext;
        } else {
            ToLog(DBERROR, "SaveInodeNext ERROR could not save %d\n", (int) InodeNext);
            abort();
        }
    }
}


LoadInodeNext()
{
    FILE *File;
    int Status;
    double Time;

    File=AFOPEN(INODENEXT, "r");
    if (File == NULL) {
        ToLog(DBERROR, "LoadInodeNext ERROR could not open InodeNext \n");
        Status= -1;
    } else {
        Status=fscanf(File, "%u", &InodeNext);
        (void) AFCLOSE(File);
    }

    if (Status != 1) {
        ToLog(DBERROR, "LoadInodeNext ERROR could not read in InodeNext \n");
        InodeNext=TimeInSeconds();
    } else {
        ToLog(DBMAJOR, "LoadInodeNext read in %d\n", (int) InodeNext);
        InodeNext += 1000;       /* safety margin just in case we died in middle of something */
        Time=TimeInSeconds();
        if (Time > InodeNext) {
            InodeNext=Time;
        }
    }

    ToLog(DBMAJOR, "LoadInodeNext using %d\n", (int) InodeNext);
}
   

/*  Would be better to test that these files/dirs exist           XXXXXXXXX
 */
int PrintConfigStuff()
{
#ifdef  ALEXVERSION 
    ToLog(DBMAJOR, "ALEXVERSION   %s\n", ALEXVERSION);
#else
    ToLog(DBMAJOR, "ALEXVERSION   %s\n", "Not a release");
#endif

    ToLog(DBMAJOR, "CACHEDIR      %s\n", CACHEDIR);
    ToLog(DBMAJOR, "LOGDIR        %s\n", LOGDIR);
    ToLog(DBMAJOR, "RUNNINGDIR    %s\n", RUNNINGDIR);
    ToLog(DBMAJOR, "HOSTSLIST     %s\n", HOSTSLIST);
    ToLog(DBMAJOR, "UIDSTR        %s\n", UIDSTR);
    ToLog(DBMAJOR, "INODENEXT     %s\n", INODENEXT);
    ToLog(DBMAJOR, "ROOTALEXINFO  %s\n", ROOTALEXINFO);
    ToLog(DBMAJOR, "COMPRESSPATH  %s\n", COMPRESSPATH);
    ToLog(DBMAJOR, "MINFREEKB     %d\n", (int) MINFREEKB);
    
}


extern int AlexInit()
{
    int i, oumask, numask;
    char HomeString[1000];

    InitReadOnlyVariables();    /* needs to be first before any Logs() */

    CheckLogDir();

    SetUidGid();

    if (chdir(RUNNINGDIR)) {                 /* really wanted chroot() as well */
        fprintf(stderr, "ERROR Can not chdir");
        exit(-1);
    }


    numask= 022;                             /* only owner can write */
    oumask = umask(numask);

    ToLog(DBERROR,"Doing AlexInit\n");                               /* opens Log */
    (void) TimeStampLog(DBMAJOR);

    LogN("AlexInit Old umask was ", oumask);
    oumask = umask(numask);
    LogN("AlexInit New umask is ", oumask);

    PrintTypes();


    PrintConfigStuff();


    (void) signal(SIGINT, CloseAndDie);
    (void) signal(SIGHUP, CloseAndDie);
    (void) signal(SIGTERM, CloseAndDie);
    (void) signal(SIGPIPE, AlarmIgnore);

    LoadHostsTable();
    LoadUidStrTable();
    LoadInodeNext();

    ToLog(DBRPC, "My PID is %d\n", getpid());

    (void) strcpy(CMUMACHINE, CACHEDIRVAR);
    (void) strcat(CMUMACHINE, "/edu/cmu");
    LENCMUMACHINE=strlen(CMUMACHINE);

    for (i=0; i< MAXFTPSOPEN; i++) {
        ClearFtpCacheEntry(i);
    }

    for (i=0; i< MaxFC; i++) {
        GHBNFailureCache[i].Name[0]=0;
    }

    for (i=0; i < MAXUIDNUM; i++) {
        KnownUsers[i]=NULL;
    }
    NumKnownUsers=0;

    flushLog();
    sprintf(HomeString, "Started Alex as %s at %f\n", ALEXSRVR, InitTime);
    PhoneHome(1, HomeString, __FILE__, __LINE__);
    flushLog();

    ToLog(DBMAJOR, "AlexInit Done\n");
}



/*  Given a directory it checkes that there is a .alex.info and maybe a .alex.dir
 *  Given a filename it fetches that file it necessary.
 */
extern int CheckFileCache(Path, UidStr)
char *Path, *UidStr;
{
    static int LastResult=AFAIL;
    static char LastPath[MAXPATH];
    static double LastStartTime;
    double StartTime, StopTime, ElapsTime;
    int Status, RetryTime;

    ToLog(DBMAJOR, "\n\n\nCheckFileCache %s %s \n", UidStr,  Path);
    StartTime=TimeStampLog(DBPERF2);

    if (DropAllStill(UidStr)) {
        return(AFAIL);
    }

    if ((LastResult == AOK) && streql(LastPath, Path) && (StartTime-LastStartTime)<30) {
        ToLog(DBPERF, "CheckFileCache had just said this was OK - time 0 %s\n", Path);
        return(LastResult);
    }

    KillOldFtps();

    if ((Path==NULL) || (*Path!='/')) {
        ToLog(DBERROR, "CheckFileCache ERROR do not like Path so returning AFAIL %s\n", Path); /* assume printf handles NULL */
        Status=AFAIL;
    } else {
        Status=HandleOneFile(Path, UidStr);
        LogT("CheckFileCache got back from HandleOneFile ", Status);
        if ((Status==AOLD) || (Status==AUPDATE) || (Status==AFAIL)) {  /* AUPDATE could save one FTP if handled XXX */
            Log2("\n\nCheckFileCache going to UpdateAlexInfo and re-call HandleOneFile", Path);
            if (Status == AFAIL) {
                RetryTime = SECSTILLRETRY;
            } else {
                RetryTime = 0;
            }
            Status=UpdateAlexInfo(Path, UidStr, REMOVEALEXDIR, RetryTime);
            if (Status != AWORKING) {
                Status=HandleOneFile(Path, UidStr);
            }
        } else if (Status == SOFTERROR) {
            Log("\n\n CheckFileCache got a SOFTERROR and is doing one retry ");
            Status=HandleOneFile(Path, UidStr);     /* try one more time */
        }

        (void) TimeStampLog(DBPERF2+1);

    }

    StopTime=TimeStampLog(DBPERF2+1);
    ElapsTime= StopTime-StartTime;

    PhoneHomeIfBeenTooLong();      /* 1 out of 100 times calls gettimeofday */

    if (Status == AWORKING) {
        ToLog(DBPERF, "CheckFileCache: time for this AWORKING was %10.5f\n\n", ElapsTime);
    } else {
        ToLog(DBPERF, "CheckFileCache: time for this finished request was %10.5f  NumFinishedFTPs=%d\n\n", ElapsTime, NumFinishedFTPs);
    }
   
    LogT("CheckFileCache done ", Status);

    flushLog();
    (void) strcpy(LastPath, Path);
    LastResult=Status; 
    LastStartTime=StartTime;
    return(Status);
}




