/* Copyright (c) 1992 Vincent Cate
 * All Rights Reserved.
 *
 * Permission to use and modify this software and its documentation
 * is hereby granted, provided that both the copyright notice and this
 * permission notice appear in all copies of the software, derivative works
 * or modified versions, and any portions thereof, and that both notices
 * appear in supporting documentation.  This software or any derivate works
 * may not be sold or distributed without prior written approval from
 * Vincent Cate.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND VINCENT CATE DISCLAIMS ALL
 * WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS.  IN NO EVENT SHALL
 * VINCENT CATE BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
 * OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE
 * OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE
 * OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Users of this software agree to return to Vincent Cate any improvements
 * or extensions that they make and grant Vincent Cate the rights to
 * redistribute these changes.
 *
 * Francis.Dupont@inria.fr added FTPPASSIVE 3/94
 *
 * Andreas Stolcke fixed up compress option 9/93
 *
 * 1/93 this file completely rewritten by
 *
 *           Juergen Hannken-Illjes (hannken@eis.cs.tu-bs.de)
 *
 * 
 * Exports:
 * 
 * Imports:
 *
 * SetLogBaseName()
 * debugLog 
 * ToLog()
 *
 * 
 */


#include "alexincs.h"
#include "alex.h"
#include "alexftp.h"

#ifndef DONTCOMPRESS
#define DONTCOMPRESS ".Z", ".z", ".gz", ".au", ".snd", ".gif", ".jpg", ".mpg", ".gsm"
#endif

/* FTP child code: the following code will be used after a fork() */

/*
 * Messages from or to FTP process. All messages are terminated by a newline.
 * FTP_CONNECTED is followed by SYSTYPE and ROOTDIRECTORY as in:
 *   'cu /usr/anon'
 * FTP_GET and FTP_DIR are followed by REMOTENAME and LOCALNAME as:
 *  'g /usr/anon/readme /alexhome/alex-cache/readme'
 * All messages other than FTP_OK may be followed by a string describing
 * the problem.
 */


static struct sockaddr_in remAddr;	/* address of remote host */
static struct sockaddr_in LocalAddr;	/* socket name of local host */
static struct sockaddr_in DataAddr;	/* same for data connection */

static FILE *cntrlIn;			/* control connection as FILE("r") */
static FILE *cntrlOut;			/* control connection as FILE("w") */
static FILE *DataIn;			/* data connection as FILE("r") */

static int  connected=0;		/* true, if we are connected */
static int  transferring=0;		/* true, if transfer is in progress */
static char RemoteType;			/* type of remote, SYS_??? */
static int  isBinary=0;			/* true, if transferring binary */

#define MAXREPLY 1024

static char replyString[MAXREPLY];		/* last reply, set by GetReply() */

char *DontCompList[] = {
	DONTCOMPRESS,
	(char *) NULL
};

#ifdef NAMECHANGE			/* change process name after fork() */

static char **Argv = NULL;     /* saved start of argv to set process title */
static char *LastArgv = NULL;  /* end of buffer to set process title */


/*
 * set title of process to '-ftp(???) to host (???)' for ps(1)
 */

static void
ChangeProcessName(dest,state)
     char *dest;
     char *state;
{
  char buf[MAXREPLY], *cp, *cq;

  sprintf(buf,"-ftp(%s) to %s",state,dest);
  if(Argv){
    if(strlen(buf) > LastArgv-*Argv-1)
      buf[LastArgv-*Argv-1] = '\0';
    for(cp = buf, cq = *Argv; *cp; cp++, cq++)
      *cq = *cp;
    while(cq < LastArgv)
      *cq++ = ' ';
  }
}

void
InitNameChangeVars(argc,argv,envp)
     int argc;
     char **argv,**envp;
{
  int i;

  Argv = argv;
  for(i = 0; envp[i]; i++)
    ;
  if(i > 0)
    LastArgv = envp[i-1]+strlen(envp[i-1]);
  else
    LastArgv = argv[argc-1]+strlen(argv[argc-1]);
}
#endif

/*
 * either we must quit or we have timed out
 * terminate gracefully
 */

static int GetReply();
static void PutCommand();
static void ClientAlive();
static void RunDownAndExit();

/*
 * send a reply to parent:
 * we use an extended scheme of FTP
 * 0:   timed out
 * 1,3: failure, if unexpected
 * 2:   success
 * 4:   temporary failure
 * 5:   failure
 * 6:   no such file or directory
 * 7    no such host
 */

#define ClientFailed()	SendReplyFunc(5, __LINE__)
#define ClientRetry()	SendReplyFunc(4, __LINE__)
#define ClientOk()	SendReplyFunc(2, __LINE__)
#define ClientNoSuchHost() SendReplyFunc(7, __LINE__)
#define SendReply(r)	SendReplyFunc((r), __LINE__)

static char replyCodes[8] = {
  FTP_EXITED, FTP_FAILED, FTP_OK, FTP_FAILED, FTP_RETRY, FTP_FAILED, FTP_NOTTHERE, FTP_NOSUCHHOST };

static void
SendReplyFunc(r,line)
     int r,line;
{
  char buf[MAXCOMMAND];

  if(r < 0 || r >= sizeof(replyCodes))
    r = 5;
  if(r == 0){
    (void) strcpy(replyString,"Timed out ");
    (void) strcat(replyString,transferring ? "during transfer" : "during connect");
  }
  sprintf(buf, "%c%s", replyCodes[r], r == 2 ? "" : replyString);
  puts(buf);
  fflush(stdout);
  ToLog(DBFTPCHILD, ">  %s @ line %d of %s\n",buf,line,__FILE__);
  if(r != 2 && r != 6)
    RunDownAndExit(0);
}

static void
RunDownAndExit(reply)
     int reply;
{
  int wasconnected;
  char buf[4];

  wasconnected = connected;
  connected = 0;

  if(wasconnected){
    ClientAlive(QUITTIMEOUT);
    if(transferring){
      sprintf(buf,"%c%c%c",IAC,IP,IAC);
      if(send(fileno(cntrlOut),buf,3,MSG_OOB) == 3)
        PutCommand("%cABOR",DM);
    }
    PutCommand("QUIT");
    (void) GetReply();
    if(reply)
      SendReply(0);
  }
  /* UnlinkLog(); */
  exit(0);
}

/*
 * time out processing, if ClientAlive() isn't called within the next
 * 'secs' seconds, send FTP_EXITED reply and exit
 */

static void
ClientTimeout()
{
  ToLog(DBFTPCHILD, "timed out\n");
  if(!connected) {
    SendReply(0);
  } else {
    RunDownAndExit(1);
  }
  exit(0);
}


static void
ClientAlive(secs)
     int secs;
{
  signal(SIGALRM,ClientTimeout);
  signal(SIGPIPE,ClientTimeout);

  alarm(secs);
}

/*
 * match last reply against set of patterns and set some flags
 * a star inside a pattern matches any number of any characters
 */

#define FLAG_NOANON	0x0001
#define FLAG_NOFILE	0x0002
#define FLAG_STRANGE	0x0004

/*  Note that "User anonymous access denied" can be followed by 
 *  "load too high" and "try again later".
 *
 *  All patterns should be in lower case as reply string is converted
 *  to lower before the comparison.
 *
 *  Only put a "*" in middle, it is as if there were stars at start and end
 *
 *  SOFTERROR ones should come first so first match on a line is all that matters.
 */
static struct pattern{
  char *pattern;
  int   flags;
} replyPattern[] = {
  "too many anonymous",                 FLAG_STRANGE,        /* SOFTERROR */
  "try again later",             	FLAG_STRANGE,        /* SOFTERROR */
  "try again in a few minutes",         FLAG_STRANGE,        /* SOFTERROR */
  "load too high",                      FLAG_STRANGE,        /* SOFTERROR */
  "this system is too busy",            FLAG_STRANGE,        /* SOFTERROR */
  "maximum number of anonyous",         FLAG_STRANGE,        /* SOFTERROR */
  "too many*users",                     FLAG_STRANGE,        /* SOFTERROR */
  "no such file or directory",		FLAG_NOFILE,
  "password required",			FLAG_NOANON,
  "login attempt*rejected",		FLAG_NOANON,
  "need password",			FLAG_NOANON,
  "login information invalid",		FLAG_NOANON,
  "anonymous ftp not permitted", 	FLAG_NOANON,
  "anonymous ftp not allowed", 		FLAG_NOANON,
  "no ftp service here",		FLAG_NOANON,
  "not logged in",			FLAG_NOANON,
  "can't set guest privileges",		FLAG_NOANON,
  "for anonymous ftp, please use",	FLAG_NOANON,
  "please use*for anonymous ftp",       FLAG_NOANON,
  "guest*not permitted",		FLAG_NOANON,         /* Login or login */
  "user*unknown",		        FLAG_NOANON,         /* anonymous or ftp */
  "user*access denied",                 FLAG_NOANON,         /* may be temporary XXXX */
  "login failed",			FLAG_NOANON,         /* may be temporary XXXX */
  "you could at least say goodbye",	FLAG_STRANGE,
  NULL,					0
};

static int CurrentFlags;		/* flags, that have been matched */

/* This used to try to match all patterns and set flags for each, but we really 
 * only need to set the flag for the first matching pattern.  So now we return.
 */
static void
AnalyzeReply()
{
  struct pattern *pp = replyPattern;
  char *cp, *cq, *s1, *s2;
  char replystr[MAXREPLY];
 
  (void) strcpy(replystr, replyString); 
  ToLower(replystr);

  while(pp->pattern){
    cq = pp->pattern;
    cp = replystr;
    while(*cp){
      if(*cp != *cq)
	cp++;
      else{				/* at least this char matches */
	s1 = cp;
	s2 = cq;
	while(*s2 && *s2 != '*' && *s1 == *s2)
	  s1++, s2++;
	if(*s2 == '*'){			/* restart, trying to match remainder */
	  cp = s1;
	  cq = s2+1;
	}else if(*s2 == '\0'){		/* pattern did match, set flags */
	  CurrentFlags |= pp->flags;
	  ToLog(DBFTPCHILD, "-----> flags set to %04x\n", CurrentFlags);
	  return;    /*  was break; */
	}else
	  cp++;
      }
    }
    pp++;
  }
}

/*
 * receive reply from remote host, return first valid reply
 * a reply is valid, if it starts with 3 digits not followed by a hyphen
 *
 * problem is that earlier text can tell us if this is a temporary error 
 * or if we can never log in here
 */

static int
GetReply()
{
  int c, Result, IsStrange;
  char *cp, StrangeText[MAXREPLY];

  IsStrange=0;

  for(;;){
    cp = replyString;
    for(;;){
      c = getc(cntrlIn);
      if(c == IAC)
	switch(c = getc(cntrlIn)){
	case WILL:
	case WONT:
	  c = getc(cntrlIn);
	  fprintf(cntrlOut,"%c%c%c",IAC,DONT,c);
	  fflush(cntrlOut);
	  break;
	case DO:
	case DONT:
	  c = getc(cntrlIn);
	  fprintf(cntrlOut,"%c%c%c",IAC,WONT,c);
	  fflush(cntrlOut);
	  break;
	}
      else if(c == EOF){
        (void) strcpy(replyString,"421 Service not available");
	break;
      }else if(c == '\n'){
	if(cp > replyString && cp[-1] == '\r')
	  cp--;
	*cp = '\0';
	break;
      }else if(cp-replyString < sizeof(replyString)-1)
	*cp++ = c;
    }

    ClientAlive(CLIENTTIMEOUT);
    ToLog(DBFTPCHILD, "<< %s\n",replyString);
    AnalyzeReply();
    if(CurrentFlags & FLAG_STRANGE) {
      if (strlen(replyString) < 10) {
          (void) strcpy(replyString,"400 Server sent strange reply");
      } 
      (void) strcpy(StrangeText, replyString);
      IsStrange=1;
      CurrentFlags &= ~FLAG_STRANGE;          
    }

    if(isdigit(replyString[0]) && isdigit(replyString[1]) &&
    	                    isdigit(replyString[2]) && replyString[3] != '-') {
      Result = replyString[0]-'0';
      if (IsStrange) {
          CurrentFlags &= ~FLAG_NOANON;             /* if strange then is FTP at other times */
          if (!streql(replyString, StrangeText)) {
              (void) strcat(replyString, StrangeText);  /* could be long - but ... */
          }
      }
      return(Result);
    }
  }
}

/*
 * send a command to remote host
 */
/*VARARGS*/
static void
PutCommand(va_alist)
     va_dcl
{
  va_list ap;
  char *fmt,buf[MAXREPLY];

  va_start(ap);
  fmt = va_arg(ap,char *);
  vsprintf(buf,fmt,ap);
  ToLog(DBFTPCHILD, ">> %s\n",buf);
  (void) strcat(buf,"\r\n");
  fputs(buf, cntrlOut);
  fflush(cntrlOut);
}

/*
 * connect to remote host
 */

static void
ConnectTo(HostName)
     char *HostName;
{
  struct servent *sp;
  struct hostent *hp;
  int s,len;

  ToLog(DBFTPCHILD, "ConnectTo starting for %s\n",HostName);

#ifdef PROXY_FTP_HOST
  if((sp = getservbyname("ftp-passthru","tcp")) == NULL)
#else  PROXY_FTP_HOST
  if((sp = getservbyname("ftp","tcp")) == NULL)
#endif PROXY_FTP_HOST
    ClientRetry();

  bzero((char *)&remAddr,sizeof(remAddr));
  remAddr.sin_addr.s_addr = inet_addr(HostName);
  if(remAddr.sin_addr.s_addr != -1){
    remAddr.sin_family = AF_INET;
    hp = NULL;
  }else{
    if((hp = gethostbyname(HostName)) == NULL){
      if(h_errno == TRY_AGAIN)
	ClientRetry();
      else if ((h_errno == HOST_NOT_FOUND) || (h_errno == NO_ADDRESS)) 
	ClientNoSuchHost();
      else 
        ClientFailed();
    }
    remAddr.sin_family = hp->h_addrtype;
    bcopy(hp->h_addr_list[0],(caddr_t)&remAddr.sin_addr,hp->h_length);
  }

  if((s = socket(remAddr.sin_family,SOCK_STREAM,0)) < 0)
    ClientRetry();

  remAddr.sin_port = sp->s_port;
  for(;;){
    ToLog(DBFTPCHILD, "Trying %s ...\n",inet_ntoa(remAddr.sin_addr));
    if(connect(s,(struct sockaddr *)&remAddr,sizeof(remAddr)) >= 0)
      break;
    if(hp && hp->h_addr_list[1]){
      hp->h_addr_list++;
      bcopy(hp->h_addr_list[0],(caddr_t)&remAddr.sin_addr,hp->h_length);
      close(s);
      if((s = socket(remAddr.sin_family,SOCK_STREAM,0)) < 0)
	ClientRetry();
    }else
      ClientRetry();
  }

  ToLog(DBFTPCHILD, "Connected to %s ...\n",inet_ntoa(remAddr.sin_addr));
  len = sizeof(LocalAddr);
  if(getsockname(s,(struct sockaddr *)&LocalAddr,&len) < 0)
    ClientRetry();

  if((cntrlIn = fdopen(s,"r")) == NULL || (cntrlOut = fdopen(s,"w")) == NULL)
    ClientRetry();
  /* setlinebuf(cntrlOut);    not portable to HPUX so we fflush() */
}

BinaryMode()
{
    int r;

    if (isBinary) return;

    PutCommand("TYPE I");
    if((r = GetReply()) != 2)
      SendReply(r);
    isBinary = 1;
}

AsciiMode()
{
    int r;

    if (!isBinary) return;

    PutCommand("TYPE A");
    if((r = GetReply()) != 2)
      SendReply(r);
    isBinary = 0;
}

/*  The result of this is cached in the parent.  There is a global
    variable here in ftpchild of "RemoteType" that holds the type. */
int GetRemoteType()
{
  char buf[MAXPATH+4];
  char *cp,*cq;

  buf[1] = SYS_UNKNOWN;
  buf[2] = 0;

  PutCommand("SYST");
  if(GetReply() == 2 && !strncmp(replyString,"215 UNIX Type: L8",17)) {
    buf[1] = SYS_UNIX;
  } else {
    PutCommand("PWD");
    if(GetReply() != 2){
      PutCommand("XPWD");
      if(GetReply() != 2)
        ClientFailed();
    }

    cp = replyString;
    while(*cp && *cp != '"')
      cp++;
    if(*cp++ == '"'){
      cq = buf+3;
      while(*cp && *cp != '"' && cq-buf < sizeof(buf))
        *cq++ = *cp++;
      if(*cp == '"'){
        *cq++ = '\0';
        buf[2] = ' ';
      }
    }

    if (buf[1] == SYS_UNKNOWN && (buf[3] == '/' || 
               (buf[3] != 0 && buf[4] == ':' && buf[5] == '/')))
      buf[1] = SYS_UNIX;                 /* d:/ is microsoft but rest is Unix */
    else if(buf[1] == SYS_UNKNOWN && (cp = strchr(buf+3,':')) && cp[1] == '[')
      buf[1] = SYS_VMS;
    if (buf[1] == SYS_UNKNOWN || buf[3] == '\0')
      ClientFailed();
  }

  return(buf[1]);
}

/*
 * login to remote host and query host type
 *   type is UNIX,   if it either responds to SYST or PWD starts with '/'
 *   type is VMS,    if PWD contains ':['
 *   type is UNKNOWN otherwise
 *
 * Right now we always use SYST and maybe PWD.  We should use hostPtr->Type    XXXX
 *     I think SYST fails alot
 */

static void
#ifdef PROXY_FTP_HOST
Login(HostName)
  char *HostName;
#else  PROXY_FTP_HOST
Login()
#endif PROXY_FTP_HOST
{
  char buf[MAXPATH+4];
  int r;

  if((r = GetReply()) != 2)
    SendReply(r);
  connected = 1;

#ifdef PROXY_FTP_HOST
  PutCommand("USER anonymous@%s", HostName);
#else  PROXY_FTP_HOST
  PutCommand("USER anonymous");
#endif PROXY_FTP_HOST
  if((r = GetReply()) == 3){
    PutCommand("PASS %s",passWord);
    r = GetReply();
  }

  if(CurrentFlags & FLAG_NOANON){
    sprintf(buf,"%c", FTP_NOANON);
    puts(buf);
    fflush(stdout);
    ToLog(DBFTPCHILD, ">  %s\n",buf);
    RunDownAndExit(0);
  }

  if(r != 2)
    SendReply(r);

  buf[0] = FTP_CONNECTED;            /* connected means RemoteType will be in buf[1] */

  if (RemoteType == SYS_UNKNOWN) {
      RemoteType = GetRemoteType();
  }
  buf[1] = RemoteType;
  buf[2] = '\0';

  if (RemoteType == SYS_UNIX) {
    BinaryMode();
  }

  puts(buf);
  fflush(stdout);
  ToLog(DBFTPCHILD, ">  %s\n",buf);
}

/*
 * we remove trailing . or /.
 * UNIX path is either 'name/name ... name/.' for a directory
 *                  or 'name/name ... name'   for a file
 *
 */

UnixPathRemoveSlashDotOrDot(path)
char *path;
{
    int len;

    len = strlen(path);

    if ((len==1) && (path[0] == '.')) {
        path[0] = 0;
        return;
    }

    if (len<3) return;

    if ((path[len-2]=='/') && (path[len-1]=='.')) {
        path[len-2]=0;
    } 
}

/*
 * replace UNIX type path by VMS type path
 */

static void
UnixPathToVMS(path)
     char *path;
{
  char *cp,*cq,*slash,tmp[MAXPATH];

  cq = tmp;
  cp = path;

  *cq++ = '[';

  while(*cp){
    slash = cq;
    *cq++ = '.';
    while(*cp && *cp != '/')
      *cq++ = *cp++;
    if(*cp == '/')
      cp++;
  }

  *cq = '\0';
  *slash = ']';
  if(!strcmp(slash,"]."))
    (void) strcpy(slash+1,"*.*");

  ToLog(DBFTPCHILD, "Unix to VMS: '%s' to '%s'\n",path,tmp);
  (void) strcpy(path,tmp);
}

/*
 * start the FTP client after fork()
 * connect to remote host, login and enter a loop to process commands
 */

extern void
StartFtpClient(HostName, RemoteTypeArg)
char *HostName;
char  RemoteTypeArg;
{
  char buf[MAXCOMMAND];
  char cmd;
  char LocalName[MAXPATH];
  char RemoteName[MAXPATH];
  char *a,*p;
  int i,c,r,s,len;
  FILE *fp;
  struct sockaddr_in from;
  int RemoteLen, LocalLen;
  int compressed, ThisHostNoCompress;
#ifdef FTPPASSIVE
  char pasv[64];
  int a1,a2,a3,a4,p1,p2;
#endif

#ifdef NAMECHANGE
  ChangeProcessName(HostName,"con");
#endif

  sprintf(buf,"%s/ftp.log.%d", LOGDIR, getpid());
  SetLogBaseName(buf);
  debugLog = NULL;
  TimeStampLog(DBFTPCHILD);

  RemoteType = RemoteTypeArg;
  ToLog(DBFTPCHILD, "HostName = %s   RemoteType = %c \n", HostName,  RemoteType);
  CurrentFlags=0;
 
  for(i = 0; i < NSIG; i++) {
    signal(i,SIG_DFL);
  }
  signal(SIGALRM,ClientTimeout);
  signal(SIGPIPE,ClientTimeout);
  alarm(CLIENTTIMEOUT);

  (void) strcpy(replyString,"Connect failed");
#ifdef PROXY_FTP_HOST
  ConnectTo(PROXY_FTP_HOST);
  Login(HostName);
#else  PROXY_FTP_HOST
  ConnectTo(HostName);
  Login();
#endif PROXY_FTP_HOST
  TimeStampLog(DBFTPCHILD);
  ToLog(DBFTPCHILD, "logged in, awaiting command\n");

#ifdef FTP_TRY_COMPRESS
  ThisHostNoCompress=0;
#else FTP_TRY_COMPRESS
  ThisHostNoCompress=1;
#endif FTP_TRY_COMPRESS


  for(;;){
#ifdef NAMECHANGE
    ChangeProcessName(HostName,"idl");
#endif

    ClientAlive(IDLETIMEOUT);

    if(fgets(buf, MAXCOMMAND, stdin) == NULL) {
      exit(1);
    }

    if(sscanf(buf,"%c%s%s", &cmd, LocalName, RemoteName) != 3)
      ClientRetry();

    ClientAlive(CLIENTTIMEOUT);
    if(cmd == FTP_QUIT){
      ToLog(DBFTPCHILD, "got QUIT command");
      RunDownAndExit(1);
    }

    if(RemoteType == SYS_VMS) {
      UnixPathToVMS(RemoteName);
      if ((cmd == FTP_GET) && HasString(RemoteName, ".gif")) {
         BinaryMode();
      } else {
         AsciiMode();
      }
    } else {
      UnixPathRemoveSlashDotOrDot(RemoteName);  
    }

    ToLog(DBFTPCHILD, "got command: %s %s -> %s\n", cmd == FTP_GET ? "get" : "dir",
				     RemoteName, LocalName);


#ifdef NAMECHANGE
  ChangeProcessName(HostName,cmd == FTP_GET ? "get" : "dir");
#endif

    if((s = socket(AF_INET,SOCK_STREAM,0)) < 0)
      ClientRetry();
#ifdef FTPPASSIVE
    bzero((char *)&DataAddr,sizeof(DataAddr));

    PutCommand("PASV");
    if(GetReply() != 2)
     ClientRetry();

    a = replyString+3;
    p = NULL;
    pasv[0] = '\0';

    while (*a != '\0') {
      if (p == NULL) {
	if (isdigit(*a)) {
	  p = pasv;
	  *p++ = *a++;
	} else
	  a++;
      } else {
	if ((*a == ')') || (*a == '\r'))
	  break;
	else
	  *p++ = *a++;
      }
    }
    if (p != NULL)
      *p = '\0';

    if (sscanf(pasv,"%d,%d,%d,%d,%d,%d",&a1,&a2,&a3,&a4,&p1,&p2) != 6) {
      ToLog(DBFTPCHILD, "bad PASV answer format \"%s\"\n", replyString);
      ClientFailed();
    }

    DataAddr.sin_family = AF_INET;
    DataAddr.sin_addr.s_addr = htonl((a1<<24)|(a2<<16)|(a3<<8)|a4);
    DataAddr.sin_port = htons((p1<<8)|p2);

    /* call connect now ! */

    if (connect(s, (struct sockaddr *)&DataAddr, sizeof(DataAddr)) < 0)
      ClientRetry();
#else
    bcopy((char *)&LocalAddr,(char *)&DataAddr,sizeof(DataAddr));
    DataAddr.sin_port = 0;
    if(bind(s,(struct sockaddr *)&DataAddr,sizeof(DataAddr)) < 0)
      ClientRetry();
    len = sizeof(DataAddr);
    if(getsockname(s,(struct sockaddr *)&DataAddr,&len) < 0)
      ClientRetry();
    if(listen(s,1) < 0)
      ClientRetry();

    a = (char *)&DataAddr.sin_addr;
    p = (char *)&DataAddr.sin_port;

    PutCommand("PORT %d,%d,%d,%d,%d,%d",a[0]&0xff,a[1]&0xff,a[2]&0xff,a[3]&0xff,
        p[0]&0xff,p[1]&0xff);
    if(GetReply() != 2)
      ClientRetry();
#endif

    RemoteLen = strlen(RemoteName);
    LocalLen = strlen(LocalName);
    /*
     * Determined whether we should try getting a compressed file
     *
     * Would be best if we only did this for large files or at least
     * if it failed once to not try again.
     *
     * XXX: Can't have single quotes in LocalName lest 'caus it's going to
     * be interpreted by /bin/sh (popen).
     */
#ifdef FTP_TRY_COMPRESS
    compressed = !ThisHostNoCompress && isBinary && (cmd == FTP_GET)
		&& !strchr(LocalName, '\'')
                && (RemoteLen + 1 < sizeof(RemoteName) - 2)
                && (LocalLen + 1 < sizeof(LocalName) - 2);
#else
    compressed = 0;
#endif
 
                                  /* rule out some formats that aren't worth compressing */
    
    {
        char *p;
        int i,len;
 
        for (i=0; compressed && ((p=DontCompList[i]) != NULL); i++) {
            len = strlen(p);
                         /* if the filename has one of the listed suffixes, don't compress */
            if (RemoteLen > len && strcmp(RemoteName + RemoteLen - len, p)==0) {
                compressed = 0;
            }
        }
    }

    if (compressed)
         strcat (RemoteName, ".Z");

retry:
    PutCommand("%s%s%s",cmd == FTP_GET ? "RETR" : "LIST", RemoteLen>0 ? " " : "", RemoteName);
    if((r = GetReply()) != 1) {
      if (compressed) {
	  /*
	   * No compressed file to be found -- just retry uncompressed
	   */
	  compressed = 0;
          ThisHostNoCompress = 1;
	  RemoteName[RemoteLen] = '\0';
	  goto retry;
      }
      if(cmd == FTP_GET && (CurrentFlags & FLAG_NOFILE) == FLAG_NOFILE){
        CurrentFlags &= ~FLAG_NOFILE;
        SendReply(6);
	close(s);
	continue;
      } else {
        SendReply(r);
      }
    }
  
    transferring = 1;
#ifdef FTPPASSIVE
    if((DataIn = fdopen(s, "r")) == NULL)
      ClientRetry();
#else
    len = sizeof(from);
    if((r = accept(s,(struct sockaddr *)&from,&len)) < 0)
      ClientRetry();
    close(s);
    if((DataIn = fdopen(r, "r")) == NULL)
      ClientRetry();
#endif
   
#ifdef FTP_TRY_COMPRESS
    if (compressed) {
      char command[sizeof(UNCOMPRESS) + 1 + MAXPATH];

      sprintf(command, "exec %s >'%s'", UNCOMPRESS, LocalName);
      if ((fp = popen(command, "w")) == NULL)
        ClientRetry();
    } else {
#else
    {
#endif
      if((fp = fopen(LocalName,"w")) == NULL)
        ClientRetry();
    }
  
    len = 0;
    while((c = getc(DataIn)) != EOF && !ferror(fp)){       /* copy from socket into file */
      if(len++ == BUFSIZ){
	ClientAlive(CLIENTTIMEOUT);           /* too slow a keep alive for very slow directories? */
	len = 0;
      }
      if(!isBinary && c == '\r'){
        if((c = getc(DataIn)) != '\n')        /* when not binary change \r\n to \n */
	  putc('\r', fp);
        if(c != EOF)
          putc(c, fp);
      }else {
        putc(c, fp);
      }
    }
  
    if(ferror(fp))
      ClientRetry();
    fflush(fp);                   /* may be a problem with finishing before shows up in filesystem */
    if (compressed) {
      if (pclose(fp) != 0) {
        ToLog(DBFTPCHILD, "uncompress failed\n");
        ClientRetry();
      }
    } else {
      fclose(fp);
    }
    fclose(DataIn);
  
    SendReply(GetReply());
    transferring = 0;
  }
}



