#include "protos.h"

/*
 * This software is Copyright (C) 1988 by Steven Dorner and the
 * University of Illinois Board of Trustees.  No warranties of any
 * kind are expressed or implied.  No support will be provided.
 * This software may not be redistributed for commercial purposes.
 * You may direct questions to nameserv@uiuc.edu
 */

#ifdef __STDC__
#include <stdarg.h>
#else
#include <varargs.h>
#endif
#include <netdb.h>
#include <sys/param.h>

static ARG *s_args = NULL;
static ARG *LastArg;

extern FILE *Input;		/* mqi.c */
extern FILE *Output;		/* mqi.c */
extern int InputType;		/* mqi.c */
extern char Foreign[80];	/* mqi.c */
extern char CommandText[];	/* language.l */
extern char *CommandSpot;	/* language.l */
extern int Daemon;		/* mqi.c */
extern char *DBState;		/* mqi.c */

#ifdef EMAIL_AUTH
extern struct hostent *TrustHp;	/* qi.c */

#endif

FILE	*TempOutput = 0;
static char *TFile = 0;

int	DiscardIt;
void	(*CommandTable[]) __P((ARG *)) =
{
	DoQuery,
	DoChange,
	DoLogin,
	DoAnswer,
	DoLogout,
	DoFields,
	DoAdd,
	DoDelete,
	DoSet,
	DoQuit,
	DoStatus,
	DoId,
	DoHelp,
	DoAnswer,
	DoInfo,
	DoAnswer,
	(void (*) __P((ARG *))) 0
};

QDIR	User = 0;		/* the currently logged in user */
int	AmHero = 0;		/* is currently logged in user the Hero? */
int	AmOpr = 0;		/* is currently logged in user the Operator? */
char	*UserAlias = 0;		/* the alias of the currently logged in user */
int	UserEnt = 0;		/* entry number of current user */

char	*ClaimUser = 0;		/* the identity the user has claimed, but not
				 * yet verified */
char	*Challenge = 0;		/* challenge string */
int	State = S_IDLE;

/*
 * Add a value to an argument list
 */
void 
AddValue(value, ttype)
	char *value;
	int ttype;
{
	if (!s_args)
	{
		LastArg = s_args = FreshArg();
		LastArg->aFirst = make_str(value);
		LastArg->aType = ttype;
	} else
	{
		if (LastArg->aType == VALUE && ttype & EQUAL)
			LastArg->aType |= ttype;
		else if (LastArg->aType & EQUAL && !(LastArg->aType & VALUE2))
		{
			LastArg->aType |= VALUE2;
			LastArg->aSecond = make_str(value);
		} else
		{
			LastArg->aNext = FreshArg();
			LastArg = LastArg->aNext;
			LastArg->aType = ttype;
			LastArg->aFirst = make_str(value);
		}
	}
}

/*
 * Complain about extraneous junk
 */
void 
Unknown(junk)
	char *junk;
{
	fprintf(Output, "%d:%s:%s\n", LR_NOCMD, junk, "Command not recognized.");
}

/*
 * Execute a command
 */
void 
DoCommand(cmd)
	int cmd;
{
#define MAX_SYSLOG 100
	char	c;
	int	which = cmd == C_CLEAR ? 6 : MAX_SYSLOG;

	c = CommandText[which];
	CommandText[which] = '\0';
	*CommandSpot = '\0';
	IssueMessage(LOG_INFO, "%s", CommandText);
	CommandText[which] = c;
	if (InputType == IT_FILE || InputType == IT_PIPE || OP_VALUE(ECHO_OP))
		PrintCommand(cmd, s_args);	/* echo the cmd */

	if (Daemon && GetState())
	{
		fprintf(Output, "555:Database shut off (%s).\n", DBState);
		exit(0);
	}
	if (TempOutput == NULL && !OpenTempOut())
	{
		fprintf(Output, "%d:couldn't open temp file.\n", LR_INTERNAL);
		IssueMessage(LOG_INFO, "Couldn't open temp file.\n");
		goto done;
	}
	if (State == S_E_PENDING)
	{
		if (DiscardIt || cmd != C_ANSWER && cmd != C_CLEAR && cmd != C_EMAIL)
		{
			DoReply(-LR_NOANSWER, "Expecting answer, clear, or email; login discarded.");
			State = S_IDLE;
			free(ClaimUser);
			ClaimUser = 0;
			free(Challenge);
			Challenge = 0;
		}
	}
	if (DiscardIt)
		DoReply(LR_SYNTAX, "Command not understood.");
	else
		(*CommandTable[cmd - 1]) (s_args);

	DumpOutput();

      done:
	FreeArgs(s_args);
	s_args = NULL;
	DiscardIt = 0;
}

/*
 * Free the argument list
 */
void 
FreeArgs(arg)
	ARG *arg;
{
	ARG	*narg;

	if (arg)
		for (narg = arg->aNext; arg; arg = narg)
		{
			narg = arg->aNext;
			if (arg->aFirst)
				free(arg->aFirst);
			if (arg->aSecond)
				free(arg->aSecond);
			free((char *) arg);
		}
}

/*
 * create a fresh argument structure
 */
ARG	*
FreshArg()
{
	ARG	*arg;

	arg = NEW(ARG);
	bzero((void *) arg, sizeof (ARG));
	return (arg);
}

/*
 * status--give the database status
 */
/*ARGSUSED*/
void 
DoStatus(arg)
	ARG *arg;
{
	char	banner[MAXPATHLEN];
	FILE	*bfp;

	(void)sprintf(banner, "%s.bnr", Database);
	if (bfp = fopen(banner, "r"))
	{
		while (fgets(banner, sizeof (banner), bfp))
		{
			char	*nl = index(banner, '\n');

			if (nl)
				*nl = 0;
			DoReply(LR_PROGRESS, banner);
		}
		(void) fclose(bfp);
	}
	if (ReadOnly)
		DoReply(LR_RONLY, "Database ready, read only (%s).", DBState);
	else
		DoReply(LR_OK, "Database ready.");
}

/*
 * id--this command is a no-op; the client issues it only to put
 * the name of the calling user into the nameserver's logfiles
 */
/*ARGSUSED*/
void 
DoId(arg)
	ARG *arg;
{
	DoReply(LR_OK, "Thanks.");
}

/*
 * quit
 */
/*ARGSUSED*/
void 
DoQuit(arg)
	ARG *arg;
{
	fprintf(Output, "%d:%s\n", LR_OK, "Bye!");
	fflush(Output);
	IssueMessage(LOG_INFO, "Done 0");
	closelog();
	exit(0);
}

/*
 * info
 */
/*ARGSUSED*/
void 
DoInfo(arg)
	ARG *arg;
{
	short	n = 0;

	DoReply(-LR_OK, "%d:maildomain:%s", ++n, MAILDOMAIN);
	DoReply(-LR_OK, "%d:mailfield:alias", ++n);
	DoReply(-LR_OK, "%d:administrator:%s", ++n, ADMIN);
	DoReply(-LR_OK, "%d:passwords:%s", ++n, PASSW);
	DoReply(LR_OK, "Ok.");
}

/*
 * Not implemented
 */
void 
NotImplemented(arg)
	ARG *arg;
{
	DoReply(500, "%s:command not implemented.", arg->aFirst);
}

/*
 * make a reply to a command
 */
/*VARARGS2*/
void 
#ifdef __STDC__
DoReply(int code, char *fmt,...)
#else /* !__STDC__ */
DoReply(code, fmt, va_alist)
	int code;
	char *fmt;
va_dcl
#endif /* __STDC__ */
{
	char	scratchFormat[256];
	va_list	args;

	(void) sprintf(scratchFormat, "%d:%s\n", code, fmt);

#ifdef __STDC__
	va_start(args, fmt);
#else /* !__STDC__ */
	va_start(args);
#endif /* __STDC__ */
	if (TempOutput == NULL && !OpenTempOut())
	{
		fprintf(Output, "%d:couldn't open temp file.\n", LR_INTERNAL);
		IssueMessage(LOG_INFO, "Couldn't open temp file.\n");
		return;
	}
	vfprintf(TempOutput, scratchFormat, args);
	va_end(args);
	{
		char	buf[4096];

#ifdef __STDC__
		va_start(args, fmt);
#else /* !__STDC__ */
		va_start(args);
#endif /* __STDC__ */
		vsprintf(buf, scratchFormat, args);
		va_end(args);
		IssueMessage(LOG_DEBUG, "%s", buf);
	}
}

/*
 * identify user
 */
void 
DoLogin(arg)
	ARG *arg;
{
	char	*me;
	char	*RandomString();

	me = arg->aFirst;
	arg = arg->aNext;	/* skip the command name */
	if (!arg)
		DoReply(LR_SYNTAX, "%s:no name given.", me);
	else if (arg->aType != VALUE)
		DoReply(LR_SYNTAX, "%s:argument invalid.", me);
	else if (arg->aNext)
		DoReply(LR_SYNTAX, "%s:extra arguments.", me);
	else
	{
		Challenge = make_str(RandomString(42));
		DoReply(LR_LOGIN, "%s", Challenge);
		ClaimUser = make_str(arg->aFirst);
		AmHero = 0;
		AmOpr = 0;
		State = S_E_PENDING;
		if (User)
			FreeDir(&User);
	}
}

/*
 * handle the answer to a challenge
 */
#define WAIT_SECS 1
void 
DoAnswer(arg)
	ARG *arg;
{
	char	*me;
	long	xtime;

	me = arg->aFirst;
	arg = arg->aNext;

	if (!ClaimUser)
		DoReply(LR_SYNTAX, "%s:there is no outstanding login.", me);
	else if (!arg)
		DoReply(LR_SYNTAX, "%s:no argument given.", me);
	else if (arg->aType != VALUE)
		DoReply(LR_SYNTAX, "%s:invalid argument type.", me);
	else if (!GonnaRead())
		/* Lock routines give their own errors */ ;
	else
	{
		AmHero = 0;
		AmOpr = 0;
		UserAlias = 0;
		xtime = time((long *) NULL);
		if (!(User = GetAliasDir(ClaimUser)))
		{
			SleepTil(xtime + WAIT_SECS);
			if (ReadOnly)
				DoReply(LR_READONLY, "login not allowed to read-only database.");
			else
				DoReply(LR_ERROR, "Login failed.");
			IssueMessage(LOG_INFO, "login: alias %s does not exist.", ClaimUser);
		} else if (((*me == 'a' || *me == 'A') && !UserMatch(arg->aFirst)) ||
#ifndef PRE_ENCRYPT
			   ((*me == 'c' || *me == 'C') && strcmp(arg->aFirst, PasswordOf(User)))
#else
			   ((*me == 'c' || *me == 'C') &&
			    strncmp(crypt(arg->aFirst, arg->aFirst), PasswordOf(User), 13))
#endif
#ifdef EMAIL_AUTH
			   || ((*me == 'e' || *me == 'E') && !OkByEmail(User, arg->aFirst))
#endif
		    )
		{
			SleepTil(xtime + WAIT_SECS);
			if (*me != 'e' && *me != 'E')
			{
				if (ReadOnly)
					DoReply(LR_READONLY, "login not allowed to read-only database.");
				else
				{
					DoReply(LR_ERROR, "Login failed.");
					IssueMessage(LOG_INFO, "Password incorrect for %s.", ClaimUser);
				}
			}
			FreeDir(&User);
		} else if (ReadOnly)
		{
			DoReply(LR_COULDA_BEEN, "login not allowed to read-only database.");
			FreeDir(&User);
		} else
		{
			char *tpnt = FINDVALUE(User, F_HERO);

			SleepTil(xtime + WAIT_SECS);
			DoReply(LR_OK, "%s:Hi how are you?", ClaimUser);
			if (*tpnt != '\0')
			{
				if (stricmp(tpnt, "opr") == 0 ||
				    stricmp(tpnt, "oper") == 0 ||
				    stricmp(tpnt, "operator") == 0)
					AmOpr = 1;
				else
					AmHero = 1;
			}
			UserAlias = FINDVALUE(User, F_ALIAS);
			UserEnt = CurrentIndex();
			IssueMessage(LOG_INFO, "%s logged in.", UserAlias);
		}
	}
	if (ClaimUser)
	{
		free(ClaimUser);
		ClaimUser = NULL;
	}
	if (Challenge)
	{
		free(Challenge);
		Challenge = NULL;
	}
	State = S_IDLE;
	Unlock();
}

/*
 * sleep til a given time
 */
void 
SleepTil(xtime)
	long xtime;
{
	unsigned span;

	span = xtime - time((long *) 0);
	if (0 < span && span < 10000)
		sleep(span);
}

/*
 * return the dir entry of the requested alias
 */
QDIR 
GetAliasDir(fname)
	char *fname;
{
	ARG	*Alist;
	ARG	*arg;
	long	*entry;
	static QDIR dirp;

	arg = Alist = FreshArg();

	arg->aType = COMMAND;
	arg->aFirst = make_str("query");
	arg->aNext = FreshArg();
	arg = arg->aNext;
	arg->aType = VALUE | EQUAL | VALUE2;
	arg->aFirst = make_str("alias");	/* should be alias */
	arg->aSecond = make_str(fname);
	(void) ValidQuery(Alist, C_QUERY);

	if ((entry = DoLookup(Alist)) != NULL && length(entry) == 1 &&
	    next_ent(*entry))
		getdata(&dirp);
	else
		dirp = NULL;

	if (entry)
		free((char *) entry);
	FreeArgs(Alist);

	return (dirp);
}

/*
 * de-identify the current user
 */
void 
DoLogout(arg)
	ARG *arg;
{
	if (arg->aNext)
		DoReply(LR_SYNTAX, "argument given on logout command.");
	else if (!User)
		DoReply(LR_ERROR, "Not logged in.");
	else
	{
		FreeDir(&User);
		DoReply(LR_OK, "Ok.");
	}
}

/*
 * list fields
 */
void 
DoFields(arg)
	ARG *arg;
{
	if (arg->aNext == NULL)
	{
		ListAllFields();
		DoReply(LR_OK, "Ok.");
	} else if (OkFields(arg->aNext))
	{
		for (arg = arg->aNext; arg; arg = arg->aNext)
			if (arg->aFD)
				ListField(arg->aFD);
		DoReply(LR_OK, "Ok.");
	} else
		DoReply(LR_SYNTAX, "Invalid field request.");
}

/*
 * List a single field
 */
void 
ListField(fd)
	FDESC *fd;
{
	char	scratch[MAX_LEN];
	char	*cp;

	(void) sprintf(scratch, "%d:%s:max %d", fd->fdId, fd->fdName, fd->fdMax);
	cp = scratch + strlen(scratch);
	if (fd->fdIndexed)
		cp += strcpc(cp, " Indexed");
	if (fd->fdLookup)
		cp += strcpc(cp, " Lookup");
	if (fd->fdPublic)
		cp += strcpc(cp, " Public");
	if (fd->fdDefault)
		cp += strcpc(cp, " Default");
	if (fd->fdAlways)
		cp += strcpc(cp, " Always");
	if (fd->fdAny)
		cp += strcpc(cp, " Any");
	if (fd->fdChange)
		cp += strcpc(cp, " Change");
	if (fd->fdSacred)
		cp += strcpc(cp, " Sacred");
	if (fd->fdTurn)
		cp += strcpc(cp, " Turn");
	if (fd->fdEncrypt)
		cp += strcpc(cp, " Encrypt");
	if (fd->fdNoPeople)
		cp += strcpc(cp, " NoPeople");
	*cp = 0;
	DoReply(-LR_OK, scratch);
	strcpy(scratch, fd->fdHelp);
	for (cp = strtok(scratch, "\n"); cp; cp = strtok((char *) NULL, "\n"))
		DoReply(-LR_OK, "%d:%s:%s", fd->fdId, fd->fdName, cp);
}

/*
 * list all fields
 */
void 
ListAllFields()
{
	FDESC **fd;

	for (fd = FieldDescriptors; *fd; fd++)
		ListField(*fd);
}

/*
 * validate arguments for field names
 */
int 
OkFields(arg)
	ARG *arg;
{
	int	bad = 0;
	int	count = 0;
	FDESC *fd;

	for (; arg; arg = arg->aNext)
	{
		count++;
		if (arg->aType != VALUE)
		{
			DoReply(-LR_SYNTAX, "argument %d:is not a field name.", count);
			bad = 1;
		} else if (!(fd = FindFD(arg->aFirst)))
		{
			DoReply(-LR_FIELD, "%s:unknown field.", arg->aFirst);
			bad = 1;
		} else
			arg->aFD = fd;
	}
	return (!bad);
}

/*
 * delete entries
 */
void 
DoDelete(arg)
	ARG *arg;
{
	long	*entries, *entp;
	int	haveError = 0;
	int	count;
	int	done;
	QDIR	dirp;

	if (!AmHero && !User)
	{
		DoReply(LR_NOTLOG, "Must be logged in to delete.");
		return;
	} else if (!UserCanDelete())
	{
		DoReply(LR_ERROR, "You may not delete entries.");
		IssueMessage(LOG_INFO, "%s is not authorized to delete entries.", UserAlias);
		return;
	}
	if (!ValidQuery(arg, C_DELETE))
	{
		DoReply(LR_SYNTAX, "Delete command not understood.");
		return;
	}
	if (!GonnaWrite())
	{
		/* GonnaWrite will issue an error message */ ;
		return;
	}
	if ((entries = DoLookup(arg)) == NULL)
	{
		DoReply(LR_NOMATCH, "No entries matched specifications.");
		Unlock();
		return;
	}
	for (count = 1, done = 0, entp = entries; *entp; count++, entp++)
	{
		if (!next_ent(*entp))
		{
			DoReply(-LR_TEMP, "Internal error.");
			haveError = 1;
			continue;
		}
		getdata(&dirp);
		if (!CanDelete(dirp))
		{
			DoReply(-LR_ERROR, "%d:%s: you may not delete this entry.",
				count, FINDVALUE(dirp, F_ALIAS));
			haveError = 1;
			IssueMessage(LOG_INFO, "%s may not delete %s.",
				     UserAlias, FINDVALUE(dirp, F_ALIAS));
			continue;
		}
		/* delete the index entries */
		MakeLookup(dirp, *entp, unmake_lookup);
		FreeDir(&dirp);

		/* mark it as dead and put it out to pasture */
		SetDeleteMark();
		set_date(1);
		store_ent();
		done++;
	}

	free((char *) entries);
	Unlock();

	if (haveError)
		DoReply(LR_ERROR, "%d errors, %d successes on delete command.",
			count - done, done);
	else
		DoReply(LR_OK, "%d entries deleted.", done);
}

/*
 * open a temp file for output
 */
int 
OpenTempOut()
{
	if (TFile == NULL)
	{
		TFile = make_str(TEMPFILE);
		mktemp(TFile);
	}
	if ((TempOutput = fopen(TFile, "w+")) == NULL)
	{
		free(TFile);
		TFile = NULL;
		return (0);
	}
	unlink(TFile);

	return (1);
}

/*
 * Dump a the stuff in TFile to output
 */
void 
DumpOutput()
{
	int	c;

	rewind(TempOutput);	/* back to the beginning */
	{
		while ((c = getc(TempOutput)) != EOF)
			putc(c, Output);
	}
	fclose(TempOutput);	/* close; already unlinked */
	TempOutput = NULL;
	fflush(Output);
}

/*
 * print the current command
 */
void 
PrintCommand(cmd, arg)
	int cmd;
	ARG *arg;
{
	fprintf(Output, "%d: %s", LR_ECHO, arg->aFirst);
	for (arg = arg->aNext;
	     arg;
	     arg = arg->aNext)
	{
		putc(' ', Output);
		if (arg->aType == RETURN)
			fputs(cmd == C_QUERY ? "return" : "make", Output);
		else
		{
			if (arg->aType & VALUE)
				fputs(arg->aFirst, Output);
#ifdef DO_TILDE
			if (arg->aType & TILD_E)
				putc('~', Output);
			else
#endif
			if (arg->aType & EQUAL)
				putc('=', Output);
			if (arg->aType & VALUE2)
				fputs(arg->aSecond, Output);
		}
	}
	putc('\n', Output);
}

/*
 * was the returned string encrypted with the appropriate password?
 */
int 
UserMatch(string)
	char *string;
{
	char	decrypted[MAXSTR];
	char	*pw = PasswordOf(User);

	if (!*pw)
		return (0);
	crypt_start(pw);
	(void) decrypt(decrypted, string);
	return (!strcmp(decrypted, Challenge));
}

/*
 * generate a random string
 */
char	*
RandomString(byteCount)
	int byteCount;
{
	static char string[MAXSTR];
	char	*cp;
	static int seeded = 0;

	if (!seeded)
	{
		seeded = 1;
		srand((int) time((long *) NULL) ^ getpid());
	}
	for (cp = string; byteCount; cp++, byteCount--)
		*cp = (rand() & 0x3f) + 0x21;

	return (string);
}

/*
 * extract the password form a dir
 */
char	*
PasswordOf(User)
	QDIR User;
{
	int	len;
	char	*password;

	/* find the user's password */
	if (!*(password = FINDVALUE(User, F_PASSWORD)))
	{
#ifdef ID_FALLBACK
		if (*(password = FINDVALUE(User, F_UNIVID)))
		{
			len = strlen(password);
			if (len > 8)
				password += len - 8;
		} else
#endif
			password = "";
	}
	return (password);
}

#ifdef EMAIL_AUTH
/*
 * figure out if a user is ok by his email address
 */
int 
OkByEmail(User, username)
	QDIR User;
	char *username;
{
	char	buf[256];
	char	*email = FINDVALUE(User, F_EMAIL);
	char	*pnt;
	int	result;

	/*
	 * Fix up email field by omitting leading whitespace and
	 * terminating it at the first white space.
	 */
	email = strcpy(malloc(strlen(email) + 1), email);
	while (isspace(*email))
		email++;
	pnt = email;
	while (!isspace(*pnt))
		pnt++;
	*pnt = '\0';
	if (!TrustHp || !*email)
		result = 1;
	else
	{
		(void) sprintf(buf, "%s@%s", username, TrustHp->h_name);
		result = stricmp(email, buf);
	}
	if (result)
	{
		if (ReadOnly)
		{
			DoReply(LR_READONLY, "login not allowed to read-only database.");
			result = 1;
		} else
			DoReply(LR_NOEMAIL, "You can't login that way.");
	}
	return (!result);
}

#endif
