/*
** identd.c                       A TCP/IP link identification protocol server
**
** Version: 1.8.1
**
** This program is in the public domain and may be used freely by anyone
** who wants to. 
**
** Last update: 23 May 1992
**
** Please send bug fixes/bug reports to: Peter Eriksson <pen@lysator.liu.se>
*/

#define VERSION "[in.identd, version 1.8.1]"

#include <stdio.h>
#include <errno.h>
#include <ctype.h>
#include <netdb.h>
#include <nlist.h>
#include <pwd.h>
#include <signal.h>
#include <syslog.h>

#ifdef HAVE_KVM
#include <kvm.h>
#else
#include "kvm.h"
#endif

#include <sys/types.h>
#include <sys/param.h>
#include <sys/socket.h>
#include <sys/socketvar.h>
#include <sys/ioctl.h>
#ifdef BSD43
#include <sys/vmmac.h>
#include <machine/pte.h>
#include <sys/dir.h>
#include <sys/inode.h>
#include <sys/time.h>
#include <sys/proc.h>
#include "paths.h"
#endif

#ifndef HPUX7
#define KERNEL
#endif

#include <sys/file.h>

#ifndef HPUX7
#undef KERNEL
#endif

#include <fcntl.h>

#ifdef ultrix
#include <sys/dir.h>
#endif

#include <sys/user.h>
  
#include <net/if.h>
#include <net/route.h>
#include <netinet/in.h>
#include <netinet/in_pcb.h>
#include <netinet/tcp.h>
#include <netinet/ip_var.h>
#include <netinet/tcp_timer.h>
#include <netinet/tcp_var.h>
#include <arpa/inet.h>

#ifdef MIPS
#include <sysv/sys/var.h>
extern int errno;
#endif


extern void *calloc();


struct nlist nl[] =
{
#ifdef hp9000s800	/* HP9000 Series 800, HP-UX 8.0 */

#define N_FILE  0  
#define N_NFILE 1
#define N_TCB   2

  { "file" },
  { "nfile" },
  { "tcb" },
  
#else
#ifdef HPUX7		/* HP-UX 7.0 */

#define N_FILE  0  
#define N_NFILE 1
#define N_TCB   2

  { "_file" },
  { "_nfile" },
  { "_tcp_cb" },

#else
#ifdef MIPS		/* MIPS RISC/OS */

#define N_FILE 0  
#define N_V    1
#define N_TCB  2

  { "_file" },
  { "_v" },
  { "_tcb" },

#else 			/* SUN or BSD or other systems */
#ifdef BSD43

#define X_PROC          0
#define X_USRPTMA       1
#define X_NPROC         2
#define X_USRPT         3
#define X_TCB		4

  { "_proc" },
  { "_Usrptmap" },
  { "_nproc" },
  { "_usrpt" },
  { "_tcb" },


#else

#define N_FILE 0  
#define N_NFILE 1
#define N_TCB 2

  { "_file" },
  { "_nfile" },
  { "_tcb" },

#endif
#endif
#endif
#endif
  { "" }
};


char *path_unix = NULL;
char *path_kmem = NULL;

kvm_t *kd;

int verbose_flag = 0;
int debug_flag = 0;
int syslog_flag = 0;

#ifdef LOG_DAEMON
int syslog_facility = LOG_DAEMON;
#endif

#ifndef BSD43

struct file *xfile;
int nfile;

#else

int nproc;
struct proc *mproc;
struct pte *Usrptma, *usrpt;
kvm_t swap, mem;

union {
        struct user user;
        char upages[UPAGES][NBPG];
} user;
#endif


#ifdef MIPS
struct var v;
#endif

int lport, fport;
struct inpcb tcb;


/*
** Error handling macros
*/
#define ERROR(fmt) \
    ((syslog_flag ? syslog(LOG_ERR, fmt) : 0), \
     (debug_flag ? (fprintf(stderr, "%d, %d: ERROR: X-DBG", lport, fport), \
		    fprintf(stderr, fmt), perror(""), 0) : \
      (printf("%d, %d: ERROR: UNKNOWN-ERROR\r\n", lport, fport), \
       exit(1), 0)))

#define ERROR1(fmt,v1) \
    ((syslog_flag ? syslog(LOG_ERR, fmt, v1) : 0), \
     (debug_flag ? (fprintf(stderr, "%d, %d: ERROR: X-DBG: ", lport, fport), \
		    fprintf(stderr, fmt, v1), perror(""), 0) : \
      (printf("%d, %d: ERROR: UNKNOWN-ERROR\r\n", lport, fport), \
       exit(1), 0)))

#define ERROR2(fmt,v1,v2) \
    ((syslog_flag ? syslog(LOG_ERR, fmt, v1, v2) : 0), \
     (debug_flag ? (fprintf(stderr, "%d, %d: ERROR: X-DBG: ", lport, fport), \
		    fprintf(stderr, fmt, v1, v2), perror(""), 0) : \
      (printf("%d, %d: ERROR: UNKNOWN-ERROR\r\n", lport, fport), \
       exit(1), 0)))


/*
** The structure passing convention for GCC is incompatible with
** Suns own C compiler, so we define our own inet_ntoa() function.
** (This should only affect GCC version 1 I think, a well, this works
** for version 2 also so why bother.. :-)
*/
#if defined(__GNUC__) && defined(__sparc__)

#ifdef inet_ntoa
#undef inet_ntoa
#endif

char *inet_ntoa(ad)
  struct in_addr ad;
{
  unsigned long int s_ad;
  int a, b, c, d;
  static char addr[20];
  
  s_ad = ad.s_addr;
  d = s_ad % 256;
  s_ad /= 256;
  c = s_ad % 256;
  s_ad /= 256;
  b = s_ad % 256;
  a = s_ad / 256;
  sprintf(addr, "%d.%d.%d.%d", a, b, c, d);
  
  return addr;
}
#endif


/*
** Return the name of the connecting host, or the IP number as a string.
*/
char *gethost(addr)
  struct in_addr *addr;
{
  struct hostent *hp;

  
  hp = gethostbyaddr(addr, sizeof(struct in_addr), AF_INET);
  if (hp)
    return hp->h_name;
  else
    return inet_ntoa(*addr);
}

/*
** Get a piece of kernel memory with error handling.
** Always return 1.
*/
int getbufkd(kdp,addr, buf, len, what)
  struct kd *kdp;
  long addr;
  char *buf;
  int len;
  char *what;
{

  if (kvm_read(kdp, addr, buf, len) < 0)
    ERROR2("getbuf: kvm_read(%08x, %d)", addr, len);
  
  return 1;
}

#define getbuf(a,b,c,d) (getbufkd(kd,a,b,c,d))

/*
** Traverse the inpcb list until a match is found.
** Returns NULL if no match.
*/
struct socket *getlist(pcbp, faddr, fport, laddr, lport)
  struct inpcb *pcbp;
  struct in_addr *faddr;
  int fport;
  struct in_addr *laddr;
  int lport;
{
  struct inpcb *head;
  

  if (!pcbp)
    return NULL;
  
  head = pcbp->inp_prev;
  do 
  {
    if ( pcbp->inp_faddr.s_addr == faddr->s_addr &&
	 pcbp->inp_laddr.s_addr == laddr->s_addr &&
	 pcbp->inp_fport        == fport &&
	 pcbp->inp_lport        == lport )
      return pcbp->inp_socket;

  } while (pcbp->inp_next != head &&
	   getbuf((long) pcbp->inp_next,
		  pcbp,
		  sizeof(struct inpcb),
		  "tcblist"));

  return NULL;
}

#ifdef BSD43	/* BSD */
struct ucred {
	u_short		cr_ruid;
};

struct ucred *getinfo(faddr, fport, laddr, lport)
  struct in_addr faddr;
  int fport;
  struct in_addr laddr;
  int lport;
{
  static struct ucred ucb;
  long addr;
  struct file lfile;
  struct socket *sockp;
  int i;

  /* -------------------- TCP PCB LIST -------------------- */

  getbuf(nl[X_TCB].n_value, &tcb, sizeof(tcb), "tcb");
  tcb.inp_prev = (struct inpcb *) nl[X_TCB].n_value;
  sockp = getlist(&tcb, faddr, fport, laddr, lport);
  if (!sockp)
    return NULL;


  /* --------------------   PROCESSES TABLE    -------------------- */

  Usrptma = (struct pte *)nl[X_USRPTMA].n_value;
  usrpt = (struct pte *) nl[X_USRPT].n_value;
  getbuf(nl[X_NPROC].n_value, &nproc, sizeof(nproc), "nproc");
  getbuf(nl[X_PROC].n_value, &addr, sizeof(addr), "&proc");

  mproc = (struct proc *) calloc(nproc, sizeof(struct proc));
  if (!mproc)
    ERROR2("getinfo: calloc(%d,%d)", nproc, sizeof(struct proc));
  
  getbuf(addr, mproc, sizeof(struct proc)*nproc, "proc[]");

  /* --------------------  SCAN PROCCESS TABLE   ----------------- */

  for (; nproc--; mproc++) {
     if(mproc->p_stat == 0)
	continue;
     if(mproc->p_stat != SZOMB && getu() == 0)
        continue;
   
  /* ---------------------   SCAN FILE TABLE  -------------------- */

     for (i = user.user.u_lastfile; i >= 0; i--) {
        if (user.user.u_ofile[i] == 0)
            continue;     /* no file here */
        getbuf(user.user.u_ofile[i], &lfile, sizeof(lfile), "lfile");
        if (lfile.f_type == DTYPE_SOCKET && 
	    (struct socket *) lfile.f_data == sockp) {
                     ucb.cr_ruid = user.user.u_uid;
                     return &ucb;
         } 		/* if */
     } 			/* for -- ofiles */
  }   			/* for -- proc */
  return NULL;  	/* nothing found */
} 			/* getinfo() */

static getu()
{
	struct pte *pteaddr, apte;
	struct pte arguutl[UPAGES+CLSIZE];
	register int i;
	int ncl;

	if ((mproc->p_flag & SLOAD) == 0) {
		getbufkd(&swap, dtob(mproc->p_swaddr), &user.user, sizeof(struct user), "user");
		return(1);
	}
	pteaddr = &Usrptma[btokmx(mproc->p_p0br) + mproc->p_szpt - 1];
	getbuf(pteaddr, &apte, sizeof(apte), "apte");

	getbufkd(&mem,ctob(apte.pg_pfnum+1)-(UPAGES+CLSIZE)*sizeof(struct pte),
		     arguutl, sizeof(arguutl), "arguutl");
	ncl = (sizeof(struct user) + NBPG*CLSIZE - 1) / (NBPG*CLSIZE);
	while (--ncl >= 0) {
		i = ncl * CLSIZE;
		getbufkd(&mem, ctob(arguutl[CLSIZE+i].pg_pfnum), user.upages[i],
			      CLSIZE*NBPG, "upage");
	}
	return(1);
}

#else		/* not BSD */

/*
** Locate the 'ucred' information for a certain connection.
*/
struct ucred *getinfo(faddr, fport, laddr, lport)
  struct in_addr faddr;
  int fport;
  struct in_addr laddr;
  int lport;
{
  static struct ucred ucb;
  long addr;
  struct socket *sockp;
  int i;

  

  /* -------------------- FILE DESCRIPTOR TABLE -------------------- */
#ifdef MIPS
  getbuf(nl[N_V].n_value, &v, sizeof(v), "v");
  nfile = v.v_file;
  addr = nl[N_FILE].n_value;
#else
  getbuf(nl[N_NFILE].n_value, &nfile, sizeof(nfile), "nfile");
  getbuf(nl[N_FILE].n_value, &addr, sizeof(addr), "&file");
#endif
  
  xfile = (struct file *) calloc(nfile, sizeof(struct file));
  if (!xfile)
    ERROR2("getinfo: calloc(%d,%d)", nfile, sizeof(struct file));
  
  getbuf(addr, xfile, sizeof(struct file)*nfile, "file[]");


  /* -------------------- TCP PCB LIST -------------------- */
  getbuf(nl[N_TCB].n_value, &tcb, sizeof(tcb), "tcb");
  tcb.inp_prev = (struct inpcb *) nl[N_TCB].n_value;
  sockp = getlist(&tcb, faddr, fport, laddr, lport);
  if (!sockp)
    return NULL;


  /*
  ** Locate the file descriptor that has the socket in question
  ** open so that we can get the 'ucred' information
  */
  for (i = 0; i < nfile; i++)
    if (xfile[i].f_count &&
	xfile[i].f_type == DTYPE_SOCKET &&
	(struct socket *) xfile[i].f_data == sockp)
    {
      getbuf(xfile[i].f_cred, &ucb, sizeof(ucb));
      return &ucb;
    }

  return NULL;
}
#endif



int main(argc,argv)
  int argc;
  char *argv[];
{
  int i, len;
  struct sockaddr_in sin;
  struct in_addr laddr, faddr;
  struct ucred *ucp;
  struct passwd *pwp;


  /*
  ** Parse the command line arguments
  */
  for (i = 1; i < argc && argv[i][0] == '-'; i++)
    while (*(++argv[i]))
      switch (*argv[i])
      {
	case 'l':
	  syslog_flag++;
	  break;
	  
        case 'V':
	  puts(VERSION);
	  exit(0);
	  break;

	case 'v':
	  verbose_flag++;
	  break;
	  
        case 'd':
	  debug_flag++;
	  break;
      }

  /*
  ** Path to kernel namelist file specified on command line
  */
  if (i < argc)
    path_unix = argv[i++];

  /*
  ** Path the kernel memory device specified on command line
  */
  if (i < argc)
    path_kmem = argv[i++];


  /*
  ** Get foreign internet address
  */
  len = sizeof(sin);
  if (getpeername(0,&sin,&len) == -1)
  {
    /*
    ** A user has tried to start us from the command line or
    ** the network link died, in which case this message won't
    ** reach to other end anyway, so lets give the poor user some
    ** errors.
    */
    perror("in.identd: getpeername()");
    exit(1);
  }
  faddr = sin.sin_addr;


  /*
  ** Get local internet address
  */
  len = sizeof(sin);
  if (getsockname(0,&sin,&len) == -1)
  {
    /*
    ** We can just die here, because if this fails then the
    ** network has died and we haven't got anyone to return
    ** errors to.
    */
    exit(1);
  }
  laddr = sin.sin_addr;


  /*
  ** Open the connection to the Syslog daemon if requested
  */
  if (syslog_flag)
  {
#ifdef LOG_DAEMON
    openlog("identd", LOG_PID, syslog_facility);
#else
    openlog("identd", LOG_PID);
#endif
    
    syslog(LOG_NOTICE, "Connection from %s", gethost(&faddr));
  }
  

  /*
  ** Open the kernel memory device
  */
  if (!(kd = kvm_open(path_unix, path_kmem, NULL, O_RDONLY, NULL)))
    ERROR("main: kvm_open");
  if ((swap.fd = open(_PATH_SWAP, O_RDONLY, 0)) <0)
    ERROR("main: swap");
  
  if ((mem.fd = open(_PATH_MEM, O_RDONLY, 0)) <0)
    ERROR("main: mem");

  /*
  ** Extract offsets to the needed variables in the kernel
  */
  if (kvm_nlist(kd, nl) != 0)
    ERROR("main: kvm_nlist");


  /*
  ** Get the local/foreign port pair from the luser
  */
  lport = fport = 0;
  if (scanf("%d , %d", &lport, &fport) != 2 ||
      lport < 1 || lport > 65535 || fport < 1 || fport > 65535)
  {
    if (syslog_flag)
      syslog(LOG_INFO, "scanf: invalid-port(s): %d, %d", lport, fport);
    
    printf("%d, %d: ERROR: INVALID-PORT\r\n", lport, fport);
    exit(0);
  }

  /*
  ** Next - get the specific TCP connection and return the
  ** 'ucred' - user credentials - information 
  */
  ucp = getinfo(&faddr, htons(fport), &laddr, htons(lport));
  if (!ucp)
  {
    if (syslog_flag)
      syslog(LOG_DEBUG, "Returned: %d, %d: NO-USER", lport, fport);
    
    printf("%d, %d: ERROR: NO-USER\r\n", lport, fport);
    exit(0);
  }

  /*
  ** Then we should try to get the username
  */
  pwp = getpwuid(ucp->cr_ruid);
  if (!pwp)
  {
    if (syslog_flag)
      syslog(LOG_WARNING, "getpwuid() could not map uid (%d) to name",
	     ucp->cr_ruid);
    
    printf("%d, %d: ERROR: UNKNOWN-ERROR\r\n", lport, fport);
    exit(1);
  }

  /*
  ** Hey! We finally made it!!!
  */
  if (syslog_flag)
    syslog(LOG_DEBUG, "Successful lookup: %d, %d: %s\n",
	   lport, fport, pwp->pw_name);
    
  printf("%d, %d: USERID: UNIX: %s\r\n", lport, fport, pwp->pw_name);
  exit(0);
}
