// BrowserPane.h
//
// Free software created 1 Feb 1992
// by Paul Burchard <burchard@math.utah.edu>.
//
// Provides a scrolling list of strings in NX_LISTMODE; i.e. its entries may be
// multiply selected in all the usual ways.  The strings are all distinct 
// (although displayed abbreviations may not be---see below); entering an
// existing string has no effect.
//
// The stringValue of the BrowserPane is a TAB-separated list of the selected 
// strings.  Selecting entries sends the action to the target.  The selection
// can be adjusted by mouse, via selectEntryAt::, or via methods which change 
// the stringValue (such as takeStringValueFrom:, appendStringValueFrom:).
// Double-clicking selects that single entry, sends the doubleAction to the 
// target, and sends an openFile:: message to the delegate (if any).  Note:
// action methods which set the string value won't send the action to the
// target if the target is the same as the sender (to avoid circularity).
// The separator character (TAB) can be changed to any non-null character.
//
// The strings may be displayed in alphabetized and/or abbreviated form.  The
// default isFirst:second: comparison method uses NXOrderStrings(); the default
// abbreviate: method is to take the final ``path component'' of a file name.
// (Note: abbreviations must not be longer than originals.)  Alphabetization 
// uses abbreviated values.  [Note...currently the setAlphabetized: method 
// isn't retroactive; call it before entering any data.]
//
// If isEditable is set, then the standard NeXT editing operations (and Delete)
// will work (pasteboard interaction uses same format as stringValue).  This
// object accepts first responder status and will allow the -copy: operation
// whether or not it is editable, and displays its first responder status by
// highlighting itself.  The delegate (if any) will be sent a removeFile:ok: 
// message upon Cut or Delete operations, and a prepFile:ok: message on
// Copy.
//
// Entries may be disabled; their text is dimmed and not selectable.  To make 
// new entries disabled by default, set the isDisabledOnEntry flag.
//
// Both the action and the doubleAction are by default takeStringValueFrom:.
//
// If the -initFrame:cellClass: initializer is used, the cell class should be 
// a subclass of TextFieldCell.
//

#import <appkit/appkit.h>

@interface BrowserPane : ScrollView
{
    id stringValue;
    id stringList;
    id delegate;
    id target;
    SEL action;
    SEL doubleAction;
    BOOL isAlphabetized;
    BOOL isAbbreviated;
    BOOL isEditable;
    BOOL isDisabledOnEntry;
    char separator;
}

- initFrame:(const NXRect *)frameRect;
- initFrame:(const NXRect *)frameRect cellClass:factoryId;
- free;

// Target, action, and value.
- setTarget:anObject;
- setAction:(SEL)aSelector;
- setDoubleAction:(SEL)aSelector;
- target;
- (SEL)action;
- (SEL)doubleAction;
- sendAction;
- sendDoubleAction;
- takeStringValueFrom:sender;
- appendStringValueFrom:sender;
- setStringValue:(const char *)aString;
- setStringValue:(const char *)aString append:(BOOL)flag;
- (char)separator;
- setSeparator:(char)c;

// Editing.
- setEditable:(BOOL)yn;
- (BOOL)isEditable;
- cut:sender;
- copy:sender;
- paste:sender;
- delete:sender;
- selectAll:sender;
- keyDown:(NXEvent *)theEvent;
- mouseDown:(NXEvent *)theEvent;

// Responding to changes in environment.
- (BOOL)acceptsFirstResponder;
- becomeFirstResponder;
- resignFirstResponder;
- read:(NXTypedStream *)stream;
- write:(NXTypedStream *)stream;
- resizeSubviews:(const NXSize *)oldSize;

// Delegated file operations.
- (int)openFile:(const char *)fileName ok:(int *)flag;
- (int)prepFile:(const char *)fileName ok:(int *)flag;
- (int)removeFile:(const char *)fileName ok:(int *)flag;

// Managing the list.
- clear;
- clearSelection;
- addEntry:(const char *)aString;
- (int)indexAddEntry:(const char *)aString;
- (const char *)entryAt:(int)row;
- removeEntryAt:(int)row;
- removeSelection;
- (unsigned)count;
- addFiles:(const char *)dir suffix:(const char *)sfx;
- selectEntryAt:(int)row append:(BOOL)yn;
- (BOOL)isEntrySelectedAt:(int)row;
- (int)indexOfEntry:(const char *)aString;
- setEntryEnabled:(BOOL)yn at:(int)row;
- (BOOL)isEntryEnabledAt:(int)row;
- setDisabledOnEntry:(BOOL)yn;
- (BOOL)isDisabledOnEntry;
- setAlphabetized:(BOOL)yn;
- (BOOL)isAlphabetized;
- (BOOL)isFirst:(const char *)firstString second:(const char *)secondString;
- setAbbreviated:(BOOL)yn;
- (BOOL)isAbbreviated;
- abbreviate:(const char *)srcString to:(char *)destString;

// Accessing Cells for enhanced functions.
- cellAt:(int)row;

@end

