;;;
;;;
;;; zenirc-netsplit.el --- Script to hide excessive spew from netsplits.

;;; Copyright (C) 1993, 1994 Ben A. Mesander

;;; Author: Ben A. Mesander <ben@gnu.ai.mit.edu>
;;;         Eric Prestemon <eric@american.edu>
;;; Maintainer: ben@gnu.ai.mit.edu
;;; Keywords: extensions
;;; Created: 1993/03/10

;;; $Id: zenirc-netsplit.el,v 1.4 1994/05/15 16:17:02 ben Exp $

;;; This program is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 2, or (at your option)
;;; any later version.
;;;
;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with this program; if not, you can either send email to this
;;; program's maintainer or write to: The Free Software Foundation,
;;; Inc.; 675 Massachusetts Avenue; Cambridge, MA 02139, USA.

;;; Commentary:

;;; This script attempts to supress excessive signon/offs and mode changes
;;; due to netsplits.

;;; Code:

(defvar zenirc-netsplit-show-server-modechanges nil
  "Set to t to enable display of server mode changes.")

(defvar zenirc-netsplit-debug nil
  "Set to t in order to enable debugging messages in the netsplit code")

;;
;; this is a list of the form
;; (("a.b.c.d e.f.g" (time stamp) first-join "nick1" ... "nickn") ...)
;; where first-join is t or nil, depending on whether or not the first
;; join from that split has been detected or not.
;;
(defvar zenirc-netsplit-list nil)

(defvar zenirc-command-wholeft-hook '(zenirc-netsplit-wholeft)
  "Hook variable for /wholeft command.")

(zenirc-add-hook 'zenirc-startup-hook 'zenirc-netsplit-init)
(zenirc-add-hook 'zenirc-server-JOIN-hook 'zenirc-netsplit-JOIN)
(zenirc-add-hook 'zenirc-server-MODE-hook 'zenirc-netsplit-MODE)
(zenirc-add-hook 'zenirc-server-QUIT-hook 'zenirc-netsplit-QUIT)
(zenirc-add-hook 'zenirc-timer-hook 'zenirc-netsplit-timer)

;;
;; initialize netsplit script
;;
(defun zenirc-netsplit-init ()
  ; netsplit list has to be buffer local
  (make-local-variable 'zenirc-netsplit-list))

;;
;; show/don't show rejoins
;;
(defun zenirc-netsplit-JOIN (proc parsedmsg)
  (let* ((i 0) (elt (nth i zenirc-netsplit-list)) 
	 (nick (zenirc-extract-nick (aref parsedmsg 1))))
    (while elt
      (if (zenirc-member nick (cdr (cdr (cdr elt))))
	  (progn
	    (setq zenirc-run-next-hook nil)
	    (if (not (car (cdr (cdr elt))))
		(progn
		  (zenirc-display-string 
		   proc (format "[info] netjoin: %s\n"
					 (car elt)))
		  (setcar (nthcdr 2 elt) t)))
	    ;; need to remove this nick, perhaps the whole entry here.
	    (if (eq 4 (length elt))
		(setq zenirc-netsplit-list (zenirc-delete 
					    elt 
					    zenirc-netsplit-list))
	      ;; check for first join here
	      (setcar (nthcdr i zenirc-netsplit-list)
		      (zenirc-delete nick elt)))
	    (setq i (length zenirc-netsplit-list)))
	(setq i (1+ i)))
      (setq elt (nth i zenirc-netsplit-list)))))

;;
;; hide mode changes from servers
;;
(defun zenirc-netsplit-MODE (proc parsedmsg)
  ;; regexp matches things with a . in them, and no ! or @ in them.
  (if (string-match "^[^@!]+\\.[^@!]+$" (aref parsedmsg 1))
      (progn
	(if zenirc-netsplit-debug
	    (zenirc-display-string proc "[debug] server mode change\n"))
	(if (not zenirc-netsplit-show-server-modechanges)
	    (setq zenirc-run-next-hook nil)))))

;;
;; detect netsplits
;;
(defun zenirc-netsplit-QUIT (proc parsedmsg)
  (let* ((split (aref parsedmsg 2)) 
	 (nick (zenirc-extract-nick (aref parsedmsg 1)))
	 (ass (assoc split zenirc-netsplit-list)))
    ;; look for arguments of the form host.name.1 host.name.2
    (if (string-match "^[^ ]+\\.[^ ]+ [^ ]+\\.[^ ]+$" split)
	(progn
	  (if ass
	      ;; element for this netsplit exists already
	      (setcdr (nthcdr (1- (length ass)) ass) (cons nick nil))
	    ;; element for this netsplit does not yet exist
	    (setq zenirc-netsplit-list
		  (cons (list split (zenirc-time-to-int 
				     (current-time-string)) nil nick)
			zenirc-netsplit-list))
	    (zenirc-display-string
	     proc (format "[info] netsplit: %s\n" split)))
	  (setq zenirc-run-next-hook nil)))))

;;
;; clean cruft from zenirc-netsplit-list older than 10 minutes
;;
(defun zenirc-netsplit-timer (proc now)
  (let* ((i 0) (elt (nth i zenirc-netsplit-list)))
    (while elt
      (if (zenirc-time< '(0 600)
			(zenirc-time-diff now (car (cdr elt))))
	  (setq zenirc-netsplit-list (delq elt zenirc-netsplit-list))
	(setq i (1+ i)))
      (setq elt (nth i zenirc-netsplit-list)))))

;;
;; show who's gone
;;
(defun zenirc-netsplit-wholeft (proc parsedcmd)
  (let* ((i 0) (elt (nth i zenirc-netsplit-list)))
    (while elt
      (zenirc-display-string
       proc (format "[info] split: %s missing: %s %s\n"
		    (car elt)
		    (mapconcat 'identity 
			       (cdr (cdr (cdr elt))) " ")
		    (if (car (cdr (cdr elt)))
			"(joining)"
		      "")))
      (setq i (1+ i))
      (setq elt (nth i zenirc-netsplit-list)))))

;;; end of zenirc-netsplit.el
