/*
 * rtg  Random Topogology Generator.
 *      Based on Waxman's edge distribution function
 *
 */
#include <stdio.h>
#include <math.h>
#include <values.h>
#include "event.h"

/*
The structure to store the points in.  It uses short as the data type as our grid
is 100 x 100 only and so each co-ordinate of the grid is less than 100.
*/
struct points {
  short x;
  short y;
 } point[N];

short d[N][N]; /* Graph Adjacency matrix (Distance)*/
short mark[N]; /* Scratch memory, to save node specific temporary information*/
/* To store the degree of each node */
short degree[N];
/* To store the list of members and sources */
int sources[N], 
    members[N];
/* To store the big router table.  This will be stored on each node in reality */
int router_table[N][N];

int fflag = 0;
int conflag = 0;
int Conflag = 0;
int pflag = 0; /* print coordinates */
int src_r_memflag = 1;

int count;
int test=0;
long seedval= 1;
float  alpha = 0.2,
	beta = 0.2,
        memp = 0.5,
        srcp = 0.1;
double target = 3.0;
double terror = 0.01;
int offset = 0;
short s_infinity();
double infinity();
int num = 50;	/* Number of nodes to generate for current graph */

/* Structure to store the state during the dijekstra's algorithm */
struct State{
  int pred;
  short length;
  int label;
}state[N];

typedef struct State STATE;
typedef STATE *STATEPTR;

/*
Structure to store the edges in.  This will be used to make the path structure as
well.
*/

struct edge{
  int u;
  int v;
  double weight;
  struct edge* next;
  struct edge* prev;
};

typedef struct edge EDGE;
typedef EDGE *EDGEPTR;
/* Functions defined for the edge structure */

/* Assigns values to an edge pointer... allocates memory */
create_edge(EDGEPTR temp, int s, int t){
   temp->u = s;
   temp->v = t;
   temp->weight = EWEIGHT;
   temp->next = (EDGEPTR) 0;
   temp->prev = (EDGEPTR) 0;
}

/*
*  Deallocates the memory for an edge e
*/
delete_edge(EDGEPTR e)
{
  free(e);
  test --;
}

/*
* This fucntions compares to see whether e1 and e2 are the same edge.  If they
* are, it returns true, else it returns false
*/
int compare_edge (EDGEPTR e1, EDGEPTR e2)
{
   if (((*e1).u==(*e2).u && (*e1).v == (*e2).v) || 
       ((*e1).u == (*e2).v && (*e1).v==(*e2).u))
        return TRUE;
   return FALSE; 

}

/* Structure for a path.  This is made as a link list of edges */

struct path{
  EDGEPTR head;   
  int len;
};
typedef struct path PATH;
typedef PATH *PATHPTR;

/* Functions that will be needed for the path structure */

/*  Assign value to the head of the path */
create_path(PATHPTR p, EDGEPTR e){
  p->head = e;
  p->len = 1;
}

/* Deallocates memmory allocated to a path */
delete_path(PATHPTR p)
{
  EDGEPTR t1, t2;
  p->len = 0;
  t1 = p->head;
  t2 = p->head;
  while(t1->next != (EDGEPTR) 0){
    t2 = t1->next;
    delete_edge(t1);
    t1 = t2;
  }  
  delete_edge(t1); 
  free(p);
  if (0) printf("Done deleting last egde\n");
}

/*  Adds the edge e to the path p.  no memory allocation done */

int add_edgeptr(EDGEPTR e, PATHPTR p)
{
  EDGEPTR cur, temp;
  if (p == (PATHPTR) 0){
    p = (PATHPTR) malloc(sizeof(PATH));
    create_path(p, e);
    return TRUE;
  }
  cur = p->head;
  while (cur->next != (EDGEPTR) 0){
       cur = cur->next;
  }
  cur->next = e;
  p->len++;
  return TRUE;
}


/* Add another edge to path p.  Allocates memory for the edge. */
int add_edge(int u, int v, PATHPTR p)
{
  EDGEPTR cur, temp; 
  cur = p->head;
  while (cur->next != (EDGEPTR) 0){
       cur = cur->next;
  }
  temp = (EDGEPTR) malloc(sizeof(EDGE));
  test ++;
  if (temp == NULL)  panic("No Memory");
  temp->u = u;
  temp->v = v;
  temp->weight = EWEIGHT;
  temp->next = (EDGEPTR) 0;
  temp->prev = cur;
  cur->next = temp;
  p->len++;
  return TRUE;
}

/*
*  This function compares an edge to find whether it is in the path p or not.  if
* it is not, then it returns true. Else it returns false.
*/
int compare_edge_to_path(EDGEPTR e, PATHPTR p)
{
   EDGEPTR cur;
   if (p == (PATHPTR) 0) return TRUE; 
   cur = p->head;
   while (cur != (EDGEPTR) 0){
        if (compare_edge(e, cur))       return FALSE;
      cur = cur->next;
   }
   return TRUE;

}
/*
* This fucntions merges the two paths to make one path.  Used in Dijekstra's 
* Algorithm later on.
*/

int merge_paths(PATHPTR p1, PATHPTR p2)
{
   EDGEPTR cur, temp;
   if (p2 == (PATHPTR) 0)  return TRUE; 
   cur = p2->head;
   while (cur != (EDGEPTR) 0){
     if (compare_edge_to_path(cur, p1)){
       if (0) printf("adding edge");
       temp = cur->next;
       cur->next = (EDGEPTR) 0;
       add_edgeptr(cur, p1);
       cur = temp;
     }
     else {
       temp = cur->next;
       delete_edge(cur);
       cur = temp;
     }
   }
   /* To free the path that was merged with p1.  Do not need to free */
   /* memory for the individual edges as they are part of p1 now */
   free(p2);
   return TRUE;

}

pprint(PATHPTR p){
  EDGEPTR cur;
  cur = p->head;
  while (cur != (EDGEPTR) 0){
    printf("% d - %d ", cur->u, cur->v);
    cur = cur->next;
  }
  printf("\n");
}

/*
        to choose a set of members and sources from the graph
   Intially, we can just choose a proportion of the nodes
   of the graph to be members and a further propotion of
   the nodes to be sources.  We are not assuming that sources
   are neccesarily members.  If the flag is set to 1, the
   sources are forced to be members with a probablity of SRCP.
*/

choose_members(float m, float s, int flag)
{
  int i;
  unsigned long prob;
  double p;
  for (i=1 ; i <= num; i++){
      members[i] = 0;
      sources[i] = 0;
   }
   // members first
   for (i=1; i <= num; i++){
      prob = lrand48();
      p = (double) prob / 0x7fffffff;
        if ( p < m ){
        members[i] = 1;
      }
   }   
   //sources now
   for (i=1; i <= num; i++){
      if (flag == 1){         // sources are members
        if (members[i] == 0) continue;
      }
      prob = lrand48();
      p = (double) prob / 0x7fffffff;
      if (p < s){
         sources[i] = 1;
      }
   }
}

/* To see whether a node is a member or not! */

int is_member(int t){
  if (members[t] == 1 || sources[t] == 1) return TRUE;
  return FALSE;
}

/* To see whether a node is a member of the list or not */

int is_member_of(int t, int* list){
  if (list[t] == 1)  return TRUE;
  return FALSE;
}

/* Functions to calculate the max, min, mem and normal degree of the graph */

float deg(){
   float sum = 0;
   int i;
   for (i=1; i<= num; i++)
       sum += (float) degree[i];
   return sum/num;
}

short Deg(int node){ return degree[node]; }

short mem_Deg(int node, int* list){
   short ret = 0;
   int i;
   for (i=1; i<= num; i++){
     if (i == node) 
       continue;
     if (d[node][i] != s_infinity() && is_member_of(i, list))
       ret ++;
   }
   return ret;
}

short max_deg(){
   short max = 0;
   int i;
   for (i =1; i<=num; i++)
   	if (degree[i] > max) max  = degree[i];
   return max;
}

short min_deg(){
  short min = s_infinity();
  int i;
  for (i=1; i<= num; i++)
   	if (degree[i] < min) min = degree[i];
   return min;
}

/* To find the predecessor of a node using Dijekstra's Algorithm */

int predecessor(int s, int t)
{
  int i, k;
  short min;
  for (i=1; i <= num; i++){
    state[i].pred = -1;
    state[i].length = s_infinity();
    state[i].label = FALSE;
  }
  state[t].length = 0;
  state[t].label = TRUE;
  k = t;
  do{
        /*printf("\nk = %d, s= %d", k, s);*/
   	for (i=1; i<=num; i++)
      	if (d[k][i] != s_infinity() && state[i].label == FALSE){
          if (state[k].length + 1 < state[i].length){
            state[i].pred = k;
            state[i].length = state[k].length + 1;
          }
        }
      k = 1;
      min = s_infinity();
      for (i=1; i <= num; i++)
      	if (state[i].label == FALSE && state[i].length < min){
          min = state[i].length;
          k = i;
        }
        state[k].label = TRUE;
   }while (k!=s);
   return state[s].pred;
}

/* Using the predecessor fucntion, trying to find the shortest path between s and
*  t.  This is again using Dijekstra's Algorithm to do so.
*/

PATHPTR shortest_path(int s, int t){
   int tnode;
   int nnode = predecessor(s, t);
   
   /* find the first node in the path.  */

   EDGEPTR temp;
   PATHPTR p;
   temp = (EDGEPTR) malloc(sizeof(EDGE));
   test ++;
   if (temp == NULL)  panic("Memory");
   p = (PATHPTR) malloc(sizeof(PATH));
   create_edge(temp, s, nnode); 
   if (0) printf("\nMaking %d -- %d\n", s, nnode);
   create_path(p, temp);
   while (nnode != t){
     tnode = nnode;
     nnode = predecessor(nnode, t);
     if (0) printf("Making %d -- %d\n", tnode, nnode);
     add_edge(tnode, nnode, p);
   }
   return p;
}

/*
* This is a routine to find the shortest path tree with all the members in
* it.  This is needed as the CBT algorithm finds the shortest path tree 
* based on the root that is passed to it as a parameter.  Another
* parameter is the list of members.
*/

PATHPTR shortest_path_tree(int root, int* list){
   int tnode;
   int flag = TRUE;  /* To see whether this is the first path in the list */
   PATHPTR ret_path, temp;
   for(tnode = 1; tnode <= num; tnode++){
     if (!is_member_of(tnode, list) || tnode == root) continue;
     if (flag){
       ret_path = shortest_path(root, tnode);
       if (0) printf("Doing Shortest Path %d to %d on %d with %d\n", root, tnode, test, ret_path->len);
       if (0) pprint(ret_path);
       flag = FALSE;
       continue;
     }
     if (0) printf("Doing the path between %d and %d\n", root, tnode);
     temp = shortest_path(root, tnode);
     if (0) printf("Merging with ");
     if (0) pprint(temp);
     merge_paths(ret_path, temp);
   }
   return ret_path;
}


/*
* This is a routine to find the estimated performance factor of a
* particular node.  This routine is written for the CBT algorithm in
* finding out the optimal center.  This does a local computation on the
* given node and figures out the fitness factor for that node. 
*/

int est_cost(int node, int level, int* list){
   int i,j;
   int cost = 0;
   int nbrs[N];
   if (level == 0) return 0;  /* Done with all levels */
   for (i = 1; i<= num; i++)
   	nbrs[i] = 0;
   nbrs[node] = 1;
   for (i = 1; i<= num; i++){
      if (d[node][i] == s_infinity() || nbrs[i] == 1)	continue;
      nbrs[i] = 1;
      for (j = 1; j <= num; j++){
       	if (is_member_of(j, list)){
          	if (router_table[i][j] == node)
            	cost = cost + 1 + est_cost(i, level-1, list);
         }
      }
   }
   return cost;
}

/*
* To choose a random node as the center.  This is going to be used as a
* reference case against other algorithms developed later on to locate the
* center. 
*/

int a_RSST(){
   return random_uniform_int(1, num);
}

/*
* The MST hueristic for the Steiner tree used by Waxman in his paper.
* Implemeneted this just as a first case to see the algorithms in work.
*/

PATHPTR h_MST(int* list){
   int i, j;
   int flag = TRUE;
   PATHPTR fin, p;
   /* fill the graph by taking nodes as the set of members and          */
   /* distances between them as the shortest path distance between them.*/
   for (i = 1; i <= num; i++){
     if (!is_member_of(i, list)) continue;
     for (j = i+1; j <= num; j++){
       if (!is_member(j)) continue;
       p = shortest_path(i,j);
       if (flag){
         fin = p;
         flag = FALSE;
       }
       else
         merge_paths(fin, p);
     }
   }
   return fin;
}

/*
*  This is an optimal algorithm to for CBT.  It finds the root as the node
*  which will form the tree with the minimum cost.  It tries all nodes as
*  possible choices of centers and thus it can be computationally very
*  expensive.  It is just used as a reference.  This returns the center
*  with the least cost tree formed of the nodes in the list.
*/

int a_OCBT(int* list){
   PATHPTR temp;
   PATHPTR best_so_far;
   int tnode = 1, best_root, best_length, temp_length;
   best_so_far = shortest_path_tree(tnode, &list[0]);
   best_length = best_so_far->len;
   best_root = 1;
   for (tnode = 2; tnode <= num; tnode++){
     temp = shortest_path_tree(tnode, &list[0]);
     temp_length = temp->len;
     if (0) printf("Comparing %d: %d with %d: %d\n", best_root, best_length, tnode, temp_length);
     if (temp_length < best_length){
         best_length = temp_length;
         best_root = tnode;
     }
     delete_path(temp);
   }
   delete_path(best_so_far);
   return best_root;
}

/*
*  The minimum shortest path hueristic.  This takes the node which has the
*  higest degree among members and choses that as a next center.
*/

int h_SPT(int level, int* list){
   int i;
   int cen = a_RSST();
   int best_cost = est_cost(cen, level, list);
   int best_degree = Deg(cen);
   for (i = 1; i <= num; i++){
    	if (is_member_of(i, list) && Deg(i) >= best_degree){
      	  if (est_cost(i, level, list) > best_cost){
            best_cost =  est_cost(i, level, list);
            cen = i;
          }
        }
   }
   return cen;
}

/*
*  This is a hueristic that we have developed.  It choses a next center, 
*  given this one, based on the (member) degree of the neighbours of the
*  current center.  It then finds the neighbour with the best fitness
*  factor and chooses that as the next center.
*/


int choose_next_center(int cur, int criterion, int level, int* list){
   int i, dis, next_center=cur;
   if (criterion == 0){
     dis = mem_Deg(next_center, list);
     for (i = 1; i <= num; i++){
       if (d[cur][i] != s_infinity()){   // a neighbour of the current center.
         if (mem_Deg(i, list) > dis && est_cost(i, level, list) > 
                          est_cost(next_center, level, list)){
   	   next_center = i;
           dis = mem_Deg(next_center, list);
   	 }
       }
     }
   }
   else if (criterion == 1){
     dis = Deg(next_center);
     for (i = 1; i <= num; i++){
       if (d[cur][i] != s_infinity()){   // a neighbour of the current center.
         if (Deg(i) > dis && est_cost(i, level, list) >
                est_cost(next_center, level, list)){
   	   next_center = i;
           dis = Deg(next_center);
   	 }
       }
     }
   }
   return next_center;
}

/* Function declaration to return the average degree of the graoh */  
float nd();  
void printadjacency() 
	{
	int i,j;
	for (i=1;i<=num;i++){
		for(j=1;j<=num;j++){
			if(d[i][j]!=s_infinity()) 
			 {printf("%d",j);
			printf(" ");
			printf("%d",d[i][j]);
			printf(" ");
			}
		}
		printf(".\n");
	}
	printf("distance\n");

	/* Above prints out the adjacency matrix, below prints
	   out the distances */
/*	for (i=1; i<=num;i++){
		for(j=1;j<=num;j++){
			if(d[i][j]!=s_infinity()) 
			 {printf("%d",d[i][j]);
			 printf(" ");}
		}
		printf("\n"); 
	}*/
}
main(argc, argv)
  int argc;
  char *argv[];
{
  extern char *optarg;
  char input;
  PATHPTR p;
  EVENTPTR event;
  char *optstring = "s:n:a:b:cf:h:p:Ct:"; 
		/* -s <seed> 16bit value for srand48() */
		/* -n <number> of nodes in the network */
		/* -p  print list of point coordinates */
		/* -a  ALPHA */
		/* -b  BETA */
		/* -f <offset>, offset of Waxman fun() starting position */
		/* -c  Produce connected graph */
                /* -t  target node degree */
                /* -C  look for and fix articulation points */
		/* -h  Usage */
  char c;
  int i,j;

  while ((c = getopt(argc, argv, optstring)) > 0)
	switch(c){
		case 'a':
			sscanf(optarg, "%f", &alpha);
			break;

		case 'b':
			sscanf(optarg, "%f", &beta);
			break;

		case 'c':
			conflag++;
			break;

		case 'h':
			usage();
			break;

		case 's':
			seedval = atol(optarg);
			break;

		case 'p':
			pflag++;
			break;

		case 'n':
			num = atoi(optarg);
			break;

		case 'f':
			fflag++;
			offset = atoi(optarg);
			break;

		case 't':
			sscanf(optarg, "%lf", &target);
			break;

		case 'C':
			Conflag++;
			break;

	}

  srand48(seedval);	
  printf("\nAbout to build the graph\n");
  build_graph(target);
// PUTTING IN ALTERATION HERE   --- RR
// WANT TO PRINTOUT TOPOLOGY HERE
	printadjacency();

  printf("Average node degree = %.2f", nd());
  printf("\n");
}



/*
 * Toss the vertex onto a rectangle (Xmax by Ymax), with integer grids.
 * The coordinates are recorded in struct coord[N].
 * If a vertex falls on the grid with someone else already there, we
 * throw it away and toss the next vertex, until we have produced 'n'
 * vertices. 
 */
roll_coord_dice(n)
  int n;
{
  int cnt = 0;
  short x, y;
  int i,redo;

  while (cnt < n){
	x = (unsigned short) (lrand48() % (Xmax - 10)) + 5;
	y = (unsigned short) (lrand48() % (Ymax - 10)) + 5;
	redo = 0;
	for (i = 0; i < cnt; i++){
		if (x == point[i].x && y == point[i].y) {
			redo = 1;
			break;
		}
	}
	if (redo == 0){		/* point[0] is not used */
		cnt++;
		point[cnt].x = x;
		point[cnt].y = y;
	}
  }
}

/*
 * Roll the dice again, and use Waxman's formula (exponential distribution)
 * to decide whether there is an edge between each pair of nodes.
 * Record the distance between nodes x and y in d[x][y] if there is
 * and edge between x and y.
 */

create_edges(n)
  int n;
{
  int i,j;
		/*
		 * Waxman's formula doesn't allow really reasonable
		 * graphs. Add an offset to force the reasonableness
		 */
  double df[N + N/2];
  unsigned long a;
  double q, s;

  for (i = 0; i <= LL; i++){
	/*
	 * Waxman's distribution function
	 */
	df[i] = beta * exp((double) (- i) / (double) (LL * alpha));
  }
  count = 0;
  /*
   * Now roll the dice and see if it shows the number below Waxman's
   * distribution function at that point. If it does, add an edge.
   */
  for (i = 1; i <= n; i++){
	for (j = i + 1; j <= n; j++) {
		s = sqrt((double)(point[i].x - point[j].x) * (point[i].x - point[j].x)
		     +(point[i].y - point[j].y)*(point[i].y - point[j].y));
		a = lrand48(); /* so that we could check what number shows up */
		q = (double) a / 0x7fffffff;
		if ( q <= df[(int) ((s - offset > 0) ? (s - offset) : 0)]) {
			d[i][j] = d[j][i] = (short) s;
			degree[i]++;
			degree[j]++;
			count ++;
		} else {
			d[i][j] = d[j][i] = s_infinity();
		};
	}
  }
  /*printf("\nCount = %d", count);*/
}

/*
 * If the graph in d[][] is partitioned, connect the disconnected node
 * to the nearest connected component (CC).
 */
connect_graph()

{
  int i, j, count = 0, n1, n2;
  double td, mind;


  for (i = 0; i <= num; i++) {
    mark[i] = 0;
  }

  /*
   * Do depth-first search, and mark the connected component that
   * node 1 belongs to
   */
  count = dfs(1, 1);

  while (count < num) {
    mind = infinity();
    for (i = 1; i <= num; i++)
      for (j = 1; j <= num; j++) 
	if (mark[i] && !mark[j]) {
	  td = DIST(i, j);
	  if (td < mind) {
	    mind = td;
	    n1 = i;
	    n2 = j;
	  }
	}
    /* mark the subtree rooted at n2 */
    count += dfs(n2, 1);
    d[n1][n2] = d[n2][n1] = (short) mind;
    degree[n1]++;
    degree[n2]++;
  }

  if (num <= 2) return;

  /* in one version of code I then made the graph 2-connected, I'll
   * not worry about that, since taking repairing articulation points
   * subsumes this.
   */
  if (Conflag)
    repair_articulation_point();

}
	
    
/*
 * co is used to distinguish different connected component
 */
dfs(v, co)
int v;
int co;
{
   int i;
   int count;

   mark[v] = co;
   count = 1;
   for (i = 1; i <= num; i++){
	if (d[v][i] != s_infinity()  && mark[i] == 0)
		count += dfs(i, co);
   }
   return(count);
}

/*
 * Compute delta x and delta y for a line from vertex i to vertex j
 */
deltaxy(i, j, rad, deltx, delty)
int i, j;
int rad;
int *deltx, *delty;
{
  double tang;


   if (abs((int) point[i].x - point[j].x) == 0) {
	*deltx = 0;
	*delty = (point[i].y < point[j].y) ? rad : -rad;
   } else {
	tang = (double)
	       (point[i].y - point[j].y)/(point[i].x - point[j].x);
	*deltx = rad / sqrt(1 + tang * tang);
	*delty = tang * rad / sqrt(1 + tang * tang);
	if (point[i].x > point[j].x){
		*deltx = (-1) * (*deltx);
		*delty = (-1) * (*delty);
	}
   }
}

/*
 * compute and return average node degree of the graph.
 */
float nd()
{
  int i,j, sum = 0;

  for (i = 1; i <= num; i++)
	for (j = 1; j <= num; j++){
		if (i != j && d[i][j] != s_infinity())
		  sum++;
	}
  return ((float) sum/num);
}


usage()
{
printf("Usage:\nrtg \t-a  ALPHA \n\t-b  BETA \n\t-s <seed> 16bit value for srand48()\n\
\t-n <number> of nodes in the network\n\
\t-c  construct connected graph\n\
\t-o  output the graph in text format\n\
\t-p  print list of point coordinates\n\
\t-h  Usage\n");
}

disp_matrix()
{
  int i, j, k;

  k = (num < 20) ? num : 20;

  for (i = 1; i <= k; i++){
	for(j = 1; j <= k; j++){
		printf("%-3d ",d[i][j]);
	}
	printf("\n");
  }
}


/* find and repair any articulation points.
 */
repair_articulation_point()

{
  int i, j, k, root, count, ok;
  int n1, n2;
  double mind, td;
  short row[N], col[N];


  for (i = 1; i <= num; i++) {
    /* 'remove' current node from the graph */
    for (j = 1; j <= num; j++) {
      row[j] = d[i][j];
      col[j] = d[j][i];
      mark[j] = 0;
      d[j][i] = d[i][j] = 0;
    }
    if (i == 1) 
      root = 2;
    else 
      root = 1;
    ok = 0;  /* indicates whether 'i' is an articulation point */
    while ( !ok ) {
      /* actually for to be more random we could chose a different
       * root each time.
       */
      count = dfs(root, 1);
      if (count == num - 1)
	ok = 1;
      else {  
	/* i is an articulation point.  will 'heal' it by picking 2 
	 * points at random, one in the tree rooted at 'root' and one
	 * not in the tree.  note:  this will improve the situation, but
	 * is not guaranteed to be sufficient, hence we have to repeat
	 * the dfs.
	 */
	/* possible strategies for healing:  pick 2 points farthest away,
	 * closest together, or at random.  I'll do random.
	 */
	printf("%d is an articulation point\n", i);
	mind = infinity();
	for (j = 1; j <= num; j++)
	  for (k = 1; k <= num; k++)
	    if (mark[j] && (!mark[k]) && k != i) {
	      td = DIST(j, k);
	      if (td < mind) {
		mind = td;
		n1 = j;
		n2 = k;
	      }
	    }
/*	r1 = random_uniform_int(1,count);
	r2 = random_uniform_int(1,num-count-1); 
	r3 = 0;
	r4 = 0;
	for (j = 1; j<= num; j++) {
	  if (mark[j]) {
	    r3++;
	    if (r3 == r1)
	      break;
	  }
	}
	r3 = j;
	for (j = 1; j <= num; j++) {
	  if (!mark[j] && j != i) {
	    r4++;
	    if (r4 == r2)
	      break;
	  }
	}
	r4 = j;
*/
	printf("correcting with link %d %d\n", n1, n2);
	d[n1][n2] = d[n2][n1] = 4;

	/* reset mark array so we can do the search again */
	for (j = 1; j <= num; j++)
	  mark[j] = 0;
      }
    }
    /* add current node back into the graph */
    for (j = 1; j <= num; j++) {
      mark[j] = 0;  /* maybe overkill */
      d[i][j] = row[j];
      d[j][i] = col[j];
    }
  }
  

}



/* return a random int uniformly distributed between min and max */

random_uniform_int(min, max)
int min;
int max;

{
  if (min > max)
    panic("random_uniform_int");

  return((lrand48() >> 5) % (max - min + 1) + min);
}


/* Print a message and cause a core dump */

panic(s)
     char *s;
{
  fputs(s, stderr);
  fflush(stderr);
  abort();
}


/* make a graph with a target node degree */

build_graph(target)
double target;

{
  int done = 0;
  double target_hi = target + target * terror;
  double target_lo = target - target * terror;
  double old_beta = beta; /* to restore it if I want */
  double degree_hi;
  double degree_lo;
  double beta_hi, beta_lo;
  double degree_xx;
  double last_diff;

  /* idea is to keep producing graphs until I have one with desired
   * level of connectivity.
   */

  /* printf("target = %f -- %f\n", target_lo, target_hi); */ 

  /* place the nodes */
  roll_coord_dice(num);

  beta = beta_lo = 0.0;

  create_edges(num);

  degree_lo = nd();

  if (degree_lo >= target_lo)
    panic("build_graph1");

  clear_edges();

  /* need to find an initial beta_hi */
  beta = 2.0;  /* who knows? */

  srand48(seedval + 3);	
  create_edges(num);
  degree_hi = nd();
  while (degree_hi <= target_lo) {
    beta *= 2.0;
    clear_edges();
    srand48(seedval + 3);	
    create_edges(num);
    if (conflag || Conflag) {
      connect_graph();
    }
    degree_hi = nd();
  }
  beta_hi = beta;
  if (beta > 2.0)
    beta_lo = beta/2.0;

/*  printf("initial beta range  = %g -- %g\n", beta_lo, beta_hi); */

  degree_xx = degree_hi;

  while (degree_xx >= target_hi || degree_xx <= target_lo) {
/*    printf("beta = %f -- %f\n", beta_lo, beta_hi); */
    beta = (beta_hi + beta_lo)/2.0;
/*    printf("***trying with beta = %g***\n", beta); */
    clear_edges();
    srand48(seedval + 3);	
    create_edges(num);
/*    if (conflag || Conflag) */
      connect_graph();
    degree_xx = nd();
/*    printf("degree = %f\n", degree_xx); */
    if (degree_xx >= target_hi)
      beta_hi = beta;
    else if (degree_xx < target_lo)
      beta_lo = beta;
    if (last_diff == (beta_hi - beta_lo)) {
      break;
    }
    last_diff = beta_hi - beta_lo;
  }

}

clear_edges()

{
  int i, j;

  for (i = 0; i <= num; i++) {
    degree[i] = 0;
    for (j = 0; j <= num; j++)
      d[i][j] = s_infinity();
  }
}


double infinity()
{
  return MAXDOUBLE;		
}

short s_infinity()
{
  return MAXSHORT;
}
