use strict;
use warnings;
use File::Basename;
use ExtUtils::MakeMaker;
use Config;
use File::Fetch;

# pick up our parameters from @ARGV
my %ARGV;
for (@ARGV) {
    if (/^(.*?)\=(.*)/) {
        $ARGV{$1} = $2;
    } else {
        $ARGV{$_} = 1;
    }
    $_ = '' if /^--gdal-config/;
    $_ = '' if /^--no-version-check/;
    $_ = '' if /^--no-downloads/;
}
# ... or from the environment
$ARGV{'--gdal-config'} = $ENV{PERL_GDAL_CONFIG} if $ENV{PERL_GDAL_CONFIG};
$ARGV{'--no-version-check'} = 1 if $ENV{PERL_GDAL_NO_VERSION_CHECK};
$ARGV{'--no-downloads'} = 1 if $ENV{PERL_GDAL_NO_DOWNLOADS};

# Obtain the version of GDAL for this module distribution.

my $my_gdal_version;
if (open(my $fh, "lib/Geo/GDAL.pm")) {
    for (<$fh>) {
        ($my_gdal_version) = /(\d+\.\d+\.\d+)/ if /GDAL_VERSION/;
    }
    close $fh;
} else {
    die "GDAL Perl modules not found, perhaps you need to run make generate?";
}

# search and decide which GDAL (gdal-config) to build against if not given

my $versions_may_not_differ = not $ARGV{'--no-version-check'};
my $downloads_are_allowed = not $ARGV{'--no-downloads'};
my $given_config = $ARGV{'--gdal-config'};
my $config;
my $not_found;
if ($ARGV{'--gdal-config'}) {
    if (-r $given_config) {
        $config = $given_config;
    } else {
        die "The given gdal-config '$given_config' does not exist or is unreadable.\n";
    }
} else {
    # scan known possible locations in the order of preference:
    my @configs;
    for ('../../apps/gdal-config',
         'c:/msys/1.0/local/bin/gdal-config',
         '/usr/local/bin/gdal-config',
         '/usr/bin/gdal-config') {
        push @configs, $_ if -r $_;
    }
    if (@configs) {
        $config = $configs[0];
        if (@configs) {
            print "Found gdal-config(s): '",join("', '", @configs),"'.\n";
            print "Will try '$config'.\n";
        } else {
            print STDERR "Did not find any gdal-config(s)\n";
            $not_found = 1;
        }
    }
}
$config = download_and_compile_gdal($my_gdal_version) if !$config && $not_found && $downloads_are_allowed;

my $uri = gdal_source_location($my_gdal_version);

die "Can't find GDAL development files. Please install them from \n$uri\n".
    "or define the location of gdal-config using --gdal-config=XXX\n".
    "or by setting environment variable PERL_GDAL_CONFIG to a true value." 
    unless $config;

my $gdal_version = get_gdal_version($config);
my $different_versions = $my_gdal_version ne $gdal_version;

die "The version in the given GDAL is different ($my_gdal_version ne $gdal_version) from\n".
    "the version these bindings were developed for. Either get correct GDAL, allow\n".
    "building against different version (with --no-version-check or by setting\n".
    "the environment variable PERL_GDAL_NO_VERSION_CHECK to a true value, but it is \n".
    "not recommended), or let me download and build GDAL for you \n"
    if 
    ($given_config && $different_versions && $versions_may_not_differ) ||
    (!$downloads_are_allowed && $different_versions && $versions_may_not_differ);

$config = download_and_compile_gdal($my_gdal_version, $gdal_version) 
    if $different_versions && !$given_config && $versions_may_not_differ && $downloads_are_allowed;

# still in the game

my ($INC, $LIB) = get_gdal_inc_lib($config);

my %object = ( 'Geo::GDAL' => 'gdal_wrap.o',
               'Geo::GDAL::Const' => 'gdalconst_wrap.o',
               'Geo::OGR' => 'ogr_wrap.o',
               'Geo::OSR' => 'osr_wrap.o'  );

if ($ARGV{'--debug'}) {
    print "LIB = $LIB\n";
    print "INC = $INC\n";
}

for my $module (keys %object) {
    my $add = $module;
    $add =~ s/:/_/g;
    my $LD = $Config{ld};
    $LD .= ' '.$ENV{CFLAGS} if $ENV{CFLAGS};
    $LD .= ' '.$ENV{LDFLAGS} if $ENV{LDFLAGS};
    my $OPTIMIZE = '';
    $OPTIMIZE .= ' '.$ENV{CFLAGS} if $ENV{CFLAGS};
    $OPTIMIZE .= ' '.$ENV{CPPFLAGS} if $ENV{CFLAGS};
    WriteMakefile( NAME => $module,
                   VERSION_FROM => 'lib/Geo/GDAL.pm',
                   ABSTRACT_FROM => 'lib/Geo/GDAL.pm',
                   AUTHOR => 'Ari Jolma <ari.jolma at gmail.com>',
                   LICENSE => 'mit',
                   META_MERGE => {
                       'meta-spec' => { version => 2 },
                       resources => {
                           repository => {
                               type => 'svn',
                               url  => 'https://svn.osgeo.org/gdal/trunk/gdal/swig/perl',
                               web  => 'https://trac.osgeo.org/gdal/browser/trunk/gdal/swig/perl',
                           },
                       },
                   },
                   MAKEFILE => 'Makefile_'.$add,
                   LIBS => $LIB,
                   INC => $INC,
                   OPTIMIZE => $OPTIMIZE,
                   LD => $LD,
                   OBJECT => $object{$module},
                   PM => { 'lib/Geo/GDAL.pm' => '$(INST_LIBDIR)/GDAL.pm',
                           'lib/Geo/OGR.pm' => '$(INST_LIBDIR)/OGR.pm',
                           'lib/Geo/OSR.pm' => '$(INST_LIBDIR)/OSR.pm',
                           'lib/Geo/GDAL/Const.pm' => '$(INST_LIBDIR)/GDAL/Const.pm' },
                   BUILD_REQUIRES => {
                       'File::Basename' => 0,
                       'ExtUtils::MakeMaker' => 0,
                       'Config' => 0,
                       'File::Fetch' => 0 },
                   PREREQ_PM => {
                       Carp => 0,
                       Encode => 0,
                       'Scalar::Util' => 0,
                       POSIX => 0 },
                   TEST_REQUIRES => {
                       'Scalar::Util' => 0,
                       'Test::More' => 0 },
        );
}

sub gdal_source_location {
    my $version = shift;
    return "http://download.osgeo.org/gdal/$version/gdal-$version.tar.gz";
}

sub download_and_compile_gdal {
    my ($version, $got_version) = @_;
    
    if (defined $got_version) {
        print STDERR <<end;

The GDAL development files we found have a different version
($got_version) than what these bindings were developed for
($version). I'm going to try to download and compile GDAL since you
did not prevent it by command line argument --no-downloads or
environment variable PERL_GDAL_NO_DOWNLOADS and you did not wish to be
adventurous by specifying --no-version-check or environment variable
PERL_GDAL_NO_VERSION_CHECK.

Note that this automatic download and compile is mostly meant only for
automatic tests etc. It is recommended that you download and configure
GDAL by hand.

end

    } else {
        print STDERR <<end;

I did not find GDAL development files. I'm going to try to download
and compile GDAL since you did not prevent it by command line argument
--no-downloads or environment variable PERL_GDAL_NO_DOWNLOADS and
because you did not specify gdal-config by --gdal-config or
environment variable PERL_GDAL_CONFIG.

Note that this automatic download and compile is mostly meant only for
automatic tests etc. It is recommended that you download and configure
GDAL by hand.

end

    }

    my $gdal = gdal_source_location($version);
    my $ret = 1;
    unless (-r "gdal-$version.tar.gz") {
        my $ff = File::Fetch->new(uri => $gdal);
        $ret = $ff->fetch() if $ff;
    }
    $ret = system("tar", "zxf", "gdal-$version.tar.gz") if $ret;
    $ret = chdir("gdal-$version") if $ret == 0;
    $ret = system("./configure") if $ret;
    $ret = system("make", "-j4") if $ret == 0;
    $ret = chdir("..") if $ret == 0;
    if ($ret) {
        return "./gdal-$version/apps/gdal-config";
    } else {
        die "Downloading and compiling of GDAL failed.";
    }
}

sub get_gdal_version {
    my $config = shift;
    my $version;
    if (-x $config) {
        chomp($version = `$config --version`);
    }
    else {
        # it is not recommended to be here
        if (open(my $fh, $config) || die "Can't open '$config': $!") {
            for (<$fh>) {
                ($version) = /(\d+\.\d+\.\d+)/ if /^CONFIG_VERSION/;
            }
            close $fh;
        }
        die "Can't find version from '$config'." unless $version;
    }
    return $version;
}

sub get_gdal_inc_lib {
    my $config = shift;
    my ($INC, $LIB) = ('', '');
    if (-x $config) {
        chomp($LIB = `$config --libs`);
        if ($LIB =~ /\.la$/) { # this is a not-yet-installed gdal-config
            my ($name,$path,$suffix) = fileparse($LIB);
            $LIB = "-L$path.libs -lgdal";
        } 
        chomp($INC = `$config --cflags`);
    }
    else {
        # again, it is not recommended to be here
        if (open(my $fh, $config) || die "Can't open '$config': $!") {
            for (<$fh>) {
                if (/^CONFIG_LIBS/) {
                    s/^CONFIG_LIBS="//;
                    s/"\s*$//;
                    if ($_ =~ /\.la$/) { 
                        # parse a libtool library file
                        $LIB .= parse_libtool_library_file_for_l($_);
                    } else {
                        $LIB .= $_;
                    }
                    $LIB .= ' ';
                }
                if (/^CONFIG_DEP_LIBS/) {
                    s/^CONFIG_DEP_LIBS="//;
                    s/"\s*$//;
                    $LIB .= $_;
                }
                if (/^CONFIG_CFLAGS/) {
                    s/^CONFIG_CFLAGS="//;
                    s/"\s*$//;
                    $INC .= $_;
                }
            }
            close $fh;
        }
    }
    return ($INC, $LIB);
}

sub parse_libtool_library_file_for_l {
    my $fn = shift;
    my $fh;
    my $l = '';
    if (open($fh, $fn)) {
        while (<$fh>) {
            if (/^dlname=(.*)/) {
                $l = $1;
                $l =~ s/^'//;
                $l =~ s/^lib/\-l/;
                $l =~ s/\..*$//;
                last;
            }
        }
        close $fh;
    }
    return $l;
}
