package App::bif::new::issue;
use strict;
use warnings;
use Bif::Mo;
use IO::Prompt::Tiny qw/prompt/;

our $VERSION = '0.1.5_5';
extends 'App::bif';

sub run {
    my $self = shift;
    my $opts = $self->opts;
    my $dbw  = $self->dbw;

    $dbw->txn(
        sub {
            my $start = time;
            $self->stop_work(
                stop => $start,
                save => 1,
            );

            $opts->{title} ||= prompt( 'Title:', '' )
              || return $self->err( 'TitleRequired', 'title is required' );

            if ( !$opts->{path} ) {

                my @paths = $dbw->xarrayrefs(
                    select     => 'n.path',
                    from       => 'projects p',
                    inner_join => 'nodes n',
                    on         => 'n.id = p.id',
                    order_by =>
                      [ 'n.kind = "project" DESC', 'p.local DESC', 'n.path', ],
                );

                if ( 0 == @paths ) {
                    return $self->err( 'NoProjectInRepo',
                        'task needs a project' );
                }
                elsif ( 1 == @paths ) {
                    $opts->{path} = $paths[0]->[0];
                }
                else {
                    $opts->{path} = prompt( 'Project:', $paths[0]->[0] )
                      || return $self->err( 'ProjectRequired',
                        'project is required' );
                }
            }

            my $pinfo = $self->get_project( $opts->{path} );

            if ( $opts->{status} ) {
                my ( $status_ids, $invalid ) =
                  $dbw->status_ids( $pinfo->{id}, 'issue', $opts->{status} );

                return $self->err( 'InvalidStatus',
                    'unknown status: ' . join( ', ', @$invalid ) )
                  if @$invalid;

                $opts->{issue_status_id} = $status_ids->[0];
            }
            else {
                $opts->{issue_status_id} = $dbw->xval(
                    select => 'id',
                    from   => 'issue_status',
                    where  => { project_id => $pinfo->{id}, def => 1 },
                );
            }

            $opts->{message} ||= $self->prompt_edit(
                opts => $self,
                txt  => "

# Things to keep in mind when reporting a software issue:
#
#   - What are goal you trying to achieve?
#
#   - What version of software and libraries are installed?
#
#   - What commands are you running?
#
#   - What (output) did you expect (to see)?
#
#   - What (output) actually occured?
#"
            );
            $opts->{id}        = $dbw->nextval('nodes');
            $opts->{change_id} = $self->new_change;

            $dbw->xdo(
                insert_into => 'func_new_issue',
                values      => {
                    change_id       => $opts->{change_id},
                    id              => $opts->{id},
                    issue_status_id => $opts->{issue_status_id},
                    title           => $opts->{title},
                },
            );

            $self->start_work(
                node_id       => $opts->{id},
                start         => $start,
                start_comment => "new issue",
                billable      => 1,
            );

            $self->stop_work(
                stop    => time,
                restore => 1,
            );

            $self->save_work(
                node_id   => $opts->{id},
                change_id => $opts->{change_id}
            );

            $self->end_change(
                id               => $opts->{change_id},
                action_format    => "new issue %s",
                action_node_id_1 => $opts->{id},
                message          => $opts->{message},
            );

        }
    );

    return $self->ok('NewIssue');
}

1;
__END__

=head1 NAME

=for bif-doc #create

bif-new-issue - add a new issue to a project

=head1 VERSION

0.1.5_5 (2015-08-13)

=head1 SYNOPSIS

    bif new issue [PATH] [TITLE...] [OPTIONS...]

=head1 DESCRIPTION

Add a new issue to a project.

=head1 ARGUMENTS

=over

=item PATH

The path of the project to which this issue applies. Prompted for if
not provided.

=item TITLE

The summary of this issue. Prompted for if not provided.

=back

=head1 OPTIONS

=over

=item --status, -s STATE

The initial status of the issue. This must be a valid status for the
project as output by the L<bif-list-status>(1) command. A default is
used if not provided.

=item --message, -m MESSAGE

The body of the issue. An editor will be invoked if not provided.

=back

=head1 SEE ALSO

L<bif>(1)

=head1 AUTHOR

Mark Lawrence E<lt>nomad@null.netE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2013-2015 Mark Lawrence <nomad@null.net>

This program is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 3 of the License, or (at your
option) any later version.

