/*
 * The author of this software is William Dorsey.
 * Copyright (c) 1993, 1994, 1995 by William Dorsey.  All rights reserved.
 *
 * THIS SOFTWARE IS BEING PROVIDED "AS IS", WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTY.  IN PARTICULAR, THE AUTHOR DOES NOT MAKE ANY CLAIM OR
 * WARRANTY OF ANY KIND CONCERNING THE MERCHANTABILITY OF THIS SOFTWARE OR
 * ITS FITNESS FOR ANY PARTICULAR PURPOSE.
 */

/*
 * sp124.c
 *
 * REVISION HISTORY
 *
 * DATE      RESPONSIBLE PARTY  DESCRIPTION
 * -------------------------------------------------------------------------
 * 93/11/12  B. Dorsey		Wrote original version
 * 94/11/17  B. Dorsey		Adapted for use in Nautilus
 * 94/11/30  B. Dorsey		Modified to use 6KHZ base sampling rate
 * 94/12/11  B. Dorsey		Modified to use fixed point arithmetic
 * 95/06/08  B. Dorsey		Renamed to sp124 from dmsp
 */

#include <stdio.h>

#include "machine.h"

#define FR_BITS		20
#define ZNORM		(1L << FR_BITS)

INT32		zmul(INT32, INT32);
INT32		zdiv(INT32, INT32);
UINT32		zsqrt(UINT32 v);

/* Operating Parameters */
#define FRAME_SIZE	108		/* 18 ms */
#define YPSIZE		6
#define YPMIN		((INT32)(0.001 * ZNORM))
#define YPMAX		((INT32)(0.250 * ZNORM))
#define FSROOT		((INT32)(10.3923 * ZNORM))	/* sqrt(FRAME_SIZE-1) */
#define	D0		((INT32)(1.5 * ZNORM))
#define D1		((INT32)(0.375 * ZNORM))
#define SPLIT		((INT32)(0.9375 * ZNORM))	/* (D0+D1)/2.0 */

static struct {
	INT32		y[2];
	INT32		ypdelta;
} handle;

static INT32 filt[4][2] = {	/* filter coefficients */
	{  ((INT32) ( 1.53 * ZNORM)), ((INT32) (-0.72 * ZNORM)) },
	{  ((INT32) ( 0.95 * ZNORM)), ((INT32) (-0.34 * ZNORM)) },
	{  ((INT32) ( 0.48 * ZNORM)), ((INT32) (-0.21 * ZNORM)) },
	{  ((INT32) (-0.63 * ZNORM)), ((INT32) (-0.36 * ZNORM)) }
};

/* Local functions */
static void PutBits(int bits, int nbits, UINT8 *buf);
static int GetBits(int nbits, UINT8 *buf);

void
sp124_init(void)
{
    /* initializations */
    handle.y[0] = 0;
    handle.y[1] = 0;
    handle.ypdelta = zdiv(YPMAX - YPMIN, ((1L << YPSIZE) - 1) * ZNORM);
}

int
sp124_encode(INT16 *x, UINT8 *bits)
{
    int		i, state, u;
    INT32	yy, y, phi0, phi1, step_size, step;

    /*
     * Compute normalized autocorrelation at lag 0 & 1
     * Compute step size based on RMS value
     */
    yy = ((INT32) x[0]) << (FR_BITS - 15);
    phi0 = zmul(yy, yy) + 1;		/* + 1 to prevent divide by 0 */
    step_size = phi1 = 0;

    for (i = 1; i < FRAME_SIZE; i++) {
        y = ((INT32) x[i]) << (FR_BITS - 15);	/* convert to fixed pt */
        phi0 += zmul(y, y);			/* autocorr at lag 0 */
        phi1 += zmul(yy, y);			/* autocorr at lag 1 */
        step_size += zmul(y - yy, y - yy);	/* part of rms calc */
        yy = y;
    }
    phi1 = zdiv(phi1, phi0);			/* normalize phi1 */

    /* select predictor state */
    if (phi1 > ((INT32) (0.7 * ZNORM)))
	state = 0;
    else if (phi1 > ((INT32) (0.4 * ZNORM)))
	state = 1;
    else if (phi1 > 0)
	state = 2;
    else
	state = 3;

    /* compute step sized based on RMS value of input */
    step_size = zsqrt(step_size);
    step_size = zdiv(step_size, FSROOT);

    /* check step size for bounds */
    if (step_size < YPMIN)
	step_size = YPMIN;
    else if (step_size > YPMAX)
	step_size = YPMAX;
	
    /* quantize step size to YPSIZE bits */
    step = zdiv(step_size - YPMIN, handle.ypdelta);
    step >>= FR_BITS;

    step_size = zmul(step << FR_BITS, handle.ypdelta);
    step_size += YPMIN;

    /* save predictor state and quantized step size in output */
    bits[0] = state + (step << 2);

    /* initialize PutBits() */
    PutBits(0, 0, &bits[1]);

    /* main loop */
    for (i = 0; i < FRAME_SIZE; i++) {
	/* apply linear predictive filter */
	y = zmul(filt[state][0], handle.y[0]);
	y += zmul(filt[state][1], handle.y[1]);

	handle.y[1] = handle.y[0];

	step = ((INT32) x[i] << (FR_BITS - 15)) - y;
	step = zdiv(step, step_size);
	if (step < 0) {
	    if (step < -SPLIT) {
                y -= zmul(D0, step_size);
	        u = 0;
	    }
	    else {
                y -= zmul(D1, step_size);
	        u = 1;
	    }
	}
	else {
            if (step < SPLIT) {
                y += zmul(D1, step_size);
                u = 2;
	    }
	    else {
                y += zmul(D0, step_size);
	        u = 3;
	    }
	}

	/* output bits for next delta */
	PutBits(u, 2, (char *)0);

	handle.y[0] = y;
    }
}

int
sp124_decode(UINT8 *bits, INT16 *x)
{
    int		i, state, v;
    INT32	y, step_size, step;

    /* get state & step size */
    state = bits[0] & 0x3;	/* ls 2 bits */
    step = bits[0] >> 2;	/* ms 6 bits */
    
    /* decode step size from quantized step size */
    step_size = zmul(step << FR_BITS, handle.ypdelta);
    step_size += YPMIN;

    /* initialize GetBits() */
    GetBits(0, &bits[1]);

    /* compute output from input bits */
    for (i = 0; i < FRAME_SIZE; i++) {
	/* apply linear predictive filter */
	y = zmul(filt[state][0], handle.y[0]);
	y += zmul(filt[state][1], handle.y[1]);

        handle.y[1] = handle.y[0];

	/* get input */
	v = GetBits(2, (char *)0);

	/* update output */
	switch (v) {
	case 0:
	    y -= zmul(D0, step_size);
	    break;
	case 1:
	    y -= zmul(D1, step_size);
	    break;
	case 2:
	    y += zmul(D1, step_size);
	    break;
	case 3:
	    y += zmul(D0, step_size);
	    break;
	}

	/* save output */
	handle.y[0] = y;
	x[i] = (INT16) (y >> (FR_BITS - 15));
    }
}

static void
PutBits(int bits, int nbits, UINT8 *buf)
{
	int	i;
	UINT8			imask;
	static UINT8		data;
	static UINT8		mask;
	static UINT8		*ptr;

	if (buf) {
		data = 0;
		mask = 0x80;
		ptr = buf;
	}
	else {
		imask = 1<<(nbits-1);
		for (i=0; i<nbits; i++) {
			if (bits & imask) {
				data |= mask;
			}
			imask >>= 1;
			mask >>= 1;
			if (mask == 0) {
				*(ptr++) = data;
				data = 0;
				mask = 0x80;
			}
		}
	}
}

static int
GetBits(int nbits, UINT8 *buf)
{
	int	i, bits;
	static UINT8		mask;
	static UINT8		*ptr;

	if (buf) {
		mask = 0x80;
		ptr = buf;
	}
	else {
		bits = 0;
		for (i=0; i<nbits; i++) {
			bits <<= 1;
			bits |= (*ptr & mask) ? 1 : 0;
			mask >>= 1;
			if (mask == 0) {
				mask = 0x80;
				ptr++;
			}
		}
	}

	return bits;
}
