/*
 * The author of this software is William Dorsey.
 * Copyright (c) 1993, 1994, 1995 by William Dorsey.  All rights reserved.
 *
 * THIS SOFTWARE IS BEING PROVIDED "AS IS", WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTY.  IN PARTICULAR, THE AUTHOR DOES NOT MAKE ANY CLAIM OR
 * WARRANTY OF ANY KIND CONCERNING THE MERCHANTABILITY OF THIS SOFTWARE OR
 * ITS FITNESS FOR ANY PARTICULAR PURPOSE.
 */

/*
 * talk.c
 *
 * REVISION HISTORY
 *
 * DATE      RESPONSIBLE PARTY  DESCRIPTION
 * -------------------------------------------------------------------------
 * 93/12/08  B. Dosrey          Module created by breakup of nautilus.c
 * 94/10/15  P. Mullarky        Allow both sides to change PTT status
 * 95/03/11  B. Dorsey          Add encryption
 */

#include <stdio.h>
#include <malloc.h>

#include "nautilus.h"

/* external variables */
extern struct coder_t coders[];	/* coder table */
extern struct param_t params;	/* operating parameters */
extern UINT8    beep[];		/* the turnaround cue tone */

void
Talk(int mode)
{
    UINT32          timer;
    int             i;
    int             done;
    int             err_count;
    int             sample_rate;
    int             frame_size;
    int             output_size;
    int             frames_per_pkt;
    struct packet_t packet;
    UINT8          *ptr;
    UINT8          *audio_data;
    UINT8          *output_data;
    INT16          *samples;

    /* initialize memory */
    sample_rate = coders[params.coder.index].sample_rate;
    frame_size = coders[params.coder.index].frame_size;
    output_size = coders[params.coder.index].output_size;
    frames_per_pkt = coders[params.coder.index].frames_per_pkt;
    audio_data = malloc(frame_size + 1);
    output_data = malloc(output_size * frames_per_pkt);
    samples = malloc(frame_size * sizeof(INT16));

    /* initialize coder */
    coders[params.coder.index].init();

    /* main loop initialization */
    err_count = done = 0;
    AudioFlow(mode);
    puts("\n***** BEGIN COMMUNICATING *****\n");
    ShowMode(mode);		/* display operating mode to user */

    /* main loop */
    do {
    	switch (mode) {
    	case TRANSMIT:
    	    /*
    	     * see if there are any incoming packets to be processed
    	     */
    	    if (IncomingPort()) {
retry1:		if (ReadPkt(&packet) == SUCCESS) {
    	            switch (packet.type) {
    	            case EOT:
			done = 1;
			break;
    	            case RECV:
			mode = RECEIVE;
			AudioFlow(mode);/* resume audio sampling */
			SendPkt(ACK, NULL, 0);
			ShowMode(mode);	/* display operating mode */
			coders[params.coder.index].init();
			ptt();		/* flush any keyboard input */
			break;
    	            case XMIT:
    	                SendPkt(ACK, NULL, 0);
			break;
    	            default:
			goto retry1;
    	            }
    	            err_count = 0;
    	        }
    	        else {
    	            if (++err_count > 6) {
    	            	error(MSG_FATAL, "Repeated errors detected");
    	            }
    	            else {
			error(MSG_WARNING, "received garbled packet");
    	            }
    	        }
    	    }

	    /*
    	     * see if the user wants to change modes
    	     */
    	    if (ptt()) {
		SendPkt(XMIT, NULL, 0);
    	        timer = Clock();	/* initialize timeout timer */
    	        do {
    	            if (Clock() - timer > 2) {
    	            	SendPkt(XMIT, NULL, 0);
    	            }
    	            ReadPkt(&packet);
    	            /*
    	             * see if a race condition where both sides are
    	             * requesting a turnaround is occuring
    	             */
    	            if (packet.type == RECV) {
    	            	SendPkt(ACK, NULL, 0);	/* fix race condition */
    	            	break;
		    }
    	            if (Clock() - timer > 6) {
    	            	error(MSG_FATAL, "No response from other side");
    	            }
    	        } while (packet.type != ACK);
    	        err_count = 0;
    	        mode = RECEIVE;
    	        AudioFlow(mode);
    	        ShowMode(mode);		/* display operating mode */
    	        coders[params.coder.index].init();
    	        ptt();			/* flush any keyboard input */
    	    }

    	    /*
    	     * see if the user wants to quit
    	     */
    	    if (quit()) {
    	    	SendPkt(EOT, NULL, 0);	/* tell other side we're quitting */
    	    	SendPkt(EOT, NULL, 0);	/* just in case... */
    	    	sleep(1);		/* let buffers empty XXX */
    	    	done = 1;
    	    }

    	    /*
    	     * exit TRANSMIT mode if appropriate
    	     */
	    if ((mode == RECEIVE) || (done == 1))
	    	break;

	    /*
	     * process incoming audio data and transmit it
	     */
	    for (i = 0, ptr = output_data; i < frames_per_pkt; i++, ptr = &ptr[output_size]) {
#if DEBUG > 0 && defined(sun)
		/* check for overflow on audio device */
		if (AudioOverflow())
		    error(MSG_WARNING, "audio overflow detected");
#endif

	     	/* read samples from the audio device */
	     	if (ReadAudio(audio_data, 1) < 0)
	     	    error(MSG_WARNING, "ReadAudio() failure!\n");

	     	/* convert samples into standard form */
	     	Raw2Std(audio_data, samples, frame_size);

	     	/* Encode audio data */
		coders[params.coder.index].encode(samples, ptr);
	    }

	    if (params.crypto.key1.type != NONE) {
		/*
		 * encrypt compressed audio data
		 */
		cfb_encrypt(&params.crypto.xmit_ctx, output_data,
			    output_size * frames_per_pkt);
	    }

	    SendPkt(DATA, output_data, output_size * frames_per_pkt);
	    break;

    	case RECEIVE:
	    /*
    	     * see if the user wants to change modes
    	     */
    	    if (ptt()) {
		SendPkt(RECV, NULL, 0);
    	        timer = Clock();	/* initialize timeout timer */
    	        do {
    	            if (Clock() - timer > 2) {
    	            	SendPkt(RECV, NULL, 0);
    	            }
    	            ReadPkt(&packet);
    	            if (Clock() - timer > 6) {
    	            	error(MSG_FATAL, "No response from other side");
    	            }
    	        } while (packet.type != ACK);
    	        err_count = 0;
    	        mode = TRANSMIT;
    	        AudioFlow(mode);
    	        ShowMode(mode);		/* display operating mode */
    	        coders[params.coder.index].init();
    	        ptt();			/* flush any keyboard input */
    	        break;
    	    }

    	    /*
    	     * see if the user wants to quit
    	     */
    	    if (quit()) {
    	    	SendPkt(EOT, NULL, 0);	/* tell other side we're quitting */
    	    	SendPkt(EOT, NULL, 0);	/* just in case... */
    	    	sleep(1);		/* let buffers empty XXX */
    	    	done = 1;
    	    	break;
    	    }

    	    /*
    	     * get a packet from transmitter or timeout
    	     */
    	    timer = Clock();		/* initialize timeout timer */
    	    while (ReadPkt(&packet) == -1) {
    	    	if (Clock() - timer > 2) {
    	    	    SendPkt(XMIT, NULL, 0);	/* ping other side */
    	    	}
    	    	if (Clock() - timer > 6) {
    	    	    error(MSG_FATAL, "No response from other side");
		}
    	    }
    	    err_count = 0;

    	    /*
    	     * process packets from transmitter
    	     */
    	    switch (packet.type) {
	    case DATA:
	    	if (packet.length == output_size * frames_per_pkt) {
		    if (params.crypto.key1.type != NONE) {
			/*
			 * decrypt incoming packet
			 */
			cfb_decrypt(&params.crypto.recv_ctx, packet.data,
			            output_size * frames_per_pkt);
		    }

	    	    /*
	    	     * decode incoming audio frames
	    	     */
	    	    for (i = 0, ptr = packet.data; i < frames_per_pkt; i++, ptr = &ptr[output_size]) {
	    	    	/* decode audio data */
	    	    	coders[params.coder.index].decode(ptr, samples);

	    		/* convert samples to raw form */
	    		Std2Raw(samples, audio_data, frame_size);

#if DEBUG > 0 && defined(sun)
			/* check for underflow on audio device */
			if (AudioUnderflow())
			    error(MSG_WARNING, "audio underflow detected");
#endif

	    		/* output samples to audio device */
	    		WriteAudio(audio_data, 1);
		    }
	    	}
	    	else {
	    	    error(MSG_WARNING, "received garbled packet");
	    	}
	    	break;
	    case XMIT:
	    	while (WriteAudio(beep, (sample_rate / 1000) * BEEP_MS / frame_size) == FAIL)
	    	     ;
	    	DrainAudio();
	    	mode = TRANSMIT;
	    	AudioFlow(mode);
	    	SendPkt(ACK, NULL, 0);
	    	ShowMode(mode);
	    	coders[params.coder.index].init();
	    	ptt();
	    	break;
	    case RECV:
	    	SendPkt(ACK, NULL, 0);
	    	break;
            case EOT:
		done = 1;
		break;
	    default:
	    	break;
    	    }
	    break;

    	} /* switch (mode) */
    } while (!done);

    puts("\n\n***** TRANSMISSION COMPLETE *****\n\n");
    free(samples);
    free(output_data);
    free(audio_data);
}
