// === Starter Application 1.0.0 === //

/* ======================================================

FUNCTION:  CheckAVS()

INPUT:     payevent - the Payevent object previously constructed that
	contains AVS data returned by the Acquirer after the auth.

RETURN:   an integer indicating which part of the AVS check failed - 
		address or zip.  Returns a 0 for no failures, 1 for 
		address fail, 2 for zip fail.

====================================================== */

function CheckAVS(payevent)
{
  var avsfail = 0;

  avs = payevent.avsResp;

  // Check for Address failure.
  if (avs.substring(0, 1) == "N")
  {
     avsfail = 1;
  }

  // Check for Zip Code failure.
  if (avs.substring(1, 2) == "N")
  {
     avsfail = 2;
  }
  return (avsfail);

} // END CheckAVS()



/* ======================================================

FUNCTION:  CreatePayEvent()

INPUT:     amount - amount of the authorization

RETURN:   a Payevent object initialized with the merchantReference, amount,
	and ID.

DESCRIPTION:   Creates a Payevent object, initializes some properties
	required for authorizations, and returns the object.  The function
	also creates a new record in the LP_PURCHASE table for the
	authorization and sets the status to "AUTHORIZING".

====================================================== */

function CreatePayEvent(amount, merchantReference)
{
  var nextPurchaseID       = GetNextPurchaseID();

  // If no merchantReference was supplied, fill it in for the merchant
  //
  if (merchantReference == null)
	merchantReference = GenerateMerchantReference();

  //
  // Create a payevent object 
  //
  payevent                 = new PayEvent(merchantReference);
  payevent.amount          = amount;

  //
  // Save payevent information in database
  //
  database.beginTransaction();

  
  error = database.execute("insert into LP_PURCHASE values (" +
             nextPurchaseID + ",'" + 
             payevent.merchantReference + "'," +
	     "null,'" +
	     project.currency + "'," +
             payevent.amount + 
             ", null, null, null, null, null, null, null, 'AUTHORIZING')"); 

  if (error)
	if ((error == 5) || (error == 7))
		PrintError("Database Error.", "Could not insert into LP_PURCHASE table.  " +
			"Error code " + error + ": " + database.majorErrorMessage());
	else if (error)
		PrintError("Database Error.", "Could not insert into LP_PURCHASE table.  " +
				"Error code " + error + " returned from the database.");

  database.commitTransaction();

  //
  // ID is not a built-in property of the payevent object.
  // It is created here to remember the purchase ID.
  //
  payevent.ID              = nextPurchaseID;

  return (payevent);

} // END CreatePayEvent()



/* ======================================================

FUNCTION:  SavePayEvent()

INPUT:     payevent - the Payevent object previously constructed that
		contains AVS data returned by the Acquirer after the auth.
	   slip     - the previously constructed slip object associated
			with the auth
	   avsFail  - the code indicating if any one (or both) AVS checks failed.
	authSucceed - the boolean value returned by the authorization
			indicating success or failure of the auth

RETURN:   none

DESCRIPTION:  Saves information about the purchase, including STATE information, in the database.  It actually updates the record previously created
for this authorization by the CreatePayEvent() and changes
the STATE of the transaction from "AUTHORIZING" to one of the following 
states:

STATE CHANGES:   

	Changes state in LP_PURCHASE table from AUTHORIZING to:

	If FailOnAVS(avsFail) returns true:		AVSFAILED
	If authSucceed is false:     			AUTHFAILED
	If authSucceed is true
	   AND FailOnAVS(avsFail) returns FALSE:  	AUTHORIZED

====================================================== */

function SavePayEvent(payevent, slip, avsFail, authSucceed, cardHolderName)
{
  database.beginTransaction();

  cursor = database.cursor("select * from LP_PURCHASE where ID = " + payevent.ID + " AND status = 'AUTHORIZING'", true);

  if (cursor.next())
  {
     cursor.cardHolderName = cardHolderName;
     cursor.authCode = payevent.authCode;
     cursor.paySvcData = payevent.paySvcData;
     cursor.avsResp = payevent.avsResp;
     cursor.eventTime = payevent.eventTime;
//     var curDate = new Date();
//     cursor.eventTime = curDate;
     cursor.slipID = slipID;	// slipID is a global var set in GenerateSlip()
    
     /* Check merchant setting in "FailOnAVS()" to see if an AVS failure
        has occured that, according to the merchant's business rules
        constitues a failed authorization based on AVS, producing a
        status of "AVSFAILED"  
     */
     if (FailOnAVS(avsFail))
     {
        cursor.status = "AVSFAILED";
     }
     else 
     {
	if (authSucceed)
        	cursor.status = "AUTHORIZED";
	else
		cursor.status = "AUTHFAILED";
     }

     if ((error = cursor.updateRow("LP_PURCHASE")))
     {
        cursor.close();
        database.rollbackTransaction();
	  if ((error == 5) || (error == 7))
		PrintError("Database Error.", "Could not update LP_PURCHASE table.  " +
			"Error code " + error + ": " + database.majorErrorMessage());
	  else if (error)
		PrintError("Database Error.", "Could not  update LP_PURCHASE table  " +
				"Error code " + error + " returned from the database.");
   

     }
  }
  else
  {
     cursor.close();
     database.rollbackTransaction();
     PrintError("Transaction not found", "No purchase transaction with ID " +
                nextPurchaseID +
                " and status AUTHORIZING in database.");
  }
  cursor.close();

  database.commitTransaction();

}  // END SavePayEvent()



/* ======================================================

FUNCTION:  Authorize

INPUT:     authData - a JavaScript object that contains
	information about the customer, credit card, and order
	description.  The specific properties of authData which
	are used in this function are:

	amount - the amount of the transcation
	currency - the currency type (e.g. USD)
	orderDesc - a description of the order
	cardHolderName - name of the card holder as it appears on the credit card.
	cardNumber - credit card number
	cardExpDate - expiration date
	cardType - credit card type
	address - address of card holder
	zip - zipcode of card holder		

        merchantReference - a reference number that the merchant
			may assign to the transaction

RETURN:   true, if the operation succeds.  

	The function does not return if the authorization fails, but 
        rather calls one of the error functions, PrintFmtError() 
	or PrintError().

====================================================== */

function Authorize( authData )
{

  var returnval = false;

// Make sure Credit Card number and Expiration date fields are NOT blank
//
if (authData.cardNumber.length && authData.cardExpDate.length)
{

   // Instantiate new objects
   //
   merchant  = new Merchant();
   terminal  = new Terminal();
   processor = new Processor(project.acquirer);
   if (processor.bad())
   {
      PrintFmtError("Failed to construct processor.",
                    processor.getStatusMessage());
   }

   //
   // Generate slip
   //

   slip = GenerateSlip(authData, processor);
   var merRef = slip.merchantReference;

   //
   // Create payevent object
   //
   payevent = CreatePayEvent(authData.amount, merRef);
   
   //
   // Authorize the transaction
   //
   if ((authResult = processor.authorize(terminal, merchant, payevent, slip)))
   {
      avsFail = CheckAVS(payevent);

      // Save transaction info to the database
      SavePayEvent(payevent, slip, avsFail, authResult, authData.cardHolderName);


      // If AVS Check fails according to rules set in "FailOnAVS", then the
      // authorization should fail with status "AVSFAIL".  Otherwise, 
      // the authorization has succeeded at this point.
      //
      if (FailOnAVS(avsFail))		// see 'failOnAVS()' description.
      {
	 PrintAVSError(avsFail);
      }
      else
      {
  	 returnval  = true;	// else, no errors so return TRUE    
      }
   }
   else
   {
      // Save transaction info to the database
      SavePayEvent(payevent, slip, false, authResult, authData.cardHolderName);
      PrintFmtError("Credit Card Authorization Failed.",
                    processor.getStatusMessage());
         }
}
else
{
   PrintError("Input Error", "Must input credit card number and expiration date");
}

	return returnval;

} // END FUNCTION Authorize()

