// === Starter Application 1.0.0 === //

/* ======================================================

FUNCTION:  PrepareToSettle()

INPUT:     batch - batch object related to the batch being settled

	   merchant,
	   terminal,
	   processor - objects, previously created, that are required
			to perform a processor.settle() operation.

RETURN:   none.

DESCRIPTION:  This function changes the state of the current batch from
	OPENED to SETTLING.  The batch is not actually settled with the
	Acquirer in this function (see Settle() below).

	      This function also attemps to finish crediting and 
	capturing any pending transactions in the CREDITING or 
	CAPTURING state in the batch.  If they cannot be completed, 
	an error may occur when the batch is actually settled in SettleBatch()
	because the totals for Credits and Captures may not add up.
	However, the administrator will receive an error indicating that
	the totals do not match, and one of the actions they should take
	is to ensure that all pending Credits or Captures have completed
	before attempting to settle the batch again.  

		It should be documented that when a settle fails, a possible
	source of the problem could be that some transactions remain 
	unresolved (i.e. remain in the CREDITING or CAPTURING state) and
	must be resolved before the batch may be settled. It should also
	be documented that the	administrator should check with the 
	Acquirer if pending  Captures or Credits remain pending after 
	several attempts to settle a batch (e.g. it could indicate 
	connection problems for example, and the responses from the 
	acquirers regarding captures/credits may not be reacing the
	terminal).

STATE CHANGES:  The batch indicated by the batch object has it's state
	in the database changed from OPENED to:
		SETTLING

====================================================== */

function PrepareToSettle(batch, merchant, terminal, processor, merchantReference)
{

   database.beginTransaction();

   //
   // Update batch status to "SETTLING" 
   // if not already in SETTLING state
   //
   batchCursor = database.cursor("select STATUS from LP_BATCH where ID = " + batch.ID, true);
     
   if (batchCursor.next())
   {
      if (batchCursor.status == "OPENED")
      {
         batchCursor.status = "SETTLING";
         if ((error = batchCursor.updateRow("LP_BATCH")))
         {
            batchCursor.close();
            database.rollbackTransaction();
	if ((error == 5) || (error == 7))
		PrintError("Database Error.", "Could not update LP_BATCH table to settling.  " +
			"Error code " + error + ": " + database.majorErrorMessage());
	else if (error)
		PrintError("Database Error.", "Could not insert into LP_BATCH table to settling.  " +
				"Error code " + error + " returned from the database.");
         }
      }
      else
      {
         if (batchCursor.status != "SETTLING")
	 {
            batchCursor.close();
            database.rollbackTransaction();
            PrintError("Transaction status error", "Batch with ID "+ batch.ID,
                       " has invalid status " + batchCursor.status +
                       " in database.");
   	 }      

      } // end "if (batchCursor.status == "OPENED")"

   } // end "if (batchCursor.next())"

   else
   {
      batchCursor.close();
      database.rollbackTransaction();
      PrintError("Batch not found", "No batch with ID " + batch.ID +
                 " in database.");
   }
   batchCursor.close();
   database.commitTransaction();



} // END PrepareToSettle()




/* ======================================================

FUNCTION:  SettleBatch()

INPUT:     batch - batch object related to the batch being settled

	   merchant,
	   terminal,
	   processor - objects, previously created, that are required
			to perform a processor.settle() operation.

RETURN:   none.

DESCRIPTION:  SettleBatch() actually peforms the settle operation via
	communication with the Acquirer across the Internet.  It reports
	an error if the operation failed for some reason.  If it succeedes,
	the state of the batch changes to SETTLED.
		

STATE CHANGES:  If the batch settle is successful the state changes from
	SETTLING to SETTLED.  Otherwise, no change takes place.

====================================================== */

function SettleBatch(batch, merchant, terminal, processor, merchantReference)
{
   //
   // Total up amount and number of transactions to settle
   //
   var totalSalesAmt = 0;
   var totalCreditAmt = 0;
   var salesCount = 0;
   var creditCount = 0;

   database.beginTransaction();

   purchaseCursor = database.cursor("select sum(amount), count(distinct ID) from LP_PURCHASE where batchID = " + batch.ID + " AND status = 'CAPTURED'");

   if (purchaseCursor.next())
   {
      if ( (purchaseCursor[0] + "") != "null")
 	     totalSalesAmt = purchaseCursor[0];
      salesCount = purchaseCursor[1];
   }
   else
   {
      purchaseCursor.close();
      database.rollbackTransaction();
      PrintError("Database Retrieval Error", "Failed to get totalSalesAmt and salesCount for batch in settlebatch function for batch #" + batch.ID);
   }
   purchaseCursor.close();


   creditCursor = database.cursor("select sum(amount), count(distinct ID) from LP_PURCHASE where batchID = " + batch.ID + " AND status = 'CREDITED'");

   if (creditCursor.next())
   {
      if( (creditCursor[0] + "") != "null")
	      totalCreditAmt = creditCursor[0];
      creditCount = creditCursor[1];
   }
   else
   {
      creditCursor.close();
      database.rollbackTransaction();
      PrintError("Database Retrieval Error", "Failed to get totalCreditAmt and creditCount for batch in settlebatch function for batch #" + batch.ID);
   }
   creditCursor.close();


    
   //
   // Set up batch object
   //

   if (merchantReference == null)
	merchantReference = GenerateMerchantReference();
   
   batch.merchantReference = merchantReference; 
   batch.totalSalesAmount = totalSalesAmt + "";
   batch.totalCreditAmount = totalCreditAmt + "";
   batch.salesCount = salesCount;
   batch.creditCount = creditCount;

   //
   // Settle batch
   //
   if (processor.settleBatch(terminal, merchant, batch))
   {
      batchCursor = database.cursor("select * from LP_BATCH where ID = " + batch.ID + " AND status = 'SETTLING'", true);

      if (batchCursor.next())
      {
         batchCursor.merchantReference = batch.merchantReference;
         batchCursor.totalSalesAmount = batch.totalSalesAmount;
         batchCursor.totalCreditAmount = batch.totalCreditAmount;
         batchCursor.salesCount = batch.salesCount;
         batchCursor.creditCount = batch.creditCount;
         batchCursor.status = "SETTLED";
	 curDate = new Date();
	 batchCursor.timeSettled = curDate;

         if ((error = batchCursor.updateRow("LP_BATCH")))
         {
            batchCursor.close();
            database.rollbackTransaction();
	if ((error == 5) || (error == 7))
		PrintError("Database Error.", "Could not update LP_BATCH table to SETTLED after successful settle.  " +
			"Error code " + error + ": " + database.majorErrorMessage());
	else if (error)
		PrintError("Database Error.", "Could not update LP_BATCH table to SETTLED after successful settle.  " +
				"Error code " + error + " returned from the database.");
         }
         PrintSettledBatch(batch);
      }
      else
      {
         batchCursor.close();
         database.rollbackTransaction();
         PrintError("Batch Not Found", "No batch with ID " + batch.ID +
                    " and status SETTLING in database.");
      }
      batchCursor.close();
   }
   else
   {
      database.rollbackTransaction();
      PrintFmtError("Failed to settle batch.", processor.getStatusMessage());
   }
   database.commitTransaction();

}  // END SettleBatch()
