/*
SKIP Source Code License Statement:
------------------------------------------------------------------
  Copyright
  Sun Microsystems, Inc.


  Copyright (C) 1994, 1995 Sun Microsystems, Inc.  All Rights
  Reserved.

  Permission is hereby granted, free of charge, to any person
  obtaining a copy of this software and associated documentation
  files (the "Software"), to deal in the Software without
  restriction, including without limitation the rights to use,
  copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software or derivatives of the Software, and to 
  permit persons to whom the Software or its derivatives is furnished 
  to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be
  included in all copies or substantial portions of the Software.

  The Software must not be transferred to persons who are not US
  citizens or permanent residents of the US or exported outside
  the US (except Canada) in any form (including by electronic
  transmission) without prior written approval from the US
  Government. Non-compliance with these restrictions constitutes
  a violation of the U.S. Export Control Laws.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
  OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
  NONINFRINGEMENT.  IN NO EVENT SHALL SUN MICROSYSTEMS, INC., BE LIABLE
  FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
  OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
  CONNECTION WITH THE SOFTWARE OR DERIVATES OF THIS SOFTWARE OR 
  THE USE OR OTHER DEALINGS IN THE SOFTWARE.

  Except as contained in this notice, the name of Sun Microsystems, Inc.
  shall not be used in advertising or otherwise to promote
  the sale, use or other dealings in this Software or its derivatives 
  without prior written authorization from Sun Microsystems, Inc.
*/

#pragma ident "@(#)dh_keygen.C	1.8 95/11/21 Sun Microsystems"

//
// This utility generates a Diffie-Hellman key-pair, based
// on the exponent size, the DH params g and p provided,
// and the seed generated from the random keystrokes input.
// It stores the secret i in a file "my_secret_i".
// It stores the public value (g^i mod p) in a file "my_nsid8_cert".
//

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include "Time.h"
#include "Bigint.h"
#include "Bstream.h"
#include "ObjId.h"
#include "SkipCert.h"
#include "X509Cert.h"
#include "HashCert.h"
#include "Sig.h"
#include "bn.h"
#include "prime.h"
#include "bn_glue.h"
#include "bnprint.h"
#include "dh_1024_params.h"
#include "dh_2048_params.h"
#include "dsa.h"
#include "utils.h"


#define DEFAULT_EXPONENT_SIZE	256	// bits
#define DEFAULT_MODULUS_SIZE	1024	// bits
#define DEFAULT_CERT_LIFETIME	2	// 2 days
#define MY_REAL_SECRET_FILE	"my_dh_secret"
#define MY_NSID8_CERT_FILE	"my_nsid8_cert"

void
usage(const char *progname)
{
	fprintf(stderr, "Usage: %s [-e exponent size (bits)] ",
			progname);
	fprintf(stderr, "\t[-f secret filename ]");
	fprintf(stderr, "\t[-l certificate lifetime (days) ]");
	fprintf(stderr, "\t[-m modulus size (1024 or 2048)]");
	exit(1);
}

main(int argc, char **argv)
{
	int	c, exp_size = DEFAULT_EXPONENT_SIZE;
	int	mod_size = DEFAULT_MODULUS_SIZE;
	int	lifetime = DEFAULT_CERT_LIFETIME;
	char	*secretFile = NULL;
	int	i, len;
	struct	BigNum sec, pub, gen, mod;
	Bstream	secret, nullbstr;
	Bigint	g, p, pubval;
	extern char *optarg;
	
	while ((c = getopt(argc, argv, "e:f:l:m:")) != EOF) {
		switch (c) {
		case 'e':
			exp_size = atoi(optarg);
			break;
		case 'f':
			secretFile = optarg;
			break;
		case 'l':
			lifetime = atoi(optarg);
			break;
		case 'm':
			mod_size = atoi(optarg);
			break;
		case '?':
		default:
			(void) usage(argv[0]);
		}
	}

	if (exp_size <= 0 || lifetime <= 0)
		(void) usage(argv[0]);

	if (mod_size != 1024 && mod_size != 2048)
		(void) usage(argv[0]);

	bnInit();
	bnBegin(&gen);
	bnBegin(&mod);
	bnBegin(&pub);
	bnBegin(&sec);

	if (mod_size == 1024) {
		g = Bigint(base_1024, sizeof(base_1024));
		p = Bigint(mod_1024, sizeof(mod_1024));
	} else {
		g = Bigint(base_2048, sizeof(base_2048));
		p = Bigint(mod_2048, sizeof(mod_2048));
	}

	if (secretFile == NULL) {
		// generate secret
		prompt_user_for_randomness();
		if (genRandBn(&sec, (unsigned)exp_size) < 0) {
			fprintf(stderr, "Error generating random number\n");
			exit(1);
		}
		BigNum_to_Bstream(&sec, &secret);
		secret.store(MY_REAL_SECRET_FILE);
	} else {
		// read the secret from the file
		secret = File_to_Bstr(secretFile);
		if (secret == nullbstr) {
			fprintf(stderr,
				"couldn't read secret from %s\n",
				secretFile);
			exit(1);
		}
	}
#ifdef DEBUG
	fprintf(stderr, "secret: %s\n",
		(const char *) secret.gethexstr());
#endif

	Bigint_to_BigNum(&g, &gen);
	Bigint_to_BigNum(&p, &mod);

	// calculate the public value
	if (bnExpMod(&pub, &gen, &sec, &mod) < 0) {
		fprintf(stderr, "Error calculating public value\n");
		exit(1);
	}
	BigNum_to_Bigint(&pub, &pubval);

				// # of days * hours * mins * secs
	unsigned long notafter = NTPNOW + (lifetime * 24 * 60 * 60);

	HashCert cert(g, p, pubval, NTPNOW, notafter);

	cert.print();
	Bstream certstrm = cert.encode();
	certstrm.store(MY_NSID8_CERT_FILE);

	bnEnd(&sec);
	bnEnd(&gen);
	bnEnd(&mod);
	bnEnd(&pub);
	exit(0);
}
